/*
 * Copyright (c) 2006 Raritan, All Rights Reserved.
 * Author: Ralf Guenther ralf.guenther@raritan.com
 */

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <sys/types.h> 
#include <sys/stat.h>

#include <pp/strstream.h>

#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>
#include <ipmitool/ipmi_strings.h>
#include <ipmitool/bswap.h>
#include <ipmi_return.h>
#include <ipmitool/ipmi_raritanoem.h>

static const struct valstr raritanoem_cmd_strs[] = {
    { IPMI_RARITANOEM_GET_FIRMWARE_VERSION,      "Get Firmware Version" },
    { IPMI_RARITANOEM_START_FIRMWARE_UPGRADE,    "Start Firmware Upgrade" },
    { IPMI_RARITANOEM_UPLOAD_FIRMWARE,           "Upload Firmware" },
    { IPMI_RARITANOEM_FLASH_FIRMWARE,            "Flash Firmware" },
    { IPMI_RARITANOEM_CANCEL_FIRMWARE_UPGRADE,   "Cancel Firmware Upgrade" },
    { IPMI_RARITANOEM_FINALIZE_FIRMWARE_UPGRADE, "Finalize Firmware Upgrade" },
    { IPMI_RARITANOEM_CONFIG_GET,                "Get Config Key" },
    { IPMI_RARITANOEM_CONFIG_SET,                "Set Config Key" },
    { IPMI_RARITANOEM_START_CONFIG_BACKUP,       "Start Config Backup" },
    { IPMI_RARITANOEM_BACKUP_CONFIG,             "Backup Config" },
    { IPMI_RARITANOEM_FINISH_CONFIG_BACKUP,      "Finish Config Backup" },
    { IPMI_RARITANOEM_START_CONFIG_RESTORE,      "Start Config Restore" },
    { IPMI_RARITANOEM_RESTORE_CONFIG,            "Restore Config" },
    { IPMI_RARITANOEM_FINISH_CONFIG_RESTORE,     "Finish Config Restore" },
    { IPMI_RARITANOEM_RESET_TO_FACTORY_DEFAULTS, "Reset To Factory Defaults" },
    { IPMI_RARITANOEM_GET_SERIAL_NUMBER,         "Get Serial Number" },
    { IPMI_RARITANOEM_SET_SERIAL_NUMBER,         "Set Serial Number" },
    { IPMI_RARITANOEM_GET_VMEDIA_STATUS,         "Get Virtual Media Status" },
    { IPMI_RARITANOEM_CLOSE_VMEDIA_SESSION,      "Close Virtual Media Session" },
    { IPMI_RARITANOEM_START_FLOPPY_IMG_UPLOAD,   "Start Floppy Image Upload" },
    { IPMI_RARITANOEM_UPLOAD_FLOPPY_IMG,         "Upload Floppy Image" },
    { IPMI_RARITANOEM_FINALIZE_FLOPPY_IMG_UPLOAD,"Finalize FLoappy Image Upload" },
    { IPMI_RARITANOEM_START_SMB_IMG_MOUNT,       "Start SMB Image Mount" },
    { IPMI_RARITANOEM_SET_SMB_IMG_PARAM,         "Set SMB Image Parameter" },
    { IPMI_RARITANOEM_FINALIZE_SMB_IMG_MOUNT,    "Finalize SMB Image Mount" },
    { 0xffff, NULL }
};

static const struct errstr {
    uint8_t cmd;
    uint8_t err;
    const char *str;
    int error_code;
} raritanoem_err_strs[] = {
    { IPMI_RARITANOEM_FLASH_FIRMWARE,   0x80, "incomplete upload", IPMI_ERROR_INCOMPLETE_UPLOAD },
    { IPMI_RARITANOEM_FLASH_FIRMWARE,   0x81, "bad CRC", IPMI_ERROR_BAD_CRC },
    { IPMI_RARITANOEM_FLASH_FIRMWARE,   0x82, "firmware invalid", IPMI_ERROR_INVALID_FIRMWARE },
    { IPMI_RARITANOEM_FLASH_FIRMWARE,   0x83, "upgrade failed", IPMI_ERROR_UPGRADE_FAILED },
    { IPMI_RARITANOEM_GET_SERIAL_NUMBER,    0x80, "no serial number set", IPMI_ERROR_NO_SERIAL_NUMBER_SET },
    { IPMI_RARITANOEM_SET_SERIAL_NUMBER,    0x80, "serial number already set", IPMI_ERROR_SERIAL_ALREADY_SET },
    { IPMI_RARITANOEM_START_FLOPPY_IMG_UPLOAD,  0x80, "no empty device", IPMI_ERROR_NO_EMPTY_DEVICE },
    { IPMI_RARITANOEM_START_FLOPPY_IMG_UPLOAD,  0x81, "invalid file size", IPMI_ERROR_INVALID_FILE_SIZE },
    { IPMI_RARITANOEM_START_FLOPPY_IMG_UPLOAD,  0x82, "SMB mount in progress", IPMI_ERROR_SMB_MOUNT_IN_PROGRESS },
    { IPMI_RARITANOEM_FINALIZE_FLOPPY_IMG_UPLOAD,  0x82, "SMB mount in progress", IPMI_ERROR_SMB_MOUNT_IN_PROGRESS },
    { IPMI_RARITANOEM_FINALIZE_FLOPPY_IMG_UPLOAD,  0x83, "incomplete upload", IPMI_ERROR_INCOMPLETE_UPLOAD },
    { IPMI_RARITANOEM_START_SMB_IMG_MOUNT,  0x80, "no empty device", IPMI_ERROR_NO_EMPTY_DEVICE },
    { IPMI_RARITANOEM_FINALIZE_SMB_IMG_MOUNT,   0x82, "SMB mount in progress", IPMI_ERROR_SMB_MOUNT_IN_PROGRESS },
    { IPMI_RARITANOEM_FINALIZE_SMB_IMG_MOUNT,   0x84, "image access failed", IPMI_ERROR_IMAGE_ACCESS_FAILED },
    { 0xff, 0xff, NULL, 0 }
};

static const char *cmd_str(unsigned char netfn, unsigned char cmd)
{
    return netfn == IPMI_NETFN_RARITANOEM ?
        val2str(cmd, raritanoem_cmd_strs) : "";
}

static const char *err_str(unsigned char err, unsigned char netfn, unsigned char cmd)
{
    const char *str = NULL;

    if (netfn == IPMI_NETFN_RARITANOEM) {
        const struct errstr *errs = raritanoem_err_strs;
        for (errs = raritanoem_err_strs; errs->str != NULL; errs++) {
            if (errs->cmd == cmd && errs->err == err) {
                str = errs->str;
                break;
            }
        }
    }

    return str ? str : val2str(err, completion_code_vals);
}

static int err_code(unsigned char err, unsigned char netfn, unsigned char cmd)
{
    if (netfn == IPMI_NETFN_RARITANOEM) {
        const struct errstr *errs = raritanoem_err_strs;
        for (errs = raritanoem_err_strs; errs->str != NULL; errs++) {
            if (errs->cmd == cmd && errs->err == err) {
                return errs->error_code;
            }
        }
    }

    return err;
}


#define SENDRECV(intf, req, rsp, err) \
    rsp = intf->sendrecv(intf, &req, err); \
    if (rsp == NULL) { \
        ipmi_printf("%s command failed\n", \
                cmd_str(req.msg.netfn, req.msg.cmd)); \
        goto exit; \
    } \
    if (rsp->ccode > 0) { \
        ipmi_printf("%s command failed: %s\n", \
                cmd_str(req.msg.netfn, req.msg.cmd), \
                err_str(rsp->ccode, req.msg.netfn, req.msg.cmd)); \
	ipmi_reset_error(error); \
	ipmi_set_error(err, err_code(rsp->ccode, req.msg.netfn, req.msg.cmd)); \
        goto exit; \
    }

static int
ipmi_raritanoem_fw_ver(struct ipmi_intf * intf, pp_ipmi_return_t* ipmi_ret, int *error)
{
    struct ipmi_rq req;
    struct ipmi_rs * rsp;

    int *ver;
    int hwid;
    char *tag, *oem;
    int oem_len;

    if (ipmi_init_oem_raritan_fw_version(ipmi_ret)) {
	ipmi_set_error(error, IPMI_ERROR_INTERNAL);
	ipmi_printf("Error: could not initialize return structure.\n");
	return -1;
    }

    memset(&req, 0, sizeof(struct ipmi_rq));
    req.msg.netfn = IPMI_NETFN_RARITANOEM;
    req.msg.cmd = IPMI_RARITANOEM_GET_FIRMWARE_VERSION;

    SENDRECV(intf, req, rsp, error);

    ver = (int*)rsp->data;
    hwid = (int)*(rsp->data + 16);
    tag = (char *)rsp->data + 17;
    oem = tag + strlen(tag) + 1;
    oem_len = rsp->data_len - (oem - (char*)rsp->data);

    pp_strappendf(&ipmi_ret->data.oem_raritan_fw_version.version,
	"%d.%d.%d", le32_to_cpu(ver[0]), le32_to_cpu(ver[1]), le32_to_cpu(ver[2]));
    ipmi_ret->data.oem_raritan_fw_version.build_no = le32_to_cpu(ver[3]);
    ipmi_ret->data.oem_raritan_fw_version.hw_id = hwid;
    pp_strappendf(&ipmi_ret->data.oem_raritan_fw_version.tag,
	"%s", tag);
    pp_strappendf(&ipmi_ret->data.oem_raritan_fw_version.oem,
	"%.*s", oem_len, oem);

exit:
    return 0;
}

static int
ipmi_raritanoem_fw_upgrade(struct ipmi_intf * intf, const char* fw_file,
			   int flags, unsigned int max_chunk_size,
			   pp_ipmi_progress_cb_func progress_func,
			   void *progress_func_param, int *error)
{
    struct ipmi_rq req;
    struct ipmi_rs * rsp;
    unsigned char buf[256];
    unsigned char *buf2 = NULL;
    int ret = -1;
    struct stat fs;
    unsigned int size;
    unsigned short reserv;
    unsigned int chunk_size;
    unsigned int offs, p, pold = 0;

    FILE *fp = ipmi_open_file_read_bin(fw_file);
    if (!fp) {
        ipmi_printf("Could not open file '%s'\n", fw_file);
	ipmi_set_error(error, IPMI_ERROR_COULD_NOT_OPEN_FILE);
        goto exit;
    }
    if (fstat(fileno(fp), &fs) < 0) {
        ipmi_printf("Could not determine file size of '%s'\n", fw_file);
	ipmi_set_error(error, IPMI_ERROR_FILE_ACCESS_ERROR);
        goto exit;
    }
    size = (int)fs.st_size;

    *(unsigned int*)buf = cpu_to_le32(size);

    if (verbose) ipmi_printf("Starting Firmware Upgrade\n");
    memset(&req, 0, sizeof(struct ipmi_rq));
    req.msg.netfn = IPMI_NETFN_RARITANOEM;
    req.msg.cmd = IPMI_RARITANOEM_START_FIRMWARE_UPGRADE;
    req.msg.data = buf;
    req.msg.data_len = sizeof(size);

    SENDRECV(intf, req, rsp, error);

    reserv = le16_to_cpu(*(unsigned short*)rsp->data);
    chunk_size = le32_to_cpu(*(unsigned int*)(rsp->data + 2));

#define hdr_len (sizeof(reserv) + sizeof(offs))
    buf2 = malloc(hdr_len + max_chunk_size);
    if (buf2 == NULL) {
	ipmi_set_error(error, IPMI_ERROR_INTERNAL);
	goto exit;
    }

    if (chunk_size > max_chunk_size) chunk_size = max_chunk_size;
    for (offs = 0; offs < size; ) {
        // fill args
	int chunk_len;

	*(unsigned short*)buf2 = cpu_to_le16(reserv);
        *(unsigned int*)(buf2 + sizeof(reserv)) = cpu_to_le32(offs);
        chunk_len = fread(buf2 + hdr_len, 1, chunk_size, fp);
	if (chunk_len == 0) {
	    ipmi_set_error(error, IPMI_ERROR_FILE_ACCESS_ERROR);
	    ipmi_printf("Error reading from firmware file.\n");
	    goto exit;
	}

	p = offs * 400 / size + 1;
	if (progress_func && p != pold) {
	    progress_func(progress_func_param, p / 4);
	    pold = p;
	}

        memset(&req, 0, sizeof(struct ipmi_rq));
        req.msg.netfn = IPMI_NETFN_RARITANOEM;
        req.msg.cmd = IPMI_RARITANOEM_UPLOAD_FIRMWARE;
        req.msg.data = buf2;
        req.msg.data_len = hdr_len + chunk_len;

        SENDRECV(intf, req, rsp, error);
	offs += chunk_len;
    }

    // fill args
    *(unsigned short*)buf = cpu_to_le16(reserv);
    buf[2] = flags & 0x07;

    if (verbose) ipmi_printf("Flashing Firmware (takes about 1min)\n");
    if (progress_func) {
	progress_func(progress_func_param,
	    PP_IPMI_OEM_RARITAN_FW_UPGRAGE_PROGRESS_START_FLASHING);
    }

    memset(&req, 0, sizeof(struct ipmi_rq));
    req.msg.netfn = IPMI_NETFN_RARITANOEM;
    req.msg.cmd = IPMI_RARITANOEM_FLASH_FIRMWARE;
    req.msg.data = buf;
    req.msg.data_len = 3;

    SENDRECV(intf, req, rsp, error);

    if (!(flags & 0x01)) { // not in validate_only mode; wait for flash
	// fill args
	*(unsigned short*)buf = cpu_to_le16(reserv);

	memset(&req, 0, sizeof(struct ipmi_rq));
	req.msg.netfn = IPMI_NETFN_RARITANOEM;
	req.msg.cmd = IPMI_RARITANOEM_FINALIZE_FIRMWARE_UPGRADE;
	req.msg.data = buf;
	req.msg.data_len = 2;

    retry:
	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL || rsp->ccode == 0xc0) {
	    // retry if Finalize timed out
	    sleep(2);
	    goto retry;
	}
	if (rsp->ccode > 0) {
	    ipmi_printf("%s command failed: %s\n",
		    cmd_str(req.msg.netfn, req.msg.cmd),
		    err_str(rsp->ccode, req.msg.netfn, req.msg.cmd));
	    ipmi_reset_error(error);
	    ipmi_set_error(error, err_code(rsp->ccode, req.msg.netfn, req.msg.cmd));
	    goto exit;
	}

	if (verbose) ipmi_printf("Firmware successfully upgraded, device is rebooting now\n");
    }
    ret = 0;
exit:
    if (fp) fclose(fp);
    if (buf2) free(buf2);
    return ret;
}

static int
ipmi_raritanoem_backup_config(struct ipmi_intf *intf, const char *filename, unsigned int max_chunk_size, int *error)
{
    struct ipmi_rq req;
    struct ipmi_rs *rsp;
    int ret = -1;
    unsigned char buf[10];
    unsigned short reserv;
    unsigned int size, size_done = 0;
    unsigned int crc_expected, crc_actual;
    unsigned int chunk_size;

    FILE *fp = ipmi_open_file_write_bin(filename);
    if (!fp) {
	ipmi_printf("Could not open file '%s'\n", filename);
	ipmi_set_error(error, IPMI_ERROR_COULD_NOT_OPEN_FILE);
	goto exit;
    }

    if (verbose) ipmi_printf("Starting Config File Download\n");
    memset(&req, 0, sizeof(struct ipmi_rq));
    req.msg.netfn = IPMI_NETFN_RARITANOEM;
    req.msg.cmd = IPMI_RARITANOEM_START_CONFIG_BACKUP;
    SENDRECV(intf, req, rsp, error);
    reserv = *(unsigned short *)rsp->data;
    size = le32_to_cpu(*(unsigned int *)(rsp->data + 2));
    crc_expected = le32_to_cpu(*(unsigned int *)(rsp->data + 6));
    chunk_size = le32_to_cpu(*(unsigned int *)(rsp->data + 10));
    if (chunk_size > max_chunk_size) chunk_size = max_chunk_size;

    memset(&req, 0, sizeof(struct ipmi_rq));
    req.msg.netfn = IPMI_NETFN_RARITANOEM;
    req.msg.cmd = IPMI_RARITANOEM_BACKUP_CONFIG;
    *(unsigned short *)buf = reserv;
    req.msg.data = buf;
    req.msg.data_len = 10;
    while (size_done < size) {
	unsigned int len = size - size_done;
	if (len > chunk_size) len = chunk_size;
	*(unsigned int *)(buf + 2) = cpu_to_le32(size_done);
	*(unsigned int *)(buf + 6) = cpu_to_le32(len);
	SENDRECV(intf, req, rsp, error);
	fwrite(rsp->data, len, 1, fp);
	size_done += len;
    }

    memset(&req, 0, sizeof(struct ipmi_rq));
    req.msg.netfn = IPMI_NETFN_RARITANOEM;
    req.msg.cmd = IPMI_RARITANOEM_FINISH_CONFIG_BACKUP;
    *(unsigned short *)buf = reserv;
    req.msg.data = buf;
    req.msg.data_len = 2;
    SENDRECV(intf, req, rsp, error);

    fflush(fp);
    if (ipmi_crc32_file(filename, &crc_actual) < 0) {
	ipmi_printf("Could not open file '%s'\n", filename);
	ipmi_set_error(error, IPMI_ERROR_COULD_NOT_OPEN_FILE);
	goto exit;
    } else if (crc_actual != crc_expected) {
	ipmi_printf("Transferred file corrupt (CRC check failed)\n");
	ipmi_set_error(error, IPMI_ERROR_BAD_CRC);
	goto exit;
    }

    ret = 0;
exit:
    if (fp) fclose(fp);
    return ret;
}

static int
ipmi_raritanoem_restore_config(struct ipmi_intf *intf, const char *filename, unsigned int max_chunk_size, int *error)
{
    struct ipmi_rq req;
    struct ipmi_rs *rsp;
    int ret = -1;
    unsigned char buf[8];
    unsigned char *buf2 = NULL;
    unsigned short reserv;
    struct stat fs;
    unsigned int size, size_done = 0;
    unsigned int chunk_size;
    unsigned int crc;
    FILE *fp;

    if (ipmi_crc32_file(filename, &crc) < 0) {
	ipmi_printf("Could not open file '%s'\n", filename);
	ipmi_set_error(error, IPMI_ERROR_COULD_NOT_OPEN_FILE);
	return -1;
    }

    fp = ipmi_open_file_read_bin(filename);
    if (!fp) {
	ipmi_printf("Could not open file '%s'\n", filename);
	ipmi_set_error(error, IPMI_ERROR_COULD_NOT_OPEN_FILE);
	goto exit;
    }
    if (fstat(fileno(fp), &fs) < 0) {
	ipmi_printf("Could not determine file size of '%s'\n", filename);
	ipmi_set_error(error, IPMI_ERROR_FILE_ACCESS_ERROR);
	goto exit;
    }
    size = (int)fs.st_size;

    if (verbose) ipmi_printf("Starting Config File Download\n");
    memset(&req, 0, sizeof(struct ipmi_rq));
    req.msg.netfn = IPMI_NETFN_RARITANOEM;
    req.msg.cmd = IPMI_RARITANOEM_START_CONFIG_RESTORE;
    *(unsigned int *)buf = cpu_to_le32(size);
    *(unsigned int *)(buf + 4) = cpu_to_le32(crc);
    req.msg.data = buf;
    req.msg.data_len = 8;
    SENDRECV(intf, req, rsp, error);
    reserv = *(unsigned short *)rsp->data;
    chunk_size = le32_to_cpu(*(unsigned int *)(rsp->data + 2));
    if (chunk_size > max_chunk_size) chunk_size = max_chunk_size;
    buf2 = malloc(6 + max_chunk_size);

    memset(&req, 0, sizeof(struct ipmi_rq));
    req.msg.netfn = IPMI_NETFN_RARITANOEM;
    req.msg.cmd = IPMI_RARITANOEM_RESTORE_CONFIG;
    req.msg.data = buf2;
    while (size_done < size) {
	unsigned int len = size - size_done;
	if (len > chunk_size) len = chunk_size;
	*(unsigned short *)buf2 = reserv;
	*(unsigned int *)(buf2 + 2) = cpu_to_le32(size_done);
	fread(buf2 + 6, len, 1, fp);
	req.msg.data_len = 6 + len;
	SENDRECV(intf, req, rsp, error);
	size_done += len;
    }

    memset(&req, 0, sizeof(struct ipmi_rq));
    req.msg.netfn = IPMI_NETFN_RARITANOEM;
    req.msg.cmd = IPMI_RARITANOEM_FINISH_CONFIG_RESTORE;
    *(unsigned short *)buf = reserv;
    buf[2] = 1; // activate config
    req.msg.data = buf;
    req.msg.data_len = 3;
    SENDRECV(intf, req, rsp, error);

exit:
    if (fp) fclose(fp);
    free(buf2);
    return ret;
}

static int
ipmi_raritanoem_sn_get(struct ipmi_intf * intf, pp_ipmi_return_t *ipmi_ret, int *error)
{
    struct ipmi_rq req;
    struct ipmi_rs * rsp;
    int ret = -1;

    if (ipmi_init_oem_raritan_sn(ipmi_ret)) {
	ipmi_set_error(error, IPMI_ERROR_INTERNAL);
	ipmi_printf("Error: could not initialize return structure.\n");
	return -1;
    }

    memset(&req, 0, sizeof(struct ipmi_rq));
    req.msg.netfn = IPMI_NETFN_RARITANOEM;
    req.msg.cmd = IPMI_RARITANOEM_GET_SERIAL_NUMBER;

    SENDRECV(intf, req, rsp, error);

    pp_strappendf(&ipmi_ret->data.oem_raritan_sn, "%.*s\n", rsp->data_len, rsp->data);
    ret = 0;

exit:
    return ret;
}

static int
ipmi_raritanoem_sn_set(struct ipmi_intf * intf, char * val, int * error)
{
    struct ipmi_rq req;
    struct ipmi_rs * rsp;
    int ret = -1;

    memset(&req, 0, sizeof(struct ipmi_rq));
    req.msg.netfn = IPMI_NETFN_RARITANOEM;
    req.msg.cmd = IPMI_RARITANOEM_SET_SERIAL_NUMBER;
    req.msg.data = (void*)val;
    req.msg.data_len = strlen(val);

    SENDRECV(intf, req, rsp, error);

    if (verbose) ipmi_printf("serial number set to: %s\n", val);
    ret = 0;

exit:
    return ret;
}

static int
ipmi_raritanoem_defaults(struct ipmi_intf * intf, int * error) {
    struct ipmi_rq req;
    struct ipmi_rs * rsp;
    int ret = -1;

    memset(&req, 0, sizeof(struct ipmi_rq));
    req.msg.netfn = IPMI_NETFN_RARITANOEM;
    req.msg.cmd = IPMI_RARITANOEM_RESET_TO_FACTORY_DEFAULTS;

    SENDRECV(intf, req, rsp, error);

    if (verbose) ipmi_printf("Device set to defaults.\n");
    ret = 0;

exit:
    return ret;
}

struct get_extended_user_parameters_rq_s {
    BITFIELD2(unsigned char, uid : 6, _rsv : 2);
    unsigned char param;
    unsigned char set_sel;
    unsigned char block_sel;
} PACKED_STRUCT;
typedef struct get_extended_user_parameters_rq_s get_extended_user_parameters_rq_t;

static int
ipmi_raritanoem_ext_user_param_get(struct ipmi_intf * intf,
				   unsigned char user_id,
				   pp_ipmi_oem_raritan_get_ext_user_params_t *user_params,
				   int *error) {
    struct ipmi_rq req;
    struct ipmi_rs * rsp;
    int ret = -1;
    get_extended_user_parameters_rq_t get_param;

    // preapre structures
    memset(&get_param, 0, sizeof(get_param));
    get_param.uid = user_id;

    memset(&req, 0, sizeof(struct ipmi_rq));
    req.msg.netfn = IPMI_NETFN_RARITANOEM;
    req.msg.cmd = IPMI_CMD_PP_GET_EXTENDED_USER_PARAMETERS;
    req.msg.data = (void *)&get_param;
    req.msg.data_len = sizeof(get_param);

    // query device UID
    get_param.param = IPMI_OEM_PP_EXT_USER_DEVICE_UID;
    get_param.set_sel = get_param.block_sel = 0;

    SENDRECV(intf, req, rsp, error);

    user_params->dev_uid = le32_to_cpu(*((uint32_t *)rsp->data));

    // done
    ret = 0;

exit:
    return ret;
}

static int
ipmi_raritanoem_ext_user_param_list(struct ipmi_intf * intf,
				    pp_ipmi_oem_raritan_list_ext_user_params_t user_params,
				    int *error) {
    int i, ret;

    for (i = 1; i < 64; i++) {
	ret = ipmi_raritanoem_ext_user_param_get(intf, i, &user_params[i], error);
	if (ret != 0) {
	    return ret;
	}
    }

    return 0;
}

int
ipmi_raritanoem_main(struct ipmi_intf * intf, int subcommand, pp_ipmi_parameter_t *params, pp_ipmi_return_t *ipmi_ret, int * error)
{
    int flags = 0, max_chunk_size;
    switch ((pp_ipmi_oem_raritan_subcommand_t) subcommand) {
	case PP_IPMI_OEM_RARITAN_FW_VERSION:
	    return ipmi_raritanoem_fw_ver(intf, ipmi_ret, error);
	case PP_IPMI_OEM_RARITAN_FW_UPGRADE:
	    flags = (params->data.oem_raritan_fw_upgrade.validate_only ? 0x01 : 0)
	          | (params->data.oem_raritan_fw_upgrade.cross_oem     ? 0x02 : 0)
		  | (params->data.oem_raritan_fw_upgrade.cross_hwid    ? 0x04 : 0);
	    max_chunk_size = params->data.oem_raritan_fw_upgrade.max_chunk_size;
	    if (max_chunk_size == 0) max_chunk_size = 70;
	    return ipmi_raritanoem_fw_upgrade(intf,
		params->data.oem_raritan_fw_upgrade.filename,
		flags, max_chunk_size,
		params->data.oem_raritan_fw_upgrade.progress_cb,
		params->data.oem_raritan_fw_upgrade.progress_cb_param,
		error);
	case PP_IPMI_OEM_RARITAN_CFG_BACKUP:
	    max_chunk_size = params->data.oem_raritan_cfg_backup.max_chunk_size;
	    if (max_chunk_size == 0) max_chunk_size = 70;
	    return ipmi_raritanoem_backup_config(intf,
		params->data.oem_raritan_cfg_backup.filename,
		max_chunk_size, error);
	case PP_IPMI_OEM_RARITAN_CFG_RESTORE:
	    max_chunk_size = params->data.oem_raritan_cfg_restore.max_chunk_size;
	    if (max_chunk_size == 0) max_chunk_size = 70;
	    return ipmi_raritanoem_restore_config(intf,
		    params->data.oem_raritan_cfg_restore.filename,
		    max_chunk_size, error);
	case PP_IPMI_OEM_RARITAN_SN_GET:
	    return ipmi_raritanoem_sn_get(intf, ipmi_ret, error);
	case PP_IPMI_OEM_RARITAN_SN_SET:
	    return ipmi_raritanoem_sn_set(intf, params->data.oem_raritan_sn_set.serial, error);
	case PP_IPMI_OEM_RARITAN_DEFAULTS:
	    return ipmi_raritanoem_defaults(intf, error);
	case PP_IPMI_OEM_RARITAN_GET_EXT_USER_PARAMS:
	    return ipmi_raritanoem_ext_user_param_get(intf, params->data.oem_raritan_get_ext_user_params,
		&ipmi_ret->data.oem_raritan_get_ext_user_params, error);
	case PP_IPMI_OEM_RARITAN_LIST_EXT_USER_PARAMS:
	    return ipmi_raritanoem_ext_user_param_list(intf,
		ipmi_ret->data.oem_raritan_list_ext_user_params, error);
	default:
	    ipmi_printf("Invalid OEM Raritan command: %d\n", subcommand);
	    ipmi_set_error(error, IPMI_ERROR_INVALID_COMMAND);
	    return -1;
    }
}
