/*
 * Copyright (c) 2003 Sun Microsystems, Inc.  All Rights Reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * You acknowledge that this software is not designed or intended for use
 * in the design, construction, operation or maintenance of any nuclear
 * facility.
 */

#include <string.h>

#include <math.h>
#include <stdio.h>
#include <assert.h>

#ifndef _WIN32
#include <unistd.h>
#endif

#include <sys/types.h>
#include <time.h>

#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_sdr.h>
#include <ipmitool/ipmi_sensor.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/ipmi_sel.h>
#include <ipmitool/ipmi_entity.h>
#include <ipmitool/ipmi_constants.h>
#include <ipmitool/ipmi_strings.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>

#ifdef WIN32
#define strdup _strdup
#endif

#include <config.h>

#include <ipmi_return.h>
#ifdef OEM_IWILL
static int sdr_max_read_len = 0x16;
#else
static int sdr_max_read_len = GET_SDR_ENTIRE_RECORD;
#endif
static struct sdr_record_list * sdr_list_head = NULL;
static struct sdr_record_list * sdr_list_tail = NULL;
static struct ipmi_sdr_iterator * sdr_list_itr = NULL;

/* sensor type codes (IPMI v1.5 table 36.3) */
#define SENSOR_TYPE_MAX 0x29
static struct {
	pp_ipmi_specific_sensor_type_t type;
	const char * string;
} sensor_type_desc[] = {
	{ PP_IPMI_SPEC_SENSOR_TYPE_RESERVED, N_("reserved") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_TEMPERATURE, N_("Temperature") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_VOLTAGE, N_("Voltage") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_CURRENT, N_("Current") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_FAN, N_("Fan") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_PHYS_SECURITY, N_("Physical Security") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_VIOL_ATTEMPT, N_("Platform Security Violation Attempt") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_PROCESSOR, N_("Processor") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_POWER_SUPPLY, N_("Power Supply") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_POWER_UNIT, N_("Power Unit") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_COOLING_DEVICE, N_("Cooling Device") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_OTHER, N_("Other") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_MEMORY, N_("Memory") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_DRIVE_SLOT, N_("Drive Slot / Bay") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_POST, N_("POST Memory Resize") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_FIRMWARE, N_("System Firmware Progress") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_EVENT_LOG_DISABLED, N_("Event Logging Disabled") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_WATCHDOG, N_("Watchdog") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_SYSEVENT, N_("System Event") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_CRIT_INT, N_("Critical Interrupt") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_BUTTON, N_("Button") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_MODULE, N_("Module / Board") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_MICROCONTROLLER, N_("Microcontroller / Coprocessor") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_ADD_IN, N_("Add-in Card") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_CHASSIS, N_("Chassis") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_CHIPSET, N_("Chip Set") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_OTHER_FRU, N_("Other FRU") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_CABLE, N_("Cable / Interconnect") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_TERMINATOR, N_("Terminator") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_BOOT_INITIATED, N_("System Boot Initiated") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_BOOT_ERROR, N_("Boot Error") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_OS_BOOT, N_("OS Boot") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_OS_STOP, N_("OS Critical Stop") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_SLOT, N_("Slot / Connector") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_ACPI_STATE, N_("System ACPI Power State") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_WATCHDOG, N_("Watchdog") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_PLATFORM_ALERT, N_("Platform Alert") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_ENTITY, N_("Entity Presence") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_MONITOR_IC, N_("Monitor ASIC / IC") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_LAN, N_("LAN") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_MANAGEMENT_HEALTH, N_("Management Subsystem Health") },
	{ PP_IPMI_SPEC_SENSOR_TYPE_BATTERY, N_("Battery") },
};

/* unit description codes (IPMI v1.5 section 37.16) */
#define UNIT_MAX	0x90
static const char * unit_desc[] = {
	N_("unspecified"),
	N_("degrees C"), N_("degrees F"), N_("degrees K"),
	N_("Volts"), N_("Amps"), N_("Watts"), N_("Joules"),
	N_("Coulombs"), N_("VA"), N_("Nits"),
	N_("lumen"), N_("lux"), N_("Candela"),
	N_("kPa"), N_("PSI"), N_("Newton"),
	N_("CFM"), N_("RPM"), N_("Hz"),
	N_("microsecond"), N_("millisecond"), N_("second"),
	N_("minute"), N_("hour"), N_("day"), N_("week"),
	N_("mil"), N_("inches"), N_("feet"), N_("cu in"),
	N_("cu feet"), N_("mm"), N_("cm"), N_("m"), N_("cu cm"), N_("cu m"),
	N_("liters"), N_("fluid ounce"),
	N_("radians"), N_("steradians"), N_("revolutions"), N_("cycles"), N_("gravities"),
	N_("ounce"), N_("pound"), N_("ft-lb"), N_("oz-in"),
	N_("gauss"), N_("gilberts"), N_("henry"), N_("millihenry"),
	N_("farad"), N_("microfarad"), N_("ohms"), N_("siemens"), N_("mole"), N_("becquerel"),
	N_("PPM"), N_("reserved"),
	N_("Decibels"), N_("DbA"), N_("DbC"),
	N_("gray"), N_("sievert"), N_("color temp deg K"),
	N_("bit"), N_("kilobit"), N_("megabit"), N_("gigabit"),
	N_("byte"), N_("kilobyte"), N_("megabyte"), N_("gigabyte"),
	N_("word"), N_("dword"), N_("qword"), N_("line"),
	N_("hit"), N_("miss"), N_("retry"), N_("reset"),
	N_("overflow"), N_("underrun"),
	N_("collision"), N_("packets"),
	N_("messages"), N_("characters"),
	N_("error"), N_("correctable error"), N_("uncorrectable error"),
};

static const int cim_rate_units[] = {
	0,
	0, 0, 0,
	0, 0, 0, 0,
	0, 0, 0,
	0, 0, 0,
	0, 0, 0,
	0, 0, 0,
	1, 2, 3, 4, 5, 6, 7,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0,
	0, 0, 0, 0, 0,
	0, 0, 0, 0,
	0, 0, 0, 0,
	0, 0, 0, 0, 0, 0,
	0, 0,
	0, 0, 0,
	0, 0, 0,
	0, 0, 0, 0,
	0, 0, 0, 0,
	0, 0, 0, 0,
	0, 0, 0, 0,
	0, 0,
	0, 0,
	0, 0,
	0, 0, 0,
};

/* utos  -  convert unsigned 32bit value to 2's complement signed
 *
 * @val:	unsigned value to convert
 * @bits:	number of bits in value
 *
 * returns 2s complement signed integer
 */
int32_t
utos(uint32_t val, int bits)
{	
	int x;

	assert(bits > 0);

	x = 1 << (bits-1);
	if (val & x) return -(int)((~val & (x-1))+1);
	return val;
}

/* ipmi_sdr_get_unit_string  -  return units for base/modifier
 *
 * @type:	unit type
 * @base:	base
 * @modifier:	modifier
 *
 * returns pointer to static string
 */
char *
ipmi_sdr_get_unit_string(uint8_t type, uint8_t base, uint8_t modifier)
{
	static char unitstr[16];

	memset(unitstr, 0, sizeof(unitstr));
	switch (type) {
	case 2:
		snprintf(unitstr, sizeof(unitstr), "%s * %s",
			 _(unit_desc[base]),
			 _(unit_desc[modifier]));
		break;
	case 1:
		snprintf(unitstr, sizeof(unitstr), "%s/%s",
			 _(unit_desc[base]),
			 _(unit_desc[modifier]));
		break;
	case 0:
	default:
		snprintf(unitstr, sizeof(unitstr), "%s",
			 _(unit_desc[base]));
		break;
	}

	return unitstr;
}

double
pp_ipmi_sdr_convert_sensor_reading_core(int m, int b, int k1, int k2, 
                                        uint8_t analog,
                                        uint8_t linearization, uint16_t val) {
	double result;

	switch (analog)
	{
	case 0:
		result = (double)(((m * val) +
				(b * pow(10, k1))) * pow(10, k2));
		break;
	case 1:
		if (val & 0x80) val ++;
		/* Deliberately fall through to case 2. */
	case 2:
		result = (double)(((m * (int8_t)val) +
				(b * pow(10, k1))) * pow(10, k2));
		break;
	default:
		/* Oops! This isn't an analog sensor. */
		return 0.0;
	}

	switch (linearization & 0x7f) {
	default:
	case SDR_SENSOR_L_LINEAR:
		break;
	case SDR_SENSOR_L_LN:
		result = log(result);
		break;
	case SDR_SENSOR_L_LOG10:
		result = log10(result);
		break;
	case SDR_SENSOR_L_LOG2:
		result = (double)(log(result) / log(2.0));
		break;
	case SDR_SENSOR_L_E:
		result = exp(result);
		break;
	case SDR_SENSOR_L_EXP10:
		result = pow(10.0, result);
		break;
	case SDR_SENSOR_L_EXP2:
		result = pow(2.0, result);
		break;
	case SDR_SENSOR_L_1_X:
		result = pow(result, -1.0); /*1/x w/o exception*/
		break;
	case SDR_SENSOR_L_SQR:
		result = pow(result, 2.0);
		break;
	case SDR_SENSOR_L_CUBE:
		result = pow(result, 3.0);
		break;
	case SDR_SENSOR_L_SQRT:
		result = sqrt(result);
		break;
	case SDR_SENSOR_L_CUBERT:
#ifdef WIN32
		// Windows doesn't have cbrt
		result = 0.0;
#else
		result = cbrt(result);
#endif
		break;
        }

	return result;
}

/* sdr_convert_sensor_reading  -  convert raw sensor reading
 *
 * @sensor:	sensor record
 * @val:	raw sensor reading
 *
 * returns floating-point sensor reading
 */
double
sdr_convert_sensor_reading(struct sdr_record_full_sensor * sensor,
			   uint16_t val)
{
	int m, b, k1, k2;

	m  = __TO_M(sensor->mtol);
	b  = __TO_B(sensor->bacc);
	k1 = __TO_B_EXP(sensor->bacc);
	k2 = __TO_R_EXP(sensor->bacc);

	return pp_ipmi_sdr_convert_sensor_reading_core(m, b, k1, k2, sensor->unit.analog, sensor->linearization, val);
}

int
sdr_convert_sensor_reading_cim(struct sdr_record_full_sensor * sensor,
			       uint16_t val)
{
	int m, b, k1;
	double result;

	m  = __TO_M(sensor->mtol);
	b  = __TO_B(sensor->bacc);
	k1 = __TO_B_EXP(sensor->bacc);

	switch (sensor->unit.analog)
	{
	case 0:
		result = (double)((m * val) + (b * pow(10, k1)));
		break;
	case 1:
		if (val & 0x80) val ++;
		/* Deliberately fall through to case 2. */
	case 2:
		result = (double)((m * (signed char)val) + (b * pow(10, k1)));
		break;
	default:
		/* Oops! This isn't an analog sensor. */
		return 0;
	}

	switch (sensor->linearization & 0x7f) {
	default:
	case SDR_SENSOR_L_LINEAR:
		break;
	case SDR_SENSOR_L_LN:
		result = log(result);
		break;
	case SDR_SENSOR_L_LOG10:
		result = log10(result);
		break;
	case SDR_SENSOR_L_LOG2:
		result = (double)(log(result) / log(2.0));
		break;
	case SDR_SENSOR_L_E:
		result = exp(result);
		break;
	case SDR_SENSOR_L_EXP10:
		result = pow(10.0, result);
		break;
	case SDR_SENSOR_L_EXP2:
		result = pow(2.0, result);
		break;
	case SDR_SENSOR_L_1_X:
		result = pow(result, -1.0); /*1/x w/o exception*/
		break;
	case SDR_SENSOR_L_SQR:
		result = pow(result, 2.0);
		break;
	case SDR_SENSOR_L_CUBE:
		result = pow(result, 3.0);
		break;
	case SDR_SENSOR_L_SQRT:
		result = sqrt(result);
		break;
	case SDR_SENSOR_L_CUBERT:
#ifdef WIN32
		// Windows doesn't have cbrt
		result = result>=0? pow(result, 1.0/3) : -pow(-result, 1.0/3);
#else
		result = cbrt(result);
#endif
		break;
        }

	return (int)result;
}

/* sdr_convert_sensor_value_to_raw  -  convert sensor reading back to raw
 *
 * @sensor:	sensor record
 * @val:	converted sensor reading
 *
 * returns raw sensor reading
 */
uint8_t
sdr_convert_sensor_value_to_raw(struct sdr_record_full_sensor * sensor,
				double val)
{
	int m, b, k1, k2;

	m  = __TO_M(sensor->mtol);
	b  = __TO_B(sensor->bacc);
	k1 = __TO_B_EXP(sensor->bacc);
	k2 = __TO_R_EXP(sensor->bacc);

	return pp_ipmi_sdr_convert_sensor_value_to_raw(m, b, k1, k2,
						       sensor->unit.analog,
						       val);
}

uint8_t
pp_ipmi_sdr_convert_sensor_value_to_raw(int m, int b, int k1, int k2,
					uint8_t analog, double val) {
	double result;

	/* only works for analog sensors */
	if (analog > 2)
		return 0;

	/* don't divide by zero */
        if (m == 0)
		return 0;

        result = (((val / pow(10, k2)) - (b * pow(10, k1))) / m);

	if ((result -(int)result) >= .5) {
		result = ceil(result);
	}
	return (uint8_t)result;
}

uint8_t
sdr_convert_sensor_value_to_raw_cim(struct sdr_record_full_sensor * sensor, int val)
{
	int m, b, k1;
	int result;

	m  = __TO_M(sensor->mtol);
	b  = __TO_B(sensor->bacc);
	k1 = __TO_B_EXP(sensor->bacc);

	/* only works for analog sensors */
	if (sensor->unit.analog > 2)
		return 0;

	/* don't divide by zero */
        if (m == 0)
		return 0;

        result = (int)((val - (b * pow(10, k1))) / m + 0.5);
	return (uint8_t)result;
}


/* ipmi_sdr_get_sensor_thresholds  -  return thresholds for sensor
 *
 * @intf:	ipmi interface
 * @sensor:	sensor number
 *
 * returns pointer to ipmi response
 */
struct ipmi_rs *
ipmi_sdr_get_sensor_thresholds(struct ipmi_intf * intf, uint8_t sensor, int * error)
{
	struct ipmi_rq req;

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_SE;
	req.msg.cmd = GET_SENSOR_THRESHOLDS;
	req.msg.data = &sensor;
	req.msg.data_len = sizeof(sensor);

	return intf->sendrecv(intf, &req, error);
}

/* ipmi_sdr_get_sensor_hysteresis  -  return hysteresis for sensor
 *
 * @intf:	ipmi interface
 * @sensor:	sensor number
 *
 * returns pointer to ipmi response
 */
struct ipmi_rs *
ipmi_sdr_get_sensor_hysteresis(struct ipmi_intf * intf, uint8_t sensor, int * error)
{
	struct ipmi_rq req;
	uint8_t rqdata[2];

	rqdata[0] = sensor;
	rqdata[1] = 0xff;	/* reserved */

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_SE;
	req.msg.cmd = GET_SENSOR_HYSTERESIS;
	req.msg.data = rqdata;
	req.msg.data_len = 2;

	return intf->sendrecv(intf, &req, error);
}
  

/* ipmi_sdr_get_sensor_reading  -  retrieve a raw sensor reading
 *
 * @intf:	ipmi interface
 * @sensor:	sensor id
 *
 * returns ipmi response structure
 */
static struct ipmi_rs *
ipmi_sdr_get_sensor_reading(struct ipmi_intf * intf, uint8_t sensor, int * error)
{
	struct ipmi_rq req;

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_SE;
	req.msg.cmd = GET_SENSOR_READING;
	req.msg.data = &sensor;
	req.msg.data_len = sizeof(sensor);

	return intf->sendrecv(intf, &req, error);
}

/* ipmi_sdr_get_sensor_reading_ipmb  -  retrieve a raw sensor reading from ipmb
 *
 * @intf:	ipmi interface
 * @sensor:	sensor id
 * @target:	IPMB target address
 *
 * returns ipmi response structure
 */
/* unused?! */
#if 0
static struct ipmi_rs *
ipmi_sdr_get_sensor_reading_ipmb(struct ipmi_intf * intf, uint8_t sensor, uint8_t target, int * error)
{
	struct ipmi_rq req;
	struct ipmi_rs * rsp;
	uint8_t save_addr;

	save_addr = intf->target_addr;
	intf->target_addr = target;

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_SE;
	req.msg.cmd = GET_SENSOR_READING;
	req.msg.data = &sensor;
	req.msg.data_len = 1;

	rsp = intf->sendrecv(intf, &req, error);
	intf->target_addr = save_addr;
	return rsp;
}
#endif /* 0 */
  
/* ipmi_sdr_get_sensor_event_status  -  retrieve sensor event status
 *
 * @intf:	ipmi interface
 * @sensor:	sensor id
 *
 * returns ipmi response structure
 */
static struct ipmi_rs *
ipmi_sdr_get_sensor_event_status(struct ipmi_intf * intf, uint8_t sensor, int * error)
{
	struct ipmi_rq req;

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_SE;
	req.msg.cmd = GET_SENSOR_EVENT_STATUS;
	req.msg.data = &sensor;
	req.msg.data_len = 1;

	return intf->sendrecv(intf, &req, error);
}

/* ipmi_sdr_get_sensor_event_enable  -  retrieve sensor event enables
 *
 * @intf:	ipmi interface
 * @sensor:	sensor id
 *
 * returns ipmi response structure
 */
static struct ipmi_rs *
ipmi_sdr_get_sensor_event_enable(struct ipmi_intf * intf, uint8_t sensor, int * error)
{
	struct ipmi_rq req;

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_SE;
	req.msg.cmd = GET_SENSOR_EVENT_ENABLE;
	req.msg.data = &sensor;
	req.msg.data_len = 1;

	return intf->sendrecv(intf, &req, error);
}
  
/* ipmi_sdr_get_sensor_type_desc  -  Get sensor type descriptor
 *
 * @type:	ipmi sensor type
 *
 * returns
 *   string from sensor_type_desc
 *   or "reserved"
 *   or "OEM reserved"
 */
const char *
ipmi_sdr_get_sensor_type_desc(const uint8_t type)
{
	static char desc[32];
	memset(desc, 0, 32);
	if (type <= SENSOR_TYPE_MAX)
		return _(sensor_type_desc[type].string);
	if (type < 0xc0)
		snprintf(desc, 32, _("reserved #%02x"), type);
	else
		snprintf(desc, 32, _("OEM reserved #%02x"), type);
	return desc;
}
  
static pp_ipmi_specific_sensor_type_t
ipmi_sdr_get_sensor_spec_type(const uint8_t type)
{
	if (type <= SENSOR_TYPE_MAX)
		return  sensor_type_desc[type].type;

	if (type < 0xc0)
		return PP_IPMI_SPEC_SENSOR_TYPE_RESERVED;

	return PP_IPMI_SPEC_SENSOR_TYPE_OEM_RESERVED;
}

/* ipmi_sdr_get_status  -  Return status indicator
 *
 * @stat:	ipmi SDR status field
 *
 * returns
 *   cr = critical
 *   nc = non-critical
 *   nr = non-recoverable
 *   ok = ok
 *   us = unspecified (not used)
 */
static pp_ipmi_sensor_status_t
ipmi_sdr_get_status(uint8_t stat)
{
	if (stat & SDR_SENSOR_STAT_LO_NR) {
		return PP_IPMI_SENSOR_STATUS_LOWER_NON_RECOVERABLE;
	} else if (stat & SDR_SENSOR_STAT_HI_NR) {
		return PP_IPMI_SENSOR_STATUS_UPPER_NON_RECOVERABLE;
	} else if (stat & SDR_SENSOR_STAT_LO_CR) {
		return PP_IPMI_SENSOR_STATUS_LOWER_CRITICAL;	
	} else if (stat & SDR_SENSOR_STAT_HI_CR) {
		return PP_IPMI_SENSOR_STATUS_UPPER_CRITICAL;	
	} else if (stat & SDR_SENSOR_STAT_LO_NC) {
		return PP_IPMI_SENSOR_STATUS_LOWER_NON_CRITICAL;
	} else if (stat & SDR_SENSOR_STAT_HI_NC) {
		return PP_IPMI_SENSOR_STATUS_UPPER_NON_CRITICAL;
	} else {
		return PP_IPMI_SENSOR_STATUS_OK;
	}
}

static pp_ipmi_sensor_status_t
ipmi_sdr_get_status_discrete(uint8_t stat1, uint8_t stat2)
{
	if (!stat1 && !stat2) {
		return PP_IPMI_SENSOR_STATUS_OK;
	} else {
		return PP_IPMI_SENSOR_STATUS_ASSERTED;	
	}
}

static const char *
ipmi_sdr_get_status_full(uint8_t stat)
{
	if (stat & SDR_SENSOR_STAT_LO_NR)
		return _("Below lower non-recoverable threshold");
	else if (stat & SDR_SENSOR_STAT_HI_NR)
		return _("Above upper non-recoverable threshold");
	else if (stat & SDR_SENSOR_STAT_LO_CR)
		return _("Below lower critical threshold");
	else if (stat & SDR_SENSOR_STAT_HI_CR)
		return _("Above upper critical threshold");
	else if (stat & SDR_SENSOR_STAT_LO_NC)
		return _("Below lower non-critical threshold");
	else if (stat & SDR_SENSOR_STAT_HI_NC)
		return _("Above upper non-critical threshold");
	else
		return _("Ok");
}

static const char *
ipmi_sdr_discrete_get_cim_current_state(uint8_t sensor_type,
	uint8_t event_type, uint8_t state1, uint8_t state2)
{
	uint8_t typ;
	struct ipmi_event_sensor_types *evt;
	const char *result = "OK";
	int max_offset = -1;

	if (event_type == 0x6f) {
		evt = sensor_specific_types;
		typ = sensor_type;
	} else {
		evt = generic_event_types;
		typ = event_type;
	}

	for (; evt->type != NULL; evt++) {
		if (evt->code != typ || evt->offset < max_offset)
			continue;

		if (evt->offset > 7) {
			if ((1<<(evt->offset-8)) & state2) {
				max_offset = evt->offset;
				result = evt->desc;
			}
		} else {
			if ((1<<evt->offset) & state1) {
				max_offset = evt->offset;
				result = evt->desc;
			}
		}
	}
	return result;
}

static void ipmi_sdr_discrete_get_cim_possible_states(vector_t *states,
	uint8_t sensor_type, uint8_t event_type, uint16_t mask)
{
	uint8_t typ;
	struct ipmi_event_sensor_types *evt;

	// default state (no bits set)
	vector_add(states, strdup("OK"));

#ifdef WORDS_BIGENDIAN
	mask = ((mask & 0xff) << 8) | (mask >> 8);
#endif

	if (event_type == 0x6f) {
		evt = sensor_specific_types;
		typ = sensor_type;
	} else {
		evt = generic_event_types;
		typ = event_type;
	}

	for (; evt->type != NULL; evt++) {
		if (evt->code != typ)
			continue;

		if ((1<<evt->offset) & mask)
                    vector_add(states, strdup(evt->desc ? evt->desc : ""));
	}
}

static const char *
ipmi_sdr_threshold_get_cim_current_state(uint8_t stat)
{
	if (stat & SDR_SENSOR_STAT_LO_NR)
		return "Lower Non-Recoverable";
	else if (stat & SDR_SENSOR_STAT_HI_NR)
		return "Upper Non-Recoverable";
	else if (stat & SDR_SENSOR_STAT_LO_CR)
		return "Lower Critical";
	else if (stat & SDR_SENSOR_STAT_HI_CR)
		return "Upper Critical";
	else if (stat & SDR_SENSOR_STAT_LO_NC)
		return "Lower Non-Critical";
	else if (stat & SDR_SENSOR_STAT_HI_NC)
		return "Upper Non-Critical";
	else
		return "OK";
}

static void ipmi_sdr_threshold_get_cim_possible_states(vector_t *states)
{
	vector_add(states, strdup("OK"));
	vector_add(states, strdup("Lower Non-Critical"));
	vector_add(states, strdup("Lower Critical"));
	vector_add(states, strdup("Lower Non-Recoverable"));
	vector_add(states, strdup("Upper Non-Critical"));
	vector_add(states, strdup("Upper Critical"));
	vector_add(states, strdup("Upper Non-Recoverable"));
}

/* ipmi_sdr_get_header  -  retreive SDR record header
 *
 * @intf:	ipmi interface
 * @itr:	sdr iterator
 *
 * returns pointer to static sensor retrieval struct
 * returns NULL on error
 */
static struct sdr_get_rs *
ipmi_sdr_get_header(struct ipmi_intf * intf,
		    struct ipmi_sdr_iterator * itr, int * error)
{
	struct ipmi_rq req;
	struct ipmi_rs * rsp;
	struct sdr_get_rq sdr_rq;
	static struct sdr_get_rs sdr_rs;
	int try = 0;

	memset(&sdr_rq, 0, sizeof(sdr_rq));
	sdr_rq.reserve_id = cpu_to_le16(itr->reservation);
	sdr_rq.id = cpu_to_le16(itr->next);
	sdr_rq.offset = 0;
	sdr_rq.length = 5;	/* only get the header */

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_STORAGE;
	req.msg.cmd = GET_SDR;
	req.msg.data = (uint8_t *)&sdr_rq;
	req.msg.data_len = sizeof(sdr_rq);

	for (try=0; try<5; try++) {
		sdr_rq.reserve_id = cpu_to_le16(itr->reservation);
		rsp = intf->sendrecv(intf, &req, error);
		if (rsp == NULL) {
			ipmi_printf("Get SDR %04x command failed\n",	itr->next);
			continue;
		}
		else if (rsp->ccode == 0xc5) {
			/* lost reservation */
			ipmi_printf("SDR reserveration %04x cancelled. "
				"Sleeping a bit and retrying...\n", itr->reservation);

			sleep(rand() & 3);

			if (ipmi_sdr_get_reservation(intf, &(itr->reservation), error) < 0) {
				ipmi_printf("Unable to renew SDR reservation\n");
				return NULL;
			}
		}
		else if (rsp->ccode > 0) {
			ipmi_printf("Get SDR %04x command failed: %s\n",
				itr->next, val2str(rsp->ccode, completion_code_vals));
			continue;
		}
		else {
			break;
		}
	}

	if (verbose > 1)
		ipmi_printf("SDR record ID   : 0x%04x\n", itr->next);

        if (rsp == NULL) 
            return NULL;
        
	memcpy(&sdr_rs, rsp->data, sizeof(sdr_rs));

	sdr_rs.next = le16_to_cpu(sdr_rs.next);
	sdr_rs.id = le16_to_cpu(sdr_rs.id);

	/* achu (chu11 at llnl dot gov): - Some boards are stupid and
	 * return a record id from the Get SDR Record command
	 * different than the record id passed in.  If we find this
	 * situation, we cheat and put the original record id back in.
	 * Otherwise, a later Get SDR Record command will fail with
	 * completion code CBh = "Requested Sensor, data, or record
	 * not present"
	 */
	if (sdr_rs.id != itr->next) {
		ipmi_printf("SDR record id mismatch: 0x%04x\n",
			sdr_rs.id);
		sdr_rs.id = itr->next;
	}
  
	if (sdr_rs.length == 0) {
		ipmi_printf("Error in SDR record id 0x%04x: invalid length %d\n",
		       itr->next, sdr_rs.length);
		return NULL;
	}

	return &sdr_rs;
}

/* ipmi_sdr_get_next_header  -  retreive next SDR header
 *
 * @intf:	ipmi interface
 * @itr:	sdr iterator
 *
 * returns pointer to sensor retrieval struct
 * returns NULL on error
 */
struct sdr_get_rs *
ipmi_sdr_get_next_header(struct ipmi_intf * intf,
			 struct ipmi_sdr_iterator * itr, int * error)
{
	struct sdr_get_rs *header;

	if (itr->next == 0xffff)
		return NULL;

	header = ipmi_sdr_get_header(intf, itr, error);
	if (header == NULL)
  		return NULL;

	itr->next = header->next;

	return header;
}

static int
ipmi_sdr_print_sensor_event_status(struct ipmi_intf * intf,
				   uint8_t sensor_num,
				   uint8_t sensor_type,
				   uint8_t event_type,
				   int numeric_fmt,
				   pp_ipmi_sdr_list_entry_t *entry,
				   int * error)
{
	struct ipmi_rs * rsp;
	int i;
	int compact = 0;
	const struct valstr assert_cond_1[] = {
		{ 0x80, "unc+" },
		{ 0x40, "unc-" },
		{ 0x20, "lnr+" },
		{ 0x10, "lnr-" },
		{ 0x08, "lcr+" },
		{ 0x04, "lcr-" },
		{ 0x02, "lnc+" },
		{ 0x01, "lnc-" },
		{ 0x00, NULL },
	};
	const struct valstr assert_cond_2[] = {
		{ 0x08, "unr+" },
		{ 0x04, "unr-" },
		{ 0x02, "ucr+" },
		{ 0x01, "ucr-" },
		{ 0x00, NULL },
	};

	rsp = ipmi_sdr_get_sensor_event_status(intf, sensor_num, error);

	if (rsp == NULL) {
		ipmi_printf("Error reading event status for sensor #%02x\n",
			sensor_num);
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Error reading event status for sensor #%02x: %s\n",
			sensor_num, val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	if (IS_READING_UNAVAILABLE(rsp->data[0])) {
		printf(" Event Status          : Unavailable\n");
		return 0;
	}
	if (IS_SCANNING_DISABLED(rsp->data[0])) {
		//printf(" Event Status          : Scanning Disabled\n");
		//return 0;
	}
	if (IS_EVENT_MSG_DISABLED(rsp->data[0])) {
		printf(" Event Status          : Event Messages Disabled\n");
		//return 0;
	}

	switch (numeric_fmt) {
	case DISCRETE_COMPACT_SENSOR:
		compact = 1;
		// fall through
	case DISCRETE_FULL_SENSOR:
		if (rsp->data_len == 2) {
			ipmi_sdr_print_discrete_state(sensor_type,
						      event_type, rsp->data[1], 0,
						      compact ?	&entry->data.compact.assertion_events :
						      		&entry->data.full.assertion_events);
		}
		else if (rsp->data_len > 2) {
			ipmi_sdr_print_discrete_state(sensor_type,
						      event_type, rsp->data[1], rsp->data[2],
						      compact ?	&entry->data.compact.assertion_events :
						      		&entry->data.full.assertion_events);
		}
		if (rsp->data_len == 4) {
			ipmi_sdr_print_discrete_state(sensor_type,
						      event_type, rsp->data[3], 0,
						      compact ?	&entry->data.compact.deassertion_events :
						      		&entry->data.full.deassertion_events);
		}
		else if (rsp->data_len > 4) {
			ipmi_sdr_print_discrete_state(sensor_type,
						      event_type, rsp->data[3], rsp->data[4],
						      compact ?	&entry->data.compact.deassertion_events :
						      		&entry->data.full.deassertion_events);
		}
		break;

	case ANALOG_SENSOR:
		for (i = 0; i < 8; i++) {
			if (rsp->data[1] & (1<<i))
				pp_strappendf(&entry->data.full.assertion_events,
					"%s ", val2str(1<<i, assert_cond_1));
		}
		if (rsp->data_len > 2) {
			for (i = 0; i < 4; i++) {
				if (rsp->data[2] & (1<<i))
					pp_strappendf(&entry->data.full.assertion_events,
						"%s ", val2str(1<<i, assert_cond_2));
			}
			if ((rsp->data_len == 4 && rsp->data[3] != 0) ||
			    (rsp->data_len > 4 && (rsp->data[3] != 0 && rsp->data[4] != 0))) {
				for (i = 0; i < 8; i++) {
					if (rsp->data[3] & (1<<i))
						pp_strappendf(&entry->data.full.deassertion_events,
							"%s ", val2str(1<<i, assert_cond_1));
				}
				if (rsp->data_len > 4) {
					for (i = 0; i < 4; i++) {
						if (rsp->data[4] & (1<<i))
							pp_strappendf(&entry->data.full.deassertion_events,
								"%s ", val2str(1<<i, assert_cond_2));
					}
				}
			}
		}
		break;

	default:
		break;
	}

	return 0;
}

static int
ipmi_sdr_print_sensor_mask(struct sdr_record_mask * mask,
			   uint8_t sensor_type,
			   uint8_t event_type,
			   int numeric_fmt,
			   pp_ipmi_sdr_list_entry_t *entry)
{
	int compact = 0;
	return 0;

	switch (numeric_fmt)
	{
	case DISCRETE_COMPACT_SENSOR:
		compact = 1;
		// fall through
	case DISCRETE_FULL_SENSOR:
		ipmi_sdr_print_discrete_state(sensor_type,
					      event_type,
					      mask->type.discrete.assert_event & 0xff,
					      (mask->type.discrete.assert_event & 0xff00) >> 8,
					      compact ?	&entry->data.compact.assertion_event_mask :
					      		&entry->data.full.assertion_event_mask);
		ipmi_sdr_print_discrete_state(sensor_type,
					      event_type,
					      mask->type.discrete.deassert_event & 0xff,
					      (mask->type.discrete.deassert_event & 0xff00) >> 8,
					      compact ?	&entry->data.compact.deassertion_event_mask :
					      		&entry->data.full.deassertion_event_mask);
		break;

	case ANALOG_SENSOR:
		if (mask->type.threshold.assert_lnr_high)
			pp_strappendf(&entry->data.full.assertion_event_mask, "lnr+ ");
		if (mask->type.threshold.assert_lnr_low)
			pp_strappendf(&entry->data.full.assertion_event_mask, "lnr- ");
		if (mask->type.threshold.assert_lcr_high)
			pp_strappendf(&entry->data.full.assertion_event_mask, "lcr+ ");
		if (mask->type.threshold.assert_lcr_low)
			pp_strappendf(&entry->data.full.assertion_event_mask, "lcr- ");
		if (mask->type.threshold.assert_lnc_high)
			pp_strappendf(&entry->data.full.assertion_event_mask, "lnc+ ");
		if (mask->type.threshold.assert_lnc_low)
			pp_strappendf(&entry->data.full.assertion_event_mask, "lnc- ");
		if (mask->type.threshold.assert_unc_high)
			pp_strappendf(&entry->data.full.assertion_event_mask, "unc+ ");
		if (mask->type.threshold.assert_unc_low)
			pp_strappendf(&entry->data.full.assertion_event_mask, "unc- ");
		if (mask->type.threshold.assert_ucr_high)
			pp_strappendf(&entry->data.full.assertion_event_mask, "ucr+ ");
		if (mask->type.threshold.assert_ucr_low)
			pp_strappendf(&entry->data.full.assertion_event_mask, "ucr- ");
		if (mask->type.threshold.assert_unr_high)
			pp_strappendf(&entry->data.full.assertion_event_mask, "unr+ ");
		if (mask->type.threshold.assert_unr_low)
			pp_strappendf(&entry->data.full.assertion_event_mask, "unr- ");

		if (mask->type.threshold.deassert_lnr_high)
			pp_strappendf(&entry->data.full.deassertion_event_mask, "lnr+ ");
		if (mask->type.threshold.deassert_lnr_low)
			pp_strappendf(&entry->data.full.deassertion_event_mask, "lnr- ");
		if (mask->type.threshold.deassert_lcr_high)
			pp_strappendf(&entry->data.full.deassertion_event_mask, "lcr+ ");
		if (mask->type.threshold.deassert_lcr_low)
			pp_strappendf(&entry->data.full.deassertion_event_mask, "lcr- ");
		if (mask->type.threshold.deassert_lnc_high)
			pp_strappendf(&entry->data.full.deassertion_event_mask, "lnc+ ");
		if (mask->type.threshold.deassert_lnc_low)
			pp_strappendf(&entry->data.full.deassertion_event_mask, "lnc- ");
		if (mask->type.threshold.deassert_unc_high)
			pp_strappendf(&entry->data.full.deassertion_event_mask, "unc+ ");
		if (mask->type.threshold.deassert_unc_low)
			pp_strappendf(&entry->data.full.deassertion_event_mask, "unc- ");
		if (mask->type.threshold.deassert_ucr_high)
			pp_strappendf(&entry->data.full.deassertion_event_mask, "ucr+ ");
		if (mask->type.threshold.deassert_ucr_low)
			pp_strappendf(&entry->data.full.deassertion_event_mask, "ucr- ");
		if (mask->type.threshold.deassert_unr_high)
			pp_strappendf(&entry->data.full.deassertion_event_mask, "unr+ ");
		if (mask->type.threshold.deassert_unr_low)
			pp_strappendf(&entry->data.full.deassertion_event_mask, "unr- ");
		break;

	default:
		break;
	}

	return 0;
}

static int
ipmi_sdr_print_sensor_event_enable(struct ipmi_intf * intf,
				   uint8_t sensor_num,
				   uint8_t sensor_type,
				   uint8_t event_type,
				   int numeric_fmt,
				   pp_ipmi_sdr_list_entry_t *entry,
				   int * error)
{
	int compact = 0;
	struct ipmi_rs * rsp;
	int i;
	const struct valstr assert_cond_1[] = {
		{ 0x80, "unc+" },
		{ 0x40, "unc-" },
		{ 0x20, "lnr+" },
		{ 0x10, "lnr-" },
		{ 0x08, "lcr+" },
		{ 0x04, "lcr-" },
		{ 0x02, "lnc+" },
		{ 0x01, "lnc-" },
		{ 0x00, NULL },
	};
	const struct valstr assert_cond_2[] = {
		{ 0x08, "unr+" },
		{ 0x04, "unr-" },
		{ 0x02, "ucr+" },
		{ 0x01, "ucr-" },
		{ 0x00, NULL },
	};

	rsp = ipmi_sdr_get_sensor_event_enable(intf, sensor_num, error);

	if (rsp == NULL) {
		ipmi_printf("Error reading event enable for sensor #%02x\n",
			sensor_num);
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Error reading event enable for sensor #%02x: %s\n",
			sensor_num, val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	if (IS_SCANNING_DISABLED(rsp->data[0])) {
		//printf(" Event Enable          : Scanning Disabled\n");
		//return 0;
	}
	if (IS_EVENT_MSG_DISABLED(rsp->data[0])) {
		printf(" Event Enable          : Event Messages Disabled\n");
		//return 0;
	}

	switch (numeric_fmt)
	{
	case DISCRETE_COMPACT_SENSOR:
		compact = 1;
		// fall through
	case DISCRETE_FULL_SENSOR:
		/* discrete */
		if (rsp->data_len == 2) {
			ipmi_sdr_print_discrete_state(sensor_type,
						      event_type, rsp->data[1], 0,
						      compact ?	&entry->data.compact.assertions_enabled :
						      		&entry->data.full.assertions_enabled);
		}
		else if (rsp->data_len > 2) {
			ipmi_sdr_print_discrete_state(sensor_type,
						      event_type, rsp->data[1], rsp->data[2],
						      compact ?	&entry->data.compact.assertions_enabled :
						      		&entry->data.full.assertions_enabled);
		}
		if (rsp->data_len == 4) {
			ipmi_sdr_print_discrete_state(sensor_type,
						      event_type, rsp->data[3], 0,
						      compact ?	&entry->data.compact.deassertions_enabled :
						      		&entry->data.full.deassertions_enabled);
		}
		else if (rsp->data_len > 4) {
			ipmi_sdr_print_discrete_state(sensor_type,
						      event_type, rsp->data[3], rsp->data[4],
						      compact ?	&entry->data.compact.deassertions_enabled :
						      		&entry->data.full.deassertions_enabled);
		}
		break;

	case ANALOG_SENSOR:
		/* analog */
		for (i = 0; i < 8; i++) {
			if (rsp->data[1] & (1<<i))
				pp_strappendf(&entry->data.full.assertions_enabled,
					"%s ", val2str(1<<i, assert_cond_1));
		}
		if (rsp->data_len > 2) {
			for (i = 0; i < 4; i++) {
				if (rsp->data[2] & (1<<i))
					pp_strappendf(&entry->data.full.assertions_enabled,
						"%s ", val2str(1<<i, assert_cond_2));
			}
			if ((rsp->data_len == 4 && rsp->data[3] != 0) ||
			    (rsp->data_len > 4 && (rsp->data[3] != 0 && rsp->data[4] != 0))) {
				for (i = 0; i < 8; i++) {
					if (rsp->data[3] & (1<<i))
						pp_strappendf(&entry->data.full.deassertions_enabled,
							"%s ", val2str(1<<i, assert_cond_1));
				}
				if (rsp->data_len > 4) {
					for (i = 0; i < 4; i++) {
						if (rsp->data[4] & (1<<i))
							pp_strappendf(&entry->data.full.deassertions_enabled,
								"%s ", val2str(1<<i, assert_cond_2));
					}
				}
			}
		}
		break;

	default:
		break;
	}

	return 0;
}

static void fill_entity(pp_ipmi_sdr_list_entry_t *entry, uint32_t id, uint32_t instance) {
	if (entry == NULL) return;
	
	entry->entity.is_valid = 1;
	entry->entity.id = id;
	entry->entity.instance = instance;
	pp_strappend(&entry->entity.string, val2str((uint16_t)id, entity_id_vals));
}

static void set_value(pp_ipmi_sensor_value_t *val, int condition, float value, int cim_value) {
	if (!val) return;
	val->valid = condition ? 1 : 0;
	val->value = value;
	val->cim_value = cim_value;
}

/* helper macro for priting analog values */
#define SENSOR_PRINT_NORMAL(TARGET, READ)							\
	set_value(&entry->data.full.TARGET, sensor->analog_flag.READ,				\
    	      (float)sdr_convert_sensor_reading(sensor, sensor->READ),				\
    	      sdr_convert_sensor_reading_cim(sensor, sensor->READ))

/* helper macros for printing sensor thresholds */
#define SENSOR_PRINT_THRESH_SDR(TARGET, READ, FLAG)						\
	set_value(&entry->data.full.TARGET,							\
    	      sensor->sensor.init.thresholds && sensor->mask.type.threshold.set.FLAG,		\
    	      (float)sdr_convert_sensor_reading(sensor, sensor->threshold.READ),		\
    	      sdr_convert_sensor_reading_cim(sensor, sensor->threshold.READ))

#define SENSOR_PRINT_THRESH_SENSOR(TARGET, READ, FLAG)						\
	set_value(&entry->data.full.TARGET,							\
    	      rsp->data[0] & FLAG,								\
    	      (float)sdr_convert_sensor_reading(sensor, rsp->data[READ]),			\
    	      sdr_convert_sensor_reading_cim(sensor, rsp->data[READ]))

/* helper macro for printing sensor hysteresis */
#define SENSOR_PRINT_HYSTERESIS(TARGET, READ)							\
	set_value(&entry->data.full.TARGET,							\
    	      sensor->sensor.init.hysteresis,							\
    	      (float)sdr_convert_sensor_reading(sensor, sensor->threshold.hysteresis.READ),	\
    	      sdr_convert_sensor_reading_cim(sensor, sensor->threshold.hysteresis.READ))


int
ipmi_sdr_print_sensor_full(struct ipmi_intf * intf,
			   struct sdr_record_full_sensor * sensor,
			   int use_sdr_thresholds,
			   pp_ipmi_sdr_list_entry_t *entry,
			   int * error)
{
	char unitstr[16], desc[17];
        const char *descr;
	int i=0, validread=1, do_unit=1;
	double val = 0.0;
	int val_cim = 0;
	struct ipmi_rs * rsp;
        uint8_t min_reading, max_reading;
        int min_reading_cim, max_reading_cim;
	uint8_t u = sensor->unit.type.base;

	if (entry == NULL || sensor == NULL)
		return -1;

	memset(desc, 0, sizeof(desc));
	snprintf(desc, (sensor->id_code & 0x1f) + 1, "%s", sensor->id_string);

        /* only handle linear sensors and linearized sensors (for now) */
        if (sensor->linearization>=SDR_SENSOR_L_NONLINEAR) {
                ipmi_printf("sensor %s non-linear!\n", desc);
		entry->data.full.reading_present = PP_IPMI_SENSOR_READING_NOT_LINEAR;
                return 0;
        }

	rsp = ipmi_sdr_get_sensor_reading(intf, sensor->keys.sensor_num, error);

	if (rsp == NULL) {
		entry->data.full.reading_present = PP_IPMI_SENSOR_READING_ERROR;
		ipmi_printf("Error reading sensor %s (#%02x)\n",
			desc, sensor->keys.sensor_num);
                return -1;
	} else if (rsp->ccode > 0) {
		entry->data.full.reading_present = PP_IPMI_SENSOR_READING_ERROR;
		if (rsp->ccode == 0xcb) {
			/* sensor not found */
		} else {
			ipmi_printf("Error reading sensor %s (#%02x): %s\n",
				desc, sensor->keys.sensor_num,
				val2str(rsp->ccode, completion_code_vals));
		}
		validread = 0;
	} else {
		if (IS_READING_UNAVAILABLE(rsp->data[1])) {
			/* sensor reading unavailable */
			entry->data.full.reading_present = PP_IPMI_SENSOR_READING_NOT_PRESENT;
			validread = 0;
		}
		else if (IS_SCANNING_DISABLED(rsp->data[1])) {
			/* Sensor Scanning Disabled */
			entry->data.full.reading_present = PP_IPMI_SENSOR_READING_SCANNING_DISABLED;
			validread = 0;
			if (rsp->data[0] != 0) {
				/* we might still get a valid reading */
				val = sdr_convert_sensor_reading(sensor, rsp->data[0]);
				val_cim = sdr_convert_sensor_reading_cim(sensor, rsp->data[0]);
				if (val != 0.0)
					validread = 1;
			}
		}
		else if (rsp->data[0] != 0) { 
			/* convert RAW reading into units */
			entry->data.full.reading_present = PP_IPMI_SENSOR_READING_OK;
			val = sdr_convert_sensor_reading(sensor, rsp->data[0]);
			val_cim = sdr_convert_sensor_reading_cim(sensor, rsp->data[0]);
		}
	}
	
        /* determine units with possible modifiers */
	memset(unitstr, 0, sizeof(unitstr));
	if (do_unit && validread) {
		i += snprintf(unitstr, sizeof(unitstr), "%s",
			ipmi_sdr_get_unit_string(sensor->unit.modifier,
				sensor->unit.type.base,
				sensor->unit.type.modifier));
	}

	entry->sdr_id = sensor->keys.sensor_num;
	entry->data.full.owner_id = sensor->keys.owner_id;
	entry->data.full.owner_lun = sensor->keys.lun;
	pp_strappend(&entry->sdr_name, sensor->id_code ? desc : "");
	fill_entity(entry, sensor->entity.id, sensor->entity.instance);

	/* Map IPMI to CIM unit -- the tables are similar, but not equal */
	/* Sources: IPMI spec v2.0, Table 43-15 and CIM Schema v2.8      */
	entry->data.full.cim_base_unit =
	    u ==  0              ?  0      :
	    u >=  1  &&  u <= 19 ?  u + 1  :
	    u >= 20  &&  u <= 22 ?  20     : // usec, msec, sec -> sec
	    u >= 23  &&  u <= 31 ?  u - 1  :
	    u >= 32  &&  u <= 34 ?  31     : // mm, cm, m -> m
	    u >= 35  &&  u <= 49 ?  u - 3  :
	    u >= 50  &&  u <= 51 ?  47     : // henry, millihenry -> henry
	    u >= 52  &&  u <= 53 ?  48     : // farad, microfarad -> farad
	    u >= 54  &&  u <= 58 ?  u - 5  :
	    u >= 60  &&  u <= 65 ?  u - 6  : // 59 = reserved
	    u >= 66  &&  u <= 69 ?  60     : // bit, kilobit, megabit, gigabit -> bit
	    u >= 70  &&  u <= 73 ?  61     : // byte, kilobyte, megabyte, gigabyte -> byte
	    u >= 74  &&  u <= 76 ?  u - 12 :
	    1; // unknown

	entry->data.full.cim_unit_modifier = __TO_R_EXP(sensor->bacc);
	if (u == 20 || u == 53) { // usec, microfarad
	    entry->data.full.cim_unit_modifier -= 6;
	} else if (u == 21 || u == 32 || u == 51) { // msec, mm, millihenry
	    entry->data.full.cim_unit_modifier -= 3;
	} else if (u == 33) { // cm
	    entry->data.full.cim_unit_modifier -= 2;
	} else if (u == 67 || u == 71) { // kilobit, kilobyte
	    entry->data.full.cim_unit_modifier += 3;
	} else if (u == 68 || u == 72) { // megabit, megabyte
	    entry->data.full.cim_unit_modifier += 6;
	} else if (u == 69 || u == 73) { // gigabit, gigabyte
	    entry->data.full.cim_unit_modifier += 9;
	}

	entry->data.full.cim_rate_units =
		sensor->unit.modifier == 1 ? cim_rate_units[sensor->unit.type.modifier] : 0;

	pp_strappend(&entry->sdr_type_sub, ipmi_sdr_get_sensor_type_desc(sensor->sensor.type));
	entry->spec_type = ipmi_sdr_get_sensor_spec_type(sensor->sensor.type);

	if (sensor->unit.analog == 3) {
		/* discrete sensor */
		entry->data.full.type = PP_IPMI_SENSOR_FULL_TYPE_DISCRETE;
		pp_strappend(&entry->sdr_type_main, _("Discrete"));

		if (validread) {
			set_value(&entry->data.full.reading, 1, (float)val, val_cim);
		}

		switch (sensor->sensor.capabilities.event_msg) {
		case 0:
			pp_strappend(&entry->data.full.evt_msg_control_string, _("Per-threshold"));
			entry->data.full.evt_msg_control = PP_IPMI_SENSOR_EVT_CTRL_PRE_TRESHOLD;
			break;
		case 1:
			pp_strappend(&entry->data.full.evt_msg_control_string, _("Entire Sensor Only"));
			entry->data.full.evt_msg_control = PP_IPMI_SENSOR_EVT_CTRL_ENTIRE;
			break;
		case 2:
			pp_strappend(&entry->data.full.evt_msg_control_string, _("Global Disable Only"));
			entry->data.full.evt_msg_control = PP_IPMI_SENSOR_EVT_CTRL_GLOBAL;
			break;
		case 3:
			pp_strappend(&entry->data.full.evt_msg_control_string, _("No Events From Sensor"));
			entry->data.full.evt_msg_control = PP_IPMI_SENSOR_EVT_CTRL_NONE;
			break;
		}

		entry->data.full.status = ipmi_sdr_get_status_discrete(rsp->data[2], rsp->data[3]);
		ipmi_sdr_print_discrete_state(sensor->sensor.type,
					      sensor->event_type, rsp->data[2], rsp->data[3],
					      &entry->data.full.status_string);
		pp_strappend(&entry->data.full.cim_current_state,
			((descr = ipmi_sdr_discrete_get_cim_current_state(
				sensor->sensor.type,
				sensor->event_type,
				rsp->data[2],
                                rsp->data[3])) != NULL) ? _(descr) : "");
		ipmi_sdr_print_sensor_mask(&sensor->mask,
					   sensor->sensor.type,
					   sensor->event_type,
					   DISCRETE_FULL_SENSOR,
					   entry);
		ipmi_sdr_print_sensor_event_status(intf,
						   sensor->keys.sensor_num,
						   sensor->sensor.type,
						   sensor->event_type,
						   DISCRETE_FULL_SENSOR,
						   entry,
						   error);
		ipmi_sdr_print_sensor_event_enable(intf,
						   sensor->keys.sensor_num,
						   sensor->sensor.type,
						   sensor->event_type,
						   DISCRETE_FULL_SENSOR,
						   entry,
						   error);

		ipmi_sdr_discrete_get_cim_possible_states(&entry->data.full.cim_possible_states,
			sensor->sensor.type, sensor->event_type, sensor->mask.type.discrete.read);
	} else {
		/* analog sensor */
		uint8_t thres;
		double reading;

		entry->data.full.type = PP_IPMI_SENSOR_FULL_TYPE_ANALOG;
		pp_strappend(&entry->sdr_type_main, _("Analog"));

		memcpy(&thres, &sensor->mask.type.threshold.set, 1);
		
		if (validread) {
			uint16_t raw_tol = __TO_TOL(sensor->mtol);
			double tol = raw_tol ? sdr_convert_sensor_reading(sensor, (uint8_t)raw_tol) / 2 : 0.0;
			int tol_cim = raw_tol ? sdr_convert_sensor_reading_cim(sensor, (uint8_t)raw_tol) / 2 : 0;

			set_value(&entry->data.full.reading, 1, (float)val, val_cim);
			set_value(&entry->data.full.tolerance, 1, (float)tol, tol_cim);
			pp_strappend(&entry->data.full.unit, unitstr);
		}
		entry->data.full.status = ipmi_sdr_get_status(rsp->data[2]);
		pp_strappend(&entry->data.full.status_string, ipmi_sdr_get_status_full(rsp->data[2]));
                pp_strappend(&entry->data.full.cim_current_state, ipmi_sdr_threshold_get_cim_current_state(rsp->data[2]));
                ipmi_sdr_threshold_get_cim_possible_states(&entry->data.full.cim_possible_states);

		SENSOR_PRINT_NORMAL(nominal_reading, nominal_read);
                SENSOR_PRINT_NORMAL(normal_minimum, normal_min);
                SENSOR_PRINT_NORMAL(normal_maximum, normal_max);

		if (use_sdr_thresholds) {
			SENSOR_PRINT_THRESH_SDR(upper_non_recoverable, upper.non_recover, unr);
			SENSOR_PRINT_THRESH_SDR(upper_critical, upper.critical, ucr);
			SENSOR_PRINT_THRESH_SDR(upper_non_critical, upper.non_critical, unc);
			SENSOR_PRINT_THRESH_SDR(lower_non_recoverable, lower.non_recover, lnr);
			SENSOR_PRINT_THRESH_SDR(lower_critical, lower.critical, lcr);
			SENSOR_PRINT_THRESH_SDR(lower_non_critical, lower.non_critical, lnc);
		} else {
			rsp = ipmi_sdr_get_sensor_thresholds(intf, sensor->keys.sensor_num, error);
			
			if (rsp == NULL) {
				ipmi_printf("Error reading sensor %s (#%02x) threshold\n",
					desc, sensor->keys.sensor_num);
			} else if (rsp->ccode > 0) {
				ipmi_printf("Error reading sensor %s (#%02x) threshold: %s\n",
					desc, sensor->keys.sensor_num,
					val2str(rsp->ccode, completion_code_vals));
			} else {
				SENSOR_PRINT_THRESH_SENSOR(upper_non_recoverable, 6, UPPER_NON_RECOV_SPECIFIED);
				SENSOR_PRINT_THRESH_SENSOR(upper_critical, 5, UPPER_CRIT_SPECIFIED);
				SENSOR_PRINT_THRESH_SENSOR(upper_non_critical, 4, UPPER_NON_CRIT_SPECIFIED);
				SENSOR_PRINT_THRESH_SENSOR(lower_non_recoverable, 3, LOWER_NON_RECOV_SPECIFIED);
				SENSOR_PRINT_THRESH_SENSOR(lower_critical, 2, LOWER_CRIT_SPECIFIED);
				SENSOR_PRINT_THRESH_SENSOR(lower_non_critical, 1, LOWER_NON_CRIT_SPECIFIED);
			}
		}

		SENSOR_PRINT_HYSTERESIS(hysteresis_pos, positive);
		SENSOR_PRINT_HYSTERESIS(hysteresis_neg, negative);

		reading = sdr_convert_sensor_reading(sensor, sensor->sensor_min);
		min_reading = (uint8_t)reading;
		min_reading_cim = sdr_convert_sensor_reading_cim(sensor, sensor->sensor_min);
		set_value(&entry->data.full.minimum_sensor_range,
            	      !((sensor->unit.analog == 0 && sensor->sensor_min == 0x00) ||
                	(sensor->unit.analog == 1 && sensor->sensor_min == 0xff) ||
                	(sensor->unit.analog == 2 && sensor->sensor_min == 0x80)),
                      (float)min_reading, min_reading_cim);

		reading = sdr_convert_sensor_reading(sensor, sensor->sensor_max);
		max_reading = (uint8_t)reading;
		max_reading_cim = sdr_convert_sensor_reading_cim(sensor, sensor->sensor_max);
		set_value(&entry->data.full.maximum_sensor_range,
            	      !((sensor->unit.analog == 0 && sensor->sensor_max == 0xff) ||
                	(sensor->unit.analog == 1 && sensor->sensor_max == 0x00) ||
                	(sensor->unit.analog == 2 && sensor->sensor_max == 0x7f)),
                      (float)max_reading, max_reading_cim);

		switch (sensor->sensor.capabilities.event_msg) {
		case 0:
			pp_strappend(&entry->data.full.evt_msg_control_string, _("Per-threshold"));
			entry->data.full.evt_msg_control = PP_IPMI_SENSOR_EVT_CTRL_PRE_TRESHOLD;
			break;
		case 1:
			pp_strappend(&entry->data.full.evt_msg_control_string, _("Entire Sensor Only"));
			entry->data.full.evt_msg_control = PP_IPMI_SENSOR_EVT_CTRL_ENTIRE;
			break;
		case 2:
			pp_strappend(&entry->data.full.evt_msg_control_string, _("Global Disable Only"));
			entry->data.full.evt_msg_control = PP_IPMI_SENSOR_EVT_CTRL_GLOBAL;
			break;
		case 3:
			pp_strappend(&entry->data.full.evt_msg_control_string, _("No Events From Sensor"));
			entry->data.full.evt_msg_control = PP_IPMI_SENSOR_EVT_CTRL_NONE;
			break;
		}

		switch (sensor->sensor.capabilities.threshold) {
		case 0:
			break;
		case 1: /* readable according to mask */
		case 2: /* readable and settable according to mask */
			if (sensor->mask.type.threshold.read.lnr)
				entry->data.full.readable_thresholds.lower_non_recoverable = 1;
			if (sensor->mask.type.threshold.read.lcr)
				entry->data.full.readable_thresholds.lower_critical = 1;
			if (sensor->mask.type.threshold.read.lnc)
				entry->data.full.readable_thresholds.lower_non_critical = 1;
			if (sensor->mask.type.threshold.read.unc)
				entry->data.full.readable_thresholds.upper_non_critical = 1;
			if (sensor->mask.type.threshold.read.ucr)
				entry->data.full.readable_thresholds.upper_critical = 1;
			if (sensor->mask.type.threshold.read.unr)
				entry->data.full.readable_thresholds.upper_non_recoverable = 1;
			break;
		case 3:
			break;
		}

		switch (sensor->sensor.capabilities.threshold) {
		case 0:
			break;
		case 1: /* readable according to mask */
		case 2: /* readable and settable according to mask */
			if (sensor->mask.type.threshold.set.lnr)
				entry->data.full.settable_thresholds.lower_non_recoverable = 1;
			if (sensor->mask.type.threshold.set.lcr)
				entry->data.full.settable_thresholds.lower_critical = 1;
			if (sensor->mask.type.threshold.set.lnc)
				entry->data.full.settable_thresholds.lower_non_critical = 1;
			if (sensor->mask.type.threshold.set.unc)
				entry->data.full.settable_thresholds.upper_non_critical = 1;
			if (sensor->mask.type.threshold.set.ucr)
				entry->data.full.settable_thresholds.upper_critical = 1;
			if (sensor->mask.type.threshold.set.unr)
				entry->data.full.settable_thresholds.upper_non_recoverable = 1;
			break;
		case 3:
			break;
		}

		if (sensor->mask.type.threshold.status_lnr ||
		    sensor->mask.type.threshold.status_lcr ||
		    sensor->mask.type.threshold.status_lnc ||
		    sensor->mask.type.threshold.status_unc ||
		    sensor->mask.type.threshold.status_ucr ||
		    sensor->mask.type.threshold.status_unr) {
			if (sensor->mask.type.threshold.status_lnr)
				entry->data.full.threshold_read_mask.lower_non_recoverable = 1;
			if (sensor->mask.type.threshold.status_lcr)
				entry->data.full.threshold_read_mask.lower_critical = 1;
			if (sensor->mask.type.threshold.status_lnc)
				entry->data.full.threshold_read_mask.lower_non_critical = 1;
			if (sensor->mask.type.threshold.status_unc)
				entry->data.full.threshold_read_mask.upper_non_critical = 1;
			if (sensor->mask.type.threshold.status_ucr)
				entry->data.full.threshold_read_mask.upper_critical = 1;
			if (sensor->mask.type.threshold.status_unr)
				entry->data.full.threshold_read_mask.upper_non_recoverable = 1;
		}

		ipmi_sdr_print_sensor_mask(&sensor->mask,
					   sensor->sensor.type,
					   sensor->event_type,
					   ANALOG_SENSOR,
					   entry);
		ipmi_sdr_print_sensor_event_status(intf,
						   sensor->keys.sensor_num,
						   sensor->sensor.type,
						   sensor->event_type,
						   ANALOG_SENSOR,
						   entry,
						   error);
		ipmi_sdr_print_sensor_event_enable(intf,
						   sensor->keys.sensor_num,
						   sensor->sensor.type,
						   sensor->event_type,
						   ANALOG_SENSOR,
						   entry,
						   error);
        }
        
        /* store value conversion information */
	entry->data.full.m = __TO_M(sensor->mtol);
	entry->data.full.b = __TO_B(sensor->bacc);
	entry->data.full.b_exp = __TO_B_EXP(sensor->bacc);
	entry->data.full.r_exp = __TO_R_EXP(sensor->bacc);
        entry->data.full.analog = sensor->unit.analog;
        entry->data.full.linearization = sensor->linearization;
	
	return 0;
}

/* ipmi_sdr_print_discrete_state  -  print list of asserted states
 *                                   for a discrete sensor
 *
 * @desc        : description for this line
 * @sensor_type	: sensor type code
 * @event_type	: event type code
 * @state	: mask of asserted states
 *
 * no meaningful return value
 */
void ipmi_sdr_print_discrete_state(uint8_t sensor_type, uint8_t event_type,
				   uint8_t state1, uint8_t state2,
				   pp_strstream_t *stream)
{
    uint8_t typ;
    struct ipmi_event_sensor_types *evt;
    int pre = 0, c = 0;
    int show_type = 0;
  
    if (state1 == 0 && state2 == 0)
	return;

    if (event_type >= 0x70 && event_type <= 0x7f) { // OEM displayed 'raw'
	pp_strappendf(stream, "0x%.2hhx%.2hhx", state2, state1);
	
    } else {
	if (event_type == 0x6f) {
	    evt = sensor_specific_types;
	    typ = sensor_type;
	} else {
	    evt = generic_event_types;
	    typ = event_type;
	}
    
	for (; evt->type != NULL; evt++) {
	    if (evt->code != typ)
		continue;
      
	    if (pre == 0) {
		pre = 1;
		if (show_type) {
		    pp_strappendf(stream, "%s", _(evt->type));
		}
	    }
      
	    if (evt->offset > 7) {
		if ((1<<(evt->offset-8)) & state2) {
		    if (pre == 1) {
			if (show_type) {
			    pp_strappend(stream, " (");
			}
			pre = 2;
		    } else if (pre == 2) {
			pp_strappend(stream, ", ");
		    }
                    pp_strappend(stream, evt->desc ? _(evt->desc) : "");
		}
	    } else {
		if ((1<<evt->offset) & state1) {
		    if (pre == 1) {
			if (show_type) {
			    pp_strappend(stream, " (");
			}
			pre = 2;
		    } else if (pre == 2) {
			pp_strappend(stream, ", ");
		    }
                    pp_strappend(stream, evt->desc ? _(evt->desc) : "");
		}
	    }
	    c++;
	}
    
	if (pre == 2 && show_type) {
	    pp_strappend(stream, ")");
	}
    }
}

/* ipmi_sdr_print_sensor_compact  -  print SDR compact record
 *
 * @intf:	ipmi interface
 * @sensor:	compact sdr record
 *
 * returns 0 on success
 * returns -1 on error
 */
int
ipmi_sdr_print_sensor_compact(struct ipmi_intf * intf,
			      struct sdr_record_compact_sensor * sensor,
			      pp_ipmi_sdr_list_entry_t *entry,
			      int * error)
{
	struct ipmi_rs * rsp;
	char desc[17];
        const char *descr;
	int validread;

	if (entry == NULL || sensor == NULL)
		return -1;

	memset(desc, 0, sizeof(desc));
	snprintf(desc, (sensor->id_code & 0x1f) + 1, "%s", sensor->id_string);
	
	/* get sensor reading */
	rsp = ipmi_sdr_get_sensor_reading(intf, sensor->keys.sensor_num, error);
	if (rsp == NULL) {
		ipmi_printf("Error reading sensor %s (#%02x)\n",
			desc, sensor->keys.sensor_num);
		validread = 0;
	} else if (rsp->ccode > 0 && rsp->ccode != 0xcd) {
		/* completion code 0xcd is special case */
		ipmi_printf("Error reading sensor %s (#%02x): %s\n",
		       desc, sensor->keys.sensor_num,
		       val2str(rsp->ccode, completion_code_vals));
		validread = 0;
	} else if (IS_SCANNING_DISABLED(rsp->data[1])) {
		ipmi_printf("Sensor %s (#%02x) scanning disabled\n",
			desc, sensor->keys.sensor_num);
		entry->data.compact.scanning_disabled = 1;
		entry->data.compact.status = PP_IPMI_SENSOR_STATUS_NO_READING;
		validread = 0;
	} else if (IS_READING_UNAVAILABLE(rsp->data[1])) {
		ipmi_printf("Sensor %s (#%02x) reading unavailable\n",
			desc, sensor->keys.sensor_num);
		entry->data.compact.status = PP_IPMI_SENSOR_STATUS_NO_READING;
		validread = 0;
	} else {
	    validread = 1;
	    entry->data.compact.status = ipmi_sdr_get_status_discrete(rsp->data[2], rsp->data[3]);
	}

	entry->sdr_id = sensor->keys.sensor_num;
	entry->data.compact.owner_id = sensor->keys.owner_id;
	entry->data.compact.owner_lun = sensor->keys.lun;
	pp_strappend(&entry->sdr_name, sensor->id_code ? desc : "");
	fill_entity(entry, sensor->entity.id, sensor->entity.instance);

	pp_strappend(&entry->sdr_type_main, _("Compact"));
	pp_strappend(&entry->sdr_type_sub, ipmi_sdr_get_sensor_type_desc(sensor->sensor.type));
	entry->spec_type = ipmi_sdr_get_sensor_spec_type(sensor->sensor.type);

	entry->data.compact.event_type = sensor->event_type;


	if (validread) {
		ipmi_sdr_print_discrete_state(sensor->sensor.type,
					      sensor->event_type,
					      rsp->data[2],
					      rsp->data[3],
					      &entry->data.compact.description);
		pp_strappend(&entry->data.compact.cim_current_state,
			((descr = ipmi_sdr_discrete_get_cim_current_state(
				sensor->sensor.type,
				sensor->event_type,
				rsp->data[2],
                                rsp->data[3])) != NULL) ? _(descr) : "");
	} else {
		pp_strappend(&entry->data.compact.cim_current_state, _("OK"));
	}

	ipmi_sdr_print_sensor_mask(&sensor->mask,
				   sensor->sensor.type,
				   sensor->event_type,
				   DISCRETE_COMPACT_SENSOR,
				   entry);
	ipmi_sdr_print_sensor_event_status(intf,
					   sensor->keys.sensor_num, 
					   sensor->sensor.type,
					   sensor->event_type,
					   DISCRETE_COMPACT_SENSOR,
					   entry,
					   error);
	ipmi_sdr_print_sensor_event_enable(intf,
					   sensor->keys.sensor_num,
					   sensor->sensor.type,
					   sensor->event_type,
					   DISCRETE_COMPACT_SENSOR,
					   entry,
					   error);

	ipmi_sdr_discrete_get_cim_possible_states(&entry->data.compact.cim_possible_states,
		sensor->sensor.type, sensor->event_type, sensor->mask.type.discrete.read);

	return 0;
}

/* ipmi_sdr_print_sensor_eventonly  -  print SDR event only record
 *
 * @intf:	ipmi interface
 * @sensor:	event only sdr record
 *
 * returns 0 on success
 * returns -1 on error
 */
int
ipmi_sdr_print_sensor_eventonly(struct ipmi_intf * intf UNUSED,
				struct sdr_record_eventonly_sensor * sensor,
				pp_ipmi_sdr_list_entry_t *entry)
{
	char desc[17];

	if (entry == NULL || sensor == NULL)
		return -1;

	memset(desc, 0, sizeof(desc));
	snprintf(desc, (sensor->id_code & 0x1f) + 1, "%s", sensor->id_string);

	entry->sdr_id = sensor->keys.sensor_num;
	entry->data.event_only.owner_id = sensor->keys.owner_id;
	entry->data.event_only.owner_lun = sensor->keys.lun;
	pp_strappend(&entry->sdr_name, sensor->id_code ? desc : "");
	fill_entity(entry, sensor->entity.id, sensor->entity.instance);

	pp_strappend(&entry->sdr_type_main, _("Event"));
	pp_strappend(&entry->sdr_type_sub, ipmi_sdr_get_sensor_type_desc(sensor->sensor_type));
	entry->spec_type = ipmi_sdr_get_sensor_spec_type(sensor->sensor_type);

	entry->data.event_only.event_type = sensor->event_type;
	
	return 0;
}

int
ipmi_sdr_print_mc_locator(struct ipmi_intf * intf UNUSED,
			  struct sdr_record_mc_locator * mc,
			  pp_ipmi_sdr_list_entry_t *entry)
{
	if (entry == NULL || mc == NULL)
		return -1;

	entry->sdr_id = -1;
	pp_strappend(&entry->sdr_name, (char *)mc->id_string);
	fill_entity(entry, mc->entity.id, mc->entity.instance);

	pp_strappend(&entry->sdr_type_main, _("MC Device locator"));
	entry->spec_type = ipmi_sdr_get_sensor_spec_type(PP_IPMI_SPEC_SENSOR_TYPE_MC_DEVICE_LOCATOR);

	entry->data.mc_locator.dev_slave_addr = mc->dev_slave_addr;
	entry->data.mc_locator.channel_num = mc->channel_num;

	entry->data.mc_locator.acpi_system_ps_notif_required = (mc->pwr_state_notif & 0x4) ? 1 : 0;
	entry->data.mc_locator.acpi_device_ps_notif_required = (mc->pwr_state_notif & 0x2) ? 1 : 0;
	entry->data.mc_locator.controler_presence_static = (mc->pwr_state_notif & 0x1) ? 1 : 0;
	entry->data.mc_locator.logs_init_agent_errors = (mc->global_init & 0x8) ? 1 : 0;

	if (!(mc->global_init & 0x3))
		pp_strappend(&entry->data.mc_locator.event_message_gen, _("Enable"));
	else if ((mc->global_init & 0x3) == 0x1)
		pp_strappend(&entry->data.mc_locator.event_message_gen, _("Disable"));
	else if ((mc->global_init & 0x3) == 0x2)
		pp_strappend(&entry->data.mc_locator.event_message_gen, _("Do Not Init Controller"));
	else
		pp_strappend(&entry->data.mc_locator.event_message_gen, _("Reserved"));

	entry->data.mc_locator.cap_chassis_device = (mc->dev_support & 0x80) ? 1 : 0;
	entry->data.mc_locator.cap_bridge = (mc->dev_support & 0x40) ? 1 : 0;
	entry->data.mc_locator.cap_ipmb_event_generator = (mc->dev_support & 0x20) ? 1 : 0;
	entry->data.mc_locator.cap_ipmi_event_receiver = (mc->dev_support & 0x10) ? 1 : 0;
	entry->data.mc_locator.cap_fru_inventory_support = (mc->dev_support & 0x08) ? 1 : 0;
	entry->data.mc_locator.cap_sel_device = (mc->dev_support & 0x04) ? 1 : 0;
	entry->data.mc_locator.cap_sdr_repository = (mc->dev_support & 0x02) ? 1 : 0;
	entry->data.mc_locator.cap_sensor_device = (mc->dev_support & 0x01) ? 1 : 0;

	return 0;
}

/* ipmi_sdr_print_sensor_generic_locator  -  print generic device locator record
 *
 * @intf:	ipmi interface
 * @gen:	generic device locator sdr record
 *
 * returns 0 on success
 * returns -1 on error
 */
int
ipmi_sdr_print_generic_locator(struct ipmi_intf * intf UNUSED,
			       struct sdr_record_generic_locator * dev,
			       pp_ipmi_sdr_list_entry_t *entry)
{
	if (entry == NULL)
		return 0;

	entry->sdr_id = -1;
	pp_strappend(&entry->sdr_name, (char *)dev->id_string);
	fill_entity(entry, dev->entity.id, dev->entity.instance);

	pp_strappend(&entry->sdr_type_main, _("Generic Device locator"));
	entry->spec_type = ipmi_sdr_get_sensor_spec_type(PP_IPMI_SPEC_SENSOR_TYPE_GENERIC_DEVICE_LOCATOR);

	entry->data.generic_locator.dev_access_addr = dev->dev_access_addr;
	entry->data.generic_locator.dev_slave_addr = dev->dev_slave_addr;
	entry->data.generic_locator.address_span = dev->addr_span;

	entry->data.generic_locator.lun = dev->lun;
	entry->data.generic_locator.bus = dev->bus;
	entry->data.generic_locator.channel_num = dev->channel_num;
	entry->data.generic_locator.dev_type = dev->dev_type;
	entry->data.generic_locator.dev_type_modifier = dev->dev_type_modifier;
	pp_strappend(&entry->data.generic_locator.dev_type_string,
		val2str((uint16_t)(dev->dev_type << 8 | dev->dev_type_modifier), entity_device_type_vals));
	
	return 0;
}

/* ipmi_sdr_print_sensor_fru_locator  -  print FRU locator record
 *
 * @intf:	ipmi interface
 * @fru:	fru locator sdr record
 *
 * returns 0 on success
 * returns -1 on error
 */
int
ipmi_sdr_print_fru_locator(struct ipmi_intf * intf UNUSED,
			   struct sdr_record_fru_locator * fru,
			   pp_ipmi_sdr_list_entry_t *entry)
{
	if (entry == NULL)
		return 0;

	entry->sdr_id = -1;
	pp_strappend(&entry->sdr_name, (char *)fru->id_string);
	fill_entity(entry, fru->entity.id, fru->entity.instance);

	pp_strappend(&entry->sdr_type_main, _("FRU Device locator"));
	entry->spec_type = ipmi_sdr_get_sensor_spec_type(PP_IPMI_SPEC_SENSOR_TYPE_FRU_DEVICE_LOCATOR);

	entry->data.fru_locator.dev_slave_addr = fru->dev_slave_addr;
	entry->data.fru_locator.dev_id = fru->device_id;
	pp_strappend(&entry->data.fru_locator.dev_id_description, 
                     fru->logical ? _("Logical FRU Device") : 
                                    _("Slave Address"));
	entry->data.fru_locator.lun = fru->lun;
	entry->data.fru_locator.bus = fru->bus;
	entry->data.fru_locator.channel_num = fru->channel_num;
	entry->data.fru_locator.dev_type = fru->dev_type;
	entry->data.fru_locator.dev_type_modifier = fru->dev_type_modifier;
	pp_strappend(&entry->data.fru_locator.dev_type_string,
		val2str((uint16_t)(fru->dev_type << 8 | fru->dev_type_modifier), entity_device_type_vals));

	return 0;
}

/* ipmi_sdr_print_sensor_entity_assoc  -  print SDR entity association record
 *
 * @intf:	ipmi interface
 * @mc:		entity association sdr record
 *
 * returns 0 on success
 * returns -1 on error
 */
static int
ipmi_sdr_print_sensor_entity_assoc(struct ipmi_intf * intf UNUSED,
				   struct sdr_record_entity_assoc * assoc UNUSED)
{
	return 0;
}

/**
 * ipmi_sdr_print_sensor_raw_reading - print raw sensor reading
 *
 * @param intf          ipmi interface
 * @param entry         raw sensor reading list entry
 *
 * @returns             0 on success, -1 on error
 */
int
ipmi_sdr_print_sensor_raw_reading(struct ipmi_intf * intf,
                                  pp_ipmi_raw_reading_list_entry_t *entry,
                                  int * error)
{
	struct ipmi_rs * rsp;

	if (entry == NULL)
		return -1;

        /* get raw sensor reading */
	rsp = ipmi_sdr_get_sensor_reading(intf, entry->id, error);

	if (rsp == NULL) {
		entry->reading_present = PP_IPMI_SENSOR_READING_ERROR;
		ipmi_printf("Error reading sensor (#%02x)\n", entry->id);
                return -1;
	} else if (rsp->ccode > 0) {
		entry->reading_present = PP_IPMI_SENSOR_READING_ERROR;
		if (rsp->ccode == 0xcb) {
			/* sensor not found */
		} else {
			ipmi_printf("Error reading sensor (#%02x): %s\n",
				    entry->id,
				    val2str(rsp->ccode, completion_code_vals));
		}
                return -1;
	} else {
		if (IS_READING_UNAVAILABLE(rsp->data[1])) {
			/* sensor reading unavailable */
			entry->reading_present = PP_IPMI_SENSOR_READING_NOT_PRESENT;
		}
		else if (IS_SCANNING_DISABLED(rsp->data[1])) {
			/* Sensor Scanning Disabled */
			entry->reading_present = PP_IPMI_SENSOR_READING_SCANNING_DISABLED;
			if (rsp->data[0] != 0) {
                               entry->raw_value = rsp->data[0];
			}
		}
		else if (rsp->data[0] != 0) {
			entry->reading_present = PP_IPMI_SENSOR_READING_OK;
			entry->raw_value = rsp->data[0];
		}
	}
        
        entry->threshold.data = rsp->data[2];
	
	return 0;
}

/**
 * ipmi_sdr_print_sensor_raw_thresholds - print raw sensor thresholds
 *
 * @param intf          ipmi interface
 * @param entry         raw sensor reading list entry
 *
 * @returns             0 on success, -1 on error
 */
int
ipmi_sdr_print_sensor_raw_thresholds(struct ipmi_intf * intf,
                                     pp_ipmi_raw_thresholds_list_entry_t *entry,
                                     int * error)
{
	struct ipmi_rs * rsp;

	if (entry == NULL)
		return -1;

        /* get raw sensor reading */
	rsp = ipmi_sdr_get_sensor_thresholds(intf, entry->id, error);

	if (rsp == NULL) {
                entry->threshold_present.data = 0x00;
		ipmi_printf("Error reading sensor (#%02x)\n", entry->id);
                return -1;
	} else if (rsp->ccode > 0) {
                entry->threshold_present.data = 0x00;
		ipmi_printf("Error reading sensor (#%02x): %s\n",
			    entry->id,
			    val2str(rsp->ccode, completion_code_vals));
                return -1;
	}
        
        entry->threshold_present.data = rsp->data[0];
        entry->lower_non_critical =     rsp->data[1];
        entry->lower_critical =         rsp->data[2];
        entry->lower_non_recoverable =  rsp->data[3];
        entry->upper_non_critical =     rsp->data[4];
        entry->upper_critical =         rsp->data[5];
        entry->upper_non_recoverable =  rsp->data[6];
	
	return 0;
}

static int
sdr_print_sensor_full(struct ipmi_intf * intf, struct sdr_record_full_sensor * rec, pp_ipmi_return_t *ipmi_ret, int * error)
{
	pp_ipmi_sdr_list_entry_t *entry = ipmi_init_sdr_list_entry(PP_IPMI_SENSOR_TYPE_FULL, ipmi_ret);
	memcpy(entry->raw,rec,sizeof(struct sdr_record_full_sensor));
	if (entry) {
		return ipmi_sdr_print_sensor_full(intf, rec, 1, entry, error);
	} else {
		return -1;
	}
}

static int
sdr_print_sensor_compact(struct ipmi_intf * intf, struct sdr_record_compact_sensor * rec, pp_ipmi_return_t *ipmi_ret, int * error)
{
	pp_ipmi_sdr_list_entry_t *entry = ipmi_init_sdr_list_entry(PP_IPMI_SENSOR_TYPE_COMPACT_SENSOR, ipmi_ret);
	memcpy(entry->raw,rec,sizeof(struct sdr_record_compact_sensor));
	if (entry) {
		return ipmi_sdr_print_sensor_compact(intf, rec, entry, error);
	} else {
		return -1;
	}
}

static int
sdr_print_sensor_eventonly(struct ipmi_intf * intf, struct sdr_record_eventonly_sensor * rec, pp_ipmi_return_t *ipmi_ret)
{
	pp_ipmi_sdr_list_entry_t *entry = ipmi_init_sdr_list_entry(PP_IPMI_SENSOR_TYPE_EVENTONLY, ipmi_ret);
	memcpy(entry->raw,rec,sizeof(struct sdr_record_eventonly_sensor));
	if (entry) {
		return ipmi_sdr_print_sensor_eventonly(intf, rec, entry);
	} else {
		return -1;
	}
}

static int
sdr_print_fru_locator(struct ipmi_intf * intf, struct sdr_record_fru_locator * rec, pp_ipmi_return_t *ipmi_ret)
{
	pp_ipmi_sdr_list_entry_t *entry = ipmi_init_sdr_list_entry(PP_IPMI_SENSOR_TYPE_FRU_DEVICE_LOCATOR, ipmi_ret);
	memcpy(entry->raw,rec,sizeof(struct sdr_record_fru_locator));
	if (entry) {
		return ipmi_sdr_print_fru_locator(intf, rec, entry);
	} else {
		return -1;
	}
}

static int
sdr_print_mc_locator(struct ipmi_intf * intf, struct sdr_record_mc_locator * rec, pp_ipmi_return_t *ipmi_ret)
{
	pp_ipmi_sdr_list_entry_t *entry = ipmi_init_sdr_list_entry(PP_IPMI_SENSOR_TYPE_MC_DEVICE_LOCATOR, ipmi_ret);
	memcpy(entry->raw,rec,sizeof(struct sdr_record_mc_locator));
	if (entry) {
		return ipmi_sdr_print_mc_locator(intf, rec, entry);
	} else {
		return -1;
	}
}

static int
sdr_print_sensor_generic_locator(struct ipmi_intf * intf, struct sdr_record_generic_locator * rec, pp_ipmi_return_t *ipmi_ret)
{
	pp_ipmi_sdr_list_entry_t *entry = ipmi_init_sdr_list_entry(PP_IPMI_SENSOR_TYPE_GENERIC_DEVICE_LOCATOR, ipmi_ret);
	memcpy(entry->raw,rec,sizeof(struct sdr_record_generic_locator));
	if (entry) {
		return ipmi_sdr_print_generic_locator(intf, rec, entry);
	} else {
		return -1;
	}
}

static int
sdr_print_sensor_entity_assoc(struct ipmi_intf * intf, struct sdr_record_entity_assoc * rec, pp_ipmi_return_t *ipmi_ret UNUSED)
{
	return ipmi_sdr_print_sensor_entity_assoc(intf, rec);
}

/* ipmi_sdr_print_rawentry  -  Print SDR entry from raw data
 *
 * @intf:	ipmi interface
 * @type:	sensor type
 * @raw:	raw sensor data
 * @len:	length of raw sensor data
 *
 * returns 0 on success
 * returns -1 on error
 */
static int
ipmi_sdr_print_rawentry(struct ipmi_intf * intf, uint8_t type,
			uint8_t * raw, int len UNUSED,
			pp_ipmi_return_t *ipmi_ret,
			int * error)
{
	int rc = 0;

	switch (type) {
	case SDR_RECORD_TYPE_FULL_SENSOR:
		rc = sdr_print_sensor_full(intf,
				(struct sdr_record_full_sensor *) raw, ipmi_ret, error);
		break;
	case SDR_RECORD_TYPE_COMPACT_SENSOR:
		rc = sdr_print_sensor_compact(intf,
				(struct sdr_record_compact_sensor *) raw, ipmi_ret, error);
		break;
	case SDR_RECORD_TYPE_EVENTONLY_SENSOR:
		rc = sdr_print_sensor_eventonly(intf,
				(struct sdr_record_eventonly_sensor *) raw, ipmi_ret);
		break;
	case SDR_RECORD_TYPE_GENERIC_DEVICE_LOCATOR:
		rc = sdr_print_sensor_generic_locator(intf,
				(struct sdr_record_generic_locator *) raw, ipmi_ret);
		break;
	case SDR_RECORD_TYPE_FRU_DEVICE_LOCATOR:
		rc = sdr_print_fru_locator(intf,
				(struct sdr_record_fru_locator *) raw, ipmi_ret);
		break;
	case SDR_RECORD_TYPE_MC_DEVICE_LOCATOR:
		rc = sdr_print_mc_locator(intf,
				(struct sdr_record_mc_locator *) raw, ipmi_ret);
		break;
	case SDR_RECORD_TYPE_ENTITY_ASSOC:
		rc = sdr_print_sensor_entity_assoc(intf,
				(struct sdr_record_entity_assoc *) raw, ipmi_ret);
		break;
	case SDR_RECORD_TYPE_OEM:
	case SDR_RECORD_TYPE_DEVICE_ENTITY_ASSOC:
	case SDR_RECORD_TYPE_MC_CONFIRMATION:
	case SDR_RECORD_TYPE_BMC_MSG_CHANNEL_INFO:
		/* not implemented */
		break;
	}
    
	return rc;
}

/* ipmi_sdr_print_listentry  -  Print SDR entry from list
 *
 * @intf:	ipmi interface
 * @entry:	sdr record list entry
 *
 * returns 0 on success
 * returns -1 on error
 */
static int
ipmi_sdr_print_listentry(struct ipmi_intf * intf,
			 struct sdr_record_list * entry,
			 pp_ipmi_return_t *ipmi_ret,
			 int * error)
{
	int rc = 0;

	switch (entry->type) {
	case SDR_RECORD_TYPE_FULL_SENSOR:
		rc = sdr_print_sensor_full(intf, entry->record.full, ipmi_ret, error);
		break;
	case SDR_RECORD_TYPE_COMPACT_SENSOR:
		rc = sdr_print_sensor_compact(intf, entry->record.compact, ipmi_ret, error);
		break;
	case SDR_RECORD_TYPE_EVENTONLY_SENSOR:
		rc = sdr_print_sensor_eventonly(intf, entry->record.eventonly, ipmi_ret);
		break;
	case SDR_RECORD_TYPE_GENERIC_DEVICE_LOCATOR:
		rc = sdr_print_sensor_generic_locator(intf, entry->record.genloc, ipmi_ret);
		break;
	case SDR_RECORD_TYPE_FRU_DEVICE_LOCATOR:
		rc = sdr_print_fru_locator(intf, entry->record.fruloc, ipmi_ret);
		break;
	case SDR_RECORD_TYPE_MC_DEVICE_LOCATOR:
		rc = sdr_print_mc_locator(intf, entry->record.mcloc, ipmi_ret);
		break;
	case SDR_RECORD_TYPE_ENTITY_ASSOC:
		rc = sdr_print_sensor_entity_assoc(intf, entry->record.entassoc, ipmi_ret);
		break;
	case SDR_RECORD_TYPE_OEM:
	case SDR_RECORD_TYPE_DEVICE_ENTITY_ASSOC:
	case SDR_RECORD_TYPE_MC_CONFIRMATION:
	case SDR_RECORD_TYPE_BMC_MSG_CHANNEL_INFO:
		/* not implemented yet */
		break;
	}

	return rc;
}

static int ipmi_sdr_update_sdr(struct ipmi_intf * intf,
	size_t idx,
	pp_ipmi_return_t *ipmi_ret,
	int *error)
{
    pp_ipmi_sdr_list_entry_t *entry = vector_get(ipmi_ret->data.sdr_list,idx);
    void *raw=entry->raw;
    int rc = 0;

    switch (entry->type) {
	case PP_IPMI_SENSOR_TYPE_FULL:
	    rc = sdr_print_sensor_full(intf,
		    (struct sdr_record_full_sensor *) raw, ipmi_ret, error);
	    break;
	case PP_IPMI_SENSOR_TYPE_COMPACT_SENSOR:
	    rc = sdr_print_sensor_compact(intf,
		    (struct sdr_record_compact_sensor *) raw, ipmi_ret, error);
	    break;
	case PP_IPMI_SENSOR_TYPE_EVENTONLY:
	    rc = sdr_print_sensor_eventonly(intf,
		    (struct sdr_record_eventonly_sensor *) raw, ipmi_ret);
	    break;
	case PP_IPMI_SENSOR_TYPE_GENERIC_DEVICE_LOCATOR:
	    rc = sdr_print_sensor_generic_locator(intf,
		    (struct sdr_record_generic_locator *) raw, ipmi_ret);
	    break;
	case PP_IPMI_SENSOR_TYPE_FRU_DEVICE_LOCATOR:
	    rc = sdr_print_fru_locator(intf,
		    (struct sdr_record_fru_locator *) raw, ipmi_ret);
	    break;
	case PP_IPMI_SENSOR_TYPE_MC_DEVICE_LOCATOR:
	    rc = sdr_print_mc_locator(intf,
		    (struct sdr_record_mc_locator *) raw, ipmi_ret);
	    break;
	default:
	    /* not implemented */
	    break;
    }

    return rc;
}

/* ipmi_sdr_print_sdr  -  iterate through SDR printing records
 *
 * intf:	ipmi interface
 * type:	record type to print
 *
 * returns 0 on success
 * returns -1 on error
 */
static int ipmi_sdr_print_sdr(struct ipmi_intf * intf,
			      uint8_t type,
			      pp_ipmi_return_t *ipmi_ret,
			      int * error)
{
	struct sdr_get_rs * header;
	struct sdr_record_list * e;
	int rc = 0;

	if (ipmi_init_sdr_list(ipmi_ret)) {
		ipmi_printf("Could not allocate return structure.\n");
		return -1;
	}

	if (verbose > 1)
		ipmi_printf("Querying SDR for sensor list\n");

	if (sdr_list_itr == NULL) {
		sdr_list_itr = ipmi_sdr_start(intf, error);
		if (sdr_list_itr == NULL) {
			ipmi_printf("Unable to open SDR for reading\n");
			return -1;
		}
	}

	for (e = sdr_list_head; e != NULL; e = e->next) {
		if (type != e->type && type != 0xff && type != 0xfe)
			continue;
		if (type == 0xfe &&
		    e->type != SDR_RECORD_TYPE_FULL_SENSOR &&
		    e->type != SDR_RECORD_TYPE_COMPACT_SENSOR)
			continue;
		if (ipmi_sdr_print_listentry(intf, e, ipmi_ret, error) < 0) 
			rc = -1;
	}

	while ((header = ipmi_sdr_get_next_header(intf, sdr_list_itr, error)) != NULL) {
		uint8_t * rec;
		struct sdr_record_list * sdrr;

		rec = ipmi_sdr_get_record(intf, header, sdr_list_itr, error);
		if (rec == NULL) {
			rc = -1;
			continue;
		}

		sdrr = malloc(sizeof(struct sdr_record_list));
		if (sdrr == NULL) {
			ipmi_printf("%s: malloc failure\n", __FUNCTION__);
			ipmi_set_error(error, IPMI_ERROR_NOT_ENOUGH_MEMORY);
			break;
		}
		memset(sdrr, 0, sizeof(struct sdr_record_list));
		sdrr->id = header->id;
		sdrr->type = header->type;
		
		switch (header->type) {
		case SDR_RECORD_TYPE_FULL_SENSOR:
			sdrr->record.full = (struct sdr_record_full_sensor *)rec;
			break;
		case SDR_RECORD_TYPE_COMPACT_SENSOR:
			sdrr->record.compact = (struct sdr_record_compact_sensor *)rec;
			break;
		case SDR_RECORD_TYPE_EVENTONLY_SENSOR:
			sdrr->record.eventonly = (struct sdr_record_eventonly_sensor *)rec;
			break;
		case SDR_RECORD_TYPE_GENERIC_DEVICE_LOCATOR:
			sdrr->record.genloc = (struct sdr_record_generic_locator *)rec;
			break;
		case SDR_RECORD_TYPE_FRU_DEVICE_LOCATOR:
			sdrr->record.fruloc = (struct sdr_record_fru_locator *)rec;
			break;
		case SDR_RECORD_TYPE_MC_DEVICE_LOCATOR:
			sdrr->record.mcloc = (struct sdr_record_mc_locator *)rec;
			break;
		case SDR_RECORD_TYPE_ENTITY_ASSOC:
			sdrr->record.entassoc = (struct sdr_record_entity_assoc *)rec;
			break;
		default:
			free(rec);
			continue;
		}

		if (type == header->type || type == 0xff ||
		    (type == 0xfe &&
		     (header->type == SDR_RECORD_TYPE_FULL_SENSOR ||
		      header->type == SDR_RECORD_TYPE_COMPACT_SENSOR))) {
			if (ipmi_sdr_print_rawentry(intf, header->type,
						    rec, header->length,
						    ipmi_ret, error) < 0) 
				rc = -1;
		}

		/* add to global record liset */
		if (sdr_list_head == NULL)
			sdr_list_head = sdrr;
		else
			sdr_list_tail->next = sdrr;

		sdr_list_tail = sdrr;
	}
	return rc;
}

/* ipmi_sdr_get_reservation  -  Obtain SDR reservation ID
 *
 * @intf:	ipmi interface
 * @reserve_id:	pointer to short int for storing the id
 *
 * returns 0 on success
 * returns -1 on error
 */
int
ipmi_sdr_get_reservation(struct ipmi_intf * intf,
			 uint16_t *reserve_id, int * error)
{
	struct ipmi_rs *rsp;
	struct ipmi_rq req;

	/* obtain reservation ID */
	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_STORAGE;
	req.msg.cmd = GET_SDR_RESERVE_REPO;
	rsp = intf->sendrecv(intf, &req, error);

	/* be slient for errors, they are handled by calling function */
	if (rsp == NULL)
		return -1;
	if (rsp->ccode > 0)
		return -1;

	*reserve_id = le16_to_cpu(((struct sdr_reserve_repo_rs *)&(rsp->data))->reserve_id);
	ipmi_printf("SDR reserveration ID %04x\n", *reserve_id);

	return 0;
}

/* ipmi_sdr_start  -  setup sdr iterator
 *
 * @intf:	ipmi interface
 *
 * returns sdr iterator structure pointer
 * returns NULL on error
 */
struct ipmi_sdr_iterator *
ipmi_sdr_start(struct ipmi_intf * intf, int * error)
{
	struct ipmi_sdr_iterator * itr;
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	struct sdr_repo_info_rs sdr_info;

	itr = malloc(sizeof(struct ipmi_sdr_iterator));
	if (itr == NULL) {
		ipmi_printf("%s: malloc failure\n", __FUNCTION__);
		ipmi_set_error(error, IPMI_ERROR_NOT_ENOUGH_MEMORY);
		return NULL;
	}

	/* get sdr repository info */
	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_STORAGE;
	req.msg.cmd = GET_SDR_REPO_INFO;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Error obtaining SDR info\n");
		free(itr);
		return NULL;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Error obtaining SDR info: %s\n",
		       val2str(rsp->ccode, completion_code_vals));
		free(itr);
		return NULL;
	}

	memcpy(&sdr_info, rsp->data, sizeof(sdr_info));

	/* IPMIv1.0 == 0x01
	 * IPMIv1.5 == 0x51
	 * IPMIv2.0 == 0x02
	 */
	if ((sdr_info.version != 0x51) &&
	    (sdr_info.version != 0x01) &&
	    (sdr_info.version != 0x02)) {
		ipmi_printf("WARNING: Unknown SDR repository "
			"version 0x%02x\n", sdr_info.version);
	}

	itr->total = sdr_info.count;
	itr->next = 0;

	if (verbose > 1) {
		ipmi_printf("SDR free space: %d\n", sdr_info.free);
		ipmi_printf("SDR records: %d\n", sdr_info.count);
	}

	if (ipmi_sdr_get_reservation(intf, &(itr->reservation), error) < 0) {
		ipmi_printf("Unable to obtain SDR reservation\n");
		free(itr);
		return NULL;
	}

	return itr;
}

/* ipmi_sdr_get_record  -  return RAW SDR record
 *
 * @intf:	ipmi interface
 * @header:	SDR header
 * @itr:	SDR iterator
 *
 * returns raw SDR data
 * returns NULL on error
 */
uint8_t *
ipmi_sdr_get_record(struct ipmi_intf * intf, struct sdr_get_rs * header,
		    struct ipmi_sdr_iterator * itr, int * error)
{
	struct ipmi_rq req;
	struct ipmi_rs * rsp;
	struct sdr_get_rq sdr_rq;
	uint8_t * data;
	int i = 0, len = header->length;

	if (len < 1)
		return NULL;

	data = malloc(len+1);
	if (data == NULL) {
		ipmi_printf("%s: malloc failure\n", __FUNCTION__);
		ipmi_set_error(error, IPMI_ERROR_NOT_ENOUGH_MEMORY);
		return NULL;
	}

	memset(data, 0, len+1);

	memset(&sdr_rq, 0, sizeof(sdr_rq));
	sdr_rq.reserve_id = cpu_to_le16(itr->reservation);
	sdr_rq.id = cpu_to_le16(header->id);
	sdr_rq.offset = 0;

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_STORAGE;
	req.msg.cmd = GET_SDR;
	req.msg.data = (uint8_t *)&sdr_rq;
	req.msg.data_len = sizeof(sdr_rq);

	/* read SDR record with partial reads
	 * because a full read usually exceeds the maximum
	 * transport buffer size.  (completion code 0xca)
	 */
	while (i < len) {
		sdr_rq.length = (len-i < sdr_max_read_len) ?
			len-i : sdr_max_read_len;
		sdr_rq.offset = i+5; /* 5 header bytes */
		if (verbose > 1)
			ipmi_printf("Getting %d bytes from SDR at offset %d\n",
			       sdr_rq.length, sdr_rq.offset);
		rsp = intf->sendrecv(intf, &req, error);
		if (rsp == NULL) {
			free (data);
			return NULL;
		}

		switch (rsp->ccode) {
		case 0xca:
			/* read too many bytes at once */
			if (sdr_max_read_len > 32)  
			    sdr_max_read_len = sdr_max_read_len >> 1;
			else if (sdr_max_read_len > 20)
			    sdr_max_read_len -= 4;
			else if (sdr_max_read_len > 10)
			    sdr_max_read_len--;
			else 
			    /* we do not want to handle transfers with less than
			       10 bytes, so bail out in this case */
			    break;  
			continue;
		case 0xc5:
			/* lost reservation */
			if (verbose > 1)
				ipmi_printf("SDR reserveration canceled. "
				       "Sleeping a bit and retrying...\n");

			sleep (rand () & 3);

			if (ipmi_sdr_get_reservation(intf, &(itr->reservation), error) < 0) {
				free (data);
				return NULL;
			}
			sdr_rq.reserve_id = cpu_to_le16(itr->reservation);
			continue;
		}

		/* special completion codes handled above */
		if (rsp->ccode > 0 || rsp->data_len == 0) {
			free(data);
			return NULL;
		}

		memcpy(data+i, rsp->data+2, sdr_rq.length);
		i += sdr_max_read_len;
	}

	return data;
}

/* ipmi_sdr_end  -  cleanup SDR iterator
 *
 * @intf:	ipmi interface
 * @itr:	SDR iterator
 *
 * no meaningful return code
 */
void
ipmi_sdr_end(struct ipmi_intf * intf UNUSED, struct ipmi_sdr_iterator * itr)
{
	if (itr)
		free(itr);
}

/* __sdr_list_add  -  helper function to add SDR record to list
 *
 * @head:	list head
 * @entry:	new entry to add to end of list
 *
 * returns 0 on success
 * returns -1 on error
 */
/* unused?! */
#if 0
static int
__sdr_list_add(struct sdr_record_list * head,
	       struct sdr_record_list * entry,
	       int * error)
{
	struct sdr_record_list * e;
	struct sdr_record_list * new;

	if (head == NULL)
		return -1;

	new = malloc(sizeof(struct sdr_record_list));
	if (new == NULL) {
		ipmi_printf("ipmitool: malloc failure\n");
		ipmi_set_error(error, IPMI_ERROR_NOT_ENOUGH_MEMORY);
		return -1;
	}
	memcpy(new, entry, sizeof(struct sdr_record_list));

	e = head;
	while (e->next)
		e = e->next;
	e->next = new;
	new->next = NULL;

	return 0;
}
#endif /* 0 */

/* __sdr_list_empty  -  low-level handler to clean up record list
 *
 * @head:	list head to clean
 *
 * no meaningful return code
 */
/* unused?! */
#if 0
static void
__sdr_list_empty(struct sdr_record_list * head)
{
	struct sdr_record_list * e, * f;
	for (e = head; e != NULL; e = f) {
		f = e->next;
		free(e);
	}
	head = NULL;
}
#endif /* 0 */

/* ipmi_sdr_list_empty  -  clean global SDR list
 *
 * @intf:	ipmi interface
 *
 * no meaningful return code
 */
void
ipmi_sdr_list_empty(struct ipmi_intf * intf)
{
	struct sdr_record_list *list, *next;

	ipmi_sdr_end(intf, sdr_list_itr);

	for (list = sdr_list_head; list != NULL; list = next) {
		switch (list->type) {
		case SDR_RECORD_TYPE_FULL_SENSOR:
			if (list->record.full)
				free(list->record.full);
			break;
		case SDR_RECORD_TYPE_COMPACT_SENSOR:
			if (list->record.compact)
				free(list->record.compact);
			break;
		case SDR_RECORD_TYPE_EVENTONLY_SENSOR:
			if (list->record.eventonly)
				free(list->record.eventonly);
			break;
		case SDR_RECORD_TYPE_GENERIC_DEVICE_LOCATOR:
			if (list->record.genloc)
				free(list->record.genloc);
			break;
		case SDR_RECORD_TYPE_FRU_DEVICE_LOCATOR:
			if (list->record.fruloc)
				free(list->record.fruloc);
			break;
		case SDR_RECORD_TYPE_MC_DEVICE_LOCATOR:
			if (list->record.mcloc)
				free(list->record.mcloc);
			break;
		case SDR_RECORD_TYPE_ENTITY_ASSOC:
			if (list->record.entassoc)
				free(list->record.entassoc);
			break;
		}
		next = list->next;
		free(list);
	}

	sdr_list_head = NULL;
	sdr_list_tail = NULL;
	sdr_list_itr = NULL;
}

/* ipmi_sdr_find_sdr_byid  -  lookup SDR entry by ID string
 *
 * @intf:	ipmi interface
 * @id:		string to match for sensor name
 *
 * returns pointer to SDR list
 * returns NULL on error
 */
struct sdr_record_list *
ipmi_sdr_find_sdr_byid(struct ipmi_intf * intf, char * id, int * error)
{
	struct sdr_get_rs * header;
	struct sdr_record_list * e;
	int found = 0;
	int idlen;

	if (id == NULL)
		return NULL;

	idlen = strlen(id);

	if (sdr_list_itr == NULL) {
		sdr_list_itr = ipmi_sdr_start(intf, error);
		if (sdr_list_itr == NULL) {
			ipmi_printf("Unable to open SDR for reading\n");
			return NULL;
		}
	}

	/* check what we've already read */
	for (e = sdr_list_head; e != NULL; e = e->next) {
		switch (e->type) {
		case SDR_RECORD_TYPE_FULL_SENSOR:
			if (!strncmp((char *)e->record.full->id_string, id,
				     __max(e->record.full->id_code & 0x1f, idlen)))
				return e;
			break;
		case SDR_RECORD_TYPE_COMPACT_SENSOR:
			if (!strncmp((char *)e->record.compact->id_string, id,
				     __max(e->record.compact->id_code & 0x1f, idlen)))
				return e;
			break;
		case SDR_RECORD_TYPE_EVENTONLY_SENSOR:
			if (!strncmp((char *)e->record.eventonly->id_string, id,
				     __max(e->record.eventonly->id_code & 0x1f, idlen)))
				return e;
			break;
		case SDR_RECORD_TYPE_GENERIC_DEVICE_LOCATOR:
			if (!strncmp((char *)e->record.genloc->id_string, id,
				     __max(e->record.genloc->id_code & 0x1f, idlen)))
				return e;
			break;
		case SDR_RECORD_TYPE_FRU_DEVICE_LOCATOR:
			if (!strncmp((char *)e->record.fruloc->id_string, id,
				     __max(e->record.fruloc->id_code & 0x1f, idlen)))
				return e;
			break;
		case SDR_RECORD_TYPE_MC_DEVICE_LOCATOR:
			if (!strncmp((char *)e->record.mcloc->id_string, id,
				     __max(e->record.mcloc->id_code & 0x1f, idlen)))
				return e;
			break;
		}
	}

	/* now keep looking */
	while ((header = ipmi_sdr_get_next_header(intf, sdr_list_itr, error)) != NULL) {
		uint8_t * rec;
		struct sdr_record_list * sdrr;

		sdrr = malloc(sizeof(struct sdr_record_list));
		if (sdrr == NULL) {
			ipmi_printf("%s: malloc failure\n", __FUNCTION__);
			ipmi_set_error(error, IPMI_ERROR_NOT_ENOUGH_MEMORY);
			break;
		}
		memset(sdrr, 0, sizeof(struct sdr_record_list));
		sdrr->id = header->id;
		sdrr->type = header->type;

		rec = ipmi_sdr_get_record(intf, header, sdr_list_itr, error);
		if (rec == NULL)
			continue;

		switch (header->type) {
		case SDR_RECORD_TYPE_FULL_SENSOR:
			sdrr->record.full = (struct sdr_record_full_sensor *)rec;
			if (!strncmp((char *)sdrr->record.full->id_string, id,
				     __max(sdrr->record.full->id_code & 0x1f, idlen)))
				found = 1;
			break;
		case SDR_RECORD_TYPE_COMPACT_SENSOR:
			sdrr->record.compact = (struct sdr_record_compact_sensor *)rec;
			if (!strncmp((char *)sdrr->record.compact->id_string, id,
				     __max(sdrr->record.compact->id_code & 0x1f, idlen)))
				found = 1;
			break;
		case SDR_RECORD_TYPE_EVENTONLY_SENSOR:
			sdrr->record.eventonly = (struct sdr_record_eventonly_sensor *)rec;
			if (!strncmp((char *)sdrr->record.eventonly->id_string, id,
				     __max(sdrr->record.eventonly->id_code & 0x1f, idlen)))
				found = 1;
			break;
		case SDR_RECORD_TYPE_GENERIC_DEVICE_LOCATOR:
			sdrr->record.genloc = (struct sdr_record_generic_locator *)rec;
			if (!strncmp((char *)sdrr->record.genloc->id_string, id,
				     __max(sdrr->record.genloc->id_code & 0x1f, idlen)))
				found = 1;
			break;
		case SDR_RECORD_TYPE_FRU_DEVICE_LOCATOR:
			sdrr->record.fruloc = (struct sdr_record_fru_locator *)rec;
			if (!strncmp((char *)sdrr->record.fruloc->id_string, id,
				     __max(sdrr->record.fruloc->id_code & 0x1f, idlen)))
				found = 1;
			break;
		case SDR_RECORD_TYPE_MC_DEVICE_LOCATOR:
			sdrr->record.mcloc = (struct sdr_record_mc_locator *)rec;
			if (!strncmp((char *)sdrr->record.mcloc->id_string, id,
				     __max(sdrr->record.mcloc->id_code & 0x1f, idlen)))
				found = 1;
			break;
		case SDR_RECORD_TYPE_ENTITY_ASSOC:
			sdrr->record.entassoc = (struct sdr_record_entity_assoc *)rec;
			break;
		default:
			free(rec);
			continue;
		}

		/* add to global record liset */
		if (sdr_list_head == NULL)
			sdr_list_head = sdrr;
		else
			sdr_list_tail->next = sdrr;

		sdr_list_tail = sdrr;

		if (found)
			return sdrr;
	}

	return NULL;
}

struct sdr_record_list *
ipmi_sdr_find_sdr_bysensorno(struct ipmi_intf * intf, int sensorno, int * error)
{
	struct sdr_get_rs * header;
	static struct sdr_record_list * e;
	int found = 0;

	if (!sdr_list_itr) {
		sdr_list_itr = ipmi_sdr_start(intf, error);
		if (!sdr_list_itr) {
			ipmi_printf("Unable to open SDR for reading\n");
			return NULL;
		}
	}

	/* check what we've already read */
	for (e = sdr_list_head; e != NULL; e = e->next) {
		switch (e->type) {
		case SDR_RECORD_TYPE_FULL_SENSOR:
			if (e->record.full->keys.sensor_num == sensorno) {
				return e;
			}
			break;
		case SDR_RECORD_TYPE_COMPACT_SENSOR:
			if (e->record.compact->keys.sensor_num == sensorno) {
				return e;
			}
			break;
		case SDR_RECORD_TYPE_EVENTONLY_SENSOR:
			if (e->record.eventonly->keys.sensor_num == sensorno) {
				return e;
			}
			break;
		}
	}

	/* now keep looking */
	while ((header = ipmi_sdr_get_next_header(intf, sdr_list_itr, error)) != NULL) {
		uint8_t * rec;
		struct sdr_record_list * sdrr;

		sdrr = malloc(sizeof(struct sdr_record_list));
		memset(sdrr, 0, sizeof(struct sdr_record_list));
		sdrr->id = header->id;
		sdrr->type = header->type;

		rec = ipmi_sdr_get_record(intf, header, sdr_list_itr, error);
		if (!rec)
			continue;

		switch (header->type) {
		case SDR_RECORD_TYPE_FULL_SENSOR:
			sdrr->record.full = (struct sdr_record_full_sensor *)rec;
			if (sdrr->record.full->keys.sensor_num == sensorno)
				found = 1;
			break;
		case SDR_RECORD_TYPE_COMPACT_SENSOR:
			sdrr->record.full = (struct sdr_record_full_sensor *)rec;
			if (sdrr->record.compact->keys.sensor_num == sensorno)
				found = 1;
			break;
		case SDR_RECORD_TYPE_EVENTONLY_SENSOR:
			sdrr->record.full = (struct sdr_record_full_sensor *)rec;
			if (sdrr->record.eventonly->keys.sensor_num == sensorno)
				found = 1;
			break;
		case SDR_RECORD_TYPE_FRU_DEVICE_LOCATOR:
			sdrr->record.fruloc = (struct sdr_record_fru_locator *)rec;
			break;
		case SDR_RECORD_TYPE_GENERIC_DEVICE_LOCATOR:
			sdrr->record.genloc = (struct sdr_record_generic_locator *)rec;
			break;
		case SDR_RECORD_TYPE_MC_DEVICE_LOCATOR:
			sdrr->record.mcloc = (struct sdr_record_mc_locator *)rec;
			break;
		case SDR_RECORD_TYPE_ENTITY_ASSOC:
			sdrr->record.entassoc = (struct sdr_record_entity_assoc *)rec;
			break;
		default:
			free(rec);
			continue;
		}

		if (!sdr_list_head)
			sdr_list_head = sdrr;
		else
			sdr_list_tail->next = sdrr;
		sdr_list_tail = sdrr;

		if (found)
			return sdrr;
	}

	return NULL;
}

const char * sdr_get_sensor_name(struct sdr_record_list *sdr) {
	if (!sdr) {
		ipmi_printf("No sensor given.\n");
		return _("Unknown");
	}
	switch (sdr->type) {
		case SDR_RECORD_TYPE_FULL_SENSOR:
			return (char *)sdr->record.full->id_string;
		case SDR_RECORD_TYPE_COMPACT_SENSOR:
			return (char *)sdr->record.compact->id_string;
		case SDR_RECORD_TYPE_EVENTONLY_SENSOR:
			return (char *)sdr->record.eventonly->id_string;
		case SDR_RECORD_TYPE_FRU_DEVICE_LOCATOR:
			return (char *)sdr->record.fruloc->id_string;
		case SDR_RECORD_TYPE_MC_DEVICE_LOCATOR:
			return (char *)sdr->record.mcloc->id_string;
	}
	return _("Unknown");
}

/*
 * ipmi_sdr_get_info
 *
 * Execute the GET SDR REPOSITORY INFO command, and populate the sdr_info
 * structure.
 * See section 33.9 of the IPMI v2 specification for details
 *
 * returns 0 on success
 *         -1 on transport error
 *         > 0 for other errors
 */
static  int
ipmi_sdr_get_info(struct ipmi_intf * intf,
		  struct get_sdr_repository_info_rsp * sdr_repository_info,
		  int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;

	memset(&req, 0, sizeof(req));

	req.msg.netfn    = IPMI_NETFN_STORAGE;           // 0x0A
	req.msg.cmd      = IPMI_GET_SDR_REPOSITORY_INFO; // 0x20
	req.msg.data     = 0;
	req.msg.data_len = 0;

	rsp = intf->sendrecv(intf, &req, error);

	if (rsp == NULL) {
		ipmi_printf("Get SDR Repository Info command failed\n");
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Get SDR Repository Info command failed: %s\n",
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	memcpy(sdr_repository_info,
		   rsp->data,
		   min(sizeof(struct get_sdr_repository_info_rsp), (size_t)rsp->data_len));

	return 0;
}

/*
 * ipmi_sdr_print_info
 *
 * Display the return data of the GET SDR REPOSITORY INFO command
 * See section 33.9 of the IPMI v2 specification for details
 *
 * returns 0 on success
 *         -1 on error
 */
static int
ipmi_sdr_print_info(struct ipmi_intf * intf, pp_ipmi_return_t *ipmi_ret, int * error)
{
	uint32_t timestamp;
	uint16_t free_space;
	pp_ipmi_sdr_info_t *info;

	struct get_sdr_repository_info_rsp sdr_repository_info;

	if (ipmi_init_sdr_info(ipmi_ret)) {
		ipmi_printf("Error: Could not allocate return structure.\n");
		return -1;
	}

	info = &ipmi_ret->data.sdr_info;

	if (ipmi_sdr_get_info(intf, &sdr_repository_info, error) != 0) {
		return -1;
	}

	info->version = sdr_repository_info.sdr_version;
	info->record_count = (sdr_repository_info.record_count_msb << 8) |
		   sdr_repository_info.record_count_lsb;
	
	free_space =
		(sdr_repository_info.free_space[1] << 8) |
		sdr_repository_info.free_space[0];
	
	switch (free_space)
	{
	case 0x0000:
		pp_strappend(&info->free_space.string, _("none (full)"));
		break;
	case 0xFFFF:
		pp_strappend(&info->free_space.string, _("unspecified"));
		break;
	case 0xFFFE:
		pp_strappend(&info->free_space.string, _("> 64Kb - 2 bytes"));
		break;
	default:
		pp_strappendf(&info->free_space.string, _("%d bytes"), free_space);
		break;
	}
	info->free_space.space = free_space;

	timestamp =
		(sdr_repository_info.most_recent_addition_timestamp[3] << 24) |
		(sdr_repository_info.most_recent_addition_timestamp[2] << 16) |
		(sdr_repository_info.most_recent_addition_timestamp[1] << 8) |
		sdr_repository_info.most_recent_addition_timestamp[0];
	ipmi_set_timestamp(timestamp, &info->most_resent_addition);

	timestamp =
		(sdr_repository_info.most_recent_erase_timestamp[3] << 24) |
		(sdr_repository_info.most_recent_erase_timestamp[2] << 16) |
		(sdr_repository_info.most_recent_erase_timestamp[1] << 8) |
		sdr_repository_info.most_recent_erase_timestamp[0];
	ipmi_set_timestamp(timestamp, &info->most_resent_erase);

	info->overflow = sdr_repository_info.overflow_flag? 1: 0;

	switch (sdr_repository_info.modal_update_support)
	{
	case 0:
		pp_strappend(&info->modal_update_support, _("unspecified"));
		break;
	case 1:
		pp_strappend(&info->modal_update_support, _("non-modal"));
		break;
	case 2:
		pp_strappend(&info->modal_update_support, _("modal"));
		break;
	case 3:
		pp_strappend(&info->modal_update_support, _("modal and non-modal"));
		break;
	default:
		pp_strappend(&info->modal_update_support, _("error in response"));
		break;
	}

	info->delete_supported = sdr_repository_info.delete_sdr_supported ? 1 : 0;
	info->partial_add_supported = sdr_repository_info.partial_add_sdr_supported ? 1 : 0;
	info->reserve_supported = sdr_repository_info.reserve_sdr_repository_supported ? 1 : 0;
	info->get_alloc_info_supported = sdr_repository_info.get_sdr_repository_allo_info_supported ? 1 : 0;

	if (sdr_repository_info.get_sdr_repository_allo_info_supported) {
		uint16_t unit_size, alloc_units;
		struct ipmi_rs * rsp;
		struct ipmi_rq req;
		
		/* get sel allocation info */
		memset(&req, 0, sizeof(req));
		req.msg.netfn = IPMI_NETFN_STORAGE;
		req.msg.cmd = GET_SDR_ALLOC_INFO;

		rsp = intf->sendrecv(intf, &req, error);
		if (!rsp || rsp->ccode) {
			ipmi_printf("error %d in Get SDR Allocation Info command\n",
			       rsp ? rsp->ccode : 0);
			return -1;
		}

		alloc_units = buf2short(rsp->data);
		if (alloc_units) {
			info->allocation_info.alloc_units.specified = 1;
			info->allocation_info.alloc_units.units = alloc_units;
		} else {
			info->allocation_info.alloc_units.specified = 0;
		}
		unit_size = buf2short(rsp->data + 2);
		if (unit_size) {
			info->allocation_info.alloc_unit_size.specified = 1;
			info->allocation_info.alloc_unit_size.unit_size = unit_size;
		} else {
			info->allocation_info.alloc_unit_size.specified = 0;
		}
		info->allocation_info.free_units = buf2short(rsp->data + 4);
		info->allocation_info.largest_free_block = buf2short(rsp->data + 6);
		info->allocation_info.max_record_size =rsp->data[7];
	}

	return 0;
}

int ipmi_sdr_main(struct ipmi_intf * intf, int subcmd, pp_ipmi_parameter_t *params, pp_ipmi_return_t *ret, int * error)
{
	uint8_t sdr_list_type = 0xff;
	srand ((unsigned int)time (NULL));

	switch ((pp_ipmi_sdr_subcommand_t) subcmd) {
		case PP_IPMI_SDR_SUBCMD_INFO:
			return ipmi_sdr_print_info(intf, ret, error);
		case PP_IPMI_SDR_SUBCMD_LIST:
			if (params && !params->is_cmdline) {
				if (params->data.sdr_list == PP_IPMI_SDR_LIST_ALL) {
					sdr_list_type = 0xff;
				} else if (params->data.sdr_list == PP_IPMI_SDR_LIST_FULL) {
					sdr_list_type = SDR_RECORD_TYPE_FULL_SENSOR;
				} else if (params->data.sdr_list == PP_IPMI_SDR_LIST_COMPACT) {
					sdr_list_type = SDR_RECORD_TYPE_COMPACT_SENSOR;
				} else if (params->data.sdr_list == PP_IPMI_SDR_LIST_EVENT) {
					sdr_list_type = SDR_RECORD_TYPE_EVENTONLY_SENSOR;
				} else if (params->data.sdr_list == PP_IPMI_SDR_LIST_MCLOC) {
					sdr_list_type = SDR_RECORD_TYPE_MC_DEVICE_LOCATOR;
				} else if (params->data.sdr_list == PP_IPMI_SDR_LIST_FRU) {
					sdr_list_type = SDR_RECORD_TYPE_FRU_DEVICE_LOCATOR;
				}
			}
			return ipmi_sdr_print_sdr(intf, sdr_list_type, ret, error);
		case PP_IPMI_SDR_SUBCMD_UPDATE:
			if(params && !params->is_cmdline)
			{
			    return ipmi_sdr_update_sdr(intf,params->data.sdr_idx,ret,error);
			}
			return -1;
		default:
			ipmi_printf("Invalid SDR command: %d\n", subcmd);
			ipmi_set_error(error, IPMI_ERROR_INVALID_COMMAND);
			return -1;
	}
	return 0;
}

