/*
 * Copyright (c) 2003 Sun Microsystems, Inc.  All Rights Reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * You acknowledge that this software is not designed or intended for use
 * in the design, construction, operation or maintenance of any nuclear
 * facility.
 */

#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <linux/i2c.h>
#include <pp/base.h>

#include <config.h>
#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>

#include <pp/base.h>

#define IPMI_FILE	"/dev/i2c-ipmi"

/* minimum time between two requests being sent - given in microseconds */
#define MIN_REQUEST_TIMEOUT 70000

static int curr_seq = 0;

static int ipmi_i2c_fd = -1;
static u_char current_bmc = 0;

static struct ipmi_rq_entry *
ipmi_req_add_entry(struct ipmi_intf * intf, struct ipmi_rq * req, int * error)
{
	struct ipmi_rq_entry * e = malloc(sizeof(struct ipmi_rq_entry));

	if (e == NULL) {
		ipmi_set_error(error, IPMI_ERROR_NOT_ENOUGH_MEMORY);
		ipmi_printf("WARNING: no memory!\n");
	}
	else {
		memset(e, 0, sizeof(struct ipmi_rq_entry));
		memcpy(&e->req, req, sizeof(struct ipmi_rq));

		e->intf = intf;
	}
	return e;
}

static struct ipmi_rq_entry *
ipmi_i2c_build_cmd(struct ipmi_intf * intf, struct ipmi_rq * req, int * error)
{
	uint8_t * msg;
	int cs, tmp;
	int len = 0;
	struct ipmi_rq_entry * entry;
	struct ipmi_session * s = intf->session;

	if (!intf->session) {
		return NULL;
	}

	if (curr_seq >= 64)
		curr_seq = 0;

	entry = ipmi_req_add_entry(intf, req, error);
	if (!entry)
		return NULL;
	
	len = req->msg.data_len + 7;
	msg = malloc(len);
	memset(msg, 0, len);

	/* ipmi message header */
	len = 0;
	cs = len;

	msg[len++] = intf->target_addr;
	msg[len++] = req->msg.netfn << 2;
	tmp = len - cs;
	msg[len++] = ipmi_csum(msg+cs, tmp);
	cs = len;
	msg[len++] = intf->my_addr ? intf->my_addr : IPMI_REMOTE_SWID;

	entry->rq_seq = curr_seq++;
	msg[len++] = entry->rq_seq << 2;
	msg[len++] = req->msg.cmd;

	/* message data */
	if (req->msg.data_len) {
 		memcpy(msg+len, req->msg.data, req->msg.data_len);
		len += req->msg.data_len;
	}

	/* second checksum */
	tmp = len - cs;
	msg[len++] = ipmi_csum(msg+cs, tmp);

	if (s->in_seq) {
		s->in_seq++;
		if (!s->in_seq)
			s->in_seq++;
	}

	entry->msg_len = len;
	entry->msg_data = msg;

	return entry;
}

static void
ipmi_i2c_check_slave_address(u_char bmc)
{
	if (bmc != current_bmc) {
		if (ioctl(ipmi_i2c_fd, I2C_SLAVE_FORCE, bmc >> 1) < 0) {
			ipmi_printf("%s(): Setting I2C slave address failed\n", ___F);
			return;
		}

		current_bmc = bmc;
	}
}

static struct ipmi_rs* ipmi_i2c_build_response(char *data, int len) {
	int x = 0;
	static struct ipmi_rs rsp;

	if (len < 7) {
		return NULL;
	}

	memset(&rsp, 0, sizeof(rsp));	
	rsp.payload.ipmi_response.netfn   = data[x] >> 2;
	rsp.payload.ipmi_response.rq_lun  = data[x++] & 0x3;
	x++;		/* checksum */
	rsp.payload.ipmi_response.rs_addr = data[x++];
	rsp.payload.ipmi_response.rq_seq  = data[x] >> 2;
	rsp.payload.ipmi_response.rs_lun  = data[x++] & 0x3;
	rsp.payload.ipmi_response.cmd     = data[x++]; 
	rsp.ccode                         = data[x++];
	
	rsp.data_len = len - 7;
	if (!rsp.ccode && rsp.data_len) {
		memcpy(&rsp.data, &data[x], rsp.data_len);
	}

#ifdef PP_IPMI_DEBUG
	printf("CC: %d\n", rsp.ccode);
	{
		int i;
		D("rsp.data: ");
		for (i = 0; i < rsp.data_len; i++) {
			printf("%02x ", rsp.data[i]);
		}
		printf("\n");
	}
#endif // PP_IPMI_DEBUG

	/* checksum follows */

	return &rsp;
}

static struct ipmi_rs * ipmi_i2c_send_cmd(struct ipmi_intf * intf, struct ipmi_rq * req, int * error)
{
	static struct timeval last_req_time = {.tv_sec = 0, .tv_usec = 0};
	unsigned int time_elapsed;
	struct ipmi_rq_entry * entry;
	uint8_t recv_message[255];
	int ret = 0;
	struct timeval to;
	fd_set i2cset;
	int have_response = 0;
	int retries = 3;

	if (!intf->opened && intf->open && intf->open(intf, error) < 0)
		return NULL;

	/* to fulfill IPMB spec section 4, we have to make
	   sure that at least 60ms elapsed since the last 
	   request */
	gettimeofday(&to, NULL);
	time_elapsed = ((to.tv_sec*1000000)+to.tv_usec)-((last_req_time.tv_sec*1000000)+last_req_time.tv_usec);

	if (time_elapsed < MIN_REQUEST_TIMEOUT) {
	    //less than 60ms elapsed
	    usleep(MIN_REQUEST_TIMEOUT-time_elapsed);
	}
	
	/* store the current time as request send time */
	gettimeofday(&last_req_time, NULL);

	/* generate the request message */
	entry = ipmi_i2c_build_cmd(intf, req, error);
	if (!entry) {
		ipmi_set_error(error, IPMI_ERROR_INTERNAL);
		ipmi_printf("Aborting send command, unable to build.\n");
		return NULL;
	}

#ifdef PP_IPMI_DEBUG
	{
		int i;
		D("Transmit: ");
		for (i = 0; i < entry->msg_len; i++) {
			printf("%02x ", entry->msg_data[i]);
		}
		printf("\n");
	}
#endif // PP_IPMI_DEBUG

	/* send the data via i2c */

	/* clear buffer, in case there is a previous message */
	if (ioctl(ipmi_i2c_fd, I2C_SLAVE_FLUSH_BUFFER, 0) < 0) {
		ipmi_printf("%s(): flushing buffer failed\n", ___F);
	}
	
	/* set slave address */
	ipmi_i2c_check_slave_address(intf->target_addr);

	while ((retries > 0) && (!have_response)) {
	    //printf("writing to i2c, retries left: %i\n", retries);
    	    /* write the data */
    	    if (write(ipmi_i2c_fd, &entry->msg_data[1], entry->msg_len - 1) != (entry->msg_len - 1)) {
		ipmi_printf("Error: could not write data to i2c.\n");
		ipmi_set_error(error, IPMI_ERROR_SEND_FAILED);
#ifdef OEM_IWILL        
        usleep(5000);
#endif /* OE_IWILL */        
		goto bail;
	    }

    	    /* wait for the answer and read it */
    	    to.tv_sec = 5;	/* wait 5 seconds */
    	    to.tv_usec = 0;
    	    FD_ZERO(&i2cset);
    	    FD_SET(ipmi_i2c_fd, &i2cset);

    	    if((select(ipmi_i2c_fd + 1, &i2cset, NULL, NULL, &to) >= 0) &&
		    FD_ISSET(ipmi_i2c_fd, &i2cset)) {
	    	have_response = 1;
	    	ret = read(ipmi_i2c_fd, recv_message, sizeof(recv_message));
	    } else {
		ipmi_set_error(error, IPMI_ERROR_NO_RESPONSE);
		ipmi_printf("Error waiting for i2c response.\n");
	    }
	    retries--;
	}

#ifdef PP_IPMI_DEBUG
	if (have_response) {
		int i;
		D("Receive: ");
		for (i = 0; i < ret; i++) {
			printf("%02x ", recv_message[i]);
		}
		printf("\n");
	}
#endif // PP_IPMI_DEBUG

 bail:
	free(entry->msg_data);
	free(entry);
#ifdef OEM_IWILL    
    usleep(2000);
#endif /* OEM_IWILL */    
	return have_response ? ipmi_i2c_build_response(recv_message, ret) : NULL;
}

static int ipmi_i2c_setup(struct ipmi_intf * intf, int * perr UNUSED)
{
        intf->cancel = 0;
        intf->session = malloc(sizeof(struct ipmi_session));
        memset(intf->session, 0, sizeof(struct ipmi_session));
        return (intf->session) ? 0 : -1;
}

static int ipmi_i2c_open(struct ipmi_intf * intf, int * perr)
{
	if (ipmi_i2c_fd == -1) {
		ipmi_i2c_fd = open(IPMI_FILE, O_RDWR);
	}
	intf->fd = ipmi_i2c_fd;
	
	if (intf->fd == -1) {
		ipmi_printf("Could not open %s.\n", IPMI_FILE);
		ipmi_set_error(perr, IPMI_ERROR_NO_CONNECTION);
		return -1;
	}

	intf->opened = 1;
	curr_seq = 0;
	return 0;
}

static void ipmi_i2c_close(struct ipmi_intf * intf UNUSED, int force UNUSED, int * perr UNUSED)
{
}

struct ipmi_intf ipmi_i2c = {
	name:		"i2c",
	desc:		"Peppercon IPMI I2C adapter",
	setup:		ipmi_i2c_setup,
	open:		ipmi_i2c_open,
	close:		ipmi_i2c_close,
	sendrecv:	ipmi_i2c_send_cmd,
	bmc_addr:	IPMI_BMC_SLAVE_ADDR,
	target_addr:	IPMI_BMC_SLAVE_ADDR,
};



