/******************************************************************************\
* cat.c                                                                        *
*                                                                              *
* CURSOR ACTIVITY TRACKING                                                     *
*                                                                              *
* Intelligent online mouse cursor detection and tracking                       *
*                                                                              *
* Copyright 2005 Peppercon AG                                                  *
* Thomas Weber tweb@peppercon.de                                               *
\******************************************************************************/

#include <pp/base.h>
#include <pp/km.h>
#include <pp/vsc.h>
#include <liberic_pthread.h>

#include "cat_debug.h"
#include "cat_internal.h"
#include "iipptr_bitmap.h"
#include "iipptr_internal.h"
#include "iipptr_cursor.h"

//#define CAT_PTR_RANDOM  ((random() & 0x7) - 4)
#define CAT_PTR_RANDOM  ((random() & 0x2) - 1)
#define CAT_BUF_SZ      100

/*
static void cat_queue_enqueue_diffmap_event(void);
static void cat_queue_enqueue_ptr_pos_event(void);
*/

static int mouse_chase = 0;
#if !defined(CAT_DISABLE_TRACKING)
static pthread_t mouse_chase_thr;
static void* mouse_chase_thr_func(void* arg);
static pthread_mutex_t mouse_chase_mtx = PTHREAD_MUTEX_INITIALIZER;

#if defined(CAT_GLOBAL_LOCALIZATION_THR)    
static int mouse_locate = 0;
static pthread_t mouse_locate_thr;
static void* mouse_locate_thr_func(void* arg);
#endif /* CAT_GLOBAL_LOCALIZATION_THR */
#endif /* !CAT_DISABLE_TRACKING */

static pthread_mutex_t pos_estim_mtx = PTHREAD_MUTEX_INITIALIZER;

int pp_cat_init(driver_t *obj) {
#if !defined(CAT_DISABLE_TRACKING)
    data_imouse_t *imouse_data;
    
    assert(obj);
    imouse_data = (data_imouse_t*)obj->data_conv_mouse.data;
    assert(imouse_data);

    /* TODO, buf sizes */
    if(imouse_data->diffmap_queue || 
//       mouse_data->ptr_pos_queue ||
       imouse_data->ptr_move_queue ||
#if defined(CAT_ADAPTIVE_LOOKUP_TABLE)
       ((data_cat_t*)imouse_data)->ll_move_queue || 
       PP_ERR == cat_queue_init(&((data_cat_t*)imouse_data)->ll_move_queue,
                                1000, NULL, "low level pointer moves") ||
#endif /* CAT_ADAPTIVE_LOOKUP_TABLE */
       PP_ERR == cat_queue_init(&imouse_data->diffmap_queue, CAT_BUF_SZ,
                                NULL /* cat_queue_enqueue_diffmap_event */,
                                "diffmaps") ||
//       PP_ERR == cat_queue_init(&cat_ptr_pos_queue, CAT_BUF_SZ,
//                                NULL /* cat_queue_enqueue_ptr_pos_event */) ||
       PP_ERR == cat_queue_init(&imouse_data->ptr_move_queue, CAT_BUF_SZ,
                                NULL /* cat_queue_enqueue_ptr_move_event */,
                                "relative moves")) {
        pp_log("cat_init: can't initialize queue\n");
        goto error;
    }

    CDMSG(CAT_CDMSG_INFO, "CAT initialized\n");
    
    return PP_SUC;
    
 error:
    abort();
    cat_cleanup(obj);
    return PP_ERR;
#endif /* CAT_DISABLE_TRACKING */
    (void)obj;
    return PP_SUC;
}

void cat_cleanup(driver_t *obj) {
    data_imouse_t *imouse_data;

    assert(obj);
    imouse_data = (data_imouse_t*)obj->data_conv_mouse.data;
    assert(imouse_data);

    cat_queue_free(imouse_data->diffmap_queue);

//    cat_queue_free(mouse_data->ptr_pos_queue);

    cat_queue_free(imouse_data->ptr_move_queue);

#if defined(CAT_ADAPTIVE_LOOKUP_TABLE)
    cat_queue_free(((data_cat_t*)imouse_data)->ll_move_queue);
#endif /* CAT_ADAPTIVE_LOOKUP_TABLE */
}

int cat_suspend(const driver_t *obj ) {
    if(mouse_chase) {
        mouse_chase = -1;

        while(mouse_chase); 
        /* wait for mouse chase thread to finish (re-) sampling */
    }
    
    pp_grab_unregister_diffmap_consumer();
    
    return PP_SUC;
}

int cat_resume(const driver_t *obj) {
    pp_grab_register_diffmap_consumer((pp_grab_diffmap_consumer_t)
                                      { (void*)obj, cat_enqueue_diffmap });

    mouse_chase = 1;
    if(eric_pthread_create(&mouse_chase_thr, 1, 128 * 1024,
                           mouse_chase_thr_func, (void*)obj)){
        pp_log("cat_init: can't create mouse chase thread\n");
        goto error;
    }

#if defined(CAT_GLOBAL_LOCALIZATION_THR)
    if(eric_pthread_create(&mouse_locate_thr, 1, 128 * 1024,
                           mouse_locate_thr_func, (void*)obj)){
        pp_log("cat_init: can't create mouse locate thread\n");
        goto error;
    }
#endif /* CAT_GLOBAL_LOCALIZATION_THR */
   
    return PP_SUC;
    
 error:
    mouse_chase = -1;
    return PP_ERR;
}

#if 0
void cat_enqueue_ptr_pos(const driver_t *obj, u_int16_t x, u_int16_t y) {
    if(!mouse_chase) {
        /* if CAT is not active, ignore */
        return;
    }
    
//    CDMSG(1, "got mouse event x: %d, y: %d\n", x, y);
    
    cat_queue_lock(cat_ptr_pos_queue);
    if(PP_ERR == cat_queue_enqueue_ptr_pos(cat_ptr_pos_queue, x, y)) {
//        CDMSG(1, "failed to append pointer position to queue\n");
    }
    cat_queue_unlock(cat_ptr_pos_queue);
}
#endif

void cat_enqueue_ptr_move(const void *that, int16_t x, int16_t y) {
    driver_t *obj = (driver_t*)that;
    data_imouse_t *imouse_data;
    
    assert(obj);
    imouse_data = (data_imouse_t*)obj->data_conv_mouse.data;
    assert(imouse_data);

    if(!mouse_chase) {
        /* if CAT is not active, ignore */
        return;
    }
    
    cat_queue_lock(imouse_data->ptr_move_queue);
    if(PP_ERR == cat_queue_enqueue_ptr_move(imouse_data->ptr_move_queue, x, y)){
//        CDMSG(1, "failed to append pointer movement to queue\n");
    }
    cat_queue_unlock(imouse_data->ptr_move_queue);
}

void cat_enqueue_diffmap(const void *that, u_int32_t *diffmap) {
#warning TODO: figure out wether the condensed diffmap makes sense
#if 1 /* condensed diffmap */
    u_int y, offset, buf_offset, line_bytes, fb_line_len, buf_line_len,
          diffmap_sz;
    fb_format_info_t fb_info;
    u_int32_t *buf;
    data_imouse_t *imouse_data;
    driver_t *obj = (driver_t*)that;
    
    assert(obj);
    imouse_data = (data_imouse_t*)obj->data_conv_mouse.data;
    assert(imouse_data);

    if(!mouse_chase) {
        /* if CAT is not active, ignore */
        return;
    }
    
    if(pp_vsc_get_fb_format(0, &fb_info) == PP_ERR) {
        CDMSG(CAT_CDMSG_WARNING, "No framebuffer info yet\n");
        return;
    }

//    CDMSG(1, "got diffmap\n");

#if defined(CAT_DEBUG) && 0 // print out diffmap
    cat_debug_print_diffmap(diffmap, VSC_DIFFMAP_SIZE, PP_FB_TILES_PER_LINE,
                            fb_info.tiles_w, fb_info.tiles_h);
#endif

    buf_line_len = OHDIV(fb_info.tiles_w, 32);
    line_bytes = buf_line_len * 4;
    fb_line_len = OHDIV(PP_FB_TILES_PER_LINE, 32);
    diffmap_sz = fb_info.tiles_h * line_bytes;
    buf = (u_int32_t*)malloc(diffmap_sz);
    for(y = 0; y < fb_info.tiles_h; ++y) {
        offset = y * fb_line_len;
        assert(offset * 4 + line_bytes <= VSC_DIFFMAP_SIZE);
        
        buf_offset = y * buf_line_len;
//printf("tweb: line_bytes = %d(%d), buf_line_len = %d(%d), fb_line_len = %d(%d), buf_offset = %d, diffmap_sz = %d, y = %d / %d\n", line_bytes, fb_info.tiles_w / 8, buf_line_len, fb_info.tiles_w / 32, fb_line_len, PP_FB_TILES_PER_LINE / 32, buf_offset, diffmap_sz, y, fb_info.tiles_h);usleep(22222);
        assert(buf_offset * 4 + line_bytes <= diffmap_sz);

        memcpy(&buf[buf_offset], &diffmap[offset], line_bytes);
    }

#if defined(CAT_DEBUG) && 0// print out buffered diffmap
    cat_debug_print_diffmap(buf, diffmap_sz, fb_info.tiles_w, 
                            fb_info.tiles_w, fb_info.tiles_h);
#endif

    cat_queue_lock(imouse_data->diffmap_queue);
    if(PP_ERR == cat_queue_enqueue_diffmap(imouse_data->diffmap_queue, buf, 
                                           0 /* don't copy */, line_bytes * 8,
                                           fb_info.tiles_w, fb_info.tiles_h)) {
//        CDMSG(1, "failed to append diffmap to queue\n");
    }
    cat_queue_unlock(imouse_data->diffmap_queue);
#else /* generic diffmap */
    fb_format_info_t fb_info;
        
    if(!mouse_chase) {
        /* if CAT is not active, ignore */
        return;
    }
    
    if(pp_vsc_get_fb_format(0, &fb_info) == PP_ERR) {
        CDMSG(CAT_CDMSG_WARNING, "No framebuffer info yet\n");
        return;
    }

//    CDMSG(1, "got diffmap\n");

#if defined(CAT_DEBUG) && 0 // print out diffmap
    cat_debug_print_diffmap(diffmap, VSC_DIFFMAP_SIZE, PP_FB_TILES_PER_LINE,
                            fb_info.tiles_w, fb_info.tiles_h);
#endif

    cat_queue_lock(imouse_data->diffmap_queue);
    if(PP_ERR == cat_queue_enqueue_diffmap(imouse_data->diffmap_queue, diffmap, 
                                           VSC_DIFFMAP_SIZE /* copy diffmap */,
                                           PP_FB_TILES_PER_LINE,
                                           fb_info.tiles_w, fb_info.tiles_h)) {
//        CDMSG(1, "failed to append diffmap to queue\n");
    }
    cat_queue_unlock(imouse_data->diffmap_queue);
#endif /* generic diffmap */
}

#if defined(CAT_ADAPTIVE_LOOKUP_TABLE)
void cat_enqueue_ll_move(const void *that, long moved_x, long moved_y,
                         long ticks_x, long ticks_y) {
    driver_t *obj = (driver_t*)that;

    assert(obj);
    imouse_data = (data_imouse_t*)obj->data_conv_mouse.data;
    assert(imouse_data);

    if(!mouse_chase) {
        /* if CAT is not active, ignore */
        return;
    }
    
    cat_queue_lock(cat_ll_move_queue);
    if(PP_ERR == cat_queue_enqueue_ll_move(cat_ll_move_queue, moved_x, moved_y,
                                           ticks_x, ticks_y)) {
        CDMSG(1, "failed to append ticks to queue\n");
    }
    cat_queue_unlock(cat_ll_move_queue);
}
#endif /* CAT_ADAPTIVE_LOOKUP_TABLE */

#if 0
static void cat_queue_enqueue_diffmap_event(void) {
    /* TODO! mutex cond: if(!cat_ptr_pos_queue_empty) cond signal bla */
    cat_diffmap_queue_empty = 0;
}

static void cat_queue_enqueue_ptr_pos_event(void) {
    /* TODO! mutex cond: if(!cat_diffmap_queue_empty) cond signal bla */
    cat_ptr_pos_queue_empty = 0;
}
#endif

#if !defined(CAT_DISABLE_TRACKING)
static void* mouse_chase_thr_func(void* arg) {
    driver_t *obj = (driver_t*)arg;
    data_imouse_t *imouse_data;
//    struct timeval now;
//    long tdiff;
    u_char diffmap_queue_empty = 1, 
//           cat_ptr_pos_queue_empty = 1,
           ptr_move_queue_empty = 1;
#if defined(CAT_GLOBAL_LOCALIZATION_THR)    
    pp_stopwatch_t mouse_locate_timeout;
#endif /* CAT_GLOBAL_LOCALIZATION_THR */    
//    pp_stopwatch_t profiler;
    
    assert(obj);
    imouse_data = (data_imouse_t*)obj->data_conv_mouse.data;
    assert(imouse_data);

    CDMSG(CAT_CDMSG_BLABLA, "may the chase begin!\n");
    
    while(mouse_chase > 0) {
//        pp_hrtime_start(&profiler);
        
        while(mouse_chase > 0) {
            if(PP_ERR != cat_queue_trylock(imouse_data->diffmap_queue)) {
                if(PP_ERR != cat_queue_trylock(imouse_data->ptr_move_queue)) {
                    diffmap_queue_empty =
                        cat_queue_is_empty(imouse_data->diffmap_queue);
                    ptr_move_queue_empty =
                        cat_queue_is_empty(imouse_data->ptr_move_queue);
                    cat_queue_unlock(imouse_data->diffmap_queue);
                    cat_queue_unlock(imouse_data->ptr_move_queue);
                    break;
                }
//else { printf("tweb: initial search, moveq already locked\n"); }
                cat_queue_unlock(imouse_data->diffmap_queue);
            }
//else { printf("tweb: initial search, diffmapq already locked\n"); }
            usleep(25000);
        }
//        cat_queue_lock(cat_ptr_pos_queue);
//        cat_ptr_pos_queue_empty = cat_queue_is_empty(cat_ptr_pos_queue);
//        cat_queue_unlock(cat_ptr_pos_queue);

#if defined(CAT_GLOBAL_LOCALIZATION_THR)    
        pp_hrtime_start(&mouse_locate_timeout);
#endif /* CAT_GLOBAL_LOCALIZATION_THR */    

        /* wait until something is enqueued */
        /* TODO: perhaps we need a timeout... 
                 e.g. shake the mouse if no remote console is connected... */
        while(mouse_chase > 0 &&
              (diffmap_queue_empty || 
//              cat_ptr_pos_queue_empty ||
               ptr_move_queue_empty)) {
#if defined(CAT_GLOBAL_LOCALIZATION_THR)    
            pp_hrtime_stop(&mouse_locate_timeout);
            if(pp_hrtime_read(&mouse_locate_timeout) >
               CAT_MOUSE_LOCATE_TIMEOUT_NS) {
                /* reset timeout and trigger global mouse localisation */
                mouse_locate = 1;
                pp_hrtime_start(&mouse_locate_timeout);
            }
#endif /* CAT_GLOBAL_LOCALIZATION_THR */    
         
#if defined(PP_FEAT_CAT)
#if defined(CAT_DEBUG)
            {
                /**
                 * shake the mouse
                 * this is a debug feature
                 * it breaks the disabling mechanisms, cat_cleanup will fail!
                 */
                u_int16_t abs_x = mouse_data->rc_abs.x;
                u_int16_t abs_y = mouse_data->rc_abs.y;
#if CAT_SHAKE // shake
                int16_t randx, randy;
                fb_format_info_t fb_info;
                
                if(pp_vsc_get_fb_format(0, &fb_info) == PP_ERR) {
                    CDMSG(CAT_CDMSG_WARNING, "No framebuffer info yet\n");
                    usleep(250000);
                    break;
                }
#if 0 // timeout
                gettimeofday(&now, NULL);
                /* calc diff in milliseconds */
                tdiff = (now.tv_sec - last_time.tv_sec) * 1000 +
                        (now.tv_usec - last_time.tv_usec) / 1000;
                if(tdiff > PP_KM_CAT_WAIT)
#endif // timeout
                {
//                    CDMSG(1, "time diff is %ld, shaking the mouse\n", tdiff);
                    randx = abs_x + CAT_PTR_RANDOM; 
                    randy = abs_y + CAT_PTR_RANDOM;
//printf("tweb: randx = %d (%d), randy = %d (%d)\n", randx, SET_WITHIN(randx, 0, (int)fb_info.g_w - 1), randy, SET_WITHIN(randy, 0, (int)fb_info.g_h - 1));
#warning do not use internal call... but exernal deadlocks on cleanup...
                    pp_km_send_ptrmove(SET_WITHIN(randx, 0, 
                                                  (int)fb_info.g_w - 1),
                                       SET_WITHIN(randy, 0, 
                                                  (int)fb_info.g_h - 1),
                                       0, 0, PP_KM_PTR_MOVE_ABSOLUTE);
                }
#else // shake
if(0)
                {
                    u_int16_t e_x, e_y;
                    
                    if(PP_ERR != cat_get_position_estimate(obj, &e_x, &e_y) &&
                       (abs(e_x - abs_x) + abs(e_y - abs_y)) >=
                           MIN_PTR_DEV_4_APPROACH) {
                        pp_km_send_ptrmove(abs_x, abs_y, 0, 0,
                                           PP_KM_PTR_MOVE_ABSOLUTE);
                    }
//printf("tweb: abs_x = %d, e_x = %d, abs_y = %d, e_y = %d\n", abs_x, e_x, abs_y, e_y);
                }
#endif // shake
            }
#endif /* CAT_DEBUG */
#endif /* PP_FEAT_CAT */

            /* wait a bit and try again... */
            usleep(25000);
    
            while(mouse_chase > 0) {
                if(PP_ERR != cat_queue_trylock(imouse_data->diffmap_queue)) {
                    if(PP_ERR != 
                       cat_queue_trylock(imouse_data->ptr_move_queue)) {
                        diffmap_queue_empty =
                            cat_queue_is_empty(imouse_data->diffmap_queue);
                        ptr_move_queue_empty =
                            cat_queue_is_empty(imouse_data->ptr_move_queue);
                        cat_queue_unlock(imouse_data->diffmap_queue);
                        cat_queue_unlock(imouse_data->ptr_move_queue);
                        break;
                    }
//else { printf("tweb: 2nd search, moveq already locked\n"); }
                    cat_queue_unlock(imouse_data->diffmap_queue);
                }
//else { printf("tweb: 2nd search, diffmapq already locked\n"); }
                usleep(25000);
            }
//        cat_queue_lock(cat_ptr_pos_queue);
//        cat_ptr_pos_queue_empty = cat_queue_is_empty(cat_ptr_pos_queue);
//        cat_queue_unlock(cat_ptr_pos_queue);
        }

        if(mouse_chase > 0 && pthread_mutex_trylock(&mouse_chase_mtx) == 0) {
            CDMSG(CAT_CDMSG_MTX, "mouse_chase_mtx locked\n");
            assert(imouse_data->tr_sample);
            assert(imouse_data->tr_resample);
            imouse_data->tr_sample(obj);
            imouse_data->tr_resample(obj);
            
//            cat_diffmap_queue_empty = 1;
//            cat_ptr_pos_queue_empty = 1;
            MUTEX_UNLOCK(&mouse_chase_mtx);
            CDMSG(CAT_CDMSG_MTX, "mouse_chase_mtx unlocked\n");
        }

/*        
        pp_hrtime_stop(&profiler);
        CDMSG(0, "sampling / resampling took %f sec (%fHz)\n",
              pp_hrtime_read(&profiler) / 1000000000.0,
              1000000000.0 / pp_hrtime_read(&profiler));
*/
    }
    
    mouse_chase = 0;
    
    return NULL;
}

#if defined(CAT_GLOBAL_LOCALIZATION_THR)
static void* mouse_locate_thr_func(void* arg) {
    driver_t *obj = (driver_t*)arg;
    data_imouse_t *imouse_data;
    fb_format_info_t fb_info;
#if defined(CAT_FB_DIRECT)
    u_int16_t *shape_fb_rgb = NULL;
#else /* CAT_FB_DIRECT */
    t_bitmap screen = {0, 0, NULL};
#endif /* CAT_FB_DIRECT */
    t_point found;
    static const u_int32_t threshold = CAT_SSD_THRESHOLD;  
    t_bitmap *shape, *mask;
#if defined(PP_FEAT_CAT)
    data_cat_t *mouse_data = (data_cat_t*)imouse_data;
#endif  /* PP_FEAT_CAT */
#if defined(PP_FEAT_KITTY_CAT)
    t_mouse_data *md;
#endif /* PP_FEAT_KITTY_CAT */

#if defined(CAT_DEBUG)
    pp_stopwatch_t watch;
#endif /* CAT_DEBUG */
    
    assert(obj);

    imouse_data = (data_imouse_t*)obj->data_conv_mouse.data;
    assert(imouse_data);
#if defined(PP_FEAT_CAT)
    assert(mouse_data->shape);
    shape = mouse_data->shape;
    
    assert(mouse_data->mask);
    mask = mouse_data->mask;

    ccd->absolute.x = mouse_data->rc_abs.x;
    ccd->absolute.y = mouse_data->rc_abs.y;
#endif /* PP_FEAT_CAT */
#if defined(PP_FEAT_KITTY_CAT)
    while(!imouse_data->current_kvm_port) {
        /* wait for PP_PROP_KVM_PORT_SWITCHED to init current_kvm_port */
        CDMSG(CAT_CDMSG_GLOBAL_LOCALIZATION_INFO,
              "wait for current port to get initialized!\n");
        usleep(250000);
    }

    assert(imouse_data->current_kvm_port);
    md = &imouse_data->current_kvm_port->mouse_data;
    assert(md);

    shape = &imouse_data->current_cursor_shape;
    mask = &imouse_data->current_cursor_mask;
#endif /* PP_FEAT_KITTY_CAT */

#if defined(CAT_FB_DIRECT)
    /* wait for driver to get initialized */
    while(obj->state == DRIVER_NOT_INITIALIZED ||
#if defined(PP_FEAT_KITTY_CAT)
          md->cursor_state != ERIC_MOUSE_CURSOR_IS_VALID ||
#endif /* PP_FEAT_KITTY_CAT */
          !shape->rgb || !mask->rgb) {
#warning TODO! timed wait / signal?
        CDMSG(CAT_CDMSG_GLOBAL_LOCALIZATION_INFO,
              "wait for driver to get initialized!\n");
        usleep(250000);
    }

    if(PP_ERR == cat_bmp2fb_rgb(&shape_fb_rgb, shape)) {
        CDMSG(CAT_CDMSG_ERROR,
              "could not convert shape to frame buffer RGB!\n");
        abort();
        return NULL;
    }
#endif /* CAT_FB_DIRECT */

    while(mouse_chase > 0) {
        if(mouse_locate != 1 || pp_vsc_get_fb_format(0, &fb_info) != 0) {
            usleep(250000);
            continue;
        }
        
#if defined(CAT_DEBUG)
        pp_hrtime_start(&watch);
#endif /* CAT_DEBUG */

        /* locate the mouse */
        CDMSG(CAT_CDMSG_GLOBAL_LOCALIZATION_INFO, 
              "global mouse pointer localisation triggered!\n");
        
        DBG_WRITE_BITMAP(mask, "debug_mouse_locate_thr_mask.bmp");
#if !defined(CAT_FB_DIRECT)
        screen.width = fb_info.g_w;
        screen.height = fb_info.g_h;
        screen.rgb = (t_rgb*)malloc(screen.width * screen.height *
                                    sizeof(t_rgb));
        
        assert(screen.rgb);
        DBG_WRITE_BITMAP(shape, "debug_mouse_locate_thr_shape.bmp");
        
        if(screen.rgb &&
           PP_ERR != fb_to_rgb_area(imouse_data->grab_client, &screen, 
                                    0, 0, screen.width, screen.height)) {
            if(PP_ERR != cat_mse32(&screen, shape, mask,
                                   0, 0, screen.width, screen.height, 
                                   threshold, &found)) {
#else /* CAT_FB_DIRECT */
        if(PP_ERR != cat_mse32_fb(imouse_data->grab_client, shape_fb_rgb, mask,
                                  0, 0, fb_info.g_w, fb_info.g_h,
                                  threshold, 0, &found)) {
#endif /* CAT_FB_DIRECT */
#if defined(CAT_DEBUG)
            pp_hrtime_stop(&watch);
            CDMSG(CAT_CDMSG_GLOBAL_LOCALIZATION_INFO,
                  "located pointer @ %d, %d, took %llu usec\n",
                  found.pos_x, found.pos_y, pp_hrtime_read(&watch) / 1000);
#endif /* CAT_DEBUG */

            if(PP_ERR != cat_get_position_estimate(obj, NULL, NULL)) {
                CDMSG(CAT_CDMSG_GLOBAL_LOCALIZATION_INFO,
                      "pointer estimation already set\n");
                goto bailout;
            }
            
            cat_queue_lock(imouse_data->ptr_move_queue);
            if(cat_queue_is_empty(imouse_data->ptr_move_queue)) {
                cat_set_position_estimate(obj, 1, found.pos_x, found.pos_y, 0);
#if defined(CAT_DEBUG)
            } else {
                CDMSG(CAT_CDMSG_GLOBAL_LOCALIZATION_INFO,
                      "pointer queue not valid any more\n");
                cat_queue_unlock(imouse_data->ptr_move_queue);
                goto bailout;
#endif /* CAT_DEBUG */
            }
            cat_queue_unlock(imouse_data->ptr_move_queue);
#if defined(CAT_DEBUG)
#if defined(CAT_DUMP_BMPS)
            {
                t_bitmap dbg_img = {0, 0, NULL};
                int dbg_left, dbg_top, dbg_width, dbg_height;
                
                dbg_left = SET_WITHIN(found.pos_x - (int)mask->width,
                                      0, (int)fb_info.g_w);
                dbg_width = SET_WITHIN(3 * (int)mask->width,
                                       0, (int)fb_info.g_w - dbg_left);
                dbg_top = SET_WITHIN(found.pos_y - (int)mask->height,
                                     0, (int)fb_info.g_h);
                dbg_height = SET_WITHIN(3 * (int)mask->height,
                                        0, (int)fb_info.g_h - dbg_top);
                
                dbg_img.width = dbg_width;
                dbg_img.height = dbg_height;
                dbg_img.rgb = (t_rgb*)malloc(dbg_width * dbg_height *
                                             sizeof(t_rgb));
                
                fb_to_rgb_area(imouse_data->grab_client, &dbg_img, 
                               dbg_left, dbg_top, dbg_width, dbg_height);
                DBG_WRITE_BITMAP(&dbg_img, "debug_mouse_locate_thr.bmp");
                free(dbg_img.rgb);
                CDMSG(CAT_CDMSG_GLOBAL_LOCALIZATION_INFO,
                      "wrote bitmap (%4d; %3d) -  (%4d; %3d)\n",
                      dbg_left, dbg_top, 
                      dbg_left + dbg_width, dbg_top + dbg_height);
            }
#endif /* CAT_DUMP_BMPS */
        } else {
            pp_hrtime_stop(&watch);
            CAT_INVALIDATE_POSITION_ESTIMATE;
            CDMSG(CAT_CDMSG_GLOBAL_LOCALIZATION_INFO,
                  "pointer not located, took %llu usec\n",
                  pp_hrtime_read(&watch) / 1000);
#endif /* CAT_DEBUG */
        }

 bailout:
#warning mouse_locate_thr running all the time!
usleep(250000);//            mouse_locate = 0;
#if !defined(CAT_FB_DIRECT)
            DBG_WRITE_BITMAP(&screen, "debug_mouse_locate_thr_screen.bmp");
#if defined(CAT_DEBUG)
        } else {
            CDMSG(CAT_CDMSG_GLOBAL_LOCALIZATION_INFO,
                  "could not get screenshot!\n");
//            abort();
#endif /* CAT_DEBUG */
        }
        
        free(screen.rgb);
        screen.rgb = NULL;
#endif /* CAT_FB_DIRECT */
    }
    
#if defined(CAT_FB_DIRECT)
    free(shape_fb_rgb);
#endif /* CAT_FB_DIRECT */
    return NULL;
}
#endif /* CAT_GLOBAL_LOCALIZATION_THR */    
#endif /* !CAT_DISABLE_TRACKING */

int64_t cat_get_position_estimate(const driver_t *obj, 
                                  u_int16_t *estim_x, u_int16_t *estim_y) {
    data_imouse_t* imouse_data;
    int64_t ret;

    assert(obj);
    imouse_data = (data_imouse_t*)obj->data_conv_mouse.data;
    assert(imouse_data);
    
    if(!mouse_chase || !imouse_data->get_pos_estim) {
        return PP_ERR;
    }
    
    CDMSG(CAT_CDMSG_MTX, "wait for pos_estim_mtx\n");
    MUTEX_LOCK(&pos_estim_mtx);
    CDMSG(CAT_CDMSG_MTX, "pos_estim_mtx locked\n");
    ret = imouse_data->get_pos_estim(obj, estim_x, estim_y);
    MUTEX_UNLOCK(&pos_estim_mtx);
    CDMSG(CAT_CDMSG_MTX, "pos_estim_mtx unlocked\n");
    
    return ret;
}

int cat_set_position_estimate(const driver_t *obj, u_char reliable,
                              u_int16_t estim_x, u_int16_t estim_y,
                              u_int64_t timestamp) {
    data_imouse_t* imouse_data;
    int ret;

    assert(obj);
    imouse_data = (data_imouse_t*)obj->data_conv_mouse.data;
    assert(imouse_data);
    
    if(!mouse_chase || !imouse_data->set_pos_estim) {
        return PP_ERR;
    }
    
    CDMSG(CAT_CDMSG_MTX, "wait for pos_estim_mtx\n");
    MUTEX_LOCK(&pos_estim_mtx);
    CDMSG(CAT_CDMSG_MTX, "pos_estim_mtx locked\n");
    ret = imouse_data->set_pos_estim(obj, reliable, 
                                     estim_x, estim_y, timestamp);
    MUTEX_UNLOCK(&pos_estim_mtx);
    CDMSG(CAT_CDMSG_MTX, "pos_estim_mtx unlocked\n");
    
    return ret;
}

#if defined(PP_FEAT_CAT)
int cat_get_position_diff(const driver_t *obj, int16_t *dx, int16_t *dy) {
    data_imouse_t* imouse_data;
    int ret;

    assert(obj);
    imouse_data = (data_imouse_t*)obj->data_conv_mouse.data;
    assert(imouse_data);
    
    if(!mouse_chase || !imouse_data->get_pos_diff) {
        return PP_ERR;
    }
    
    CDMSG(CAT_CDMSG_MTX, "wait for pos_estim_mtx\n");
    MUTEX_LOCK(&pos_estim_mtx);
    CDMSG(CAT_CDMSG_MTX, "pos_estim_mtx locked\n");
    ret = imouse_data->get_pos_diff(obj, dx, dy);
    MUTEX_UNLOCK(&pos_estim_mtx);
    CDMSG(CAT_CDMSG_MTX, "pos_estim_mtx unlocked\n");
    
    return ret;
}
#endif /* PP_FEAT_CAT */

#if 0
/* adjust transition table */
void cat_translation_adjust(const driver_t *obj, 
                            int16_t rc_diff, int16_t real_diff) {
    data_cat_t* mouse_data;
    int16_t i;
    
    if(rc_diff < 0) {
        rc_diff *= -1;
    }
    if(real_diff < 0) {
        real_diff *= -1;
    }

    assert(obj);
    mouse_data = (data_cat_t*)obj->data_conv_mouse.data;
    assert(mouse_data);

    /**
     * kme sent rc_diff movement, mouse moved real_diff
     * -> set tt for real_diff to rc_diff
     */
    mouse_data->tt[real_diff] = (mouse_data->tt[real_diff] * 
                                 (float)mouse_data->ttc[real_diff] + rc_diff) /
                                 (float)++mouse_data->ttc[real_diff];
                               
    for(i = real_diff - 1; i >= 0 &&
                           mouse_data->tt[i] > mouse_data->tt[real_diff]; --i) {
        mouse_data->tt[i] = mouse_data->tt[real_diff];
    }
    
    /**
     * adjust tt[rc_diff] by chance
     */
    mouse_data->tt[rc_diff] = (mouse_data->tt[rc_diff] * 
                               ((float)mouse_data->ttc[rc_diff] + 
                                (float)rc_diff / ((float)real_diff + 1e-50))) /
                               (float)++mouse_data->ttc[rc_diff];
                               
    for(i = rc_diff - 1; i >= 0 &&
                         mouse_data->tt[i] > mouse_data->tt[rc_diff]; --i) {
        mouse_data->tt[i] = mouse_data->tt[rc_diff];
    }
    
    CDMSG(0, "absolute move %4d pixels => real %4d pixels: "
             "tt[%4d] = %4d, tt[%4d] = %4d\n",
             rc_diff, real_diff, 
             rc_diff, (int16_t)mouse_data->tt[rc_diff],
             real_diff, (int16_t)mouse_data->tt[real_diff]);
//for(i = 0; i < 25; ++i)printf("tweb: tt[%4d] = %f\n", i, mouse_data->tt[i]);
}

int16_t cat_translation_lookup(const driver_t *obj, int16_t diff) {
    signed char sign = diff < 0 ? -1 : 1;
    u_int16_t lookup = diff * sign;
    data_cat_t* mouse_data;

    assert(obj);
    mouse_data = (data_cat_t*) obj->data_conv_mouse.data;
    assert(mouse_data);
    assert(mouse_data->tt);
    
    return((int16_t)mouse_data->tt[lookup] * sign);
}
#endif

void cat_reset_tracking(const driver_t *obj) {
    data_imouse_t* imouse_data;

    assert(obj);
    imouse_data = (data_imouse_t*)obj->data_conv_mouse.data;
    assert(imouse_data);
    assert(imouse_data->tr_reset);
    
    imouse_data->tr_reset(obj);
}

int cat_update_mouse_position(const driver_t *obj) {
    int ret = PP_ERR;
    u_int16_t estim_x, estim_y;
    data_imouse_t *mouse_data;
    t_mouse_data *md;
    
    assert(obj);
    mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    assert(mouse_data);
    md = &mouse_data->current_kvm_port->mouse_data;
    assert(md);
    
    pp_hrtime_stop(&mouse_data->update_timer);
    if(pp_hrtime_read(&mouse_data->update_timer) >> 20 > 100 &&
       PP_ERR != cat_get_position_estimate(obj, &estim_x, &estim_y)) {
        long diff_x, diff_y;

        diff_x = (md->cur_x - (estim_x * MOUSE_RASTER));
        diff_y = (md->cur_y - (estim_y * MOUSE_RASTER));

        /* correct position if we have more than 10px citiblock diff */
        if(abs(diff_x) + abs(diff_y) > 10 * MOUSE_RASTER) {
            /* correct current position */
            md->cur_x -= diff_x;
            md->cur_y -= diff_y;
            
            ret = PP_SUC;
        }

        pp_hrtime_start(&mouse_data->update_timer);
    }
    
    /* invalidate applied position estimate! */
    CAT_INVALIDATE_POSITION_ESTIMATE;
    
    return ret;
}

int cat_get_mousecursor_shape(driver_t *obj) {
    int update;
    data_imouse_t *mouse_data;
    t_mouse_data *md;
    
    assert(obj);
    mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    assert(mouse_data);
    md = &mouse_data->current_kvm_port->mouse_data;
    assert(md);
    
 cursor_valid_loop:
    update = md->cursor_state != ERIC_MOUSE_CURSOR_IS_VALID;
    if(update) {
        if(md->cursor_state != ERIC_MOUSE_CURSOR_NOT_VALID) {
            /* seems, we are getting cursor... wait a bit and try again */
            usleep(125000); /* 2.5x CURSOR_SNAP_SLEEP */
            goto cursor_valid_loop;
        }
        /* ok, cursor is invalid, get new one */
        get_mousecursor_shape(obj);
        md->cur_x =  MOUSE_RASTER/2;
        md->cur_y = -MOUSE_RASTER/2;
    }
    
    return update;
}
    
    
