/******************************************************************************\
* cat_internal.h                                                               *
*                                                                              *
* CURSOR ACTIVITY TRACKING                                                     *
*                                                                              *
* Intelligent online mouse cursor detection and tracking                       *
*                                                                              *
* Copyright 2005 Peppercon AG                                                  *
* Thomas Weber tweb@peppercon.de                                               *
\******************************************************************************/

#ifndef __CAT_INTERNAL_H__
#define __CAT_INTERNAL_H__

/* do not include if no CAT feature is present! */
#if defined(PP_FEAT_CAT) || defined(PP_FEAT_KITTY_CAT)

/*
#if !defined(NDEBUG)
#warning remove me! debug only! there are already some cat methods...
#include "../../libesmtp/concatenate.h"
#endif
*/

#include <sys/types.h>

#include <pp/kvm.h>

#include "driver.h"
#include "iipptr_bitmap.h"
//#include "iipptr_internal.h"

/********************************** FEATURES **********************************/

/**
 * enable debugging for CAT
 */
#if !defined(NDEBUG)
# define noCAT_DEBUG
#endif /* !NDEBUG */

/**
 * global pointer localization
 */
#define CAT_GLOBAL_LOCALIZATION_THR
 
#if defined(PP_FEAT_CAT)

/**
 * time in ms to wait until starting auto recognition if no mouse event is
 * enqueued
 */
#define PP_KM_CAT_WAIT 250

/**
 * use intelligent sync for CAT
 */
#define PP_KM_CAT_INTELLI

/**
 * maximum citiblock distance of tracked pointer position and internal pointer
 * position to be handled as synchronized
 */
#define PP_KM_CAT_MAX_ASYNC_CB 10

/**
 * maximum number of "ticks" to send to the host unpartitioned
 * (PS/2 supports a value of 0xFF, 0xAA is not supported by some hosts)
 * (see data_conv_mouse_intelli_internal.c:move_mouse_direct)
 */
#define PP_KM_CAT_MAX_TICKS_DIRECT 0xff 

#endif /* PP_FEAT_CAT */

/**
 * work directly on framebuffer instead of taking a screenshot (i.e. copying the
 * whole image)
 * saves memory, no need to copy, but algorithms are inefficient (TODO!)
 * attention: no framebuffer in local test!
 */
#if !defined(CAT_BLOCK_MATCHING_LOCAL) 
# define CAT_FB_DIRECT
#endif

/**
 * do not only draw samples from hypotheses with large probability but also
 * spread samples around "obvious" locations in screen, e.g. the last known 
 * position, the last absolute position, ...
 */
# define CAT_CONDENSATION_INTELLI_SAMPLE

/**
 * do not generate explicit movement commands but correct user movement
 */
#define CAT_APPROACH_ON_THE_FLY

/**
 * do not invalidate resampling results but backtrace movement
 */
#define noCAT_CONDENSATION_BACKTRACE_PTR_MOVE

/**
 * do not aquire mousecursor shape but wait for ERIC_MOUSE_CURSOR_IS_VALID
 */
#define CAT_CONDENSATION_USE_ERIC_MOUSE_CURSOR_STATE

/**
 * adapt lookup table on the fly
 */
#define CAT_ADAPTIVE_LOOKUP_TABLE

/**
 * threshold for cropping the mouse cursor masks
 * (in general and for intelli sync)
 */
#define CAT_CROP_CHANNEL_THRESHOLD 0x1f
//#define CAT_CROP_CHANNEL_THRESHOLD 0x3f
#define CAT_CROP_CHANNEL_INTELLI_THRESHOLD 0x3f

/**
 * threshold for SSD matching (max mean div per px and channel)
 */
#if defined(PRODUCT_ERIC2) // no A/D sampling...
# define CAT_SSD_THRESHOLD 32
#else /* !PRODUCT_ERIC2 */
# define CAT_SSD_THRESHOLD 64
#endif /* !PRODUCT_ERIC2 */

/**
 * minimum mouse pointer pointer deviation for approach in pixel
 */
#define MIN_PTR_DEV_4_APPROACH 3

/********************************* CAT QUEUE **********************************/

typedef enum {
    CAT_QUEUE_TIME_BEFORE,
    CAT_QUEUE_TIME_IS,
    CAT_QUEUE_TIME_AFTER
} cat_queue_time_type_t;

typedef enum {
    CAT_QUEUE_DIFFMAP_ENTRY,
    CAT_QUEUE_PTR_POS_ENTRY,
    CAT_QUEUE_PTR_MOVE_ENTRY,
#if defined(CAT_ADAPTIVE_LOOKUP_TABLE)
    CAT_QUEUE_LL_MOVE_ENTRY,
#endif /* CAT_ADAPTIVE_LOOKUP_TABLE */
} cat_queue_entry_type_t;

typedef struct {
    u_int16_t   size;           // data size in bytes
    u_char      line_len;       // total length of line in hex-tiles 
    u_char      width;          // number of hex-tiles in x direction
    u_char      height;         // number of hex-tiles in y direction
    u_int32_t*  data;           // the diffmap
} __attribute__((__packed__)) cat_queue_diffmap_entry_t;

typedef struct {
    u_int16_t   x;
    u_int16_t   y;
} cat_queue_ptr_pos_entry_t;

typedef struct {
    int16_t     x;
    int16_t     y;
} cat_queue_ptr_move_entry_t;

typedef struct {
    int        step;
    long       ticks_x;
    long       ticks_y;
    long       moved_x;
    long       moved_y;
} cat_queue_ll_move_entry_t;

typedef struct {
    cat_queue_entry_type_t              type;
    u_int64_t                           timestamp;
    union {
        cat_queue_diffmap_entry_t       diffmap;
        cat_queue_ptr_pos_entry_t       ptr_pos;
        cat_queue_ptr_move_entry_t      ptr_move;
#if defined(CAT_ADAPTIVE_LOOKUP_TABLE)
        cat_queue_ll_move_entry_t       ll_move;
#endif /* CAT_ADAPTIVE_LOOKUP_TABLE */
    }                                   data;
} cat_queue_entry_t;

typedef struct {
    size_t              size;   // maxium number of elems to be enqueued
    u_int16_t           elems;  // number of elems currently enqueued
    u_int16_t           first;                  // number of first entry
    u_int16_t           last;                   // number of last entry
    pthread_mutex_t     mtx;                    // mutex for queue access
    void                (*notify)(void);        // enqueue notification hook
    cat_queue_entry_t** data;                   // the entries
    char*               name;                   // identifier... very useful ;-)
} cat_queue_t;

/**
 * init a new cat queue for size entries
 * catq is set to a pointer to obj queue
 * if not NULL, notify cb is called on cat_queue_enqueue
 */
int cat_queue_init(cat_queue_t **catq, size_t size, void (*notify)(void),
                   char *name);

/**
 * free cat queue and all of its elements
 */
void cat_queue_free(cat_queue_t *catq);

/**
 * dequeue and destroy all entries of queue
 */
void cat_queue_clear(cat_queue_t *catq);

/**
 * cat queue mutex locking
 */
void cat_queue_lock(cat_queue_t *catq);
int cat_queue_trylock(cat_queue_t *catq);
void cat_queue_unlock(cat_queue_t *catq);

/**
 * query if cat queue is empty
 */
int cat_queue_is_empty(cat_queue_t *catq);

/**
 * enqueue cat queue entry in cat queue
 */
int cat_queue_enqueue(cat_queue_t *catq, cat_queue_entry_t *cqe);

/**
 * enqueue a diffmap in cat queue
 */
int cat_queue_enqueue_diffmap(cat_queue_t *catq, u_int32_t *diffmap, 
                              u_int size, u_int line_len,
                              u_int width, u_int height);

/**
 * enqueue a pointer position in cat queue
 */
int cat_queue_enqueue_ptr_pos(cat_queue_t *catq, u_int16_t x, u_int16_t y);

/**
 * enqueue a pointer move in cat queue
 */
int cat_queue_enqueue_ptr_move(cat_queue_t *catq, int16_t x, int16_t y);

#if defined(CAT_ADAPTIVE_LOOKUP_TABLE)
/**
 * enqueue a pointer move in cat queue
 */
int cat_queue_enqueue_ll_move(cat_queue_t *catq, long moved_x, long moved_y,
                              long ticks_x, long ticks_y);
#endif /* CAT_ADAPTIVE_LOOKUP_TABLE */

/**
 * dequeues first entry of cat queue
 */
cat_queue_entry_t* cat_queue_dequeue(cat_queue_t *catq);

/**
 * dequeues first entry of cat queue
 */
cat_queue_entry_t* cat_queue_dequeue_timed(cat_queue_t *catq, u_int64_t time,
                                           cat_queue_time_type_t comparator);

/**
 * create diffmap entry for cat queue
 * if size is not 0, size bytes of diffmap are copied, diffmap pointer is used
 * otherwhise 
 */
cat_queue_entry_t* cat_queue_create_diffmap_entry(u_int32_t *diffmap, 
                                                  u_int size, u_int line_len,
                                                  u_int width, u_int height);

/**
 * create pointer position entry for cat queue
 */
cat_queue_entry_t* cat_queue_create_ptr_pos_entry(u_int16_t x, u_int16_t y);

/**
 * create pointer move entry for cat queue
 */
cat_queue_entry_t* cat_queue_create_ptr_move_entry(int16_t x, int16_t y);

#if defined(CAT_ADAPTIVE_LOOKUP_TABLE)
/**
 * create low level move entry for cat queue
 */
cat_queue_entry_t* cat_queue_create_ll_move_entry(long moved_x, long moved_y,
                                                  long ticks_x, long ticks_y);
#endif /* CAT_ADAPTIVE_LOOKUP_TABLE */

/**
 * free cat queue entry and stored data
 */
void cat_queue_destroy_entry(cat_queue_entry_t *cqe);

/**
 * free cat queue diffmap entry and stored data
 */
void cat_queue_destroy_diffmap_entry(cat_queue_entry_t *cqe);

/**
 * free cat queue pointer position entry and stored data
 */
void cat_queue_destroy_ptr_pos_entry(cat_queue_entry_t *cqe);

/**
 * free cat queue pointer move entry and stored data
 */
void cat_queue_destroy_ptr_move_entry(cat_queue_entry_t *cqe);

#if defined(CAT_ADAPTIVE_LOOKUP_TABLE)
/**
 * free cat queue low level move entry and stored data
 */
void cat_queue_destroy_ll_move_entry(cat_queue_entry_t *cqe);
#endif /* CAT_ADAPTIVE_LOOKUP_TABLE */

/********************************** TRACKING **********************************/

typedef struct {
    u_int16_t   x;      // absolute x-position
    u_int16_t   y;      // absolute y-position
    u_char      weight; // weight of sample
} __attribute__((__packed__)) cat_sample_t;

typedef struct {
    u_int32_t       x;  // absolute x-position
    u_int32_t       y;  // absolute y-position
    float           p;  // probability value
    float           w;  // weight of sample
    u_int64_t       t;  // timestamp of sample
} cat_sample32_t;

/******************************** condensation ********************************/

/* CAT_CONDENSATION_SAMPLE_COUNT must be 2^n - 1 !!! 
   see initialize_sample_from_samples for example */
//#define CAT_CONDENSATION_SAMPLE_COUNT           0x01 // nice to debug ;-)
//#define CAT_CONDENSATION_SAMPLE_COUNT           0x0f // 15
#define CAT_CONDENSATION_SAMPLE_COUNT           0x3f // 63
//#define CAT_CONDENSATION_SAMPLE_COUNT           0xff // 255
//#define CAT_CONDENSATION_SAMPLE_COUNT           0x01ff // 511
//#define CAT_CONDENSATION_SAMPLE_COUNT           0x03ff // 1023
//#define CAT_CONDENSATION_SAMPLE_COUNT           0x07ff // 2047
/* sample weight minimum is 1, 0 is uninitialized */
#define CAT_CONDENSATION_SAMPLE_INIT_WEIGHT     (4 | 0x80) // weight | init bit
#define CAT_CONDENSATION_SAMPLE_MAX_WEIGHT      20
#define CAT_CONDENSATION_SAMPLE_WEIGHT_INC      1
#define CAT_CONDENSATION_SAMPLE_WEIGHT_DEC      2

typedef struct {
    BITFIELD5(u_char,           initialize : 1, /* initialize samples */
                                reset: 1,       /* reset samples */
                                reliable : 1,   /* is estim reliable? */
                                rreliable : 1,  /* was estim reliable on res? */
                                reserved : 4);
    cat_sample32_t*             samples;        /* list of samples */
    cat_sample32_t*             hypos;          /* list of position hypos */
    cat_queue_ptr_pos_entry_t   absolute;       /* latest absolute position */
    cat_queue_ptr_move_entry_t  relative;       /* latest relative move */
    cat_queue_ptr_pos_entry_t   estim;          /* pose estimation data */
    u_int64_t                   estim_time;     /* pose estimation timestamp */
} cat_condensation_data_t;

typedef struct {
    BITFIELD3(u_char,           initialize : 1, /* initialize samples */
                                reliable : 1,   /* is estim reliable? */
                                reserved : 6);
    cat_sample_t*               samples;        /* list of samples */
    cat_queue_ptr_pos_entry_t   absolute;       /* latest absolute position */
    cat_queue_ptr_move_entry_t  relative;       /* latest relative move */
    cat_queue_ptr_pos_entry_t   estim;          /* pose estimation data */
    
} cat_condensation_simple_data_t;

/**
 * condensation lifecycle
 */
int cat_condensation_init(driver_t *obj);
void cat_condensation_cleanup(driver_t *obj);

/**
 * condensation callbacks
 */
void cat_condensation_sample(driver_t *obj);
void cat_condensation_resample(driver_t *obj);
void cat_condensation_reset(const driver_t *obj);
int64_t cat_condensation_get_position_estimate(const driver_t *obj,
                                               u_int16_t *estim_x, 
                                               u_int16_t *estim_y);
int cat_condensation_set_position_estimate(const driver_t *obj,
                                           u_char reliable,
                                           u_int16_t estim_x, 
                                           u_int16_t estim_y,
                                           u_int64_t timestamp);
#if defined(PP_FEAT_CAT)
int cat_condensation_get_position_diff(const driver_t *obj,
                                       int16_t *dx, int16_t *dy);
#endif /* PP_FEAT_CAT */

/***************************** simple condensation ****************************/

#define SIMPLE_CONDENSATION_DEBUG_SAMPLES

/**
 * simple condensation lifecycle
 */
int cat_condensation_simple_init(driver_t *obj);
void cat_condensation_simple_cleanup(driver_t *obj);

/**
 * simple condensation callbacks
 */
void cat_condensation_simple_sample(driver_t *obj);
void cat_condensation_simple_resample(driver_t *obj);
int64_t cat_condensation_simple_get_position_estimate(const driver_t *obj,
                                                      u_int16_t *estim_x, 
                                                      u_int16_t *estim_y);

/******************************* BLOCK MATCHING *******************************/

#define CAT_SSD_MAX_DIFF 3072 // 5 bit squared per channel
#define CAT_BLOCK_MATCHING_MAX_ITER 4

#if !defined(CAT_FB_DIRECT)

/**
 * Calculates the sum of squared differences of a mask/shape at position
 * (x_offset, y_offset) in screen.
 * The maximum value may be ssd_min, as obj specifies the current minimum.
 * Search increment is given by iter.
 * If successful, returns SSD, PP_ERR otherwhise.
 */
int cat_ssd32_sub(t_bitmap *screen, t_bitmap *shape, t_bitmap *mask, 
                  u_int32_t x_offset, u_int32_t y_offset, 
                  u_int32_t ssd_min, u_int32_t iter);

/**
 * Calculates the minimum mean squared error of a mask/shape within screen from
 * (search_left, search_top) to (search_width, search_height).
 * A maximum mean divergence per pixel and channel is specified by threshold.
 * If successfull, returns minimum MSE and found is set to corresponding
 * position, returns PP_ERR otherwhise.
 */
int cat_mse32(t_bitmap *screen, t_bitmap *shape, t_bitmap *mask, 
              u_int32_t search_left, u_int32_t search_top,
              u_int32_t search_width, u_int32_t search_height,
              u_int32_t threshold, t_point *found);

#else /* CAT_FB_DIRECT */

/**************************** frame buffer based ******************************/

/**
 * converts a bitmap into frame buffer 16 bit rgb
 */
int cat_bmp2fb_rgb(u_int16_t **ret, const t_bitmap *bmp);

/**
 * Calculates the minimum mean squared error of a mask/shape within frame buffer
 * from (search_left, search_top) to (search_width, search_height).
 * A maximum mean divergence per pixel and channel is specified by threshold.
 * If successfull, returns minimum MSE and found is set to corresponding
 * position, returns PP_ERR otherwhise.
 */
int cat_mse32_fb(pp_grab_client_t *grab_client, 
                 const u_int16_t *shape, const t_bitmap *mask, 
                 u_int32_t search_left, u_int32_t search_top,
                 u_int32_t search_width, u_int32_t search_height, 
                 u_int32_t threshold, int32_t iter, t_point *found);

/**
 * obj function tries to find the mouse cursor on the screen in a
 * horizontal stripe of cursor size at the top of the screen,
 * provided for building the acceleration table
 * (see iipptr_cursor.c:find_cursor_for_table)
 */
int cat_find_cursor_for_table_fb(const driver_t *obj, const u_int16_t *shape, 
                                 const t_point *center, t_point *found);
#endif /* CAT_FB_DIRECT */

/******************************** CLUSTERING **********************************/

/**
 * cluster samples by a simple density clustering algorithm
 * cluster_ids has to be of same dimensions as samples, holds cluster_id of
 * corresponding sample
 * returns number of clusters generated
 * TODO! choose a more efficient one...
 */
int cat_cluster_simple_density(int *cluster_ids, const cat_sample32_t *samples,
                               u_int32_t max_diff_x, u_int32_t max_diff_y);

/**
 * get dimensions (x_min, y_min) - (x_max, y_max) of cluster
 * returns number of elements if cluster exists, PP_ERR otherwhise
 */
int cat_cluster_get_dimensions(int *x_min, int *y_min, int *x_max, int *y_max,
                               int cluster_id, const int *cluster_ids, 
                               const cat_sample32_t *samples);

/**
 * get size of cluster
 */
int cat_cluster_get_size(int cluster_id, const int *cluster_ids);

/**
 * alligns cluster dimensions to given image size max_x, max_y
 */
int cat_cluster_align_dimensions(u_int32_t *left, u_int32_t *top, 
                                 u_int32_t *width, u_int32_t *height,
                                 u_int32_t x_min, u_int32_t y_min, 
                                 u_int32_t x_max, u_int32_t y_max,
                                 u_int32_t max_diff_x, u_int32_t max_diff_y,
                                 u_int32_t max_x, u_int32_t max_y);

/**
 * wraps above... see code...
 */
int cat_cluster_get_search_space_for_cluster(u_int32_t *left, 
                                             u_int32_t *top, 
                                             u_int32_t *width, 
                                             u_int32_t *height,
                                             int cluster_id, 
                                             const int *cluster_ids, 
                                             const cat_sample32_t *samples,
                                             u_int32_t shape_width, 
                                             u_int32_t shape_height,
                                             u_int32_t grow_x, 
                                             u_int32_t grow_y,
                                             u_int32_t max_width, 
                                             u_int32_t max_height);

/********************************** GENERIC ***********************************/

#define CAT_MOUSE_LOCATE_TIMEOUT_NS 5000000000LL // 5s

#if defined(PP_FEAT_CAT)
typedef struct {
#if defined(PP_KM_CAT_INTELLI)
    data_imouse_t               imouse; // convenience feature
    int                         state; // tmp!!!
#endif /* PP_KM_CAT_INTELLI */

    /* generic vars */
    km_kvm_entry_t*            kvm_port_state_list_units[PP_KVM_MAX_UNIT_COUNT];
    km_kvm_entry_t*             current_kvm_port;
    pthread_mutex_t             kvm_port_mtx;
//    local_propchange_listener_t propchange_listener;
    pp_grab_client_t*           grab_client;

    /* latest absolute position from rc */
    cat_queue_ptr_pos_entry_t   rc_abs;
    
    /* lookup table */
    pthread_mutex_t             lt_mtx;
    u_int*                      lt;     /* lookup table */
    u_int*                      ltc;    /* averaged item count */
   
    /* queues */
//    cat_queue_t*                diffmap_queue;
    cat_queue_t*                ptr_pos_queue;
//    cat_queue_t*                ptr_move_queue;
#if defined(CAT_ADAPTIVE_LOOKUP_TABLE)
    cat_queue_t*                ll_move_queue;
    
    cat_queue_ptr_pos_entry_t   ll_move_pos;
    pp_stopwatch_t              ll_move_timer;
#endif /* CAT_ADAPTIVE_LOOKUP_TABLE */

    /* tracking */
    int                         (*get_pos_diff)(const driver_t *obj,
                                                int16_t *dx, int16_t *dy);
    
    /* pointer info */
/*
    int                         num_pointers;
    t_bitmap**                  shapes;
    t_bitmap**                  masks;
*/
    t_bitmap*                   shape;
    t_bitmap*                   mask;
} data_cat_t;
#endif /* CAT_FB_DIRECT */

/* cat lifecycle */
int pp_cat_init(driver_t *obj); /* pp_ to avoid linking probs with libesmtp */
void cat_cleanup(driver_t *obj);
int cat_suspend(const driver_t *obj);
int cat_resume(const driver_t *obj);

/* get current position estimate */
int64_t cat_get_position_estimate(const driver_t *obj,
                                  u_int16_t *estim_x, u_int16_t *estim_y);

/* set current position estimate */
int cat_set_position_estimate(const driver_t *obj, u_char reliable,
                              u_int16_t estim_x, u_int16_t estim_y,
                              u_int64_t timestamp);

#if defined(PP_FEAT_CAT)
/* get current position diff */
int cat_get_position_diff(const driver_t *obj, int16_t *dx, int16_t *dy);
#endif /* PP_FEAT_CAT */

#if 0
/* adjust transition table */
void cat_translation_adjust(const driver_t *obj, 
                            int16_t lookup, int16_t diff);

/* get value from transition table */
int16_t cat_translation_lookup(const driver_t *obj, int16_t diff);
#endif

/**
 * calls tracking algo reset hook
 */
void cat_reset_tracking(const driver_t *obj);

/**
 * if a valid position estimate is present, cur_x and cur_y position coordinates
 * are updated
 * returns update state: updated (PP_SUC) or not (PP_ERR)
 */
int cat_update_mouse_position(const driver_t *obj);

/**
 * assures that cursor is valid, gets it otherwhise
 * returns update state: shape updated (1) or shape untouched (0)
 */
int cat_get_mousecursor_shape(driver_t *obj);

/**
 * enqueue absolute mouse pointer position received from remote console
 */
void cat_enqueue_ptr_pos(const void *that, u_int16_t x, u_int16_t y);

/**
 * enqueue relative mouse pointer movement received from remote console
 */
void cat_enqueue_ptr_move(const void *that, int16_t x, int16_t y);

/**
 * enqueue rfb diffmap
 */
void cat_enqueue_diffmap(const void *that, u_int32_t *diffmap);

#if defined(CAT_ADAPTIVE_LOOKUP_TABLE)
/**
 * enqueue ticks
 */
void cat_enqueue_ll_move(const void *that, long moved_x, long moved_y, 
                         long ticks_x, long ticks_y);
#endif /* CAT_ADAPTIVE_LOOKUP_TABLE */

/************************************ MISC ************************************/

/* min and max */
#ifndef MAX
# define MAX(x, y) ((x) < (y) ? (y) : (x))
#endif
#ifndef MIN
# define MIN(x, y) ((x) > (y) ? (y) : (x))
#endif

/* divide with overhead:  5 / 2 = 2  <=>  OHDIV(5, 2) = 3 */
#define OHDIV(t, r) (((t) + (r) - 1) / (r))

/* set value within given range */
#define SET_WITHIN(i, min, max) \
            ((i) < (min) ? (min) : ((i) > (max) ? (max) : (i)))
            
#define CAT_INVALIDATE_POSITION_ESTIMATE \
            cat_set_position_estimate(obj, 0, 0, 0, 0)

/***************************** diffmap processing *****************************/

/**
 * init a sample at a position in diffmap, where changes occured
 */
int cat_diffmap_initialize_sample(cat_sample32_t *sample,
                                  const cat_queue_diffmap_entry_t *cqde,
                                  const fb_format_info_t *fb_info);

/**
 * init a sample at a position in diffmap near bias, where changes occured
 */
int cat_diffmap_initialize_sample_biased(cat_sample32_t *sample,
                                         int bias_x, int bias_y,
                                         const cat_queue_diffmap_entry_t *cqde,
                                         const fb_format_info_t *fb_info);

/**
 * calculate absolute position from diffmap offset
 */
int cat_diffmap_get_abs_coord(u_int32_t *x, u_int32_t *y,
                              u_char idx, u_int32_t mask, u_char line_len,
                              const fb_format_info_t *fb_info);

/**
 * calculate diffmap offset from absolute position
 */
int cat_diffmap_get_offset_from_abs_coord(u_char *idx, u_int32_t *mask,
                                          u_char line_len, 
                                          u_int32_t x, u_int32_t y,
                                          const fb_format_info_t *fb_info);

/**
 * calculate diffmap position from absolute position
 */
int cat_diffmap_get_coord_from_abs_coord(u_char *dmx, u_char *dmy,
                                         u_int32_t fbx, u_int32_t fby,
                                         const fb_format_info_t *fb_info);

/**
 * calculate diffmap offset from diffmap position
 */
int cat_diffmap_get_offset_from_diffmap_coord(u_char *idx, u_int32_t *mask,
                                              u_char line_len, 
                                              u_char dmx, u_char dmy,
                                              const fb_format_info_t *fb_info);

int cat_get_merged_diffmap(cat_queue_diffmap_entry_t *cqde,
                           cat_queue_diffmap_entry_t *cqdle,
                           cat_queue_t *diffmap_queue);

/******************************** intelli sync ********************************/

#if defined(PP_FEAT_CAT)
int cat_intelli_get_mouse_universe(driver_t* obj);
int cat_intelli_get_mousecursor_shape(driver_t* obj);
int cat_intelli_move_mouse_diff(driver_t* obj, long diff_x, long diff_y, 
                                const unsigned char buttonmask, int* moved);
int cat_intelli_move_mouse_diff_core(driver_t* obj, long diff_x, long diff_y, 
                                     const unsigned char buttonmask, int* moved,
                                     cat_queue_ll_move_entry_t *ctx);
int cat_intelli_move_mouse(driver_t *obj, int x, int y, int z UNUSED,
                           const unsigned char buttonmask);
#endif /* PP_FEAT_CAT */

/******************************** lookup table ********************************/

#if defined(PP_FEAT_CAT)
void cat_lookup_table_init(driver_t *obj);
void cat_lookup_table_reset(driver_t *obj);
void cat_lookup_table_adjust(driver_t *obj);
void cat_lookup_table_add(driver_t *obj, int s_ticks, int s_diff);
void cat_lookup_table_set(driver_t *obj, int s_ticks, int s_diff);
void cat_lookup_table_scale(driver_t *obj, float scale, long min, long max);
int cat_lookup_table_out2in(driver_t *obj, int s_diff);
int cat_lookup_table_in2out(driver_t *obj, int s_ticks);
#endif /* PP_FEAT_CAT */

/**************************** kitty cat specifics *****************************/

#if defined(PP_FEAT_KITTY_CAT) && !defined(CAT_BLOCK_MATCHING_LOCAL)
# if !defined(CAT_FB_DIRECT)
#  define CAT_FB_DIRECT
# endif
# if !defined(CAT_CONDENSATION_INTELLI_SAMPLE)
#  define CAT_CONDENSATION_INTELLI_SAMPLE
# endif
# if defined(CAT_ADAPTIVE_LOOKUP_TABLE)
#  undef CAT_ADAPTIVE_LOOKUP_TABLE
# endif
#endif /* PP_FEAT_KITTY_CAT && !CAT_BLOCK_MATCHING_LOCAL */

/************************************ END *************************************/

#endif /* PP_FEAT_CAT || PP_FEAT_KITTY_CAT */

#endif /* __CAT_INTERNAL_H__ */
