/******************************************************************************\
* cat_lookup_table.c                                                           *
*                                                                              *
* CURSOR ACTIVITY TRACKING                                                     *
*                                                                              *
* Implements lookup table for intelligent online mouse cursor detection and    *
* tracking                                                                     *
*                                                                              *
* Copyright 2005 Peppercon AG                                                  *
* Thomas Weber tweb@peppercon.de                                               *
\******************************************************************************/

#include <math.h>

#include "cat_internal.h"
#include "cat_debug.h"

#if defined(CAT_LOOKUP_TABLE_DEBUG)
/* init to measured values (XP EPP) */
#define LT_SIZE 62 /* 61 measured values + 1 tailing supporting point */
#else /* CAT_LOOKUP_TABLE_DEBUG */
/* max incl. 0 ticks + 1 tailing supporting point */
#define LT_SIZE (PP_KM_CAT_MAX_TICKS_DIRECT + 2)
#endif /* CAT_LOOKUP_TABLE_DEBUG */

#define LT_TAIL (LT_SIZE - 1)

void cat_lookup_table_init(driver_t *obj) {
    /* TODO! unit / port specific! */
    /* TODO! fixpoint arithmetics! */
    u_int *lt;
    u_int *ltc;
    data_cat_t* mouse_data;

    assert(obj);
    mouse_data = (data_cat_t*) obj->data_conv_mouse.data;
    assert(mouse_data);
    
    /* alloc lookup table */
    lt = (u_int*)malloc(sizeof(u_int) * LT_SIZE);
    ltc = (u_int*)malloc(sizeof(u_int) * LT_TAIL); /* no count for tail! */

#if defined(CAT_LOOKUP_TABLE_DEBUG)
    { /* init to measured values */
        int i;

#if 0 /* WinXP EPP */    
        lt[0] = 0000;
        lt[1] = 0000;
        lt[2] = 2000;
        lt[3] = 2000;
        lt[4] = 4000;
        lt[5] = 5000;
        lt[6] = 6000;
        lt[7] = 8000;
        lt[8] = 10000;
        lt[9] = 11000;
        lt[10] = 13000;
        lt[11] = 14000;
        lt[12] = 16000;
        lt[13] = 17000;
        lt[14] = 20000;
        lt[15] = 23000;
        lt[16] = 26000;
        lt[17] = 29000;
        lt[18] = 33000;
        lt[19] = 36000;
        lt[20] = 39000;
        lt[21] = 43000;
        lt[22] = 45000;
        lt[23] = 49000;
        lt[24] = 52000;
        lt[25] = 55000;
        lt[26] = 59000;
        lt[27] = 62000;
        lt[28] = 64000;
        lt[29] = 69000;
        lt[30] = 71000;
        lt[31] = 74000;
        lt[32] = 78000;
        lt[33] = 81000;
        lt[34] = 84000;
        lt[35] = 88000;
        lt[36] = 91000;
        lt[37] = 93000;
        lt[38] = 98000;
        lt[39] = 100000;
        lt[40] = 104000;
        lt[41] = 106000;
        lt[42] = 110000;
        lt[43] = 114000;
        lt[44] = 116000;
        lt[45] = 120000;
        lt[46] = 123000;
        lt[47] = 126000;
        lt[48] = 130000;
        lt[49] = 132000;
        lt[50] = 136000;
        lt[51] = 139000;
        lt[52] = 142000;
        lt[53] = 146000;
        lt[54] = 149000;
        lt[55] = 152000;
        lt[56] = 155000;
        lt[57] = 158000;
        lt[58] = 162000;
        lt[59] = 165000;
        lt[60] = 168000;
        lt[61] = 170000;
#endif /* WinXP EPP */
#if 1 /* WinXP 1.5x */
	for(i = 0; i < 62; ++i) {
	    lt[i] = i * 1500 /* = 1.5 * MOUSE_RASTER */;
	}
#endif /* WinXP 1.5x */
        
        for(i = 0; i < 61; ++i) {
            ltc[i] = 1;
        }
    }
#else /* CAT_LOOKUP_TABLE_DEBUG */
    /* initialize lookup table */
    lt[0] = 0;
    lt[LT_TAIL] = LT_SIZE * 2 * MOUSE_RASTER;
    memset(ltc, 0, sizeof(u_int) * LT_TAIL);
#endif /* CAT_LOOKUP_TABLE_DEBUG */

    pthread_mutex_init(&mouse_data->lt_mtx, NULL);    
    mouse_data->lt = lt;
    mouse_data->ltc = ltc;
}

void cat_lookup_table_reset(driver_t *obj) {
#if !defined(CAT_LOOKUP_TABLE_DEBUG)
    /* TODO! unit / port specific! */
    data_cat_t* mouse_data;

    assert(obj);
    mouse_data = (data_cat_t*) obj->data_conv_mouse.data;
    assert(mouse_data);
    assert(mouse_data->lt);
    assert(mouse_data->ltc);
    
    MUTEX_LOCK(&mouse_data->lt_mtx);
    
    memset(mouse_data->lt, 0, sizeof(u_int) * LT_SIZE);
    memset(mouse_data->ltc, 0, sizeof(u_int) * LT_SIZE);
    
    MUTEX_UNLOCK(&mouse_data->lt_mtx);
#else /* CAT_LOOKUP_TABLE_DEBUG */
    (void)obj;
#endif /* CAT_LOOKUP_TABLE_DEBUG */
}

#if !defined(CAT_LOOKUP_TABLE_DEBUG)
static void lookup_table_adjust_core(u_int *lt, u_int *ltc) {
    int prev = 0, current = 0, next = 0;

    for(current = prev + 1; current < LT_TAIL && !ltc[current]; ++current);
    if(current == LT_TAIL) {
        printf("tweb: lookup_table_adjust failed looking up initial current\n");
        return;
    }
    
    for(next = current + 1; next < LT_TAIL && !ltc[next]; ++next);
    if(next == LT_TAIL) {
        printf("tweb: lookup_table_adjust failed looking up initial next\n");
        return;
    }
    
    while(1) {
        if(lt[current] < lt[prev] || 
           (lt[current] > lt[next] && lt[next] >= lt[prev])) {
            /* invalidate */
            /* TODO! Handle according to ltc?? */
#warning TODO!!!
            lt[current] = 0;
            ltc[current] = 0;
            current = prev;
        }
        
        /* get next next */
        prev = current;
        current = next;
        for(++next; next < LT_TAIL && !ltc[next]; ++next);
        if(next >= LT_TAIL) {
            break;
        }
    }
    
    /* adjust lookup table tail */
    lt[LT_TAIL] = lt[prev] + ((lt[current] - lt[prev]) / (current - prev)) *
                             (LT_TAIL - prev);
}
#endif /* CAT_LOOKUP_TABLE_DEBUG */

void cat_lookup_table_adjust(driver_t *obj) {
#if defined(CAT_LOOKUP_TABLE_DEBUG)
    (void)obj;
#else /* CAT_LOOKUP_TABLE_DEBUG */
    data_cat_t* mouse_data;
    u_int *ltc;
    u_int *lt;

    assert(obj);
    mouse_data = (data_cat_t*) obj->data_conv_mouse.data;
    assert(mouse_data);
    assert(mouse_data->lt);
    assert(mouse_data->ltc);
    
    lt = mouse_data->lt;
    ltc = mouse_data->ltc;
    
    MUTEX_LOCK(&mouse_data->lt_mtx);
    
    lookup_table_adjust_core(lt, ltc);

    MUTEX_UNLOCK(&mouse_data->lt_mtx);
#endif /* CAT_LOOKUP_TABLE_DEBUG */
}

void cat_lookup_table_scale(driver_t *obj, float scale, long min, long max) {
    (void)obj;
    (void)scale;
    (void)min;
    (void)max;
#if !defined(CAT_LOOKUP_TABLE_DEBUG)
    data_cat_t* mouse_data;
    u_int *ltc;
    u_int *lt;
    long i;

    assert(obj);
    mouse_data = (data_cat_t*) obj->data_conv_mouse.data;
    assert(mouse_data);
    assert(mouse_data->lt);
    assert(mouse_data->ltc);
    
    assert(min <= max);
    
    lt = mouse_data->lt;
    ltc = mouse_data->ltc;
    
    if(min < 1) {
        min = 1;
    }
    if(max > PP_KM_CAT_MAX_TICKS_DIRECT) {
        max = PP_KM_CAT_MAX_TICKS_DIRECT;
    }
    
    MUTEX_LOCK(&mouse_data->lt_mtx);
    
    for(i = min; i <= max; ++i) {
        if(lt[i]) {
#if defined(CAT_DEBUG)
            long oldval = lt[i];
#endif /* CAT_DEBUG */
            lt[i] = floor((float)lt[i] * scale);
            CDMSG(CAT_CDMSG_LOOKUP_TABLE_INFO,
                  "scale lt[%3ld] from %6ld (scale %f) to %6d (scale %f)",
                  i, oldval, (float)oldval / (i * MOUSE_RASTER),
                  lt[i], (float)lt[i] / (i * MOUSE_RASTER));
        }
    }
    lookup_table_adjust_core(lt, ltc);
    
    MUTEX_UNLOCK(&mouse_data->lt_mtx);
#endif /* !CAT_LOOKUP_TABLE_DEBUG */
}

void cat_lookup_table_add(driver_t *obj, int s_ticks, int s_diff) {
#if defined(CAT_LOOKUP_TABLE_DEBUG)
    (void)obj;
    (void)s_ticks;
    (void)s_diff;
#else /* CAT_LOOKUP_TABLE_DEBUG */
    data_cat_t* mouse_data;
    u_int *ltc;
    u_int *lt;
    int sign_ticks = s_ticks < 0 ? -1 : 1;
    int sign_diff = s_diff < 0 ? -1 : 1;
    u_int u_ticks = s_ticks * sign_ticks;
    u_int u_diff = s_diff * sign_diff;
    
    if(s_ticks == 0) {
        return;
    }
    
    if(sign_ticks != sign_diff) {
        /* strange */
        printf("tweb: cat_lookup_table_add: sign of diff and ticks differ!\n");
        assert(abs(s_diff) < 10); /* for large |diff|this is really strange */
        return;
    }
    
    assert(u_ticks < LT_TAIL);

    assert(obj);
    mouse_data = (data_cat_t*) obj->data_conv_mouse.data;
    assert(mouse_data);
    assert(mouse_data->lt);
    assert(mouse_data->ltc);
    
    lt = mouse_data->lt;
    ltc = mouse_data->ltc;
    
    MUTEX_LOCK(&mouse_data->lt_mtx);
    
    lt[u_ticks] = (lt[u_ticks] * ltc[u_ticks] + u_diff * MOUSE_RASTER) /
                  ++ltc[u_ticks];
    CDMSG(CAT_CDMSG_LOOKUP_TABLE_INFO,
          "added diff = %d to tick %d, is now %d (%d)\n",
          u_diff, u_ticks, lt[u_ticks], ltc[u_ticks]);
    lookup_table_adjust_core(lt, ltc);
    
    MUTEX_UNLOCK(&mouse_data->lt_mtx);
#endif /* CAT_LOOKUP_TABLE_DEBUG */
}

void cat_lookup_table_set(driver_t *obj, int s_ticks, int s_diff) {
#if defined(CAT_LOOKUP_TABLE_DEBUG)
    (void)obj;
    (void)s_ticks;
    (void)s_diff;
assert(0);
#else /* CAT_LOOKUP_TABLE_DEBUG */
    data_cat_t* mouse_data;
    u_int *ltc;
    u_int *lt;
    int sign_ticks = s_ticks < 0 ? -1 : 1;
    int sign_diff = s_diff < 0 ? -1 : 1;
    u_int u_ticks = s_ticks * sign_ticks;
    u_int u_diff = s_diff * sign_diff;
    
    if(sign_ticks != sign_diff) {
        /* strange */
        printf("tweb: cat_lookup_table_set: sign of diff and ticks differ!\n");
        assert(abs(s_diff) < 10); /* for large |diff| this is really strange */
        return;
    }
    
    assert(u_ticks < LT_TAIL);

    assert(obj);
    mouse_data = (data_cat_t*) obj->data_conv_mouse.data;
    assert(mouse_data);
    assert(mouse_data->lt);
    assert(mouse_data->ltc);
    
    lt = mouse_data->lt;
    ltc = mouse_data->ltc;
    
    MUTEX_LOCK(&mouse_data->lt_mtx);
    
    lt[u_ticks] = u_diff * MOUSE_RASTER;
    ltc[u_ticks] = 1;
    CDMSG(CAT_CDMSG_LOOKUP_TABLE_INFO || 1,
          "set diff = %d to tick %d\n", u_diff, u_ticks);
    
    MUTEX_UNLOCK(&mouse_data->lt_mtx);
#endif /* CAT_LOOKUP_TABLE_DEBUG */
}

int cat_lookup_table_out2in(driver_t *obj, int s_diff) {
    data_cat_t* mouse_data;
    u_int *ltc;
    u_int *lt;
    int ret;
    int sign = s_diff < 0 ? -1 : 1;
    int u_diff = s_diff * sign;

    assert(obj);
    mouse_data = (data_cat_t*) obj->data_conv_mouse.data;
    assert(mouse_data);
    assert(mouse_data->lt);
    assert(mouse_data->ltc);
    
    lt = mouse_data->lt;
    ltc = mouse_data->ltc;
    
    MUTEX_LOCK(&mouse_data->lt_mtx);
    if(u_diff > (int)lt[LT_TAIL]) {
        ret = (LT_TAIL - 1) * sign;
        CDMSG(CAT_CDMSG_LOOKUP_TABLE_INFO,
              "diff larger than max -> returning max ticks\n");
    } else if(u_diff < (int)lt[1]) {
        ret = 0;
        CDMSG(CAT_CDMSG_LOOKUP_TABLE_INFO,
              "diff smaller than min -> returning 0 ticks\n");
    } else {
        int i, prev = 0, next = LT_TAIL;
        
        for(i = LT_TAIL - 1; i > 0; --i) {
            if(ltc[i]) {
                if((int)lt[i] > u_diff) {
                    next = i;
                } else {
                    prev = i;
                    break;
                }
            }
        }
        
        /* floored linar interpolation, as we only support u_char ticks */
        ret = prev + (((u_diff - lt[prev]) * (next - prev)) /
                      (lt[next] - lt[prev]) * sign);
//printf("prev (%d) = %d; next (%d) = %d; u_diff = %d => u_ret = %d\n", prev, lt[prev], next, lt[next], u_diff, ret * sign);usleep(20000);
        assert(ret < LT_TAIL);
        CDMSG(CAT_CDMSG_LOOKUP_TABLE_INFO,
              "interpolated %d subpixels to %d ticks\n", s_diff, ret);
        assert(ret < LT_TAIL);
    }
    
    MUTEX_UNLOCK(&mouse_data->lt_mtx);
   
    return ret;
}

int cat_lookup_table_in2out(driver_t *obj, int s_ticks) {
    data_cat_t* mouse_data;
    u_int *ltc;
    u_int *lt;
    int ret;
    int sign = s_ticks < 0 ? -1 : 1;
    u_int u_ticks = s_ticks * sign;

    assert(u_ticks < LT_TAIL);
    
    assert(obj);
    mouse_data = (data_cat_t*) obj->data_conv_mouse.data;
    assert(mouse_data);
    assert(mouse_data->lt);
    assert(mouse_data->ltc);
    
    lt = mouse_data->lt;
    ltc = mouse_data->ltc;
    
    MUTEX_LOCK(&mouse_data->lt_mtx);
    if(ltc[u_ticks]) {
        ret = lt[u_ticks] * sign;
        CDMSG(CAT_CDMSG_LOOKUP_TABLE_INFO,
              "translated %d ticks to %d subpixels\n", s_ticks, ret);
    } else {
        u_int u;
        int prev, next;
        
        for(u = 1, prev = 0; u < u_ticks; ++u) {
            if(ltc[u]) {
                prev = u;
            }
        }
        for(u = LT_SIZE - 2, next = LT_TAIL; u > u_ticks; --u) {
            if(ltc[u]) {
                next = u;
            }
        }
        
        ret = (lt[prev] + ((u_ticks - prev) * (lt[next] - lt[prev])) /
                           (next - prev)) * sign;
        CDMSG(CAT_CDMSG_LOOKUP_TABLE_INFO,
              "interpolated %d ticks to %d subpixels\n", s_ticks, ret);
//printf("prev (%d) = %d; next (%d) = %d; u_ticks = %d => u_ret = %d\n", prev, lt[prev], next, lt[next], u_ticks, ret * sign);
    }
    MUTEX_UNLOCK(&mouse_data->lt_mtx);
    
    return ret;
}    

