/* system includes */
#include <stdlib.h>
#include <sys/types.h>
#include <stdio.h>

/* firmware includes */
#include <pp/base.h>
#include <pp/intl.h>
#include <liberic_config.h>
#include <pp/cfg.h>
#include <liberic_misc.h>
#include <pp/kvm.h>
#include <pp/km.h>
#include <pp/rfb.h>
#include <pp/propchange.h>
#include <pp/xdefs.h>

/* local includes */
#include "driver.h"
#include "iipptr_data.h"
#include "iipptr_internal.h"

#if defined(PP_FEAT_KITTY_CAT)
# include <pp/grab.h>
# include "cat_internal.h"
# include "iipptr_cursor.h"
#endif /* PP_FEAT_KITTY_CAT */

/* obj implements the real intelligent mouse mode (calculates
   and measures the accel curve

   history note: from iipptrbase.c

   usage note: - relative input -> single mouse mode
	       - absolute input -> intelligent mouse mode
*/


/* object functions */
static int resume(driver_t* obj);
static int suspend(driver_t* obj);
static int cleanup(driver_t* obj);
static int alloc_port_state_for_kvm_unit(driver_t* obj, int kvm_unit);
static int reset_port_state_for_kvm_unit(driver_t* obj, int kvm_unit);
static int free_port_state_for_kvm_unit(driver_t* obj, int kvm_unit);
static int reconfigure(driver_t* obj);
static void reconfigure_port(driver_t* obj, u_char kvm_unit, u_short kvm_port);
static int send_ptrmove(driver_t* obj,
		 const int x, const int y, const int z, 
		 const unsigned char buttonmask,
		 const unsigned char abs_rel);
static int sync_mouse (driver_t* obj, const int sync_type);
static int is_equal(driver_t* obj, driver_t* d);

/* local object functions */
static int single_mouse_move(driver_t* obj, const int x, const int y, const int z, 
			     const unsigned char buttonmask);
static int intelligent_mouse_move(driver_t* obj,
				  const int x, const int y, const int z, 
				  const unsigned char buttonmask);
static void set_current_kvm_port(driver_t* obj);

static void propchange_handler(pp_propchange_listener_t * listener, 
			       unsigned short prop, unsigned short prop_flags);


static const char* mouse_mode_key = "unit[%u].port[%u].mouse.mode";
static const char* mouse_direct_sc_key = "unit[%u].port[%u].mouse.direct_scaling";

int init_data_conv_imouse(driver_t* obj) {
    data_imouse_t *mouse_data;
    int kvm_unit;

    assert(obj);
    /* sets the object functions */
    obj->data_conv_mouse.id = __FUNCTION__;
    obj->data_conv_mouse.is_equal = is_equal;

    obj->data_conv_mouse.resume = resume;
    obj->data_conv_mouse.suspend = suspend;
    obj->data_conv_mouse.cleanup = cleanup;
    obj->data_conv_mouse.reconfigure = reconfigure;
    obj->data_conv_mouse.send_ptrmove = send_ptrmove;
    obj->data_conv_mouse.sync_mouse = sync_mouse;

    /* initializes our private data */
    obj->data_conv_mouse.data = calloc(1, sizeof(data_imouse_t));
    mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    assert(mouse_data);
    pthread_mutex_init(&mouse_data->move_mouse_corner_flag_mtx, NULL);
    mouse_data->current_cursor_shape.rgb = NULL;
    mouse_data->current_cursor_bg_shape.rgb = NULL;
    mouse_data->current_cursor_mask.rgb = NULL;
    mouse_data->grab_client = NULL;
    MUTEX_LOCK(&mouse_data->move_mouse_corner_flag_mtx);
    mouse_data->move_mouse_corner_flag = 0;
    MUTEX_UNLOCK(&mouse_data->move_mouse_corner_flag_mtx);
  
    /* init for all ports */
    //    cnt = eric_misc_kvm_get_nr_ports();
    pthread_mutex_init(&mouse_data->kvm_port_mtx, NULL);    
    for (kvm_unit=0; kvm_unit < PP_KVM_MAX_UNIT_COUNT; kvm_unit++) {
	if (pp_kvm_is_unit_present(kvm_unit)) {
	    alloc_port_state_for_kvm_unit(obj, kvm_unit);
	    reset_port_state_for_kvm_unit(obj, kvm_unit);
	}
    }

    pp_propchange_init_listener((pp_propchange_listener_t *)&mouse_data->propchange_listener, propchange_handler);
    mouse_data->propchange_listener.obj = obj;
    pp_propchange_add((pp_propchange_listener_t *)&mouse_data->propchange_listener, PP_PROP_KVM_PORT_SWITCHED);
    pp_propchange_add((pp_propchange_listener_t *)&mouse_data->propchange_listener, PP_PROP_VIDEO_MODE_CHANGED);
    pp_propchange_add((pp_propchange_listener_t *)&mouse_data->propchange_listener, PP_PROP_KVM_UNIT_ADD_OR_REMOVE);
    
#if defined(PP_FEAT_KITTY_CAT)
    /* create grabber client */
    mouse_data->grab_client = pp_grab_new_client(0, 0);

    /* use image understanding tracking */
    cat_condensation_init(obj);
    
    if(pp_cat_init(obj) == PP_ERR) {
	pp_log("Initializing CAT failed.\n");
        cleanup(obj);
	return -1;
    }
#endif /* PP_FEAT_KITTY_CAT */
    
    return 0;
}

static int resume(driver_t* obj) {
    data_imouse_t *mouse_data;
    assert(obj);
    mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    assert(mouse_data);
    
    /* reset the mouse position on the first move after resume
       (maybe we switched from USB or another driver and its now
       not in sync anmore)
       FIXME: better solution would be to update the movement
       counters even if the intelli driver is inactive */
    mouse_data->move_mouse_corner_flag = 1;

#if defined(PP_FEAT_KITTY_CAT)
    return cat_resume(obj);
#endif /* PP_FEAT_KITTY_CAT */

    return 0;
}

#if defined(PP_FEAT_KITTY_CAT)
static int suspend(driver_t* obj ) {
    return cat_suspend(obj);
}
#else
static int suspend(driver_t* /* obj */) {
    return 0;
}
#endif /* PP_FEAT_KITTY_CAT */

static int is_equal(driver_t* obj, driver_t* d) {
    assert(obj);
    assert(d);
    
    if (!strcmp(obj->data_conv_mouse.id, d->data_conv_mouse.id)) {
	return 1;
    } else {
	return 0;
    }
}

static int cleanup(driver_t* obj)
{
    int kvm_unit;
    data_imouse_t *mouse_data;

    assert(obj);
    mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    assert(mouse_data);
    
#if defined(PP_FEAT_KITTY_CAT)
    cat_cleanup(obj);

    /* remove grabber client */
    pp_grab_remove_client(mouse_data->grab_client);
#endif /* PP_FEAT_KITTY_CAT */

    pp_propchange_remove_all((pp_propchange_listener_t *)&mouse_data->propchange_listener);
    
    /* free kbd state mem for all units */
    for (kvm_unit=0; kvm_unit < PP_KVM_MAX_UNIT_COUNT; kvm_unit++) {
	free_port_state_for_kvm_unit(obj, kvm_unit);
    }

    /* free all data structures */
    free(mouse_data);

    return 0;
}

static int
alloc_port_state_for_kvm_unit(driver_t* obj, int kvm_unit)
{
    data_imouse_t *mouse_data;
    void *ptr;
    
    assert(obj);
    mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    assert(mouse_data);

    if (mouse_data->kvm_port_state_list_units[kvm_unit] == NULL) {
	/* allocate mem for mouse state for max nr of ports in obj unit */
	ptr = mouse_data->kvm_port_state_list_units[kvm_unit] =
	    (km_kvm_entry_t *) calloc(PP_KVM_MAX_UNIT_PORT_COUNT, sizeof(km_kvm_entry_t));
	KD("%s: %s(): alloc, pointer %d: %p\n", __FILE__, ___F, kvm_unit, ptr);
	if (!ptr) {
	    pp_log_err("%s: %s: allocating memory failed", __FILE__, ___F);
	    exit(1);
	}
    } else {
	KD("%s: %s(): unit %d was already allocated: %p\n", __FILE__, ___F, kvm_unit, mouse_data->kvm_port_state_list_units[kvm_unit]);
    }

    return 0;
}

/* resets the states of all ports in KVM_UNIT (if allocated) */
static int
reset_port_state_for_kvm_unit(driver_t* obj, int kvm_unit)
{
    int kvm_port, j;
    km_kvm_entry_t *unit, *port;   /* mouse data entries for ports of one unit */
    data_imouse_t *mouse_data;
    
    assert(obj);
    mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    assert(mouse_data);

    if (mouse_data->kvm_port_state_list_units[kvm_unit]) {
	KD("%s: %s(): resetting port state for unit %d\n", __FILE__, ___F, kvm_unit);

	/* initialize all ports in obj unit */
	unit = mouse_data->kvm_port_state_list_units[kvm_unit];
	if (unit) {
	    for (kvm_port = 0; kvm_port < PP_KVM_MAX_UNIT_PORT_COUNT; ++kvm_port) {
		port = &unit[kvm_port];
		port->index = kvm_port;
		port->u_index = kvm_unit;

		/* initialize mouse values */
		port->mouse_data.cur_x = 500;
		port->mouse_data.cur_y = -500;
		port->mouse_data.accel_steps = 1;
		for (j = 0; j < MAX_ACCEL_STEPS; ++j) {
		    port->mouse_data.m[j] = 1.0 * MOUSE_RASTER;
		    port->mouse_data.ymax[j] = 0;
		}
		port->mouse_data.fm_corr = non_mouse_corr; // 1:1 correction fkt;
		reconfigure_port(obj, kvm_unit, kvm_port);

		/* load mouse settings for obj port */	
		port->mouse_data.state = ERIC_MOUSE_IS_VALID;
		if (load_mouse_universe(obj, port) != PP_SUC) {
		    port->mouse_data.state = ERIC_MOUSE_NOT_YET;
		}
	    }
	}
    }

    return 0;
}

static int
free_port_state_for_kvm_unit(driver_t* obj, int kvm_unit)
{
    data_imouse_t * mouse_data;
    void * p;
    
    assert(obj);
    mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    assert(mouse_data);

    p = mouse_data->kvm_port_state_list_units[kvm_unit];
    /* unit not present -> release mem */
    KD("%s: %s: %d: freeing, pointer %p\n", __FILE__, ___F, kvm_unit, p);
    free(p);
    mouse_data->kvm_port_state_list_units[kvm_unit] = NULL;

    return 0;
}

static int
reconfigure(driver_t* obj)
{
    u_char kvm_unit;
    u_short kvm_port;

    assert(obj);
    
    for (kvm_unit = 0; kvm_unit < PP_KVM_MAX_UNIT_COUNT; ++kvm_unit) {	
        for (kvm_port = 0; kvm_port < PP_KVM_MAX_UNIT_PORT_COUNT; ++kvm_port) {
	    reconfigure_port(obj, kvm_unit, kvm_port);
	}
    }

    set_current_kvm_port(obj);

    return 0;
}

static void
reconfigure_port(driver_t* obj, u_char kvm_unit, u_short kvm_port)
{
    data_imouse_t* mouse_data;
    km_kvm_entry_t * unit, *port;
    char *var_val;

    assert(obj);
    mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    assert(mouse_data);

    unit = mouse_data->kvm_port_state_list_units[kvm_unit];
    if (!unit) return;
    port = &unit[kvm_port];
    
    /* get mouse mode for obj port */
    pp_cfg_get(&var_val, mouse_mode_key, kvm_unit, kvm_port);
    if (!strcmp(var_val, PP_CD_PORTPARAMS_MOUSE_MODE_DIRECT_STR)) {
	KD("set port %d to 1:n mode\n", kvm_port);
	port->mouse_data.mode = ERIC_MOUSE_OFF;
	port->mouse_data.cursor_state = ERIC_MOUSE_CURSOR_NOT_VALID;
	/* set direct mouse scaling for obj port */
	pp_cfg_get_float(&port->mouse_data.scaling,
			 mouse_direct_sc_key, 
			 kvm_unit, kvm_port);
    } else {
	KD("set port %d to auto\n", kvm_port);
	port->mouse_data.mode = ERIC_MOUSE_ON;
	port->mouse_data.cursor_state = ERIC_MOUSE_CURSOR_NOT_VALID;
	port->mouse_data.scaling = 1.0; // no scaling
    }
    free(var_val);
}

static int send_ptrmove(driver_t* obj,
			const int x, const int y, const int z, 
			const unsigned char buttonmask,
			const unsigned char abs_rel) {

    /* don't move mouse, because:
       - ptr is off TODO !obj->is_ptron(obj) || 
       - we are currently syncing */
    data_imouse_t* mouse_data;
    t_mouse_data* md;
    assert(obj);
    mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    assert(mouse_data);
    MUTEX_LOCK(&mouse_data->kvm_port_mtx);
    /* throw away mouse events as long as current port is invalid */
    if (mouse_data->current_kvm_port == NULL) {
	KD("%s: no current kvm port selected\n", ___F);
	goto bail;
    }
    md = &mouse_data->current_kvm_port->mouse_data;
    MUTEX_UNLOCK(&mouse_data->kvm_port_mtx);
    assert(md);
    
    if ((md->state == ERIC_MOUSE_GETTING)) {
	return 0;
    }
    
    if (abs_rel == PP_KM_PTR_MOVE_RELATIVE) {
	/* call functions for single mouse mode */
	return single_mouse_move(obj, x, y, z, buttonmask);
    } else {
	return intelligent_mouse_move(obj, x, y, z, buttonmask);
    }

    return 0;
bail:
    MUTEX_UNLOCK(&mouse_data->kvm_port_mtx);
    return 0;
}

static int single_mouse_move(driver_t* obj, const int x, const int y, const int z, 
			     const unsigned char buttonmask) {
    return move_mouse_direct(obj, x, y, z, buttonmask, 0);
}

static int intelligent_mouse_move(driver_t* obj,
				  const int x, const int y, const int z, 
				  const unsigned char buttonmask) {
    static const char* fn = ___F;
    data_imouse_t* mouse_data;
    t_mouse_data* md;
    int ret;
    km_event_t event;
    struct list_head *ptr;
    km_encoder_desc_data_t *enc_data;
    
    assert(obj);
    mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    assert(mouse_data);
    MUTEX_LOCK(&mouse_data->kvm_port_mtx);
    if (mouse_data->current_kvm_port == NULL) {
	KD("%s: no current kvm port selected\n", fn);
	goto bail;
    }
    md = &mouse_data->current_kvm_port->mouse_data;
    MUTEX_UNLOCK(&mouse_data->kvm_port_mtx);
    assert(md);

    KD("intelli_ptrmove: (x,y,z): (%d,%d,%d), button: %x, mode: %d, state: %d\n", x,y,z,buttonmask, md->mode, md->state);

    MUTEX_LOCK(&mouse_data->move_mouse_corner_flag_mtx);
    if (mouse_data->move_mouse_corner_flag) {
	move_mouse_corner(obj, mouse_data->old_buttonmask);
	mouse_data->move_mouse_corner_flag = 0;
    }
    MUTEX_UNLOCK(&mouse_data->move_mouse_corner_flag_mtx);

    // #warning what is necessary in case of only intelligent mouse?
    if ((md->mode  == ERIC_MOUSE_OFF) ||
	(md->state == ERIC_MOUSE_NOT_YET) ||
	(md->state == ERIC_MOUSE_NOT_NEEDED)) {
#if defined(PP_FEAT_KITTY_CAT)
        long move_x, move_y;
        
        /* kitty cat needs cursor shape which cannot be loaded */
        if(!cat_get_mousecursor_shape(obj)) {
            /* shape is valid, update position
               if we updated shape, position is (0, 0) */
            cat_update_mouse_position(obj);
        }
        
        move_x = x - (md->cur_x / MOUSE_RASTER);
        move_y = y - (md->cur_y / MOUSE_RASTER);
        
        cat_enqueue_ptr_move(obj, move_x, move_y);
	ret = move_mouse_direct(obj, move_x, move_y, z, buttonmask, 1);
#else /* !PP_FEAT_KITTY_CAT */
	ret = move_mouse_direct(obj, 
				x-md->cur_x/MOUSE_RASTER,
				y-md->cur_y/MOUSE_RASTER, z,
				buttonmask, 1);
#endif /* !PP_FEAT_KITTY_CAT */
	//	MUTEX_UNLOCK(&context->kvm_port->mtx);
	return ret;
    }

    /* we don't know about the mouse, so let's get knowledge */
    if (md->state == ERIC_MOUSE_NOT_VALID) {
        event.type = KM_EVENT_MOUSE_SYNC;
        event.event.mouse_sync.type = KM_MOUSE_SYNC_IN_PROGRESS;
        MUTEX_LOCK(&km_encoder_hook_mtx);
        list_for_each(ptr, &km_encoder_hook_list) {
            enc_data = list_entry(ptr, km_encoder_desc_data_t, listnode);
            if (enc_data->encoder_hook && enc_data->encoder_hook->km_event) {
            	enc_data->encoder_hook->km_event(enc_data->encoder_hook, obj->data_link, &event);
            }
        }
        MUTEX_UNLOCK(&km_encoder_hook_mtx);
	ret = get_mouse_universe(obj);
	switch(ret) {
	  case ERIC_MOUSE_UNIVERSE_OK:
              event.type = KM_EVENT_MOUSE_SYNC;
              event.event.mouse_sync.type = KM_MOUSE_SYNC_SUCCESS;
              MUTEX_LOCK(&km_encoder_hook_mtx);
              list_for_each(ptr, &km_encoder_hook_list) {
                  enc_data = list_entry(ptr, km_encoder_desc_data_t, listnode);
                  if (enc_data->encoder_hook && enc_data->encoder_hook->km_event) {
                      enc_data->encoder_hook->km_event(enc_data->encoder_hook, obj->data_link, &event);
                  }
              }
              MUTEX_UNLOCK(&km_encoder_hook_mtx);
	      break;
	  case ERIC_MOUSE_UNIVERSE_FAILED:
	      md->state = ERIC_MOUSE_NOT_YET;
              event.type = KM_EVENT_MOUSE_SYNC;
              event.event.mouse_sync.type = KM_MOUSE_SYNC_FAILED;
              MUTEX_LOCK(&km_encoder_hook_mtx);
              list_for_each(ptr, &km_encoder_hook_list) {
                  enc_data = list_entry(ptr, km_encoder_desc_data_t, listnode);
                  if (enc_data->encoder_hook && enc_data->encoder_hook->km_event) {
                      enc_data->encoder_hook->km_event(enc_data->encoder_hook, obj->data_link, &event);
                  }
              }
              MUTEX_UNLOCK(&km_encoder_hook_mtx);
	      //MUTEX_UNLOCK(&context->kvm_port->mtx);
	      return 0;   	      
	  case ERIC_MOUSE_UNIVERSE_WARN_GOOD:
              event.type = KM_EVENT_MOUSE_SYNC;
              event.event.mouse_sync.type = KM_MOUSE_SYNC_AMBIGUOUS_GOOD;
              MUTEX_LOCK(&km_encoder_hook_mtx);
              list_for_each(ptr, &km_encoder_hook_list) {
                  enc_data = list_entry(ptr, km_encoder_desc_data_t, listnode);
                  if (enc_data->encoder_hook && enc_data->encoder_hook->km_event) {
                      enc_data->encoder_hook->km_event(enc_data->encoder_hook, obj->data_link, &event);
                  }
              }
              MUTEX_UNLOCK(&km_encoder_hook_mtx);
	      break;
	  case ERIC_MOUSE_UNIVERSE_WARN_BAD:
              event.type = KM_EVENT_MOUSE_SYNC;
              event.event.mouse_sync.type = KM_MOUSE_SYNC_AMBIGUOUS_BAD;
              MUTEX_LOCK(&km_encoder_hook_mtx);
              list_for_each(ptr, &km_encoder_hook_list) {
                  enc_data = list_entry(ptr, km_encoder_desc_data_t, listnode);
                  if (enc_data->encoder_hook && enc_data->encoder_hook->km_event) {
                      enc_data->encoder_hook->km_event(enc_data->encoder_hook, obj->data_link, &event);
                  }
              }
              MUTEX_UNLOCK(&km_encoder_hook_mtx);
	      break;
	}
	if (save_mouse_universe(obj) != PP_SUC) {
	    pp_log_err("%s: could not save mouse universe\n", fn);
	}
#if defined(PP_FEAT_KITTY_CAT)
    } else {
        /* kitty cat needs cursor shape which cannot be loaded */
        cat_get_mousecursor_shape(obj);
#endif /* PP_FEAT_KITTY_CAT */
    }

    ret = move_mouse(obj, x, y, z, buttonmask);

    return ret;
bail:
    MUTEX_UNLOCK(&mouse_data->kvm_port_mtx);
    return 0;
}

static int
sync_mouse(driver_t* obj, const int sync_type)
{
    int ret = 0;

    data_imouse_t* mouse_data;
    t_mouse_data* md;
    
    assert(obj);
    mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    assert(mouse_data);
    MUTEX_LOCK(&mouse_data->kvm_port_mtx);
    if (mouse_data->current_kvm_port == NULL) {
	KD("%s: no current kvm port selected\n", ___F);
	goto bail;
    }
    md = &mouse_data->current_kvm_port->mouse_data;
    MUTEX_UNLOCK(&mouse_data->kvm_port_mtx);
    assert(md);
    
    if (md->mode == ERIC_MOUSE_ON
	&& (sync_type == PP_KM_MOUSE_SYNC_HARD ||
	    (sync_type == PP_KM_MOUSE_SYNC_NORM &&
	     (md->state == ERIC_MOUSE_NOT_YET || md->state == ERIC_MOUSE_VALID_UNSYNC)))) {
	
	/* clear cursor and table -> redetect on next movement */
	md->cursor_state = ERIC_MOUSE_CURSOR_NOT_VALID;
	md->state	 = ERIC_MOUSE_NOT_VALID;
	
    } else if (sync_type == PP_KM_MOUSE_SYNC_NORM
	       || sync_type == PP_KM_MOUSE_SYNC_FAST) {
	/* move the mouse to its old position */
	long x = md->cur_x / MOUSE_RASTER;
	long y = md->cur_y / MOUSE_RASTER;
	move_mouse_corner(obj, mouse_data->old_buttonmask);
	if ((md->state == ERIC_MOUSE_NOT_NEEDED) ||
	    (md->state == ERIC_MOUSE_NOT_YET) ||
	    (md->mode  == ERIC_MOUSE_OFF)) {
	    move_mouse_direct(obj, x, y, 0, mouse_data->old_buttonmask, 1);
	} else {
	    move_mouse(obj, x, y, 0, mouse_data->old_buttonmask);
	}
    } else if (sync_type == PP_KM_MOUSE_SYNC_ACCEL_STATE) {
	char * accel_opt = NULL;
	u_char kvm_unit;
	u_short kvm_port;
	/* turn direct mouse mode on or off, reread scaling factor */
	/* FIXME: use (unit, port) instead */
	if (pp_kvm_get_unit_port_for_data_link(obj->data_link, &kvm_unit, &kvm_port) == 0) {
	    pp_cfg_get(&accel_opt, mouse_mode_key, kvm_unit, kvm_port);
	    if (!strcmp("direct", accel_opt)) {
		md->mode  = ERIC_MOUSE_OFF;
		pp_cfg_get_float(&md->scaling, mouse_direct_sc_key, kvm_port);
	    } else {
		int eqz_preset;
		int eqz_enabled;
		md->mode = ERIC_MOUSE_ON;
	    
		// check whether gud equalizer options have changed
		pp_cfg_get_int(&eqz_preset, "unit[%u].port[%u].mouse.gud_eqz_preset", kvm_unit, kvm_port);
		pp_cfg_is_enabled(&eqz_enabled, "unit[%u].port[%u].mouse.gud_eqz", kvm_unit, kvm_port);
		if (eqz_enabled) {
		    if (md->fm_corr.id == 0) md->state = ERIC_MOUSE_NOT_YET;
		} else if (md->fm_corr.id != 0) {
		    md->fm_corr = non_mouse_corr;
		    md->state = ERIC_MOUSE_NOT_YET;
		}
                md->scaling = 1.0; // no scaling
	    }
	    free(accel_opt);
	}
    }

    return ret;
bail:
    MUTEX_UNLOCK(&mouse_data->kvm_port_mtx);
    return 0;
}


static void
propchange_handler(pp_propchange_listener_t * listener, 
		   unsigned short prop, unsigned short /* prop_flags */)
{
    local_propchange_listener_t *mylistener;
    driver_t* obj;
    data_imouse_t* mouse_data;
    int kvm_unit;
    
    mylistener = (local_propchange_listener_t *)listener;
    obj = mylistener->obj;
    assert(obj);
    mouse_data = (data_imouse_t *)obj->data_conv_mouse.data;
    assert(mouse_data);

#if defined(PP_FEAT_KITTY_CAT)
    cat_reset_tracking(obj);
#endif /* PP_FEAT_KITTY_CAT */

    switch (prop) {
      case PP_PROP_KVM_PORT_SWITCHED:
	set_current_kvm_port(obj);
	break;
      case PP_PROP_VIDEO_MODE_CHANGED:
	MUTEX_LOCK(&mouse_data->move_mouse_corner_flag_mtx);
	mouse_data->move_mouse_corner_flag = 1;
	MUTEX_UNLOCK(&mouse_data->move_mouse_corner_flag_mtx);
	break;
      case PP_PROP_KVM_UNIT_ADD_OR_REMOVE:
	assert(obj);
	mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
	assert(mouse_data);
	for (kvm_unit = 0; kvm_unit < PP_KVM_MAX_UNIT_COUNT; ++kvm_unit) {
	    if (mouse_data->kvm_port_state_list_units[kvm_unit] == NULL && pp_kvm_is_unit_present(kvm_unit)) {
		// unit was added
		alloc_port_state_for_kvm_unit(obj, kvm_unit);
		reset_port_state_for_kvm_unit(obj, kvm_unit);
	    } else if (mouse_data->kvm_port_state_list_units[kvm_unit] != NULL && !pp_kvm_is_unit_present(kvm_unit)) {
		// unit was removed
		free_port_state_for_kvm_unit(obj, kvm_unit);
	    }
	}
	/*
	 * We need to call this here because PP_PROP_KVM_PORT_SWITCHED
	 * might be received earlier and is ignored in this case.
	 */
	set_current_kvm_port(obj);
	break;
      default:
	pp_log("%s: %s(): Unhandled property %hu received.\n", __FILE__, ___F, prop);
	break;
    }
}

static void
set_current_kvm_port(driver_t* obj)
{
    data_imouse_t* mouse_data;
    u_char kvm_unit;
    u_short kvm_port;

    assert(obj);
    mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    assert(mouse_data);
    if (pp_kvm_get_unit_port_for_data_link(obj->data_link, &kvm_unit, &kvm_port) == 0) {
	if (mouse_data->kvm_port_state_list_units[kvm_unit] != NULL) {
	    KD("%s(): set current_kvm_port for data_link %hhu: to %hu (unit %hhu)\n", ___F, obj->data_link, kvm_port, kvm_unit);
	    MUTEX_LOCK(&mouse_data->kvm_port_mtx);
	    mouse_data->current_kvm_port = &mouse_data->kvm_port_state_list_units[kvm_unit][kvm_port];
    	    MUTEX_UNLOCK(&mouse_data->kvm_port_mtx);
	}
    } else {
	KD("%s(): pp_kvm_unit_port_data_link() failed\n", ___F);
	MUTEX_LOCK(&mouse_data->kvm_port_mtx);
	mouse_data->current_kvm_port = NULL;
    	MUTEX_UNLOCK(&mouse_data->kvm_port_mtx);
    }
}
