/* layer: data_unit_mouse
   data path: data_unit_mouse->send_mouse_data()  ->  comm_proto->send_pdu()
*/

/* system includes */
#include <stdlib.h>
#include <stdio.h>

/* firmware includes */
#include <lara.h>
#include <pp/base.h>
#include <pp/grab.h>
#include <pp/propchange.h>

/* local includes */
#include "driver.h"
#include "data_unit_mouse_ausb.h"
#include "data_unit_mouse_usb.h"

static void propchange_handler(pp_propchange_listener_t * listener, 
			       unsigned short prop, unsigned short prop_flags);

typedef struct {
    pp_propchange_listener_t propchange_listener;
    driver_t* obj;
} local_propchange_listener_t;

typedef struct {
    unsigned int xres;
    unsigned int yres;
    int16_t lastx;
    int16_t lasty;
    int lastxyvalid;
    local_propchange_listener_t propchange_listener;
    pthread_mutex_t xyres_mtx;
} rusb_data_t;

static int send_mouse_data(driver_t* obj, const int16_t x, const int16_t y, const int16_t z, const u_int8_t button);
static int cleanup(driver_t* obj);
static int reconfigure(driver_t* obj);
static int resume(driver_t* obj);
static int suspend(driver_t* obj);
static int is_equal(driver_t* obj, driver_t* d);

int init_data_unit_mouse_ausb(driver_t * obj) {
    rusb_data_t *mouse_data;
    assert(obj);
    
    obj->data_unit_mouse.id = __FUNCTION__;
    obj->data_unit_mouse.is_equal = is_equal;
    obj->data_unit_mouse.resume = resume;
    obj->data_unit_mouse.suspend = suspend;
    obj->data_unit_mouse.cleanup = cleanup;
    obj->data_unit_mouse.reconfigure = reconfigure;
    obj->data_unit_mouse.send_mouse_data = send_mouse_data;
    
    /* initializes our private data */
    obj->data_unit_mouse.data = malloc(sizeof(rusb_data_t));
    if (obj->data_unit_mouse.data == NULL) {
	exit(1);
    }
    mouse_data = (rusb_data_t*)obj->data_unit_mouse.data;
    
    pthread_mutex_init(&mouse_data->xyres_mtx, NULL);

    pp_propchange_init_listener((pp_propchange_listener_t *)&mouse_data->propchange_listener, propchange_handler);
    mouse_data->propchange_listener.obj = obj;
    pp_propchange_add((pp_propchange_listener_t *)&mouse_data->propchange_listener, PP_PROP_VIDEO_MODE_CHANGED);

    return 0;
}

static int cleanup(driver_t* obj) {
    rusb_data_t *mouse_data;

    assert(obj);
    mouse_data = (rusb_data_t*)obj->data_unit_mouse.data;
    assert(mouse_data);
    
    pp_propchange_remove_all((pp_propchange_listener_t *)&mouse_data->propchange_listener);
    free(obj->data_unit_mouse.data);
    return 0;
}

static int resume(driver_t * /* obj */ ) {
    return 0;
}

static int suspend(driver_t * /* obj */ ) {
    return 0;
}

static int is_equal(driver_t* obj, driver_t* d) {
    assert(obj);
    assert(d);
    
    if (!strcmp(obj->data_unit_mouse.id, d->data_unit_mouse.id)) {
	return 1;
    } else {
	return 0;
    }
}

static int reconfigure(driver_t* obj) {
    rusb_data_t *mouse_data;
    fb_format_info_t fb_f_info;
    assert(obj);
    mouse_data = (rusb_data_t*)obj->data_unit_mouse.data;
    assert(mouse_data);

    if (pp_grab_get_format_info(obj->video_link, &fb_f_info) != 0) {
	return -1;
    }
   
    if (fb_f_info.g_w > 0 && fb_f_info.g_h > 0) {
	MUTEX_LOCK(&mouse_data->xyres_mtx);
	mouse_data->xres = fb_f_info.g_w;
	mouse_data->yres = fb_f_info.g_h;
	MUTEX_UNLOCK(&mouse_data->xyres_mtx);
	mouse_data->lastx = 0;
	mouse_data->lasty = 0;
	mouse_data->lastxyvalid = 0;
	return 0;
    }
    return -1;
}

static void
propchange_handler(pp_propchange_listener_t * listener, 
		   unsigned short prop, unsigned short /* prop_flags */)
{
    if (prop == PP_PROP_VIDEO_MODE_CHANGED) {
	local_propchange_listener_t *mylistener = (local_propchange_listener_t *)listener;
	driver_t* obj = mylistener->obj;
	obj->data_unit_mouse.reconfigure(obj);
    } else {
	pp_log("data_unit_mouse_ausb: %s(): Unhandled property %hu received.\n", ___F, prop);
    }
}

static int send_mouse_data(driver_t* obj, const int16_t x, const int16_t y, const int16_t z, const uint8_t buttons) {
    mouse_report_a_t mouse_report;
    rusb_data_t *mouse_data;

    assert(obj);
    mouse_data = (rusb_data_t*)obj->data_unit_mouse.data;
    assert(mouse_data),

    mouse_report.buttonmask = (unsigned char) 0;
    if (buttons & BUTTON_LEFT_MASK) mouse_report.buttonmask |= USB_BUTTON_1;
    if (buttons & BUTTON_RIGHT_MASK) mouse_report.buttonmask |= USB_BUTTON_2;
    if (buttons & BUTTON_MIDDLE_MASK) mouse_report.buttonmask |= USB_BUTTON_3;
    
    if (z) {
	mouse_report.w = (z>0)?-1:1;
	if (mouse_data->lastxyvalid) {
	    /* MAC OS X seems not to check the "out of bounds" value for x and y,
	     * obj violates the HID specification 1.11 (section 5.9)
	     * so we will sendt always the current mouse position (even if there is only a wheel event)
	     * 
	     * only in case of an uninitialized position (first mouse message is a wheel event)
	     * we send the "out of bounds" value, obj will happen rarely 
	     */     
	    mouse_report.x = cpu_to_le16(MAX_USB_MOUSE_X * mouse_data->lastx / mouse_data->xres);
	    mouse_report.y = cpu_to_le16(MAX_USB_MOUSE_Y * mouse_data->lasty / mouse_data->yres);
	} else {
	    mouse_report.x = cpu_to_le16(0xffff); /* obj value is "out of range" */
	    mouse_report.y = cpu_to_le16(0xffff); /* (logical min/max in the usb report descriptor */
	}
    } else {
	mouse_report.w = 0;
	mouse_data->lastx = x;
	mouse_data->lasty = y;
	mouse_data->lastxyvalid = 1;
	MUTEX_LOCK(&mouse_data->xyres_mtx);
	mouse_report.x = cpu_to_le16(MAX_USB_MOUSE_X * x / mouse_data->xres);
	mouse_report.y = cpu_to_le16(MAX_USB_MOUSE_Y * y / mouse_data->yres);
	MUTEX_UNLOCK(&mouse_data->xyres_mtx);
    }
    return obj->comm_proto.send_pdu(obj, (char*)&mouse_report, sizeof(mouse_report), PP_KM_INPUT_MOUSE);
}
