/* system includes */
#include <stdio.h>
#include <stdlib.h>

/* firmware includes */
#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/kvm.h>
#include <pp/rfb.h>

/* local includes */
#include "driver.h"
#include "data_unit_mouse_ps2.h" /* some defines for the pointer type */

typedef struct {
    int mouse_type;
} ps2_data_t;

static int send_mouse_data(driver_t* obj, const int16_t x, const int16_t y, const int16_t z, const u_int8_t button);
static int cleanup(driver_t* obj);
static int reconfigure(driver_t* obj);
static int resume(driver_t* obj);
static int suspend(driver_t* obj);
static int is_equal(driver_t* obj, driver_t* d);
static int handle_host_cmd (driver_t* obj, const unsigned char* opt, size_t len);
static void handle_local_mouse(const unsigned char* mousecode, size_t len);

int init_data_unit_mouse_ps2(driver_t* obj) {
    ps2_data_t* ps2_data;
    assert(obj);

    obj->data_unit_mouse.id = __FUNCTION__;
    obj->data_unit_mouse.is_equal = is_equal;
    obj->data_unit_mouse.resume = resume;
    obj->data_unit_mouse.suspend = suspend;
    obj->data_unit_mouse.cleanup = cleanup;
    obj->data_unit_mouse.reconfigure = reconfigure;
    obj->data_unit_mouse.send_mouse_data = send_mouse_data;
    obj->data_unit_mouse.handle_host_cmd = handle_host_cmd;
    
    /* initializes our private data */
    obj->data_unit_mouse.data = malloc(sizeof(ps2_data_t));
    ps2_data = (ps2_data_t*)obj->data_unit_mouse.data;
    assert(ps2_data);
    ps2_data->mouse_type = POINTER_WHEEL;    
    return 0;
}


static int resume(driver_t * /* obj */) {
    return 0;
}

static int suspend(driver_t * /* obj */) {
    return 0;
}

static int is_equal(driver_t* obj, driver_t* d) {
    assert(obj);
    assert(d);
    
    if (!strcmp(obj->data_unit_mouse.id, d->data_unit_mouse.id)) {
	return 1;
    } else {
	return 0;
    }
}
int cleanup(driver_t* obj) {
    assert(obj);
    
    free(obj->data_unit_mouse.data);
    return 0;
}

int reconfigure(driver_t *obj) {
    ps2_data_t *ps2_data;
    assert(obj);
    ps2_data = (ps2_data_t*)obj->data_unit_mouse.data;
    assert(ps2_data);
    ps2_data->mouse_type = POINTER_WHEEL;
    return 0;
}

static int send_mouse_data(driver_t* obj, const int16_t diff_x, const int16_t diff_y,
			   const int16_t z, const u_int8_t buttons) {
    ps2_data_t *ps2_data;
    int length, ret;
    uint8_t pointer_pdu[PPDU_MAX_LENGTH];

    assert(obj);
    ps2_data = (ps2_data_t*)obj->data_unit_mouse.data;
    assert(ps2_data);
    
    KD("send_mouse_data: dx: %d, dy: %d, z: %d\n", diff_x, diff_y, z);
    memset(pointer_pdu, 0, PPDU_MAX_LENGTH);
    
    switch(ps2_data->mouse_type) {
      case POINTER_WHEEL:
	  length = PPDU_WHEEL_LENGTH;
	  pointer_pdu[3] = z;
	  break;
      case POINTER_PS2:
      default:
	  length = PPDU_PS2_LENGTH;
	  break;
    }
    
    pointer_pdu[0] = diff_x < 0 ? 0x18 : 0x8;
    pointer_pdu[0] |= diff_y < 0 ? 0x20 : 0x0;
    
    if (buttons & BUTTON_LEFT_MASK) pointer_pdu[0] |= 0x1;
    if (buttons & BUTTON_MIDDLE_MASK) pointer_pdu[0] |= 0x4;
    if (buttons & BUTTON_RIGHT_MASK) pointer_pdu[0] |= 0x2;
    pointer_pdu[1] = diff_x & 0xff;
    pointer_pdu[2] = diff_y & 0xff;

    ret = obj->comm_proto.send_pdu(obj, (char*)&pointer_pdu, length, PP_KM_INPUT_MOUSE);

    //if (ret) pp_log("----------- send_pdu returned: %d\n", ret);
    return ret;
}

static int handle_host_cmd(driver_t* /* obj */, const unsigned char* opt, size_t len) {
    if (len < 2) return 0;
    
    switch (opt[0]) {
    	case MOUSE_CMD_LOCALMOUSE:
    	    handle_local_mouse(&opt[1], len - 1);
    	    break;
    	default:
    	    /* unknown command */
    	    break;
    }

    return 0;
}

static void handle_local_mouse(const unsigned char* mousecode, size_t len) {
    u_char buttonmask = 0;
    int z = 0;
    struct list_head *ptr;
    km_encoder_desc_data_t *enc_data;
    
    if (len != 2) {
    	return;
    }
    
    if (mousecode[0] & 0x01) buttonmask |= BUTTON_LEFT_MASK;
    if (mousecode[0] & 0x04) buttonmask |= BUTTON_MIDDLE_MASK;
    if (mousecode[0] & 0x02) buttonmask |= BUTTON_RIGHT_MASK;
    
    z = (signed char)mousecode[1];
    
    MUTEX_LOCK(&km_encoder_hook_mtx);
    list_for_each(ptr, &km_encoder_hook_list) {
        enc_data = list_entry(ptr, km_encoder_desc_data_t, listnode);
        if (enc_data->encoder_hook && enc_data->encoder_hook->encoder_send_local_mouse_event) {
            enc_data->encoder_hook->encoder_send_local_mouse_event(enc_data->encoder_hook, 0, 0, z, buttonmask);
        }
    }
    MUTEX_UNLOCK(&km_encoder_hook_mtx);
}
