#ifndef _KM_DRIVER_H
#define _KM_DRIVER_H

/* system includes */
#include <sys/types.h>

/* our public interface */
#include <pp/features.h>

#include <pp/km.h>
#include <pp/usb.h>

/* TODO: obj should go in a lib global include */
/* ----------------- debug -------------------------------- */
#define noKBD_DEBUG
#ifdef KBD_DEBUG
#define KD(fmt, args...)	{ printf(fmt, ##args); }
#else
#define KD(fmt, args...)
#endif

typedef enum {
    TARGET_TYPE_ERICFD,
    TARGET_TYPE_I2C,
    TARGET_TYPE_DUMMY /* does nothing */ 
} target_type_t;
   
typedef struct {
    int dummy;
} target_para_unix_t;

typedef struct {
} target_para_chardev_t;
    
typedef struct {
    target_type_t target_type;
    char* path;
    union {
	target_para_unix_t unixdomain;
	target_para_chardev_t ericdev;
    } para;
} target_para_t;

typedef enum {
    TARGET_ERIC,
    TARGET_KME,
    TARGET_KMESENSE,
    TARGET_USB
} driver_type_t;

typedef enum {
    DRIVER_NOT_INITIALIZED,
    /* all other are initialized */
    DRIVER_SUSPENDED,
    DRIVER_RUNNING
} driver_state_t;

typedef struct _driver_t driver_t;

typedef struct {
    /* intelligent, direct, aUSB */
    void* data;
    const char* id;
    int (*is_equal) (driver_t* obj, driver_t* d);
    int (*suspend) (driver_t*);
    int (*resume) (driver_t*);
    int (*cleanup) (driver_t* obj);
    int (*reconfigure) (driver_t* obj);
    int (*reset_driver) (driver_t* obj);
    int (*sync_mouse) (driver_t* obj, const int sync_type);
    int (*send_ptrmove) (driver_t* obj,
			 const int x, const int y, const int z, 
			 const unsigned char buttonmask,
			 const unsigned char abs_rel);
} data_conv_mouse_t;

typedef struct {
    void* data;
    const char* id;
    int (*is_equal) (driver_t* obj, driver_t* d);
    int (*suspend) (driver_t*);
    int (*resume) (driver_t*);
    int (*cleanup) (driver_t* obj);
    int (*reconfigure) (driver_t* obj);
    int (*reset_driver) (driver_t* obj);
    int (*send_keycodes) (driver_t* obj,
			  const unsigned char keys[], const int length);
} data_conv_kbd_t;

typedef struct {
    /* ps2, USB */
    void* data;
    const char* id;
    int (*is_equal) (driver_t* obj, driver_t* d);
    int (*suspend) (driver_t*);
    int (*resume) (driver_t*);
    int (*cleanup) (driver_t* obj);
    int (*reconfigure) (driver_t* obj);
    int (*send_kbd_data) (driver_t*, const u_char key[], const int length, void* status);
    const char * (*get_info) (driver_t*, const char* parameter);
    int (*handle_host_cmd) (driver_t*, const unsigned char* opt, size_t len);
    int (*reset_driver) (driver_t* obj);
    int (*get_led_status) (driver_t* obj);
} data_unit_kbd_t;

typedef struct {
    /* ps2, rUSB, aUSB */
    void* data;
    const char* id;
    int (*is_equal) (driver_t* obj, driver_t* d);
    int (*suspend) (driver_t*);
    int (*resume) (driver_t*);
    int (*cleanup) (driver_t* obj);
    int (*reconfigure) (driver_t* obj);
    int (*send_mouse_data) (driver_t* obj, const int16_t x, const int16_t y, const int16_t z, const u_int8_t button);
    int (*handle_host_cmd) (driver_t*, const unsigned char* opt, size_t len);
    int (*reset_driver) (driver_t* obj);
} data_unit_mouse_t;

typedef struct {
    void* data;
    const char* id;
    int (*is_equal) (driver_t* obj, driver_t* d);
    int (*send_pdu) (driver_t*, const void* pdu, const int length, const input_t input);
    int (*suspend) (driver_t*);
    int (*resume) (driver_t*);
    int (*reconfigure) (driver_t* obj);
    int (*ping) (driver_t* obj, char** info);
    int (*cleanup) (driver_t* obj);
    int (*reset_driver) (driver_t* obj);
    int (*send_cmd_ack) (driver_t* obj);
    int (*rw_pin) (driver_t* obj, int is_write, int pin, int *val);
} comm_proto_t;

typedef struct {
    void* data;
    const char* id;
    int (*is_equal) (driver_t* obj, driver_t* d);
    int (*get_open_fd) (driver_t*);
    int (*reconfigure) (driver_t* obj);
    int (*close_fd) (driver_t*);
    int (*cleanup) (driver_t* obj);
    int (*use_file) (driver_t* obj, const char* file);
} target_helper_t;

struct _driver_t {
    u_char data_link;
    u_char video_link;

    int (*is_equal)(driver_t *obj, driver_t* d);
    driver_state_t state;
    pthread_mutex_t state_mtx;
    data_conv_mouse_t data_conv_mouse;
    data_conv_kbd_t data_conv_kbd;
    
    data_unit_mouse_t data_unit_mouse;
    data_unit_kbd_t data_unit_kbd;
    comm_proto_t comm_proto;
    target_helper_t target_helper;
};

/* Global variables */
typedef struct _km_encoder_desc_data_t {
    struct list_head listnode;
    pp_km_encoder_desc_t* encoder_hook;
} km_encoder_desc_data_t;
extern pthread_mutex_t km_encoder_hook_mtx;
extern struct list_head km_encoder_hook_list;

extern volatile int ignore_usb_propchange;
extern pthread_mutex_t usb_propchange_mutex;
extern pthread_cond_t usb_suspend_cond;
extern pthread_cond_t usb_resume_cond;

/*********************************************/
/* intern functions - mostly driver specific */
/*********************************************/

/* driver factory functions */
int init_data_conv_kbd(driver_t* obj);
int init_data_conv_mouse_ausb(driver_t* obj);
int init_data_conv_imouse(driver_t* obj);
int init_data_conv_dmouse(driver_t* obj);
#if defined(PP_FEAT_CAT)
int init_data_conv_cat(driver_t* obj);
#endif /* PP_FEAT_CAT */

int init_data_unit_mouse_ausb(driver_t* obj);
int init_data_unit_mouse_rusb(driver_t* obj);
int init_data_unit_mouse_ps2(driver_t* obj);
int init_data_unit_kbd_usb(driver_t* obj);
int init_data_unit_kbd_ps2(driver_t* obj, int default_scset);

int init_comm_proto_usb(driver_t* obj, usb_device_t dev, usb_device_mouse_type_t mouse_type);
int init_comm_proto_pppp_i2c(driver_t* obj, int reset_ioctl_fd);
int init_comm_proto_minicom(driver_t* obj);
int init_comm_proto_cim(driver_t* obj, int link, int cim_type, int port);

int init_target_helper_ericfd(driver_t* obj, target_para_t* tpt);
int init_target_helper_i2c(driver_t* obj, target_para_t* tpt);
int init_target_helper_dummy(driver_t* obj, target_para_t* tpt);

#endif /* _KM_DRIVER_H */
