/* system includes */
#include <arpa/inet.h>

#include <pthread.h>
#include <pp/base.h>
#include <pp/propchange.h>
#include <pp/kvm.h>
#include <pp_kernel_common.h>
#include <pp/um.h>

/* local includes */
#include "kvm_driver.h"
#include "kvm_defaults.h"
#include "debug.h"

using namespace pp;

KvmDriver::KvmDriver()
{
    pthread_mutex_t init_mtx = PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;
    u_char i, unit;

    DP_VERBOSE("%s called\n", ___F);

    /* assign initializer struct (not a pointer) */
    this->_currPortMtx = init_mtx;
    this->_switchPortMtx = init_mtx;
    this->_portsInUnitMtx = init_mtx;

    for (i = 0; i < PP_KVM_MAX_CHANNEL_COUNT; ++i) {
		this->_currPort[i] = this->_currUnit[i] = 0;
    }

    for (unit = 0; unit < PP_KVM_MAX_UNIT_COUNT; ++unit) {
		this->_portCountInUnit[unit] = 0;
    }
}

KvmDriver::~KvmDriver()
{
    /* destroy non-inherited members */
    /* destroy superclass */
}

int
KvmDriver::getUnitAndPortForKvmNode(u_char kvm_node, int *unit_out, int *port_out)
{
    pthread_mutex_t *mtx = &this->_currPortMtx;

    assert(unit_out);
    assert(port_out);

    if (kvm_node < PP_KVM_MAX_CHANNEL_COUNT) {
	MUTEX_LOCK(mtx);
	/* return the cached unit/port index (e.g. as last sent by kvm switch) */
	*unit_out = this->_currUnit[kvm_node];
	*port_out = this->_currPort[kvm_node];
	MUTEX_UNLOCK(mtx);
    }

    return 0;
}

int
KvmDriver::getUnitAndPortForDataLink(int data_link_id, int *unit_out, int *port_out)
{
    return getUnitAndPortForKvmNode(data_link_id, unit_out, port_out);
}

int
KvmDriver::getUnitAndPortForVideoLink(int video_link_id, int *unit_out, int *port_out)
{
    return getUnitAndPortForKvmNode(video_link_id, unit_out, port_out);
}

u_short
KvmDriver::GetUnitPortCount( u_char unit )
{
    u_short count = 0;
    pthread_mutex_t *mtx = &this->_portsInUnitMtx;

	if (unit < PP_KVM_MAX_UNIT_COUNT) {
		MUTEX_LOCK(mtx);
		count = this->_portCountInUnit[unit];
		MUTEX_UNLOCK(mtx);
	}

    return count;
}

void
KvmDriver::SetUnitPortCount( u_char unit, u_short count )
{
    pthread_mutex_t *mtx = &this->_portsInUnitMtx;

	if (unit < PP_KVM_MAX_UNIT_COUNT) {
		DP_NOTICE("%s(): unit %hhu: setting num of hosts to %hu\n", ___F, unit, count);

		MUTEX_LOCK(mtx);
		this->_portCountInUnit[unit] = count;
		MUTEX_UNLOCK(mtx);

		/* notify listeners */
		pp_propchange_enqueue(PP_PROP_KVM_UNIT_ADD_OR_REMOVE, 0);
	}
}

void
KvmDriver::PortChanged( u_char channel, u_char unit, u_short port )
{
	if (channel < PP_KVM_MAX_CHANNEL_COUNT && unit < PP_KVM_MAX_UNIT_COUNT) {
		DP_VERBOSE("%s: unit/port changed on channel %hhu to %hhu/%hu\n", ___F, channel, unit, port);

		MUTEX_LOCK(&this->_currPortMtx);
		this->_currUnit[channel] = unit;
		this->_currPort[channel] = port;
		MUTEX_UNLOCK(&this->_currPortMtx);

		/* notify listeners */
		if (channel != (u_char)pp_misc_get_kvm_node()) {
			pp_propchange_enqueue(PP_PROP_KVM_PORT_SWITCHED, 0);
		}
	}
}

int
KvmDriver::getAnyVscVideoLinkForDataLink(int data_link_id, int *video_link_id)
{
    if (data_link_id >= 0) {
	/* subclasses of this have the notion of "kvm nodes" rather than
	 * video/data links; each node has DL == VL by definition */
	*video_link_id = data_link_id;
	return PP_SUC;
    } else {
	return PP_ERR;
    }
}

int
KvmDriver::IsHostAtPort(u_char unit UNUSED, u_short port UNUSED)
{
    /*
     * Return true for _each_ port.  Subclasses may overwrite this, e.g. if
     * they are able to detect hosts.
     */
    return 1;
}

int
KvmDriver::IsPortAllowedForUser(const char * user UNUSED, u_char unit UNUSED, u_short port UNUSED)
{
    /* all users allowed on all ports (unless subclass says otherwise) */
    return 1;
}

bool
KvmDriver::isSwitchToPortAllowed(const char *user, u_char unit, u_short port)
{
    int user_ok, good_port;

    user_ok =  IsPortAllowedForUser(user, unit, port);
    good_port = IsHostAtPort(unit, port);

    return user_ok && good_port;
}

void
KvmDriver::ChangeIpAddress( uint32_t ip, int ip_valid, uint32_t mask, int mask_valid, uint32_t gw, int gw_valid)
{
    char iptext[INET_ADDRSTRLEN];
    char masktext[INET_ADDRSTRLEN];
    char gwtext[INET_ADDRSTRLEN];

    if (( ip_valid  && (!inet_ntop(AF_INET, &ip, iptext, sizeof(iptext)))) ||
        ( mask_valid && (!inet_ntop(AF_INET, &mask, masktext, sizeof(masktext)))) ||
        ( gw_valid  && (!inet_ntop(AF_INET, &gw, gwtext, sizeof(gwtext))))
       ) {
       pp_log("wrong IP address format: %08x, %08x, %08x\n", ip, mask, gw);
    }

    /* reject 255.255.255.255 as netmask */
    if (mask == 0xffffffff) {
	pp_log("%s(): wrong netmask : 255.255.255.255\n", ___F);
	return;
    }
    pp_cfg_set("None", "network.ip_auto_config_proto");
    pp_cfg_set(iptext, "network.ipaddr");
    pp_cfg_set(masktext, "network.netmask");
    pp_cfg_set(gwtext, "network.gateway");
    pp_cfg_save(DO_FLUSH);
}

int
KvmDriver::setLinkClientLinks(LinkClient* client, u_char data_link, u_char video_link)
{
    int ret = PP_ERR;

    /* DL and VL both represent the kvm_node and must hence be equal */
    if (data_link != video_link) goto bail;

    /* most devices just have one kvm_node (may be different in subclasses) */
    if (data_link != 0) goto bail;

    MUTEX_LOCK(&m_mtx);

    client->SetVideoLinkId(video_link);
    client->SetDataLinkId(data_link);

    MUTEX_UNLOCK(&m_mtx);

    ret = PP_SUC;
bail:
    return ret;
}
