/**
 * retrieves MO Profiles out of the ldap database
 *   particular for a specific user and a specific MO
 *
 * Author: Thomas Breitfeld <thomas@peppercon.de>
 * Copyright 2003 Peppercon AG
 */

#include <stdio.h>
#include <stdlib.h>

#include <pp/base.h>
#include <pp/vector.h>
#include <pp/tokenizer.h>
#include <pp/utf.h>
#include <liberic_misc.h>
#include <liberic_config.h>
#include <pp/ldap_prof.h>
#include <pp/profile.h>
#include <pp/intl.h>
#include <pp/acl.h>
#include <pp/cfg.h>
#include <pp/cfgx.h>
#include <pp/um.h>

#include "ldap_cache.h"
#include "ldap_intern.h"

/* sanity check */
#if !defined(PP_BOARD_PEMX) && !defined(PP_FEAT_DEVICE) && !defined(PP_BOARD_KXNG)
# error wrong board
#endif

#define DEBUG
#ifdef DEBUG
# define D(fmt, args...) printf(fmt, ##args);
#else
# define D(fmt, args...)    
#endif

typedef enum {
    LDAP_SERVER_TYPE_USER,
    LDAP_SERVER_TYPE_NORBOX
} ldap_server_type_t;

typedef struct {
    char* nbx_bdn;
    char* nbx_name;
    int ref_cnt;
    pthread_mutex_t ref_cnt_mtx;
} moinfo_t;

static volatile int pp_ldap_initialized = 0;

static pp_ldap_hash_cache_t userdn_cache;
static pp_ldap_hash_cache_t groups_cache;
static pp_ldap_hash_cache_t profiles_cache;
static pp_ldap_hash_cache_t assigned_profiles_cache;
static pp_ldap_hash_cache_t assigned_devprof_cache;
static pp_ldap_single_cache_t clusters_cache;
static pp_ldap_single_cache_t moinfo_cache;

pp_hash_i_t* temp_uids=NULL;

// assigned device profiles change listener
// (used for delayed activation of dev profile changes
static void (*assigned_devprof_update_listener)(void);

static char* nbx_base_dn            = NULL;  // kinda base dn cache
static char* user_base_dn           = NULL;  // kinda user base dn cache
const char* emx_srv_type            = "emx";
const char* nds_srv_type            = "nds";
const char* ads_srv_type            = "ads";
const char* std_nbx_base_dn         = "dc=norbox,dc=peppercon,dc=com";
const char* ca_root_cert	    = "/etc/e-RIC/ssl/ca/ca_ldap.cer";

const char* nbx_usrou_attrvalue     = "Users";

const char* nbx_usr_attrtype        = "uid";
const char* nbx_usr_objectclass     = "OpenLDAPperson";

const char* nbx_grpou_attrvalue     = "Groups";

const char* nbx_grp_attrtype        = "cn";
const char* nbx_grp_ocvalue	    = "groupOfUniqueNames";
const char* nbx_grpmember_attrtype  = "uniqueMember";

const char* nbx_moou_attrvalue      = "Devices";

const char* nbx_id_attrtype         = "ppId";
const char* nbx_eui64_attrtype      = "ppEUI64";

const char* nbx_clt_attrtype        = "cn";
const char* nbx_clt_ocvalue	    = "groupOfUniqueNames";
const char* nbx_cltmember_attrtype  = "uniqueMember";


const char* nbx_usrassign_ocvalue   = "ppUserAssignment";
const char* nbx_devassign_ocvalue   = "ppDeviceAssignment";
const char* nbx_assignprof_attrtype = "ppAliasedProfile";
const char* nbx_assigngrp_attrtype  = "ppAliasedGroup";
const char* nbx_assignusr_attrtype  = "ppAliasedUser";
const char* nbx_assigndev_attrtype  = "ppAliasedDevice";
const char* nbx_assignclt_attrtype  = "ppAliasedCluster";
const char* nbx_profile_attrtype    = "cn";
const char* nbx_usrprofile_ocvalue  = "ppUserProfile";
const char* nbx_devprofile_ocvalue  = "ppDeviceProfile";

const char* nbx_devprofassign_def   = "default";
const char* nbx_cdlid_attrtype      = "ppCdlIdentifier";
const char* nbx_cdldata_attrtype    = "ppCdlData";

// routines for accessing the caches
static int lookup_mo_info(moinfo_t **moinfo, pp_ldap_t* ld, const char* eui);
static int lookup_user_dn(char **dn, pp_ldap_t* ld, const char* name);
static int get_group_memberships(char ***groups, pp_ldap_t* ld_user, const char* uid);
static int get_cluster_memberships(char ***clusters, pp_ldap_t* ld, const char* eui64);
static int get_profile(pp_profile_t** prof, pp_ldap_t* ld, const char* profid);
static int get_assigned_profile(pp_profile_t** prof, pp_ldap_t* ld, const char* moid,
				const char* uid);
static int get_assigned_devprof(pp_profile_t** prof, pp_ldap_t* ld, const char* moid);

// basic ldap routines for filling the caches
static int ldget_mo_info(moinfo_t **moinfo, pp_ldap_t* ld, const char* eui);
static int ldget_user_dn(char **dn, pp_ldap_t* ld, const char* name);
static int ldget_user_groups(char ***groups, pp_ldap_t* ld, const char* uid);
static int ldget_mo_clusters(char ***clusters, pp_ldap_t* ld, const char* eui64);
static int ldget_profile(pp_profile_t** prof, pp_ldap_t* ld, const char* profdn);
static int find_assigned_profile(pp_profile_t** prof, pp_ldap_t* ld, const char* mouid);
static int find_assigned_devprof(pp_profile_t** prof, pp_ldap_t* ld, const char* moid);

// ldap search utilities for getting profiles
static int ldsearch_usrgrp_profalias(LDAPMessage **res, pp_ldap_t *ld, const char *uname, const char *moid);
static char** extract_attrval_from_ldres(LDAP* ld, LDAPMessage *res, 
                                         const char* attrtype);
static char* extract_dn_from_ldres(LDAP* ld, LDAPMessage *res, LDAPMessage** entry);
static int find_user_profile(pp_profile_t** prof, pp_ldap_t *ld, LDAPMessage* ldm,
			     const char *usrgrp_attr,const char *devclt_attr,const char* uid);
static int read_profile(pp_profile_t** prof, pp_ldap_t *ld, LDAPMessage *entry);
static int read_acl(pp_ldap_t *ld, LDAPMessage *entry, pp_profile_t* prof);
static int read_settings(pp_ldap_t *ld, LDAPMessage *entry, pp_profile_t* prof);
static int read_name(pp_ldap_t* ld, LDAPMessage* entry, char** name);

// other utils
static int ld_connect(pp_ldap_t* ld, ldap_server_type_t ldap_server_type);
static int ld_connect_server(LDAP** ld, const char* server, const char* bdn,
			     const char* bpw, int* bound);
static int ld_bind_server(int* bound, LDAP* ld, const char* dn, const char* pw);
static int ld_authenticate(int *is_auth, pp_ldap_t* ld, const char* dn, const char* pw);
static void ld_disconnect(pp_ldap_t * ld);
static moinfo_t* create_moinfo(const char* bdn, const char* name);
static moinfo_t* duplicate_moinfo(moinfo_t* info);
static void destroy_moinfo(void* _p);
static char** duplicate_strstr(char** strs);
static void free_strstr(void* strs); // free a string of strings
static int ld_get_errno(LDAP* ld);
static char* get_nbx_base_dn(void);
static char* get_user_base_dn(void);
static int generate_ldap_env(pp_ldap_t* ld, const char *name, 
                             char **p_dn, char **p_filter);

// profile operations
static int ld_profile_union(pp_profile_t **prof, const char *uname, 
                            vector_t *profs, int forward);
#if 0
static int ld_profile_settings_union(pp_profile_t* nprof, pp_profile_t* prof);
#endif

// error handling
static const char * ldap_errors[] = {
    /*00*/ N_("An internal LDAP error occured."),
    /*01*/ N_("The user has no LDAP profile assigned."),
    /*02*/ N_("The LDAP configuration is inconsistent."),
    /*03*/ N_("The LDAP server could not be contacted."),
    /*04*/ N_("An LDAP operation failed."),
    /*05*/ N_("The LDAP server rejected authentication."),
    /*06*/ N_("The LDAP lookup user could not be authenticated."),
    /*07*/ N_("The LDAP manager user could not be authenticated."),
    /*08*/ N_("The LDAP profile tag doesn't exist."),
    /*09*/ N_("The Device Profile is assigned ambiguously."),
    /*10*/ N_("The LDAP Entries are inconsistent."),
};

static int errno_base = -1;
inline int pp_ldap_errno_base() { 
    assert(errno_base > 0);
    return errno_base;
}

int pp_ldap_has_errno(int error) {
    return error >= errno_base &&
	error < (int)(errno_base + sizeof(ldap_errors) / sizeof(*ldap_errors));
}

static const char* get_error_string(const int error) {
    return ldap_errors[error - errno_base];
}

/*
 * public methods used to run this module
 ************************************************/
int
pp_ldap_init()
{
    int ret;
    const char * fn = ___F;
    assert(!pp_ldap_initialized);

    temp_uids=pp_hash_create_i(20);

    // register errnos for myself
    errno_base = pp_register_errnos(sizeof(ldap_errors) / sizeof(*ldap_errors),
				    get_error_string);
    ret = PP_LDAP_ERR_INTERNAL_ERROR;
    
    if (0 > 
    pp_ldap_single_cache_init
	(&moinfo_cache, PP_LDAP_CACHE_DONT_NULL, 0,
	(int(*)(void**, pp_ldap_t*, const char*))ldget_mo_info,
	(void*(*)(void*))duplicate_moinfo, destroy_moinfo)) {
	pp_log("%s(): modn_cache init failed\n", fn);
	return ret;
    }
    
    if(0 > pp_ldap_hash_cache_init(&userdn_cache, PP_LDAP_CACHE_DONT_NULL, 0,
	            (int(*)(void**, pp_ldap_t*, const char*))ldget_user_dn,
				   (void*(*)(void*))strdup,
				   ldap_memfree)) {
	pp_log("%s(): userdn_cache init failed\n", fn);
	return ret;
    }
    if(0 > pp_ldap_hash_cache_init(&groups_cache, PP_LDAP_CACHE_DONT_NULL, 0,
	            (int(*)(void**, pp_ldap_t*, const char*))ldget_user_groups,
				   (void*(*)(void*))duplicate_strstr,
				   free_strstr)) {
	pp_log("%s(): groups_cache init failed\n", fn);
	return ret;
    }
    if(0 > pp_ldap_hash_cache_init(&profiles_cache, PP_LDAP_CACHE_DONT_NULL, 0,
	            (int(*)(void**, pp_ldap_t*, const char*))ldget_profile,
				   (void*(*)(void*))pp_profile_duplicate,
				   pp_profile_release)) {
	pp_log("%s(): profiles_cache init failed\n", fn);
	return ret;
    }
    if(0 > pp_ldap_hash_cache_init(&assigned_profiles_cache,
				   PP_LDAP_CACHE_DONT_NULL, 0,
	      (int(*)(void**, pp_ldap_t*, const char*))find_assigned_profile,
				   (void*(*)(void*))pp_profile_duplicate,
				   pp_profile_release)) {
	pp_log("%s(): assigned_profiles_cache init failed\n", fn);
	return ret;
    }
    
    if (0 > 
    pp_ldap_single_cache_init
	(&clusters_cache, PP_LDAP_CACHE_DONT_NULL, 0,
	 (int(*)(void**, pp_ldap_t*, const char*))ldget_mo_clusters,
	 (void*(*)(void*))duplicate_strstr, free_strstr)) {
	pp_log("%s(): clusters_cache init failed\n", fn);
	return ret;
    }

    if (0 > pp_ldap_hash_cache_init(&assigned_devprof_cache,
				    PP_LDAP_CACHE_DO_NULL,
				    PP_LDAP_ERR_NO_SUCH_TAG,
	 (int(*)(void**, pp_ldap_t*, const char*))find_assigned_devprof,
	 (void*(*)(void*))pp_profile_duplicate, pp_profile_release)) {
	pp_log("%s(): assigned_devprof_cache init failed\n", fn);
	return ret;
    }
    
    ret = PP_LDAP_ERR_NO_ERROR;
    pp_ldap_initialized = 1;
    
    return ret;
}

void pp_ldap_cleanup(void) {
    assert(pp_ldap_initialized);

    pp_ldap_clear_caches();

    pp_hash_delete_i(temp_uids);

    pp_unregister_errnos(errno_base);
    
    pp_ldap_initialized = 0;
}
    

/*
 * returns true if erla is PEMX discovered and
 * and params are there for getting infos out of the LDAP server
 * 
 * This function can also be called without initializing,
 * i.e. pp_ldap_init doesn't need to be called,
 * however result will be false then
 */
int
pp_ldap_is_active(UNUSED char** lds, UNUSED char** lds_user)
{
    if (!pp_ldap_initialized) return 0;

    return 0;
}

/*
 * returns true if erla is not PEMX discovered but
 * the authentication mode is "ldap" and
 * there is a valid ldap user server in the config file
 *
 * This function can also be called without initializing,
 * i.e. pp_ldap_init doesn't need to be called,
 * however result will be false then
 */
int pp_ldap_mini_is_active(char** lds_user) {
    char *user_ldap_server = NULL, *auth_mode = NULL;
    int ret = 0;
    
    if (!pp_ldap_initialized || pp_ldap_is_active(NULL, NULL)) {
	return 0;
    }

    pp_cfg_get(&auth_mode, "auth_mode");
    if (!strcmp(auth_mode, "ldap")
	&& PP_SUCCED(pp_cfg_get_nodflt(&user_ldap_server, "ldap.host"))) {
	ret = 1;
    }
    free(auth_mode);
    if (lds_user && user_ldap_server) {
	*lds_user = user_ldap_server;
    } else {
	free(user_ldap_server);
    }
    return ret;
}

/*
 * returns the NORBOX assigend MO name
 */
char* pp_ldap_get_nbx_moname(void) {
    pp_ldap_t ld = PP_LDAP_T_INITIALIZER;
    char* name = NULL;
    if (pp_ldap_is_active(NULL, NULL)) {
	moinfo_t* info;
        lookup_mo_info(&info, &ld, pp_eui64_str);
	if(info != NULL && info->nbx_name != NULL) {
	    name = strdup(info->nbx_name);
	    destroy_moinfo(info);
	}
    }
    ld_disconnect(&ld);
    return name;
}

/*
 * Name:	pp_ldap_authenticate()
 *
 * Description:	This routine will authenticate us via an bind operation on
 *		the LDAP server and caches the DN with the login as key.
 */
int
pp_ldap_authenticate(int *auth, const char* moid, 
                     const char* name, const char* passwd, char **groupname) {
    pp_ldap_t ld = PP_LDAP_T_INITIALIZER_USER;
    char* ldap_server = NULL;
    char* ldap_server_type = NULL;
    char* ads_domain_name = NULL;
    char* dn = NULL;
    int bound = 0, ret = PP_ERR, is_auth = 0;

    assert(moid && name && passwd);
    D("pp_ldap_authenticate: %s %s\n", name, passwd);

    if (groupname) *groupname = NULL;

    /* ld.user will be freed by ld_disconnect */
    ld.user = strdup(name);

    if(0 > ld_connect(&ld, LDAP_SERVER_TYPE_USER))
        return ret;
    
    if (!pp_ldap_mini_is_active(NULL)) {
	//ldap_server_type = pp_dp_get_bm_prop(PP_BM_USER_LDAP_SERVER_TYPE);
	//ads_domain_name = pp_dp_get_bm_prop(PP_BM_ADS_DOMAIN_NAME);
	pp_cfg_get_at_layer_nodflt(PP_PROFILE_LOCAL, &ldap_server_type, "pemx.user_ldap.server_type");
	pp_cfg_get_at_layer_nodflt(PP_PROFILE_LOCAL, &ads_domain_name, "pemx.user_ldap.ads_domain_name");
    } else {
        pp_cfg_get_nodflt(&ldap_server_type, "ldap.server_type");
	pp_cfg_get_nodflt(&ads_domain_name, "ldap.ads_domain_name");
    }
    if (!ldap_server_type) {
	errno = PP_LDAP_ERR_CONFIG_ERROR;
	goto bail;
    }

    // for ADS you must bind before you can issue searches
    if (!strcmp(ldap_server_type, ads_srv_type) && ads_domain_name) {
	char _dn[1024];
	/* construct DN for Active Directory */
	snprintf(_dn, sizeof(_dn), "%s@%s", name, ads_domain_name);
	if (PP_ERR == (ret = ld_authenticate(&is_auth, &ld, _dn, passwd)) || !is_auth) 
            goto bail;
	/* lookup the user dn */
	if (PP_ERR == (ret = lookup_user_dn(&dn, &ld, name))) 
            goto bail;
	bound = 1;
    } else if (!strcmp(ldap_server_type, nds_srv_type) ||
	       !strcmp(ldap_server_type, emx_srv_type) ||
	       !strcmp(ldap_server_type, "generic")) {
	/* lookup the user dn */
	if (PP_ERR == (ret = lookup_user_dn(&dn, &ld, name))) 
            goto bail;
	if (PP_ERR == (ret = ld_authenticate(&is_auth, &ld, dn, passwd)) || !is_auth) 
            goto bail;
	bound = 1;
    }

    D("pp_ldap_authenticate: %s authenticated\n", name);
    
    if(bound) {
	if (!ld.use_minimal_ldap) {
	    /* if there is no profile, deny authentication */
	    pp_profile_t* prof = NULL;
	    if(PP_ERR == (ret = get_assigned_profile(&prof, &ld, moid, name)) || !prof) 
		goto bail;
	    else
		pp_profile_release(prof);
	}
	*auth = 1;
        ret = PP_SUC;
	pp_ldap_create_temp_uid(name);
        
    // try to extract Raritan group from LDAP attributes
    // (only if requested by callee)
        if (groupname)
        {
            char** vals;
            LDAPMessage *res, *entry;
            char *filter = NULL, *search_dn = NULL;
            const char * attrs[] = { "rciusergroup", "memberOf", NULL };
                  
            assert(groupname);
            
            if(generate_ldap_env(&ld, name, &search_dn, &filter) != PP_SUC) {
                goto bailout;
            }
        
            /* search user */
            D("pp_ldap_authenticate: %s %s\n", search_dn, filter);
            if (ldap_search_s(ld.ld_user, search_dn, LDAP_SCOPE_SUBTREE,
                              filter, attrs, 0, &res)  != LDAP_SUCCESS ) {
                ldap_perror(ld.ld_user,
                            "pp_ldap_authenticate: ldap_search_s");
                errno = PP_LDAP_ERR_OPERATION_FAILED;
                goto bailout;
            }
        
            /* should return exactly one user dn */
            if (ldap_count_entries(ld.ld_user, res) != 1)
                goto bailout;
        
            /* get that one */
            if (NULL == (entry = ldap_first_entry(ld.ld_user, res))) 
                goto bailout;
            
            /* search for rciusergroup */
            if ((vals = ldap_get_values(ld.ld_user, entry, 
                                        "rciusergroup")) != NULL) {
                *groupname = strdup(vals[0]);
		ldap_value_free(vals);
                D("pp_ldap_authenticate: got groupname %s\n", *groupname);
            } else {
                /* no rciusergroup set */
                *groupname = NULL;
                D("pp_ldap_authenticate: "
                  "failed to get groupname from rciusergroup, returning NULL\n");
            }

	    if (*groupname == NULL){
		/* no rciusergroup attribute. try alternative way */

		if (!strcmp(ldap_server_type, ads_srv_type)){
		    /* ads, search memberOf */
		    if ((vals = ldap_get_values(ld.ld_user, entry,
                                                "memberOf")) != NULL) {
			int i=0, gid;
			char *group_dn, *ptr1, *ptr2;
			while(vals[i] && !*groupname) {
			    group_dn = strdup(vals[i]);
			    D("pp_ldap_authenticate: memberOf -- [%s]\n", group_dn);
			    if((ptr1 = strtok_r(group_dn, "=,", &ptr2)) && 
				(ptr1 = strtok_r(NULL, "=,", &ptr2))) {
				if(PP_ERR != (gid = pp_um_group_get_gid(ptr1))) {
				    *groupname = strdup(ptr1);
				    D("pp_ldap_authenticate: got groupname[%s] from memeberOf\n", *groupname);
				}
			    }
			    free(group_dn);
			    i++;
			}
			ldap_value_free(vals);
		    }
		}
	    }
            
 bailout:
            if (res) ldap_msgfree(res);
            free(search_dn);
            free(filter);
        }
    }

 bail:
    ld_disconnect(&ld);
    free(dn);
    free(ldap_server);
    free(ldap_server_type);
    free(ads_domain_name);
    return ret;
}

int
pp_ldap_get_profile(pp_profile_t **prof, const char* moid, const char* uid) {
    pp_ldap_t ld = PP_LDAP_T_INITIALIZER_USER;
    int ret = PP_SUC;

    assert(moid && uid);
    
    /* ld.user will be freed by ld_disconnect */
    ld.user = strdup(uid);
    
    if(PP_ERR == (ret = get_assigned_profile(prof, &ld, moid, uid)) || !*prof)
	goto bailout;
    
 bailout:
    ld_disconnect(&ld);
    return ret;
}

int pp_ldap_get_devprofile(pp_profile_t **prof, const char* moid) {
    pp_ldap_t ld = PP_LDAP_T_INITIALIZER_DEV;
    int ret = PP_SUC; 

    assert(moid);

    ret = get_assigned_devprof(prof, &ld, moid);
    
    ld_disconnect(&ld);
    
    return ret;
}

void pp_ldap_clear_caches() {
    pp_ldap_cache_clear((pp_ldap_cache_t*)&userdn_cache);
    pp_ldap_cache_clear((pp_ldap_cache_t*)&groups_cache);
    pp_ldap_cache_clear((pp_ldap_cache_t*)&profiles_cache);
    pp_ldap_cache_clear((pp_ldap_cache_t*)&assigned_profiles_cache);
    pp_ldap_cache_clear((pp_ldap_cache_t*)&clusters_cache);
    pp_ldap_cache_clear((pp_ldap_cache_t*)&moinfo_cache);
    pp_ldap_cache_clear((pp_ldap_cache_t*)&assigned_devprof_cache);
    free(nbx_base_dn);
    nbx_base_dn = NULL;
    free(user_base_dn);
    user_base_dn = NULL;
}

/*
 * device profile changes may not be actived immediately,
 * however, we are not able to make that decision, so in
 * case it should be postponed, somebody should register
 * for the update notification and call
 * pp_ldap_assigned_devprof_clear_caches() at his
 * convinience
 */
void pp_ldap_assigned_devprof_add_listener(void(*changed_listener)(void)) {
    assert(assigned_devprof_update_listener == NULL);
    assigned_devprof_update_listener = changed_listener;
}

/*
 * currently device profiles are simply stored in the
 * profiles cache, so ne special action is taken for them
 * see also pp_ldap_devprofile_clear_caches()
 */
void pp_ldap_assigned_devprof_clear_caches() {
    pp_ldap_cache_clear((pp_ldap_cache_t*)&profiles_cache);
    pp_ldap_cache_clear((pp_ldap_cache_t*)&assigned_devprof_cache);    
}

/* ------------------------------------------------------- *
 * private methods realizing the stuff above               *
 * ------------------------------------------------------- */

// returns 0 in case of success, -1 in case of error
static int
ld_connect(pp_ldap_t* ld, ldap_server_type_t ldap_server_type) {
    char *ldps = NULL, *ldus = NULL, *pmgr_dn, *pmgr_pw, *ulu_dn, *ulu_pw;
    int identical = 0, ret = -1;

    // check if already connected, if yes, just return 
    if((ldap_server_type == LDAP_SERVER_TYPE_NORBOX && ld->ld != NULL)
       || (ldap_server_type == LDAP_SERVER_TYPE_USER && ld->ld_user != NULL))
	return 0;

    // get user ldap server name and nbx server name
    if (!pp_ldap_is_active(&ldps, &ldus)) {
	if (ldap_server_type == LDAP_SERVER_TYPE_USER
	    && pp_ldap_mini_is_active(&ldus)) {
	    ld->use_minimal_ldap = 1;
	} else {
	    errno = PP_LDAP_ERR_CONFIG_ERROR;
	    return -1;
	}
    }
    
    // check whether user and profile server are identical
    // in case there are a MGR_DN and MGR_PW or
    // LOOKUP_DN and LOOKUP_PW given, we need to build
    // up two seperate connections unconditionally
    //pmgr_dn = pp_dp_get_bm_prop(PP_BM_NORBOX_LDAP_MANAGER_DN);
    //pmgr_pw = pp_dp_get_bm_prop(PP_BM_NORBOX_LDAP_MANAGER_PW);
    //ulu_dn = pp_dp_get_bm_prop(PP_BM_LDAP_LOOKUP_USER_DN);
    //ulu_pw = pp_dp_get_bm_prop(PP_BM_LDAP_LOOKUP_USER_PW);
    pmgr_dn=pmgr_pw=ulu_dn=ulu_pw=NULL;
    if(ldus != NULL && ldps != NULL && !strcmp(ldus, ldps)
       && (pmgr_dn == NULL || pmgr_pw == NULL)
       && (ulu_dn == NULL || ulu_pw == NULL))
	identical = 1;
#if defined(PP_FEAT_RARITAN_STYLE_LDAP_SETTINGS)
    {
        /* get root DN and secret */
        char *secret = NULL, *root_dn = NULL;
        
        pp_cfg_get_nodflt(&root_dn, "ldap.base_dn");
        pp_cfg_get_nodflt(&secret, "ldap.secret");
        if(root_dn && *root_dn && secret) {
            /* secret may be "" */
            ulu_dn = root_dn;
            ulu_pw = secret;
        }
        
        /* no need to free root_dn and secret! free ulu_dn and ulu_pw! */        
    }
#endif /* PP_FEAT_RARITAN_STYLE_LDAP_SETTINGS */


    // establish connection
    if(ldap_server_type == LDAP_SERVER_TYPE_USER) {
	D("ld_connect: TYPE_USER\n");
	if(PP_ERR == (ret = ld_connect_server(&ld->ld_user, ldus, ulu_dn, ulu_pw,
					      &ld->ld_user_is_bound))) {
	    if (errno == PP_LDAP_ERR_AUTH_FAILED)
		errno = PP_LDAP_ERR_LOOKUP_AUTH_FAILED;
	    goto bailout;
	}
	ld->ld_user_server = ldus; ldus = NULL;
	if(identical) {
	    ld->ld = ld->ld_user;
	    ld->ld_server = ldps;  ldps = NULL;
	}
    } else {
	D("ld_connect: TYPE_NORBOX\n");
	D("ld_connect: %s %s %s\n",ldps,pmgr_dn,pmgr_pw);
	if(PP_ERR == (ret = ld_connect_server(&ld->ld, ldps, pmgr_dn, pmgr_pw,
					      &ld->ld_is_bound))) {
	    if (errno == PP_LDAP_ERR_AUTH_FAILED)
		errno = PP_LDAP_ERR_MGR_AUTH_FAILED;
	    goto bailout;
	}
	ld->ld_server = ldps;          ldps = NULL;
	if(identical) {
	    ld->ld_user = ld->ld;
	    ld->ld_user_server = ldus; ldus = NULL;
	}
    }
    ret = 0;
 bailout:
    free(ldus);
    free(ldps);
    free(pmgr_dn);
    free(pmgr_pw);
    return ret;
}

/*
 * ld_connect_server may also bind a connection in case dn and pw
 * are given. This is intended for external norbox ldap server
 * that cannot be browsed annonymously
 */
static int ld_connect_server(LDAP** ldap, const char* server, const char* bdn,
			     const char* bpw, int* bound) {
    LDAP* ld;
    int ldap_version = LDAP_VERSION3;
    int ret = PP_SUC;
    int use_ssl, port;
    #define MAX_LDAP_SERVER_URI 80
    char server_uri[MAX_LDAP_SERVER_URI];
    
    if(server == NULL) {
        ret = PP_ERR;
        goto error;
    }
    
    if(PP_ERR == (ret = pp_cfg_is_enabled_at_layer(PP_PROFILE_LOCAL, &use_ssl,
                                                   "ldap.ssl.enabled"))) {
        errno = PP_LDAP_ERR_CONFIG_ERROR;
        goto error;
    }
    
    if(use_ssl) {
        /* open SSL connection */
        D("SSL active, try to establish secure connection\n");

        if(PP_ERR == (ret = pp_cfg_get_int_at_layer(PP_PROFILE_LOCAL, &port,
                                                    "ldap.ssl.port"))) {
            errno = PP_LDAP_ERR_CONFIG_ERROR;
            goto error;
        }

        {
	    /* check server certificate.
	     * Note: server host name must be domain name. Because
	     *       according rfc2830.txt, host name must match
	     *       the server's identity as presented in the server's
	     *       Certificate message, in order to prevent 
	     *       man-in-the-middle attacks.
             */
	    D("ld_connect_server: enable cert check -- [%s]\n", ca_root_cert);
	    if ( ldap_set_option(NULL, LDAP_OPT_X_TLS_CACERTFILE, ca_root_cert) 
		    != LDAP_OPT_SUCCESS) {
		ldap_perror(ld,"setting root certificate:");
		errno = PP_LDAP_ERR_CONFIG_ERROR;
                ret = PP_ERR;
		goto error;
	    }
        }
    } else {
        /* open unencrypted connection */
        D("SSL not active, try to establish unencrypted connection\n");
        
        if(PP_ERR == (ret = pp_cfg_get_int_at_layer(PP_PROFILE_LOCAL, &port,
                                                    "ldap.port"))) {
            errno = PP_LDAP_ERR_CONFIG_ERROR;
            goto error;
        }
    }
    
    snprintf(server_uri, MAX_LDAP_SERVER_URI, "%s://%s:%d", use_ssl ? "ldaps" : "ldap",
                                           server, port);

    D("ld_connect_server: %s %s %s\n", server_uri, bdn, bpw);
    /* open ldap connection */
    if (ldap_initialize(&ld, server_uri) != LDAP_SUCCESS) {
	pp_log_err("ld_connect_server: ldap_open");
	errno = PP_LDAP_ERR_CONN_FAILED;
        ret = PP_ERR;
	goto error;
    }
    ldap_perror(ld, "ld_connect_server: succeeded ldap_initialize");
    
    /* set protocol version */
    if (ldap_set_option(ld, LDAP_OPT_PROTOCOL_VERSION,
			&ldap_version) != LDAP_OPT_SUCCESS ) {
	ldap_perror(ld, "ld_connect_server: ldap_set_option(VERSION)");
	ldap_unbind(ld);
	errno = PP_LDAP_ERR_CONN_FAILED;
	ret = PP_ERR;
        goto error;
    }
    ldap_perror(ld, "ld_connect_server: succeeded ldap_set_option(VERSION)");

    /* bind in case dn and pw are given */
    if (bdn != NULL && bpw != NULL) {
	if (PP_ERR == (ret = ld_bind_server(bound, ld, bdn, bpw))) {
	    ldap_unbind(ld);
	    goto error;
	}
    }
    
    *ldap = ld;
 error:
    return ret;
}

static int ld_bind_server(int *bound, LDAP* ld, const char* dn, const char* pw) {
    if (ldap_simple_bind_s(ld, dn, pw) != LDAP_SUCCESS) {
	ldap_perror(ld, "ldap_simple_bind_s");
	errno = PP_LDAP_ERR_AUTH_FAILED;
        *bound = 0;
	return PP_ERR;
    }
    *bound = 1;
    return PP_SUC;
}

/* this is the core authentication function
 * Note: it is only applicable for the user ldap server!
 * thwo cases:
 *
 * if lookup user was used allready to bind the normal user server connection
 *    or if there is no user server connection at all
 *    in these cases we simply open and bind another connection and close
 *    it immediately afterwards
 * else
 *    we bind the already open connection to the given user and
 *    use it from that time on for the lookup user.
 * this way chances are higher that we actually can browse the ldap db
 *
 * returns PP_ERR on connection failure, PP_SUC otherwhise
 * is_auth is set to 1 for authentified user, 0 otherwhise
 */
static int ld_authenticate(int *is_auth, pp_ldap_t* ld, const char* dn, const char* pw) {
    int ret = PP_SUC, tmp;
    LDAP* uld = NULL;
    *is_auth = 0;
    if (ld->ld_user == NULL || ld->ld_user_is_bound) {
	char* user_server;
	if (ld->ld_user_server == NULL) {
	    if (!pp_ldap_is_active(NULL, &user_server) &&
		!pp_ldap_mini_is_active(&user_server)) {
		errno = PP_LDAP_ERR_CONFIG_ERROR;
                ret = PP_ERR;
                goto error;
	    }
	} else 
            user_server = strdup(ld->ld_user_server);
	D("ld_authenticate: ld_connect_server: temporary TYPE_USER %s %s %s\n",
	  user_server, dn, pw);
	if(PP_ERR != (ret = ld_connect_server(&uld, user_server, 
                                              dn, pw, &tmp))) {
	    *is_auth = 1;
            D("ld_authenticate: unbind\n");
            ldap_unbind_s(uld); // no need to keep this connection
        }
	free(user_server);
    } else {
	D("ld_authenticate: ld_bind_server: TYPE_USER %s %s\n", dn, pw);
	if (PP_ERR != (ret = ld_bind_server(&ld->ld_user_is_bound, ld->ld_user, dn, pw)))
	    *is_auth = 1;
    }
 error:
    return ret;
}

static void ld_disconnect(pp_ldap_t* ld) {
    if (ld->ld_user) {
	if (ld->ld_user != ld->ld) {
	    D("ld_disconnect: user\n");
	    ldap_unbind_s(ld->ld_user);
	}
	free(ld->ld_user_server);
    }
    if (ld->ld) {
	D("ld_disconnect: nbx\n");
	ldap_unbind_s(ld->ld);
	free(ld->ld_server);
    }
    ld->ld = ld->ld_user = NULL;
    ld->ld_server = ld->ld_user_server = NULL;
    free(ld->user);
}

static int lookup_mo_info(moinfo_t** moinfo, pp_ldap_t* ld, const char* eui64) {
    return pp_ldap_cache_get((void**)moinfo, (pp_ldap_cache_t*)&moinfo_cache, ld, eui64);
}

static int lookup_user_dn(char **dn, pp_ldap_t * ld, const char* name) {
    return pp_ldap_cache_get((void**)dn, (pp_ldap_cache_t*)&userdn_cache, ld, name);
}

static int
get_group_memberships(char ***groups, pp_ldap_t* ld, const char* uid) {
    return pp_ldap_cache_get((void**)groups, (pp_ldap_cache_t*)&groups_cache, ld, uid);
}

static int
get_cluster_memberships(char ***members, pp_ldap_t* ld, const char* eui64) {
    return pp_ldap_cache_get((void**)members, (pp_ldap_cache_t*)&clusters_cache,
				     ld, eui64);
}

static int get_profile(pp_profile_t **prof, pp_ldap_t* ld, const char* profid) {
    return pp_ldap_cache_get((void**)prof, (pp_ldap_cache_t*)&profiles_cache, ld, profid);
}

static int
get_assigned_profile(pp_profile_t **prof, pp_ldap_t* ld, const char* moid, 
                     const char* uid) {
    char* mouid_ptr = NULL;
    size_t len = 0;
    int ret;
    pp_dsprintf(&mouid_ptr, &len, 0, "%s,%s", moid, uid);
    ret = pp_ldap_cache_get((void**)prof, (pp_ldap_cache_t*)&assigned_profiles_cache,
			    ld, mouid_ptr);
    free(mouid_ptr);
    return ret;
}

static int
get_assigned_devprof(pp_profile_t **prof, pp_ldap_t* ld, const char* moid) {
    int ret;
    ret = pp_ldap_cache_get((void**)prof, (pp_ldap_cache_t*)&assigned_devprof_cache,
			    ld, moid);
    return ret;
}

static int
find_assigned_profile(pp_profile_t **prof, pp_ldap_t * ld, const char* mouid) {
    pp_profile_t* uprof = NULL;
    LDAPMessage* usrgrp_mo_profs = NULL;
    int ret;
    char *moid = NULL, *uname=NULL;

    D("find_assigned_profile: fetch %s\n", mouid);
    if(PP_ERR == (ret = ld_connect(ld, LDAP_SERVER_TYPE_NORBOX)))
        goto bailout;

    /* devide mouid into its moid and uname part */
    moid = strdup(mouid);
    uname = strchr(moid, ',');
    *uname++ = '\0';
     
    /*
     * check for the profiles in the order of their priorities.
     * ones a profile has been found we will use it and not
     * search further, what also means the acls of lower priorities
     * have no effect at all
     */
    // get mo associations
    if(PP_ERR == (ret = ldsearch_usrgrp_profalias(&usrgrp_mo_profs, ld, uname, moid))
       || !usrgrp_mo_profs)
	goto bailout;

    // ceck for user and group in mo association
    // check for a user profile first
    // if there is none check for a group profile
    // if that fails as well check the clusters
    // the priority of profiles:
    // 1. user on device
    // 2. group on device
    // 3. user on cluster
    // 4. group on cluster

    if(!(PP_ERR != (ret = find_user_profile(&uprof, ld, usrgrp_mo_profs, 
			nbx_assignusr_attrtype,nbx_assigndev_attrtype, uname)) && uprof)) 
	if(!(PP_ERR != (ret = find_user_profile(&uprof, ld, usrgrp_mo_profs, 
			    nbx_assigngrp_attrtype,nbx_assigndev_attrtype, uname)) && uprof)) 
	    if(!(PP_ERR != (ret = find_user_profile(&uprof, ld, usrgrp_mo_profs, 
				nbx_assignusr_attrtype,nbx_assignclt_attrtype, uname)) && uprof)) 
		ret = find_user_profile(&uprof, ld, usrgrp_mo_profs, 
			nbx_assigngrp_attrtype,nbx_assignclt_attrtype, uname);
    
    ldap_msgfree(usrgrp_mo_profs);
    *prof = uprof;
 bailout:
    free(moid);
    return ret;
}

/*
 * search an LDAP message for a group profile alias and resolves it
 * the ldap message is assumed to contain ppAlias object types
 * we won't check that again and only UserGroups of the desirged ids
 */
int 
find_user_profile(pp_profile_t **prof, pp_ldap_t *ld, LDAPMessage *ldm,const char *usrgrp_attr,const char *devclt_attr,const char *uname) {
    LDAPMessage *entry;
    pp_profile_t *uprof = NULL;
    int n,ret = PP_SUC;
    char **vals = NULL;
    vector_t *profs;
    
    if(0 > (n = ldap_count_entries(ld->ld, ldm))) {
	errno = PP_LDAP_ERR_OPERATION_FAILED;
	ret=PP_ERR;
	goto error;
    }
    
    profs = vector_new(NULL, 16, pp_profile_release);

    for(entry = ldap_first_entry(ld->ld, ldm); entry != NULL;
	entry = ldap_next_entry(ld->ld, entry)) {
	
	ldap_value_free(vals);
	// check whether this is a user/group entry
	if(NULL == (vals = ldap_get_values(ld->ld,entry,usrgrp_attr))
	   || NULL == vals[0])
	    continue;
	
	ldap_value_free(vals);
	// check whether this is a device/cluster entry
	if(NULL == (vals = ldap_get_values(ld->ld,entry,devclt_attr))
	   || NULL == vals[0])
	    continue;
	
	ldap_value_free(vals);
	// get the profile alias
	if(NULL == (vals = ldap_get_values(ld->ld, entry, nbx_assignprof_attrtype))
	   || NULL == vals[0]) continue;
	
	// resolve and read the alias and add it to our vector
	if(PP_ERR != get_profile(&uprof, ld, vals[0]) && uprof) {
	    vector_add(profs, uprof);
            ret = PP_SUC;
	}
    }
    ldap_value_free(vals);
    
    D("find_user_profile %s %s %d\n",usrgrp_attr,devclt_attr,vector_size(profs));
    
    if(ret != PP_ERR) 
    {
	if(vector_size(profs)==0) 
	{
	    vector_delete(profs);
	    *prof=NULL;
	} 
	else if(vector_size(profs)==1) //only one profile found, return it
	{
	    *prof=(pp_profile_t*)vector_get(profs,0);
	    vector_remove_dont_delete(profs,0);
	} 
	else // more than one profile found, returning a merge
	{
	    ret=ld_profile_union(prof,uname,profs,1);
	    vector_delete(profs);
	}
    }
    else {
        vector_delete(profs);
	*prof=NULL;
    }
 error:
    return ret;
}

static int
find_assigned_devprof(pp_profile_t** prof, pp_ldap_t* ld, const char* moid) 
{
    pp_profile_t *dprof=NULL;
    int ret=PP_ERR;
    char **clts=NULL,**clt=NULL;
    char *flt=NULL;
    int flt_len=0;
    LDAPMessage *res=NULL,*entry=NULL;
    char **vals=NULL;

    if(PP_ERR == (ret = ld_connect(ld, LDAP_SERVER_TYPE_NORBOX)))
        goto bailout;

    if(PP_ERR == (ret = get_cluster_memberships(&clts, ld, moid)))
        goto bailout;

    pp_dsprintf(&flt, &flt_len, 0,
		"(&(objectclass=%s)(|(ppAliasedDevice=%s)",
		nbx_devassign_ocvalue, moid);
    for(clt = clts; *clt != NULL; clt++) 
    {
	pp_dstrcat(&flt, &flt_len, "(ppAliasedCluster=");
	pp_dstrcat(&flt, &flt_len, *clt);
	pp_dstrcat(&flt, &flt_len, ")");
    }
    pp_dstrcat(&flt, &flt_len, "))");

    D("find_assigned_devprof: %s\n",flt);
    
    if(ldap_search_s(ld->ld,get_nbx_base_dn(),LDAP_SCOPE_SUBTREE,flt,NULL,0,&res)!=LDAP_SUCCESS)
    {
	ldap_perror(ld->ld,"find_assigned_devprof: ldap_search_s");
	errno=PP_LDAP_ERR_OPERATION_FAILED;
	ret=PP_ERR;
	goto bailout;
    }

    // try to find a assignment to the device
    for(entry=ldap_first_entry(ld->ld,res);entry!=NULL;entry=ldap_next_entry(ld->ld,entry))
    {
	ldap_value_free(vals);
	// check wether this is a device assignment
	if(NULL == (vals = ldap_get_values(ld->ld,entry,nbx_assigndev_attrtype))
	   || NULL == vals[0])
	    continue;
	
	ldap_value_free(vals);
	// get the profile alias
	if(NULL == (vals = ldap_get_values(ld->ld, entry, nbx_assignprof_attrtype))
	   || NULL == vals[0]) continue;
	
	D("find_assigned_devprof: found assignment for %s\n",vals[0]);

	// resolve the profile
	if(PP_ERR != get_profile(&dprof, ld, vals[0]) && dprof) {
            ret = PP_SUC;
	    break;
	}
    }
    ldap_value_free(vals);

    if(ret==PP_SUC && dprof)
    {
	*prof=dprof;
	goto bailout;
    }

    // in case there are more than one device profile assignments via clusters
    if(ldap_count_entries(ld->ld,res)>1)
    {
	D("find_assigned_devprof: devprof ambiguity!!\n");
	errno = PP_LDAP_ERR_DEVPROF_AMBIGUOUS;
	ret=PP_ERR; 
	goto bailout;
    }

    if((entry=ldap_first_entry(ld->ld,res))==NULL)
	goto bailout;
    
    // get the profile alias
    if(NULL == (vals = ldap_get_values(ld->ld, entry, nbx_assignprof_attrtype))
       || NULL == vals[0]) goto bailout;
	
	// resolve the profile
	if(PP_ERR != get_profile(&dprof, ld, vals[0]) && dprof) {
            ret = PP_SUC;
	}
    
    ldap_value_free(vals);

    if(ret != PP_ERR)
        *prof = dprof;

bailout:
    free_strstr(clts);
    free(flt);
    ldap_msgfree(res);
    return ret;
}

static int
ldget_profile(pp_profile_t** prof, pp_ldap_t* ld, const char* profname) {
    LDAPMessage *res, *entry;
    int ret = PP_ERR;
    char *flt=NULL;
    int flt_len=0;

   pp_dsprintf(&flt, &flt_len, 0,
		"(&(objectclass=%s)(cn=%s))",
		nbx_usrprofile_ocvalue, profname);

   D("ldget_profile: fetch %s %s\n", profname,flt);
    if(0 > ld_connect(ld, LDAP_SERVER_TYPE_NORBOX))
        goto bailout;

    if (ldap_search_s(ld->ld, get_nbx_base_dn(), LDAP_SCOPE_SUBTREE, flt, NULL, 0, &res)
	!= LDAP_SUCCESS ) {
	ldap_perror(ld->ld, "ldget_profile: ldap_search_s");
	goto bailout;
    }

    if (ldap_count_entries(ld->ld, res) != 1)
        goto bailout;
    if (NULL == (entry = ldap_first_entry(ld->ld, res))) 
        goto bailout;
    ret = read_profile(prof, ld, entry);
    free(flt);
    ldap_msgfree(res);
    
    return ret;
    
 bailout:
    free(flt);
    if (ld_get_errno(ld->ld) == LDAP_NO_SUCH_OBJECT) /* ldap_search_s failed */
        errno = PP_LDAP_ERR_SRV_INCONSTISTENT;
    else
        errno = PP_LDAP_ERR_OPERATION_FAILED;
    return ret;
}

int 
read_profile(pp_profile_t** prof, pp_ldap_t *ld, LDAPMessage *entry) {
    pp_profile_t* profile = NULL;
    char *name = NULL;

    if(0 > read_name(ld, entry, &name)) 
        goto bailout;
    
    /* create empty user profile */
    assert(ld->type != PP_PROFILE_NONE);
    if ((profile = pp_profile_create(pp_mallocator_heap(), ld->type, 
                                     name ? name : "noname")) == NULL)
        goto bailout;

    if(0 > read_acl(ld, entry, profile)) 
        goto bailout;
    if(0 > read_settings(ld, entry, profile)) 
        goto bailout;
    
    *prof = profile;

    return PP_SUC;
    
 bailout:
    pp_profile_release(profile);
    free(name);
    return PP_ERR;
}

/*
 * reads the name of a ldap profile and stores it in the according
 * field of the profile structure
 */
static int
read_name(pp_ldap_t* ld, LDAPMessage* entry, char** name) {
    char** vals;
    int ret = PP_SUC;
    
    if ((vals = ldap_get_values(ld->ld, entry, nbx_profile_attrtype)) !=NULL) {
        *name = vals[0] ? strdup(vals[0]) : NULL;
	ldap_value_free(vals);
    } else if(ld_get_errno(ld->ld) !=  LDAP_DECODING_ERROR) {
	ldap_perror(ld->ld, "read_name: ldap_get_values");
	errno = PP_LDAP_ERR_OPERATION_FAILED;	
	ret = PP_ERR;
    } else
        *name = NULL;
    return ret;
}

/*
 * parses the acl data in entry and build new-cfg-compliant acl entries.
 * acl objects are generated by acl lib
 */

static int
read_acl(pp_ldap_t *ld, LDAPMessage *entry, pp_profile_t* prof) {
    char** vals;
    int i, ret = 0;
    pp_acl_t *acl;

    assert(prof);

    acl=pp_acl_create();

    if ((vals = ldap_get_values(ld->ld, entry, "ppAcl")) != NULL) {
	for (i = 0; vals[i] != NULL; i++) {
	    char * tok_ptr;
	    char * obj;
	    char * perm;
	    
	    if ((obj = strtok_r(vals[i], ":", &tok_ptr)) != NULL) {
		obj = strdup(obj);
	    }
	    
	    perm = strtok_r(NULL, ":", &tok_ptr);
	    while (perm) {
		if (perm[0] == '+' || perm[0] == '-') {
		    if(pp_acl_add_entry(acl, obj, &perm[1], perm[0] != '+')==PP_ERR)
			pp_log_err("read_acl: Reading ACL entry (%s:%s) failed\n", obj, perm);
		} else {
		    pp_log("read_acl: ACL entry (%s:%s) parse error\n",
			   obj, perm);
		}
		perm = strtok_r(NULL, ":", &tok_ptr);
	    }
	    
	    free(obj);
	}
	ldap_value_free(vals); vals = NULL;
	pp_profile_set_ldap_acl(prof,acl);
    } else if(ld_get_errno(ld->ld) !=  LDAP_DECODING_ERROR) {
	ldap_perror(ld->ld, "read_acl: ldap_get_values");
	errno = PP_LDAP_ERR_OPERATION_FAILED;	
	ret = PP_ERR;
    }    
    return ret;
}

static int
read_settings(pp_ldap_t *ld, LDAPMessage *entry, pp_profile_t* prof) {
    int i, ret = PP_SUC;
    char** vals;
    if ((vals = ldap_get_values(ld->ld, entry, "ppSetting")) != NULL) {
	for (i = 0; vals[i] != NULL; ++i) {
	    char key[MAX_OPT_KEY_LEN + 1];
	    char value[MAX_OPT_VALUE_LEN + 1];
	    char* valptr;
	    char* setting;
	    int n;

	    if ((setting = pp_utf8_to_latin(vals[i], NULL)) == NULL) {
		pp_log("read_settings: pp_utf8_to_latin() failed\n");
		continue;
	    }
	    
            n = sscanf(setting, "%128[^= \t\n]=%2048[^\n]", key, value);

	    free(setting);
	    
	    if (n < 1) {
		pp_log("read_settings: Option parse error occured.\n");
	    } else {
		if (n < 2) valptr = NULL;
		else valptr = value;
		if(valptr) {
                    pp_profile_set(valptr, prof, key, NULL);
//printf("tweb: setting '%s=%s'\n", key, valptr);
		}
	    }
	}
	ldap_value_free(vals); vals = NULL;
    } else if(ld_get_errno(ld->ld) !=  LDAP_DECODING_ERROR) {
	ldap_perror(ld->ld, "read_settings: ldap_get_values");
	errno = PP_LDAP_ERR_OPERATION_FAILED;	
	ret = PP_ERR;
    }    
    return ret;
}

static int
ldsearch_usrgrp_profalias(LDAPMessage** msg, pp_ldap_t *ld, const char* uid,
			  const char* moid) {
    LDAPMessage *res;
    char *flt = NULL, **grps = NULL, **grp,**clts=NULL,**clt;
    int flt_len = 0, ret;

    if(PP_ERR == (ret = get_group_memberships(&grps, ld, uid)))
        goto error;

    if(PP_ERR == (ret = get_cluster_memberships(&clts, ld, moid)))
        goto error;

    // construct a filter to search for all relevant profiles based on
    // user, groups, device, clusters
    
    pp_dsprintf(&flt, &flt_len, 0,
		"(&(objectclass=%s)(|(ppAliasedUser=%s)",
		nbx_usrassign_ocvalue, uid);
    for(grp = grps; *grp != NULL; ++grp) {
	pp_dstrcat(&flt, &flt_len, "(ppAliasedGroup=");
	pp_dstrcat(&flt, &flt_len, *grp);
	pp_dstrcat(&flt, &flt_len, ")");
    }
    pp_dstrcat(&flt, &flt_len, ")(|(ppAliasedDevice=");
    pp_dstrcat(&flt, &flt_len, moid);
    pp_dstrcat(&flt, &flt_len, ")");
    for(clt = clts; *clt != NULL; ++clt) {
	pp_dstrcat(&flt, &flt_len, "(ppAliasedCluster=");
	pp_dstrcat(&flt, &flt_len, *clt);
	pp_dstrcat(&flt, &flt_len, ")");
    }
    pp_dstrcat(&flt, &flt_len, "))");


    /* retrieve profiles from ldap server */
    D("ldsearch_usrgrp_profalias: %s\n", flt);
    if (ldap_search_s(ld->ld, get_nbx_base_dn(), LDAP_SCOPE_SUBTREE, flt, NULL, 0, &res)
	!= LDAP_SUCCESS ) {
	ldap_perror(ld->ld, "ldsearch_usrgrp_profalias: ldap_search_s");
	errno = PP_LDAP_ERR_OPERATION_FAILED;
        ret = PP_ERR;
        goto error;
    }
    
    ret = PP_SUC;
    *msg = res;
 error:
    free_strstr(grps);
    free(flt);
    return ret;
}

static int
ldget_user_groups(char*** groups, pp_ldap_t *ld, const char* uid) {
    LDAPMessage *res = NULL;
    char *bdn = NULL, *flt = NULL, *user_dn;
    int flt_len = 0, bdn_len = 0, ret;
    char* rattrs[] = { NULL, NULL };
    char* server_type;
    
    D("ldget_user_groups: fetch %s\n", uid);
    
    if(PP_ERR == (ret = lookup_user_dn(&user_dn, ld, uid))) 
        goto bail;

    /* look for the attributenames etc. according to our current servertype */
	if(PP_ERR==pp_cfg_get_at_layer_nodflt(PP_PROFILE_LOCAL,&server_type,"pemx.user_ldap.server_type")
	    //    if (NULL == (server_type =  pp_dp_get_bm_prop(PP_BM_USER_LDAP_SERVER_TYPE))
	|| !strcmp(server_type, emx_srv_type)) {  // our beloved pemx.. :-)
	pp_dsprintf(&bdn, &bdn_len, 0, "ou=%s,%s",
		    nbx_grpou_attrvalue, get_nbx_base_dn());
	pp_dsprintf(&flt, &flt_len, 0,
		    "(&(objectclass=groupOfUniqueNames)(uniqueMember=%s))",
		    user_dn);
	rattrs[0] = strdup(nbx_grp_attrtype);
    } else { // something else than pemx
	const char *grp_objectclass, *grp_name_attrtype, *grp_memb_attrtype;
	const char *grp_search_filter;

	grp_objectclass="groupOfUniqueNames";
	grp_name_attrtype="cn";
	grp_memb_attrtype="uniqueMember";
	grp_search_filter="";
	
	pp_dsprintf(&bdn, &bdn_len, 0, "%s", get_user_base_dn());
	pp_dsprintf(&flt, &flt_len, 0, "(&(objectclass=%s)(%s=%s)%s)",
		    grp_objectclass, grp_memb_attrtype, user_dn,
		    grp_search_filter ? grp_search_filter : "");
	rattrs[0] = strdup(grp_name_attrtype);
    }
    free(server_type);

    D("ldget_user_groups: bdn: %s flt: %s rattr: %s\n", bdn, flt, rattrs[0]);
    if(0 > ld_connect(ld, LDAP_SERVER_TYPE_USER)) {
        ret = PP_ERR;
        goto bail;
    }
        
    /* retrieve groups from ldap server */
    if (ldap_search_s(ld->ld_user, bdn, LDAP_SCOPE_ONELEVEL, flt,
		      NULL, 0, &res) != LDAP_SUCCESS) {
	ldap_perror(ld->ld, "ldget_user_groups: ldap_search_s");
	errno = PP_LDAP_ERR_OPERATION_FAILED;
        ret = PP_ERR;
	goto bail;
    }

    *groups = extract_attrval_from_ldres(ld->ld_user, res, rattrs[0]);
    ret = *groups ? PP_SUC : PP_ERR;
    ldap_msgfree(res);

 bail:
    free(bdn);
    free(flt);
    free(user_dn);
    free(rattrs[0]);
    return ret;
}

static int ldget_mo_clusters(char*** clusters, pp_ldap_t *ld, const char* eui64) {
    LDAPMessage *res;
    moinfo_t* moinfo;
    char *bdn = NULL, *flt = NULL;
    int flt_len = 0, bdn_len = 0, ret;
    const char* rattrs[] = { nbx_clt_attrtype, NULL };
    
    D("ldget_mo_clusters: fetch %s\n", eui64);

    if (PP_ERR == (ret = lookup_mo_info(&moinfo, ld, eui64)))
        return PP_ERR;
	
    if(0 > ld_connect(ld, LDAP_SERVER_TYPE_NORBOX)) 
        return PP_ERR;

    pp_dsprintf(&bdn, &bdn_len, 0,
		"ou=%s,%s", nbx_moou_attrvalue, get_nbx_base_dn());
    pp_dsprintf(&flt, &flt_len, 0,
		"(&(objectclass=ppCluster)(uniqueMember=%s))",
		moinfo->nbx_bdn);

    D("ldget_mo_clusters: %s %s\n", bdn, flt);
    /* retrieve clusters from ldap server */
    if (ldap_search_s(ld->ld, bdn, LDAP_SCOPE_ONELEVEL, flt, rattrs, 0, &res)
	!= LDAP_SUCCESS ) {
	ldap_perror(ld->ld, "ldget_mo_clusters: ldap_search_s");
	errno = PP_LDAP_ERR_OPERATION_FAILED;
        ret = PP_ERR;
	goto bailout;
    }
    *clusters = extract_attrval_from_ldres(ld->ld, res, rattrs[0]);
    ret = *clusters ? PP_SUC : PP_ERR;
    ldap_msgfree(res);

 bailout:
    destroy_moinfo(moinfo);
    free(bdn);
    free(flt);
    return ret;
}

static char**
extract_attrval_from_ldres(LDAP* ld, LDAPMessage *res, const char* attr) {
    LDAPMessage *entry;
    char** retvals;
    int n, i;
    if(0 > (n = ldap_count_entries(ld, res))) {
	ldap_perror(ld, "extract_attrval_from_ldres: ldap_count_entries");
	errno = PP_LDAP_ERR_OPERATION_FAILED;
	return NULL;
    }
    retvals = malloc((n + 1) * sizeof(char*));
    for(entry = ldap_first_entry(ld, res), i = 0; entry != NULL;
	entry = ldap_next_entry(ld, entry)) {
	char** vals;
	if(NULL == (vals = ldap_get_values(ld, entry, attr))
	   || NULL == vals[0]) continue;
	retvals[i++] = strdup(vals[0]);
	ldap_value_free(vals);
    }
    retvals[i] = NULL;
    return retvals;
}

static int ldget_mo_info(moinfo_t** moinfo, pp_ldap_t* ld, const char* eui64) {
    LDAPMessage *res = NULL, *entry;
    char *bdn = NULL, *flt = NULL, *modn = NULL;
    int flt_len = 0, bdn_len = 0, ret = PP_ERR;
    const char* rattrs[] = { "dn", "cn", NULL };
    char** vals;

    D("ldget_mo_info: fetch %s\n", eui64);

    if(0 > ld_connect(ld, LDAP_SERVER_TYPE_NORBOX))
        return PP_ERR;

    pp_dsprintf(&bdn, &bdn_len, 0,
		"ou=%s,%s", nbx_moou_attrvalue, get_nbx_base_dn());
    pp_dsprintf(&flt, &flt_len, 0, "(&(objectClass=ppDevice)(%s=%s))",
		nbx_eui64_attrtype, eui64);

    D("ldget_mo_info: %s %s\n", bdn, flt);

    /* retrieve MO from ldap server */
    if (ldap_search_s(ld->ld, bdn, LDAP_SCOPE_ONELEVEL, flt, rattrs, 0, &res)
	!= LDAP_SUCCESS ) {
	ldap_perror(ld->ld, "ldget_mo_info: ldap_search_s");
	errno = PP_LDAP_ERR_OPERATION_FAILED;
	goto bailout;
    }

    /* get the dn out of the result */
    if(NULL == (modn = extract_dn_from_ldres(ld->ld, res, &entry)))
	goto bailout;

    assert(modn);
    D("ldget_mo_info: %s\n", modn);

    /* get the mo name out of same entry as dn */
    /* NOTE: this stupid LDAP API has no way of getting the ld_errno,
     * so I'll treat any error as a 'attr is not present'  */
    if(NULL != (vals = ldap_get_values(ld->ld, entry, rattrs[1]))) {
	D("ldget_mo_info: %s\n", vals[0]);
	*moinfo = create_moinfo(modn, vals[0]);
	ldap_value_free(vals);
    } else {
	*moinfo = create_moinfo(modn, NULL);
    }
    ret = PP_SUC;

 bailout:
    free (modn);
    if (res) ldap_msgfree(res);
    free (bdn);
    free (flt);
    return ret;
}

static int generate_ldap_env(pp_ldap_t* ld, const char *name, 
                             char **p_dn, char **p_filter) {
    char filter[256];
    const char * filter_pattern = "(&(%s=%s)(objectclass=%s)%s)";
#if defined(PP_FEAT_RARITAN_STYLE_LDAP_SETTINGS)
    char * base_search = NULL;
#else /* PP_FEAT_RARITAN_STYLE_LDAP_SETTINGS */
    char * ldap_base_dn = NULL;
#endif /* PP_FEAT_RARITAN_STYLE_LDAP_SETTINGS */
    char * user_subfilter = NULL;
    char * login_attrtype = NULL;
    char * user_objectclass = NULL;
    char * ldap_server_type = NULL;
    int ret = PP_ERR;

    assert(name);
    assert(p_dn);
    assert(p_filter);

#if defined(PP_FEAT_RARITAN_STYLE_LDAP_SETTINGS)
    pp_cfg_get(&base_search, "ldap.base_search");
    user_subfilter = strdup(""); // lazy one
    pp_cfg_get(&ldap_server_type, "ldap.server_type");
    (void)ld;
#else /* PP_FEAT_RARITAN_STYLE_LDAP_SETTINGS */
    if (!ld->use_minimal_ldap) {
        pp_cfg_get_at_layer_nodflt(PP_PROFILE_STACK(PP_PROFILE_LOCAL),
				   &ldap_server_type,
				   "pemx.user_ldap.server_type");
	pp_cfg_get_at_layer_nodflt(PP_PROFILE_STACK(PP_PROFILE_LOCAL),
				   &ldap_base_dn, "pemx.user_ldap.base_dn");
	pp_cfg_get_at_layer_nodflt(PP_PROFILE_STACK(PP_PROFILE_LOCAL),
				   &login_attrtype,
				   "pemx.user_ldap.login_attrtype");
	pp_cfg_get_at_layer_nodflt(PP_PROFILE_STACK(PP_PROFILE_LOCAL),
				   &user_objectclass,
				   "pemx.user_ldap.user_objectclass");
	pp_cfg_get_at_layer_nodflt(PP_PROFILE_STACK(PP_PROFILE_LOCAL),
				   &user_subfilter,
				   "pemx.user_ldap.user_search_subfilter");
    } else {
        pp_cfg_get(&ldap_server_type, "ldap.server_type");
	pp_cfg_get(&ldap_base_dn, "ldap.base_dn");
	pp_cfg_get(&login_attrtype, "ldap.login_attrtype");
	pp_cfg_get(&user_objectclass, "ldap.user_objectclass");
	pp_cfg_get(&user_subfilter, "ldap.user_search_subfilter");
    }
#endif /* PP_FEAT_RARITAN_STYLE_LDAP_SETTINGS */
    
    if (!ldap_server_type || !pp_strcmp_safe(ldap_server_type,"")
#ifndef PP_FEAT_RARITAN_STYLE_LDAP_SETTINGS
	|| !ldap_base_dn || !pp_strcmp_safe(ldap_base_dn,"")
#endif /* !PP_FEAT_RARITAN_STYLE_LDAP_SETTINGS */
        ) {
	errno = PP_LDAP_ERR_CONFIG_ERROR;
	goto bail;
    }

    if (!pp_strcmp_safe(login_attrtype, "")) {
	free(login_attrtype);
	login_attrtype = NULL;
    }
    if (!pp_strcmp_safe(user_objectclass, "")) {
	free(user_objectclass);
	user_objectclass = NULL;
    }
    
    /* set some defaults */
    if (!strcmp(ldap_server_type, emx_srv_type)) {
	if (!login_attrtype) login_attrtype = strdup(nbx_usr_attrtype);
	if (!user_objectclass) user_objectclass = strdup(nbx_usr_objectclass);
    } else if (!strcmp(ldap_server_type, ads_srv_type)) {
	if (!login_attrtype) login_attrtype = strdup("sAMAccountName");
	if (!user_objectclass) user_objectclass = strdup("user");
    } else if (!strcmp(ldap_server_type, nds_srv_type) ||
	       !strcmp(ldap_server_type, "generic")) {
	if (!login_attrtype) login_attrtype = strdup("uid");
	if (!user_objectclass) user_objectclass = strdup("inetOrgPerson");
    } 

    /* sanity checks */
    if (!login_attrtype || !user_objectclass) {
	errno = PP_LDAP_ERR_CONFIG_ERROR;
	goto bail;
    }
    
#ifndef PP_FEAT_RARITAN_STYLE_LDAP_SETTINGS
    if(user_subfilter == NULL) user_subfilter = strdup("");
#endif /* !PP_FEAT_RARITAN_STYLE_LDAP_SETTINGS */

    /* generate filter string */
    snprintf(filter, sizeof(filter), filter_pattern, login_attrtype,
	     name, user_objectclass, user_subfilter);
    *p_filter = strdup(filter);
#if defined(PP_FEAT_RARITAN_STYLE_LDAP_SETTINGS)
    *p_dn = base_search;
    base_search = NULL;
#else /* PP_FEAT_RARITAN_STYLE_LDAP_SETTINGS */
    *p_dn = ldap_base_dn;
    ldap_base_dn = NULL;
#endif /* PP_FEAT_RARITAN_STYLE_LDAP_SETTINGS */

    ret = PP_SUC;

 bail:
    free(ldap_server_type);
    free(login_attrtype);
    free(user_objectclass);
#if defined(PP_FEAT_RARITAN_STYLE_LDAP_SETTINGS)
    free(base_search);
#else /* PP_FEAT_RARITAN_STYLE_LDAP_SETTINGS */
    free(ldap_base_dn);
    free(user_subfilter);
#endif /* !PP_FEAT_RARITAN_STYLE_LDAP_SETTINGS */

    return ret;
}

static int ldget_user_dn(char** dn, pp_ldap_t * ld, const char * name) {
    LDAPMessage *res = NULL;
    char * filter = NULL;
    char * search_dn = NULL;
    const char * attrs[] = { "dn", NULL };
    int ret = PP_ERR;

    D("ldget_user_dn: fetch %s\n", name);
    
    if(0 > ld_connect(ld, LDAP_SERVER_TYPE_USER))
        return PP_ERR;

    if(generate_ldap_env(ld, name, &search_dn, &filter) != PP_SUC) {
        goto bail;
    }

    /* search user */
    D("ldget_user_dn: %s %s\n", search_dn, filter);
    if (ldap_search_s(ld->ld_user, search_dn, LDAP_SCOPE_SUBTREE,
		      filter, attrs, 0, &res)  != LDAP_SUCCESS ) {
	ldap_perror(ld->ld_user, "lookup_user_dn: ldap_search_s");
	errno = PP_LDAP_ERR_OPERATION_FAILED;
	goto bail;
    }
    /* get the dn out of the result */
    *dn = extract_dn_from_ldres(ld->ld_user, res, NULL);
    ret = *dn ? PP_SUC : PP_ERR;
    D("ldget_user_dn: %s\n", *dn);
    
 bail:
    if (res) ldap_msgfree(res);
    free(search_dn);
    free(filter);

    return ret;
}

/*
 * in case entry is a valid pointer ( != NULL)
 * entry will contain the result entry, out of which the dn was
 * taken, in case of error the pointer will be left unchanged
 */
char* extract_dn_from_ldres(LDAP* ld, LDAPMessage *res, LDAPMessage **pentry) {
    LDAPMessage* entry;
    int n;
    char *dn = NULL;
    
    /* check if we have found at least entry */
    if ((n = ldap_count_entries(ld, res)) < 0) {
	ldap_perror(ld, "extract_dn_from_ldres: ldap_count_entries");
        goto bailout;
    }

    if (n > 0) {
	/* extract the entry from the search result */
	if ((entry = ldap_first_entry(ld, res)) == NULL) {
	    ldap_perror(ld, "extract_dn_from_ldres: ldap_first_entry");
            goto bailout;
	}

	/* get the DN of the entry */
	if(NULL == (dn = ldap_get_dn(ld, entry))) {
	    ldap_perror(ld, "extract_dn_from_ldres: ldap_get_dn");
            goto bailout;
	}

	/* return entry for later use */
	if(pentry != NULL) *pentry = entry;
    } else
        goto bailout;
    
    return dn;
    
 bailout:
    errno = PP_LDAP_ERR_OPERATION_FAILED;
    return NULL;
}
 
static moinfo_t* create_moinfo(const char* bdn, const char* name) {
    pthread_mutexattr_t err_check_mtx_attr;
    int mutex_error;
    moinfo_t* p;

    p = (moinfo_t *)malloc(sizeof(moinfo_t));
    memset(p, 0, sizeof(moinfo_t));
    p->nbx_bdn = strdup(bdn);
    if(name)
        p->nbx_name = strdup(name);
    
    // create error checking mutex
    mutex_error = pthread_mutexattr_init(&err_check_mtx_attr);
    assert(mutex_error == 0);
    mutex_error = pthread_mutexattr_settype(&err_check_mtx_attr,
                                            PTHREAD_MUTEX_ERRORCHECK_NP);
    assert(mutex_error == 0);
    mutex_error = pthread_mutex_init(&p->ref_cnt_mtx, &err_check_mtx_attr);
    p->ref_cnt = 1;

    return p;
}

/*
 * mo info is reference counted,
 */
static moinfo_t* duplicate_moinfo(moinfo_t* info) {
    MUTEX_LOCK(&info->ref_cnt_mtx);
    assert(info->ref_cnt > 0);
    info->ref_cnt++;
    MUTEX_UNLOCK(&info->ref_cnt_mtx);
    return info;
}

static void destroy_moinfo(void* p) {
    moinfo_t* info = (moinfo_t*)p;
    if (info == NULL) return;
    MUTEX_LOCK(&info->ref_cnt_mtx);
    assert(info->ref_cnt > 0);
    if(--info->ref_cnt == 0) {
	int mtxerr;
	// we are the one and only, so unlock is safe
	MUTEX_UNLOCK(&info->ref_cnt_mtx);
	mtxerr = pthread_mutex_destroy(&info->ref_cnt_mtx);
	assert(mtxerr == 0);
	free (info->nbx_bdn);
	free (info->nbx_name);
	free (info);
	return;
    }
    MUTEX_UNLOCK(&info->ref_cnt_mtx);
}

// duplicate a string of strings
static char** duplicate_strstr(char** strs) {
    int i, n;
    char** res;
    for(n = 0; strs[n] != NULL; ++n);
    res = malloc((n + 1) * sizeof(char*));
    for(i = 0; i < n; ++i) {
	res[i] = strdup(strs[i]);
    }
    res[i] = NULL;
    return res;
}

// free a string of strings
static void free_strstr(void* strs) {
    char** strp = (char**)strs;
    char*  str;
    if(strs != NULL) {
	while(NULL != (str = *strp++)) {
	    free(str);
	}
	free(strs);
    }
}

static int ld_get_errno(LDAP* ld) {
    int lderr;
    ldap_get_option(ld, LDAP_OPT_ERROR_NUMBER, &lderr);
    return lderr;
}

static char* get_nbx_base_dn() {
    if (nbx_base_dn == NULL) {
	if (PP_FAILED(pp_cfg_get_at_layer_nodflt(PP_PROFILE_LOCAL,
						     &nbx_base_dn,
						     "pemx.pemx_ldap.base_dn"))) {
	    nbx_base_dn = strdup(std_nbx_base_dn);
	}
    }
    return nbx_base_dn;
}

static char* get_user_base_dn() {
    if (user_base_dn == NULL) {
	if (PP_FAILED(pp_cfg_get_at_layer_nodflt(PP_PROFILE_LOCAL,
						     &user_base_dn,
						     "pemx.user_ldap.base_dn"))) {
	    user_base_dn = strdup(std_nbx_base_dn);
	}
    }
    return user_base_dn;
}

/**
 * extends cfg_get_profile_type_stack_core for the use of LDAP profiles
 */
int pp_cfg_get_profile_type_stack_core(pp_profile_type_stack_t *prof_type_stack, 
                                       pp_profile_type_t prof_type,
                                       const vector_t* key_comps) {
    int ret = PP_SUC, uid = -1;
    pp_profile_type_t ldap_prof_type;
    pp_profile_type_t real_type = PP_REAL_PROFILE(prof_type);
    int effective = PP_PROFILE_IS_STACK(prof_type);
    pp_profile_type_stack_t pts = PP_PROFILE_TYPE_STACK_T_INITIALIZER;
    
    /* only if LDAP prof is affected */
    if(real_type >= PP_PROFILE_LDAP_DEV && pp_ldap_is_active(NULL, NULL) && 
       /* do not get LDAP profile for superuser */
       (PP_ERR == pp_cfg_is_user_key(&uid, key_comps) || uid)) {
        if (effective) {
            ldap_prof_type = uid>0 ? PP_PROFILE_LDAP_USER : PP_PROFILE_LDAP_DEV;
        } else {
            ldap_prof_type = prof_type;
        }
        
        pts.profile_type_stack[0] = ldap_prof_type;
        pts.profile_type_stack[1] = PP_PROFILE_NONE;
        pts.size = 1;
    }

    if(effective)
        ret = __pp_cfg_get_profile_type_stack_core(&pts, prof_type, key_comps);
    
//    if(ret != PP_ERR)
    {
        // copy local profile type stack
        memcpy(prof_type_stack->profile_type_stack, pts.profile_type_stack,
               sizeof(pp_profile_type_t) * pts.size);
        prof_type_stack->size = pts.size;
    }

    return ret;
}
 
/**
 * extends cfg_get_profile_stack_core for the use of LDAP profiles
 */
int pp_cfg_get_profile_stack_core(vector_t** prof_stack,
			          pp_profile_type_t prof_type,
			          const vector_t *key_comps) {
    vector_t* ps = vector_new(NULL, PP_PROFILE_COUNT, pp_profile_release);
    pp_profile_t* p = NULL;
    int ret = PP_ERR, only_ldap = 0, uid = -1;
    pp_profile_type_t ldap_prof_type;
    pp_profile_type_t real_type = PP_REAL_PROFILE(prof_type);
    int effective = PP_PROFILE_IS_STACK(prof_type);
    
    /* only if LDAP prof is affected */
    if(real_type >= PP_PROFILE_LDAP_DEV && pp_ldap_is_active(NULL, NULL) && 
       /* do not get LDAP profile for superuser */
       (PP_ERR == pp_cfg_is_user_key(&uid, key_comps) || uid)) {
        if (effective) {
            ldap_prof_type = uid>0 ? PP_PROFILE_LDAP_USER : PP_PROFILE_LDAP_DEV;
        } else {
            ldap_prof_type = prof_type;
            only_ldap = 1;
        }
       
        if (PP_ERR != (ret = pp_cfg_get_profile_core(&p, ldap_prof_type,
                                                     key_comps, uid))) {
            assert(p);
            vector_add(ps, p);
        } else if (ldap_prof_type != PP_PROFILE_LDAP_DEV) {
            vector_delete(ps);
            return ret;
        }
    }
   
    if(!only_ldap)
        ret = __pp_cfg_get_profile_stack_core(&ps, prof_type, key_comps);
    
    /**
     * if an error occured (in pp_ldap_get_devprofile for only LDAP case or in   
     * __pp_cfg_get_profile_stack_core for the local one), delete temp vector 
     * and return, prof_stack stays untouched
     */
    if(ret == PP_ERR) {
        vector_delete(ps);
        return ret;
    }
    
    if(*prof_stack) {
        /* if prof_stack is already initialized, merge vectors */
        vector_addvec(*prof_stack, ps, NULL);
        vector_delete(ps);
    } else {
        /* else use the vector just created */
        *prof_stack = ps;
    }
    
    return ret;
}

int
pp_cfg_get_profile_core(pp_profile_t** prof, pp_profile_type_t prof_type,
		 const vector_t* key_comps UNUSED, int uid) {
    int ret = PP_ERR;
    char *user;
    pp_profile_type_t real_type = PP_REAL_PROFILE(prof_type);
    assert(!PP_PROFILE_IS_STACK(prof_type));
    switch (real_type) {
        case PP_PROFILE_LDAP_USER:
	    user=pp_ldap_get_temp_user(uid);
	    if(!user) {
		ret=PP_ERR;
		break;
	    }
            pp_ldap_get_profile(prof, pp_eui64_str, user);
            ret = *prof ? PP_SUC : PP_ERR;
            break;
        case PP_PROFILE_LDAP_DEV:
	    /* tags have vanished for dev profiles, always NULL */
            pp_ldap_get_devprofile(prof, pp_eui64_str);
            ret = *prof ? PP_SUC : PP_ERR;
            break;
      default:
	  ret = __pp_cfg_get_profile_core(prof, prof_type, key_comps, uid);
	  break;
    }
    return ret;
}

/******************** LDAP specific profile operations ********************/
#if 0
static int 
ld_acl_load_from_prof(pp_acl_t **acl, pp_profile_t *prof, const char *uname) {
    const char* obj;
    const char* perm;
    pp_acl_t* nacl = pp_acl_create();
    cfg_entry_t *cfge, *acle;
    pp_ditr_t *iter, *acliter;
    pp_dict_t* d;
    vector_t* key_comps = pp_cfg_key_expand("user[%s].acl[", uname);

    cfge = pp_cfg_key_resolve_exact(&d, prof, key_comps);
    assert(cfge->type == CFG_DICT);
    pp_cfg_key_comps_destroy(key_comps);
    
    iter = pp_ditr_new(cfge->value.dict);
    while(pp_ditr_next(iter)) {
        obj = (char*)pp_ditr_key(iter);
        acle = (cfg_entry_t*)pp_ditr_data(iter);
        assert(acle->type == CFG_DICT);
		
        /* get user._e_.username.acl._e_.aclname._e_ dict from acl dict */
        if(NULL == (acle = (cfg_entry_t*)pp_dict_search(acle->value.dict, 
                                                        pp_cfg_vect_elems_comp)))
            goto bailout;
        assert(acle->type == CFG_DICT);
		
        acliter = pp_ditr_new(acle->value.dict);
        while(pp_ditr_next(acliter)) {
            acle = (cfg_entry_t*)pp_ditr_data(acliter);
            assert(acle->type == CFG_STR);
            /* finally, we got the perms */
            perm = acle->value.string;
            if (perm[0] == '+' || perm[0] == '-') {
				//printf("tweb: Adding ACL entry '%s: %s'\n", obj, perm);
                if(0 > pp_acl_add_entry(nacl, obj, &perm[1], perm[0] != '+')) {
                    pp_log("Adding ACL entry '%s: %s' failed: %s\n",
                           obj, perm, pp_error_string(errno));
                }
            } else {
                pp_log("ACL parse error\n");
            }
        }
    }
	
    *acl = nacl;
    return PP_SUC;
    
 bailout:
    pp_acl_destroy(nacl);
    return PP_ERR;
}
#endif
/*
 * Parameter forward means we merge with raising index of the
 * profiles in the overgiven vector, i.e. 0, 1, 2
 * If forward is false, we merge in the oposite order.
 * This is important, as the 'left sided' profile of the
 * profile-union operation has kinda higher priority for settings
 */
static int 
ld_profile_union(pp_profile_t **prof, const char *uname, vector_t *profs, int forward) {
    pp_profile_t *nprof, *cprof, *cdp;
    vector_t doneprofs;
    unsigned int i, j, start, end, profs_sz, doneprofs_sz;
    int inc, found;
    pp_acl_t *nacl;

    vector_new(&doneprofs, 16, NULL);
    
    profs_sz = vector_size(profs);
    if(NULL == profs || 0 == profs_sz)
        return PP_ERR;

    if((nprof = pp_profile_create(pp_mallocator_heap(), PP_PROFILE_LDAP_USER, 
                                  uname ? uname : "unioned_noname")) == NULL)
	return PP_ERR;
    
    nacl = pp_acl_create();

    // adjust loop bounds for direction
    if (forward) {
	start = 0; end = vector_size(profs); inc = 1;
    } else {
	start = vector_size(profs) - 1; end = 0 - 1; inc = -1;
    }
    
    for(i = start; i != end; i += inc) {
	found = 0;
	cprof = vector_get(profs, i);
	// check whether it's a duplicate of something we merged before
        doneprofs_sz = vector_size(&doneprofs);
	for(j = 0; j < doneprofs_sz; j++) {
	    cdp = vector_get(&doneprofs, j);
	    if(cdp == cprof) {
		found = 1;
		break;
	    }
	}
/* TODO: no LDAP with RAASIP yet */
#if 0
	if(!found) {
	    // unify with nprof
            if(pp_acl_union(nacl, cprof->ldap_acl, 1 /*optimistic*/) == PP_ERR)
		goto bailout;
            if(ld_profile_settings_union(nprof, cprof) == PP_ERR)
                goto bailout;
	    // remember in doneprofs, so we wont unify same thing again
	    vector_add(&doneprofs, cprof);
	}
#endif
    }
    vector_delete(&doneprofs);
    nprof->ldap_acl=nacl;
    *prof = nprof;
    return PP_SUC;
 //bailout:
    vector_delete(&doneprofs);
    pp_profile_release(nprof);
    pp_acl_destroy(nacl);
    return PP_ERR;
}

/*
 * union of settings will be calculated as follows
 * - all scalar settings not part of s1 but of s2 will be added to s1
 * - scalar setting types that are present in s1 and s2 will be
 *   left unchanged, no matter whether they are different or not
 * - vector settings of s2 that are not there in s1, i.e. the basename
 *   is not present at all, will be entirely copied to s1
 * - vector settings of s1 will be left unchanged, even if
 *   s2 contains the same vector base name with different elements
 * (this is done by copying the whole configs dictionary from prof
 * to new prof without overwriting existing cfg_keys)
 * allways returns PP_SUC, since we don't care, if we really copy or not
 */
#if 0
static int
ld_profile_settings_union(pp_profile_t* nprof, pp_profile_t* prof) {
    cfg_entry_t *e;
    pp_ditr_t* ditr;
    char *key;
    
    /* loop over username dict of prof */
    ditr = pp_ditr_new(prof->configs);
    while(pp_ditr_next(ditr)) {
        key = (char*)pp_ditr_key(ditr);
        e = pp_ditr_data(ditr);
        pp_cfg_copy_entry(prof, nprof->configs, key, e, 0);
    }

    return PP_SUC;
}
#endif /* 0 */

static int temp_id=50000;

int 
pp_ldap_create_temp_uid(const char* name)
{
	int id=temp_id++;
	return pp_hash_set_entry_i(temp_uids,id,strdup(name),&free);
}

void 
pp_ldap_remove_temp_uid(int uid)
{
	pp_hash_delete_entry_i(temp_uids,uid);
}

int
pp_ldap_get_temp_uid(const char* name)
{
    unsigned int key=0;
    for(key=pp_hash_get_first_key_i(temp_uids); key>0;
	key=pp_hash_get_next_key_i(temp_uids)) 
	if(!strcmp(name,(char*)pp_hash_get_entry_i(temp_uids,key)))
	    break;

    if(key>0)
	return key;
    else
	return -1;
}

char*
pp_ldap_get_temp_user(int uid)
{
    return (char*) pp_hash_get_entry_i(temp_uids,uid);
}

