#include <pthread.h>
#include <openssl/bio.h>

#include <pp/propchange.h>
#include <pp/rfb.h>
#include <liberic_net.h>
#include <pp_kernel_common.h>
#include <pp_netevent_proto.h>

#include "netevent_intern.h"

/******************************************************************************
* table with erla properties and netevent properties                          *
******************************************************************************/

static struct {
    int netevent_prop;
    int pp_prop;
} prop_table[] = {
    {	NETEVENT_VIDEO_MODE_HAS_CHANGED,	PP_PROP_VIDEO_MODE_CHANGED	},
    {	NETEVENT_KEYBOARD_RESET,		PP_PROP_KEYBOARD_RESET		},
    {	NETEVENT_KVM_SESSION_OPENED,		PP_PROP_KVM_SESSION_OPENED	},
    {	NETEVENT_KVM_SESSION_CLOSED,		PP_PROP_KVM_SESSION_CLOSED	},
};

static int get_pp_prop_from_netevent_prop(int netprop) {
    unsigned int i, size = sizeof(prop_table) / sizeof(prop_table[0]);
    for (i = 0; i < size; i++) {
    	if (prop_table[i].netevent_prop == netprop) {
    	    return prop_table[i].pp_prop;
    	}
    }
    return -1;
}

static int get_netevent_prop_from_pp_prop(int pp_prop) {
    unsigned int i, size = sizeof(prop_table) / sizeof(prop_table[0]);
    for (i = 0; i < size; i++) {
    	if (prop_table[i].pp_prop == pp_prop) {
    	    return prop_table[i].netevent_prop;
    	}
    }
    return -1;
}

/******************************************************************************
* functions for property change handling                                      *
******************************************************************************/

static void
ne_propchange_listener_func (ne_propchange_listener_t * listener,
			     unsigned short _prop, unsigned short prop_flags)
{
    int prop = get_netevent_prop_from_pp_prop(_prop);
    if (prop < 0) return;
    
    if (prop == NETEVENT_KVM_SESSION_OPENED || prop == NETEVENT_KVM_SESSION_CLOSED) {
        prop_flags = pp_rfb_get_non_forensics_connection_count();
    }
    D ("prop (%u,%u) recieved\n", prop, prop_flags);
    ne_write_event(listener, prop, prop_flags);
}

void
ne_propchange_init (ne_propchange_listener_t ** listener, BIO * bio)
{
    D ("ne_propchange_init\n");
    listener[0] =
	(ne_propchange_listener_t *)
	malloc (sizeof (ne_propchange_listener_t));
    memset(listener[0], 0, sizeof(ne_propchange_listener_t));
    pp_propchange_init_listener ((pp_propchange_listener_t *) listener[0],
				 (changed_prop_t)
				 ne_propchange_listener_func);
    listener[0]->bio = bio;
    listener[0]->events = pp_hash_create_i(20);
}

void
ne_propchange_free (ne_propchange_listener_t * listener)
{
    D ("ne_propchange_free\n");
    pp_propchange_remove_all ((pp_propchange_listener_t *) listener);
    pp_hash_delete_i(listener->events);
    free (listener);
}

static void
ne_propchange_add (ne_propchange_listener_t * listener, unsigned _prop)
{
    /* special handling for local input events */
    if (_prop == NETEVENT_LOCAL_MOUSE_EVENT) {
    	listener->want_mouse = 1;
    	return;
    }
    if (_prop == NETEVENT_LOCAL_KEYBOARD_EVENT) {
    	listener->want_keyboard = 1;
    	return;
    }
    
    /* default propchange handling */
    int prop = get_pp_prop_from_netevent_prop(_prop);
    if (prop < 0) return;
    
    if (_prop == NETEVENT_KVM_SESSION_OPENED) {
        ne_write_event(listener, _prop, pp_rfb_get_non_forensics_connection_count());
    }

    D ("ne_propchange_add(%u)\n", prop);
    pp_propchange_add ((pp_propchange_listener_t *) listener, prop);
}

static void
ne_propchange_remove (ne_propchange_listener_t * listener, unsigned _prop)
{
    /* special handling for local input events */
    if (_prop == NETEVENT_LOCAL_MOUSE_EVENT) {
    	listener->want_mouse = 0;
    	return;
    }
    if (_prop == NETEVENT_LOCAL_KEYBOARD_EVENT) {
    	listener->want_keyboard = 0;
    	return;
    }
    
    /* default propchange handling */
    int prop = get_pp_prop_from_netevent_prop(_prop);
    if (prop < 0) return;
    
    D ("ne_propchange_add(%u)\n", prop);
    pp_propchange_remove ((pp_propchange_listener_t *) listener, prop);
}

void ne_propchange_register(ne_propchange_listener_t * listener, unsigned prop, int unregister) {
    if (prop == NETEVENT_PING_PONG) {
    	ne_write_event(listener, NETEVENT_PING_PONG, 0);
    	return;
    }
    if (unregister) {
    	if (pp_hash_get_entry_i(listener->events, prop) != NULL) {
    	    ne_propchange_remove(listener, prop);
    	    pp_hash_delete_entry_i(listener->events, prop);
    	} else {
    	    D("ne_propchange_register: event to unregister not yet registered.\n");
    	}
    } else {
    	if (pp_hash_get_entry_i(listener->events, prop) == NULL) {
    	    pp_hash_set_entry_i(listener->events, prop, (void *)1, NULL);
    	    ne_propchange_add(listener, prop);
    	} else {
    	    D("ne_propchange_register: event to register already registered.\n");
    	}
    }
}

/******************************************************************************
* functions for local input event handling                                    *
******************************************************************************/

static void ne_local_key_event(pp_km_encoder_desc_t *this, unsigned char key);
static void ne_local_mouse_event(pp_km_encoder_desc_t *this, int x, int y, int z, unsigned char bm);

static pp_km_encoder_desc_t netevent_km_encoder = {
    encoder_send_local_key_event:	ne_local_key_event,
    encoder_send_local_mouse_event:	ne_local_mouse_event,
};

void ne_km_encoder_init (ne_km_encoder_desc_t *encoder, ne_propchange_listener_t * listener) {
    memset(encoder, 0, sizeof(ne_km_encoder_desc_t));
    encoder->desc = netevent_km_encoder;
    encoder->listener = listener;
    pp_km_connect_encoder((pp_km_encoder_desc_t *)encoder);
}

void ne_km_encoder_cleanup (ne_km_encoder_desc_t *encoder) {
    pp_km_disconnect_encoder((pp_km_encoder_desc_t *)encoder);
}

static void ne_local_key_event(pp_km_encoder_desc_t *this, unsigned char key UNUSED) {
    ne_km_encoder_desc_t *desc = (ne_km_encoder_desc_t *)this;
    if (desc->listener->want_keyboard) {
    	ne_write_event(desc->listener, NETEVENT_LOCAL_KEYBOARD_EVENT, 0);
    }
}

static void ne_local_mouse_event(pp_km_encoder_desc_t *this, int x UNUSED, int y UNUSED,
				 int z UNUSED, unsigned char bm UNUSED) {
    ne_km_encoder_desc_t *desc = (ne_km_encoder_desc_t *)this;
    if (desc->listener->want_mouse) {
    	ne_write_event(desc->listener, NETEVENT_LOCAL_MOUSE_EVENT, 0);
    }
}
