#include <unistd.h>
#include <pp/base.h>
#include <pp/powerswitch.h>
#include <pp/cfg.h>
#include <pp/serial.h>
#ifdef PP_FEAT_PCI_ADC
#include <pp/sense.h>
#endif
#include <liberic_config.h>
#include <liberic_notify.h>
#include <liberic_cron.h>
#include "liberic_misc.h"
#include "power_switch.h"
#include "power_switch_spc.h"
#include "power_switch_intern.h"
#ifdef PP_FEAT_PSU_ADAPT
 #include "power_switch_psu_adapt.h"
#endif
#include "power_switch_ipm_gpio.h"
#include "power_switch_ipm_tty.h"
#include "power_switch_les.h"
#include "power_switch_lesms.h"
#include "power_switch_btec.h"
#include "power_switch_sentry.h"
#include "power_switch_smart.h"
#include "power_switch_fjsms.h"
#include <pp/intl.h>

#define MAX_LOCK_TRIALS		 3

#define __PS_CTX_LOCK(port_ctx, ret)					\
    {									\
	int lock_trials = MAX_LOCK_TRIALS;				\
	while (lock_trials--) {						\
	    int mtx_ret;						\
	    if ((mtx_ret = pthread_mutex_trylock(&port_ctx->lock)) == 0) { \
		break;							\
	    } else {							\
		if (lock_trials == 0 || mtx_ret != EBUSY) return ret;	\
	    }								\
	    usleep(500000);						\
	}								\
    }

#define __PS_CTX_UNLOCK(port_ctx) MUTEX_UNLOCK(&port_ctx->lock);

/* FIXME: cleanup the chaos, use features.h, or whatever, plz. */
static power_switch_entry_t switches[] = {
#if defined(PRODUCT_XX01IP_ANY)
    { POWER_SWITCH_LESMS_NAME, power_switch_lesms_init, power_switch_lesms_cleanup, PP_POWER_PORT_ID_SERIAL_1 },
    { POWER_SWITCH_SMART_NAME, power_switch_smart_init, power_switch_smart_cleanup, PP_POWER_PORT_ID_SERIAL_1 },
#elif defined(PRODUCT_ERIC2)
    { POWER_SWITCH_IPMGPIO_NAME, power_switch_ipmgpio_init, power_switch_ipmgpio_cleanup, PP_POWER_PORT_ID_SERIAL_1 },
#elif (defined(PRODUCT_RC1) || defined(PRODUCT_ERICXP) || defined(PRODUCT_ERICG4) || defined(PRODUCT_LARAXP)) && !defined(OEM_TANGTOP)
    { POWER_SWITCH_IPMGPIO_NAME, power_switch_ipmgpio_init, power_switch_ipmgpio_cleanup, PP_POWER_PORT_ID_SERIAL_1 },
    { POWER_SWITCH_INTERN_NAME, power_switch_intern_init, power_switch_intern_cleanup, PP_POWER_PORT_ID_INTERNAL },
#else
#if !defined(OEM_TANGTOP)
    { POWER_SWITCH_SENTRY_NAME, power_switch_sentry_init, power_switch_sentry_cleanup, PP_POWER_PORT_ID_SERIAL_1 },
    { POWER_SWITCH_BTEC_NAME, power_switch_btec_init, power_switch_btec_cleanup, PP_POWER_PORT_ID_SERIAL_1 },
    { POWER_SWITCH_IPMGPIO_NAME, power_switch_ipmgpio_init, power_switch_ipmgpio_cleanup, PP_POWER_PORT_ID_SERIAL_1 },
    { POWER_SWITCH_SPC_NAME, power_switch_spc_init, power_switch_spc_cleanup, PP_POWER_PORT_ID_SERIAL_2 },
    { POWER_SWITCH_LES_NAME, power_switch_les_init, power_switch_les_cleanup, PP_POWER_PORT_ID_SERIAL_2 },
    { POWER_SWITCH_LESMS_NAME, power_switch_lesms_init, power_switch_lesms_cleanup, PP_POWER_PORT_ID_SERIAL_2 },
    { POWER_SWITCH_SMART_NAME, power_switch_smart_init, power_switch_smart_cleanup, PP_POWER_PORT_ID_SERIAL_2 },
#endif    
# ifdef PRODUCT_ERIC2
    {POWER_SWITCH_FJSMS_NAME,  power_switch_fjsms_init,  power_switch_fjsms_cleanup, SERIAL_PORT_1},
# endif /* PRODUCT_ERIC2 */
#endif
#if defined(PP_FEAT_POWER_IPM220)
# if defined(PP_FEAT_IPM220_ON_SERIAL2) && defined(PP_FEAT_SERIAL_PORT_2)
    { POWER_SWITCH_IPMTTY_NAME, power_switch_ipmtty_init, power_switch_ipmtty_cleanup, PP_POWER_PORT_ID_SERIAL_1 | PP_POWER_PORT_ID_SERIAL_2 },
# else
    { POWER_SWITCH_IPMTTY_NAME, power_switch_ipmtty_init, power_switch_ipmtty_cleanup, PP_POWER_PORT_ID_SERIAL_1 },
# endif
#endif
#if !defined(PRODUCT_XX01IP_ANY) && !defined(PRODUCT_RC1) && !defined(PRODUCT_ERICXP) && !defined(PRODUCT_ERICG4) && !defined(PRODUCT_LARAXP)
# if defined(PRODUCT_ERIC2) || defined(PRODUCT_ERICXP) || defined(PRODUCT_ERICG4) || defined(PRODUCT_LARA) || defined (PRODUCT_ASMIDC) && defined (OEM_LENOVO)
    { POWER_SWITCH_INTERN_NAME,  power_switch_intern_init,  power_switch_intern_cleanup, PP_POWER_PORT_ID_INTERNAL },
# else /* !PRODUCT_ERIC2 && !PRODUCT_ERICXP */
    { POWER_SWITCH_INTERN_NAME,  power_switch_intern_init,  power_switch_intern_cleanup, PP_POWER_PORT_ID_SERIAL_2 },
# endif /* !PRODUCT_ERIC2 && !PRODUCT_ERICXP */
#endif /* !(PRODUCT_XX01IP_ANY || PRODUCT_RC1 || PRODUCT_ERICXP || PRODUCT_LARAXP) */

#if defined(PP_FEAT_PSU_ADAPT)
    { POWER_SWITCH_PSU_ADAPT_NAME, power_switch_psu_adapt_init, power_switch_psu_adapt_cleanup, PP_POWER_PORT_ID_PSU_ADAPT },
#endif

#ifdef PP_FEAT_POWER_SWITCH_FJSMS
    { POWER_SWITCH_FJSMS_NAME, power_switch_fjsms_init, power_switch_fjsms_cleanup, PP_POWER_PORT_ID_SERIAL_1 },
#endif
#ifdef OEM_TANGTOP
    { POWER_SWITCH_IPWR_NAME, power_switch_ipwr_init, power_switch_ipwr_cleanup, PP_POWER_PORT_ID_SERIAL_1 | PP_POWER_PORT_ID_SERIAL_2 },
#endif /* OEM_TANGTOP */
    /* ---- add more supported power switches here ---- */
    { "", NULL, NULL, 0 }
};

static port_ctx_t port_ctxs[MAX_PORT_CONTEXTS];

static pthread_mutex_t power_switch_hash_lock = PTHREAD_MUTEX_INITIALIZER;
static pp_hash_t * power_switch_hash = NULL;

static const char * reconf_errmsg_tmpl = N_("Could not %s external power option for serial port %d."
#ifdef PP_FEAT_POWER_OTHER_EXTERNAL					    
					    "<br>Make sure your <a href=/power_switch.asp>power control settings</a> "
					    "are correct."
#endif
					    );

static power_switch_t * get_current_switch(u_int port_id);
static int port_id_to_idx(u_int port_id);

static int serial_externalpower_init(const char * device_path, u_int id, u_int fd, char ** errstr_p);
static int serial_externalpower_uninit(const char * device_path, u_int id, u_int fd, char ** errstr_p);
static int _serial_externalpower_init_uninit(u_int id, int do_init, char ** errstr_p);
#if (defined(PP_FEAT_PCI_ADC) && defined(PP_FEAT_PSU_ADAPT))
static void power_check_current_state(void);
#endif

#define POLL_PCI_VOLTAGE     1 // poll every x seconds

/* save info if power was on, to turn of standby voltage when power down*/
int power_was_on = 0;
time_t standby_off_time = 0;

/*
 * Initialize each of supported power switches
 */
int
pp_power_switch_init(void)
{
    power_switch_t * ps;
    power_switch_entry_t *entry;
    int ret = PP_ERR;
    int number_switches = sizeof(switches)/sizeof(power_switch_entry_t);

    if ((power_switch_hash = pp_hash_create(number_switches * 3)) == NULL) {
	goto error;
    }
   
    /* allocate and init entries */
    for (entry = switches; entry && entry->init; ++entry) {
	ps = malloc(sizeof(power_switch_t));
	memset(ps, 0, sizeof(power_switch_t));
	ps->allowed_port_ids = entry->allowed_port_ids;
	
	if (entry->init(ps)) goto error;

	if (pp_hash_set_entry(power_switch_hash, entry->name,
			      ps, entry->cleanup)) {
	    goto error;
	}
    }

    /* initialize all port contexts */
    {
	pthread_mutexattr_t mutexattr;
	int i;
	pthread_mutexattr_init(&mutexattr);
	if (pthread_mutexattr_settype(&mutexattr, PTHREAD_MUTEX_RECURSIVE_NP)) {
	    /* this shouldn`t fail */
	    assert(0);
	    exit(1);
	}
	memset(&port_ctxs, 0, sizeof(port_ctxs));
	for (i = 0; i < MAX_PORT_CONTEXTS; ++i) {
	    pthread_mutex_init(&port_ctxs[i].lock, &mutexattr);
	}
    }

    /* initialize port contexts */
#if defined(PP_FEAT_POWER_CTRL_ATX)
    pp_power_init(PP_POWER_PORT_ID_INTERNAL);
#endif

#if defined(PP_FEAT_PSU_ADAPT)
    pp_power_init(PP_POWER_PORT_ID_PSU_ADAPT);
#endif

    pp_power_init(PP_POWER_PORT_ID_SERIAL_1);

#ifdef PP_FEAT_SERIAL_PORT_2
    pp_power_init(PP_POWER_PORT_ID_SERIAL_2);
#endif

    pp_serial_register_port_callbacks(PP_SERIAL_USAGE_EXTERNALPOWER, serial_externalpower_init, serial_externalpower_uninit);

    ret = PP_SUC;

 error:    
    return ret;
}

#if (defined(PP_FEAT_PCI_ADC) && defined(PP_FEAT_PSU_ADAPT))
void
pp_power_check_current_state(void)
{
    /* check pci voltage to turn off standby voltage if power is off*/
    power_check_current_state();
}
#endif

/*
 * Cleanup and close power switches
 */
void
pp_power_switch_cleanup(void)
{
    pp_serial_unregister_port_callbacks(PP_SERIAL_USAGE_EXTERNALPOWER);

    /* uninitialize port contexts */
    pp_power_uninit(PP_POWER_PORT_ID_INTERNAL);
    pp_power_uninit(PP_POWER_PORT_ID_PSU_ADAPT);
    pp_power_uninit(PP_POWER_PORT_ID_SERIAL_1);
    pp_power_uninit(PP_POWER_PORT_ID_SERIAL_2);

    if (power_switch_hash) {
	pp_hash_delete(power_switch_hash);
	power_switch_hash = NULL;
    }
}

int
pp_power_switch_is_supported(const char * name, u_int port_id)
{
    power_switch_t * ps = NULL;

    MUTEX_LOCK(&power_switch_hash_lock);

    if (name) {
	ps = (power_switch_t *)pp_hash_get_entry(power_switch_hash, name);
	if (ps && !(port_id & ps->allowed_port_ids)) ps = NULL;
    }

    MUTEX_UNLOCK(&power_switch_hash_lock);

    return (ps ? 1 : 0);
}

int
pp_power_init(u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    int ret = PP_ERR;

    __PS_CTX_LOCK(port_ctx, PP_ERR);

    /* return error if no switch configured */
    if ((port_ctx->ps = get_current_switch(port_id)) == NULL) {
	goto bail;
    }

    /* set the port_id */
    port_ctx->ps_data.port_id = port_id;

    /* call the init routine of the switch */
    ret = port_ctx->ps->init ? port_ctx->ps->init(port_ctx->ps, &port_ctx->ps_data) : PP_SUC;

 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

int
pp_power_uninit(u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    int ret = PP_ERR;

    __PS_CTX_LOCK(port_ctx, PP_ERR);

    /* return error if no switch configured */
    if (port_ctx->ps == NULL) goto bail;

    /* call the logout routine of the switch before uninit */
    pp_power_logout(port_id);
    port_ctx->logged_in = 0;

    /* call the uninit routine of the switch */
    if (port_ctx->ps->uninit) port_ctx->ps->uninit(port_ctx->ps, &port_ctx->ps_data);

    port_ctx->ps = NULL;
    port_ctx->ps_data.port_id = 0;

    ret = PP_SUC;

 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

int
pp_power_login(u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    int ret = PP_ERR;

    __PS_CTX_LOCK(port_ctx, PP_ERR);
   
    /* return error if no switch configured */
    if (ps == NULL) {
	if (PP_FAILED(pp_power_init(port_id))) goto bail;
	ps = port_ctx->ps;
    }

    /* call the login routine of the switch if not already logged in */
    ret = (ps->login && !port_ctx->logged_in) ? ps->login(ps, &port_ctx->ps_data) : PP_SUC;
    if (PP_SUCCED(ret)) port_ctx->logged_in = 1;

 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

int
pp_power_logout(u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    int ret = PP_ERR;

    __PS_CTX_LOCK(port_ctx, PP_ERR);

    /* return error if no switch configured */
    if (ps == NULL) goto bail; /* no assert here */

    /* call the logout routine of the switch if we are logged in */
    ret = (ps->logout && port_ctx->logged_in) ? ps->logout(ps, &port_ctx->ps_data) : PP_SUC;
    if (PP_SUCCED(ret)) port_ctx->logged_in = 0;

 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

int
pp_power_lookup_devices(u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    int ret = PP_ERR;

    __PS_CTX_LOCK(port_ctx, PP_ERR);

    /* return error if no switch configured or if it is not "open" */
    if (ps == NULL || !port_ctx->logged_in) goto bail;

    if (ps->lookup_devices) {
	ret = ps->lookup_devices(ps, &port_ctx->ps_data);
    }
    
 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

/*
 * Get capabilities of this power switch
 * (i.e. switch (hopefully), get state, ...)
 */
int
pp_power_get_caps(u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    int ret = PP_ERR;

    __PS_CTX_LOCK(port_ctx, PP_ERR);

    /* return error if no switch configured */
    if (ps == NULL) goto bail;

    ret = ps->caps;

 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

int
pp_power_button(int port UNUSED, int device_id, u_char pb_enabled, int seq, u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    int ret = PP_ERR;

    __PS_CTX_LOCK(port_ctx, PP_ERR);

    /* return error if no switch configured or if it is not "open" */
    if (ps == NULL || !port_ctx->logged_in) goto bail;
    
    if (ps->power_button) {
	ret = ps->power_button(ps, &port_ctx->ps_data, port_id, device_id, pb_enabled, seq);
    }

 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

#if (defined(PP_FEAT_PCI_ADC) && defined(PP_FEAT_PSU_ADAPT))

static void
power_check_current_state(void)
{
    struct tm tm;
    time_t clock;
    float voltage;
    int current;
    u_int duration;
  
    if (standby_off_time > 0) {
	time(&clock);    
	localtime_r(&clock, &tm);

	if (clock > standby_off_time) {
	    standby_off_time = 0;
	    pp_power_login(PP_POWER_PORT_ID_PSU_ADAPT);
	    pp_power_stdby_switch(0, 0, 1, 0, PP_POWER_PORT_ID_PSU_ADAPT);
	    pp_power_logout(PP_POWER_PORT_ID_PSU_ADAPT);
	}
    } else {

	if (!PP_FAILED(pp_sense_read_adc_value_raw(&voltage, PCI_VOLTAGE_12V))) {

	    if (voltage > 10) {
#if defined POWER_DEBUG
		if (!power_was_on) { pp_log("libpp_powerswitch: power is ON\n"); }
#endif	    
		/* power is on*/
		current = 1;
		power_was_on = 1;
	    } else {
		/* power is off*/
#if defined POWER_DEBUG
		if (power_was_on) { pp_log("libpp_powerswitch: power is OFF\n"); }
#endif	    
		current = 0;
	    }

	    if (!current && power_was_on) {
#if defined POWER_DEBUG
		pp_log("libpp_powerswitch: power is OFF, but was ON\n");
#endif	 		
		power_was_on = 0;
		pp_cfg_get_int(&duration, "ps.power_stdby_offtime");
	    
		if (duration < MIN_POWER_STDBY_DURATION || duration > MAX_POWER_STDBY_DURATION)
		    { duration = DEF_POWER_STDBY_DURATION; }

		time(&clock);    
		localtime_r(&clock, &tm);
		standby_off_time = clock + (duration / 1000);

         	pp_power_login(PP_POWER_PORT_ID_PSU_ADAPT);
		pp_power_stdby_switch(0, 0, 0, 0, PP_POWER_PORT_ID_PSU_ADAPT);
      	        pp_power_logout(PP_POWER_PORT_ID_PSU_ADAPT);
	    }
	}
    }

    time(&clock);    
    localtime_r(&clock, &tm);
    clock = 3600 * tm.tm_hour + 60 * tm.tm_min + tm.tm_sec + POLL_PCI_VOLTAGE;
    eric_cron_insert_action(clock, power_check_current_state);
}
#endif /* PP_FEAT_PCI_ADC && PP_FEAT_PSU_ADAPT */

/*
 * Switch power for a specific port on or off.
 * @port: port id, normally 0 - 7
 * @device: used for cascaded devices
 * @on: off = 0, on = 1
 * @seq: if supported (POWER_CAP_SWITCH_SEQ) this switches all ports on or off
 * @port_id: PP_PORT_CTX_*
 */
int
pp_power_switch(int port, int device_id, u_char on, int seq, u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    int ret = PP_ERR;
    
    __PS_CTX_LOCK(port_ctx, PP_ERR);

    if (ps == NULL || !port_ctx->logged_in) goto bail;

#if defined(PP_FEAT_PSU_ADAPT)
    u_int duration = 1000;
    float voltage_a, voltage_b;

    /* No power button press when standby voltage is off */
    if (pp_sense_read_adc_value_raw(&voltage_b, PCI_VOLTAGE_3V_AUX)) 	goto bail;
    if (voltage_b < 2) 	goto bail;

    voltage_b=0;
    pp_sense_read_adc_value_raw(&voltage_b, PCI_VOLTAGE_12V);
    
    if (ps->power_switch) ret = ps->power_switch(ps, &port_ctx->ps_data, port, device_id, on, seq);

    /* We have to wait, because a new power button press (e.g turn the host on) */
    /* during standby voltage is off could cause problems!  */

    usleep(100000);

    pp_sense_read_adc_value_raw(&voltage_a, PCI_VOLTAGE_12V);    
    
#ifdef POWER_DEBUG
    pp_log("libpp_powerswitch: pp_power_switch before: %f V\n", voltage_b);
    pp_log("libpp_powerswitch: pp_power_switch after : %f V)\n", voltage_a);
#endif /* POWER_DEBUG */
    
    if ((voltage_b > 10) && (voltage_a < 10)) {

	pp_cfg_get_int(&duration, "ps.power_stdby_offtime");
    	if (duration < MIN_POWER_STDBY_DURATION || duration > MAX_POWER_STDBY_DURATION) duration = DEF_POWER_STDBY_DURATION;
	
	duration = (duration + 2 * POLL_PCI_VOLTAGE * 1000 + 500);
	goto sleep;
    }

#else /* !PP_FEAT_PSU_ADAPT */
    /* return error if no switch configured or if it is not "open" */
   
    if (ps->power_switch) ret = ps->power_switch(ps, &port_ctx->ps_data, port, device_id, on, seq);
#endif /* !PP_FEAT_PSU_ADAPT */
    
 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;

#if defined(PP_FEAT_PSU_ADAPT)
 sleep:
    /* unlock the mutex because other functions want to turn off standby voltage and isolate power button*/
    __PS_CTX_UNLOCK(port_ctx);    
    /* sleep here to aviod problems by clicks on other buttons in frontend, while standby power is turned off*/

    usleep(duration * 1000);
    return ret;
#endif /* PP_FEAT_PSU_ADAPT */
}

int
pp_power_switch_all(int device_id, u_char on, u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    int ret = PP_ERR;

    __PS_CTX_LOCK(port_ctx, PP_ERR);

    /* return error if no switch configured or if it is not "open" */
    if (ps == NULL || !port_ctx->logged_in) goto bail;
    
    if (ps->power_switch_all) {
	ret = ps->power_switch_all(ps, &port_ctx->ps_data, device_id, on);
    }

 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

int
pp_power_switch_simultan(int device_id, u_short port_values, u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    int ret = PP_ERR;

    __PS_CTX_LOCK(port_ctx, PP_ERR);

    /* return error if no switch configured or if it is not "open" */
    if (ps == NULL || !port_ctx->logged_in) goto bail;
    
    if (ps->power_switch_simultan) {
	ret = ps->power_switch_simultan(ps, &port_ctx->ps_data, device_id, port_values);
    }

 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

int
pp_power_stdby_switch(int port, int device_id, u_char on, int seq, u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    int ret = PP_ERR;
    int pb_enabled;

    if (!on) {
	usleep(1000);
        pp_power_button(port, device_id, 0 , seq, port_id);
    }

    __PS_CTX_LOCK(port_ctx, PP_ERR);
        /* return error if no switch configured or if it is not "open" */
    if (ps == NULL || !port_ctx->logged_in) {
	goto bail;
    }

    if (ps->stdby_switch) ret = ps->stdby_switch(ps, &port_ctx->ps_data, port, device_id, on, seq);

    __PS_CTX_UNLOCK(port_ctx);

    if (on) {
	pp_cfg_is_enabled(&pb_enabled, "ps.power_button_enabled");
	usleep(1000);
        pp_power_button(port, device_id, pb_enabled , seq, port_id);
    }
    
    return ret;
    
 bail:

    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

int
pp_power_cycle(int port, int device_id, u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    int ret = PP_ERR;
    u_int duration;
     
    __PS_CTX_LOCK(port_ctx, PP_ERR);

    /* return error if no switch configured or if it is not "open" */
    if (ps == NULL || !port_ctx->logged_in) goto bail;
    
    pp_cfg_get_int(&duration, "ps.power_cycle_offtime");

    if (duration < MIN_POWER_CYCLE_DURATION || duration > MAX_POWER_CYCLE_DURATION) {
	duration = DEF_POWER_CYCLE_DURATION;
    }

    if (ps->power_switch) {
	ret = ps->power_switch(ps, &port_ctx->ps_data, port, device_id, 0, 0);
	if (!ret) {
	    usleep(duration * 1000);
	    ret = ps->power_switch(ps, &port_ctx->ps_data, port, device_id, 1, 0);
	}
    }
	
 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

int
pp_power_reset(int port, int device_id, u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    int ret = PP_ERR;

    __PS_CTX_LOCK(port_ctx, PP_ERR);

    /* return error if no switch configured or if it is not "open" */
    if (ps == NULL || !port_ctx->logged_in) goto bail;
    
    if (ps->do_reset) {
	ret = ps->do_reset(ps, &port_ctx->ps_data, port, device_id);
    }

 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

int
pp_power_switch_kvm_ports(u_char kvm_unit, u_short kvm_port, int on)
{
    /*
     * FIXME: This routine must be synchronized to kvm_powerport_* changes!
     */
    static pthread_mutex_t __lock = PTHREAD_MUTEX_INITIALIZER;
    int ret = PP_SUC, serial_port, port_id, port, dev;
    u_int i, port_cnt;
    char *val, *cur_sw_id_str;
    char serial_port_str[MAX_OPT_KEY_LEN + 1];
    char switch_id_str[MAX_OPT_KEY_LEN + 1];
    char dev_str[MAX_OPT_KEY_LEN + 1];
    char port_str[MAX_OPT_KEY_LEN + 1];

    MUTEX_LOCK(&__lock);

    /* get kvm power entry count */
    pp_cfg_get_uint(&port_cnt, "unit[%u].port[%u].powerports._s_",
                    kvm_unit, kvm_port);
    for (i = 0; i < port_cnt; i++) {
        /* FIXME: */
	if (pp_cfg_get(&val, "unit[%u].port[%u].powerports[%u]", 
                       kvm_unit, kvm_port, i) == PP_SUC
	    && strlen(val) <= MAX_OPT_VALUE_LEN
	    && sscanf(val, "%[^:]:%[^:]:%[^:]:%s", serial_port_str, dev_str, port_str, switch_id_str) == 4) {

	    serial_port = pp_strtol_10(serial_port_str, -1, NULL) + 1;
	    dev = pp_strtol_10(dev_str, -1, NULL);
	    port = pp_strtol_10(port_str, -1, NULL);
	    port_id = pp_power_map_serial_port_to_id(serial_port);
	    if (serial_port <= 0 || dev < 0 || port < 0 || port_id <= 0) {
		continue; /* FIXME: error handling */
	    }

	    /* pp_power_open_login manage to open the serial port only once */
	    // pp_power_open_login(serial_n);
		
	    cur_sw_id_str = pp_power_get_name(0, 0, PWR_OBJ_SWITCH_ID, port_id);
	    if (cur_sw_id_str && !strcmp(cur_sw_id_str, switch_id_str)) {
		if (pp_power_switch(port, dev, on, 0, port_id)) ret = PP_ERR;
	    }
	    free(cur_sw_id_str);
	}
	free(val);
    }

    MUTEX_UNLOCK(&__lock);
    return ret;
}

/*
 * Get the current state for a specific power port
 */
int
pp_power_get_state(int port, int device_id, u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    int ret = -1;

    __PS_CTX_LOCK(port_ctx, -1);
    /* return error if no switch configured or if it is not "open" */
    if (ps == NULL || !port_ctx->logged_in) goto bail;

    /* call the get_state routine of the switch */
    if (ps->get_state) {
	ret = ps->get_state(ps, &port_ctx->ps_data, port, device_id);
    }

 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

/*
 * Returns if Tangtop powerswitch ports are switch with our without delay time
 */
int
pp_power_get_delay_enable(int device_id, u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    int ret = -1;

    __PS_CTX_LOCK(port_ctx, -1);
    /* return error if no switch configured or if it is not "open" */
    if (ps == NULL || !port_ctx->logged_in) {
	goto bail;
    }

    /* call the get_state routine of the switch */
    if (ps->get_delay_enable) {
	ret = ps->get_delay_enable(ps, &port_ctx->ps_data, device_id);
    }

 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

/*
 * Enables or disables Tangtop powerswitch delay times
 */
int
pp_power_set_delay_enable(int device_id, u_char enable, u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    int ret = -1;

    __PS_CTX_LOCK(port_ctx, -1);
    /* return error if no switch configured or if it is not "open" */
    if (ps == NULL || !port_ctx->logged_in) {
	goto bail;
    }

    /* call the get_state routine of the switch */
    if (ps->set_delay_enable) {
	ret = ps->set_delay_enable(ps, &port_ctx->ps_data, device_id, enable);
    }

 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

/*
 * Get the number of ports, devices or external relays depandend
 */
int
pp_power_get_count(int device_id, int object, u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    int ret = 0;

    __PS_CTX_LOCK(port_ctx, 0);

    /* return error if no switch configured or if it is not "open" */
    if (ps == NULL || !port_ctx->logged_in) goto bail;

    /* call the get_count routine of the switch */
    ret = ps->get_count ? ps->get_count(ps, &port_ctx->ps_data, device_id, object) : 0;

 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

/*
 * Returns software version of power switch firmware
 */
int
pp_power_get_sw_version(int device_id, u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    int ret = 0;

    __PS_CTX_LOCK(port_ctx, 0);

    /* return error if no switch configured or if it is not "open" */
    if (ps == NULL || !port_ctx->logged_in) {
	goto bail;
    }

    /* call the get_name routine of the switch */
    if (ps->get_sw_version) {
	ret = ps->get_sw_version(ps, &port_ctx->ps_data, device_id);
    }

 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

/*
 * Get the name of a specific port
 */
char*
pp_power_get_current(int device_id, u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    char * ret = NULL;

    __PS_CTX_LOCK(port_ctx, NULL);

    /* return error if no switch configured or if it is not "open" */
    if (ps == NULL || !port_ctx->logged_in) {
	goto bail;
    }

    /* call the get_name routine of the switch */
    if (ps->get_current) {
	ret = ps->get_current(ps, &port_ctx->ps_data, device_id);
    }

 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

/*
 * Get the name of a specific port
 */
char*
pp_power_get_name(int port, int device_id, int object, u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    char * ret = NULL;

    __PS_CTX_LOCK(port_ctx, NULL);

    /* return error if no switch configured or if it is not "open" */
    if (ps == NULL || !port_ctx->logged_in) goto bail;

    /* call the get_name routine of the switch */
    if (ps->get_name) {
	ret = ps->get_name(ps, &port_ctx->ps_data, port, device_id, object);
    }

 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

/*
 * Get the name of a specific port
 */
int
pp_power_set_name(int port, int device_id, int object, u_int port_id, const char* name)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    int ret = 0;

    __PS_CTX_LOCK(port_ctx, 0);

    /* return error if no switch configured or if it is not "open" */
    if (ps == NULL || !port_ctx->logged_in) goto bail;

    /* call the get_name routine of the switch */
    if (ps->set_name) {
	ret = ps->set_name(ps, &port_ctx->ps_data, port, device_id, object, name);
    }

 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

/*
 * Get the delay of a specific port
 */
u_short*
pp_power_get_port_delays(int port, int device_id, u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    u_short * ret = NULL;

    __PS_CTX_LOCK(port_ctx, NULL);

    /* return error if no switch configured or if it is not "open" */
    if (ps == NULL || !port_ctx->logged_in) goto bail;

    /* call the get_name routine of the switch */
    if (ps->get_port_delays) {
	ret = ps->get_port_delays(ps, &port_ctx->ps_data, port, device_id);
    }

 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

/*
 * Get the delay of a specific port
 */
int
pp_power_set_port_delays(int port, int device_id, u_int port_id, u_short* delay)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    int ret = 0;

    __PS_CTX_LOCK(port_ctx, 0);

    /* return error if no switch configured or if it is not "open" */
    if (ps == NULL || !port_ctx->logged_in) goto bail;

    /* call the get_name routine of the switch */
    if (ps->set_port_delays) {
	ret = ps->set_port_delays(ps, &port_ctx->ps_data, port, device_id, delay);
    }

 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

char*
pp_power_get_data(int device_id, int channel, int type, u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    char * ret = NULL;

    __PS_CTX_LOCK(port_ctx, NULL);

    /* return error if no switch configured or if it is not "open" */
    if (ps == NULL || !port_ctx->logged_in) goto bail;

    /* call the get_data routine of the switch */
    if (ps->get_data) {
	ret = ps->get_data(ps, &port_ctx->ps_data, device_id, channel, type);
    }

 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

/*
 * Get temperature from the current powerswitch and the selected device.
 * If cascading is not supported device should be 0.
 */
char*
pp_power_get_temperature(int device_id, u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    char * ret = NULL;

    __PS_CTX_LOCK(port_ctx, NULL);

    /* return error if no switch configured or if it is not "open" */
    if (ps == NULL || !port_ctx->logged_in) goto bail;

    /* call the get_temperature routine of the switch */
    if (ps->get_temperature) {
	ret = ps->get_temperature(ps, &port_ctx->ps_data, device_id);
    }

 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

/*
 * Switch external relay.
 */
int
pp_power_ext_switch(int port, int device_id, u_char on, u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    int ret = PP_ERR;

    __PS_CTX_LOCK(port_ctx, PP_SUC);

    /* return error if no switch configured or if it is not "open" */
    if (ps == NULL || !port_ctx->logged_in) goto bail;

    /* call the ext_switch routine of the switch */
    if (ps->ext_switch) {
	ret = ps->ext_switch(ps, &port_ctx->ps_data, port, device_id, on);
    }

 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

/*
 * Get state of an external relay.
 */
int
pp_power_get_ext_state(int port, int device_id, u_int port_id)
{
    port_ctx_t * port_ctx = &port_ctxs[port_id_to_idx(port_id)];
    power_switch_t * ps = port_ctx->ps;
    int ret = -1;

    __PS_CTX_LOCK(port_ctx, 0);

    /* return error if no switch configured or if it is not "open" */
    if (ps == NULL || !port_ctx->logged_in) goto bail;

    /* call the ext_state routine of the switch */
    if (ps->get_ext_state) {
	ret = ps->get_ext_state(ps, &port_ctx->ps_data, port, device_id);
    }

 bail:
    __PS_CTX_UNLOCK(port_ctx);
    return ret;
}

char *
pp_power_get_short_name(const char *id)
{
    power_switch_t * ps;

    MUTEX_LOCK(&power_switch_hash_lock);
    ps = (power_switch_t *) pp_hash_get_entry(power_switch_hash, id);
    MUTEX_UNLOCK(&power_switch_hash_lock);

    return (ps && ps->get_name) ? ps->get_name(ps, NULL, 0 ,0, PWR_OBJ_SWITCH_SHORT_NAME) : NULL;
}

char *
pp_power_get_long_name(const char *id)
{
    power_switch_t * ps;

    MUTEX_LOCK(&power_switch_hash_lock);
    ps = (power_switch_t *) pp_hash_get_entry(power_switch_hash, id);
    MUTEX_UNLOCK(&power_switch_hash_lock);

    return (ps && ps->get_name) ? ps->get_name(ps, NULL, 0 ,0, PWR_OBJ_SWITCH_LONG_NAME) : NULL;
}

u_int
pp_power_map_serial_port_to_id(u_int serial_port)
{
    if (serial_port == 1) {
	return PP_POWER_PORT_ID_SERIAL_1;
    } else if (serial_port == 2) {
	return PP_POWER_PORT_ID_SERIAL_2;
    } else {
	return 0;
    }
}

static power_switch_t *
get_current_switch(u_int port_id)
{
    power_switch_t * ps = NULL;
    char * opt = NULL;
    char * ext_str = NULL;

    MUTEX_LOCK(&power_switch_hash_lock);

    switch (port_id) {
      case PP_POWER_PORT_ID_PSU_ADAPT:
	  ext_str = strdup("psu_adapt");
	  break;
      case PP_POWER_PORT_ID_INTERNAL:
	  ext_str = strdup("intern");
	  break;
      case PP_POWER_PORT_ID_SERIAL_1:
	  pp_cfg_get(&opt, "serialport[0]._c_");
	  if (strcmp(opt, "externalpower")) goto bail;
	  pp_cfg_get_nodflt(&ext_str, "ps.serial[0]._c_");
	  break;
      case PP_POWER_PORT_ID_SERIAL_2:
	  pp_cfg_get(&opt, "serialport[1]._c_");
	  if (strcmp(opt, "externalpower")) goto bail;
	  pp_cfg_get_nodflt(&ext_str, "ps.serial[1]._c_");
	  break;
      default:
	  break;
    }

    if (ext_str) {
        ps = (power_switch_t*) pp_hash_get_entry(power_switch_hash, ext_str);
	if (ps && !(port_id & ps->allowed_port_ids)) ps = NULL;
    }

 bail:
    MUTEX_UNLOCK(&power_switch_hash_lock);
    free(opt); 
    free(ext_str);
    return ps;
}

static int
port_id_to_idx(u_int port_id)
{
    int idx = ffs(port_id) - 1;
    assert(idx >= 0);
    return idx;
}

static int
serial_externalpower_init(const char * device_path UNUSED, u_int id, u_int fd UNUSED, char ** errstr_p)
{
    return _serial_externalpower_init_uninit(id, 1, errstr_p);
}

static int
serial_externalpower_uninit(const char * device_path UNUSED, u_int id, u_int fd UNUSED, char ** errstr_p)
{
    return _serial_externalpower_init_uninit(id, 0, errstr_p);
}

static int
_serial_externalpower_init_uninit(u_int id, int do_init, char ** errstr_p)
{
    u_int port_id;
    int ret;

    pp_log("%s(): id=%u, do_init=%d\n", ___F, id, do_init);

    if (id == 0) {
	port_id = PP_POWER_PORT_ID_SERIAL_1;
    } else if (id == 1) {
	port_id = PP_POWER_PORT_ID_SERIAL_2;
    } else {
	assert(0);
	if (errstr_p) *errstr_p = strdup(_("Internal error."));
	return PP_ERR;
    }

    ret = do_init ? pp_power_init(port_id) : pp_power_uninit(port_id);
    if (ret == PP_ERR && errstr_p) {
	asprintf(errstr_p, _(reconf_errmsg_tmpl), do_init ? _("activate") : _("deactivate"), id + 1);
    }

    return ret;
}
