#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>

#include <pp/base.h>
#include <pp/cfg.h>
#include <liberic_pthread.h>
#include <liberic_misc.h>
#include <pp/powerswitch.h>
#include <gpio_hilevel_functions.h>

#include "debug.h"
#include "power_switch.h"

static int init(power_switch_t *this, ps_data_t * ps_data);
static void uninit(power_switch_t *this, ps_data_t * ps_data);
static int get_count(power_switch_t *this, ps_data_t * ps_data, int device_id, int object);
static char* get_name(power_switch_t *this, ps_data_t * ps_data, int idx, int device_id, int object);
static int power_switch(power_switch_t *this, ps_data_t * ps_data, int port, int device, u_char on, int seq);
static int get_state(power_switch_t *this, ps_data_t * ps_data, int port, int device);
static int set_dtr(ps_data_t * ps_data, int dtr);

/* -------------------- exported functions ---------------------------- */

int
power_switch_ipmgpio_init(power_switch_t *this)
{
    this->init          = init;
    this->uninit        = uninit;
    this->power_switch  = power_switch;
    this->get_state     = get_state;
    this->get_count     = get_count;
    this->get_name      = get_name;

    this->caps = POWER_CAP_SWITCH | POWER_CAP_GET_STATE;
    return 0;
}

void
power_switch_ipmgpio_cleanup(void* this)
{
    power_switch_t *entry = (power_switch_t*) this;

    free(entry);
}

static int
init(power_switch_t *this UNUSED, ps_data_t * ps_data)
{
#ifdef PP_BOARD_KIRA
    if (ps_data->port_id == PP_POWER_PORT_ID_SERIAL_1) {
	ps_data->ipmgpio.serial_devname = "/dev/ttyS0";
    } else if (ps_data->port_id == PP_POWER_PORT_ID_SERIAL_2) {
	ps_data->ipmgpio.serial_devname = "/dev/ttyS1";
    } else {
	pp_log_err("Unsupported serial port %d\n", ps_data->port_id);
	return PP_ERR;
    }

    if ((ps_data->ipmgpio.fd = ps_get_fd_by_port_id(ps_data->port_id)) < 0) {
	pp_log_err("Could not get fd for port %d\n", ps_data->port_id);
        return PP_ERR;
    }
#else
    /* configure UART0_DTR(gpio 27) pin as a gpio */
    pp_gpio_set_alternate(PP_GPIO_DEV_IBM, 0, 1 << PP_GPIO_IPM220_PWR);
#endif

    ps_data->ipmgpio.power = 0;
#if !defined(PRODUCT_ERIC2) && !defined(PRODUCT_ERICXP) && !defined(PRODUCT_ERICG4)
    if (ps_data->port_id == PP_POWER_PORT_ID_SERIAL_1) {
        ps_data->ipmgpio.state_var_name = "ps.serial[0].ipmgpio.state";
    } else if (ps_data->port_id == PP_POWER_PORT_ID_SERIAL_2) {
        ps_data->ipmgpio.state_var_name = "ps.serial[1].ipmgpio.state";
    } else {
        ps_data->ipmgpio.state_var_name = NULL;
    }
    if (ps_data->ipmgpio.state_var_name) {
	int state = 0;
	pp_cfg_is_enabled_nodflt(&state, ps_data->ipmgpio.state_var_name);
	ps_data->ipmgpio.power = state ? 1 : 0;
	power_switch(this, ps_data, 0, 0, state ? 1 : 0, 0);
    }
#endif /* !PRODUCT_ERIC2 && !PRODUCT_ERICXP */
    return PP_SUC;
}

static void
uninit(power_switch_t *this UNUSED, ps_data_t * ps_data)
{
#ifdef PP_BOARD_KIRA
    if (ps_data->ipmgpio.fd >= 0) {
	ps_data->ipmgpio.fd = -1;
    }
#else
    (void)ps_data; /* avoid warning */
    /* configure UART0_DTR(gpio 27) pin as DTR (used for modem) */
    pp_gpio_set_alternate(PP_GPIO_DEV_IBM, 1 << PP_GPIO_IPM220_PWR, 0);
#endif
}

static int
get_count(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	  int device_id UNUSED, int object)
{
    switch (object) {
      case PWR_OBJ_PORT: return 1;
      case PWR_OBJ_DEVICE: return 1;
      default: return 0;  
    }
}

static char *
get_name(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	 int idx UNUSED, int device_id UNUSED, int object)
{
    switch (object) {
      case PWR_OBJ_PORT:
      case PWR_OBJ_SWITCH_SHORT_NAME:
      case PWR_OBJ_SWITCH_LONG_NAME:
	  return ps_get_pswitch_name(POWER_SWITCH_IPMGPIO_NAME, object);
      case PWR_OBJ_SWITCH_ID:
	  return strdup(POWER_SWITCH_IPMGPIO_NAME);
      default:
	  return NULL;
    }
}

static int
power_switch(power_switch_t *this UNUSED, ps_data_t * ps_data,
	     int port UNUSED, int device UNUSED, unsigned char on, int seq UNUSED)
{
    if (!ps_data->ipmgpio.power && !on) {
        /* turn off power */
	set_dtr(ps_data, 0);
#if !defined(PRODUCT_ERIC2) && !defined(PRODUCT_ERICXP) && !defined(PRODUCT_ERICG4)
        if (ps_data->ipmgpio.state_var_name) {
            pp_cfg_set_enabled(0, ps_data->ipmgpio.state_var_name);
            pp_cfg_save(DO_FLUSH);
        }
#endif
        ps_data->ipmgpio.power = 1;
        pp_propchange_enqueue(PP_PROP_PWR_OFF, 0);
    } else if (ps_data->ipmgpio.power && on) {
        /* turn on power */
	set_dtr(ps_data, 1);
#if !defined(PRODUCT_ERIC2) && !defined(PRODUCT_ERICXP) && !defined(PRODUCT_ERICG4)
        if (ps_data->ipmgpio.state_var_name) {
            pp_cfg_set_enabled(1, ps_data->ipmgpio.state_var_name);
            pp_cfg_save(DO_FLUSH);
        }
#endif
        ps_data->ipmgpio.power = 0;
        pp_propchange_enqueue(PP_PROP_PWR_ON, 0);
    }

    return PP_SUC;
}

static int
get_state(power_switch_t *this UNUSED, ps_data_t * ps_data,
	  int port UNUSED, int device UNUSED)
{
    return !ps_data->ipmgpio.power;
}

static int
set_dtr(ps_data_t * ps_data, int dtr)
{
    int ret = PP_ERR;
    
#ifdef PP_BOARD_KIRA
    int mcr = 0;
    
    if (ps_data->ipmgpio.fd < 0 ||
	ioctl(ps_data->ipmgpio.fd, TIOCMGET, &mcr) != 0) {

	pp_log_err("Error setting DTR=%d (TIOCMGET)\n", dtr);
	goto bail;
    }

    if (dtr) {
	mcr &= ~TIOCM_DTR;
    } else {
	mcr |= TIOCM_DTR;
    }
    
    if (ioctl(ps_data->ipmgpio.fd, TIOCMSET, &mcr) != 0) {
	pp_log_err("Error setting DTR=%d (TIOCMSET)\n", dtr);
	goto bail;
    }

 bail:    
#else
    (void)ps_data; /* avoid warning */
    pp_gpio_bit_set(PP_GPIO_DEV_IBM, PP_GPIO_IPM220_PWR, dtr);
#endif

    ret = PP_SUC;
    
    return ret;
}
