#include <stdlib.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <liberic_misc.h>
#include <liberic_config.h>
#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/powerswitch.h>
#include <pp/termios.h>

#include "debug.h"
#include "power_switch.h"

static int init(power_switch_t *this, ps_data_t * ps_data);
static int login(power_switch_t *this, ps_data_t * ps_data);
static int logout(power_switch_t *this, ps_data_t * ps_data);
static int get_count(power_switch_t *this, ps_data_t * ps_data, int device_id, int object);
static char* get_name(power_switch_t *this, ps_data_t * ps_data, int idx, int device_id, int object);
static int power_switch(power_switch_t *this, ps_data_t * ps_data, int port, int device, u_char on, int seq);

static int spc_get_login(ps_spc_t * ps_spc);
static int spc_get_prompt(ps_spc_t * ps_spc);

/* -------------------- exported functions ---------------------------- */

int
power_switch_spc_init(power_switch_t *this)
{
    this->init		= init;
    this->login		= login;
    this->logout	= logout;
    this->get_count	= get_count;
    this->get_name	= get_name;
    this->power_switch	= power_switch;

    this->caps = POWER_CAP_SWITCH;
    return PP_SUC;
}

void
power_switch_spc_cleanup(void* this)
{
    power_switch_t *entry = (power_switch_t*) this;

    free(entry);
}

static int
init(power_switch_t *this UNUSED, ps_data_t * ps_data)
{
    if (ps_data->port_id == PP_POWER_PORT_ID_SERIAL_1) {
	ps_data->spc.user_var_name = "ps.serial[0].spc.user";
	ps_data->spc.pass_var_name = "ps.serial[0].spc.pass";
    } else if (ps_data->port_id == PP_POWER_PORT_ID_SERIAL_2) {
	ps_data->spc.user_var_name = "ps.serial[1].spc.user";
	ps_data->spc.pass_var_name = "ps.serial[1].spc.pass";
    } else {
	ps_data->spc.user_var_name = NULL;
	ps_data->spc.pass_var_name = NULL;
    }

    if (PP_FAILED(pp_base_set_tty_params(ps_data->spc.fd, 9600, "", 8, 0, 0, 0))) {
	return PP_ERR;
    }

    return PP_SUC;
}

static int
login(power_switch_t *this UNUSED, ps_data_t * ps_data)
{
    ps_spc_t * ps_spc = &ps_data->spc;
    char rsp_buf[128];
    char * user = NULL;
    char * pass = NULL;
    ssize_t cmd_len;
    int ret = PP_ERR;

    /* first try to get prompt without login */
    if (PP_FAILED(spc_get_prompt(&ps_data->spc))) {
	/* try to login two times*/
	if (PP_FAILED(spc_get_login(&ps_data->spc))) {
	    if (PP_FAILED(spc_get_login(&ps_data->spc))) {
		D(D_ERROR, "SPC: %s(): login failed\n", ___F);
		goto error;
	    }
	}

	if (!ps_spc->user_var_name
	    || PP_FAILED(pp_cfg_get_nodflt(&user, ps_spc->user_var_name))) {
	    goto error;
	}

	if (!ps_spc->pass_var_name
	    || PP_FAILED(pp_cfg_get_nodflt(&pass, ps_spc->pass_var_name))) {
	    goto error;
	}

	cmd_len = strlen(user);
	if (write(ps_spc->fd, user, cmd_len) != cmd_len
	    || write(ps_spc->fd, "\r", 1) != 1) {
	    D(D_ERROR, "SPC: %s(): sending username failed\n", ___F);
	    goto error;
	}
	if (PP_FAILED(ps_read_response(ps_spc->fd, 0, "Password:", 2, 0,
				       rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING))) {
	    D(D_ERROR, "SPC: %s(): waiting for password prompt failed\n", ___F);
	    goto error;
	}

	cmd_len = strlen(pass);
	if (write(ps_spc->fd, pass, cmd_len) != cmd_len
	    || write(ps_spc->fd, "\r", 1) != 1) {
	    D(D_ERROR, "SPC: %s(): sending password failed\n", ___F);
	    goto error;
	}
    }

    ret = PP_SUC;

 error:
    free(user);
    free(pass);
    return ret;
}

static int
logout(power_switch_t *this UNUSED, ps_data_t * ps_data)
{
    ps_spc_t * ps_spc = &ps_data->spc;
    const char * cmd_buf = "quit\r";
    ssize_t cmd_len = strlen(cmd_buf);
    int ret = PP_ERR;

    if (PP_FAILED(spc_get_prompt(&ps_data->spc))) {
	D(D_ERROR, "SPC: %s(): getting prompt failed\n", ___F);
	goto error;
    } else if (write(ps_spc->fd, cmd_buf, cmd_len) != cmd_len) {
	D(D_ERROR, "SPC: %s(): sending 'quit' command failed\n", ___F);
	goto error;
    }

    ret = PP_SUC;

 error:
    return ret;
}

static int
get_count(power_switch_t *this UNUSED,  ps_data_t * ps_data UNUSED,
	  int device_id UNUSED, int object)
{
    switch (object) {
      case PWR_OBJ_PORT: return 16;
      case PWR_OBJ_DEVICE: return 1;	  
      default: return 0;  
    }
}

static char *
get_name(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	 int idx, int device_id UNUSED, int object)
{
    const char * blocks = "abcd";
    char * val = NULL;

    switch (object) {
      case PWR_OBJ_PORT:
	  asprintf(&val, "%c%d", blocks[idx / 4], (idx % 4) + 1);
	  return val;
      case PWR_OBJ_SWITCH_SHORT_NAME:
      case PWR_OBJ_SWITCH_LONG_NAME:
	  return ps_get_pswitch_name(POWER_SWITCH_SPC_NAME, object);
      case PWR_OBJ_SWITCH_ID:
	  return strdup(POWER_SWITCH_SPC_NAME);
      default:
	  return NULL;
    }
}

static int
power_switch(power_switch_t *this UNUSED, ps_data_t * ps_data,
	     int port, int device UNUSED, u_char on, int seq UNUSED)
{
    char cmd_buf[16];
    char expect_buf[32];
    char rsp_buf[128];
    const char * action;
    const char * blocks = "abcd";
    char block;
    ssize_t cmd_len;
    int nr;
    int ret = PP_ERR;
    
    if (spc_get_prompt(&ps_data->spc)) goto error;

    action = on ? "on" : "off";

    block = blocks[port / 4];
    nr = (port % 4) + 1;
    
    snprintf(cmd_buf, sizeof(cmd_buf), "%s port .%1c%1d\r", action, block, nr);
    cmd_len = strlen(cmd_buf);
    if (write(ps_data->spc.fd, cmd_buf, cmd_len) != cmd_len) {
	D(D_ERROR, "SPC: %s(): sending command failed\n", ___F);
	goto error;
    }

    snprintf(expect_buf, sizeof(expect_buf), "1 port(s) turned %s\r", action);
    if (PP_SUCCED(ret = ps_read_response(ps_data->spc.fd, 0, expect_buf, 1, 0,
					 rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING))) {
	pp_propchange_enqueue(on ? PP_PROP_PWR_ON : PP_PROP_PWR_OFF, 0);
    }

 error:
    return ret;
}

/* -------------------- internal functions ---------------------------- */

static int
spc_get_prompt(ps_spc_t * ps_spc)
{
    char rsp_buf[128];
    
    if (write(ps_spc->fd, "\r", 1) != 1) return PP_ERR;
    
    return ps_read_response(ps_spc->fd, 0, "SPC:", 0, 500000,
			    rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING);
}

static int
spc_get_login(ps_spc_t * ps_spc)
{
    char rsp_buf[128];
    
    if (write(ps_spc->fd, "\e\r\r", 3) != 3) return PP_ERR;
    
    return ps_read_response(ps_spc->fd, 0, "Username:", 4, 0,
			    rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING);
}
