/**
 * PP_RDM.cpp
 * 
 * The main file for accessing RDM functionality in PP firmware. 
 * Initializes the RDM database, keeps the RDM ports up to date
 * (powerports may be added dynamically), holds the local DDA
 * objects for the RDM database and opens the CSC connection.
 * 
 * (c) 2006 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#include "assert.h"

#include "pp/RDM.h"
#include "pp/RDM_Database.h"
#include "pp/RDM_Definitions.h"
#include "pp/RDM_DDA_Device.h"
#include "pp/RDM_DDA_Port.h"
#include "pp/RDM_DDA_Path.h"

#include "pp/CSC.h"
#include "pp/PP_RDM.h"
#include "pp/CC.h"

#include "RDM_Protocol.h"
#include "RDM_Event_Protocol.h"
#include "PP_RDM_Power.h"
#include "PP_RDM_Session.h"
#include "PP_RDM_System.h"
#include "PP_DDA_objects.h"

#include "powerport_libpp_powerswitch.h"

#include "Device.h"
#include "DeviceSettings.h"
#include "Port.h"


/* existing power devices that can be mapped */
#define POWER_DEVICE_INTERNAL   0
#define POWER_DEVICE_SERIAL1    1
#define POWER_DEVICE_SERIAL2    2
#define POWER_DEVICE_IPMI       3
#define POWER_DEVICE_MAX        4


/* static device id string */
char                     device_id[RDM_MAX_ID];

/* rdm variables */
CRDM			 *rdm;                    // needed by RDM_Protocol.cpp, maybe change
CSessionManager          *gSessionManager;	  // needed by RDM_session
CCommandCenter		 *commandCenter;
/* rdm services */
static CRDM_Database*    dbService = NULL;
static CRDM_Power*       powerService = NULL;
static CRDM_System*      systemService = NULL;
static CRDM_Session*     sessionService = NULL;
static CRDM_Notify*      notifyService = NULL;

/* DDA + related variables */
//static PP_DDA_Device*       DDA_device;                /* DDA device object */ 
//static PP_DDA_KVMport*      kvm_port;              /* our kvm port */
static PP_DDA_PowerPort*    device_port_list[4];   /* a linked list of powerports for each existing device */

/* CSC + related variables */
static CCSC                 *csc;
CRDM_Protocol_Factory       rdmProtocol;
CRDM_Event_Protocol_Factory rdmEventProtocol;





static Device*           device;
static DeviceSettings*   deviceSettings;
static Port*             kvmPort;



/* internal prototypes */
static void createRDMRoot();
static void initializePowerports();

/**
 * Open and initialize the rdm database.
 */
int pp_rdm_initialize_database() {
    int result;

    rdm = new CRDM();
    gSessionManager = new CSessionManager(rdm);
    csc = new CCSC(gSessionManager);
    
    /* initialize RDM */
    rdm->db = new CRDM_Secure_SXDB();
    commandCenter = new CCommandCenter(rdm);
    
    /* initialize the DB */
    strcpy(device_id, "IP-Reach");
    createRDMRoot();

    device         = new Device(rdm, device_id);
    deviceSettings = new DeviceSettings(rdm, device);
    kvmPort        = new Port(rdm, device);

    //DDA_device = new PP_DDA_Device(&rdm, device_id, "KVMSwitch", "EricG4", 1);
    initializePowerports();

    // FIXME: add serial ports ?

    /* create services */
    dbService = new CRDM_Database(rdm);
    powerService = new PP_RDM_Power(rdm);
    systemService = new PP_RDM_System(rdm, commandCenter);
    sessionService = new PP_RDM_Session(rdm);
    notifyService = new CRDM_Notify();
    notifyService->SetRDM(rdm);

    device->initialize(dbService);
    deviceSettings->initialize(dbService);
    kvmPort->initialize(dbService);
    

    result = csc->Initialize();
    assert( result >= 0);
    // FIXME: type, model, product, version, name, CCclusterName
    csc->SetDeviceDescription( "IP-Reach", "KX_KIM", "KX101", "1", "MyTestDevice", NULL );
    //csc.SetDeviceDescription( "Eric", "Eric G4", "Eric G4", "1", "MyTestDevice", NULL );
    csc->AddProtocol( &rdmProtocol );
    csc->AddProtocol( &rdmEventProtocol );

    rdm->nm.Notify("System", RDM_EC_System_Startup, NULL);
    rdm->nm.Notify("System", RDM_EC_System_Ready, NULL);

    return PP_SUC;
}

/**
 * Cleanup and the rdm database.
 */
void pp_rdm_cleanup_database() {
    rdm->nm.Notify("System", RDM_EC_System_Shutdown, NULL);

    delete sessionService;
    delete powerService; 
    delete systemService; 
    delete notifyService; 
    delete dbService; 
    delete gSessionManager;
    delete rdm;
}

/**
 * Get a DDA powerport object by its RDM datbase id.
 */
PP_DDA_PowerPort*
getPowerPortById(const char* id) {
    int i;
    PP_DDA_PowerPort* act = NULL;
    
    for (i=0; i<POWER_DEVICE_MAX; i++) {
        act = device_port_list[i];
        
        while (act != NULL) {
            if (strcmp(act->GetID(), id) == 0) {
                // we have found the powerport with the correct id
                return act;
            }
            act = act->next;
        }
    }
    
    return NULL;
}

/**
 * Get a pointer to the used (global) RDM object.
 */
CRDM* 
getRDM() {
    return rdm;
}

extern "C" {
/**
 * Return whether device is CC managed or not.
 */
int  pp_rdm_is_managed() {
    if (commandCenter->IsEnabled() && commandCenter->IsHeartBeatGood())  {
        return 1;
    } else {
    	// not CC managed or heartbeat bad - allow local logins
        return 0;
    }
}

/**
 * authenticate an existing rdm session (for rfb) by checking
 * session_id and session_key
 */
int pp_rdm_authenticate_session(const char* session_id, const char* session_key) {
    CSession* session;
    int ret = PP_ERR;
    
    printf("pp_rdm_authenticate_session(%s, %s)\n", session_id, session_key);
    
    session = gSessionManager->GetSession(session_id);
    if (session == NULL) return PP_ERR;
    
    if (strcmp(session->GetSessionKey(), session_key) == 0) {
        // session exists, key matches
        ret = PP_SUC;
    }
    
    session->Release();
    
    return ret;
}
}  // extern "C"

/**
 * Get the id of the <device> node.
 */
const char* getDeviceID() {
    return device_id;
}


/********************************************************/
/* internal functions                                   */
/********************************************************/

/**
 * Create the basic structure of the rdm database
 */
void createRDMRoot() {
        CSXDB_Element   *pNode;

        // Create System Root
        pNode = (CSXDB_Element *) rdm->db->Root();

        // Create /System
        pNode = (CSXDB_Element *) pNode->AddElement("System");
        pNode->AddAttribute("SecurityID","ReadOnly");
        pNode->AddAttribute("id","System");

        // Create /System/Sessions
        pNode = (CSXDB_Element *) pNode->AddElement("Sessions");
        pNode->AddAttribute("SecurityID","ReadOnly");
}

/**
 * Initialize the powerport objects. Will also add config-listener to
 * keep this information up to date.
 */
void initializePowerports() {
    int i;
    
    for (i=0; i<POWER_DEVICE_MAX; i++) {
        device_port_list[i] = NULL;
    }
    
    // FIXME: debug only, discover devices here
    // FIXME: fix powerports
    //device_port_list[0] = new PP_RDM_PowerPort_libpp_powerswitch(rdm, device, 1, 0, 0);
    //device_port_list[0] = new PP_RDM_PowerPort_libpp_powerswitch(rdm, device, 4, 0, 0);
    
    // FIXME: add config system listener for powerport related keys
}

