/**	
 * This file contains the code that manages /System/Device/Port.
 * It corresponds to one port object and must be subclassed to
 * represent different port types.
 * 
 * (c)2006 Raritan,  georg.hoesch@raritan.com
 */

#ifndef __RDM_PORT_H__
#define __RDM_PORT_H__

#include "pp/RDM_Definitions.h"
#include "PP_RDM_BranchManager.h"
#include "Device.h"


/* port status defines from RDM specification */
#define PORT_STATUS_INACTIVE       0
#define PORT_STATUS_ACTIVE         1
#define PORT_STATUS_BUSY_OCCUPIED  2

/** the internal representation of port data in the RDM database */
typedef	struct
{
    // attributes
    const char* pID;		// id of the port (mandatory)
    const char* pClass;		// Class attribute (KVM|Serial|Outlet) (mandatory)
    const char* pType;		// Type attribute (optional)
    const char* connection;	// currently not supported (mandatory if connection exists)
    //const char* model;	// currently not supported (optional)
    int         internal;	// 1 = This is an internal port
    int         capacity;	// usually 1
    int         status;		// Status attribute
    // elements
    char* pName;		// Name element
    //const char* associatedPort;   // currently not supported
    //const char* associatedOutlet; // currently not supported
    //const char* path;             // currently not supported
    //const char* dataRate;         // currently not supported
    //const char* parity;           // currently not supported
    //const char* dataBits;         // currently not supported
    //const char* stopBits;         // currently not supported
    //const char* flowControl;      // currently not supported
    //const char* applicationId;    // currently not supported
    //const char* detectMode;       // currently not supported
    const char* pSecurityID;	// Security ID
    const char* pBM;		// Branch Manager attribute, required internally
} portData_t;

/**
 * A port object that manages a <Port> branch in RDM. It initializes all
 * values and it handles the updates by implementing a branchmanager based
 * on PP_RDM_BranchManager. Writable values are stored in the config system.
 */
class Port: public PP_RDM_BranchManager
{
public:
    /**
     * Create a new port object. Needs to be initialized with
     * initialize() before usage.
     * 
     * @param rdm           reference to rdm
     * @param parentDevice  parent device of this port
     */
    Port(CRDM* rdm, Device* parentDevice);
    virtual ~Port();
    
    /**
     * return unique BranchManager name, same as getDeviceID() for this class.
     */
    virtual
    const char*
    GetName();

    /**
     * Get the id of this port, \0 terminated string. Must be overwritten by
     * subclass with concrete id, this class does not provide an id.
     */
    virtual
    const char*
    getPortID();

    /**
     * Load the default values from the config system and write them to
     * RDM. No need to call this explicity until absolutely required.
     */
    int                                 // 0 or RDM error code
    loadValues();
    
    /**
     * Stores the writable values from RDM to the config system. No need
     * to call this explicitly until absolutely required.
     */
    int                                 // 0 or RDM error code
    saveValues( );

#if 0
    /**
     * Load data in the internal representation from an XML database.
     * No need to call this explicity until absolutely required.
     */
    int                                 // 0 or RDM error code
    loadRDM(portData_t* portData);
    
    /**
     * Save data in the internal representation to an XML database.
     * No need to call this explicitly until absolutely required.
     */
    int                                 // 0 or RDM error code
    saveRDM(portData_t* portData);

    /**
     * Load data in the internal representation from the config system.
     * No need to call this explicity until absolutely required.
     */
    int                                 // 0 or RDM error code
    loadCfgSys(portData_t* portData);
    
    /**
     * Save data in the internal representation to the config system.
     * No need to call this explicitly until absolutely required.
     */
    int                                 // 0 or RDM error code
    saveCfgSys(portData_t* portData);
#endif
    
    /**
     * Get the node for this Port from the rdm database.
     * Returns NULL if node is not found.
     */
    CSXDB_Node* getPortNode();
    
protected:
    
    /** 
     * Implements branchmanager Update().
     */
    virtual
    int                                 // 0 or RDM Error code
    Update (
	CSession    *pUserSession,      // User session or NULL for super user
	CSXDB_Node  *pNode,             // The Node to be appended too
	CSXDB_Node  *pData              // Ptr to data to append
    );
    
    virtual
    int                                 // 0 or RDM error code
    validatePort(portData_t* portData); // port data structure

private:
    Device* parentDevice;               // reference to our parent device 
};


#endif // __RDM_PORT_H__


