#include <pthread.h>
#include <stdarg.h>
#include <string.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <openssl/md5.h>

#include <pp/cfg.h>
#include <pp/intl.h>
#include <liberic_notify.h>
#include <liberic_pthread.h>
#include <liberic_session.h>
#include <pp/bio.h>
#include <pp/rfb.h>
#include <pp/kvm.h>
#include "predictor.h"
#include "debug.h"
#include <pp/um.h>
#include <pp/ipmi.h>
#include <pp/propchange.h>
#include <pp/hal_common.h>
#include <pp/hal_kx2.h>


/******************************************************************************
* data                                                                        *
******************************************************************************/

rfb_cl_t *      clients_head     = NULL;
pthread_mutex_t clients_head_mtx = PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;

eric_session_int_id_t  exclusive_session = 0;
rfb_cl_t *             exclusive_clp = NULL;
pthread_mutex_t        exclusive_mode_mtx = PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP;

#define ERIC_CLIENT_NICE	-8

static int rfb_send_users_count(unsigned char channel, unsigned int sessions);
static void client_remove(rfb_cl_t * clp);
static const char* get_quit_reason_msg(int quit_reason);
static volatile unsigned int connection_count = 0;
static volatile unsigned int non_forensics_connection_count = 0;

/******************************************************************************
* client initialization and clanup                                            *
******************************************************************************/

rfb_cl_t * rfb_create_client_structure(pp_net_conn_data_t * conn_data) {
    rfb_cl_t * clp = NULL;

    clp = (rfb_cl_t *)malloc(sizeof(*clp));
    memset(clp, 0, sizeof(*clp));
    clp->conn_data = conn_data;
    clp->current_bio = conn_data->bio;
    /* FIXME: better use -1 as "undefined" value? - not differing across types */
    clp->video_link = 0xff; // max value for u_char, means "undefined"
    clp->data_link = 0xff;
    pp_bio_get_peer_name(conn_data->bio, clp->client_ip, sizeof(clp->client_ip));
    
    return clp;
}

int rfb_init_client_structure(rfb_cl_t *clp) {
    int err;
    
    pp_log("Connection from client %s\n", clp->client_ip);

    clp->do_exit = 0;

    clp->init_handshake_done = 0;
    clp->fbu_possible = 1;
    clp->fbu_possible_release = 0;
    clp->fb_update_requested = 0;
    clp->need_fb_fmt_update = 0;
    clp->need_fb_update = clp->full_fb_update = 1;
    if ((clp->s2c_queue = pp_queue_alloc(QUEUE_TYPE_STANDARD)) == NULL) {
        rfb_send_quit(clp, rfbQuitInternalError);
	pp_log("%s(): pp_queue_alloc() failed.\n", ___F);
	return -1;
    }

    err = pthread_mutex_init(&clp->s2c_queue_mtx, &recursive_mtx_attr);
    assert(err == 0);
    err = pthread_cond_init(&clp->s2c_queue_cond, NULL);
    assert(err == 0);

    clp->enc.pref_enc = rfbEncodingRaw;

    err = pthread_mutex_init(&clp->upd_mtx, &err_check_mtx_attr);
    assert(err == 0);
    err = pthread_mutex_init(&clp->enc_setup_mtx, &err_check_mtx_attr);
    assert(err == 0);
    err = pthread_mutex_init(&clp->fbu_possible_mtx, &err_check_mtx_attr);
    assert(err == 0);
    
    /* FIXME: do we need to protect this? */
    memcpy(&clp->pix_fmt, &rfb_pix_fmt, sizeof(clp->pix_fmt));
    clp->translate_lookup_table = NULL;
    clp->client_pix_fmt_change_req = 0;
    clp->client_encoding_change_req = 0;

    /* initialize client specific tight stuff */
    clp->tightBeforeBufSize = clp->tightAfterBufSize = 0;
    clp->tightBeforeBuf = clp->tightAfterBuf = NULL;
    clp->prevRowBuf = NULL;
    clp->tightCacheSize = 0;
    clp->useTightCache = 0;
    clp->cache_lists = NULL;
    clp->cache_memory = NULL;
    clp->cache_memory_size = 0;
    clp->cache_memory_next = NULL;

    clp->enc_req.tightCompressLevel = TIGHT_DEFAULT_COMPRESSION;
    clp->enc_req.tightXBitFullColor = TIGHT_DEFAULT_X_BIT_FULL_COLOR;
    clp->enc_req.tightCacheEnabled = 0;
    
    clp->fb_width_old = 0;
    clp->fb_height_old = 0;

    clp->fb_start_ofs_x = 0;
    clp->fb_start_ofs_y = 0;
    clp->hwenc_setup_needed = 1;
    
    clp->data_link = 0;
    clp->video_link = 0;
    //clp->target_number = 0;

    clp->next = NULL;

    pred_init(clp);
    
#ifdef TIME_MEASURE
    clp->time_pixel = clp->time_ns = clp->time_ns_req = 0;
#endif
    
    return 0;
}

int rfb_check_connection_allowed(rfb_cl_t * clp) {
    /* FIXME: the next two look like a race condition during connect, check! */
    
    /* ---- quit if exclusive mode is active --------------------------- */
    char *value=NULL;

    if (PP_SUCCED(pp_cfg_get(&value, "security.rc_ssl"))) {
	if (!strcmp(value, "force")) {
	    if (!clp->conn_data->ssl) {
		eric_notify_post_event("Connection to Remote Console failed: Unable to connect via SSL", "console", PP_NOTIFY_EVENT_GENERIC);
		free(value);
		return rfbQuitNoForcedSSLConn;
	    }
	}
    }

    free(value);

	/* don't quit the SAS hardware client! */
    if (!(clp->connection_flags & rfbConnectionFlagSasHW)) {
    	MUTEX_LOCK(&exclusive_mode_mtx);
    	if (exclusive_session && !(clp->connection_flags & rfbConnectionFlagSasHW)) {
	    MUTEX_UNLOCK(&exclusive_mode_mtx);
	    eric_notify_post_event("Connection to Remote Console failed: exclusive access active.", "console", PP_NOTIFY_EVENT_GENERIC);
            return rfbQuitExclAccess;
    	}
    	MUTEX_UNLOCK(&exclusive_mode_mtx);
    }

#ifdef PRODUCT_XX01IP_ANY
    /* ---- quit if user is not allowed to access current KVM port ----- */
    {
	u_char unit;
	u_short port;
	
	if (PP_FAILED(pp_kvm_get_unit_port_for_video_link(clp->video_link, &unit, &port))) {
	    pp_log("cannot determine port from video_link (%d), access denied\n", clp->video_link);
	    return rfbQuitKVMPortDenied;
	}
	if (!pp_kvm_switch_allowed(eric_session_get_user(clp->session),
                                   unit, port)) {
            eric_notify_post_event("Connection to Remote Console failed: access to this kvm port denied.", "console", PP_NOTIFY_EVENT_GENERIC);
            return rfbQuitKVMPortDenied;
	}
    }
#endif
    return 0;
}

int rfb_prepare_connection(rfb_cl_t * clp) {
    pp_rfb_session_data_t * sd;    

    /* ---- set rfb session data structure ----------------------------- */

    if ((sd = rfb_alloc_session_data()) == NULL) {
        rfb_send_quit(clp, rfbQuitInternalError);
	pp_log("%s(): Cannot allocate session data.\n", ___F);
	return -1;
    }

    sd->rc_active = 1;
    sd->clp = clp;
    sd->session = clp->session;
    sd->session_time = clp->session_time;
    sd->connection_flags = clp->connection_flags;
    MUTEX_LOCK(&rfb_session_data_mtx);
    list_add(&sd->listnode, &rfb_session_data_list);
    MUTEX_UNLOCK(&rfb_session_data_mtx);
    clp->sd = sd;
    
    /* ---- create writer thread --------------------------------------- */

    if (eric_pthread_create(&clp->writer_thd, 0, 128 * 1024,
			    rfb_writer, clp) != 0) {
	pp_log("%s(): Cannot create writer thread.\n", ___F);
	return rfbQuitInternalError;
    }

    /* ---- set our nice level ----------------------------------------- */

    if (nice(ERIC_CLIENT_NICE) == -1) {
	/* just report - we do not exit since it's not so critical */
	pp_log_err("%s(): nice()", ___F); 
    }

    return 0;
}

int rfb_start_grabbing(rfb_cl_t * clp) {
    if (rfb_needs_proxy_connection(clp)) {
        D(D_VERBOSE, "Connecting to different channel, no need to attach to grabber\n");
        return 0;
    }

    /* ---- register at the grabber ------------------------------------ */
    if ((clp->grab_client = pp_grab_new_client(clp->video_link, GRAB_VARIABLE_MEM_DESC)) == NULL) {
	pp_log("%s(): Failed to add grab client.\n", ___F);
	return rfbQuitTooManyClients;
    }

    return 0;    
}

void rfb_client_cleanup_handler(void * _clp) {
    rfb_cl_t * clp = (rfb_cl_t *)_clp;
    int i;
    char logmsg[PP_NOTIFY_MAX_MSG_LEN+1];

    pp_log("rfb reader cleanup ...\n");

    client_remove(clp);
   
#if defined(PP_FEAT_SESSION_REDIRECTOR)
    rfb_sas_cleanup(clp);
#endif /* PP_FEAT_SESSION_REDIRECTOR */

    clp->do_exit = 1;

    /* wakeup writer */
    MUTEX_LOCK(&clp->s2c_queue_mtx);
    if (pthread_cond_signal(&clp->s2c_queue_cond) != 0) {
	pp_log("%s(): pthread_cond_signal() failed\n", ___F);
    }
    MUTEX_UNLOCK(&clp->s2c_queue_mtx);

    /* wait for writers death */
    pthread_join(clp->writer_thd, NULL);

    if (clp->grab_client != NULL) {
	pp_grab_remove_client(clp->grab_client);
    }

    if (PP_FAILED(pp_kvm_release_link_client(clp->link_client))) {
	pp_log("cannot release link_client\n");
    }
    
    /* write exit reason to client in case it's something special */
    if (clp->sd && clp->sd->exit_reason) {
        snprintf(logmsg, sizeof(logmsg), "Connection to %s failed: %s.", clp->client_ip, get_quit_reason_msg(clp->sd->exit_reason));
        eric_notify_post_event(logmsg, "console", PP_NOTIFY_EVENT_GENERIC);
	rfb_send_quit(clp, clp->sd->exit_reason);
    }

    if (rfb_context.rfb_ptr_clp == clp) rfb_context.rfb_ptr_clp = NULL;

    REGION_UNINIT(&clp->req_reg);
    REGION_UNINIT(&clp->ack_reg);
    
    /* Release the compression state structures if any. */
    for (i = 0; i < 4; i++) {
	if (clp->zsActive[i]) deflateEnd(&clp->zsStruct[i]);
    }
    if (clp->zsActive_hw) deflateEnd(&clp->zsStruct_hw);

    free(clp->tightAfterBuf);
    free(clp->tightBeforeBuf);
    free(clp->prevRowBuf);
    rfb_deinit_cache_lists(clp);
    
    pp_queue_free(clp->s2c_queue);
    free(clp->translate_lookup_table);
    
#if defined(PP_FEAT_SESSION_REDIRECTOR)
    rfb_sas_send_closed_kvm_session(clp);
#endif /* PP_FEAT_SESSION_REDIRECTOR */

    if (clp->sd) clp->sd->rc_active = 0;
    rfb_free_session_data(clp->sd);
    if (clp->session) eric_session_release(clp->session);

    if (exclusive_session && exclusive_session == clp->session) {
	exclusive_session = 0;
	exclusive_clp = NULL;
    }

    pred_cleanup(clp);

    pthread_mutex_destroy(&clp->fbu_possible_mtx);
    pthread_mutex_destroy(&clp->upd_mtx);
    pthread_mutex_destroy(&clp->enc_setup_mtx);
    pthread_mutex_destroy(&clp->s2c_queue_mtx);
    pthread_cond_destroy(&clp->s2c_queue_cond);
	
    rfb_proxy_disconnect(clp);
    eric_net_close(clp->conn_data->bio, WAIT);
    pp_log("Connection to client %s closed.\n", clp->client_ip);
    
    snprintf(logmsg, sizeof(logmsg), "Connection to client %s closed.", clp->client_ip);
    eric_notify_post_event(logmsg, "console", PP_NOTIFY_EVENT_GENERIC);    
    
    if ((clp->connection_flags & rfbConnectionFlagSasHW) == 0) {
        non_forensics_connection_count--;
    }
    
    free(clp->conn_data);
    free(clp);

    connection_count--;
    /* FIXME: this fails on KX2, since link_client is gone */
    rfb_send_users_count(clp->video_link, connection_count);
#if defined(PRODUCT_KX2)
    if ( pp_kvm_get_client_cnt_on_video_link ( clp->video_link ) <= 0 ) 
	pp_hal_kx2_set_user_led_by_video_link(clp->video_link , PP_HAL_KX2_USER_LED_OFF);   
#endif 
    pp_propchange_enqueue(PP_PROP_KVM_SESSION_CLOSED, 0);
    
    pp_log("... rfb reader cleanup done.\n");
}

int rfb_client_insert(rfb_cl_t * clp) {
    rfb_cl_t * _clp;
    int ret = 0;

    MUTEX_LOCK(&clients_head_mtx);

    for (_clp = clients_head; _clp; _clp = _clp->next) {
	if (_clp == clp) {
	    pp_log("%s(): Cannot insert client twice.\n", ___F);
	    ret = -1;
	    goto done;
	}
    }

    clp->next = clients_head;
    clients_head = clp;

 done:
    MUTEX_UNLOCK(&clients_head_mtx);

    if (!ret) {
    	connection_count++;
    	rfb_send_users_count(clp->video_link, connection_count);
    	if ((clp->connection_flags & rfbConnectionFlagSasHW) == 0) {
    	    non_forensics_connection_count++;
    	}
    	pp_propchange_enqueue(PP_PROP_KVM_SESSION_OPENED, 0);
    }
    
    return ret;
}

static void client_remove(rfb_cl_t * clp) {
    rfb_cl_t * prev = NULL;
    rfb_cl_t * _clp;

    MUTEX_LOCK(&clients_head_mtx);

    for (_clp = clients_head; _clp; prev = _clp, _clp = _clp->next) {
	if (_clp == clp) {
	    if (prev) {
		prev->next = clp->next;
	    } else {
		clients_head = clp->next;
	    }
	    break;
	}
    }

    MUTEX_UNLOCK(&clients_head_mtx);
}

/******************************************************************************
* client handling stuff                                                       *
******************************************************************************/

int rfb_get_exclusive_video(rfb_cl_t * asking_clp) {
    rfb_cl_t * clp;

    MUTEX_LOCK(&clients_head_mtx);
    for (clp = clients_head; clp; clp = clp->next) {
	if (clp->enc_req.video_optimized && clp != asking_clp) {
            MUTEX_UNLOCK(&clients_head_mtx);
            return 0;
        }
    }
    MUTEX_UNLOCK(&clients_head_mtx);
    
    return 1;    
}

pp_rfb_session_data_t **
rfb_get_session_data_from_session(eric_session_int_id_t s) {
    struct list_head *ptr, *sd_list = &rfb_session_data_list;
    pp_rfb_session_data_t **ret = NULL;
    pp_rfb_session_data_t *sd;
    int i = 0;
    int ret_cnt = 0;
    int error = 1;

    MUTEX_LOCK(&rfb_session_data_mtx);
    for (ptr = sd_list->next; ptr != sd_list; ptr = ptr->next) {
	sd = list_entry(ptr, pp_rfb_session_data_t, listnode);
	if (sd->session != s) continue;
	
	if (pp_check_and_alloc_mem((void*)&ret, sizeof(*ret), &ret_cnt, i+1) < 0) {
            goto bail;
        }

	ret[i++] = sd;
    }
    if (pp_check_and_alloc_mem((void*)&ret, sizeof(*ret), &ret_cnt, i+1) < 0) {
	goto bail;
    }
    ret[i] = NULL;
    error = 0;
    
 bail:
    if (error) {
	free(ret);
	ret = NULL;
    }
    MUTEX_UNLOCK(&rfb_session_data_mtx);
    return ret;
}

/******************************************************************************
* messages for all clients                                                    *
******************************************************************************/

void rfb_for_all_clients(client_index_type_t client_type, u_short client_index, void (*callback)(rfb_cl_t *, va_list), ...) {
    va_list    args;
    rfb_cl_t * clp;

    MUTEX_LOCK(&clients_head_mtx);

    for (clp = clients_head; clp; clp = clp->next) {
	switch (client_type) {
	    case PP_RFB_CLIENT_TYPE_TARGET:
		{
		    u_char unit_i;
		    u_short port_i;// index in loop
		    if (PP_FAILED(pp_kvm_get_unit_and_port(clp->link_client, &unit_i, &port_i))) {
			pp_log("cannot determine port from link_client, skip this client\n");
			continue;
		    }

		    if (port_i != client_index) continue;
		    break;
		}
	    case PP_RFB_CLIENT_TYPE_VIDEO_LINK:
		if ((u_char)client_index != clp->video_link) continue;
		break;	
	    case PP_RFB_CLIENT_TYPE_DATA_LINK:
		if ((u_char)client_index != clp->data_link) continue;
		break;
	    case PP_RFB_CLIENT_TYPE_ALL:
		break;
	}
	va_start(args, callback);
	callback(clp, args);
	va_end(args);
    }

    MUTEX_UNLOCK(&clients_head_mtx);
}

void rfb_send_utf8string_v(rfb_cl_t * clp, va_list args) {
    if (rfb_enqueue_utf8string(clp, va_arg(args, void*), va_arg(args, int)) < 0) {
	D(D_ERROR, "%s(): WARNING: couldn't send utf8 string\n", ___F);
    }
}

void rfb_set_translate_function_v(rfb_cl_t * clp, va_list args UNUSED) {
    rfb_set_translate_function(clp);
}

void rfb_queue_fb_format_update_v(rfb_cl_t * clp, va_list args UNUSED) {
    MUTEX_LOCK(&clp->s2c_queue_mtx);
    if (!rfb_needs_proxy_connection(clp)) {
    	clp->need_fb_fmt_update = 1;
    	if (pthread_cond_signal(&clp->s2c_queue_cond) != 0) {
	    pp_log("%s(): pthread_cond_signal() failed\n", ___F);
    	}
    }
    MUTEX_UNLOCK(&clp->s2c_queue_mtx);
}

void rfb_schedule_update_v(rfb_cl_t * clp, va_list args UNUSED) {
    MUTEX_LOCK(&clp->s2c_queue_mtx);
    clp->need_fb_update = 1;
    if (pthread_cond_signal(&clp->s2c_queue_cond) != 0) {
	pp_log("%s(): pthread_cond_signal() failed\n", ___F);
    }
    MUTEX_UNLOCK(&clp->s2c_queue_mtx);
}

void rfb_send_osd_state_v(rfb_cl_t * clp, va_list args UNUSED) {
    rfb_send_osd_state(clp);
}

void rfb_notify_grab_avail_v(rfb_cl_t * clp, va_list args) {
    unsigned char avail;

    avail = va_arg(args, int);

    MUTEX_LOCK(&clp->fbu_possible_mtx);
    if (avail) {
	clp->fbu_possible_release = 1;
	if (pthread_cond_signal(&clp->s2c_queue_cond) != 0) {
	    pp_log("%s(): pthread_cond_signal() failed\n", ___F);
	}
    }
    MUTEX_UNLOCK(&clp->fbu_possible_mtx);
}

void rfb_send_video_settings_v(rfb_cl_t * clp, va_list args UNUSED) {
    rfb_send_vs_update(clp);
}

#if !defined(PP_FEAT_VSC_PANEL_INPUT)
void rfb_send_video_quality_v(rfb_cl_t * clp, va_list args UNUSED) {
    rfb_send_vq_update(clp);
}
#endif

/******************************************************************************
* local helper functions                                                      *
******************************************************************************/

static int rfb_send_users_count(unsigned char video_link, unsigned int sessions) {
#define __PP_MAX_COUNT_STRING 20
    char count[__PP_MAX_COUNT_STRING];

    u_char unit_i;
    u_short port_i;// index in loop
    if (PP_FAILED(pp_kvm_get_unit_port_for_video_link(video_link, &unit_i, &port_i))) {
	pp_log("cannot determine port from video_link (%d), don't send users_count\n", video_link);
	return -1;
    }
    snprintf(count, __PP_MAX_COUNT_STRING, "%d", sessions);
    pp_rfb_send_command(PP_RFB_CLIENT_TYPE_TARGET, port_i, "rc_users", count);//target,vl
    D(D_VERBOSE, "rc_users: %s\n", count);
    return 0;
#undef __PP_MAX_COUNT_STRING
}


static const char* get_quit_reason_msg(int quit_reason) {
    switch (quit_reason) {
      case 0x01: return "no permission";
      case 0x02: return "exclusive access active";
      case 0x03: return "manually rejected";
      case 0x04: return "server password disabled";
      case 0x05: return "loopback connection is senseless";
      case 0x06: return "authentication failed";
      case 0x07: return "access to this kvm port denied";
      case 0x08: return "too many clients active simultaneously";
    }
    return "unknown reason";  
}

unsigned int 
pp_rfb_get_connection_count(void)
{
	return connection_count;
}

unsigned int 
pp_rfb_get_non_forensics_connection_count(void)
{
	return non_forensics_connection_count;
}
