#ifndef _RFB_H
#define _RFB_H

/*
 * rfb.h - header file for RFB DDX implementation.
 */
/*
 *  Copyright (C) 1999 AT&T Laboratories Cambridge.  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

#include <endian.h>
#include <unistd.h>
#include <pthread.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <semaphore.h>
#include <zlib.h>
#include <openssl/ssl.h>
#include <pp/base.h>
#include <pp/grab.h>
#include <liberic_net.h>
#include <pp/rfb.h>
#include <pp/vsc.h>
#include <pp/km.h>
#include <pp/kvm.h>
#include <liberic_session.h>
#include <pp_rfb_proto.h>
#include <lara.h>
#include "tight.h"

#undef TIME_MEASURE
#define MAX_ENCODINGS  10

/* enable 24 and 32 bit support */
#define ENABLE_24_AND_32_BIT	0

#if !ENABLE_24_AND_32_BIT
#define ERROR_24_AND_32_BIT	{ D(D_ERROR, "Support for 24 and 32 bit not enabled."); return -1; }
#endif /* !ENABLE_24_AND_32_BIT */

static const unsigned char AA_OSD_PROGRESS_MAX  = 9;
static const unsigned int  OSD_MSG_TOUT         = 2000;
static const unsigned int  OSD_MODE_MSG_TOUT    = 5000;

struct _rfb_cl_t;

struct _pp_rfb_session_data_t {
    struct list_head listnode;
    eric_session_int_id_t session;	/* the session to which this data belongs */
    int rc_active;
    int disconnect;
    int exit_reason;
    u_int32_t connection_flags;
    struct _rfb_cl_t *clp;
    time_t session_time;
};

typedef int (*rfb_send_rect_softenc_fn_t)(struct _rfb_cl_t * clp,
					  int x, int y, int w, int h);

/*
 * rfbTranslateFnType is the type of translation functions.
 */

typedef void (*rfb_translate_fn_t)(char * table, rfbPixelFormat * in,
				   rfbPixelFormat * out,
				   char * iptr, char * optr,
				   u_char tile_pitch,
				   int width, int height);

/* ------------------------------------------------------------------------- *
 * structures and defines to handle cache in tight encoding                  *
 * ------------------------------------------------------------------------- */

#define RFB_NUM_RECTS_HASH_LIST			8
#define RFB_NUM_RECTS_SHORT_TERM_CACHE		4
#define RFB_NUM_RECTS_LONG_TERM_CACHE		4
#define RFB_RECT_SHORT_2_LONG_TERM_CACHE	3

/*
 * GET_PTR_INTO_FB returns a pointer into the frame buffer
 */
#ifdef PP_FEAT_VSC_HW_ENCODING

# define GET_PTR_INTO_FB(clp, x, y)								\
    (clp->fb_start + ((y - clp->fb_start_ofs_y) * PP_FB_TILE_WIDTH   * (clp)->fb_tile_pitch +	\
                      (x - clp->fb_start_ofs_x) * PP_FB_TILE_HEIGHT) * (clp)->fb_bpp / 8)
#else

# define GET_PTR_INTO_FB(clp, x, y)						\
    (clp->grab_client->buf + ((y) * PP_FB_TILE_WIDTH * (clp)->fb_tiles_w +	\
			      (x) * PP_FB_TILE_HEIGHT) * (clp)->fb_bpp / 8)
#endif

typedef struct _rfb_cache_list_value_t {
    u_int32_t hash_new, hash_sdbm, col;
    int8_t counter, client_id;
    struct _rfb_cache_list_value_t * next;    
} rfb_cache_list_value_t;

typedef struct _rfb_cache_lists_t {
    rfb_cache_list_value_t * hash_list;
    rfb_cache_list_value_t * short_term_cache;
    rfb_cache_list_value_t * long_term_cache;
} rfb_cache_lists_t;

typedef struct {
    short pref_enc;
    rfb_send_rect_softenc_fn_t send_rect_softenc_fn;

    int tightCompressLevel;
    u_int8_t tightXBitFullColor;
    int tightCacheEnabled;
    u_char video_optimized;
} rfb_enc_params;

typedef enum {
    PRED_TYPE_NONE,
    PRED_TYPE_SW,
    PRED_TYPE_HW
} pred_type_t;

/* ------------------------------------------------------------------------- *
 * client structure                                                          *
 * ------------------------------------------------------------------------- */

typedef struct _rfb_cl_t {
    pp_net_conn_data_t * conn_data;
    pthread_t writer_thd;
    eric_session_int_id_t session;
    pp_rfb_session_data_t * sd;
    char client_ip[INET6_ADDRSTRLEN];
    volatile int do_exit;

    u_char video_link;
    u_char data_link;

    // u_char target_number;
    pp_kvm_link_client_t link_client;

    BIO *proxy_bio;
    BIO *current_bio;

    unsigned char kvm_node;

    time_t session_time;
    int fb_update_requested;
    int need_fb_fmt_update;
    int need_fb_update;
    int full_fb_update;
    int init_handshake_done;
    u_int32_t connection_flags;
    pp_queue_t * s2c_queue;
    pthread_mutex_t s2c_queue_mtx;
    pthread_cond_t s2c_queue_cond;

    /* lock between grab failing and disabling fb update tries
       and fb updates getting possible again (callback from grabber) */
    pthread_mutex_t fbu_possible_mtx;
    int fbu_possible;
    int fbu_possible_release;
   
    /* encoding setup (asynchronous) */
    pthread_mutex_t enc_setup_mtx;    
    rfb_enc_params enc;
    rfb_enc_params enc_req;
    rfbPixelFormat pix_fmt;
    rfbPixelFormat pix_fmt_req;
    u_char client_pix_fmt_change_req;
    u_char client_encoding_change_req;
    
    u_char hwenc_subenc;
    u_char hwenc_zlib_level;
    u_char hwenc_setup_needed;
    
    /* helper for calculation of the offset to framebuffer */
    u_char *fb_start;
    u_char  fb_tile_pitch;
    u_short fb_start_ofs_x;
    u_short fb_start_ofs_y;
    
    /* tight encoding or hw enc through zlib
       client specific data - preserve zlib streams' state for each client */
    z_stream zsStruct[4];
    int zsActive[4];
    int zsLevel[4];
    z_stream zsStruct_hw;
    int zsActive_hw;
    int zsLevel_hw;
    
    int usePixelFormat24; /* set on every rfbSendRectEncodingTight() call. */
    int paletteNumColors, paletteMaxColors;
    u_int32_t monoBackground, monoForeground;
    PALETTE palette;
    /* tight caching */
    int tightCacheSize, useTightCache;
    rfb_cache_lists_t ** cache_lists;
    void * cache_memory;
    size_t cache_memory_size;
    void * cache_memory_next;

    /* Pointers to dynamically-allocated buffers. */
    int tightBeforeBufSize;
    char * tightBeforeBuf;
    char * tightXBitFullColorBuf;
    int tightAfterBufSize;
    char * tightAfterBuf;
    int * prevRowBuf;
    
    /*
     * As part of the FramebufferUpdateRequest, a client can express
     * interest in a subrectangle of the whole framebuffer.  This is
     * stored in the requestedRegion member.  In the normal case this is
     * the whole framebuffer if the client is ready, empty if it's not.
     */

    RegionRec req_reg;
    RegionRec ack_reg;

    pthread_mutex_t upd_mtx;

    u_int fb_width, fb_width_old;
    u_int fb_height, fb_height_old;
    u_int fb_width_pd;
    u_int fb_height_pd;
    u_int fb_tiles_w;
    u_int fb_tiles_h;
    u_int fb_bpp;
    int   fb_is_unsupported; // framebuffer format is unsupported by
                             // the capturing device

    pp_grab_client_t *grab_client;

    /*
     * translateFn points to the translation function which is used to
     * copy and translate a rectangle from the framebuffer to an output
     * buffer.
     */

    rfb_translate_fn_t translate_fn;
    
    rfb_translate_fn_t rawvsc_translate_fn;
    int rawvsc_use_serverpixelformat;
    char * translate_lookup_table;

    /* software LRLE encoding */
    rfb_translate_fn_t lrle_translate_fn;
    char * lrle_translate_lookup_table;

    /*
     * updateBuf must be big enough to send at least one whole line of the
     * framebuffer.  So for a max screen width of say 2K with 32-bit pixels
     * this means 8K minimum.
     * For Raw encoding and VSC devices, a multiple of 16 lines (this means
     * actually at least 16) must fit into the buffer. These devices support
     * a maximum of 16 bits (2 bytes) per pixel, which results in 4K for a
     * 2K screen width and 64K for 16 lines.
     */
#define UPDATE_BUF_SIZE 65536
    char update_buf[UPDATE_BUF_SIZE];
    size_t ub_len;

    /* compression type/level prediction */

    // bandwidth measurement
    u_int64_t pred_bwidth_stage1, pred_bwidth_stage2; // time the ticks arrived
    unsigned int pred_bwidth_bytes; // bytes used for bandwidth measurement
    u_int32_t pred_bwidth;	    // measured bandwidth (bytes/sec)
    unsigned char pred_bwidth_state; // send bandwidth request asap
    
    // compression time and data size, correction factors
    pp_stopwatch_t pred_watch_real; // stopwatch for compression time
    u_int64_t pred_t_ns;       // temp storage for compression time during update
    unsigned long pred_t_real; // compression time in us
    unsigned long pred_v_real; // compressed data size
    unsigned long pred_pixels; // pixel count
    unsigned char pred_code;   // encoding used for this update
    time_t pred_time;	       // last time we predicted
    float pred_fv_korr;	       // correction factor for data size
    float pred_ft_korr;	       // correction factor for compression time
    
    FILE* pred_file;	       // measure file output (csv)

    pred_type_t pred_type;     // clients wishes to use a predictor
    
    int vs_update_now;
    int vs_update_allowed;
    int vq_update_now;

#ifdef TIME_MEASURE
    pp_stopwatch_t time_all;
    pp_stopwatch_t time_net;
    pp_stopwatch_t time_req;
    u_long time_pixel;
    u_int64_t time_ns_all;
    u_int64_t time_ns_net;
    u_int64_t time_ns_req;
    u_int64_t time_ns;
#endif
    
#if defined(PP_FEAT_SESSION_REDIRECTOR)
    pp_queue_t * sas_input_queue;
    pthread_mutex_t sas_input_queue_mtx;
    pthread_cond_t sas_input_queue_cond;
    int sas_exit_input_thd;
    pthread_t sas_input_thd;
    vector_t *modifier_vector;
    vector_t *unfiltered_keys_vector;
#endif /* PP_FEAT_SESSION_REDIRECTOR */

    struct _rfb_cl_t * next;    
} rfb_cl_t;

/* ------------------------------------------------------------------------- *
 * input event structure
 * used to queue keyboard/pointer events, combine pointer events and
 * prioritize keyboard events
 * ------------------------------------------------------------------------- */
typedef struct {
    rfb_cl_t *clp;    
    u_int8_t type;
    union {
    	u_int8_t keycode;
    	struct {
    	    u_int16_t x;
    	    u_int16_t y;
    	    u_int16_t z;
    	    u_int8_t button_mask;
    	} mouse_event;
    	u_int8_t sync_type;
    } data;
    struct timeval arrived;
    eric_session_int_id_t session;
} input_event_t;

/* ------------------------------------------------------------------------- *
 * port (video/input path) structure
 * ------------------------------------------------------------------------- */
typedef struct {
     unsigned char	id;
     pthread_t		input_thd;
     int		input_thd_run;
     sem_t		input_sem;
     pp_queue_t*	input_queue;
     pthread_mutex_t	input_queue_mtx;
     int		tight_disable_gradient;
     pthread_mutex_t	osd_state_mtx;
     pp_blank_screen_t	blank_state;
     char*		osd_message;
     unsigned short	osd_timeout_ms;
     rfb_cl_t *         rfb_ptr_clp;
} rfb_context_t;

extern rfb_context_t rfb_context;

/* --------------------------------------------------------------------------*
 * connection parameter structure                                            *
 * ------------------------------------------------------------------------- */
typedef vector_t rfb_connection_parameter_list_t;

/* ------------------------------------------------------------------------- *
 * macros                                                                    *
 * ------------------------------------------------------------------------- */

/*
 * FB_UPDATE_PENDING is used to test whether there is a framebuffer update
 * needing to be sent to the client.
 */

#define FB_UPDATE_PENDING(clp)	REGION_NOTEMPTY(&(clp)->mod_reg)

/*
 * SAFE_REGION_INIT creates an empty region (i.e. a region with no areas)
 * if it is given a rectangle with a width or height of zero. It appears
 * that REGION_INTERSECT does not quite do the right thing with zero-width
 * rectangles, but it should with completely empty regions.
 */

#define SAFE_REGION_INIT(preg, rect, size)			\
	{							\
		if (((rect)) &&					\
		    (((rect)->x2 == (rect)->x1) ||		\
		     ((rect)->y2 == (rect)->y1))) {		\
			REGION_INIT((preg), NullBox, 0);	\
		} else {					\
			REGION_INIT((preg), (rect), (size));	\
		}						\
	}

/* ------------------------------------------------------------------------- *
 * misc inline functions
 * ------------------------------------------------------------------------- */

inline static int
rfb_read(rfb_cl_t * clp, void * buf, size_t len)
{
    return eric_net_read_exact(clp->current_bio, buf, len, 0);
}

inline static int
rfb_read_no_timeout(rfb_cl_t * clp, void * buf, size_t len)
{
    return eric_net_read_exact(clp->current_bio, buf, len, 1);
}

/* write function writes to clp->current_bio */
inline static int
rfb_write(rfb_cl_t * clp, const void * buf, size_t len)
{
    return eric_net_write_exact(clp->current_bio, buf, len, 0);
}

/* write function writes always to the client bio; must be used to write from writer thread */
inline static int
rfb_write_writer(rfb_cl_t * clp, const void * buf, size_t len)
{
    return eric_net_write_exact(clp->conn_data->bio, buf, len, 0);
}

/* ------------------------------------------------------------------------- *
 * rfb.c
 * ------------------------------------------------------------------------- */

/* functions for connecting the rfb as encoder to the vsc */
void rfb_vsc_event(unsigned char video_link, pp_vsc_event_t* event);

/* functions for connecting the rfb as encoder to the km */
void rfb_send_kbdlayout_to_all(pp_km_encoder_desc_t *enc, unsigned char data_link, char* kbd);
void rfb_km_event(pp_km_encoder_desc_t *enc, unsigned char channel, km_event_t* event);

void rfb_set_color_translation(unsigned char video_link, fb_color_info_t * fb_c_info);
void rfb_queue_fb_format_update(unsigned char video_link, fb_format_info_t *fb_f_info);
void rfb_schedule_update(unsigned char video_link);
void rfb_notify_grab_avail(unsigned char video_link);

pp_rfb_session_data_t ** rfb_get_session_data_from_session(eric_session_int_id_t s);

void rfb_send_utf8_string(unsigned char channel, char *msg, int len);
void rfb_send_video_settings(unsigned char channel);
#if !defined(PP_FEAT_VSC_PANEL_INPUT)
void rfb_send_video_quality(unsigned char channel);
#endif
#if defined(PP_FEAT_DISABLE_KVM_OVER_IP)
int rfb_kvm_ip_disabled(void);
#endif

rfb_connection_parameter_list_t *rfb_create_connection_parameter_list(void);
void rfb_free_connection_parameter_list(rfb_connection_parameter_list_t *list);
void rfb_add_connection_parameter(rfb_connection_parameter_list_t *list, const char *name, const char *value);
size_t rfb_connection_parameter_list_size(rfb_connection_parameter_list_t *list);
const char *rfb_connection_parameter_name(rfb_connection_parameter_list_t *list, unsigned int idx);
const char *rfb_connection_parameter_value(rfb_connection_parameter_list_t *list, unsigned int idx);

#if defined(PP_FEAT_SESSION_REDIRECTOR)
void rfb_sas_session_opened(eric_session_int_id_t id);
void rfb_sas_session_closed(eric_session_int_id_t id);
void rfb_sas_send_local_key(pp_km_encoder_desc_t *enc, u_char key);
void rfb_sas_send_local_mouse(pp_km_encoder_desc_t *enc, int x, int y, int z, u_char buttonmask);
void rfb_sas_send_kvm_switch_event(eric_session_int_id_t session,
				   u_char channel, u_char unit, u_short port);
void rfb_sas_login_failure(const char * user, const char * ip_str);
#endif /* PP_FEAT_SESSION_REDIRECTOR */

/* ------------------------------------------------------------------------- *
 * init.c
 * ------------------------------------------------------------------------- */

extern pthread_mutexattr_t err_check_mtx_attr;
extern pthread_mutexattr_t recursive_mtx_attr;
extern struct list_head rfb_session_data_list;
extern pthread_mutex_t rfb_session_data_mtx;

pp_rfb_session_data_t * rfb_alloc_session_data(void);
void rfb_free_session_data(pp_rfb_session_data_t * sd);

/* ------------------------------------------------------------------------- *
 * writer.c
 * ------------------------------------------------------------------------- */

void * rfb_writer(void * _clp);
int rfb_writer_enqueue_pdu(rfb_cl_t * clp, void * pdu, size_t pdu_len);

/* ------------------------------------------------------------------------- *
 * clients.c
 * ------------------------------------------------------------------------- */

extern eric_session_int_id_t exclusive_session;
extern rfb_cl_t *exclusive_clp;
extern pthread_mutex_t exclusive_mode_mtx;
extern rfb_cl_t *clients_head;
extern pthread_mutex_t clients_head_mtx;

int rfb_get_exclusive_video(rfb_cl_t * asking_clp);

rfb_cl_t * rfb_create_client_structure(pp_net_conn_data_t * conn_data);
int rfb_init_client_structure(rfb_cl_t *clp);
int rfb_check_connection_allowed(rfb_cl_t * clp);
int rfb_prepare_connection(rfb_cl_t * clp);
int rfb_start_grabbing(rfb_cl_t * clp);
int rfb_client_insert(rfb_cl_t * clp);
void rfb_client_cleanup_handler(void * _clp);
#ifdef PRODUCT_XX01IP_ANY
void rfb_disconnect_if_kvm_port_denied(eric_session_int_id_t s, va_list args);
#endif

void rfb_for_all_clients(client_index_type_t type, u_short client_index, void (*callback)(rfb_cl_t *, va_list), ...);
void rfb_send_utf8string_v(rfb_cl_t * clp, va_list args);
void rfb_set_translate_function_v(rfb_cl_t * clp, va_list args);
void rfb_queue_fb_format_update_v(rfb_cl_t * clp, va_list args);
void rfb_schedule_update_v(rfb_cl_t * clp, va_list args);
void rfb_send_osd_state_v(rfb_cl_t * clp, va_list args);
void rfb_notify_grab_avail_v(rfb_cl_t * clp, va_list args);
void rfb_send_video_settings_v(rfb_cl_t * clp, va_list args);
#if !defined(PP_FEAT_VSC_PANEL_INPUT)
void rfb_send_video_quality_v(rfb_cl_t * clp, va_list args);
#endif /* !PP_FEAT_VSC_PANEL_INPUT */

/* ------------------------------------------------------------------------- *
 * sas.c
 * ------------------------------------------------------------------------- */

u_int32_t rfb_adjust_connection_flags(rfb_cl_t *clp, u_int32_t flags);

#if defined(PP_FEAT_SESSION_REDIRECTOR)

int rfb_sas_init(rfb_cl_t * clp);
void rfb_sas_cleanup(rfb_cl_t * clp);

// message to all clients at one channel; use -1 as channel if you want to send to all channels
void rfb_for_all_sas_clients(int channel, void (*callback)(rfb_cl_t *, va_list), ...);
int rfb_sas_send_open_sessions(rfb_cl_t * clp);
void rfb_sas_login_failure_v(rfb_cl_t * clp, va_list args);
void rfb_sas_send_opened_web_session_v(rfb_cl_t * clp, va_list args);
void rfb_sas_send_closed_web_session_v(rfb_cl_t * clp, va_list args);
void rfb_sas_send_new_kvm_session(rfb_cl_t * new_clp);
void rfb_sas_send_closed_kvm_session(rfb_cl_t * closed_clp);
void rfb_sas_send_kvm_exclusive(rfb_cl_t * excl_clp, int exclusive);
void rfb_sas_send_kbd_event(u_int8_t keycode, eric_session_int_id_t session, rfb_cl_t *causing_clp);
void rfb_sas_send_mouse_event(u_int16_t x, u_int16_t y, u_int16_t z,
			      u_int8_t buttons, u_int8_t type,
			      eric_session_int_id_t session,
			      rfb_cl_t *causing_clp);
void rfb_sas_send_mousesync_event(u_int8_t type, eric_session_int_id_t session, rfb_cl_t *causing_clp);
void rfb_sas_send_kvm_switch_event_v(rfb_cl_t * clp, va_list args);

#endif /* PP_FEAT_SESSION_REDIRECTOR */

/* ------------------------------------------------------------------------- *
 * rfb_handler.c                                                             *
 * ------------------------------------------------------------------------- */

int rfb_authenticate(rfb_cl_t *clp, eric_session_int_id_t *session);
int rfb_send_connection_parameters(rfb_cl_t * clp);
int rfb_negotiate_protocol_version(rfb_cl_t * clp);
int rfb_initialize(rfb_cl_t * clp);
int rfb_main_protocol_handler(rfb_cl_t * clp);
int rfb_send_vs_update(rfb_cl_t * clp);
int rfb_send_vq_update(rfb_cl_t * clp);
int rfb_send_bandwidth_request(rfb_cl_t * clp, unsigned int length);
int rfb_send_osd_state(rfb_cl_t * clp);
int rfb_send_kbd_layout(rfb_cl_t * clp);
void rfb_setup_encoding(rfb_cl_t * clp, u_long enc_word,
			rfb_send_rect_softenc_fn_t softenc_fn,
			u_char use_cache);
/* The following routines must only be called from within the writer thread! */
void rfb_prepare_encoding(rfb_cl_t * clp);
int rfb_send_fb_format_update(rfb_cl_t * clp);
int rfb_send_update_buf(rfb_cl_t * clp);
int rfb_send_fb_update(rfb_cl_t * clp, int clip_mod_region, int full_update);

/* ------------------------------------------------------------------------- *
 * rfb_proto.c                                                               *
 * ------------------------------------------------------------------------- */

int rfb_read_message_type(rfb_cl_t * clp);

/* read messages - Client -> Server */
int rfb_read_connection_init_string(rfb_cl_t * clp);
int rfb_read_protocol_version(rfb_cl_t * clp, int *major, int *minor);
int rfb_read_login_pdu(rfb_cl_t * clp, char *user, u_int32_t *flags);
int rfb_read_auth_response_pdu(rfb_cl_t * clp, char *response, size_t len, size_t min_len);
int rfb_read_init_pdu(rfb_cl_t * clp, u_char *kvm_node, u_char *unit, u_int16_t *port, u_int16_t *flags);
int rfb_read_set_pixel_format_pdu(rfb_cl_t * clp, rfbPixelFormat *fmt);
int rfb_read_set_encodings_pdu(rfb_cl_t * clp, u_int32_t **encodings);
int rfb_read_fb_update_req_pdu(rfb_cl_t * clp, int *incremental, BoxRec *box);
int rfb_read_keyboard_event_pdu(rfb_cl_t * clp, input_event_t *event);
int rfb_read_pointer_event_pdu(rfb_cl_t * clp, input_event_t *event);
int rfb_read_utf8string_pdu(rfb_cl_t * clp, char **string, int *len);
int rfb_read_user_prop_change_pdu(rfb_cl_t * clp, char ** key, char ** value);
int rfb_read_global_prop_change_pdu(rfb_cl_t * clp, char ** key, char ** value);
int rfb_read_mousesync_pdu(rfb_cl_t * clp);
char* rfb_read_cursor_change_pdu(rfb_cl_t * clp);
int rfb_read_ping_pdu(rfb_cl_t * clp, u_int32_t *serial);
int rfb_read_bandwidth_tick_pdu(rfb_cl_t * clp);
int rfb_read_kvm_switch_pdu(rfb_cl_t * clp);
int rfb_read_video_refresh_pdu(rfb_cl_t * clp);
int rfb_read_video_settings_pdu(rfb_cl_t * clp, u_int8_t *event, u_int16_t *value);
int rfb_read_video_settings_request_pdu(rfb_cl_t * clp);
int rfb_read_video_quality_pdu(rfb_cl_t * clp, u_int8_t *event, u_int8_t *value);
int rfb_read_video_quality_request_pdu(rfb_cl_t * clp);
int rfb_read_rc_mode_request_pdu(rfb_cl_t * clp);

/* read messages - Server -> Client */
int rfb_read_auth_caps(rfb_cl_t * clp, unsigned int *caps);
int rfb_read_auth_challenge(rfb_cl_t * clp, char *challenge, size_t expected_len);
int rfb_read_auth_successful(rfb_cl_t * clp, u_int32_t *flags);
int rfb_read_quit(rfb_cl_t * clp, u_int8_t *reason);
int rfb_read_osd_state(rfb_cl_t * clp, char **msg, u_int8_t *blank, u_int16_t *timeout, struct timeval *tv);
int rfb_read_kbd_layout(rfb_cl_t * clp, char** kbd);
rfb_connection_parameter_list_t *rfb_read_connection_parameters(rfb_cl_t * clp);
int rfb_read_fb_format_update(rfb_cl_t * clp, int *fb_width, int *fb_height, u_int8_t *unsupported, rfbPixelFormat *fmt, struct timeval *tv);

/* write messages - Server -> Client */
int rfb_send_quit(rfb_cl_t * clp, int reason);
int rfb_send_protocol_version(rfb_cl_t * clp, int major, int minor);
int rfb_send_auth_caps(rfb_cl_t * clp, unsigned int caps);
int rfb_send_auth_successful(rfb_cl_t * clp, u_int32_t flags);
int rfb_send_auth_challenge(rfb_cl_t * clp, char *challenge, size_t len);
#if defined(PP_FEAT_SESSION_REDIRECTOR)
int rfb_send_sas_error(rfb_cl_t * clp, int errorcode);
#endif /* PP_FEAT_SESSION_REDIRECTOR */

/* write messages - Client -> Server */
int rfb_send_connection_init_string(rfb_cl_t * clp);
int rfb_send_init_pdu(rfb_cl_t * clp, u_char kvm_node, u_char unit, u_int16_t port, u_int16_t flags);
int rfb_send_login_pdu(rfb_cl_t * clp, const char *username, u_int8_t method);
int rfb_send_auth_response_pdu(rfb_cl_t * clp, char *response, size_t len);

/* write messages (called from writer thread) - Server -> Client */
int rfb_writer_send_ack_pixel_format(rfb_cl_t * clp, rfbPixelFormat *fmt);
int rfb_writer_send_fb_format_update(rfb_cl_t * clp, u_int8_t unsupported,
				     u_int16_t width, u_int16_t height,
				     rfbPixelFormat *fmt);
int rfb_writer_send_update_buf(rfb_cl_t * clp, char *buf, size_t len);

/* enqueue messages - Server -> Client */
void rfb_enqueue_message_v(rfb_cl_t * clp, va_list args);
int rfb_enqueue_utf8string(rfb_cl_t* clp, const void* msg, u_int16_t length);
int rfb_enqueue_connection_parameters(rfb_cl_t * clp, rfb_connection_parameter_list_t *params);
int rfb_enqueue_fb_format_update(rfb_cl_t * clp, rfbPixelFormat *fmt);
int rfb_enqueue_fb_format_update_params(rfb_cl_t * clp, int fb_width, int fb_height, u_int8_t unsupported,
                                        rfbPixelFormat *fmt, struct timeval *tv);
int rfb_enqueue_ping_request(rfb_cl_t * clp, u_int32_t serial);
int rfb_enqueue_ping_reply(rfb_cl_t * clp, u_int32_t serial);
int rfb_enqueue_vs_update(rfb_cl_t * clp, video_settings_t *settings);
int rfb_enqueue_vq_update(rfb_cl_t * clp, int noise_filter);
int rfb_enqueue_bandwidth_request(rfb_cl_t * clp, unsigned int length);
int rfb_enqueue_message(rfb_cl_t *clp, const char *msg);
int rfb_enqueue_osd_state(rfb_cl_t * clp, const char *msg, u_int8_t blank, u_int16_t timeout);
int rfb_enqueue_osd_state_time(rfb_cl_t * clp, const char *msg, u_int8_t blank, u_int16_t timeout, struct timeval *tv);
int rfb_enqueue_kbd_layout(rfb_cl_t * clp, char* kbd);
#if defined(PP_FEAT_SESSION_REDIRECTOR)
int rfb_enqueue_sas_error(rfb_cl_t * clp, int errorcode);
int rfb_enqueue_sas_existing_session(rfb_cl_t * clp, const char *user, const char *ip,
				     int self, time_t login, eric_session_int_id_t session);
int rfb_enqueue_sas_existing_kvm_session(rfb_cl_t * clp, eric_session_int_id_t session,
					 rfb_cl_t *new_clp, int exclusive, time_t login_time);
int rfb_enqueue_sas_generic_event(rfb_cl_t * clp, int type, eric_session_int_id_t session);
int rfb_enqueue_sas_new_kvm_session(rfb_cl_t * clp, eric_session_int_id_t session, rfb_cl_t *causing_clp, time_t session_time);
int rfb_enqueue_sas_kvm_session_event(rfb_cl_t * clp, int type, eric_session_int_id_t session, rfb_cl_t *causing_clp);
int rfb_enqueue_sas_input_event(rfb_cl_t * clp, input_event_t *ie, eric_session_int_id_t session, rfb_cl_t *causing_clp);
int rfb_enqueue_sas_new_session_event(rfb_cl_t * clp, const char *user, const char *ip,
				      time_t login, eric_session_int_id_t session);
int rfb_enqueue_sas_login_failure_event(rfb_cl_t * clp, const char *user, const char *ip);
int rfb_enqueue_sas_kvm_switch_event(rfb_cl_t * clp, eric_session_int_id_t session,
				     u_char channel, u_char unit, u_short port);
#endif /* PP_FEAT_SESSION_REDIRECTOR */

char * rfb_create_rfb_message_pdu(const char* msg, size_t *size);
char * rfb_create_command_pdu(const char* command, const char* params, size_t *size);
char * rfb_create_kbd_layout_pdu(char* kbd, size_t *size);
char * rfb_create_fb_update_pdu(int n_upd_reg_rects, int with_timestamp,
				int full_update, size_t *size);
char * rfb_create_fb_update_rect_header(int x, int y, int w, int h,
					u_int32_t enc, size_t *size);
char * rfb_create_raw_vsc_hdr(u_int8_t flags, size_t *size);
char * rfb_create_hwenc_hdr(u_int32_t hw_size, size_t *size);

/* ------------------------------------------------------------------------- *
 * input.c                                                                   *
 * ------------------------------------------------------------------------- */

void* rfb_input_thread(void* arg);
int rfb_enqueue_input_event(input_event_t *event);

/* ------------------------------------------------------------------------- *
 * translate.c                                                               *
 * ------------------------------------------------------------------------- */

extern rfbPixelFormat rfb_pix_fmt;
extern u_int8_t tightLUT1BitFullColorBlackWhite[];
extern u_int8_t tightLUT2BitFullColorGrayscale[];
extern u_int8_t tightLUT4BitFullColorGrayscale[];
extern u_int8_t tightLUT4BitFullColor16Colors[];

int rfb_set_translate_function(rfb_cl_t * clp);

/* ------------------------------------------------------------------------- *
 * raw.c
 * ------------------------------------------------------------------------- */

int rfb_send_rect_raw(rfb_cl_t * clp, int x, int y, int w, int h);
int rfb_send_rect_rawvsc(rfb_cl_t * clp, int x, int y, int w, int h);
void rfb_translate_none_rawvsc(char * table, rfbPixelFormat * in,
			       rfbPixelFormat * out, char * iptr, char * optr,
			       u_char tile_pitch, int width, int height);

/* ------------------------------------------------------------------------- *
 * hextile.c
 * ------------------------------------------------------------------------- */

int rfb_send_rect_hextile(rfb_cl_t * clp, int x, int y, int w, int h);

/* ------------------------------------------------------------------------- *
 * tight.c
 * ------------------------------------------------------------------------- */

#define TIGHT_DEFAULT_COMPRESSION      6
#define TIGHT_DEFAULT_X_BIT_FULL_COLOR 8

int rfb_num_coded_rects_tight(rfb_cl_t * clp, int x, int y, int w, int h);
int rfb_send_rect_tight(rfb_cl_t * clp, int x, int y, int w, int h);

/* ------------------------------------------------------------------------- *
 * tight_cache.c
 * ------------------------------------------------------------------------- */

// hash functions
u_int32_t do_hash_new(u_int8_t *data, int size);
u_int32_t do_hash_sdbm(u_int8_t *data, int size);
// support functions
void out_cache_lists(rfb_cache_lists_t * cache_lists);
int rfb_init_cache_lists(rfb_cl_t * clp, int new_tight_cache_size);
void rfb_deinit_cache_lists(rfb_cl_t * clp);
// search functions
rfb_cache_list_value_t * rfb_search_cache_list_lt(int8_t * client_id, 
	rfb_cache_lists_t * cache_lists, int num, u_int32_t hash_new, u_int32_t hash_sdbm, 
	u_int32_t col);
rfb_cache_list_value_t * rfb_search_cache_list_st(int8_t * client_id, 
	rfb_cache_lists_t * cache_lists, int num, u_int32_t hash_new, u_int32_t hash_sdbm, 
	u_int32_t col, rfb_cache_list_value_t * lt_2l);
rfb_cache_list_value_t * rfb_search_cache_list_hl(int8_t * client_id, 
	rfb_cache_lists_t * cache_lists, int num, u_int32_t hash_new, u_int32_t hash_sdbm, 
	u_int32_t col, rfb_cache_list_value_t * lt_2l, rfb_cache_list_value_t * st_2l);
// insert function
void rfb_insert_short_term(int8_t * client_id, rfb_cache_lists_t * cache_lists, 
	u_int32_t hash_new, u_int32_t hash_sdbm, u_int32_t col, 
	rfb_cache_list_value_t * st_2l, rfb_cache_list_value_t * hl_2l);

/* ------------------------------------------------------------------------- *
 * hwenc.c
 * ------------------------------------------------------------------------- */

int  rfb_hwenc_init(void);
void rfb_hwenc_cleanup(void);

/**
 * the update function for hardware encoding of the request
 * region, requesting a number of hardware encoded rects
 * directly for our client
 */
int  rfb_send_update_hwenc(rfb_cl_t * clp, RegionRec *req_reg, int full_update);

#ifdef PP_FEAT_VSC_HW_ENCODING
extern vsc_encoding_desc_t hwenc_subencs[];
#else
int  rfb_send_rect_lrle(rfb_cl_t * clp, int x, int y, int w, int h);
int  rfb_lrle_init_tables(u_char subformat);
#endif

/* ------------------------------------------------------------------------- *
 * swenc.c
 * ------------------------------------------------------------------------- */

#if !defined(PP_FEAT_VSC_HW_ENCODING)
int rfb_send_update_softenc(rfb_cl_t * clp, RegionRec *req_reg, int full_update);
#endif /* !defined(PP_FEAT_VSC_HW_ENCODING) */

/* ------------------------------------------------------------------------- *
 * proxy.c
 * ------------------------------------------------------------------------- */

int rfb_needs_proxy_connection(rfb_cl_t * clp);
int rfb_can_be_connected_by_proxy(rfb_cl_t * clp);

int rfb_proxy_connect_to_channel(rfb_cl_t * clp);
void rfb_proxy_run(rfb_cl_t * clp);

void rfb_proxy_disconnect(rfb_cl_t * clp);

#endif /* !_RFB_H */
