/*
 * tabletranstemplate.c - template for translation using lookup tables.
 *
 * This file shouldn't be compiled.  It is included multiple times by
 * translate.c, each time with different definitions of the macros IN and OUT.
 *
 * For each pair of values IN and OUT, this file defines two functions for
 * translating a given rectangle of pixel data.  One uses a single lookup
 * table, and the other uses three separate lookup tables for the red, green
 * and blue values.
 *
 * I know this code isn't nice to read because of all the macros, but
 * efficiency is important here.
 */

/*
 *  Copyright (C) 1999 AT&T Laboratories Cambridge.  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

#if !defined(IN) || !defined(OUT)
#error "This file shouldn't be compiled."
#error "It is included as part of translate.c"
#endif

#if (IN != 24)
#  define IN_T CONCAT3E(u_int,IN,_t)
#endif
#define OUT_T CONCAT3E(u_int,OUT,_t)
#define SwapIN(x) CONCAT2E(bswap_,IN(x))
#define rfbTranslateWithSingleTableINtoOUT \
				CONCAT4E(rfbTranslateWithSingleTable,IN,to,OUT)
#define rfbTranslateWithRGBTablesINtoOUT \
				CONCAT4E(rfbTranslateWithRGBTables,IN,to,OUT)
#define rfbTranslateRawVSCWithSingleTableINtoOUT \
				CONCAT4E(rfbTranslateRawVSCWithSingleTable,IN,to,OUT)
#define rfbTranslateRawVSCWithRGBTablesINtoOUT \
				CONCAT4E(rfbTranslateRawVSCWithRGBTables,IN,to,OUT)

/* HextileContinuousMemoryLayout (HCML) mode:
   Framebuffer is not a linear one, but contains hextiles ordered
   consecutively, means it starts with PP_FB_TILE_WIDTH * PP_FB_TILE_HEIGHT
   pixels of the first hextile, after this the second hextile on the first row
   follows... . Framebuffer is padded in both x and y direction, means we
   only have full hextiles in the buffer and have to possibly skip a part
   while reading.
   Translation of a single complete hextile is the trivial special case.
   Other translations linearize the framebuffer content while reading, so
   we skip around after tile lines, whole lines and hextile rows:
   TILESKIP   fixed skip after reading a single tile line to the beginning
	      of the same tile line of the next tile
   eol_skip   for horizontally incomplete tiles to get tile aligned again
   line_skip  after reading a single line from all tiles, skip to the next
   row_skip   PP_FB_TILE_HEIGHT lines completed, jump to the next tile row
*/
   
#define TILESKIP (PP_FB_TILE_WIDTH * PP_FB_TILE_HEIGHT - PP_FB_TILE_WIDTH)

/*
 * rfbTranslateWithSingleTableINtoOUT translates a rectangle of pixel data
 * using a single lookup table.
 */

#if (IN == 24)

static void
rfbTranslateWithSingleTableINtoOUT (char * table,
				    rfbPixelFormat * in UNUSED,
				    rfbPixelFormat * out UNUSED,
				    char * iptr, char * optr,
				    u_char tile_pitch,
				    int width, int height)
{
    OUT_T *op = (OUT_T *)optr;
    OUT_T *t = (OUT_T *)table;

    if (width != PP_FB_TILE_WIDTH || height != PP_FB_TILE_HEIGHT) {

	int eol_skip  = (width % PP_FB_TILE_WIDTH) ? width - (width % PP_FB_TILE_WIDTH) : 0;
	int line_skip = -(width * PP_FB_TILE_HEIGHT - PP_FB_TILE_WIDTH);
	int row_skip  = (tile_pitch - 1) * PP_FB_TILE_SIZE;
	int w_single, line=0;
	
	while (line++ < height) {
	    w_single = width;
	    
	    while (w_single-- > 0) {
#if (__BYTE_ORDER == __BIG_ENDIAN)
		u_int32_t i = bswap_32(*(u_int32_t *)iptr) & 0xFFFFFF;
#else
		u_int32_t i = bswap_32(*(u_int32_t *)iptr) >> 8;
#endif
		*(op++) = t[i];
		iptr += 3;
		
		if (w_single % PP_FB_TILE_WIDTH == 0) iptr += TILESKIP * 3;
	    }

	    /* skip partial rect at end of line */
	    iptr += eol_skip * 3;
	    /* start next pixel line */
	    iptr += line_skip * 3;
	    
	    if (line % PP_FB_TILE_HEIGHT == 0) iptr += row_skip * 3;
	}
    } else {
	int count = PP_FB_TILE_WIDTH * PP_FB_TILE_HEIGHT;
	
	while (count-- > 0) {
#if (__BYTE_ORDER == __BIG_ENDIAN)
	    u_int32_t i = bswap_32(*(u_int32_t *)iptr) & 0xFFFFFF;
#else
	    u_int32_t i = bswap_32(*(u_int32_t *)iptr) >> 8;
#endif
	    *(op++) = t[i];
	    iptr += 3;
	}
    }
}

static void
rfbTranslateRawVSCWithSingleTableINtoOUT (char * table,
				          rfbPixelFormat * in UNUSED,
				          rfbPixelFormat * out UNUSED,
				          char * iptr, char * optr,
					  u_char tile_pitch,
				          int width, int height)
{
    OUT_T * t = (OUT_T *) table;
    OUT_T *op = (OUT_T *) optr;
    
    while (height > 0) {
    	int x = 0;
    	int pixelsToCopy = min(height, PP_FB_TILE_HEIGHT) * width;
    	while(x < width * PP_FB_TILE_HEIGHT) {
#if (__BYTE_ORDER == __BIG_ENDIAN)
	    u_int32_t i = bswap_32(*(u_int32_t *)iptr + 3 * x) & 0xFFFFFF;
#else
	    u_int32_t i = bswap_32(*(u_int32_t *)iptr + 3 * x) >> 8;
#endif
	    *(op + x) = t[i];
	    x++;
    	}
	iptr += tile_pitch * PP_FB_TILE_SIZE * 3;
	op += pixelsToCopy;
	height -= PP_FB_TILE_HEIGHT;
    }
}

#else /* IN != 24 */

static void
rfbTranslateWithSingleTableINtoOUT (char * table,
				    rfbPixelFormat * in UNUSED,
				    rfbPixelFormat * out UNUSED,
				    char * iptr, char * optr,
				    u_char tile_pitch,
				    int width, int height)
{
    IN_T *ip = (IN_T *)iptr;
    OUT_T *op = (OUT_T *)optr;
    OUT_T *t = (OUT_T *)table;

    if (width != PP_FB_TILE_WIDTH || height != PP_FB_TILE_HEIGHT) {
	int eol_skip  = (width % PP_FB_TILE_WIDTH) ? width - (width % PP_FB_TILE_WIDTH) : 0;
	int line_skip = -(width * PP_FB_TILE_HEIGHT - PP_FB_TILE_WIDTH);
	int row_skip  = (tile_pitch - 1) * PP_FB_TILE_SIZE;
	int w_single, line=0;

	while (line++ < height) {
	    w_single = width;
	    
	    while (w_single-- > 0) {
		*(op++) = t[*ip++];
		//*(op++) = t[SwapIN(*ip++)];
		if (w_single % PP_FB_TILE_WIDTH == 0) ip += TILESKIP;
	    }

	    /* skip partial rect at end of line */
	    ip += eol_skip;   
	    /* start next pixel line */
	    ip += line_skip;
	    
	    if (line % PP_FB_TILE_HEIGHT == 0) ip += row_skip;
	}
    } else {
	int count = PP_FB_TILE_WIDTH * PP_FB_TILE_HEIGHT;
	
	while (count-- > 0) {
	    *(op++) = t[*ip++];
	    //*(op++) = t[SwapIN(*ip++)];
	}
    }
}

static void
rfbTranslateRawVSCWithSingleTableINtoOUT (char * table,
				          rfbPixelFormat * in UNUSED,
				          rfbPixelFormat * out UNUSED,
				          char * iptr, char * optr,
					  u_char tile_pitch,
				          int width, int height)
{
    OUT_T * t = (OUT_T *) table;
    OUT_T *op = (OUT_T *) optr;
    IN_T * ip = (IN_T *) iptr;
   
    while (height > 0) {
    	int x = 0;
    	int pixelsToCopy = min(height, PP_FB_TILE_HEIGHT) * width;
    	while(x < width * PP_FB_TILE_HEIGHT) {
	    *(op + x) = t[*(ip + x)];
	    x++;
    	}	
	ip += tile_pitch * PP_FB_TILE_SIZE;
	op += pixelsToCopy;
	height -= PP_FB_TILE_HEIGHT;
    }
}

#endif /* IN != 24 */

/*
 * rfbTranslateWithRGBTablesINtoOUT translates a rectangle of pixel data
 * using three separate lookup tables for the red, green and blue values.
 */

#if (IN == 24)

static void
rfbTranslateWithRGBTablesINtoOUT (char * table, rfbPixelFormat * in,
				  rfbPixelFormat * out UNUSED,
				  char * iptr, char * optr,
				  u_char tile_pitch,
				  int width, int height)
{
    OUT_T *op = (OUT_T *)optr;
    OUT_T *redTable = (OUT_T *)table;
    OUT_T *greenTable = redTable + in->redMax_be16 + 1;
    OUT_T *blueTable = greenTable + in->greenMax_be16 + 1;

    if (width != PP_FB_TILE_WIDTH || height != PP_FB_TILE_HEIGHT) {
	
	int eol_skip  = (width % PP_FB_TILE_WIDTH) ? width - (width % PP_FB_TILE_WIDTH) : 0;
	int line_skip = -(width * PP_FB_TILE_HEIGHT - PP_FB_TILE_WIDTH);
	int row_skip  = (tile_pitch - 1) * PP_FB_TILE_SIZE;
	int w_single, line=0;

	while (line++ < height) {
	    w_single = width;
	    
	    while (w_single-- > 0) {
#if (__BYTE_ORDER == __BIG_ENDIAN)
		u_int32_t i = bswap_32(*(u_int32_t *)iptr) & 0xFFFFFF;
#else
		u_int32_t i = bswap_32(*(u_int32_t *)iptr) >> 8;
#endif
		*(op++) = (redTable[(i >> in->redShift_8) & in->redMax_be16] |
			   greenTable[(i >> in->greenShift_8) & in->greenMax_be16] |
		           blueTable[(i >> in->blueShift_8) & in->blueMax_be16]);

iptr += 3;
		
		if (w_single % PP_FB_TILE_WIDTH == 0) iptr += TILESKIP * 3;
	    }

	    /* skip partial rect at end of line */
	    iptr += eol_skip * 3;
	    /* start next pixel line */
	    iptr += line_skip * 3;
	    
	    if (line % PP_FB_TILE_HEIGHT == 0) iptr += row_skip * 3;
	}
    } else {
	int count = PP_FB_TILE_WIDTH * PP_FB_TILE_HEIGHT;
	
	while (count-- > 0) {
#if (__BYTE_ORDER == __BIG_ENDIAN)
	    u_int32_t i = bswap_32(*(u_int32_t *)iptr) & 0xFFFFFF;
#else
	    u_int32_t i = bswap_32(*(u_int32_t *)iptr) >> 8;
#endif
	    *(op++) = (redTable[(i >> in->redShift_8) & in->redMax_be16] |
		       greenTable[(i >> in->greenShift_8) & in->greenMax_be16] |
		       blueTable[(i >> in->blueShift_8) & in->blueMax_be16]);

	    iptr += 3;
	}
    }
}

static void
rfbTranslateRawVSCWithRGBTablesINtoOUT (char * table,
					rfbPixelFormat * in UNUSED,
					rfbPixelFormat * out UNUSED,
					char * iptr, char * optr,
					u_char tile_pitch,
					int width, int height)
{
    OUT_T *redTable = (OUT_T *)table;
    OUT_T *greenTable = redTable + in->redMax_be16 + 1;
    OUT_T *blueTable = greenTable + in->greenMax_be16 + 1;
    OUT_T *op = (OUT_T *) optr;
    
    while (height > 0) {
    	int x = 0;
    	int pixelsToCopy = min(height, PP_FB_TILE_HEIGHT) * width;
    	while(x < width * PP_FB_TILE_HEIGHT) {
#if (__BYTE_ORDER == __BIG_ENDIAN)
	    u_int32_t i = bswap_32(*(u_int32_t *)iptr + 3 * x) & 0xFFFFFF;
#else
	    u_int32_t i = bswap_32(*(u_int32_t *)iptr + 3 * x) >> 8;
#endif
	    *(op + x) = (redTable[(i >> in->redShift_8) & in->redMax_be16] |
		        greenTable[(i >> in->greenShift_8) & in->greenMax_be16] |
		        blueTable[(i >> in->blueShift_8) & in->blueMax_be16]);
	    x++;
    	}
	iptr += tile_pitch * PP_FB_TILE_SIZE * 3;
	op += pixelsToCopy;
	height -= PP_FB_TILE_HEIGHT;
    }
}

#else /* IN != 24 */

static void
rfbTranslateWithRGBTablesINtoOUT (char * table, rfbPixelFormat * in,
				  rfbPixelFormat * out UNUSED,
				  char * iptr, char * optr,
				  u_char tile_pitch,
				  int width, int height)
{
    IN_T *ip = (IN_T *)iptr;
    OUT_T *op = (OUT_T *)optr;
    OUT_T *redTable = (OUT_T *)table;
    OUT_T *greenTable = redTable + in->redMax_be16 + 1;
    OUT_T *blueTable = greenTable + in->greenMax_be16 + 1;

    if (width != PP_FB_TILE_WIDTH || height != PP_FB_TILE_HEIGHT) {

	int eol_skip  = (width % PP_FB_TILE_WIDTH) ? width - (width % PP_FB_TILE_WIDTH) : 0;
	int line_skip = -(width * PP_FB_TILE_HEIGHT - PP_FB_TILE_WIDTH);
	int row_skip  = (tile_pitch - 1) * PP_FB_TILE_SIZE;
	int w_single, line=0;

	while (line++ < height) {
	    w_single = width;
	    
	    while (w_single-- > 0) {
		IN_T i = *ip++;
		//IN_T i = SwapIN(*ip++);

		*(op++) = (redTable[(i >> in->redShift_8) & in->redMax_be16] |
			   greenTable[(i >> in->greenShift_8) & in->greenMax_be16] |
			   blueTable[(i >> in->blueShift_8) & in->blueMax_be16]);
		
		if (w_single % PP_FB_TILE_WIDTH == 0) ip += TILESKIP;
	    }

	    /* skip partial rect at end of line */
	    ip += eol_skip;   
	    /* start next pixel line */
	    ip += line_skip;
	    
	    if (line % PP_FB_TILE_HEIGHT == 0) ip += row_skip;
	}
    } else {
	int count = PP_FB_TILE_WIDTH * PP_FB_TILE_HEIGHT;
	
	while (count-- > 0) {
	    IN_T i = *ip++;
	    //IN_T i = SwapIN(*ip++);
	    
	    *(op++) = (redTable[(i >> in->redShift_8) & in->redMax_be16] |
		       greenTable[(i >> in->greenShift_8) & in->greenMax_be16] |
		       blueTable[(i >> in->blueShift_8) & in->blueMax_be16]);
	}
    }
}

static void
rfbTranslateRawVSCWithRGBTablesINtoOUT (char * table,
					rfbPixelFormat * in,
					rfbPixelFormat * out UNUSED,
					char * iptr, char * optr,
					u_char tile_pitch,
					int width, int height)
{
    OUT_T *redTable = (OUT_T *)table;
    OUT_T *greenTable = redTable + in->redMax_be16 + 1;
    OUT_T *blueTable = greenTable + in->greenMax_be16 + 1;
    OUT_T *op = (OUT_T *) optr;
    IN_T * ip = (IN_T *) iptr;
    
    while (height > 0) {
    	int x = 0;
    	int pixelsToCopy = min(height, PP_FB_TILE_HEIGHT) * width;
    	while(x < width * PP_FB_TILE_HEIGHT) {
	    IN_T i = *(ip + x);
	    *(op + x) = (redTable[(i >> in->redShift_8) & in->redMax_be16] |
		        greenTable[(i >> in->greenShift_8) & in->greenMax_be16] |
		        blueTable[(i >> in->blueShift_8) & in->blueMax_be16]);
	    x++;
    	}
	ip += tile_pitch * PP_FB_TILE_SIZE;
	op += pixelsToCopy;
	height -= PP_FB_TILE_HEIGHT;
    }
}

#endif /* IN != 24 */

#undef IN_T
#undef OUT_T
#undef rfbTranslateWithSingleTableINtoOUT
#undef rfbTranslateWithRGBTablesINtoOUT
