#include <fcntl.h>
#include <termios.h>
#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/serial.h>
#include "serial_internal.h"
#include "debug.h"

typedef struct {
    const char * string;
    pp_serial_usage_t usage;
} serial_usage_map_entry_t;

serial_usage_map_entry_t serial_usage_map[] = {
    { "disabled", PP_SERIAL_USAGE_DISABLED },
    { "modem", PP_SERIAL_USAGE_MODEM },
    { "configlogin", PP_SERIAL_USAGE_CONFIGLOGIN },
    { "ipmi", PP_SERIAL_USAGE_IPMI },
    { "passthrough", PP_SERIAL_USAGE_PASSTHROUGH },
    { "externalpower", PP_SERIAL_USAGE_EXTERNALPOWER },
};

static int initialized = 0;

pp_hash_i_t * serial_ports_hash = NULL;
serial_port_callbacks_t serial_port_callbacks[PP_SERIAL_USAGE_COUNT];
pthread_mutex_t serial_lock = PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP;

static void free_port(void * _sp);
static serial_port_t * alloc_port(const char * device_path, int has_hw_flowctrl, u_int id);
static int serial_modem_uninit(const char * device_path, u_int id, u_int fd, char ** errstr_p);
static int serial_configlogin_uninit(const char * device_path, u_int id, u_int fd, char ** errstr_p);

int
pp_serial_init(void)
{
    if (!initialized) {
	if ((serial_ports_hash = pp_hash_create_i(17)) == NULL) {
	    pp_log("%s(): pp_hash_create_i() failed\n", ___F);
	    return PP_ERR;
	}
	memset(&serial_port_callbacks, 0, sizeof(serial_port_callbacks));
	pp_serial_register_port_callbacks(PP_SERIAL_USAGE_MODEM, NULL, serial_modem_uninit);
	pp_serial_register_port_callbacks(PP_SERIAL_USAGE_CONFIGLOGIN, NULL, serial_configlogin_uninit);
	serial_reconf_init();
	initialized = 1;
    }
    return PP_SUC;
}

void
pp_serial_cleanup(void)
{
    MUTEX_LOCK(&serial_lock);
    pp_serial_unregister_port_callbacks(PP_SERIAL_USAGE_MODEM);
    pp_serial_unregister_port_callbacks(PP_SERIAL_USAGE_CONFIGLOGIN);
    if (serial_ports_hash) {
	pp_hash_delete(serial_ports_hash);
	serial_ports_hash = NULL;
    }
    serial_reconf_cleanup();
    initialized = 0;
    MUTEX_UNLOCK(&serial_lock);
}

int
pp_serial_register_port(const char * device_path, int has_hw_flowctrl, u_int id)
{
    serial_port_t * sp = alloc_port(device_path, has_hw_flowctrl, id);

    if (sp) {
	MUTEX_LOCK(&serial_lock);
	pp_hash_set_entry_i(serial_ports_hash, id, sp, free_port);
	MUTEX_UNLOCK(&serial_lock);
	return PP_SUC;
    }

    pp_log("registering port %s failed\n", device_path);
    
    return PP_ERR;
}

void
pp_serial_unregister_port(u_int id)
{
    MUTEX_LOCK(&serial_lock);
    pp_hash_delete_entry_i(serial_ports_hash, id);
    MUTEX_UNLOCK(&serial_lock);
}

void
pp_serial_register_port_callbacks(pp_serial_usage_t usage, pp_serial_init_func_t init, pp_serial_uninit_func_t uninit)
{
    MUTEX_LOCK(&serial_lock);
    if (usage < PP_SERIAL_USAGE_COUNT) {
	serial_port_callbacks[usage].init = init;
	serial_port_callbacks[usage].uninit = uninit;
    }
    MUTEX_UNLOCK(&serial_lock);
}

void
pp_serial_unregister_port_callbacks(pp_serial_usage_t usage)
{
    MUTEX_LOCK(&serial_lock);
    if (usage < PP_SERIAL_USAGE_COUNT) {
	serial_port_callbacks[usage].init = NULL;
	serial_port_callbacks[usage].uninit = NULL;
    }
    MUTEX_UNLOCK(&serial_lock);
}

int
pp_serial_port_get(u_int id)
{
    int fd = -1;
    serial_port_t * sp;

    MUTEX_LOCK(&serial_lock);
    sp = (serial_port_t *)pp_hash_get_entry_i(serial_ports_hash, id);
    if (sp) fd = sp->fd;
    MUTEX_UNLOCK(&serial_lock);
    return fd;
}

pp_serial_usage_t
str_to_usage(const char * str)
{
    if (str) {
	size_t i, map_entry_cnt = sizeof(serial_usage_map) / sizeof(serial_usage_map_entry_t);
	for (i = 0; i < map_entry_cnt; ++i) {
	    if (!strcmp(str, serial_usage_map[i].string)) {
		return serial_usage_map[i].usage;
	    }
	}
    }

    return PP_SERIAL_USAGE_DISABLED;
}

static void
free_port(void * _sp)
{
    serial_port_t * sp = (serial_port_t *)_sp;

    assert(sp);

    if (serial_port_callbacks[sp->usage].uninit) {
	if (PP_FAILED(serial_port_callbacks[sp->usage].uninit(sp->device_path, sp->id, sp->fd, NULL))) {
	    pp_log("%s(): uninit failed (usage=%u)\n", ___F, sp->usage);
	}
    }
    if (sp->fd >= 0) close(sp->fd);
    free(sp->device_path);
    free(sp);
}


static serial_port_t *
alloc_port(const char * device_path, int has_hw_flowctrl, u_int id)
{
    serial_port_t * sp;
    struct termios tty_data; 

    int n;

    assert(device_path);

    sp = calloc(1, sizeof(serial_port_t));
    sp->id = id;
    sp->usage = PP_SERIAL_USAGE_DISABLED;
    sp->device_path = strdup(device_path);
    if ((sp->fd = open(device_path, O_RDWR | O_NDELAY)) < 0
	|| (n = fcntl(sp->fd, F_GETFL, 0)) < 0
	|| fcntl(sp->fd, F_SETFL, n & ~O_NDELAY) < 0) {
	pp_log_err("%s(): open() or fcntl() failed", ___F);
	free_port(sp);
	return NULL;
    }

    if (tcgetattr(sp->fd, &tty_data) == 0) {	
	/* we do not want to echo input on disabled serial ports, so switch off canonical mode and echo */
	tty_data.c_lflag &= ~(ICANON | ECHO);
	tcsetattr(sp->fd, TCSANOW, &tty_data);
    }

    sp->has_hw_flowctrl = has_hw_flowctrl;
    sp->need_reconfigure = 1;
    return sp;
}

static int
serial_modem_uninit(const char * device_path, u_int id UNUSED, u_int fd UNUSED, char ** errstr_p UNUSED)
{
    char cmdline[128];
    snprintf(cmdline, sizeof(cmdline), "kill `fuser %s | pid_name_filter.sh mgetty` > /dev/null 2>&1", device_path);
    pp_system(cmdline);
    sleep(1);
    return PP_SUC;
}

static int
serial_configlogin_uninit(const char * device_path, u_int id UNUSED, u_int fd UNUSED, char ** errstr_p UNUSED)
{
    char cmdline[128];
    snprintf(cmdline, sizeof(cmdline), "kill `fuser %s | pid_name_filter.sh getty login` > /dev/null 2>&1", device_path);
    pp_system(cmdline);
    sleep(1);
    return PP_SUC;
}

