#include <stdlib.h>
#include <unistd.h>
#include <setup_proto_internal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <scsi/sg.h>
#include <scsi/scsi_ioctl.h>

static int initialized = 0;
static int sock_fd = -1;

int
pp_setup_proto_init()
{
    const char * fn = ___F;
    int one = 1;    
    struct sockaddr_in sa;

    if (initialized) return 0;

    _pp_setup_proto_register_errnos();

    /* create socket */
    if ((sock_fd = socket(PF_INET, SOCK_DGRAM, 0)) < 0) {
	pp_log_err("%s(): socket(PF_INET) failed", fn);
	return -1;
    }

    /* allow broadcasting */
    if (setsockopt(sock_fd, SOL_SOCKET, SO_BROADCAST, &one, sizeof(one)) < 0) {
	pp_log_err("%s(): setsockopt(SO_BROADCAST) failed", fn);
	return -1;
    }

    /* bind socket */
    memset(&sa, 0, sizeof(sa));
    sa.sin_family = AF_INET;
    sa.sin_addr.s_addr = htonl(INADDR_ANY);
    sa.sin_port = htons(PP_SP_CLI_PORT);
    if (bind(sock_fd, (struct sockaddr*)&sa, sizeof(sa)) < 0) {
	pp_log_err("%s(): bind() failed", fn);
	return -1;
    }
    return 0;
}

void
pp_setup_proto_cleanup()
{
    if (sock_fd != -1) {
	close(sock_fd);
	sock_fd = -1;
    }
    _pp_setup_proto_unregister_errnos();
    initialized = 0;
}

int
pp_setup_proto_req_send(pp_setup_req_t * req)
{
    char * pdu = NULL;
    setup_req_hdr_t * hdr;
    char * payload;
    ssize_t pdu_len;
    struct sockaddr_in sa;
    int ret = -1;

    /* allocate pdu */
    if ((pdu = malloc(PP_SP_REQ_MTU)) == NULL) goto bail;

    hdr = (setup_req_hdr_t *)pdu;
    payload = pdu + sizeof(setup_req_hdr_t);

    /* write payload into pdu */
    if ((pdu_len = pp_prop_write(&req->props, payload,
				 PP_SP_REQ_MTU - sizeof(setup_req_hdr_t))) < 0) {
	goto bail;
    }

    /* write header into pdu */
    hdr->magic = htonl(PP_SP_MAGIC);
    hdr->version = PP_SP_VERSION;
    hdr->command = req->command;
    hdr->id = htonl(req->id);
    memcpy(hdr->dev_id, req->dev_id, sizeof(hdr->dev_id));
    pdu_len += sizeof(setup_req_hdr_t);

    /* send pdu */
    memset(&sa, 0, sizeof(sa));
    sa.sin_family = AF_INET;
    sa.sin_addr.s_addr = htonl(INADDR_BROADCAST);
    sa.sin_port = htons(PP_SP_SRV_PORT);
    if (sendto(sock_fd, pdu, pdu_len, 0,
	       (struct sockaddr *)&sa, sizeof(sa)) < 0) {
	pp_log_err("%s(): sendto() failed", ___F);
	goto bail;
    }

    ret = 0;

 bail:
    free(pdu);
    return ret;
}

pp_setup_rsp_t *
pp_setup_proto_rsp_recv(int timeout, int * timedout)
{
    const char * fn = ___F;
    pp_setup_rsp_t * rsp = NULL;
    char * pdu = NULL;
    ssize_t pdu_len;
    struct sockaddr_in sa;
    socklen_t sa_len = sizeof(sa);
   
    if (timedout) *timedout = 0;

    /* allocate pdu */
    if ((pdu = malloc(PP_SP_RSP_MTU)) == NULL) return NULL;

    /* set receive timeout */
    if (timeout > 0) {
	struct timeval tv;
	tv.tv_sec = 0;
	tv.tv_usec = timeout;
	if (setsockopt(sock_fd, SOL_SOCKET, SO_RCVTIMEO,
			       (void *)&tv, sizeof(tv)) == -1) {
            pp_log_err("%s(): setsockopt(SO_RCVTIMEO) failed", fn);	    
        }
    }
    
    /* receive pdu */
    if ((pdu_len = recvfrom(sock_fd, pdu, PP_SP_RSP_MTU, 0,
			    (struct sockaddr *)&sa, &sa_len)) < 0) {	   	    
	if (errno == EAGAIN && timedout) {
	    *timedout = 1;
	} else {
	    pp_log_err("%s(): recvfrom() failed", fn);
	}
	goto bail;

    }

    rsp = pp_setup_proto_rsp_create(pdu, pdu_len);

 bail:
    free(pdu);
    return rsp;
}

static unsigned char scsi_request[] = { 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00 };

static int
scsi_send_request(int fd, pp_setup_req_t *req)
{
    char *payload, *pdu;
    setup_req_hdr_t *hdr;
    int pdu_len;
    int ret = PP_ERR;

    unsigned char sense_buffer[32];
    struct sg_io_hdr io_hdr;

    pdu = malloc(PP_SP_REQ_MTU);
    if (!pdu) goto bail;
    hdr = (setup_req_hdr_t *)pdu;
    payload = pdu + sizeof(setup_req_hdr_t);

    if ((pdu_len = pp_prop_write(&req->props, payload,
		PP_SP_REQ_MTU - sizeof(setup_req_hdr_t))) < 0) goto bail;

    hdr->magic = htonl(PP_SP_MAGIC);
    hdr->version = PP_SP_VERSION;
    hdr->command = req->command;
    hdr->id = htonl(req->id);
    memcpy(hdr->dev_id, req->dev_id, sizeof(hdr->dev_id));
    pdu_len += sizeof(setup_req_hdr_t);

    memset(&io_hdr, 0, sizeof(struct sg_io_hdr));
    memset(sense_buffer, 0, sizeof(sense_buffer));

    io_hdr.interface_id = 'S';
    io_hdr.cmd_len = sizeof(scsi_request);
    io_hdr.mx_sb_len = sizeof(sense_buffer);
    io_hdr.dxfer_direction = SG_DXFER_TO_DEV;
    io_hdr.dxfer_len = pdu_len;
    io_hdr.dxferp = pdu;
    io_hdr.cmdp = scsi_request;
    io_hdr.sbp = sense_buffer;
    io_hdr.timeout = 2000;

    ioctl(fd, SG_IO, &io_hdr);
    ret = PP_SUC;

bail:
    free(pdu);
    return ret;
}

static unsigned char scsi_response[] = { 0xc1, 0x00, 0x00, 0x00, 0x00, 0x00 };

static pp_setup_rsp_t *
scsi_get_response(int fd, int retries)
{
    unsigned char sense_buffer[32];
    struct sg_io_hdr io_hdr;
    char *pdu;
    pp_setup_rsp_t *rsp = NULL;

    pdu = malloc(PP_SP_REQ_MTU);
    if (!pdu) return NULL;

    while (retries--) {
	memset(&io_hdr, 0, sizeof(struct sg_io_hdr));
	memset(sense_buffer, 0, sizeof(sense_buffer));

	io_hdr.interface_id = 'S';
	io_hdr.cmd_len = sizeof(scsi_response);
	io_hdr.mx_sb_len = sizeof(sense_buffer);
	io_hdr.dxfer_direction = SG_DXFER_FROM_DEV;
	io_hdr.dxfer_len = PP_SP_RSP_MTU;
	io_hdr.dxferp = pdu;
	io_hdr.cmdp = scsi_response;
	io_hdr.sbp = sense_buffer;
	io_hdr.timeout = 2000;

	ioctl(fd, SG_IO, &io_hdr);
	if (io_hdr.resid < (int)io_hdr.dxfer_len) {
	    rsp = pp_setup_proto_rsp_create(pdu, io_hdr.dxfer_len);
	    goto bail;
	}
	sleep(1);
    }

bail:
    free(pdu);
    return rsp;
}

pp_setup_rsp_t *
pp_setup_proto_scsi_request_response(int fd, pp_setup_req_t *req, int retries)
{
    if (PP_FAILED( scsi_send_request(fd, req) )) return NULL;
    return scsi_get_response(fd, retries);
}

