#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <setup_proto_internal.h>
#include <pp/intl.h>

/* ------------------------------------------------------------------------- *
 * error stuff
 * ------------------------------------------------------------------------- */

static const char * sp_errors[] = {
    /*00*/ N_("An unknown error occured."),
    /*01*/ N_("Permission denied."),
    /*02*/ N_("Authentication failed."),
    /*03*/ N_("An parameter error occured."),
};

int pp_setup_proto_errno_base = 0;

u_int8_t broadcast_mac[6] = { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };

static const char*
get_error_string(int error) {
    return sp_errors[error - pp_setup_proto_errno_base];
}

void
_pp_setup_proto_register_errnos()
{
    if (!(pp_setup_proto_errno_base > 0)) {
        // register errnos for myself, acl and profile
        pp_setup_proto_errno_base = pp_register_errnos(sizeof(sp_errors)/
						       sizeof(*sp_errors),
						       get_error_string);
    }
}

void
_pp_setup_proto_unregister_errnos()
{
    if (pp_setup_proto_errno_base > 0) {
	pp_unregister_errnos(pp_setup_proto_errno_base);
	pp_setup_proto_errno_base = 0;
    }
}

/* ------------------------------------------------------------------------- *
 * code dealing with requests
 * ------------------------------------------------------------------------- */

pp_setup_req_t *
pp_setup_proto_req_create(const char * if_name, void * data, size_t len,
			  u_int8_t * src_hw_addr, in_addr_t src_in_addr)
{
    setup_req_hdr_t * setup_req_hdr = NULL;
    pp_setup_req_t * req;
    u_int8_t own_mac[6];

    /* check data if not NULL */
    if (data) {

	/* check length */
	if (len < sizeof(setup_req_hdr_t)) {
	    /* FIXME: error handling */
	    return NULL;
	}

	setup_req_hdr = (setup_req_hdr_t *)data;

	/* check magic */
	if (ntohl(setup_req_hdr->magic) != PP_SP_MAGIC) {
	    /* FIXME: error handling */
	    return NULL;
	}

	/* check version */
	if (setup_req_hdr->version > PP_SP_VERSION) {
	    /* FIXME: error handling */
	    return NULL;
	}

	/*
	 * Check if this packet is for us.
	 * NOTE: If no if_name is specified we assume it is for us.
	 */
	if (if_name) {
	    if (pp_get_mac(if_name, own_mac) == -1) return NULL;
	    if (memcmp(setup_req_hdr->dev_id, own_mac, sizeof(setup_req_hdr->dev_id))) {
		/* pings may use the broadcast MAC to ping all erlas at once */
		if (memcmp(setup_req_hdr->dev_id, broadcast_mac, sizeof(setup_req_hdr->dev_id))) return NULL;
		if (setup_req_hdr->command != PP_SP_CMD_PING) return NULL;
	    }
	}
    }

    /* allocate the request structure */
    if ((req = (pp_setup_req_t *)malloc(sizeof(pp_setup_req_t))) == NULL) {
	/* FIXME: error handling */
	return NULL;
    }

    /* initialize the request structure */
    memset(req, 0, sizeof(pp_setup_req_t));
    req->src_in_addr = src_in_addr;
    if (src_hw_addr) memcpy(req->src_hw_addr, src_hw_addr, sizeof(req->src_hw_addr));
    memset(req->dev_id, 0xff, sizeof(req->dev_id));
    if (pp_prop_new(&req->props, 20) == NULL) {
	/* FIXME: error handling */
	free(req);
	return NULL;
    }

    if (data) {
	ssize_t payload_len = (ssize_t)len - sizeof(setup_req_hdr_t);
	memcpy(req->dev_id, setup_req_hdr->dev_id, sizeof(req->dev_id));
	req->command = setup_req_hdr->command;
	req->id = ntohl(setup_req_hdr->id);
	if (payload_len < 0
	    || pp_prop_read(&req->props,
			    data + sizeof(setup_req_hdr_t),
			    payload_len) < 0) {
	    pp_prop_delete(&req->props);
	    free(req);
	    req = NULL;
	}
    }

    return req;
}

void
pp_setup_proto_req_free(pp_setup_req_t * req)
{
    if (req) {
	pp_prop_delete(&req->props);
	free(req);
    }
}

int
pp_setup_proto_req_set_property(pp_setup_req_t * req, const char * key, const char * val)
{
    if (!key || !val) return -1;

    return pp_prop_set(&req->props, key, strdup(val));
}

const char *
pp_setup_proto_req_get_property(pp_setup_req_t * req, const char * key)
{
    if (!key) return NULL;

    return pp_prop_get(&req->props, key);
}

/* ------------------------------------------------------------------------- *
 * code dealing with responses
 * ------------------------------------------------------------------------- */

pp_setup_rsp_t *
pp_setup_proto_rsp_create(void * data, size_t len)
{
    setup_rsp_hdr_t * setup_rsp_hdr = NULL;
    pp_setup_rsp_t * rsp;

    /* check data if not NULL */
    if (data) {

	/* check length */
	if (len < sizeof(setup_rsp_hdr_t)) {
	    /* FIXME: error handling */
	    return NULL;
	}

	setup_rsp_hdr = (setup_rsp_hdr_t *)data;

	/* check magic */
	if (ntohl(setup_rsp_hdr->magic) != PP_SP_MAGIC) {
	    /* FIXME: error handling */
	    return NULL;
	}

	/* check version */
	if (setup_rsp_hdr->version > PP_SP_VERSION) {
	    /* FIXME: error handling */
	    return NULL;
	}
    }

    /* allocate the response structure */
    if ((rsp = (pp_setup_rsp_t *)malloc(sizeof(pp_setup_rsp_t))) == NULL) {
	/* FIXME: error handling */
	return NULL;
    }

    /* initialize the response structure */
    memset(rsp, 0, sizeof(pp_setup_rsp_t));
    memset(rsp->dev_id, 0xff, sizeof(rsp->dev_id));
    if (pp_prop_new(&rsp->props, 20) == NULL) {
	/* FIXME: error handling */
	free(rsp);
	return NULL;
    }

    if (data) {
	ssize_t payload_len = (ssize_t)len - sizeof(setup_rsp_hdr_t);
	memcpy(rsp->dev_id, setup_rsp_hdr->dev_id, sizeof(rsp->dev_id));
	rsp->id = ntohl(setup_rsp_hdr->id);
	rsp->response_code = setup_rsp_hdr->response_code;
	if (payload_len < 0 || pp_prop_read(&rsp->props,
					    data + sizeof(setup_rsp_hdr_t),
					    payload_len) < 0) {
	    pp_prop_delete(&rsp->props);
	    free(rsp);
	    rsp = NULL;
	}
    }

    return rsp;
}

void
pp_setup_proto_rsp_free(pp_setup_rsp_t * rsp)
{
    if (rsp) {
	pp_prop_delete(&rsp->props);
	free(rsp);
    }
}

int
pp_setup_proto_rsp_set_property(pp_setup_rsp_t * rsp, const char * key, const char * val)
{
    if (!key || !val) return -1;

    return pp_prop_set(&rsp->props, key, strdup(val));
}

const char *
pp_setup_proto_rsp_get_property(pp_setup_rsp_t * rsp, const char * key)
{
    if (!key) return NULL;

    return pp_prop_get(&rsp->props, key);
}
