/**
 * User manangement routines
 * This library implements RAASIP compatible usermanagement and replaces
 * the old libpp_um (that was based on permission inheritance).
 * 
 * All functions in this library operate (unless otherwise noted)
 * on the config system. That means they operate either on the local
 * flashdisk (fully writable) or on a remote ldap server (mostly
 * readonly).
 *
 * (c) 2006 Peppercon AG, geo@peppercon.de, tweb@peppercon.de
 */

// system includes
#include <openssl/evp.h>

#ifdef PP_FEAT_RPCCFG
# include <pp/rpc_ripc.h>
#endif
// fw includes
#include <pp/um.h>
#include <pp/intl.h>

// local includes
#include "um_intern.h"
#include "um_cfg_keys.h"
 

/*
 * Library management
 *********************************
 */

static int errno_base = 0; // triggers assertion failed if not registered
inline int pp_um_errno_base() { assert(errno_base > 0); return errno_base; }

static const char * pp_um_errors[] = {
    /* 0*/ N_("An internal error occured."),
    /* 1*/ N_("Authentication failed."),
    /* 2*/ N_("The user is blocked."),
    /* 3*/ N_("The user name is invalid."),
    /* 4*/ N_("The group name is invalid."),
    /* 5*/ N_("This operation cannot be performed for builtin users."),
    /* 6*/ N_("This operation cannot be performed for builtin groups."),
    /* 7*/ N_("The ACL object doesn't exist."),
    /* 8*/ N_("The user already exists."),
    /* 9*/ N_("The group already exists."),
    /*10*/ N_("The user doesn't exist."),
    /*11*/ N_("The group doesn't exist."),
    /*12*/ N_("The group is individual."),
    /*13*/ N_("The device is managed by CC-SG."),
    /*14*/ N_("No plaintext password."),
    /*15*/ N_("The password didn't change."),
    /*16*/ N_("Maximum user count reached, cannot create new user."),
    /*17*/ N_("The user is disabled."),
    /*18*/ N_("The IPMI UID is invalid or already taken."),
};
#if 0
/* 6*/ N_("The ACL doesn't exist."),
/* 8*/ N_("This operation cannot be performed for groups."),
/*10*/ N_("This operation cannot be performed if central user management is used."),
/*11*/ N_("Too many users and groups already existing, cannot create a new one."),
/*13*/ N_("Can't write into read-only Profile."),
/*14*/ N_("The local Profile does not exist."),
/*15*/ N_("The local tagged profile already exists."),
/*16*/ N_("The local tagged profile doesn't exist."),
/*18*/ N_("Web access disabled for this user."),
#endif


static const char* get_error_string(const int error) {
    return pp_um_errors[error - errno_base];
}

int pp_um_init() {
    int ret = PP_SUC;
    assert(errno_base == 0);
    
    if (PP_ERR == (ret = pp_acl_init()))
	return ret;
    
    errno_base = pp_register_errnos(sizeof(pp_um_errors) /
                                    sizeof(*pp_um_errors), get_error_string);

    ret |= um_user_init();
    ret |= um_group_init();

    pp_cfg_add_change_listener(um_user_set_pw_change_timestamp_cb,
                               "user[?].pw_hash");
#if defined(PP_FEAT_STRONG_PASSWORDS)
    pp_cfg_add_change_listener(um_user_strong_pw_notify_cb,
                               "security.strong_pw");
    pp_cfg_add_change_listener(um_user_strong_pw_got_enabled_cb,
                               strong_pw_enabled_key);
    pp_cfg_add_change_listener(um_user_strong_pw_cb,
                               "security.strong_pw.length");
    pp_cfg_add_change_listener(um_user_strong_pw_got_enabled_cb,
                               "security.strong_pw.charset");
#endif /* PP_FEAT_STRONG_PASSWORDS */

    return ret;
}

void pp_um_cleanup(void) {
    assert(errno_base != 0);

    pp_cfg_rem_change_listener(um_user_set_pw_change_timestamp_cb,
                               "user[?].pw_hash");
#if defined(PP_FEAT_STRONG_PASSWORDS)
    pp_cfg_rem_change_listener(um_user_strong_pw_notify_cb,
                               "security.strong_pw");
    pp_cfg_rem_change_listener(um_user_strong_pw_got_enabled_cb,
                               strong_pw_enabled_key);
    pp_cfg_rem_change_listener(um_user_strong_pw_cb,
                               "security.strong_pw.length");
    pp_cfg_rem_change_listener(um_user_strong_pw_got_enabled_cb,
                               "security.strong_pw.charset");
#endif /* PP_FEAT_STRONG_PASSWORDS */

    um_group_cleanup();
    um_user_cleanup();

    pp_unregister_errnos(errno_base);
    errno_base = 0;

    pp_acl_cleanup();
}

int pp_um_has_errno(int error) {
    return error >= errno_base &&
        error < (int)(errno_base + sizeof(pp_um_errors) /
		      sizeof(*pp_um_errors));
}

static const u_char * crypt_key_buf =
    "\x90\x2b\xbc\x95\x3d\xbc\x44\x79\xbb\x88\x92\x52\x58\xed\x16\xf5"
    "\x6b\xdf\x1b\x65\x59\x40\x1b\xb9\x86\xc6\x7a\x64\x88\xd6\xb0\x94"
    "\x37\x6f\xb1\x22\x37\x07\xce\xf3\x35\x3b\xab\x0c\x7f\x39\xbf\x4f"
    "\x66\x40\x66\x92\x6e\x0e\x0a\xcf\x14\x23\x48\xfb\xb2\x88\x1d\xbe"
    "\x36\xe8\x43\x69\x49\xe3\x1d\xca\xe1\x83\x63\x35\xd4\x82\xfe\x8c"
    "\xf4\xe0\x65\x8f\xaa\x2e\x81\x6a\xb9\x4d\x43\xa1\x39\x00\x2a\xe2"
    "\x26\x7e\x66\x7c\xc2\xf4\xe1\xe6\x78\xda\xb7\xe9\x22\x29\x4b\x49"
    "\xf2\xd9\xc3\xd8\x20\x58\x68\x7a\xe3\x3f\x44\xb5\x93\xe4\x3a\x14";

static int
um_crypt(char * dest, const char * src, size_t len, int do_encrypt)
{
    static EVP_CIPHER_CTX ctx;
    static const EVP_CIPHER *cipher;
    static u_char key[EVP_MAX_KEY_LENGTH], iv[EVP_MAX_IV_LENGTH];
    static int init = 0;
    int update_len, final_len;

    if (!init) {
	cipher = EVP_aes_128_cfb();
	EVP_BytesToKey(cipher, EVP_md5(), NULL, 
                       crypt_key_buf, strlen(crypt_key_buf), 1, key, iv);
	EVP_CIPHER_CTX_init(&ctx);
	init = 1;
    }

    EVP_CipherInit_ex(&ctx, cipher, NULL, key, iv, do_encrypt);

    if (!EVP_CipherUpdate(&ctx, dest, &update_len, src, len)) {
	pp_log("%s(): EVP_CipherUpdate() failed", ___F);
	return -1;
    }

    if (!EVP_CipherFinal_ex(&ctx, &dest[update_len], &final_len)) {
	pp_log("%s(): EVP_CipherFinal() failed", ___F);
	return -1;
    }

    return 0;
}

char* pp_um_passwd_encrypt(const char* passwd)
{
    char plain[1 + 64]; // salt + passwd
    char encr[1 + 64]; // encr code (same size)
    char *mem, *code;
    BIO *bio;
    int sz = (sizeof(encr) + 2) / 3 * 4; // base64 enc size

    if (strlen(passwd) > 64) return NULL; // passwd too long

    // encrypting
    srand(time(NULL));
    plain[0] = rand() & 0xff; // set salt value
    strncpy(plain + 1, passwd, sizeof(plain) - 1); // add passwd padded with 0s
    if (um_crypt(encr, plain, sizeof(plain), 1) < 0) return NULL;

    // base64-encoding
    bio = BIO_new(BIO_f_base64());
    BIO_set_flags(bio, BIO_FLAGS_BASE64_NO_NL);
    bio = BIO_push(bio, BIO_new(BIO_s_mem()));
    BIO_write(bio, encr, sizeof(encr));
    BIO_flush(bio);
    sz = BIO_get_mem_data(bio, &mem);
    code = malloc(sz + 1);
    memcpy(code, mem, sz);
    code[sz] = '\0';
    BIO_free_all(bio);

    return code;
}

char* um_passwd_decrypt(const char* code)
{
    char plain[1 + 64 + 1]; // salt + passwd + tailing 0
    char encr[1 + 64]; // encr code (same size)
    char * code_copy = strdupa(code); /* implizitely freed */
    char *passwd;
    BIO *bio;
    int sz;

    // base64-decoding
    bio = BIO_new(BIO_f_base64());
    BIO_set_flags(bio, BIO_FLAGS_BASE64_NO_NL);
    bio = BIO_push(bio, BIO_new_mem_buf(code_copy, -1));
    sz = BIO_read(bio, encr, sizeof(encr));
    BIO_free_all(bio);

    // decrypting
    if (um_crypt(plain, encr, sz, 0) < 0) return NULL;
    plain[sz] = '\0'; // add tailing 0 for savety
    passwd = strdup(plain + 1); // copy passwd ignoring salt

    return passwd;
}

int um_is_transformable(const char *type, int id, const char* transformation) {
    int transformable, ret;
    
    /* some checks, will not be performed in final */
    assert(!strcmp(type, "user") || !strcmp(type, "group"));
    assert(!strcmp(transformation, "rename") ||
           !strcmp(transformation, "delete") ||
           !strcmp(transformation, "modify"));
    
    if(PP_SUC == (ret = pp_cfg_is_enabled(&transformable, transformable_key,
                                          type, id, transformation))) {
        ret = transformable ? PP_SUC : PP_ERR;
    }
    
    return ret;
}

