/**
 * User manangement routines
 * This library implements RAASIP compatible usermanagement and replaces
 * the old libpp_um (that was based on permission inheritance).
 * 
 * All functions in this library operate (unless otherwise noted)
 * on the config system. That means they operate either on the local
 * flashdisk (fully writable) or on a remote ldap server (mostly
 * readonly).
 *
 * (c) 2006 Peppercon AG, geo@peppercon.de, tweb@peppercon.de
 */

// system includes
#include <openssl/md5.h>
#include <openssl/evp.h>
#include <pthread.h>
#if defined(PP_FEAT_CASE_INSENSITIVE_LOGINS)
#include <ctype.h> // isupper, tolower
#endif /* PP_FEAT_CASE_INSENSITIVE_LOGINS */

// fw includes
#include <pp/um.h>
#include <pp/ldap_prof.h>
#include <liberic_config.h>
#include <liberic_session.h>

// local includes
#include "um_intern.h"
#include "um_cfg_keys.h"

#ifdef PP_FEAT_RDM_SUPPORT
#include "pp/PP_RDM_c.h"
#endif

static pp_hash_t *um_uid_by_username = NULL;
static pthread_mutex_t um_user_mtx = PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP;

static int next_free_uid = -2;

static inline int get_next_free_uid(void) {
    return next_free_uid++;
}

int um_user_init() {
    assert(um_uid_by_username == NULL);
    
    um_uid_by_username = pp_hash_create(100);
    um_load_all_users();

    return PP_SUC;
}

void um_user_cleanup(void) {
    assert(um_uid_by_username);

    pp_hash_delete(um_uid_by_username);
    um_uid_by_username = NULL;
}

int pp_um_user_authenticate(const char* username,
                            const char* password,
                            int flags,
                            int *server_sid,
			    int *gid) {
#ifndef NDEBUG
    pp_log("FIXME! use pp_um_user_authenticate_with_ip instead of pp_um_user_authenticate\n");
#endif /* NDEBUG */
    return pp_um_user_authenticate_with_ip_str(username, password, flags,
                                               NULL, server_sid,
                                               gid);
}

#if 0
int pp_um_user_authenticate_with_ip(const char* username,
                                    const char* password,
                                    int flags,
                                    struct sockaddr *user_ip,
                                    socklen_t user_ip_len,
                                    int *server_sid) {
    int ret;
    (void)username;(void)password;(void)flags;(void)user_ip;(void)user_ip_len;(void)server_sid;
    abort(); // TODO: implement me
    // inet_ntoa or something... (e.g. see libpp_rdm_eric/src/ppUserObject.cpp)
    // pp_um_user_authenticate_with_ip_str(username, password, flags, user_ip_str, server_sid);
    return ret;
}
#endif

int pp_um_user_authenticate_with_ip_str(const char* username,
                                        const char* password,
                                        int flags,
                                        const char* user_ip_str,
                                        int *server_sid,
					int *p_gid) {
#define SET_AUTH_SUCCESSFUL(success)    \
    if (success) {                      \
        ret = PP_SUC;                   \
    } else {                            \
        ret = PP_ERR;                   \
        errno = PP_UM_ERR_AUTH_FAILED;  \
    }

    char md5_hash[MD5_DIGEST_LENGTH];
    char md5_hash_str[MD5_DIGEST_LENGTH*2+1];
    char *pw_hash = NULL;
    int ret = PP_ERR;
    int auth, uid = PP_ERR, local_auth = 1, is_root = 0;
    int gid = PP_ERR;
#if defined(PP_FEAT_CASE_INSENSITIVE_LOGINS)
    char *username_dc, *c;
#endif /* PP_FEAT_CASE_INSENSITIVE_LOGINS */

#if defined(PP_FEAT_BASIC_REMOTE_AUTHENTICATION)
#   define PGROUPNAME NULL // don't need a group name from remote auth server
#else /* !PP_FEAT_BASIC_REMOTE_AUTHENTICATION */
    char *groupname = NULL;
#   define PGROUPNAME (&groupname)
#endif /* !PP_FEAT_BASIC_REMOTE_AUTHENTICATION */

    if (username == NULL) { return ret; }
    if (password == NULL) { password = ""; }
    if (server_sid) *server_sid = 0;

#if defined(PP_FEAT_CASE_INSENSITIVE_LOGINS)
    /* convert username to lower case */
    
    username_dc = strdup(username);
    username = username_dc;
    
    for(c = username_dc; *c; ++c) {
        if(isupper(*c)) {
            *c = tolower(*c);
        }
    }
#endif /* PP_FEAT_CASE_INSENSITIVE_LOGINS */

#ifdef PP_FEAT_RDM_SUPPORT
    if (pp_rdm_is_managed() != 0) {
        // we are managed by CC. Don't allow any local logins.
        errno = PP_UM_ERR_DEVICE_IS_CC_MANAGED;
        ret = PP_ERR;
        goto bail;
    }
#endif

/* FIXME: TODO user (un)blocking and user disable stuff for LDAP/norbox/RADIUS */

    pp_um_user_is_root(&is_root, username);

#if defined(PP_FEAT_BASIC_REMOTE_AUTHENTICATION)
    // basic remote auth uses local user profiles!
    /* user exists? */
    if (PP_ERR == (uid = pp_um_user_get_uid(username))) {
        ret = PP_ERR;
        errno = PP_UM_ERR_AUTH_FAILED;
        goto bail;
    }
#endif /* PP_FEAT_BASIC_REMOTE_AUTHENTICATION */

    /* check whether to use LDAP for authentication */
    if (!is_root && (PP_SUC == pp_um_user_ldap_auth(username))) {
	/* authenticate against LDAP server */
        local_auth = 0;
        
	if(PP_ERR != (ret = pp_ldap_authenticate(&auth, pp_eui64_str, 
                                                 username, password,
                                                 PGROUPNAME))) {
#if defined(PP_FEAT_BASIC_REMOTE_AUTHENTICATION)
            // take group of _local_ user
            gid = pp_um_user_get_gid(username);
#else /* !PP_FEAT_BASIC_REMOTE_AUTHENTICATION */
            /* validate groupname coming from AS */
            gid = um_group_validate_remote(groupname);
#endif /* !PP_FEAT_BASIC_REMOTE_AUTHENTICATION */
            if (p_gid) *p_gid = gid;

            SET_AUTH_SUCCESSFUL(auth);
#if defined(PP_FEAT_COMMON_SECURITY_REMOTE_AS)
        } else if(errno == PP_LDAP_ERR_CONN_FAILED) {
            /* fall back to local authentication, if remote AS is unreachable */
            local_auth = 1;
#endif /* PP_FEAT_COMMON_SECURITY_REMOTE_AS */
        }
    }
#if defined(PP_FEAT_AUTH_RADIUS)
    else if (!is_root && pp_um_radius_is_active()){
	/* use radius for authentication */
        local_auth = 0;
	if(PP_ERR != (ret = pp_um_radius_auth_request(&auth, username, password,
                                                      PGROUPNAME, server_sid))) {
            SET_AUTH_SUCCESSFUL(auth);

#if defined(PP_FEAT_BASIC_REMOTE_AUTHENTICATION)
            // take group of _local_ user
            gid = pp_um_user_get_gid(username);
#else /* !PP_FEAT_BASIC_REMOTE_AUTHENTICATION */
	    /* validate groupname coming from AS */
	    gid = um_group_validate_remote(groupname);
#endif /* !PP_FEAT_BASIC_REMOTE_AUTHENTICATION */
	    if(p_gid) *p_gid = gid;

#if defined(PP_FEAT_COMMON_SECURITY_REMOTE_AS)
        } else {
            /* fall back to local authentication, if remote AS is unreachable */
            local_auth = 1;
#endif /* PP_FEAT_COMMON_SECURITY_REMOTE_AS */
	}
    }
#endif
    if(local_auth) {
	/*
	 * use the local user management for authentication
	 */

#if !defined(PP_FEAT_BASIC_REMOTE_AUTHENTICATION)
        // has already been tested above for basic remote auth
        /* user exists? */
        if (PP_ERR == (uid = pp_um_user_get_uid(username))) {
            ret = PP_ERR;
            errno = PP_UM_ERR_AUTH_FAILED;
            goto bail;
        }
#endif /* !PP_FEAT_BASIC_REMOTE_AUTHENTICATION */

	gid = pp_um_user_get_gid(username);
	if(p_gid) *p_gid = gid;

	if (!(flags & PP_UM_AUTH_IGNORE_BLOCKING)) {
	    /* check whether this user exceeded
	     * the max nr. of login attempts    */
	    if (pp_um_user_is_blocked(username, NULL)) {
		errno = PP_UM_ERR_USER_IS_BLOCKED;
		goto bail;
	    }
	}
	{
	    int enabled;
	    pp_cfg_is_enabled(&enabled, usr_enabled_key, uid);
	    if (!enabled) {
		errno = PP_UM_ERR_USER_DISABLED;
		goto bail;
	    }
	}

	if (PP_FAILED(ret = pp_cfg_get_nodflt(&pw_hash, usr_pw_hash_key, uid))){
	    goto bail;
	}

        if (flags & PP_UM_AUTH_PASSWD_AS_MD5_HASH) {
                /* password is already a MD5 hash */
                SET_AUTH_SUCCESSFUL(!strcasecmp(password, pw_hash));
        } else {
#if !defined(PP_STANDALONE_TEST)
        	MD5(password, strlen(password), md5_hash);
        	pp_md5_hash_to_str(md5_hash, md5_hash_str);
#endif /* !PP_STANDALONE_TEST */

        	SET_AUTH_SUCCESSFUL(!strcmp(md5_hash_str, pw_hash));
        }

	if (!(flags & PP_UM_AUTH_IGNORE_BLOCKING)) {
	    um_user_handle_blocking(username, ret == PP_SUC);
	}
    }

    /* if autentication already failed, don't check IP! */
    if(ret != PP_ERR && user_ip_str) {
        /* if IP is given, check if user may log on from that address */
	if (gid == PP_ERR) {
	    /* invalid gid */
	    pp_log("gid is invalid, can not check IP, so authorization failed\n");
	    ret = PP_ERR;
	    goto bail;
	}
        SET_AUTH_SUCCESSFUL(PP_SUCCED(um_group_is_allowed_from_ip_str(gid, user_ip_str)));
    }

    if(ret != PP_ERR) {
#if !defined(PP_FEAT_BASIC_REMOTE_AUTHENTICATION)
        /* FIXME: how to store last login time for remote AS?! */
        if (uid != PP_ERR && local_auth) {
#endif /* !PP_FEAT_BASIC_REMOTE_AUTHENTICATION */
            /* save last login time */
            u_long last_login;

            pp_cfg_get_ulong(&last_login, usr_current_login_time_key, uid);
            pp_cfg_set_ulong(last_login, usr_last_login_time_key, uid);
            pp_cfg_set_ulong((ulong)time(NULL), usr_current_login_time_key, uid);
#if !defined(PP_FEAT_BASIC_REMOTE_AUTHENTICATION)
        }
#endif /* !PP_FEAT_BASIC_REMOTE_AUTHENTICATION */
    }
    
 bail:
    free(pw_hash);
#if defined(PP_FEAT_CASE_INSENSITIVE_LOGINS)
    free(username_dc);
#endif /* PP_FEAT_CASE_INSENSITIVE_LOGINS */
#if !defined(PP_FEAT_BASIC_REMOTE_AUTHENTICATION)
    free(groupname);
#endif /* !PP_FEAT_BASIC_REMOTE_AUTHENTICATION */
    return ret;
}

int pp_um_user_has_permission(const char* username,
                              const char* permission_name,
                              const char* permission_value) {
    int gid;
    
#if !defined(PP_FEAT_BASIC_REMOTE_AUTHENTICATION)
    char *auth_mode;
    eric_session_int_id_t s;

    // permissions are taken from the session instead of the group for
    // remote authentication (except for basic mode)
    pp_cfg_get(&auth_mode, "auth_mode");
    if (strcmp(auth_mode, "common")) {
	// remote auth
	if((s = eric_session_get_by_login(username)) != 0){
	    if(PP_SUC == eric_session_has_permission(s, 
						    permission_name, 
						    permission_value)){
		eric_session_release(s);
		free(auth_mode);
		return PP_SUC;
	    }
	    eric_session_release(s);
	    // else, still need to check local auth 
	    // if remote auth server unreachable
	}
	// in case, there is no session associated to 
	// need to check local auth
    }
    free(auth_mode);
#endif /* !PP_FEAT_BASIC_REMOTE_AUTHENTICATION */

    // local auth
    if(PP_ERR == (gid = pp_um_user_get_gid(username))) {
	// todo errno?
	return PP_ERR;
    }
    
    return um_group_has_permission(gid, permission_name, permission_value);
}

/*
 * Note: This API is only for local authentication.
 *       For remote authentication (LADP or Radius),
 *       should get gid stored in session,
 *       then check permission by gid.
 */
int pp_um_user_get_permission(const char* username,
                              const char* permission_name,
                              char** permission_value) {
    int gid;
    
    assert(username);
    assert(permission_name);
    assert(permission_value);
    
    if(PP_ERR == (gid = pp_um_user_get_gid(username))) {
        // todo errno?
        return PP_ERR;
    }

    return um_group_get_permission(gid, permission_name, permission_value);
}

int pp_um_user_set_permission(const char* username,
                              const char* permission_name,
                              const char* permission_value) {
    int gid;
    
    if(PP_ERR == (gid = pp_um_user_get_gid(username))) {
        // todo errno?
        return PP_ERR;
    }

    return um_group_set_permission(gid, permission_name, permission_value);
}

int pp_um_usermanagement_possible() {
/* FIXME: implement me */
    return PP_ERR;
}

int pp_um_passwordchange_reasonable(const char* username) {
(void)username;
/* FIXME: implement me */
    return PP_ERR;
}

static int find_free_ipmi_uid(void)
{
    int *uid_p;
    char uid_used[64];
    int i;

    memset(uid_used, 0, sizeof(uid_used));
    /* um_user_mtx is already locked */
    uid_p = pp_hash_get_first_entry(um_uid_by_username);
    while (uid_p) {
	int ipmi_uid;
	if (PP_SUCCED( pp_cfg_get_int(&ipmi_uid, usr_ipmi_UID, *uid_p) )) {
	    uid_used[ipmi_uid] = 1;
	}
	uid_p = pp_hash_get_next_entry(um_uid_by_username);
    }

    /* 1 = anonymous user => start searching at 2 */
    for (i = 2; i < 64; i++) {
	if (!uid_used[i]) return i;
    }

    return -1;
}

static int check_ipmi_uid_available(int ipmi_uid)
{
    int *uid_p;
    if (ipmi_uid < 1 || ipmi_uid > 63) return 0;
    /* um_user_mtx is already locked */
    uid_p = pp_hash_get_first_entry(um_uid_by_username);
    while (uid_p) {
	int iuid;
	if (PP_SUCCED( pp_cfg_get_int(&iuid, usr_ipmi_UID, *uid_p) )) {
	    if (ipmi_uid == iuid) return 0;
	}
	uid_p = pp_hash_get_next_entry(um_uid_by_username);
    }
    return 1;
}

int pp_um_user_create(const char* username, const char* password, int flags,
                      const char* groupname, const char* dialback, int ipmi_uid) {
    int ret = PP_SUC;
    int uid = -1, *uid_p;
    
    MUTEX_LOCK(&um_user_mtx);
    
    assert(username);
    assert(um_uid_by_username);
    
    /* new name unique? */
    if (pp_hash_get_entry(um_uid_by_username, username) != NULL) {
        errno = PP_UM_ERR_USER_ALREADY_EXISTS;
        ret = PP_ERR;
        goto bail;
    }
    
    uid = get_next_free_uid();
    assert(uid > 0);

    if (ipmi_uid == 0) {
	/* no ipmi_uid given; find a free one */
	ipmi_uid = find_free_ipmi_uid();
	if (ipmi_uid < 0) {
	    errno = PP_UM_ERR_TOO_MANY_USERS;
	    ret = PP_ERR;
	    goto bail;
	}
    } else {
	if (!check_ipmi_uid_available(ipmi_uid)) {
	    errno = PP_UM_ERR_INVALID_IPMI_UID;
	    ret = PP_ERR;
	    goto bail;
	}
    }

    uid_p = (int*)malloc(sizeof(int));
    *uid_p = uid;
    pp_hash_set_entry(um_uid_by_username, username, uid_p, free);
    
    /* set login */
    ret = pp_cfg_set(username, usr_login_key, uid);

    /* set password */
    if(password) {
        switch (flags) {
            case PP_UM_AUTH_PASSWD_AS_MD5_HASH:
                ret = pp_cfg_set(password, usr_pw_hash_key, uid);
                break;
            case PP_UM_AUTH_NO_FLAGS:
                ret = um_user_set_password(uid, password, flags);
                break;
            default:
                ret = PP_ERR;
                assert(0);
                goto bail;
        }
#ifdef PP_FEAT_STRONG_PASSWORDS
        pp_um_strong_pw_add_to_history(username, password, flags);
#endif /* PP_FEAT_STRONG_PASSWORDS */
    }
    
    if(groupname) {
        pp_um_group_state_t state;
        
        if(PP_ERR != pp_um_group_is_individual_for_user(groupname, NULL,
                                                        &state) &&
           state != PP_UM_GROUP_INDIVIDUAL_TAKEN) {
            ret = pp_um_user_set_group(username, groupname);
        } else {
            errno = PP_UM_ERR_GROUP_IS_INDIVIDUAL;
            ret = PP_ERR;
            goto bail;
        }
    } else {
        /* no group given, generate individual one */
        
        char *grouptmp;
        
        if(PP_SUC == (ret = um_user_create_individual_group(username,
                                                            &grouptmp))) {
            ret = pp_um_user_set_group(username, grouptmp);
            free(grouptmp);
        }
    }
    
#if defined(PP_FEAT_HAS_DIALBACK)
    if(dialback) {
        ret = pp_cfg_set(dialback, usr_dialback_key, uid);
    }
#else /* !PP_FEAT_HAS_DIALBACK */
    (void)dialback;
#endif /* !PP_FEAT_HAS_DIALBACK */

    pp_cfg_set_int(ipmi_uid, usr_ipmi_UID, uid);

 bail:
    if(ret == PP_ERR && uid > 0) {
        /* something's wrong, do not create user -> cover the tracks */
        pp_cfg_remove(usr_key, uid);
        pp_hash_delete_entry(um_uid_by_username, username);
    }
 
    MUTEX_UNLOCK(&um_user_mtx);
    return ret;
}

int pp_um_user_copy(const char* username_src, const char* username_dest) {
    int ret = PP_SUC;
    char uid_buffer[10];
    int uid, uid_dest, *uid_p, gid;
    pp_um_group_state_t state;
    char *groupname = NULL;

    assert(username_src);
    assert(username_dest);
    assert(um_uid_by_username);
    
    MUTEX_LOCK(&um_user_mtx);
    
    /* old name exists? */
    if (PP_ERR == (uid = pp_um_user_get_uid(username_src))) {
        ret = PP_ERR;
        goto bail;
    }

    /* new name unique? */
    if (pp_hash_get_entry(um_uid_by_username, username_dest) != NULL ||
        !strcmp(username_src, username_dest)) {
        errno = PP_UM_ERR_USER_ALREADY_EXISTS;
        ret = PP_ERR;
        goto bail;
    }
    
    uid_dest = get_next_free_uid();
    snprintf(uid_buffer, 9, "%d", uid_dest);
    if(PP_ERR == (ret = pp_cfg_copy(uid_buffer, usr_key, uid))) {
        /* todo errno?? */
        goto bail;
    }

    pp_cfg_set(username_dest, usr_login_key, uid_dest);
    uid_p = (int*)malloc(sizeof(int));
    *uid_p = uid_dest;
    pp_hash_set_entry(um_uid_by_username, username_dest, uid_p, free);
    
    /* is group individual? */
    if(PP_ERR != (ret = pp_um_user_get_group(username_src, &groupname)) &&
       PP_ERR != (ret = pp_um_group_is_individual_for_user(groupname, NULL,
                                                           &state)) &&
       state != PP_UM_GROUP_GENERIC) {
        char newgroup[65];
        
        *newgroup = um_individual_group_prefix;
        strncpy(newgroup + 1, username_dest, sizeof(newgroup) - 1);
        
        /* group is individual, copy and assign to destination user */
        if(PP_ERR != (ret = pp_um_group_copy(groupname, newgroup)) &&
           PP_ERR != (gid = pp_um_group_get_gid(newgroup))) {
            ret = pp_cfg_set_int(gid, usr_group_id_key, uid_dest);
        } else {
            /* something is wrong, delete destination users group id */
            ret = pp_cfg_remove(usr_group_id_key, uid_dest);
            /* TODO! leave some hint, that group could not be set? 
                     delete new user? failed or succeeded? */
        }
   }
   /* FIXME! copy individual group and assign to new user */
    
 bail:
    MUTEX_UNLOCK(&um_user_mtx);
    free(groupname);
    
    return ret;
}


int pp_um_user_rename(const char* username_current, const char* username_new) {
    int ret = PP_SUC;
    int uid, *uid_p;
    
    assert(username_current);
    assert(username_new);
    assert(um_uid_by_username);
    
    /* no rename if current and new name are identical */
    if (!strcmp(username_current, username_new)) {
        return PP_SUC;
    }

    MUTEX_LOCK(&um_user_mtx);
    
    /* old name exists? */
    if (PP_ERR == (uid = pp_um_user_get_uid(username_current))) {
        ret = PP_ERR;
        goto bail;
    }

    /* new name unique? */
    if (pp_hash_get_entry(um_uid_by_username, username_new) != NULL) {
        errno = PP_UM_ERR_USER_ALREADY_EXISTS;
        ret = PP_ERR;
        goto bail;
    }
    
    /* builtin user? */
    if (um_user_is_renameable(uid) == PP_ERR) {
        errno = PP_UM_ERR_NOT_FOR_BUILTIN_USR;
        ret = PP_ERR;
        goto bail;
    }
    
    pp_cfg_set(username_new, usr_login_key, uid);

    pp_hash_delete_entry(um_uid_by_username, username_current);
    uid_p = (int*)malloc(sizeof(int));
    *uid_p = uid;
    pp_hash_set_entry(um_uid_by_username, username_new, uid_p, free);
    
 bail:
    MUTEX_UNLOCK(&um_user_mtx);
    
    return ret;
}

int pp_um_user_delete(const char* username) {
    int uid;
    int ret = PP_ERR;
    
    assert(um_uid_by_username);

    MUTEX_LOCK(&um_user_mtx);

    /* name exists? */
    if (PP_ERR == (uid = pp_um_user_get_uid(username))) {
        goto bail;
    }

    if(uid < 0) {
        errno = PP_UM_ERR_INVALID_USERNAME;
    } else if(um_user_is_deleteable(uid) == PP_ERR) {
        errno = PP_UM_ERR_NOT_FOR_BUILTIN_USR;
    } else {
        char *group = NULL;
        
        if(PP_SUC == pp_um_user_get_group(username, &group) &&
           *group == um_individual_group_prefix) {
            /* user has individual group, delete that as well */
            pp_um_group_delete(group);
        }
        free(group);
        
        if(PP_SUC == (ret = pp_cfg_remove(usr_key, uid))) {
            pp_hash_delete_entry(um_uid_by_username, username);
            pp_cfg_save(DO_FLUSH);
        }
    }
    
 bail:
    MUTEX_UNLOCK(&um_user_mtx);
    return ret;
}

int pp_um_user_set_enabled(const char* username, int enabled)
{
    int ret = PP_ERR, uid;
    assert(username);

    MUTEX_LOCK(&um_user_mtx);
    if ((uid = pp_um_user_get_uid(username)) != PP_ERR) {
        ret = pp_cfg_set_enabled(enabled, usr_enabled_key, uid);
    }
    MUTEX_UNLOCK(&um_user_mtx);

    return ret;
}

int
pp_um_user_set_password(const char* username, const char* password, int flags) {
    int uid;

    assert(username);

    /* name exists? */
    if (PP_ERR == (uid = pp_um_user_get_uid(username))) {
        return PP_ERR;
    }

#if defined(PP_FEAT_STRONG_PASSWORDS)
    {
        int usr_do_check = 1;
        int cfg_do_check;
        
        pp_cfg_is_enabled(&cfg_do_check, strong_pw_enabled_key);

#if defined(PP_FEAT_STRONG_PASSWORDS_ONLY_ROOT)
	pp_um_user_is_root(&usr_do_check, username);
#endif
        
        if(cfg_do_check && usr_do_check) {
            char *old_password = NULL;
              
            /* never set non plaintext passwords unchecked! */
            assert((flags & PP_UM_AUTH_STRONG_PW_CHECKED) ||
                   !(flags & PP_UM_AUTH_PASSWD_AS_MD5_HASH));
            
            pp_cfg_get_nodflt(&old_password, usr_pw_hash_key, uid);
              
            if(!(flags & PP_UM_AUTH_STRONG_PW_CHECKED) &&
               PP_ERR == um_strong_pw_check_for_user_core(username, password,
                                                          NULL, NULL)) {
                free(old_password);
                return PP_ERR;
            }
            
            free(old_password);
        }
    }
#endif /* PP_FEAT_STRONG_PASSWORDS */

    return um_user_set_password(uid, password, flags);
}

int um_user_set_password(int uid, const char* password, int flags) {
    int ret = PP_ERR;
    char *pw_encr = NULL;
    char md5_hash[MD5_DIGEST_LENGTH];
    char md5_hash_str[MD5_DIGEST_LENGTH * 2 + 1];

    assert(password);
    
    /* to store encrypted passwords, we need them not hashed! */
    assert(!(flags & PP_UM_AUTH_PASSWD_AS_MD5_HASH));

    MUTEX_LOCK(&um_user_mtx);

    // encrypt passwd
    pw_encr = pp_um_passwd_encrypt(password);
    if (!pw_encr) {
        errno = PP_UM_ERR_INTERNAL_ERROR;
        goto exit;
    }

    // calc MD5 hash for passwd
    MD5(password, strlen(password), md5_hash);
    pp_md5_hash_to_str(md5_hash, md5_hash_str);

    // check if password change is forced
    if(flags & PP_UM_AUTH_PASSWD_FORCE_CHANGE) {
        char *pw_hash = NULL;
        
        if(PP_ERR != pp_cfg_get(&pw_hash, usr_pw_hash_key, uid) &&
            !strcmp(pw_hash, md5_hash_str)) {
            /* the password didn't change! */
            free(pw_hash);
            errno = PP_UM_ERR_PASSWORD_DIDNT_CHANGE;
            goto exit;
        }
        
        free(pw_hash);            
    }
    
    // all ok, store passwords to config
    if (PP_FAILED(pp_cfg_set(pw_encr, usr_pw_encr_key, uid)) ||
        PP_FAILED(pp_cfg_set(md5_hash_str, usr_pw_hash_key, uid))) {
        errno = PP_UM_ERR_INTERNAL_ERROR;
        goto exit;
    }
    
    pp_cfg_set_enabled(0, usr_need_pw_change_key, uid);
    pp_cfg_save(DO_FLUSH);
    ret = PP_SUC;
    
exit:
    free(pw_encr);
    MUTEX_UNLOCK(&um_user_mtx);
    return ret;
}

int pp_um_user_set_group(const char* username, const char* group) {
    int uid = pp_um_user_get_uid(username);
    int gid;
    int ret = PP_ERR;
    pp_um_group_state_t state;
    char *oldgroup = NULL;
    char *tmpgroup = NULL;
    
    if(PP_ERR != pp_um_user_get_group(username, &oldgroup) && 
       ((group && !strcmp(oldgroup, group)) ||
        (!group && (*oldgroup == um_individual_group_prefix)))) {
        /* group didn't change, oh sunny day! */
        return PP_SUC;
    }
    
    if(!group) {
        /* create new individual group for user */
        um_user_create_individual_group(username, &tmpgroup);
        group = tmpgroup;
    }
    
    gid = pp_um_group_get_gid(group);

    if(uid < 0) {
        errno = PP_UM_ERR_INVALID_USERNAME;
    } else if(gid < 0) {
        errno = PP_UM_ERR_INVALID_GROUPNAME;
    } else if(um_user_is_modifiable(uid) == PP_ERR) {
        errno = PP_UM_ERR_NOT_FOR_BUILTIN_USR;
    } else if(PP_ERR == pp_um_group_is_individual_for_user(group, username,
                                                           &state) ||
              state == PP_UM_GROUP_INDIVIDUAL_TAKEN) {
        errno = PP_UM_ERR_GROUP_IS_INDIVIDUAL;
    } else {
        ret = pp_cfg_set_int(gid, usr_group_id_key, uid);
        if(*oldgroup == um_individual_group_prefix) {
            /* no need for user's old individual group any more */
            pp_um_group_delete(oldgroup);
        }
        pp_cfg_save(DO_FLUSH);
    }
    
    free(oldgroup);
    free(tmpgroup);
    
    return ret;
}

vector_t* pp_um_get_all_users() {
    return pp_um_get_all_users_by_gid(-1);
}

vector_t* pp_um_get_all_users_by_gid(int gid) {
    vector_t* users = NULL;
    const char *user;
    
    assert(um_uid_by_username);
    
    MUTEX_LOCK(&um_user_mtx);
    
/* FIXME: CC integration */
/* FIXME: LDAP */
    users = vector_new(NULL, 10, free);
    
    for(user = pp_hash_get_first_key(um_uid_by_username); user;
        user = pp_hash_get_next_key(um_uid_by_username)) {
        int user_gid;
        if(gid < 0 /* all groups */ ||
           (PP_ERR != (user_gid = pp_um_user_get_gid(user)) &&
            user_gid == gid /* get users of specified gid only */ )) {
            vector_add(users, strdup(user));
        }
    }
    
    if(vector_size(users)) {
        vector_qsort(users, str_compare);
    } else {
        vector_delete(users);
        users = NULL;
    }

    MUTEX_UNLOCK(&um_user_mtx);
    return users;
}

vector_t* um_get_all_uids_by_gid(int gid) {
    vector_t* uids = NULL;
    char *user;
    void *uid_p;
    
    assert(um_uid_by_username);
    
    MUTEX_LOCK(&um_user_mtx);
    
/* FIXME: CC integration */
/* FIXME: LDAP */
    uids = vector_new(NULL, 10, NULL);
    
    for(pp_hash_get_first_key_and_content(um_uid_by_username, &user, &uid_p); 
        user;
        pp_hash_get_next_key_and_content(um_uid_by_username, &user, &uid_p)) {
        int user_gid;
        if(gid < 0 /* all groups */ ||
           (PP_ERR != (user_gid = pp_um_user_get_gid(user)) &&
            user_gid == gid /* get users of specified gid only */ )) {
            vector_add_int(uids, *(int*)uid_p);
        }
    }
    
    if(vector_size(uids)) {
//        vector_qsort(users, str_compare);
    } else {
        vector_delete(uids);
        uids = NULL;
    }

    MUTEX_UNLOCK(&um_user_mtx);
    return uids;
}

int pp_um_user_get_uid(const char* username) {
    int *uid;
    
    assert(um_uid_by_username);

/* FIXME: CC integration */
/* FIXME: LDAP */
    if(!username) {
        errno = PP_UM_ERR_INTERNAL_ERROR;
        return PP_ERR;
    }
    
    if(NULL != (uid = (int*)pp_hash_get_entry(um_uid_by_username, username))) {
        return *uid;
    }
    
    errno = PP_UM_ERR_USER_DOESNT_EXIST;
    return PP_ERR;
}

int pp_um_user_is_enabled(const char* username)
{
    int ret = -1, uid;
    assert(username);

    MUTEX_LOCK(&um_user_mtx);
    if ((uid = pp_um_user_get_uid(username)) != PP_ERR) {
        pp_cfg_is_enabled_nodflt(&ret, usr_enabled_key, uid);
    }
    MUTEX_UNLOCK(&um_user_mtx);

    return ret;
}

int pp_um_user_get_password(const char* username, char** password) {
    char *passwd = NULL;
    char *pw_encr = NULL;
    int ret = PP_ERR, uid;

    assert(username);
    assert(password);

    MUTEX_LOCK(&um_user_mtx);

    /* name exists? */
    if (PP_ERR == (uid = pp_um_user_get_uid(username))) {
        goto exit;
    }

    // read encryped passwd from config
    if (PP_FAILED(pp_cfg_get_nodflt(&pw_encr, usr_pw_encr_key, uid))) {
        errno = PP_UM_ERR_NO_PLAINTEXT_PASSWD;
        goto exit;
    }

    // decrypt passwd
    passwd = um_passwd_decrypt(pw_encr);
    if (!passwd) {
        errno = PP_UM_ERR_INTERNAL_ERROR;
        goto exit;
    } else {
        ret = PP_SUC;
        *password = passwd;
    }

exit:
    if (pw_encr) free(pw_encr);
    MUTEX_UNLOCK(&um_user_mtx);
    return ret;
}

int pp_um_user_get_group(const char* username, char** group) {
    char *groupname;
    int ret = PP_ERR, gid = -1;
    
    if(PP_ERR != (gid = pp_um_user_get_gid(username)) &&
       PP_ERR != (ret = pp_cfg_get_nodflt(&groupname, grp_name_key, gid))) {
        *group = groupname;
    }
    
    return ret;
}

int pp_um_user_get_gid(const char* username) {
    int ret = PP_ERR, gid = PP_ERR, uid = PP_ERR;

    if(!username || !*username) {
        errno = PP_UM_ERR_INTERNAL_ERROR;
        return PP_ERR;
    }

    /* get uid and gid */
    if(PP_ERR != (uid = pp_um_user_get_uid(username))) {
        pp_cfg_get_int(&gid, usr_group_id_key, uid);

        /* valid local user -> either gid is valid or 1 (<Unknown>) */
        ret = gid != PP_ERR ? gid : 1;
    }
    
    return ret;
}

int pp_um_user_is_root(int* is_root, const char* username) {
    int ret;
    
    if(PP_ERR != (ret = pp_um_user_get_uid(username))) {
        *is_root = ret == 0;
        ret = PP_SUC;
    }
    
    return ret;
}

int pp_um_user_is_blocked(const char* name, int* mins_left) {
    u_long fail_time, current_time, time_diff, block_time;
    int login_fails, max_login_fails;
    int ret = 0;
    int uid;

    uid = pp_um_user_get_uid(name);
    
    if (uid < 0) {
	errno = PP_UM_ERR_INTERNAL_ERROR;
	goto bailout;
    }

    /* check if blocking is disabled */
    if (PP_FAILED(pp_cfg_get_int_nodflt(&max_login_fails, 
                                        antibf_max_login_fails_key))) {
	goto bailout;
    }

    /* decide if user is blocked */
    pp_cfg_get_int(&login_fails, usr_login_fails_key, uid);
    if (login_fails >= max_login_fails) {
	ret = 1;
	if (mins_left) *mins_left = -1;
        
	/* check if still in blocking time */
	if (PP_SUCCED(pp_cfg_get_ulong_nodflt(&block_time,
                                              antibf_block_time_key))) {
	    current_time = fail_time = (u_long)time(NULL);
	    pp_cfg_get_ulong_nodflt(&fail_time, usr_login_fail_time_key, uid); /* error status ignored - ok */
	    time_diff = current_time - fail_time;

	    if (mins_left) {
                *mins_left = block_time - (time_diff / 60);
            }

	    if ((time_diff / 60) > block_time) {
                /* block time has expired */
		pp_cfg_remove(usr_login_fails_key, uid);
		pp_cfg_save(DONT_FLUSH);
		ret = PP_SUC;
	    }
	}
    }
    
 bailout:
    return ret;
}

int pp_um_user_unblock(const char* name) {
    int ret, uid;
    
    /* user exists? */
    if (PP_ERR == (uid = pp_um_user_get_uid(name))) {
        ret = PP_ERR;
    } else if ((ret = pp_cfg_remove(usr_login_fails_key, uid)) != PP_ERR) {
	ret = pp_cfg_save(DO_FLUSH);
    }

    return ret;
}

/*
 * returns true only in case nbx-ldap or minimal ldap is enabled
 */
int pp_um_user_ldap_auth(const char *name) {
    int ret = PP_ERR;
    if (
/* FIXME: LDAP */
//        pp_ldap_is_active(NULL, NULL) || 
        pp_ldap_mini_is_active(NULL)) {
        MUTEX_LOCK(&um_user_mtx);
        if(pp_um_user_get_uid(name)) {
            /* uid != 0 -> user isn't admin -> LDAP is active 
             * at this point, it doesn't matter if user exists, as LDAP does
             * authentication for us */
            ret = PP_SUC;
        }
        MUTEX_UNLOCK(&um_user_mtx);
    }
    return ret;
}

int um_user_handle_blocking(const char* name, int auth_result) {
    int uid;
    int login_fails = 0;
    int ret = PP_ERR;
    pp_cfg_flush_t do_flush = DO_FLUSH;

    uid = pp_um_user_get_uid(name);
    if (uid < 0) goto bail;
    
    if (auth_result == 1) {
	/* user was authenticated, reset login fail counter */
	pp_cfg_get_int(&login_fails, usr_login_fails_key, uid);
	if (login_fails == 0) {
	    do_flush = DONT_FLUSH;
	} else {
	    pp_cfg_set_int(0, usr_login_fails_key, uid);
	}
    } else {
	/* user auth failed, increase login fail counter, set timestamp */
	pp_cfg_get_int(&login_fails, usr_login_fails_key, uid);
	pp_cfg_set_int(login_fails + 1, usr_login_fails_key, uid);
	pp_cfg_set_ulong((ulong)time(NULL), usr_login_fail_time_key, uid);
    }
    
    pp_cfg_save(do_flush);
    ret = PP_SUC;

 bail:
    return ret;
}

void um_load_all_users() {
    pp_profile_type_t prof[] = { PP_PROFILE_LOCAL, PP_PROFILE_LOCAL_CODE, PP_PROFILE_LOCAL_OEM };
    pp_cfg_iter_t* iter;
    const char *uidstr;
    char *username;
    unsigned int i;
    int uid;

    assert(um_uid_by_username);

    for (i = 0; i < sizeof(prof)/sizeof(pp_profile_type_t); ++i) {
	if (PP_ERR != pp_cfg_iter_create_at_layer(&iter, prof[i], usr_vector)) {
	    while (pp_cfg_iter_next(iter, &uidstr, NULL)) {
                int *uid_p;
		int ipmi_uid;
                
		uid = pp_strtol(uidstr, -1, 10, NULL);
		if (uid < 0) {
		    // The config key is in old 'user[login]' config format.
                    // TODO! what to do now? start conversion again?
		    
		    continue;
		}
		
                /* do not get at layer, fallback for default user(s) */
		if(PP_ERR == pp_cfg_get_nodflt(&username, usr_login_key, uid)) {
                    pp_log("user name not set for user '%d'\n", uid);
                    continue;
                }

		// check wether already there
		if (pp_hash_get_entry(um_uid_by_username, username) != NULL) {
		    free(username);
		    continue;
		}

		// check for IPMI uid, try to assign one if necessary
		if (PP_FAILED( pp_cfg_get_int(&ipmi_uid, usr_ipmi_UID, uid) )) {
		    if ((ipmi_uid = find_free_ipmi_uid()) >= 0) {
			pp_cfg_set_int(ipmi_uid, usr_ipmi_UID, uid);
		    }
		}

		uid_p = (int*)malloc(sizeof(int));
                *uid_p = uid;
                pp_hash_set_entry(um_uid_by_username, username, uid_p, free);
                if (uid >= next_free_uid) {
                    next_free_uid = uid + 1;
                }
                
		free(username);
	    }
	    pp_cfg_iter_destroy(iter);
	}
    }
    pp_cfg_save(DO_FLUSH);
}

int um_user_create_individual_group(const char *username, char **groupname) {
    char groupbuf[65];
    int created = 0;
    int ret;

    snprintf(groupbuf, sizeof(groupbuf), "%c%s",
             um_individual_group_prefix, username);
     
    /* create new generic group of specified name */
    while(!created) {
        if(PP_SUC == (ret = pp_um_group_create(groupbuf, 1))) {
            created = 1;
        } else if (errno != PP_UM_ERR_GROUP_ALREADY_EXISTS || 
                   sizeof(groupbuf) < strlen(groupbuf) + 1) {
            /* maximum size of groupname reached, break */
            errno = PP_UM_ERR_INTERNAL_ERROR;
            goto bail;
        } else {
            /* group of that name exists, append '_' to groupname
               and try again */
            snprintf(groupbuf, sizeof(groupbuf), "%s_", groupbuf);
        }
    }
    
    if(ret == PP_SUC && groupname) {
        *groupname = strdup(groupbuf);
    }
    
 bail:
    return ret;
}

int pp_um_user_need_pw_change(const char* username) {
    int ret = PP_ERR;
    int pw_aging;
    int uid;
    int need_pw_change;
    
    assert(username);
    
    /* user exists? */
    if(PP_ERR == (uid = pp_um_user_get_uid(username))) {
        goto bail;
    }
    
    /* first, check if user's need_pw_change-flag is set for some reason */
    if(PP_ERR != pp_cfg_is_enabled(&need_pw_change, usr_need_pw_change_key, uid)
       && need_pw_change) {
        /* need pw change, skip further checks */
        ret = PP_SUC;
        goto bail;
    }
    
    pp_cfg_is_enabled(&pw_aging, "security.pw_aging.enabled");

    if(pw_aging) {
        /* password aging is enabled, now check if we need to change pw */ 
    
        u_long pw_change = 0, interval;
    
        /* pw_change is 0 by default if not existant, no cfg default */
        pp_cfg_get_ulong_nodflt(&pw_change, usr_pw_change_key, uid);

        if(PP_ERR != (ret = pp_cfg_get_ulong(&interval,
                                             "security.pw_aging.interval"))) {
            /* check the time diff */
            u_long timediff = (u_long)time(NULL) /* now */ - pw_change;
            
            /* if timediff in seconds > interval in days... */
            if(timediff > (interval * 86400)) {
                /* need pw change, skip further checks */
                ret = PP_SUC;
                goto bail;
            } else {
                ret = PP_ERR;
                errno = PP_UM_ERR_NO_ERROR;
            }
        }
    }

 bail:
    return ret;
}

int um_user_set_pw_change_timestamp_cb(pp_cfg_chg_ctx_t *ctx) {
    int ret = PP_SUC;
    
    if(ctx && ctx->key_comp_vec) {
        int i, kcv_sz = vector_size(ctx->key_comp_vec);
        
        for(i = 0; i < kcv_sz; ++i) {
            vector_t *key_comps = (vector_t*)vector_get(ctx->key_comp_vec, i);
            int uid, key_comps_sz = vector_size(key_comps);
            const char *comp3, *comp4;
            
            /* bit strange, but ok... */
            if(key_comps_sz == 4 && 
               NULL != (comp4 = pp_cfg_get_key_comp_name_at_idx(key_comps, 3))
               && !strcmp(comp4, "pw_hash") &&
               NULL != (comp3 = pp_cfg_get_key_comp_name_at_idx(key_comps, 2))
               && (uid = pp_strtol(comp3, PP_ERR, 10, NULL)) != PP_ERR) {
                ret = pp_cfg_set_ulong((ulong)time(NULL), 
                                       usr_pw_change_key, uid);
                if (!PP_FAILED(ret)) {
                    pp_cfg_save(DO_FLUSH);
                }
            }
        }
    }
    
   return ret;
}

int pp_um_user_is_allowed_from_ip_str(const char* username, const char* ip) {
    int targid;
    
    if(PP_ERR == (targid = pp_um_user_get_gid(username))) {
        // todo errno?
        return PP_ERR;
    }
    
    return um_group_is_allowed_from_ip_str(targid, ip);
}

