/* system includes */
#include <sys/ioctl.h>
#include <fcntl.h>
#include <poll.h>
#include <sys/mman.h>
#include <unistd.h>
#include <stdlib.h>
#include <pp/intl.h>
#include <stdio.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/un.h>

/* firmware includes */
#include <pp/base.h>
#include <pp/cfg.h>
#include <liberic_misc.h>
#include <liberic_pthread.h>
#include <pp/kvm.h>
#include <pp/setup_proto.h>
#include <pp/km.h>
#include <pp/xdefs.h>

/* our public interface */
#include <pp/usb.h>
#include "usb.h"

#define noUSB_DEBUG	1

#define MAX_DESCRIPTORS 1
#define MAX_BLOCKSIZE	2048
#define MMAP_BLOCKSIZE	(800 << 20) /* 800MB -- large enough for a regular CD image */

#define DISABLE_IF_NO_IMAGE_KEY	"vfloppy.usb_ms_disable_if_no_image"

/* global variables */
static const char * pp_usb_errors[] = {
    /*  0 */ N_("An internal error occured."),
    /*  1 */ N_("Accessing the image file failed."),
    /*  2 */ N_("An image is already in use by another user."),
    /*  3 */ N_("No USB chip on board."),
    /*  4 */ N_("Mounting the Windows SMB share failed."),
    /*  5 */ N_("Invalid path to image file."),
};

static int errno_base = 0; // triggers assertion failed if not registered

static int initialized = 0;

static int usbfds[PP_FEAT_NUM_USB_DEVICES];

/* TODO: Get these values from config file */
static int dev_kbd = 0;
static int dev_mouse = 0;
#if PP_FEAT_USB_MASS_STORAGE_NO > 0
static struct {
    int usb;
    int index;
} dev_ms[PP_FEAT_USB_MASS_STORAGE_NO] = {
    { 0, 0 },
# if PP_FEAT_USB_MASS_STORAGE_NO > 1
    { 0, 1 },
# endif
# if PP_FEAT_USB_MASS_STORAGE_NO > 2
    { 0, 2 },
# endif
# if PP_FEAT_USB_MASS_STORAGE_NO > 3
    { 0, 3 },
# endif
};
#endif

#ifdef PP_FEAT_MASS_STORAGE
static pthread_t usb_thread;

/* per-mass-storage-device variables */
typedef enum {
    SETUP_NONE,
    SETUP_WORKING,
    SETUP_DONE
} setup_request_state_t;

typedef struct {
    char pp_setup_req_packet[PP_SP_REQ_MTU];
    size_t pp_setup_req_packet_len;
    char pp_setup_rsp_packet[PP_SP_RSP_MTU];
    size_t pp_setup_rsp_packet_len;

    pthread_mutex_t image_mtx;
    int msbofd;
    int page_size_mask;
    off_t mmapped_file_length;
    off_t mmapped_file_offset;
    uint8_t* mmapped_file_data;
    int mmap_flags;
    char *smbmount_path;
    pp_usb_image_type_t act_image;
    usb_device_type_t act_image_type;
    usb_device_ms_t act_state;
    char* g_name;
    char* g_path;
    char* g_host;
    char* g_share;
    char* g_user;
    off_t img_size;

    setup_request_state_t pp_setup_request_state;
} mass_storage_t;

static mass_storage_t mass_storages[PP_FEAT_USB_MASS_STORAGE_NO];

/* function definitions */
static int usb_set_mass_storage(u_int ms_index, usb_device_ms_t state);
static int set_file(u_int ms_index, int fd, const char * filename, unsigned int readonly);
static void* usb_ms_bo_thread_func(void* data);
static int usb_ms_release_data(u_int ms_index);
#endif /* PP_FEAT_MASS_STORAGE */

/* function implementations */

int
pp_usb_errno_base()
{
    assert(errno_base > 0);
    return errno_base;
}

static const char *
get_error_string(const int error)
{
    return pp_usb_errors[error - errno_base];
}

#ifdef PP_FEAT_MASS_STORAGE
static int
usb_ms_disabled_if_no_image(void)
{
    int enabled;
    pp_cfg_is_enabled(&enabled, DISABLE_IF_NO_IMAGE_KEY);
    return enabled;
}

static int
disable_no_image_change_cb(pp_cfg_chg_ctx_t *ctx UNUSED)
{
    u_int ms_index;
    for (ms_index = 0; ms_index < PP_FEAT_USB_MASS_STORAGE_NO; ++ms_index) {
	mass_storage_t * ms = &mass_storages[ms_index];
    	MUTEX_LOCK(&ms->image_mtx);
	if (ms->act_image == PP_USB_MS_IMAGE_NONE) {
	    usb_set_mass_storage(ms_index,
				 usb_ms_disabled_if_no_image() ? usb_device_ms_disabled : usb_device_ms_enabled);
	}
	MUTEX_UNLOCK(&ms->image_mtx);
    }
    return PP_SUC;
}

u_int
pp_usb_get_ms_count(void)
{
    return PP_FEAT_USB_MASS_STORAGE_NO;
}
#endif /* PP_FEAT_MASS_STORAGE */

static int reset_device(int usb)
{
    if (usbfds[usb] == -1) return 1;
    if (ioctl(usbfds[usb], LARAUSBRESET, NULL) == -1) {
	pp_log("USB device #%d reset failed\n", usb);
	return 1;
    }
    return 0;
}

int
pp_usb_reset_device(void)
{
    int ret = 0;
    if (usbfds[dev_kbd] > 0) {
	ret |= reset_device(dev_kbd);
    }
    if (usbfds[dev_mouse] > 0 && dev_mouse != dev_kbd) {
	ret |= reset_device(dev_mouse);
    }
    return ret;
}

int pp_usb_get_led_state(void)
{
    uint8_t led_state = 0;
    if (usbfds[dev_kbd] == -1) return -1;
    if (ioctl(usbfds[dev_kbd], LARAUSBGETLEDSTATE, &led_state) < 0) {
        pp_log("get led state failed\n");
        return -1;
    }
    return (int)led_state;
}

#ifdef PP_FEAT_MASS_STORAGE
int
pp_usb_enable_setup_proto(int enabled)
{
#ifdef PP_FEAT_USB
    uint8_t en = enabled ? 1 : 0;
    int i;
    int ret = 0;
    for (i = 0; i < PP_FEAT_NUM_USB_DEVICES; i++) {
	if (usbfds[i] > 0) {
	    ret |= ioctl(usbfds[i], LARAUSBENABLESETUPPROTO, &en);
	}
    }
    return ret;
#else
    (void)enabled;
    return 0;
#endif
}
#endif /* PP_FEAT_MASS_STORAGE */
    
static int
set_device(int usb, usb_device_t dev, usb_device_kbd_type_t keyboard_type, usb_device_mouse_type_t mouse_type, usb_device_ms_t mass_storage[PP_FEAT_USB_MASS_STORAGE_NO])
{
#ifdef PP_FEAT_USB
    lara_usb_device_t d;
    int ret = -1;
    int fd = usbfds[usb];

#ifdef PP_FEAT_MASS_STORAGE
    int i;
    for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
    	MUTEX_LOCK(&mass_storages[i].image_mtx);
    }
#else
    (void)mass_storage;
#endif /* PP_FEAT_MASS_STORAGE */

    d.dev = dev;
    d.mouse_type = mouse_type;
    d.keyboard_type = keyboard_type;
    
#ifdef PP_FEAT_MASS_STORAGE
    for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	d.mass_storage[i] = mass_storage[i];
    }
#endif /* PP_FEAT_MASS_STORAGE */

    if (fd == -1) goto bail;
    if (ioctl(fd, LARAUSBSETDEVICE, &d) == -1) {
	pp_log_err("%s(): ioctl(LARAUSBSETDEVICE) failed\n", ___F);
	goto bail;
    }
    ret = 0;
 bail:
#ifdef PP_FEAT_MASS_STORAGE
    for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
    	MUTEX_UNLOCK(&mass_storages[i].image_mtx);
    }
#endif /* PP_FEAT_MASS_STORAGE */

    return ret;
#else
    (void)usb;
    (void)dev;
    (void)keyboard_type;
    (void)mouse_type;
    (void)mass_storage;
    return -1;
#endif
}

int pp_usb_set_km(usb_device_kbd_type_t keyboard_type, usb_device_mouse_type_t mouse_type)
{
    int i;
    int ret = 0;
    usb_device_ms_t ms[PP_FEAT_USB_MASS_STORAGE_NO];
    for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	ms[i] = usb_device_ms_unchanged;
    }
    if (dev_kbd == dev_mouse) {
	ret |= set_device(dev_kbd, usb_device_unchanged, keyboard_type, mouse_type, ms);
    } else {
	ret |= set_device(dev_kbd, usb_device_unchanged, keyboard_type, peppercon_mouse_type_unchanged, ms);
	ret |= set_device(dev_mouse, usb_device_unchanged, usb_device_kbd_unchanged, mouse_type, ms);
    }
    return ret;
}

#ifdef PP_FEAT_MASS_STORAGE
static int
usb_set_mass_storage(u_int ms_index, usb_device_ms_t state)
{
    int i;
    usb_device_ms_t ms[PP_FEAT_USB_MASS_STORAGE_NO];

    printf("usb_set_mass_storage: index: %d, state: %d\n", ms_index, state);
    mass_storages[ms_index].act_state = state;

    for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	ms[i] = usb_device_ms_unchanged;
    }
    ms[dev_ms[ms_index].index] = state;
    return set_device(dev_ms[ms_index].usb, usb_device_unchanged, usb_device_kbd_unchanged, peppercon_mouse_type_unchanged, ms);
}

pp_usb_image_type_t
pp_usb_ms_get_image_type(u_int ms_index)
{
    if (ms_index >= PP_FEAT_USB_MASS_STORAGE_NO) return -1;
    return mass_storages[ms_index].act_image;
}

static void
usb_check_file(u_int ms_index, usb_device_type_t new_type, usb_device_ms_t state)
{
    printf("newtype: %d, mass_storages[%d].act_image_type: %d\n", new_type, ms_index, mass_storages[ms_index].act_image_type);
    if (state != usb_device_ms_unchanged && state != mass_storages[ms_index].act_state) {
	usb_set_mass_storage(ms_index, state);
    } else if (new_type != mass_storages[ms_index].act_image_type) {
	if (new_type != PP_USB_IMAGE_TYPE_NONE || mass_storages[ms_index].act_image_type == PP_USB_IMAGE_TYPE_FIXED) {
	    reset_device(dev_ms[ms_index].usb);
	}
    }
    if (state != usb_device_unchanged) {
	mass_storages[ms_index].act_state = state;
    }
    if (new_type != PP_USB_IMAGE_TYPE_NONE || mass_storages[ms_index].act_image_type == PP_USB_IMAGE_TYPE_FIXED) {
	mass_storages[ms_index].act_image_type = new_type;
    }
}

static usb_device_type_t
get_image_type_from_file(int fd)
{
    int rv;
#define TAGLEN 5
#define ISO9660_MAGIC_POS 0x8001
#define FLOPPY_SIZE_TRESHOLD (2880*1024)
    const char isotag[TAGLEN] = "CD001";
    char imagetag[TAGLEN];

    struct stat s;
    
    rv = fstat(fd, &s);
    if (rv != 0) {
	pp_log_err("can't fstat image file");
	goto fail;
    }
    if (s.st_size < ISO9660_MAGIC_POS + TAGLEN) {
	// smaller than 9k, assume floppy
	return PP_USB_IMAGE_TYPE_FLOPPY;
    }
    rv = lseek(fd, ISO9660_MAGIC_POS, SEEK_SET);
    if (rv == -1) {
	pp_log_err("can't lseek image file");
	goto fail;
    }
    memset(imagetag, 0, sizeof(imagetag));
    rv = read(fd, imagetag, sizeof(imagetag));
    if (rv == -1) {
	pp_log_err("can't read image file for magic");
	goto fail;
    }
    if (memcmp(imagetag, isotag, sizeof(imagetag)) == 0) {
	// iso TAG found
	pp_log("ISO image\n");
	return PP_USB_IMAGE_TYPE_ISO;
    }
    pp_log("no ISO image\n");

    // it is definitively not an ISO image
    if (s.st_size <= FLOPPY_SIZE_TRESHOLD) {
	// assume floppy
	return PP_USB_IMAGE_TYPE_FLOPPY;
    }
 fail:
    // and fall through
    return PP_USB_IMAGE_TYPE_REMOVABLE;
#undef FLOPPY_SIZE_TRESHOLD
#undef ISO9660_MAGIC_POS
#undef TAGLEN
}

#ifndef MSP_USE_ADAPTER
/*
 * (re)mmap the ms->msbofd file descriptor so that the block specified
 * by offset/length can be accessed.
 *
 * NOTE: At the first call to this function ms->mmappped_file_data must
 *       be NULL and ms->mmapped_file_size and ms->mmap_flags need to
 *       be set to correct values.
 */
static int
mmap_file(mass_storage_t *ms, off_t offset, size_t length)
{
    off_t res_offset;
    size_t res_length;

    if (length > MMAP_BLOCKSIZE) return PP_ERR;
    if (offset + length > ms->mmapped_file_length) return PP_ERR;

    res_length = min(ms->mmapped_file_length, (off_t)MMAP_BLOCKSIZE);
    res_offset = min(offset, ms->mmapped_file_length-res_length) & ms->page_size_mask;

    if (ms->mmapped_file_data) {
	if ( offset >= ms->mmapped_file_offset &&
	     offset + length <= ms->mmapped_file_offset + res_length ) {
	    /* nothing to be done */
	    return PP_SUC;
	}
	munmap(ms->mmapped_file_data, res_length);
    }

    ms->mmapped_file_data = mmap(ms->mmapped_file_data, res_length,
		    ms->mmap_flags, MAP_SHARED, ms->msbofd, res_offset);
    ms->mmapped_file_offset = res_offset;
    return ms->mmapped_file_data == MAP_FAILED? PP_ERR : PP_SUC;
}
#endif

/*
 * set_file must be called with image_lock locked!
 *
 * NOTE: The filename must not necessarily exist.
 *	 It is just used for propagation to frontend!
 */
static int
set_file(u_int ms_index, int fd, const char * filename, unsigned int readonly)
{
    const char * fn = ___F;
    struct stat statinfo;
    off_t size;
    uint32_t suggested_blocksize;
    file_info_t file_info;
    int usbfd;

    mass_storage_t * ms;
    
    if (ms_index >= PP_FEAT_USB_MASS_STORAGE_NO) return -1;
    usbfd = usbfds[dev_ms[ms_index].usb];
    
    ms = &mass_storages[ms_index];

    assert(fd >= 0);
    assert(ms->msbofd == -1);
    assert(ms->mmapped_file_data == NULL && ms->mmapped_file_length == 0);

    if (usbfd == -1) goto bail;
    
    memset(&file_info, 0, sizeof(file_info));
    
    ms->msbofd = fd;

    snprintf(file_info.filename, sizeof(file_info.filename), "%s", filename);
    file_info.dev_no = dev_ms[ms_index].index;
    file_info.image_type = get_image_type_from_file(ms->msbofd);
    file_info.readonly = (uint8_t) readonly;

    switch (file_info.image_type) {
      case PP_USB_IMAGE_TYPE_REMOVABLE:
      case PP_USB_IMAGE_TYPE_FIXED:
	  suggested_blocksize = 512;
	  break;
      case PP_USB_IMAGE_TYPE_ISO:
	  suggested_blocksize = 2048; /* we emulate a CDROM */
	  break;
      default:
	  suggested_blocksize = 512; /* fall back */
	  break;
    }
    
    if (fstat(ms->msbofd, &statinfo) == -1) {
	file_info.image_type = PP_USB_IMAGE_TYPE_NONE;
	goto bail;
    }

    size = statinfo.st_size;

    if (size % suggested_blocksize) {
	pp_log("%s(): file is not a multiple of blocksize: %d\n", fn, suggested_blocksize);
	file_info.image_type = PP_USB_IMAGE_TYPE_NONE;
	goto bail;
    }

#ifndef MSP_USE_ADAPTER
    ms->mmapped_file_length = size;
    ms->mmap_flags = readonly? PROT_READ : PROT_READ|PROT_WRITE;
    if ( PP_FAILED(mmap_file(ms, 0, min((uint32_t)size, (uint32_t)MMAP_BLOCKSIZE))) ) {
	pp_log_err("%s(): Can't mmap '%s', giving up", fn, filename);
	file_info.image_type = PP_USB_IMAGE_TYPE_NONE;
	goto bail;
    }
#endif

    file_info.blocklen = suggested_blocksize;
    file_info.lba = (size / suggested_blocksize) - 1;

    /* enable mass storage and reenumerate if image type has changed */
    usb_check_file(ms_index, file_info.image_type, usb_device_ms_enabled);
    
    //pp_log("file_info.blocklen = %x, lba: %x\n",
    //	   file_info.blocklen, file_info.lba);
    if (ioctl(usbfd, LARAUSBSETFILE, &file_info) == -1) {
	pp_log_err("%s(): ioctl(LARAUSBSETFILE) failed", fn);
	file_info.blocklen = 0;
	file_info.lba = 0;
	file_info.image_type = PP_USB_IMAGE_TYPE_NONE;
	/* disable mass storage if necessary */
	usb_set_mass_storage(ms_index,
			     usb_ms_disabled_if_no_image() ? usb_device_ms_disabled : usb_device_ms_enabled);
    }
    
 bail:
    if (file_info.image_type == PP_USB_IMAGE_TYPE_NONE) {
	if (ms->mmapped_file_data) {
	    if (ms->mmapped_file_data != MAP_FAILED) {
		munmap(ms->mmapped_file_data,
			min(ms->mmapped_file_length, (off_t)MMAP_BLOCKSIZE));
	    }
	    ms->mmapped_file_data = NULL;
	    ms->mmapped_file_length = 0;
	}
	if (ms->msbofd != -1) {
	    close(ms->msbofd);
	    ms->msbofd = -1;
	}
	return -1;
    } else {
	return 0;
    }
}

int
pp_usb_ms_set_image(u_int ms_index, int fd, const char* path, size_t size, const char* name, int * error)
{
    const char * fn = ___F;
    file_info_t file_info;
    mass_storage_t * ms;
    int ret = -1;
    int usbfd;
    
    if (ms_index >= PP_FEAT_USB_MASS_STORAGE_NO) return -1;
    usbfd = usbfds[dev_ms[ms_index].usb];

    ms = &mass_storages[ms_index];

    assert(path && size > 0);

    memset(&file_info, 0, sizeof(file_info));

    MUTEX_LOCK(&ms->image_mtx);

    if (usbfd == -1) goto bail;
    
    if (ms->act_image != PP_USB_MS_IMAGE_NONE) {
	if (ms->act_image == PP_USB_MS_IMAGE_MEM) {
	    // there is another memory image, release it
	    usb_ms_release_data(ms_index);
	} else {
	    if (error) *error = PP_USB_ERR_IMAGE_IN_USE;
	    goto bail;
	}
    }

    pp_log("%s(): set image file '%s', size: %u\n", fn, path, size);

    if (set_file(ms_index, fd, path, 0) == 0) {
	ms->act_image = PP_USB_MS_IMAGE_MEM;
	ms->g_name = strdup(name);
	ms->img_size = size;
	ret = 0;
    } else {
	if (error) *error = PP_USB_ERR_IMAGE_ACCESS_ERROR;
	ms->act_image = PP_USB_MS_IMAGE_NONE;	
    }
    
 bail:
    MUTEX_UNLOCK(&ms->image_mtx);
    return ret;
}

static int call_smbmount(const char *host, const char *share, const char *user, const char *pw, const char *smbmount_path)
{
    int ret = -1;
    int status, pid;
    __sighandler_t save_quit, save_int, save_chld;
    struct rlimit rlim;
    int i, max_fd;

    // create mountpoint if necessary
    mkdir(smbmount_path, 0700);

    save_quit = signal(SIGQUIT, SIG_IGN);
    save_int = signal(SIGINT, SIG_IGN);
    save_chld = signal(SIGCHLD, SIG_DFL);

    pid = fork();
    if (pid == 0) {
	/* child */
	/* get max fd nr */
	if (!getrlimit(RLIMIT_NOFILE, &rlim)) {
	    max_fd = rlim.rlim_max;
	} else {
	    max_fd = 256;
	}
	for (i = 3; i < max_fd; i++) {
	    close(i);
	}
	
	char smb_path[512];
	signal(SIGQUIT, SIG_DFL);
	signal(SIGINT, SIG_DFL);
	signal(SIGCHLD, SIG_DFL);
	snprintf(smb_path, sizeof(smb_path), "//%s/%s", host, share);
	clearenv();
	if (user && *user) {
	    char username[32];
	    snprintf(username, sizeof(username), "username=%s", user);
	    setenv("PASSWD", pw, 1);
	    execlp("smbmount", "smbmount", smb_path, smbmount_path, "-o", username, (char*)NULL);
	} else {
	    printf("%s %s %s %s %s %s\n", "smbmount", "smbmount", smb_path, smbmount_path, "-o", "username=guest,guest");
	    execlp("smbmount", "smbmount", smb_path, smbmount_path, "-o", "username=guest,guest", (char*)NULL);
	}
	exit(255);
    } else if (pid > 0) {
	/* parent */
	if (TEMP_FAILURE_RETRY(waitpid(pid, &status, 0)) == pid) ret = status;
    }

    signal(SIGQUIT, save_quit);
    signal(SIGINT, save_int);
    signal(SIGCHLD, save_chld);
    return ret;
}

int
pp_usb_ms_set_smb_image(u_int ms_index, const char* host, const char* share, const char* rpath, const char* user, const char* pw, int * error)
{
    const char * fn = ___F;
    char cmd_line[32];
    int rv, ret = -1;
    char * path = strndupa(rpath, 512);
    mass_storage_t * ms;
    int initial = 1;
    int fd = -1;
    int usbfd;
   
    printf("set smb_image: %s, %s, %s\n", host, share, path);
    
    if (ms_index >= PP_FEAT_USB_MASS_STORAGE_NO) return -1;
    usbfd = usbfds[dev_ms[ms_index].usb];
    
    ms = &mass_storages[ms_index];
    
    MUTEX_LOCK(&ms->image_mtx);

    if (ms->act_image != PP_USB_MS_IMAGE_NONE) {
	if (ms->act_image == PP_USB_MS_IMAGE_SMB) {
	    // there is another smb image, release it
	    usb_ms_release_data(ms_index);
	    initial = 0;
	} else {
	    if (error) *error = PP_USB_ERR_IMAGE_IN_USE;
	    goto bail;
	}
    }
    
    // replace \ with /
    if (path) {
	int i = 0;
	while (path[i++]) {
	    if (path[i-1] == '\\') {
		path[i-1] = '/';
	    }
	}
    }
    
    pp_log("%s(): set samba image, host: %s, share: %s, path: %s, user: %s, pw: %s, smbmount_path: %s\n", fn,
	   host, share, path, user, pw, ms->smbmount_path);
    rv = call_smbmount(host, share, user, pw, ms->smbmount_path);

    if (rv == 0) {
	char file_path[256];
	snprintf(file_path, sizeof(file_path), "/%s/%s", ms->smbmount_path, path);
	if ((fd = open(file_path, O_RDONLY)) < 0) {
	    pp_log_err("%s(): open(%s) failed", ___F, file_path);
	    if (error) *error = PP_USB_ERR_IMAGE_NOT_FOUND;
	    ms->act_image = PP_USB_MS_IMAGE_NONE;
	} else {
	    rv = set_file(ms_index, fd, file_path, 1);
	    if (rv == 0) {
		struct stat st;
		ms->g_host = strdup(host);
		ms->g_path = strdup(path);
		ms->g_share = strdup(share);
		ms->g_user = strdup(user);
		ms->act_image = PP_USB_MS_IMAGE_SMB;
		if (fstat(fd, &st) == 0) {
		    ms->img_size = st.st_size;
		} else {
		    ms->img_size = 0;
		}
		ret = 0;
	    } else {
		close(fd);
		if (error) *error = PP_USB_ERR_IMAGE_ACCESS_ERROR;
		ms->act_image = PP_USB_MS_IMAGE_NONE;
	    }
	}
    } else {
	if (error) *error = PP_USB_ERR_SHARE_NOT_FOUND;
	ms->act_image = PP_USB_MS_IMAGE_NONE;
    }

 bail:
    if (ms->act_image == PP_USB_MS_IMAGE_NONE) {
	/* sanity - may even fail */
	if (!initial) {
    	    file_info_t file_info;
    	    file_info.blocklen = 0;
    	    file_info.lba = 0;
    	    file_info.image_type = PP_USB_IMAGE_TYPE_NONE;
    	    if (ioctl(usbfd, LARAUSBSETFILE, &file_info) == -1) {
		pp_log_err("%s(): ioctl(LARAUSBSETFILE) failed", fn);
	    }
	    /* disable mass storage if necessary */
	    usb_set_mass_storage(ms_index,
			      usb_ms_disabled_if_no_image() ? usb_device_ms_disabled : usb_device_ms_enabled);
	}

	snprintf(cmd_line, sizeof(cmd_line), "umount %s", ms->smbmount_path);
	pp_system(cmd_line);
	free(ms->g_host); ms->g_host = NULL;
	free(ms->g_path); ms->g_path = NULL;
	free(ms->g_share); ms->g_share = NULL;
	free(ms->g_user); ms->g_user = NULL;
    }
    MUTEX_UNLOCK(&ms->image_mtx);
    return ret;
}

int
usb_ms_set_msp(u_int ms_index, int *error, int initial)
{
    const char * fn = ___F;
    uint32_t suggested_blocksize;
    file_info_t file_info;
    mass_storage_t * ms;
    int success = 0;
    int usbfd;
    
    if (ms_index >= PP_FEAT_USB_MASS_STORAGE_NO) {
	pp_log("%s(): ms_index (%d) >= max_mass_storage_no (%d)", fn, ms_index, PP_FEAT_USB_MASS_STORAGE_NO);
	return -1;
    }
    usbfd = usbfds[dev_ms[ms_index].usb];
    
    ms = &mass_storages[ms_index];
    suggested_blocksize = msp_data[ms_index].msp_sector_size;

    memset(&file_info, 0, sizeof(file_info));
    
    MUTEX_LOCK(&ms->image_mtx);

    if (usbfd == -1) {
    	goto bail;
    }
    
    if (initial && ms->act_image != PP_USB_MS_IMAGE_NONE) {
	if (error) *error = PP_USB_ERR_IMAGE_IN_USE;
	goto bail;
    }

    pp_log("%s(): set msp image for %d\n", fn, ms_index);
    snprintf(file_info.filename, sizeof(file_info.filename), "%s_%d", MSP_ADAPTER_FRONTEND_FILE_BASE, ms_index);
    file_info.dev_no = dev_ms[ms_index].index;
    file_info.blocklen = suggested_blocksize;
    file_info.lba = msp_data[ms_index].msp_last_sectorno;
    file_info.image_type = usb_msp_get_image_type(ms_index);
    file_info.readonly = msp_data[ms_index].msp_disc_ro;

    /* let the msp backend setup its file information */
    if (!initial && usb_msp_cleanup_file_info(ms_index) != 0) {
        *error = PP_USB_ERR_INTERNAL_ERROR;
        goto bail;
    }
    if (file_info.image_type != PP_USB_MS_IMAGE_NONE &&
        usb_msp_setup_file_info(ms_index) != 0) {
	*error = PP_USB_ERR_INTERNAL_ERROR;
	goto bail;
    }
    
    /* enable mass storage */
    usb_check_file(ms_index, file_info.image_type, usb_device_ms_enabled);
    
    //pp_log("file_info.blocklen = %x, lba: %x, enable: %x\n",
    //	   file_info.blocklen, file_info.lba, file_info.enable);
    if (ioctl(usbfd, LARAUSBSETFILE, &file_info) == -1) {
	pp_log_err("%s(): ioctl(LARAUSBSETFILE) failed", fn);
	if (error) *error = PP_USB_ERR_INTERNAL_ERROR;
	file_info.blocklen = 0;
	file_info.lba = 0;
	file_info.image_type = PP_USB_IMAGE_TYPE_NONE;

	/* disable mass storage if necessary */
	usb_set_mass_storage(ms_index,
			     usb_ms_disabled_if_no_image() ? usb_device_ms_disabled : usb_device_ms_enabled);
	goto bail;
    } 

    success = 1;
    ms->act_image = PP_USB_MS_IMAGE_MSP;
    ms->img_size = (off_t)msp_data[ms_index].msp_last_sectorno * msp_data[ms_index].msp_sector_size;

 bail:
    MUTEX_UNLOCK(&ms->image_mtx);
    return success ? 0 : -1;
}

static int
usb_ms_release_data(u_int ms_index)
{
    char cmd_line[32];
    mass_storage_t * ms;
    file_info_t file_info;
    int ret = 0;
    pp_usb_image_type_t old_image;
    int usbfd = usbfds[dev_ms[ms_index].usb];
	
    *cmd_line = '\0';
    ms = &mass_storages[ms_index];

    /* FIXME(miba): this is not optimal anymore, rearrange! */
    switch (ms->act_image) {
      case PP_USB_MS_IMAGE_SMB:
	  snprintf(cmd_line, sizeof(cmd_line), "umount %s", ms->smbmount_path);
	  free(ms->g_host); ms->g_host = NULL;
	  free(ms->g_path); ms->g_path = NULL;
	  free(ms->g_share); ms->g_share = NULL;
	  free(ms->g_user); ms->g_user = NULL;
	  /* fall through! */
      case PP_USB_MS_IMAGE_MEM:
	  if (ms->mmapped_file_data) {
	      munmap(ms->mmapped_file_data,
		      min(ms->mmapped_file_length, (off_t)MMAP_BLOCKSIZE));
	      ms->mmapped_file_data = NULL;
	  }
	  if (ms->msbofd != -1) {
	      close(ms->msbofd);
	      ms->msbofd = -1;
	  }
	  ms->mmapped_file_length = 0;
	  free(ms->g_name); ms->g_name = NULL;
	  /* fall through! */
      case PP_USB_MS_IMAGE_MSP:
	  old_image = ms->act_image;
	  if (ms->act_image == PP_USB_MS_IMAGE_MSP) {
	      usb_msp_disconnect(ms_index);
	  }
	  file_info.blocklen = 0;
	  file_info.lba = 0;
	  file_info.image_type = PP_USB_IMAGE_TYPE_NONE;
	  file_info.readonly = 1;
	  file_info.dev_no = dev_ms[ms_index].index;

	  if (ioctl(usbfd, LARAUSBSETFILE, &file_info) == -1) {
	      pp_log_err("%s(): ioctl(LARAUSBSETFILE) failed", ___F);
	      ret = -1;
	  }
	  if (ms->msbofd != -1) {
	      close(ms->msbofd);
	      ms->msbofd = -1;
	  } 
	  /* do an unmount after telling the usb driver to release the image */
	  if (strlen(cmd_line) > 0) pp_system(cmd_line);

	  ms->act_image = PP_USB_MS_IMAGE_NONE;
	  ms->img_size = 0;

	  /* let the msp backend cleanup its file information */
	  if (old_image == PP_USB_MS_IMAGE_MSP) {
	      usb_msp_cleanup_file_info(ms_index);
	  }

	  break;
      case PP_USB_MS_IMAGE_NONE:
	  break;
      default:
	  assert(0);
    }
    return ret;
}

int
pp_usb_ms_unset_image(u_int ms_index, int * error)
{
    mass_storage_t * ms;
    int ret = -1;
    int usbfd;
    
    if (ms_index >= PP_FEAT_USB_MASS_STORAGE_NO) return -1;
    usbfd = usbfds[dev_ms[ms_index].usb];
    
    ms = &mass_storages[ms_index];

    MUTEX_LOCK(&ms->image_mtx);

    if (usbfd < 0 || (ret = usb_ms_release_data(ms_index)) != 0) {
	if (error) *error = PP_USB_ERR_INTERNAL_ERROR;
    }

    /* disable mass storage if necessary */
    if (usb_ms_disabled_if_no_image()) {
	usb_check_file(ms_index, PP_USB_IMAGE_TYPE_NONE, usb_device_ms_disabled);
    } else {
	usb_check_file(ms_index, PP_USB_IMAGE_TYPE_NONE, usb_device_ms_enabled);
    }

    MUTEX_UNLOCK(&ms->image_mtx);

    return ret;
}

const char *
pp_usb_ms_get_image_info(u_int ms_index, const char* type)
{
    const char * ret = NULL;
    mass_storage_t * ms;
    
    if (ms_index >= PP_FEAT_USB_MASS_STORAGE_NO) return "";
    
    ms = &mass_storages[ms_index];

    assert(type);
    MUTEX_LOCK(&ms->image_mtx);
    if (!strcmp(type, "name")) {
	ret = ms->g_name;
    } else if (!strcmp(type, "host")) {
	ret = ms->g_host;
    } else if (!strcmp(type, "share")) {
	ret = ms->g_share;
    } else if (!strcmp(type, "path")) {
	ret = ms->g_path;
    } else if (!strcmp(type, "user")) {
	ret = ms->g_user;
    } else if (!strcmp(type, "redir_ip")) {
    	ret = usb_msp_get_connected_ip(ms_index);
    } else if (!strcmp(type, "redir_ro")) {
    	ret = msp_data[ms_index].msp_disc_ro ? "read-only" : "write support";
    } else if (!strcmp(type, "redir_type")) {
    	ret = usb_msp_get_disc_type(ms_index);
    } else if (!strcmp(type, "type")) {
	switch (ms->act_image) {
	  case PP_USB_MS_IMAGE_MEM:
	      ret = "file";
	  case PP_USB_MS_IMAGE_SMB:
	      ret = "smb";
	  case PP_USB_MS_IMAGE_MSP:
	      ret = "msp";
	  default:
	      ret = "none";
	}
    }
    MUTEX_UNLOCK(&ms->image_mtx);

    return ret;
}

off_t
pp_usb_ms_get_image_size(u_int ms_index)
{
    return mass_storages[ms_index].img_size;
}

static int
init_mass_storages(void)
{
    int i;
    
    for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
    	mass_storage_t * ms = &mass_storages[i];
    	pthread_mutexattr_t mutex_attr;
    	int ret;
    	
    	if ((ret = pthread_mutexattr_init(&mutex_attr))) {
    	    return ret;
    	}
    	if ((ret = pthread_mutexattr_settype(&mutex_attr, PTHREAD_MUTEX_RECURSIVE_NP))) {
    	    return ret;
    	}
    	if ((ret = pthread_mutex_init(&ms->image_mtx, &mutex_attr))) {
    	    return ret;
    	}
    	
    	ms->msbofd = -1;
    	ms->page_size_mask = ~(getpagesize() - 1);
    	ms->mmapped_file_length = 0;
    	ms->mmapped_file_data = NULL;
    	asprintf(&ms->smbmount_path, "/smb-%d", i);
    	ms->act_image = PP_USB_MS_IMAGE_NONE;
    	ms->act_image_type = PP_USB_IMAGE_TYPE_NONE;
    	ms->act_state = usb_ms_disabled_if_no_image()? usb_device_ms_disabled : usb_device_ms_enabled;
    	ms->g_path = NULL;
    	ms->g_host = NULL;
    	ms->g_share = NULL;
    	ms->g_user = NULL;
    }
    
    return 0;
}

static void deinit_mass_storages(void)
{
    int i;

    for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	mass_storage_t * ms = &mass_storages[i];
	pthread_mutex_destroy(&ms->image_mtx);
	free(ms->smbmount_path);
	free(ms->g_path);
	free(ms->g_host);
	free(ms->g_share);
	free(ms->g_user);
	memset(ms, 0, sizeof(mass_storage_t));
    }
}

#endif /* PP_FEAT_MASS_STORAGE */

int
pp_usb_init(void)
{
    if (!initialized) {

#ifdef PP_FEAT_MASS_STORAGE
	if (init_mass_storages()) {
	    pp_log_err("%s(): Can't initialize mass storage structures", ___F);
	    goto bail;
	}

	if (usb_msp_init()) {
	    pp_log_err("%s(): Can't initialize msp protocol library", ___F);
	    goto bail;
	}
#endif /* PP_FEAT_MASS_STORAGE */
	
	assert(errno_base == 0);

	// register errnos for myself, settings, acl and profile
	errno_base = pp_register_errnos(sizeof(pp_usb_errors) /
					sizeof(*pp_usb_errors),
					get_error_string);

	int i;
	for (i = 0; i < PP_FEAT_NUM_USB_DEVICES; i++) {
	    char dev[20];
	    char * input_type = NULL;
	    usb_device_mouse_type_t mouse_type = peppercon_mouse_type_none;
	    usb_device_kbd_type_t keyboard_type = usb_device_kbd_none;
	    usb_device_ms_t ms[PP_FEAT_USB_MASS_STORAGE_NO];

	    snprintf(dev, 20, "/dev/lara_usb-%d", i);
	    usbfds[i] = open(dev, O_RDWR);
	    if (usbfds[i] == -1) {
		pp_log_err("%s(): Can't open %s", ___F, dev);
	    }

	    /*
	     * This is the first time the usb device can be initialised to
	     * avoid a second initialising with different parameters by the
	     * km lib, we'll do our best to configure it according to the
	     * options
	     */

	    if (i == dev_kbd) {
		keyboard_type = usb_device_kbd_enabled;
	    }

	    if (i == dev_mouse) {
		pp_cfg_get(&input_type, "input_type");
		if (input_type) {
		    if (!strcmp(input_type, PP_CD_INPUT_TYPE_PS2_STR)) {
			mouse_type = peppercon_mouse_type_none;
		    } else {
			// usb or auto
			char * usb_type = NULL;

			pp_cfg_get(&usb_type, "usb_type");
			if (usb_type && !strcmp(usb_type, PP_CD_USB_TYPE_ABSOLUTE_STR)) {
			    mouse_type = peppercon_mouse_type_absolute;
			} else {
			    mouse_type = peppercon_mouse_type_relative;
			}
			free(usb_type);
		    }
		    free(input_type);
		} else {
		    // config broken - should not happen, assume PS/2
		    mouse_type = peppercon_mouse_type_none;
		}
	    }

#ifdef PP_FEAT_MASS_STORAGE
	    {
		int j;
		for (j = 0; j < PP_FEAT_USB_MASS_STORAGE_NO; j++) {
		    ms[j] = usb_device_ms_disabled;
		}
		for (j = 0; j < PP_FEAT_USB_MASS_STORAGE_NO; j++) {
		    if (i == dev_ms[j].usb) {
			int idx = dev_ms[j].index;
			ms[idx] = usb_ms_disabled_if_no_image() ? usb_device_ms_disabled : usb_device_ms_enabled;
		    }
		}
	    }
#endif /* PP_FEAT_MASS_STORAGE */

#ifdef PRODUCT_RIPCKIMXN
	    set_device(i, usb_device_ripcbelkin, usb_device_kbd_none, peppercon_mouse_type_none, ms);
#else
	    set_device(i, usb_device_peppercon, keyboard_type, mouse_type, ms);
#endif
	}

#ifdef PP_FEAT_MASS_STORAGE
	pp_cfg_add_change_listener(disable_no_image_change_cb, DISABLE_IF_NO_IMAGE_KEY);
#endif /* PP_FEAT_MASS_STORAGE */

	initialized = 1;
    }
#ifdef PP_FEAT_MASS_STORAGE
 bail:
#endif /* PP_FEAT_MASS_STORAGE */
    return 0;
}

void
pp_usb_cleanup(void)
{
    /* FIXME: join thread */
    int i;
#ifdef PP_FEAT_MASS_STORAGE
    deinit_mass_storages();
#endif
    for (i = 0; i < PP_FEAT_NUM_USB_DEVICES; i++) {
	if (usbfds[i] != -1) {
	    close(usbfds[i]);
	    usbfds[i]= -1;
	}
    }
    if (errno_base > 0) {
	pp_unregister_errnos(errno_base);
	errno_base = 0;
    }
    initialized = 0;
}

#ifdef PP_FEAT_MASS_STORAGE
int
pp_usb_start_ms_bo_thread(void)
{
    if (eric_pthread_create(&usb_thread, 0, 65536, usb_ms_bo_thread_func, NULL)) {
	return PP_ERR;
    }
    return PP_SUC;
}
#endif /* PP_FEAT_MASS_STORAGE */

/* small help function to print out some bytes (hexadecimal encoded) */
#if 0
static void
debug_show_packet(const char *packet, u_int size)
{
    u_int i;
    
    for (i = 0; i < size; ++i) {
	if (!(i % 16)) printf("%04x: ", i);
	printf("0x%02x ", packet[i]);
	if (!((i+1) % 16)) printf("\n");
    }
    printf("\n");
}
#endif /* 0 */

int
pp_usb_is_possible(void)
{
    return (usbfds[dev_kbd] == -1 && usbfds[dev_mouse] != -1) ? 0 : 1;
}

int
pp_usb_send_data(u_int8_t channel, u_int8_t type, const void * data, size_t length)
{
    cmd_struct_t cmd;
    u_int16_t len;
    cmd.command.channel = channel;
    cmd.command.type = type;
    int fd;

    len = length > PP_USB_MAX_HID_DATA_LEN ? PP_USB_MAX_HID_DATA_LEN : length;

    cmd.command.length = len;
    memcpy(&cmd.data, data, len);
    fd = type == 1 ? usbfds[dev_kbd] : usbfds[dev_mouse];
    if (fd == -1) return -1;
    if (ioctl(fd, LARAUSBSENDDATA, &cmd) < 0) {
	pp_log_err("%s(): ioctl(LARAUSBSENDDATA) failed", ___F);
	return -1;
    }
    return 0;
}

usb_device_state_t
pp_usb_get_device_state(void)
{
    usb_device_state_t state_kbd, state_mouse;
    if (usbfds[dev_kbd] < 0 || usbfds[dev_mouse] < 0) return PP_USB_DEVSTAT_ERROR;

    if (ioctl(usbfds[dev_kbd], LARAUSBGETDEVICESTATE, &state_kbd) < 0) {
	pp_log("get device state failed\n");
	return PP_USB_DEVSTAT_ERROR;
    }

    if (usbfds[dev_mouse] == usbfds[dev_kbd]) {
	state_mouse = state_kbd;
    } else if (ioctl(usbfds[dev_mouse], LARAUSBGETDEVICESTATE, &state_mouse) < 0) {
	pp_log("get device state failed\n");
	return PP_USB_DEVSTAT_ERROR;
    }

    if (state_kbd == PP_USB_DEVSTAT_CONFIGURED
	    || state_mouse == PP_USB_DEVSTAT_CONFIGURED) {
	return PP_USB_DEVSTAT_CONFIGURED;
    }
    return PP_USB_DEVSTAT_SUSPENDED;
}

#ifdef USB_DEBUG
static void
dump_sector(file_response_t *file_response)
{
    int i;
    printf("sector data:\n");
    if(file_response->response == FILE_DATA_OK) {
    	for(i = 0; i < 30; i++) {
    	    printf("0x%02x ", file_response->data[i]);
    	}
    	printf("\n");
    } else {
    	printf("bad data!\n");
    }
}
#endif

#ifdef PP_FEAT_MASS_STORAGE
static void *
usb_ms_bo_thread_func(void* data UNUSED)
{
#ifdef MSP_USE_ADAPTER

    /* New kernel module structure (ARM only, at the moment):
       This thread's only purpose is receiving setup protocol requests
       from the scsi handler and sending back responses */
    struct sockaddr_un sa;
    struct iovec iov;
    struct msghdr msg;

    unlink("/tmp/setup-proto.socket");
    int sock = socket(AF_UNIX, SOCK_DGRAM, 0);
    sa.sun_family = AF_UNIX;
    strcpy(sa.sun_path, "/tmp/setup-proto.socket");
    bind(sock, (struct sockaddr *)&sa, sizeof(sa));

    memset(&msg, 0, sizeof(msg));
    msg.msg_iov = &iov;
    msg.msg_iovlen = 1;
    msg.msg_name = &sa;
    msg.msg_namelen = sizeof(sa);

    for (;;) { /* FIXME: Allow thread termination */
	int r;
	int dev_no = 0; // who cares?
	mass_storage_t * ms = &mass_storages[dev_no];
	
	iov.iov_base = ms->pp_setup_req_packet;
	iov.iov_len = PP_SP_REQ_MTU;

	r = recvmsg(sock, &msg, 0);
	if (r < 0) continue;

	ms->pp_setup_req_packet_len = r;
	pp_setup_proto_req_handle_usb(dev_no, ms->pp_setup_req_packet, ms->pp_setup_req_packet_len);

	iov.iov_base = ms->pp_setup_rsp_packet;
	iov.iov_len = ms->pp_setup_rsp_packet_len;

	sendmsg(sock, &msg, 0);
    }

#else /* MSP_USE_ADAPTER */

    const char * fn = ___F;
    int should_running = 1;
    int i;
    struct pollfd fds[MAX_DESCRIPTORS];
    int usbfd = usbfds[0]; // only one device controller supported on PPC

#ifdef PRODUCT_RIPCKIMXN
    usb_usrspace_data_t belkinrequest;
#endif

    assert(usbfd != -1);

    for (i = 0; i < MAX_DESCRIPTORS; i++) {
        fds[i].fd = -1;
    }
    
    fds[0].fd = usbfd;
    fds[0].events = POLLIN;
    
    /* default disable device */
    while (should_running) {
	int dev_no;
	file_request_control_t request;
	file_response_t file_response;
	size_t length;
	off_t start_addr;
	
	if ((i = poll(fds, MAX_DESCRIPTORS, 1000)) == 0 || (i < 0 && errno == EINTR)) {
            continue;
        } else if (i < 0) {
            pp_log_err("%s(): poll()", fn);
            break;
        }

#ifdef PRODUCT_RIPCKIMXN
	if (ioctl(usbfd, LARAUSBGETBELKIN, &belkinrequest) != -1) {
	    switch (belkinrequest.function) {
	      case PP_KVM_PORT_CHANGED:
		  pp_kvm_port_changed(belkinrequest.arguments.pp_kvm_port_changed_data.channel,
				      belkinrequest.arguments.pp_kvm_port_changed_data.old_bank,
				      belkinrequest.arguments.pp_kvm_port_changed_data.old_host,
				      belkinrequest.arguments.pp_kvm_port_changed_data.new_bank,
				      belkinrequest.arguments.pp_kvm_port_changed_data.new_host,
				      belkinrequest.arguments.pp_kvm_port_changed_data.control
				      );
		  break;
	      case PP_KVM_DEVICE_STATUS:
		  pp_kvm_set_local_video_state(belkinrequest.arguments.pp_kvm_device_status_data.channel,
					       belkinrequest.arguments.pp_kvm_device_status_data.status);
		  break;
	      case PP_KVM_CONSOLE_BEEP:
		  pp_kvm_console_beep(belkinrequest.arguments.pp_kvm_console_beep_data.channel,
				      belkinrequest.arguments.pp_kvm_console_beep_data.pitch,
				      belkinrequest.arguments.pp_kvm_console_beep_data.time);
		  break;
	      case PP_KVM_AUTOSCAN:
		  pp_kvm_autoscan(belkinrequest.arguments.pp_kvm_autoscan_data.channel,
				  belkinrequest.arguments.pp_kvm_autoscan_data.status);
		  break;
	      case PP_KVM_IP_ADDRESS:
		  pp_kvm_ip_address(belkinrequest.arguments.pp_kvm_ip_address_data.ip,
                                    belkinrequest.arguments.pp_kvm_ip_address_data.valid & KVM_IP_ADDRESS_VALID_IP,
                                    belkinrequest.arguments.pp_kvm_ip_address_data.mask,
                                    belkinrequest.arguments.pp_kvm_ip_address_data.valid & KVM_IP_ADDRESS_VALID_MASK,
                                    belkinrequest.arguments.pp_kvm_ip_address_data.gw,
                                    belkinrequest.arguments.pp_kvm_ip_address_data.valid & KVM_IP_ADDRESS_VALID_GW);

		  break;
	      case PP_KVM_LED_CHANGE:
		  pp_kvm_led_change(belkinrequest.arguments.pp_kvm_led_change_data.channel,
				    belkinrequest.arguments.pp_kvm_led_change_data.scroll_lock,
				    belkinrequest.arguments.pp_kvm_led_change_data.num_lock,
				    belkinrequest.arguments.pp_kvm_led_change_data.caps_lock);
		  break;
	      case PP_KVM_SET_UNIT_HOSTS:
		  {
		      	int hosts;
			hosts = belkinrequest.arguments.pp_kvm_set_unit_hosts_data.hosts;
			/* hosts == -1 means, that the unit was removed
			   internally we set hosts=0 for this unit */
			if (hosts == -1) {
			   hosts = 0;
			}
			pp_kvm_set_unit_port_count(belkinrequest.arguments.pp_kvm_set_unit_hosts_data.unit, hosts);
		  }
		  break; 
	      default:
		  pp_log_err("%s(): Unknown function id", fn);
		  break;
	    }
	}
#endif /* PRODUCT_RIPCKIMXN */

        for (dev_no = 0; dev_no < PP_FEAT_USB_MASS_STORAGE_NO; dev_no++) {
	    mass_storage_t * ms = &mass_storages[dev_no];
	    request.dev_no = dev_no;
	    if (ioctl(usbfd, LARAUSBGETREQUEST, &request) == -1) {
	    	pp_log_err("%s(): ioctl(LARAUSBGETREQUEST)", fn);
	    	break;
	    }

#ifdef USB_DEBUG
	    pp_log("got request from kernel: cmd: %x id: %08x bl: %08x, ba: %08x, l: %04x (blocks)\n",
	           request.request.cmd,
		   request.request.id,
	           request.request.block_length,
	           request.request.block_address,
	           request.request.length);
#endif /* USB_DEBUG */
	    switch (request.request.cmd) {
	      case PP_USB_NOTHING:
		  break;
	      case PP_USB_WRITE:
	      case PP_USB_READ:
		  MUTEX_LOCK(&ms->image_mtx);
		  if (ms->act_image == PP_USB_MS_IMAGE_MSP) {
		      if (request.request.cmd == PP_USB_READ) {
			  usb_msp_request_data(dev_no, &request.request, &file_response);
		      } else {
			  usb_msp_save_write_data(dev_no, &request.request, &file_response);
		      }
		  } else {
		      length = (size_t)request.request.block_length * request.request.length;
		      start_addr = (off_t)request.request.block_address * request.request.block_length;
		      if (start_addr + length <= ms->mmapped_file_length) {
			  if (ms->mmapped_file_data) mmap_file(ms, start_addr, length);
			  file_response.response = FILE_DATA_OK;
			  file_response.length = length;
			  file_response.data = ms->mmapped_file_data + start_addr - ms->mmapped_file_offset;
		      } else {
			  // request too much data
			  file_response.response = FILE_DATA_BAD;
		      }
		  }
		  
#if 0 && defined(USB_DEBUG)
		  dump_sector(&file_response);
#endif
		  file_response.dev_no = request.dev_no;
		  if (ioctl(usbfd, LARAUSBSETRESPONSE, &file_response) < 0) {
		      pp_log_err("%s(): ioctl(LARAUSBSETRESPONSE)", fn);
		      break;
		  }
		  MUTEX_UNLOCK(&ms->image_mtx);
		  break;
	      case PP_USB_WRITE_DONE:
		  MUTEX_LOCK(&ms->image_mtx);
		  if (ms->act_image == PP_USB_MS_IMAGE_MSP) {
		      usb_msp_do_write_data(dev_no, &request.request);
		  }
		  MUTEX_UNLOCK(&ms->image_mtx);
		  break;
	      case PP_USB_SETUP_PROTO_REQ:
	          ms->pp_setup_request_state = SETUP_WORKING;
	          length = request.request.block_length * request.request.length;
	          start_addr = request.request.block_address * request.request.block_length;
	          if (start_addr + length <= sizeof(ms->pp_setup_req_packet)) {
		      file_response.response = FILE_DATA_OK;
		      file_response.length = length;
		      ms->pp_setup_req_packet_len = length;
		      file_response.data = ms->pp_setup_req_packet + start_addr;
	          } else {
		      // request too much data
		      file_response.response = FILE_DATA_BAD;
	          }
	          file_response.dev_no = request.dev_no;
	          if (ioctl(usbfd, LARAUSBSETRESPONSE, &file_response) < 0) {
		      pp_log_err("%s(): ioctl()", fn);
		      break;
	          }
	          break;
	      case PP_USB_SETUP_PROTO_REQ_DONE:
#if 0
		  debug_show_packet(ms->pp_setup_req_packet, ms->pp_setup_req_packet_len);
#endif
		  pp_setup_proto_req_handle_usb(request.dev_no, ms->pp_setup_req_packet, ms->pp_setup_req_packet_len);
		  break;
	      case PP_USB_SETUP_PROTO_RSP:
		  if (ms->pp_setup_request_state != SETUP_DONE) {
		      pp_log("set proto not ready yet (answer via USB)\n");
		      file_response.response = FILE_DATA_BAD;
		  } else {
		      length = request.request.block_length * request.request.length;
		      if (length > ms->pp_setup_rsp_packet_len) {
			  file_response.response = FILE_DATA_OK_SHORT;
		      } else {
			  file_response.response = FILE_DATA_OK;
		      }
		      start_addr = request.request.block_address * request.request.block_length;
		      file_response.length = ms->pp_setup_rsp_packet_len;
		      file_response.data = ms->pp_setup_rsp_packet + start_addr;
		  }
		  file_response.dev_no = request.dev_no;
		  if (ioctl(usbfd, LARAUSBSETRESPONSE, &file_response) < 0) {
		      pp_log_err("%s(): ioctl(LARAUSBSETRESPONSE)", fn);
		      break;
		  }
		  break;
	    }
	}
    }
    return NULL;

#endif
}

int
pp_usb_set_setup_rsp_packet(u_int ms_index, char* pdu, size_t len)
{
    mass_storage_t * ms;
    
    if (ms_index >= PP_FEAT_USB_MASS_STORAGE_NO) return -1;
    
    ms = &mass_storages[ms_index];

    if (len > sizeof(ms->pp_setup_rsp_packet)) {
	return -1;
    } else {
	memcpy(ms->pp_setup_rsp_packet, pdu, len);
	ms->pp_setup_rsp_packet_len = len;
	printf("setup_done, len: %d\n", ms->pp_setup_rsp_packet_len);
#if 0
	debug_show_packet(pdu, len);
#endif
	ms->pp_setup_request_state = SETUP_DONE;
	return 0;
    }
}
#endif /* PP_FEAT_MASS_STORAGE */
