/* Helper functions for Virtual Media Management via IPMI */

#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <pthread.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <pp/error.h>
#include <pp/usb.h>

#define TMP_FILENAME "/mnt/tmpfs/vfloppy.XXXXXX"

struct pp_usb_vfup_ctx_s {
    int device_id;
    int size_total;
    int size_done;
    char *display_name;
    char *filename;
    int fd;
};

struct pp_usb_smbmnt_ctx_s {
    int device_id;
    pp_usb_smbmnt_status_t status;
    pthread_t helper_thread;
    char *host;
    char *share;
    char *path;
    char *user;
    char *pass;
};

/**
 * Start a VFloppy image upload into a temporary file
 * @param device_id Virtual media device ID (0..PP_FEAT_USB_MASS_STORAGE_NO-1)
 * @param size Total image size in bytes
 * @param name Display name for image
 * @return VFloppy upload context, NULL in case of error
 */
pp_usb_vfup_ctx_t *pp_usb_vfup_start(int device_id, int size, const char *name)
{
    pp_usb_vfup_ctx_t *ctx = malloc(sizeof(pp_usb_vfup_ctx_t));
    ctx->device_id = device_id;
    ctx->size_total = size;
    ctx->size_done = 0;
    ctx->display_name = strdup(name);
    ctx->filename = strdup(TMP_FILENAME);
    ctx->fd = mkstemp(ctx->filename);
    return ctx->fd < 0? NULL : ctx;
}

/**
 * Cancel an active image upload, delete image file
 * @param ctx VFloppy upload context
 */
void pp_usb_vfup_delete(pp_usb_vfup_ctx_t *ctx)
{
    close(ctx->fd);
    unlink(ctx->filename);
    free(ctx->display_name);
    free(ctx->filename);
    free(ctx);
}

/**
 * Add a chunk to an active VFloppy image upload; chunks must be added in
 * gapless incremental order.
 * @param ctx VFloppy upload context
 * @param offset Chunk offset inside image file
 * @param chunk Chunk data
 * @param size Chunk size
 * @return PP_SUC or PP_ERR
 */
int pp_usb_vfup_add_chunk(pp_usb_vfup_ctx_t *ctx, int offset,
	char *chunk, int size)
{
    if (ctx->size_done != offset) return PP_ERR;
    if (ctx->size_done + size > ctx->size_total) return PP_ERR;

    if (write(ctx->fd, chunk, size) != size) {
	lseek(ctx->fd, ctx->size_done, SEEK_CUR);
	return PP_ERR;
    }

    ctx->size_done += size;
    return PP_SUC;
}

/**
 * Check whether a VFloppy image is uploaded completely
 * @param ctx VFloppy upload context
 * @return 0 (not yet complete) or 1 (complete)
 */
int pp_usb_vfup_complete(pp_usb_vfup_ctx_t *ctx)
{
    return (ctx->size_done == ctx->size_total);
}

/**
 * Activate a completely uploaded vfloppy image
 * @param ctx VFloppy upload context
 * @return PP_SUC or PP_ERR
 */
int pp_usb_vfup_activate(pp_usb_vfup_ctx_t *ctx)
{
    int error;
    if (ctx->size_done != ctx->size_total) return PP_ERR;

    pp_usb_ms_unset_image(ctx->device_id, &error);
    if (PP_FAILED(pp_usb_ms_set_image(ctx->device_id, ctx->fd,
		ctx->filename, ctx->size_total, ctx->display_name, &error))) {
	return PP_ERR;
    }

    pp_usb_vfup_delete(ctx);
    return PP_SUC;
}


/**
 * Initialize an SMB helper for the given virtual media device
 * @param device_id Virtual media device ID (0..PP_FEAT_USB_MASS_STORAGE_NO-1)
 * @return SMB helper context
 */
pp_usb_smbmnt_ctx_t *pp_usb_smbmnt_start(int device_id)
{
    pp_usb_smbmnt_ctx_t *ctx = malloc(sizeof(pp_usb_smbmnt_ctx_t));
    memset(ctx, 0, sizeof(pp_usb_smbmnt_ctx_t));
    ctx->device_id = device_id;
    ctx->status = PP_USB_SMBMNT_STATUS_IDLE;
    return ctx;
}

/**
 * Delete an SMB helper, reap terminated helper thread
 * @param ctx SMB helper context
 * @return PP_SUC or PP_ERR
 * Note: A busy helper cannot be terminated
 */
int pp_usb_smbmnt_close(pp_usb_smbmnt_ctx_t *ctx)
{
    void *thread_ret;
    switch (ctx->status) {
	case PP_USB_SMBMNT_STATUS_IDLE:
	    break;
	case PP_USB_SMBMNT_STATUS_BUSY:
	    return PP_ERR;
	case PP_USB_SMBMNT_STATUS_SUCCESS:
	case PP_USB_SMBMNT_STATUS_FAILED:
	    pthread_join(ctx->helper_thread, &thread_ret);
    }
    free(ctx);
    return PP_SUC;
}

/**
 * Set a parameter for the given SMB helper
 * @param ctx SMB helper context
 * @param param Parameter selector (host, share, path, user or pass)
 * @param value New value for parameter
 */
void pp_usb_smbmnt_set_param(pp_usb_smbmnt_ctx_t *ctx,
	pp_usb_smbmnt_param_t param, const char *value)
{
    char **ptr = NULL;
    switch (param) {
	case PP_USB_SMBMNT_PARAM_HOST:
	    ptr = &ctx->host;
	    break;
	case PP_USB_SMBMNT_PARAM_SHARE:
	    ptr = &ctx->share;
	    break;
	case PP_USB_SMBMNT_PARAM_PATH:
	    ptr = &ctx->path;
	    break;
	case PP_USB_SMBMNT_PARAM_USER:
	    ptr = &ctx->user;
	    break;
	case PP_USB_SMBMNT_PARAM_PASS:
	    ptr = &ctx->pass;
    }

    if (ptr) {
	free (*ptr);
	*ptr = strdup(value);
	printf("pp_usb_smbmnt_set_param(): param = %d, value = %s\n", param, *ptr);
    }
}

/* Helper background thread function */
static void *mnt_helper(void *arg)
{
    pp_usb_smbmnt_ctx_t *ctx = arg;
    int err, ret;
    const char *user, *pass;

    ctx->status = PP_USB_SMBMNT_STATUS_BUSY;
    user = ctx->user? ctx->user : "";
    pass = ctx->pass? ctx->pass : "";
    ret = pp_usb_ms_set_smb_image(ctx->device_id, ctx->host,
	    ctx->share, ctx->path, user, pass, &err);
    ctx->status = PP_SUCCED(ret)?
	PP_USB_SMBMNT_STATUS_SUCCESS : PP_USB_SMBMNT_STATUS_FAILED;

    return NULL;
}

/**
 * Initiate the smbmount helper background thread; at least the host,
 * share and path parameters must be configured before calling this.
 * @param ctx SMB helper context
 * @return PP_SUC or PP_ERR
 */
int pp_usb_smbmnt_activate(pp_usb_smbmnt_ctx_t *ctx)
{
    if (!ctx->host || !ctx->share || !ctx->path ||
	    pthread_create(&ctx->helper_thread, NULL, mnt_helper, ctx) != 0) {
	return PP_ERR;
    }
    return PP_SUC;
}

/**
 * Get the status of the background thread
 * @param ctx SMB helper context
 * @return Status code
 */
pp_usb_smbmnt_status_t pp_usb_smbmnt_get_status(pp_usb_smbmnt_ctx_t *ctx)
{
    return ctx->status;
}

