 /*********************************************************************
 * libpp_vsc: adc_io.c
 *
 * low level register access functions for ADC
 *
 ********************************************************************/

#include <pp/base.h>
#include <pp/i2c.h>
#include <pp/features.h>
#include <linux/i2c.h>
#include <linux/i2c-dev.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include "adc_io.h"
#include "debug.h"

#define ADC_DEVICE	"/dev/i2c-adc"

static int i2c_fd = -1;

static int
adc_config(vsc_context_t * context) {
    /* switch mux to correct channel if necessary */
    if (context->adc_i2c_mux_addr) {
	if (ioctl(i2c_fd, I2C_SLAVE, context->adc_i2c_mux_addr) < 0) {
	    pp_log_err("IOCTL I2C_SLAVE failed");
	    return -1;
	}
	if (i2c_smbus_write_byte(i2c_fd, (0x1 << context->adc_i2c_mux_channel)) < 0) {
	    return -1;
	}
    }
    /* set i2c address */
    if (ioctl(i2c_fd, I2C_SLAVE, context->adc_i2c_addr) < 0) {
	pp_log_err("IOCTL I2C_SLAVE failed");
	return -1;
    }
    return 0;
}

int
adc_io_init(void)
{
    if ((i2c_fd = open(ADC_DEVICE, O_RDWR)) < 0) {
    	pp_log_err("Error Opening I2C device %i.\n",errno);
    	return -1;
    }
    
    return 0;
}

void
adc_io_cleanup(void)
{
    if (i2c_fd > 0) {
    	close(i2c_fd);
    }
}

u_int8_t
adc_rx_byte(vsc_context_t *context, u_int8_t reg)
{
    int ret;

    if (adc_config(context) < 0) { return -1; }
    if ((ret = i2c_smbus_read_byte_data(i2c_fd, reg)) < 0) {
	D(D_ERROR, "Error in adc_rx_byte from %02x\n", reg);
    }

    return ret;
}

u_int16_t
adc_rx_word(vsc_context_t *context, u_int8_t reg)
{
    int ret;

    if (adc_config(context) < 0) { return -1; }
    if ((ret = i2c_smbus_read_word_data(i2c_fd, reg)) < 0) {
	D(D_ERROR, "Error in adc_rx_word from %02x\n", reg);
    }

    return ret;
}

void
adc_tx_byte(vsc_context_t *context, u_int8_t reg, u_int8_t data)
{
    if (adc_config(context) < 0) { return; }
    if (i2c_smbus_write_byte_data(i2c_fd, reg, data) < 0) {
	D(D_ERROR, "Error in adc_tx_byte from %02x\n", reg);
    }
}

void
adc_tx_byte_masked(vsc_context_t *context, u_int8_t reg, u_int8_t mask, u_int8_t data) {
    // not thread safe, but only executed whithin one thread, so it is ok
    u_int8_t d;
    d = adc_rx_byte(context, reg);
    d &= ~mask;
    d |= (data & mask);
    adc_tx_byte(context, reg, d);
}

void
adc_tx_word(vsc_context_t *context, u_int8_t reg, u_int16_t data)
{
    if (adc_config(context) < 0) { return; }
    if (i2c_smbus_write_word_data(i2c_fd, reg, data) < 0) {
	D(D_ERROR, "Error in adc_tx_word from %02x\n", reg);
    }
}
