#include <pp/grab.h>

#include "debug.h"
#include "offset_pic.h"

typedef struct s_rgb {
	unsigned short b;	/* blue */
	unsigned short g;	/* green */
	unsigned short r;	/* red */
} t_rgb;


static int get_largest_ofs_x(pp_grab_client_t* grab_client, t_rgb *snap, int start, int width,
			     int *largets_ofs);
static int get_largest_ofs_y(pp_grab_client_t* grab_client, t_rgb *snap, int start, int height,
			     int *largets_ofs);
static int fb_to_rgb_area(pp_grab_client_t *grab_client, t_rgb *snap,
			  u_short left, u_short top, u_short width, u_short height);

static int compare_pixel(t_rgb *p1, t_rgb *p2);

#define PIXEL_THRESHOLD 12

int
offset_pic_measurement(vsc_context_t *context,
		       u_int16_t *ofsX, u_int16_t *ofsY)
{
    fb_format_info_t fb_f_info;
    pp_grab_client_t* grab_client;
    int ret = PP_ERR, valX = 0 , valY = 0, start, end, line, row, width, height;
    t_rgb *snap = NULL;

    u_char video_link = (u_char )context->id;

    if ((grab_client = pp_grab_new_client(video_link, GRAB_FIX_MEM_DESC_OFFSET)) == NULL) {
	goto bail;
    }  
    
    if (pp_grab_get_format_info(video_link, &fb_f_info) != 0) {
	goto bail;
    }

    
    start = (fb_f_info.g_h_pd / 4);
    end   = 3 * (fb_f_info.g_h_pd / 4);
    width = fb_f_info.g_w_pd / 4;

    if (start % PP_FB_TILE_HEIGHT) start = (start / PP_FB_TILE_HEIGHT) * PP_FB_TILE_HEIGHT;
    if (end % PP_FB_TILE_HEIGHT) end = (end / PP_FB_TILE_HEIGHT) * PP_FB_TILE_HEIGHT;
    if (width % PP_FB_TILE_WIDTH) width = (width / PP_FB_TILE_WIDTH) * PP_FB_TILE_WIDTH;
    
    snap = (t_rgb *) malloc(width * PP_FB_TILE_HEIGHT * sizeof(t_rgb));
    
    D(D_BLABLA, "%s: looking vertically from %d to %d, width = %d\n",
      ___F, start, end, width);
    
    for (line = start; line <= end; line += PP_FB_TILE_HEIGHT) {
	int new_ofs;
	if (PP_FAILED(get_largest_ofs_x(grab_client, snap, line, width, &new_ofs))) {
	    goto bail;
	}
	if (new_ofs > valX) {
	    valX = new_ofs;
	}
    }

    D(D_VERBOSE, "%s: got maximum horizontal offset of %d\n", ___F, valX);

    start  = (fb_f_info.g_w_pd / 4);
    end   = 3 * (fb_f_info.g_h_pd / 4);
    height = fb_f_info.g_h_pd / 4;

    if (start % PP_FB_TILE_WIDTH) start = (start / PP_FB_TILE_WIDTH) * PP_FB_TILE_WIDTH;
    if (end % PP_FB_TILE_WIDTH) end = (end / PP_FB_TILE_WIDTH) * PP_FB_TILE_WIDTH;
    if (height % PP_FB_TILE_HEIGHT) height = (height / PP_FB_TILE_HEIGHT) * PP_FB_TILE_HEIGHT;
    
    snap = realloc(snap, width * PP_FB_TILE_HEIGHT * sizeof(t_rgb));

    D(D_BLABLA, "%s: looking horizontally from %d to %d, height = %d\n",
      ___F, start, end, height);

    for (row = start; row <= end; row += PP_FB_TILE_WIDTH) {
	int new_ofs;
	if (PP_FAILED(get_largest_ofs_y(grab_client, snap, row, height, &new_ofs))) {
	    goto bail;
	}
	if (new_ofs > valY) {
	    valY = new_ofs;
	}
    }

    D(D_VERBOSE, "%s: got maximum vertical offset of %d\n", ___F, valY);
    
    ret = PP_SUC;

 bail:
    free(snap);
    if (ofsX) *ofsX = valX;
    if (ofsY) *ofsY = valY;
    pp_grab_remove_client(grab_client);
    return ret;
}

static int
get_largest_ofs_x(pp_grab_client_t* grab_client, t_rgb *snap, int start, int width,
		  int *largest_ofs)
{
    int ret = PP_ERR;
    int x, y, ofs, min_ofs = width;
    
    if (fb_to_rgb_area(grab_client, snap, 0, start, width, PP_FB_TILE_HEIGHT) != 0) {
	goto bail;
    }

    for (y = 0; y < PP_FB_TILE_HEIGHT; y++) {
	t_rgb *line = &snap[y * width];
	t_rgb pixel = line[0];
	
	ofs = 0;
	
	for (x = 1; x < width; x++) {
	    if (compare_pixel(&line[x], &pixel) != 0) {
		break;
	    }

	    ofs++;
	}

	if (ofs < min_ofs) {
	    min_ofs = ofs + 1;
	}
    }

    D(D_BLABLA, "%s: Tiles from %d (width %d) gave max ofs %d\n", ___F, start, width, min_ofs);
    if (largest_ofs) *largest_ofs = min_ofs;
    ret = PP_SUC;
    
 bail:
    return ret;
}

static int get_largest_ofs_y(pp_grab_client_t* grab_client, t_rgb *snap, int start, int height,
			     int *largest_ofs)
{
    int ret = PP_ERR;
    int x, y, ofs, min_ofs = height;
    
    if (fb_to_rgb_area(grab_client, snap, start, 0, PP_FB_TILE_WIDTH, height) != 0) {
	goto bail;
    }

    for (x = 0; x < PP_FB_TILE_WIDTH; x++) {
	t_rgb *row = &snap[x];
	t_rgb pixel = *row;

	ofs = 0;

	for (y = 1; y < height; y++) {
	    if (compare_pixel(&row[y * PP_FB_TILE_WIDTH], &pixel) != 0) {
		break;
	    }

	    ofs++;
	}

	if (ofs < min_ofs) {
	    min_ofs = ofs + 1;
	}
    }

    D(D_BLABLA, "%s: Tiles from %d (height %d) gave max ofs %d\n", ___F, start, height, min_ofs);
    if (largest_ofs) *largest_ofs = min_ofs;
    ret = PP_SUC;
    
 bail:
    return ret;
    
}


/* this function convertes a whole block of pixels in the current framebuffer
   into the t_rgb structure */
/* FIXME: move this to a common include with km iipptr stuff, maybe in grab? */
static int
fb_to_rgb_area(pp_grab_client_t *grab_client, t_rgb *snap,
	       u_short left, u_short top, u_short width, u_short height)
{
    int val, ret = 0;
    u_int16_t * i_16_ptr;        
    int offset_bmp=0;
    fb_color_info_t fb_c_info;
    u_short x, y, w, h;

    pp_grab_get_color_info(grab_client->video_link,&fb_c_info);
    assert(fb_c_info.bpp == 16);
    
    x = (left / PP_FB_TILE_WIDTH) * PP_FB_TILE_WIDTH;
    y = (top / PP_FB_TILE_HEIGHT) * PP_FB_TILE_HEIGHT;
    w = ((width + left - x + PP_FB_TILE_WIDTH - 1) / PP_FB_TILE_WIDTH) *
        PP_FB_TILE_WIDTH;
    h = ((height + top - y + PP_FB_TILE_HEIGHT - 1) / PP_FB_TILE_HEIGHT) *
        PP_FB_TILE_HEIGHT;
	    
    if (pp_grab_request_rect(grab_client, GRAB_REQ_FLAGS_IGNORE_DIFFMAP, x, y, w, h) != 0) {
	ret = -1;
	goto bail;
    }

    // add returned offset to the fb start -> aligned rect start
    i_16_ptr = (u_int16_t *)(grab_client->buf + 
                             grab_client->rect_offset * sizeof(u_int16_t));
    
    {
	int bol_skip  = left % PP_FB_TILE_WIDTH;
	int eol_skip  = ((bol_skip + width) % PP_FB_TILE_WIDTH) ?
                        PP_FB_TILE_WIDTH - ((bol_skip + width) %
                                            PP_FB_TILE_WIDTH) : 0;
	int line_skip = -(((bol_skip + width + PP_FB_TILE_WIDTH - 1) / PP_FB_TILE_WIDTH - 1) * PP_FB_TILE_WIDTH * PP_FB_TILE_HEIGHT);
	int row_skip  = (grab_client->rect_pitch - PP_FB_TILE_WIDTH) * PP_FB_TILE_HEIGHT;
	int w_single, line=0;
        int y_offset = top % PP_FB_TILE_HEIGHT;
	
	// goto start pixel, still in the same first hextile
	i_16_ptr += y_offset * PP_FB_TILE_WIDTH + (left % PP_FB_TILE_WIDTH);
	
	while (line++ < height) {
	    w_single = 0;
	    
	    while (w_single++ < width) {
		val = *i_16_ptr++;
		snap[offset_bmp].r = ((val >> (fb_c_info.blue_bits +
					       fb_c_info.green_bits)) 
				      << (8 - fb_c_info.red_bits))
                                            & 0xFF;
		snap[offset_bmp].g = ((val >> fb_c_info.blue_bits)
				      << (8 - fb_c_info.green_bits))
				      & 0xFF;
		snap[offset_bmp].b = (val << (8 - fb_c_info.blue_bits))
				     & 0xFF;
		offset_bmp++;

		if ((w_single + bol_skip) % PP_FB_TILE_WIDTH == 0 &&
                    w_single < width) {
		    i_16_ptr += (PP_FB_TILE_WIDTH * PP_FB_TILE_HEIGHT) -
                                PP_FB_TILE_WIDTH;
                }
	    }
	    
	    i_16_ptr += eol_skip;  // skip partial rect at end of line
	    i_16_ptr += line_skip; // start next pixel line
	    
	    if ((line + y_offset) % PP_FB_TILE_HEIGHT == 0) {
                i_16_ptr += row_skip;
            }
	    
	    i_16_ptr += bol_skip;  // skip partial rect at beginning of line
	}
    }

 bail:
    pp_grab_release_buffer(grab_client);
    return ret;
}

static int
compare_pixel(t_rgb *p1, t_rgb *p2)
{
    if (((abs(p1->r - p2->r) +
	  abs(p1->g - p2->g) +
	  abs(p1->b - p2->b)) / 3) > PIXEL_THRESHOLD) {

	return 1;
    }

    return 0;
}
