 /*********************************************************************
 * libpp_vsc: settings.c
 *
 * contains functions to handle video settings, get/set or update 
 * settings in user-interaction through rfb library
 *
 ********************************************************************/

#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/vsc.h>
#include <lara.h>
#include "adc.h"
#include "context.h"
#include "debug.h"
#include "mode.h"
#include "settings.h"
#include "timer.h"
#include "vsc_funcs.h"
#include "vsc_io.h"

/**
 * currently supported video settings,
 * sequence needs to correspond with their IDs
 */
/* FIXME: remove defaults here as we have them in cdl now, how to implement reset then? */
vsc_setting_info_t vsc_setting_infos[] = {
#if !defined(PP_FEAT_VSC_PANEL_INPUT)
    {
	id:		VIDEO_SETTING_BRIGHTNESS,
	type:		VSC_SETTING_KVM_LOCAL,
	min:		0,
	max:		127,
	def:		80,
	increment:	1,
	diff_mask_off:	1,
	adjust_vsc_ofs: 0,
	cfg_key:	"unit[%u].port[%u].video.adc_brightness",
	set_function:	adc_set_brightness,
	get_function:	adc_get_brightness
    },
    {
	id:		VIDEO_SETTING_CONTRAST_R,
	type:		VSC_SETTING_KVM_LOCAL,
	min:		0,
	max:		255,
	def:		100,
	increment:	1,
	diff_mask_off:	1,
	adjust_vsc_ofs: 0,
	cfg_key:	"unit[%u].port[%u].video.adc_contrast.r",
	set_function:	adc_set_contrast_r,
	get_function:	adc_get_contrast_r
    },
    {
	id:		VIDEO_SETTING_CONTRAST_G,
	type:		VSC_SETTING_KVM_LOCAL,
	min:		0,
	max:		255,
	def:		100,
	increment:	1,
	diff_mask_off:	1,
	adjust_vsc_ofs: 0,
	cfg_key:	"unit[%u].port[%u].video.adc_contrast.g",
	set_function:	adc_set_contrast_g,
	get_function:	adc_get_contrast_g
    },
    {
	id:		VIDEO_SETTING_CONTRAST_B,
	type:		VSC_SETTING_KVM_LOCAL,
	min:		0,
	max:		255,
	def:		100,
	increment:	1,
	diff_mask_off:	1,
	adjust_vsc_ofs: 0,
	cfg_key:	"unit[%u].port[%u].video.adc_contrast.b",
	set_function:	adc_set_contrast_b,
	get_function:	adc_get_contrast_b
    },    
    {
	id:		VIDEO_SETTING_CLOCK,
	type:		VSC_SETTING_MODE_LOCAL,
	min:		0,
	max:		2048,
	def:		0,
	increment:	2,
	diff_mask_off:	0,
	adjust_vsc_ofs: 1,
	cfg_key:	NULL,
	set_function:	adc_set_clocks,
	get_function:	adc_get_clocks
    },
    {
	id:		VIDEO_SETTING_PHASE,
	type:		VSC_SETTING_MODE_LOCAL,
	min:		0,
	max:		31,
	def:		0,
	increment:	1,
	diff_mask_off:	0,
	adjust_vsc_ofs: 0,
	cfg_key:	NULL,	
	set_function:	adc_set_phase,
	get_function:	adc_get_phase
    },
#endif /* !PP_FEAT_VSC_PANEL_INPUT */
    {
	id:		VIDEO_SETTING_OFFSET_X,
	type:		VSC_SETTING_MODE_LOCAL,
	min:		0,
	max:		255,
	def:		0,
	increment:	1,
	diff_mask_off:	0,
	adjust_vsc_ofs: 1,
	cfg_key:	NULL,
	set_function:	vsc_set_offset_x,
	get_function:	vsc_get_offset_x
    },
    {
	id:		VIDEO_SETTING_OFFSET_Y,
	type:		VSC_SETTING_MODE_LOCAL,
	min:		0,
	max:		255,
	def:		0,
	increment:	1,
	diff_mask_off:	0,
	adjust_vsc_ofs: 1,
	cfg_key:	NULL,
	set_function:	vsc_set_offset_y,
	get_function:	vsc_get_offset_y
    },
    /* ------------ end marker, keep ------------------ */
    {
	id:		-1,
	type:		0,
	min:		0,
	max:		0,
	def:		0,
	increment:	0,
	diff_mask_off:	0,
	adjust_vsc_ofs: 0,
	cfg_key:	NULL,
	set_function:	NULL,
	get_function:	NULL
    },
};

/* allow only one external thread to mess with the video settings */
static pthread_mutex_t settings_external_mtx  = PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;

/* lock set and save of global and kvm local settings (saved in config fs) */
static pthread_mutex_t global_settings_mtx    = PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;
static pthread_mutex_t kvm_local_settings_mtx = PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;

static const int DIFF_THRESHOLD_WAIT_MS = 2000;

static vsc_setting_info_t* get_setting_info_from_id(unsigned char id);

/* ----------------- functions available to outside--------------------------- */

int
pp_vsc_get_video_settings(unsigned char video_link, video_settings_t * settings)
{
    vsc_context_t *context = &vsc_contexts[video_link];
    unsigned short resX, resY, freqV;
    int ret = PP_ERR;

    assert(settings);
    
    MUTEX_LOCK(&settings_external_mtx);
    if (context->vsc_active == PP_FALSE) goto bail;
    
    memset(settings, 0x00, sizeof(video_settings_t));

    if (mode_get_res_info(context, &resX, &resY, NULL, &freqV) == 0) {
	vsc_setting_info_t *info;
	    
#ifdef PP_FEAT_VSC_PANEL_INPUT
	settings->brightness	= 0;
	settings->contrast_red	= 0;
	settings->contrast_green	= 0;
	settings->contrast_blue	= 0;
	settings->clock		= 0;
	settings->phase		= 0;	
#else /* !PP_FEAT_VSC_PANEL_INPUT */
	info = get_setting_info_from_id(VIDEO_SETTING_BRIGHTNESS);
	if (info) settings->brightness = info->get_function(context);
	info = get_setting_info_from_id(VIDEO_SETTING_CONTRAST_R);
	if (info) settings->contrast_red = info->get_function(context);
	info = get_setting_info_from_id(VIDEO_SETTING_CONTRAST_G);
	if (info) settings->contrast_green = info->get_function(context);
	info = get_setting_info_from_id(VIDEO_SETTING_CONTRAST_B);
	if (info) settings->contrast_blue = info->get_function(context);
	info = get_setting_info_from_id(VIDEO_SETTING_CLOCK);
	if (info) settings->clock = info->get_function(context);
	info = get_setting_info_from_id(VIDEO_SETTING_PHASE);
	if (info) settings->phase = info->get_function(context);
#endif /* !PP_FEAT_VSC_PANEL_INPUT */
	info = get_setting_info_from_id(VIDEO_SETTING_OFFSET_X);
	if (info) settings->offset_x = info->get_function(context);
	info = get_setting_info_from_id(VIDEO_SETTING_OFFSET_Y);
	if (info) settings->offset_y = info->get_function(context);
	
	settings->res_x   = resX;
	settings->res_y   = resY;
	settings->refresh = (freqV + 5) / 10;

	settings->y_max_offset = 255; //we don't need this for VSC, just compatibility
	ret = PP_SUC;
    }	

 bail:
    MUTEX_UNLOCK(&settings_external_mtx);
    return ret;
}

int
pp_vsc_set_video_setting(unsigned char video_link, unsigned char setting, unsigned short value)
{
    vsc_context_t *context = &vsc_contexts[video_link];
    vsc_setting_info_t *info;
    int ret = 0;

    MUTEX_LOCK(&settings_external_mtx);
    if (context->vsc_active == PP_FALSE) goto bail;
    D(D_BLABLA, "vsc_set_video_setting: setting %02x, value %04x\n", setting, value);

    if ((info = get_setting_info_from_id(setting)) == NULL) {
	D(D_VERBOSE, "vsc_set_video_setting: unknown setting %d\n", setting);
	ret = -1;
	goto bail;
    }  
    
    if (info->diff_mask_off) {
	vsc_zero_diff_threshold(context);
	timer_set(context, DIFF_THRESHOLD_TIMER, DIFF_THRESHOLD_WAIT_MS);
    }

    // enforce certain increment
    value = (value / info->increment) * info->increment;
    
    ret = info->set_function(context, value);
    
    if (info->adjust_vsc_ofs) {
	vsc_correct_offsets(context);
    }
 
    if (ret != 0) {
	D(D_ERROR, "Error during vsc setting %d\n", setting);
    }

 bail:
    MUTEX_UNLOCK(&settings_external_mtx);
    return ret;
}

int
pp_vsc_save_video_settings(unsigned char video_link, vsc_setting_type_t types)
{
    vsc_context_t *context = &vsc_contexts[video_link];

    MUTEX_LOCK(&settings_external_mtx);
    if (context->vsc_active == PP_FALSE) goto bail;
    D(D_VERBOSE, "vsc_save_video_settings types=%02x\n", types);
    if (types & VSC_SETTING_GLOBAL) {
	save_global_settings(context);
    }

    if (types & VSC_SETTING_KVM_LOCAL) {
	save_kvm_local_settings(context);
    }

    if (types & VSC_SETTING_MODE_LOCAL) {
	save_mode_local_settings(context);
    }

    pp_cfg_save(DO_FLUSH);

 bail:
    MUTEX_UNLOCK(&settings_external_mtx);
    return 0;
}

int
pp_vsc_cancel_video_settings(unsigned char video_link, vsc_setting_type_t types)
{
    vsc_context_t *context = &vsc_contexts[video_link];
 
    if (context->vsc_active == PP_FALSE) goto bail;
    D(D_VERBOSE, "vsc_cancel_video_settings types=%02x\n", types);

    MUTEX_LOCK(&settings_external_mtx);
    if (types & VSC_SETTING_GLOBAL) {
	set_global_settings(context);
    }

    if (types & VSC_SETTING_KVM_LOCAL) {
	set_kvm_local_settings(context);
    }

    if (types & VSC_SETTING_MODE_LOCAL) {
	context->force_mode_setup = PP_TRUE;
    }
    MUTEX_UNLOCK(&settings_external_mtx);

    send_video_settings_propchange(context);

 bail:
    return 0;
}

int
pp_vsc_reset_video_settings(unsigned char video_link, vsc_setting_type_t types)
{
    vsc_context_t *context = &vsc_contexts[video_link];

    if (context->vsc_active == PP_FALSE) goto bail;
    D(D_VERBOSE, "vsc_reset_video_settings types=%02x\n", types);
    
    MUTEX_LOCK(&settings_external_mtx);
    if (types & VSC_SETTING_GLOBAL) {
	reset_global_settings(context);
    }
    if (types & VSC_SETTING_KVM_LOCAL) {
	reset_kvm_local_settings(context);
    }
    if (types & VSC_SETTING_MODE_LOCAL) {
	mode_reset_current(context);
    }
    if (types & VSC_SETTING_MODE_ALL) {
	mode_reset_all(context);
    }
    
 bail:
    MUTEX_UNLOCK(&settings_external_mtx);
    return 0;
}

int
pp_vsc_trigger_autoadjust(unsigned char video_link)
{
    vsc_context_t *context = &vsc_contexts[video_link];

    if (context->vsc_active == PP_FALSE) goto bail;
    context->aa_data.request = PP_TRUE;

 bail:
    return 0;
}

int
save_global_settings(vsc_context_t *context)
{
    vsc_setting_info_t *info = vsc_setting_infos;

    MUTEX_LOCK(&global_settings_mtx);
    D(D_VERBOSE, "save_global_settings\n");
    while (info->id != -1) {
	if (info->type == VSC_SETTING_GLOBAL) {	    	    
	    unsigned short val = info->get_function(context);
	    char val_str[6];

	    snprintf(val_str, sizeof(val_str), "%d", val);
	    if (PP_ERR == pp_cfg_set_short(val, info->cfg_key)) {
		D(D_ERROR, "Error during save of vsc setting %d\n", info->id);
	    }
	}
	info++;	
    }

    MUTEX_UNLOCK(&global_settings_mtx);
    return 0;
}

int
save_kvm_local_settings(vsc_context_t *context)
{
    vsc_setting_info_t *info = vsc_setting_infos;

    MUTEX_LOCK(&kvm_local_settings_mtx);
    D(D_VERBOSE, "save_kvm_local_settings\n");
    while (info->id != -1) {
	if (info->type == VSC_SETTING_KVM_LOCAL) {	    	    
	    unsigned short val = info->get_function(context);
	    char val_str[6];

	    snprintf(val_str, sizeof(val_str), "%d", val);
	    
	    if (PP_ERR == pp_cfg_set_short(val, info->cfg_key, context->kvm_unit,
					   context->kvm_port)) {
		D(D_ERROR, "Error during save of vsc setting %d\n", info->id);
	    }
	}
	info++;
    }

    MUTEX_UNLOCK(&kvm_local_settings_mtx);
    return 0;
}

int
reset_global_settings(vsc_context_t *context)
{
    vsc_setting_info_t *info = vsc_setting_infos;

    MUTEX_LOCK(&global_settings_mtx);
    D(D_VERBOSE, "reset_global_settings\n");
    while (info->id != -1) {
	if (info->type == VSC_SETTING_GLOBAL) {
	    int ret = info->set_function(context, info->def);
	    if (ret != 0) {
		D(D_ERROR, "Error during reset of vsc setting %d\n", info->id);
	    }
	}
	info++;	
    }

    MUTEX_UNLOCK(&global_settings_mtx);
    return 0;
}

int
reset_kvm_local_settings(vsc_context_t *context)
{
    vsc_setting_info_t *info = vsc_setting_infos;

    MUTEX_LOCK(&kvm_local_settings_mtx);
    D(D_VERBOSE, "reset_kvm_local_settings\n");
    while (info->id != -1) {
	if (info->type == VSC_SETTING_KVM_LOCAL) {
	    int ret = info->set_function(context, info->def);
	    if (ret != 0) {
		D(D_ERROR, "Error during reset of vsc setting %d\n", info->id);
	    }
	}
	info++;
    }

    MUTEX_UNLOCK(&kvm_local_settings_mtx);
    return 0;
}

int
save_mode_local_settings(vsc_context_t *context)
{
    int ret;
    
    
    D(D_VERBOSE, "save_mode_local_settings\n");
    vsc_setting_info_t *info;
    u_short myclock = 0, phase = 0, ofs_x = 0, ofs_y = 0;

    info = get_setting_info_from_id(VIDEO_SETTING_CLOCK);
    if (info) myclock = info->get_function(context);
    info = get_setting_info_from_id(VIDEO_SETTING_PHASE);
    if (info) phase = info->get_function(context);
    info = get_setting_info_from_id(VIDEO_SETTING_OFFSET_X);
    if (info) ofs_x = info->get_function(context);
    info = get_setting_info_from_id(VIDEO_SETTING_OFFSET_Y);
    if (info) ofs_y = info->get_function(context);
	       
    ret = mode_save_settings(context, myclock, phase, ofs_x, ofs_y);
    
    return ret;
}

int
set_global_settings(vsc_context_t *context)
{
    vsc_setting_info_t *info = vsc_setting_infos;

    MUTEX_LOCK(&global_settings_mtx);
    D(D_VERBOSE, "set_global_settings\n");
    while (info->id != -1) {
	if (info->type == VSC_SETTING_GLOBAL) {
	    int ret;
	    unsigned short opt;

            if(info->cfg_key) {
                pp_cfg_get_short(&opt, info->cfg_key);
	    } else {
                opt = info->def;
	    }

	    ret = info->set_function(context, opt);
	    if (ret != 0) {
		D(D_ERROR, "Error during set of vsc setting %d\n", info->id);
	    }
	}
	info++;	
    }

    MUTEX_UNLOCK(&global_settings_mtx);
    return 0;
}

int
set_kvm_local_settings(vsc_context_t *context)
{
    vsc_setting_info_t *info = vsc_setting_infos;

    MUTEX_LOCK(&kvm_local_settings_mtx);
    D(D_VERBOSE, "set_kvm_local_settings\n");
    while (info->id != -1) {
	if (info->type == VSC_SETTING_KVM_LOCAL) {
	    unsigned short opt;
	    int ret;

            if(info->cfg_key) {
                pp_cfg_get_short(&opt, info->cfg_key,
                                        context->kvm_unit, context->kvm_port);
	    } else {
                opt = info->def;
	    }

	    ret = info->set_function(context, opt);
	    if (ret != 0) {
		D(D_ERROR, "Error during set of vsc setting %d\n", info->id);
	    }
	}
	info++;
    }

    MUTEX_UNLOCK(&kvm_local_settings_mtx);
    return 0;
}

void
send_video_settings_propchange(vsc_context_t *context UNUSED)
{
    pp_propchange_enqueue(PP_PROP_VIDEO_SETTINGS_UPDATE, 0);
}

/* ----------------- internal helper functions --------------------------- */

static vsc_setting_info_t*
get_setting_info_from_id(unsigned char id)
{
    vsc_setting_info_t* info = vsc_setting_infos;

    while (info && info->id != -1 && info->id != id) {
	info++;
    }

    if (info->id == -1) info = NULL;

    return info;
}
