/*********************************************************************
 * libpp_vsc: vsc.c
 *
 * higher level VSC functions
 *
 ********************************************************************/

#include <sys/ioctl.h>
#include <unistd.h>

#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/grab.h>
#include <pp/rfb.h>
#include <lara.h>
#include <pp/intl.h>

#include "adc.h"
#include "debug.h"
#include "mode.h"
#include "offset_pic.h"
#include "settings.h"
#include "timer.h"
#include "vsc_funcs.h"
#include "vsc_io.h"
#include "main.h"

static const unsigned int  MAX_BPT		= 256;
static const unsigned int  MEASURE_SLEEP	= 100000;
static const unsigned int  AA_CLOCK_RETRIES	= 10;
static const unsigned int  AA_PHASE_RETRIES	= 2;
static const unsigned int  AA_CLOCK_RANGE	= 1;
static const unsigned int  AA_CLOCK_RANGE_BF	= 10;
static const unsigned char AA_PHASE_RANGE	= 32;
static const unsigned char AA_PHASE_STEP	= 1;
static const unsigned char AA_PHASE_STEP_BF	= 4;
static const unsigned char AA_CLOCK_STEP	= 2;
static const unsigned char AA_LEN_TOLERANCE	= 10;// percent
static const unsigned char DIFF_THRESHOLD_DEF   = 4; // intensity difference per color
#ifdef PP_FEAT_VSC_PANEL_INPUT
static const unsigned char DIFF_THRESHOLD_OFF   = 0;
#else
static const unsigned char DIFF_THRESHOLD_OFF   = 1;
#endif
static const unsigned char DIFF_THRESHOLD_IRE   = 0;

#define VSC_SDRAM_INIT_WORD	SDR_BURST_LEN(SDR_BURST_LEN_FULL)		| \
				SDR_BURST_TYPE(SDR_BURST_TYPE_SEQUENTIAL)	| \
				SDR_CAS(2)
			    
/* for calculating black pixel
   threshold from black pixel value */
#define R_MSK   0xF800
#define G_MSK   0x07E0
#define B_MSK   0x001F
#define R_POS   11
#define G_POS	5
#define B_POS	0
#define R_SHFT  3
#define G_SHFT	2
#define B_SHFT	3

static void aa_idle(vsc_context_t *context, aa_data_t *aa);
static void aa_offset(vsc_context_t *context, aa_data_t *aa);

#if !defined(PP_FEAT_VSC_PANEL_INPUT)
static void aa_clock(vsc_context_t *context, aa_data_t *aa);
static void aa_phase(vsc_context_t *context, aa_data_t *aa);
static void aa_osd_progress(vsc_context_t *context, aa_data_t *aa);
static void aa_ire(vsc_context_t *context, aa_data_t *aa);
#endif

typedef struct {
    u_short major;
    u_short minor;
    vsc_features_t features;
} vsc_feature_entry_t;

const vsc_features_t vsc_features_default = {
    .extd_sync_switch = 0,
    .sdram_initword   = 0,
    .display_enable   = 1,
};

static vsc_feature_entry_t vsc_feature_table[] = {
    { .major = 2,	.minor = 0,
      .features = {
	    .extd_sync_switch = 0,
	    .sdram_initword   = 0,
	    .display_enable   = 1,
	},
    },   
    { .major = 2,	.minor = 1,
      .features = {
	    .extd_sync_switch = 1,
	    .sdram_initword   = 1,
	    .display_enable   = 1,
	},
    },   
    { .major = 3,	.minor = 0,
      .features = {
	    .extd_sync_switch = 1,
	    .sdram_initword   = 1,
	    .display_enable   = 1,
	    /* FIXME: possibly add CRC diff engine */
	},
    },
    /* end of list marker */
    { .major = 0, .minor = 0,
      .features = {
	    .extd_sync_switch = 0,
	    .sdram_initword   = 0,
	    .display_enable   = 0,
	},
    }
};

void
vsc_check_version_features(vsc_context_t *context)
{
    u_long vr = vsc_rx_reg(context, VSC_REG_VR);
    u_char major = (vr & VR_MAJOR_MASK) >> VR_MAJOR_SHIFT;
    u_char minor = (vr & VR_MINOR_MASK) >> VR_MINOR_SHIFT;
    vsc_feature_entry_t *entry = vsc_feature_table;
    
    D(D_NOTICE, "VSC version reg 0x%08lx -> V %02x.%02x\n", vr, major, minor);

    while (entry->major != 0) {
	if (entry->major == major && entry->minor == minor) {
	    context->feats = entry->features;
	    break;
	}
	entry++;
    }

    if (entry->major == 0) {
	context->feats = vsc_features_default;
    }

#if defined(PP_BOARD_KIRA)
    /* special case, in the first KIRA100 R01 chips the
       display_enable signal is not working */
    
    if (major == 2 && minor == 0) {
	u_char minor_rev = 0;
	
	/* read PMU register to see if we have R01.1 and use display_enable */
	if (ioctl(eric_fd, PPIOCGETKIRAMINORREVISION, &minor_rev) == 0) {
	    D(D_VERBOSE, "Found KIRA100 R01 minor revision %d\n", minor_rev);
	} else {
	    D(D_ERROR, "Error determining KIRA100 R01 minor revision, falling back to R01.0\n");
	}

	context->feats.display_enable = (minor_rev > 0) ? 1 : 0;
    }
#endif
    
    D(D_VERBOSE, "VSC Features...\n"
      " extd_sync_switch = %d\n"
      " sdram_initword   = %d\n"
      " display_enable   = %d\n",
      context->feats.extd_sync_switch,
      context->feats.sdram_initword,
      context->feats.display_enable);
}

#ifdef VSC_SDRAM_INIT_WORD
void
vsc_init_sdram(vsc_context_t *context)
{
    u_long sr_pre, sr_post;
    
    if (context->feats.sdram_initword) {
	D(D_VERBOSE, "Initializing SDRAM with init word 0x%08x\n", VSC_SDRAM_INIT_WORD);
	sr_pre = vsc_rx_reg(context, VSC_REG_SR);
	vsc_tx_reg(context, VSC_REG_SDR, VSC_SDRAM_INIT_WORD);
	sr_post = vsc_rx_reg(context, VSC_REG_SR);
    }
}
#else
void
vsc_init_sdram(vsc_context_t *context UNUSED)
{
    /* nothing to do */
}
#endif

int
vsc_measure_ofs_len(vsc_context_t *context,
		    u_int16_t *ofsX, u_int16_t *ofsY, u_int16_t *lenX,
		    u_int16_t *totalX, u_int16_t *totalY)
{
    vsc_measures_t measures;
    int i;
    unsigned short bpv, bpt;
    u_int16_t offset = 0xFFFF, length = 0x0000;

#if !defined(PP_FEAT_VSC_PANEL_INPUT)
    unsigned short brightness;
    
    /* set adc offset to zero, it helps us find the black level */
    brightness = adc_get_brightness(context);
    adc_set_brightness(context, 0);
#endif

    /* if the caller wishes to know, do an additional
       run with threshold 0 to get the total line length */
    if (totalX) {
	vsc_set_bpt(context, 0);
	if (vsc_get_ofs_len_measures(context, &measures) != 0) {
	    goto error_out;
	}
	
	for (i = 0; i < measures.x_count; i++) {
	    if (measures.lenX[i] > length) {
		length = measures.lenX[i];
	    }
	}

	*totalX = length;
	
	D(D_VERBOSE, "Measured TotalH=%d\n", *totalX);
	length = 0x0000;
    }
    
    usleep(MEASURE_SLEEP);
    
    /* calculate black pixel threshold from
       black pixel on framestart */
    bpv = vsc_rx_reg(context, VSC_REG_BPV);
    bpt = ((((bpv & R_MSK) >> R_POS) << R_SHFT) +
	   (((bpv & G_MSK) >> G_POS) << G_SHFT) +
	   (((bpv & B_MSK) >> B_POS) << B_SHFT));
    
    D(D_NOTICE, "BPV=%04x -> BPT=%d\n", bpv, bpt);
    vsc_set_bpt(context, bpt);
   
    if (vsc_get_ofs_len_measures(context, &measures) != 0) {
	goto error_out;
    }

    /* Horizontal Measurement works like this:
       - get x_count line measurements from kernel
       - take the one which leads to the largest area,
         (minimum offset, maximum length)
    */
    for (i = 0; i < measures.x_count; i++) {
	/*
	D(D_BLABLA, "%3d: ofs=%d, len=%d\n",
	  i, measures.ofsX[i], measures.lenX[i]);
	*/
	
	if (measures.ofsX[i] < offset) {
	    offset = measures.ofsX[i];
	}
	if (measures.lenX[i] > length) {
	    length = measures.lenX[i];
	}
    }

    if (ofsX) {
	*ofsX = offset;
	D(D_VERBOSE, "Measured OffsetH=%d\n", *ofsX);
    }

    if (ofsY) {
	*ofsY = measures.ofsY;
	D(D_VERBOSE, "Measured OffsetV=%d\n", *ofsY);
    }

    if (totalY) {
	*totalY = measures.totalY;
	D(D_VERBOSE, "Measured TotalV=%d\n", *totalY);
    }
    
    if (lenX) {
	*lenX = length;
	D(D_VERBOSE, "Measured LengthH=%d\n", *lenX);
    }

#if !defined(PP_FEAT_VSC_PANEL_INPUT)
    adc_set_brightness(context, brightness);
#endif
    return 0;
        
 error_out:
    D(D_ERROR, "Measuring Picture failed\n");
    if (ofsX) *ofsX = 0;
    if (ofsY) *ofsY = 0;
    if (totalX) *totalX = 0;
    if (totalY) *totalY = 0;
    if (lenX) *lenX = 0;
#if !defined(PP_FEAT_VSC_PANEL_INPUT)
    adc_set_brightness(context, brightness);
#endif
    return -1;
}

void
vsc_set_sync_input(vsc_context_t *context,
		   vsc_hsync_input_t h, vsc_vsync_input_t v)
{
    u_int32_t new = 0;

    if (h == HSYNC_ADC) new |= MR_HSYNC_ADC;
    if (v == VSYNC_ADC) new |= MR_VSYNC_ADC;
 
    D(D_VERBOSE, "vsc_set_sync_input: H=%s,V=%s\n",
      (h == HSYNC_ADC) ? "ADC" : "VGA",
      (v == VSYNC_ADC) ? "ADC" : "VGA");
   
    vsc_tx_masked(context, VSC_REG_MR, new, MR_HSYNC_ADC | MR_VSYNC_ADC);
}

void
vsc_set_sync_sample_input(vsc_context_t *context,
			  vsc_hsync_input_t h, vsc_vsync_input_t v)
{
    u_int32_t new = 0;
    
    assert(context->feats.extd_sync_switch);

    if (h == HSYNC_ADC) new |= MR_HSYNC_ADC;
    if (v == VSYNC_ADC) new |= MR_VSYNC_ADC;
 
    D(D_VERBOSE, "vsc_set_sync_SAMPLE_input: H=%s,V=%s\n",
      (h == HSYNC_ADC) ? "ADC" : "VGA",
      (v == VSYNC_ADC) ? "ADC" : "VGA");
   
    vsc_tx_masked(context, VSC_REG_MR, new, MR_HSYNC_ADC | MR_VSYNC_ADC);
}

void
vsc_set_sync_irq_input(vsc_context_t *context,
		       vsc_hsync_input_t h, vsc_vsync_input_t v)
{
    u_int32_t new = 0;
    
    assert(context->feats.extd_sync_switch);

    if (h == HSYNC_ADC) new |= MR_HSIRQ_ADC;
    if (v == VSYNC_ADC) new |= MR_VSIRQ_ADC;
 
    D(D_VERBOSE, "vsc_set_sync_IRQ_input: H=%s,V=%s\n",
      (h == HSYNC_ADC) ? "ADC" : "VGA",
      (v == VSYNC_ADC) ? "ADC" : "VGA");
   
    vsc_tx_masked(context, VSC_REG_MR, new, MR_HSIRQ_ADC | MR_VSIRQ_ADC);
}

void
vsc_set_sync_reg_input(vsc_context_t *context,
		       vsc_hsync_input_t h, vsc_vsync_input_t v)
{
    u_int32_t new = 0;
    
    assert(context->feats.extd_sync_switch);

    if (h == HSYNC_ADC) new |= MR_HSHOW_ADC;
    if (v == VSYNC_ADC) new |= MR_VSHOW_ADC;
 
    D(D_VERBOSE, "vsc_set_sync_REG_input: H=%s,V=%s\n",
      (h == HSYNC_ADC) ? "ADC" : "VGA",
      (v == VSYNC_ADC) ? "ADC" : "VGA");
   
    vsc_tx_masked(context, VSC_REG_MR, new, MR_HSHOW_ADC | MR_VSHOW_ADC);
}

#ifdef PP_FEAT_VSC_PANEL_INPUT
void
vsc_correct_offsets(vsc_context_t *context UNUSED)
{
}
#else /* !PP_FEAT_VSC_PANEL_INPUT */
/*
  correct vsc sampling offsets to
  a) avoid distortion of the visible area horizontally
     (offset + active length has to be < adc clocks)
  b) avoid old screen garbage at the bottom of the screen
     (offset + active length has to be < total frame length)
*/
void
vsc_correct_offsets(vsc_context_t *context)
{
    unsigned short total  = adc_get_clocks(context);
    unsigned short length = vsc_rx_reg_shadowed(context, VSC_REG_HL);
    unsigned short offset = vsc_get_offset_x(context);
    pp_bool_t error;
    
    /* horizontal */
    if (offset + length > total - 2) {
	offset = total - length - 2;
	D(D_VERBOSE, "vsc_correct_offset x=%d\n", offset);
 	vsc_set_offset_x(context, offset);
     }

    /* vertical */
    length = vsc_rx_reg_shadowed(context, VSC_REG_VL);
    total  = vsc_rx_reg_secure(context, VSC_REG_MVT, &error);
    offset = vsc_get_offset_y(context);

    if ((error == PP_FALSE) && 
	(length < total)    &&
	(offset + length > total)) {
	offset = total - length;
	D(D_VERBOSE, "vsc_correct_offset y=%d\n", offset);
	vsc_set_offset_y(context, offset);
    }    
}
#endif /* !PP_FEAT_VSC_PANEL_INPUT */

vsc_output_state_t
vsc_output_state(vsc_context_t *context)
{
    vsc_output_state_t ret = OUTPUT_VALID;
    
    /* check if VSC is programmed */
    if (context->vsc_active == PP_FALSE) {
	ret = OUTPUT_INVALID;
    }
	
    /* check if we have both syncs */
    if (context->sync_signals != HV) {
	ret = OUTPUT_INVALID;
    }

    /* check if video mode is known */
    if (context->current_mode == NULL) {
	ret = OUTPUT_INVALID;
    }

    /* a mode was set up correctly */
    if (context->mode_valid == PP_FALSE) {
	ret = OUTPUT_INVALID;
    }
    
    return ret;
}

video_signal_state_t
vsc_has_signal(vsc_context_t *context)
{
    if (context->vsc_active == PP_FALSE) {
	return VIDEO_SIGNAL_UNKNOWN;
    }
    
    switch (context->sync_signals) {
      case HV:
 	  return VIDEO_SIGNAL_ON;
      case H:
      case V:
      case NONE:
	  return VIDEO_SIGNAL_OFF;
      default:
	  return VIDEO_SIGNAL_UNKNOWN;
    }
}

/**************************************************************
 * AutoAdjustment functions/states
 **************************************************************/

/* handles one(!) autoadjustment step, called from the
   main state machine, keeps its own state in context:
   -AA_CLOCK_PRE - adjust clock so that xlength fits
   -AA_PHASE	 - get best phase using rescan error reg
   -AA_CLOCK_POST- readjust clock after phase correction
		   because it may be off +/- 1,
		   if length doesn't fit anymore,
		   go to AA_PHASE again because it may
		   have been measured wrong (chicken/egg)		   
   -AA_OFFSET	 - adjust offsets
*/

#ifdef PP_FEAT_VSC_PANEL_INPUT
void
vsc_aa_advance(vsc_context_t *context)
{
    aa_data_t *aa = &context->aa_data;

    if (context->feats.display_enable) {
	D(D_NOTICE, "AA not necessary when using display_enable\n");
	return;
    }
    
    switch (aa->state) {
      case AA_IDLE:
	  aa_idle(context, aa);
	  break;
      case AA_OFFSET:
	  aa_offset(context, aa);
	  break;
      default:
	  /* do nothing */
	  break;
    }
}

void
vsc_aa_stop(vsc_context_t *context)
{
    context->aa_data.state = AA_IDLE;
}

static void
aa_idle(vsc_context_t *context, aa_data_t *aa)
{
    // check if we may do aa, else just leave
    if (context->vsc_active == PP_FALSE ||
	context->current_mode == NULL ||
	context->sync_signals == V ||
	context->sync_signals == NONE) {
	D(D_ERROR, "Auto adjustment not possible right now\n");
	return;
    }

    if (vsc_encoder_hook && vsc_encoder_hook->vsc_event) {
	pp_vsc_event_t event;
    
        event.type = VSC_EVENT_AA_ADJUSTMENT;
        event.event.aa_progress.type = VSC_AA_IN_PROGRESS;
        vsc_encoder_hook->vsc_event(context->id, &event);
    }

    aa->state = AA_OFFSET;
}

static void
aa_offset(vsc_context_t *context, aa_data_t *aa)
{
    u_int16_t ofsX, ofsY;

    vsc_set_offset_x(context, 0);
    vsc_set_offset_y(context, 0);

    sleep(1);
    
    if (PP_SUCCED(offset_pic_measurement(context, &ofsX, &ofsY))) {
	D(D_VERBOSE, "offset_pic_measurement gave %d,%d\n", ofsX, ofsY);

	vsc_set_offset_x(context, ofsX);
	vsc_set_offset_y(context, ofsY);
    } else {
	D(D_VERBOSE, "offset_pic_measurement failed\n");
    }

    send_video_settings_propchange(context);
    save_mode_local_settings(context);
    
    pp_cfg_save(DO_FLUSH);

    if (vsc_encoder_hook && vsc_encoder_hook->vsc_event) {
	pp_vsc_event_t event;
	
        event.type = VSC_EVENT_AA_ADJUSTMENT;
        event.event.aa_progress.type = VSC_AA_SUCCESS;
        vsc_encoder_hook->vsc_event(context->id, &event);
    }
    aa->state = AA_IDLE;
    vsc_aa_stop(context);
}

#else /* !PP_FEAT_VSC_PANEL_INPUT */

void
vsc_aa_advance(vsc_context_t *context)
{
    aa_data_t *aa = &context->aa_data;
	
    switch (aa->state) {
      case AA_IDLE:
	  aa_idle(context, aa);
	  break;
      case AA_CLOCK_PRE:
      case AA_CLOCK_POST:
	  aa_clock(context, aa);
	  break;
      case AA_PHASE:
	  aa_phase(context, aa);
	  break;
      case AA_OFFSET:
	  aa_offset(context, aa);
	  break;
      case AA_IRE:
	  aa_ire(context, aa);
	  break;  
    }
}

void
vsc_aa_stop(vsc_context_t *context)
{
    pp_vsc_event_t event;
    
    while (context->aa_data.grab_suspend) {
	pp_grab_suspend(context->id, GRAB_RESUME);
	context->aa_data.grab_suspend--;
    }

    if (context->aa_data.state != AA_IDLE) {
	D(D_VERBOSE, "AA interrupted, restoring settings\n");
        if (vsc_encoder_hook && vsc_encoder_hook->vsc_event) {
            event.type = VSC_EVENT_AA_ADJUSTMENT;
            event.event.aa_progress.type = VSC_AA_INTERRUPTED;
            vsc_encoder_hook->vsc_event(context->id, &event);
        }

	// reset to old values, aa did not finish
	vsc_set_offset_x(context, context->aa_data.ofsx_bak);
	vsc_set_offset_y(context, context->aa_data.ofsy_bak);
	adc_set_clocks(context, context->aa_data.clock_bak);
	adc_set_phase(context, context->aa_data.phase_bak);
    }

    vsc_set_diff_threshold(context);
    context->aa_data.state = AA_IDLE;
}

static void
aa_idle(vsc_context_t *context, aa_data_t *aa)
{
    // check if we may do aa, else just leave
    if (context->vsc_active == PP_FALSE ||
	context->current_mode == NULL ||
	context->sync_signals == V ||
	context->sync_signals == NONE) {
	D(D_ERROR, "Auto adjustment not possible right now\n");
	return;
    }

    if (context->aa_data.ire_process) {
	D(D_VERBOSE, "AA_IDLE -> AA_IRE\n");
	pp_grab_suspend(context->id, GRAB_SUSPEND);
	vsc_tx_reg(context, VSC_REG_DTH, DIFF_THRESHOLD_IRE);
	aa->state = AA_IRE;
	return;
    }
    
    // save old values in case we are interrupted
    context->aa_data.ofsx_bak = vsc_get_offset_x(context);
    context->aa_data.ofsy_bak = vsc_get_offset_y(context);
    context->aa_data.clock_bak= adc_get_clocks(context);
    context->aa_data.phase_bak= adc_get_phase(context);

    // initialize aa specific variables, lets go
    pp_hrtime_start(&aa->time);
    adc_set_phase(context, AA_PHASE_RANGE/2);
    aa_osd_progress(context, aa);
    aa->clock_retry = AA_CLOCK_RETRIES;
    aa->brute_force = PP_FALSE;
    aa->grab_suspend = 0;

    aa->state = AA_CLOCK_PRE;
}

static void
aa_clock(vsc_context_t *context, aa_data_t *aa)
{
    u_int16_t ofsX, ofsY, lenX, resX, totalY;
    int diff;
    pp_vsc_event_t event;
    
    switch (aa->state) {
      case AA_CLOCK_PRE:
	  aa->phase_retry = AA_PHASE_RETRIES;
	  aa->do_phase    = PP_TRUE;
	  //fallthrough
      case AA_CLOCK_POST:
	  if (--aa->clock_retry == 0) {
	    // too many tries to get near the correct length, abort
	    D(D_NOTICE, "AA retrycount reached, aborting\n");
            if (vsc_encoder_hook && vsc_encoder_hook->vsc_event) {
                event.type = VSC_EVENT_AA_ADJUSTMENT;
                event.event.aa_progress.type = VSC_AA_FAILED;
                vsc_encoder_hook->vsc_event(context->id, &event);
            }
         
	      aa->state = AA_IDLE;
	      vsc_aa_stop(context);
	      break;
	  }

	  /* get target length and measure horizontal length,
	     check if its too far away
	     (no line on screen has the full non-black length) */
	  if (mode_get_res_info(context, &resX, NULL, NULL, NULL) != 0) {
	      aa->state = AA_IDLE;
	      vsc_aa_stop(context);
	      break;
	  }
	  vsc_measure_ofs_len(context, &ofsX, &ofsY, &lenX, NULL, &totalY);
	  D(D_VERBOSE, "AA 1st OfsX,Y=%d,%d LenX=%d, TotalY=%d\n",
	    ofsX, ofsY, lenX, totalY);
      
	  diff = lenX - resX;
	  if (abs(diff) > resX * AA_LEN_TOLERANCE/100) {
	      /* if the length is too far away,
		 use the old TFT style AA (check clock/phase ranges only) */
	      D(D_NOTICE, "AA X length too far away, switching to brute force mode\n");
	      aa->brute_force = PP_TRUE;
	  }
  
	  if (abs(diff) <= 1 || aa->brute_force == PP_TRUE) {
	      /* length fits, what to do now:
		 -we are in AA_CLOCK_PRE : do phase measurement always
		 -we are in AA_CLOCK_POST: do phase measurement only
		 when a length difference > 0 was seen before and
		 we did not too many retries before
	      */
	      if (aa->do_phase) {
		  // calculate and initialize ranges for phase AA
		  if (aa->brute_force == PP_TRUE) {
		      aa->clock		= adc_get_clocks(context) - AA_CLOCK_RANGE_BF/2;
		      aa->clock_max	= aa->clock + AA_CLOCK_RANGE_BF;
		  } else {
		      aa->clock		= adc_get_clocks(context) - AA_CLOCK_RANGE/2;
		      aa->clock_max	= aa->clock + AA_CLOCK_RANGE;
		  }

		  aa->clock     = aa->clock & ~0x01;
		  aa->clock_max = aa->clock_max & ~0x01;
		  aa->phase	= 0;
		  aa->min_ire	= 0xFFFFFFFF;
  
		  adc_set_clocks(context, aa->clock);
		  vsc_set_offset_x(context, ofsX);
		  vsc_set_offset_y(context, ofsY);
  
		  vsc_correct_offsets(context);
		  aa->grab_suspend++;		  
		  pp_grab_suspend(context->id, GRAB_SUSPEND);

		  vsc_zero_diff_threshold(context);

		  aa->state = AA_PHASE;
	      } else {
		  aa->state = AA_OFFSET;
	      }
	  } else {
	      /* length is not okay, adjust clocks so we get near the
		 right one.
		 if we are in AA_CLOCK_POST, remember that another
		 phase measurement is needed after the clock was
		 corrected */
	      u_int16_t clocks = adc_get_clocks(context);
	      u_int16_t clocks_new = clocks - diff;

	      clocks_new = clocks_new & ~0x01;
	      
	      adc_set_clocks(context, clocks_new);
	      vsc_correct_offsets(context);
	      D(D_VERBOSE, "Try %2d: XDiff = %d, Clock %d -> %d\n",
		AA_CLOCK_RETRIES-aa->clock_retry, diff, clocks, clocks_new);

	      if (aa->state == AA_CLOCK_POST && aa->phase_retry > 0) {
		  aa->do_phase = PP_TRUE;
	      }
	  }
      default:
	  /* nothing to do */
	  break;
    }
}

static void
aa_phase(vsc_context_t *context, aa_data_t *aa)
{
    u_int32_t ire;

    u_char video_link = context->id;

    adc_set_phase(context, aa->phase);
        
    /* the same phase for all VSCs, use the first one */
    if (ioctl(pp_grab_fd(video_link), PPIOCVSCAUTOADJUSTSTEP, NULL) == -1) {
	D(D_ERROR, "AA PPIOCVSCAUTOADJUSTSTEP failed\n");
	vsc_aa_stop(context);
	return;
    }
    ire = vsc_rx_reg(context, VSC_REG_IRE);
    D(D_VERBOSE, "AA Clock=%4d,Phase=%02d->IRE=%d\n",
      aa->clock, aa->phase, ire);

    aa_osd_progress(context, aa);

    // get phase with minimum ImageRescanError
    if (ire < aa->min_ire) {
	aa->min_ire = ire;
	aa->clock_opt = aa->clock;
	aa->phase_opt = aa->phase;
    }
    aa->phase += (aa->brute_force == PP_TRUE) ? AA_PHASE_STEP_BF : AA_PHASE_STEP;
    if (aa->phase >= AA_PHASE_RANGE) {
	aa->phase = 0;
	aa->clock += AA_CLOCK_STEP;
	adc_set_clocks(context, aa->clock);
    }
    if (aa->clock > aa->clock_max) {
		  
	D(D_VERBOSE, "Optimum: Clock=%d,Phase=%d\n",
	  aa->clock_opt, aa->phase_opt);

	adc_set_clocks(context, aa->clock_opt);
	adc_set_phase(context, aa->phase_opt);

	vsc_set_diff_threshold(context);
	while (aa->grab_suspend) {
	    pp_grab_suspend(context->id, GRAB_RESUME);
	    aa->grab_suspend--;
	}

	aa->phase_retry--;
	aa->do_phase = PP_FALSE;
	aa->clock_retry = AA_CLOCK_RETRIES;

	aa->state = (aa->brute_force == PP_TRUE) ? AA_OFFSET : AA_CLOCK_POST;
    }
}

static void
aa_offset(vsc_context_t *context, aa_data_t *aa)
{
    u_int16_t ofsX, ofsY, lenX, totalY;
    unsigned int timediff;
    pp_vsc_event_t event;
	      
    // measure offsets
    vsc_measure_ofs_len(context, &ofsX, &ofsY, &lenX, NULL, &totalY);
    D(D_VERBOSE, "AA 2nd OfsX,Y=%d,%d LenX=%d, TotalY=%d\n",
      ofsX, ofsY, lenX, totalY);
	      
    vsc_set_offset_x(context, ofsX);
    vsc_set_offset_y(context, ofsY);
    vsc_correct_offsets(context);
	           
    send_video_settings_propchange(context);

    pp_hrtime_stop(&aa->time);
    timediff = pp_hrtime_read(&aa->time) / 1000000;
    D(D_VERBOSE, "AA done, t=%d ms\n", timediff);

    save_mode_local_settings(context);
    pp_cfg_save(DO_FLUSH);

    if (vsc_encoder_hook && vsc_encoder_hook->vsc_event) {
        event.type = VSC_EVENT_AA_ADJUSTMENT;
        event.event.aa_progress.type = VSC_AA_SUCCESS;
        vsc_encoder_hook->vsc_event(context->id, &event);
    }
    aa->state = AA_IDLE;
    vsc_aa_stop(context);
}

static void
aa_osd_progress(vsc_context_t *context, aa_data_t *aa UNUSED)
{
    pp_vsc_event_t event;
    
    if (vsc_encoder_hook && vsc_encoder_hook->vsc_event) {
        event.type = VSC_EVENT_AA_ADJUSTMENT;
        event.event.aa_progress.type = VSC_AA_IN_PROGRESS;
        vsc_encoder_hook->vsc_event(context->id, &event);
    }
}

static void
aa_ire(vsc_context_t *context, aa_data_t *aa)
{
    u_int32_t ire;
    u_char video_link = context->id;
    
    if (aa->ire_process == PP_FALSE) {
	D(D_VERBOSE, "Stopping IRE");
	goto stop_ire;
    }
    
    if (ioctl(pp_grab_fd(video_link), PPIOCVSCAUTOADJUSTSTEP, NULL) == -1) {
	D(D_ERROR, "AA PPIOCVSCAUTOADJUSTSTEP (for IRE) failed\n");
	goto stop_ire;
    }
    ire = vsc_rx_reg(context, VSC_REG_IRE);
    DCH(D_ALWAYS, "IRE=0x%08x\n", ire);

    return;
    
 stop_ire:
    pp_grab_suspend(context->id, GRAB_RESUME);
    vsc_set_diff_threshold(context);
    aa->state = AA_IDLE;    
    return;
}

#endif /* !PP_FEAT_VSC_PANEL_INPUT */

void
vsc_update_sync_deltas(vsc_context_t *context)
{
    int delta_h, delta_v;

    if (pp_cfg_get_int(&delta_h, "video.vsc.sync_delta.h") != PP_SUC ||
	pp_cfg_get_int(&delta_v, "video.vsc.sync_delta.v") != PP_SUC) {
	D(D_ERROR, "Could not get sync deltas\n");
    }

    D(D_VERBOSE, "vsc_update_sync_deltas: h=%d, v=%d\n", delta_h, delta_v);

    context->sync_delta_h = delta_h;
    context->sync_delta_v = delta_v;
    
    vsc_tx_reg(context, VSC_REG_HSD, context->sync_delta_h);
    vsc_tx_reg(context, VSC_REG_VSD, context->sync_delta_v);
}

void
vsc_update_diff_thresholds(void)
{
    vsc_set_diff_threshold(&vsc_contexts[0]);
}

void
vsc_update_sun_mode(vsc_context_t *context)
{
    int sun_mode;

    context->sun_mode = PP_FALSE;
    pp_cfg_is_enabled(&sun_mode, "video.vsc.sun_mode");
    context->sun_mode = sun_mode ? PP_TRUE : PP_FALSE;
    D(D_VERBOSE, "vsc_update_sun_mode %d\n", context->sun_mode);
}

void
vsc_set_diff_threshold(vsc_context_t *context)
{
#ifdef PP_FEAT_VSC_PANEL_INPUT
    int dth = 0;
#else
    int dth = DIFF_THRESHOLD_DEF;

    pp_cfg_get_int(&dth, "video.diff_tolerance");
#endif
    
    D(D_VERBOSE, "vsc_set_diff_threshold: %d\n", dth);
    vsc_tx_reg(context, VSC_REG_DTH, dth);
}

void
vsc_zero_diff_threshold(vsc_context_t *context)
{
    D(D_VERBOSE, "vsc_zero_diff_threshold: %d\n", DIFF_THRESHOLD_OFF);
    vsc_tx_reg(context, VSC_REG_DTH, DIFF_THRESHOLD_OFF);
}

void
vsc_sync_check_reset(vsc_context_t *context)
{
    if (ioctl(context->fd, PPIOCVSCRSTSYNCSPEED, NULL) == -1) {
	if (errno != EAGAIN) {
	    D(D_ERROR, "Error during ioctl(PPIOCVSCRSTSYNCSPEED): %s\n",
	      strerror(errno));
	}
    }       
}

void
vsc_syncs_on(vsc_context_t *context, pp_bool_t *hs, pp_bool_t *vs)
{
    vsc_sync_speed_t sync_speed;
    *hs = *vs = PP_TRUE;

    vsc_sync_check_reset(context);

    // waiting a single frame time (worst case 200us) is enough here
    usleep(200);
    
    if (ioctl(context->fd, PPIOCVSCGETSYNCSPEED, &sync_speed) == -1) {
	if (errno != EAGAIN) {
	    D(D_ERROR, "Error during ioctl(PPIOCVSCGETSYNCSPEED): %s\n",
	      strerror(errno));
	}
    }

    *hs = (sync_speed.hs_slow) ? PP_FALSE : PP_TRUE;
    *vs = (sync_speed.vs_slow) ? PP_FALSE : PP_TRUE;
}

void
vsc_sync_irq_state(vsc_context_t *context, pp_bool_t state)
{
    DCH(D_VERBOSE, "vsc_sync_irq_state: %d\n", state);
    
#ifdef PP_FEAT_VSC_PANEL_INPUT
    (void)context;
    (void)state;
    DCH(D_VERBOSE, "vsc_sync_irq_state: ignored for panel input\n");
    return;
#endif

    if (context->feats.extd_sync_switch) {
	DCH(D_VERBOSE, "vsc_sync_irq_state: ignored for ext sync switch\n");
	return;
    }

    if (state == PP_TRUE) {
	if (vsc_get_sync_irq(context, PP_FALSE)) {
	    D(D_BLABLA,"ignored pending sync irq\n");
	}
	context->ignore_irq = PP_FALSE;
    } else {
	context->ignore_irq = PP_TRUE;
    }
}

/**************************************************************
 * set/get functions for settings struct with common interface
 **************************************************************/

int
vsc_set_offset_x(vsc_context_t *context, unsigned short offset)
{    
    DCH(D_BLABLA, "vsc_set_offset_x %d\n", offset);
    vsc_tx_reg(context, VSC_REG_HO, offset);
   
    return 0;
}

int
vsc_set_offset_y(vsc_context_t *context, unsigned short offset)
{
    DCH(D_BLABLA, "vsc_set_offset_y %d\n", offset);
    vsc_tx_reg(context, VSC_REG_VO, offset);

    return 0;
}

int
vsc_set_bpt(vsc_context_t *context, unsigned short threshold)
{
    if (threshold > MAX_BPT) threshold = MAX_BPT;
/* FIXME: hack, must be investigated, automatic black level measure doesn't seem to work correctly */
#if defined(PRODUCT_KX2)
    threshold = 8; 
#endif
    D(D_BLABLA, "vsc_set_bpt: %d\n", threshold);
    vsc_tx_reg(context, VSC_REG_BPT, threshold);

    return 0;
}

unsigned short
vsc_get_offset_x(vsc_context_t *context)
{   
    return vsc_rx_reg_shadowed(context, VSC_REG_HO);
}

unsigned short
vsc_get_offset_y(vsc_context_t *context)
{
    return vsc_rx_reg_shadowed(context, VSC_REG_VO);
}

unsigned short
vsc_get_bpt(vsc_context_t *context)
{
    return vsc_rx_reg_shadowed(context, VSC_REG_BPT);
}

unsigned char
vsc_use_2nd_sync(vsc_context_t *context)
{
#ifdef LARA_KACY
    return ((vsc_rx_reg(context, VSC_REG_SR) & SR_DCIM) == SR_DCIM);
#else
    (void)context;
    return 0;
#endif
}
