#include "fault.h"
#include "wsman.h"
#include "xml_writer.h"

/******************************************************************************
* fault data                                                                  *
******************************************************************************/

/* details code lists */
static char* fault_details_action_not_supported[] = {
    /* FAULT_DETAIL_NOT_SUPPORTED */
    NULL,
    /* FAULT_DETAIL_NOT_ACTION_MISMATCH */
    "ActionMismatch",
    /* FAULT_DETAIL_READ_ONLY */
    "ReadOnly",
};

static char* fault_details_dest_unreachable[] = {
    /* FAULT_DETAIL_INVALID_RESOURCE_URI */
    "InvalidResourceURI",
};

static char* fault_details_encoding_limit[] = {
    /* FAULT_DETAIL_URI_LIMIT_EXCEEDED */
    "URILimitExceeded",
    /* FAULT_DETAIL_MAX_ENVELOPE_SIZE */
    "MaxEnvelopeSize",
    /* FAULT_DETAIL_MAX_ENVELOPE_SIZE_EXCEEDED */
    "MaxEnvelopeSizeExceeded",
    /* FAULT_DETAIL_SERVICE_ENVELOPE_LIMIT */
    "ServiceEnvelopeLimit",
    /* FAULT_DETAIL_SELECTOR_LIMIT */
    "SelectorLimit",
    /* FAULT_DETAIL_OPTION_LIMIT */
    "OptionLimit",
    /* FAULT_DETAIL_CHARACTER_SET */
    "CharacterSet",
    /* FAULT_DETAIL_UNREPORTABLE_SUCCESS */
    "UnreportableSuccess",
    /* FAULT_DETAIL_WHITESPACE */
    "WhiteSpace",
    /* FAULT_DETAIL_ENCODING_TYPE */
    "EncodingType",
    /* FAULT_DETAIL_MINIMUM_ENVELOPE_LIMIT */
    "MinimumEnvelopeLimit",
};

static char* fault_details_ep_unavailable[] = {
    /* FAULT_DETAIL_NONE */
    NULL,
    /* FAULT_DETAIL_RESOURCE_OFFLINE */
    "ResourceOffline",
};

static char* fault_details_unable_to_process[] = {
    /* FAULT_DETAIL_NONE */
    NULL,
    /* FAULT_DETAIL_UNUSABLE_ADDRESS */
    "UnusableAddress",
};

static char* fault_details_filter_unavailable[] = {
    /* FAULT_DETAIL_NONE */
    NULL,
    /* FAULT_DETAIL_FILTERING_REQUIRED */
    "FilteringRequired",
};

static char* fault_details_invalid_bookmark[] = {
    /* FAULT_DETAIL_BOOKMARK_EXPIRED */
    "Expired",
    /* FAULT_DETAIL_BOOKMARK_INVALID */
    "Invalid",
};

static char* fault_details_invalid_options[] = {
    /* FAULT_DETAIL_OPTION_NOT_SUPPORTED */
    "NotSupported",
    /* FAULT_DETAIL_OPTION_INVALID_NAME */
    "InvalidName",
    /* FAULT_DETAIL_OPTION_INVALID_VALUE */
    "InvalidValue",
};

static char* fault_details_invalid_parameter[] = {
    /* FAULT_DETAIL_PARAMETER_TYPE_MISMATCH */
    "TypeMismatch",
    /* FAULT_DETAIL_PARAMETER_INVALID_NAME */
    "InvalidName",
};

static char* fault_details_invalid_representation[] = {
    /* FAULT_DETAIL_REPRESENTATION_INVALID_VALUES */
    "InvalidValues",
    /* FAULT_DETAIL_REPRESENTATION_MISSING_VALUES */
    "MissingValues",
    /* FAULT_DETAIL_REPRESENTATION_INVALID_NAMESPACE */
    "InvalidNamespace",
    /* FAULT_DETAIL_REPRESENTATION_INVALID_FRAGMENT */
    "InvalidFragment",
};

static char* fault_details_invalid_selectors[] = {
    /* FAULT_DETAIL_NONE */
    NULL,
    /* FAULT_DETAIL_SELECTOR_INSUFFICIENT_SELECTORS */
    "InsufficientSelectors",
    /* FAULT_DETAIL_SELECTOR_UNEXPECTED_SELECTORS */
    "UnexpectedSelectors",
    /* FAULT_DETAIL_SELECTOR_TYPE_MISMATCH */
    "TypeMismatch",
    /* FAULT_DETAIL_SELECTOR_INVALID_VALUE */
    "InvalidValue",
    /* FAULT_DETAIL_SELECTOR_AMBIGUOUS_SELECTORS */
    "AmbiguousSelectors",
    /* FAULT_DETAIL_SELECTOR_DUPLICATE_SELECTORS */
    "DuplicateSelectors",
};

static char* fault_details_rename_failure[] = {
    /* FAULT_DETAIL_INVALID_RESOURCE_URI */
    "InvalidResourceURI",
    /* FAULT_DETAIL_TARGET_ADDRESS_EXISTS */
    "TargetAlreadyExists",
    /* FAULT_DETAIL_RENAME_INVALID_ADDRESS */
    "InvalidAddress",
    /* FAULT_DETAIL_INVALID_SELECTOR_ASSIGNMENT */
    "InvalidSelectorAssignment"
};

static char* fault_details_unsupported_feature[] = {
    /* FAULT_DETAIL_UNSUPPORTED_AUTHORIZATION_MODE */
    "AuthorizationMode",
    /* FAULT_DETAIL_UNSUPPORTED_ADDRESSING_MODE */
    "AddressingMode",
    /* FAULT_DETAIL_UNSUPPORTED_ACK */
    "Ack",
    /* FAULT_DETAIL_UNSUPPORTED_OPERATION_TIMEOUT */
    "OperationTimeout",
    /* FAULT_DETAIL_UNSUPPORTED_LOCALE */
    "Locale",
    /* FAULT_DETAIL_UNSUPPORTED_EXPIRATION_TIME */
    "ExpirationTime",
    /* FAULT_DETAIL_UNSUPPORTED_FRAGMENT_LEVEL_ACCESS */
    "FragmentLevelAccess",
    /* FAULT_DETAIL_UNSUPPORTED_DELIVERY_RETRIES */
    "DeliveryRetries",
    /* FAULT_DETAIL_UNSUPPORTED_HEARTBEATS */
    "Heartbeats",
    /* FAULT_DETAIL_UNSUPPORTED_BOOKMARKS */
    "Bookmarks",
    /* FAULT_DETAIL_UNSUPPORTED_MAX_ELEMENTS */
    "Maxelements",
    /* FAULT_DETAIL_UNSUPPORTED_MAX_TIME */
    "MaxTime",
    /* FAULT_DETAIL_UNSUPPORTED_MAX_ENVELOPE_SIZE */
    "MaxEnvelopeSize",
    /* FAULT_DETAIL_UNSUPPORTED_MAX_ENVELOPE_POLICY */
    "MaxEnvelopePolicy",
    /* FAULT_DETAIL_UNSUPPORTED_FILTERING_REQUIRED */
    "FilteringRequired",
    /* FAULT_DETAIL_UNSUPPORTED_INSECURE_ADDRESS */
    "InsecureAddress",
    /* FAULT_DETAIL_UNSUPPORTED_FORMAT_MISMATCH */
    "FormatMismatch",
    /* FAULT_DETAIL_UNSUPPORTED_FORMAT_SECURITY_TOKEN */
    "FormatSecurityToken",
    /* FAULT_DETAIL_UNSUPPORTED_MISSING_VALUES */
    "MissingValues",
    /* FAULT_DETAIL_UNSUPPORTED_INVALID_VALUES */
    "InvalidValues",
    /* FAULT_DETAIL_UNSUPPORTED_INVALID_NAMESPACE */
    "InvalidNamespace",
    /* FAULT_DETAIL_UNSUPPORTED_RENAME */
    "Rename",
    /* FAULT_DETAIL_UNSUPPORTED_OPTION_SET */
    "OptionSet",
};

#define FAULT_CODE_SENDER	"s:Sender"
#define FAULT_CODE_RECEIVER	"s:Receiver"

typedef struct {
    char *uri;
    char *code;
    char *subcode;
    char *reason;
    char **details;
} fault_desc_t;

static fault_desc_t faults[NUMBER_FAULTS] = {
    /* FAULT_SUCCESS */
    {
    	FAULT_URI_ADDRESSING,
    	"",
    	"",
    	"",
    	NULL,
    },
    /* FAULT_NOT_UNDERSTOOD */
    {
    	FAULT_URI_ADDRESSING,
    	"s:MustUnderStand",
    	"",
    	"Header not understood",
    	NULL,	/* no details here */
    },
    /* FAULT_ACCESS_DENIED */
    {
    	FAULT_URI_MANAGEMENT,
    	FAULT_CODE_SENDER,
    	"wsman:AccessDenied",
    	"The sender was not authorized to access the resource",
    	NULL,	/* no details here */
    },
    /* FAULT_NO_ACK */
    {
    	FAULT_URI_MANAGEMENT,
    	FAULT_CODE_SENDER,
    	"wsman:NoAck",
    	"The receiver did not acknowledge the event delivery",
    	NULL,	/* no details here */
    },
    /* FAULT_ACTION_NOT_SUPPORTED */
    {
    	FAULT_URI_ADDRESSING,
    	FAULT_CODE_SENDER,
    	"wsa:ActionNotSupported",
    	"The action is not supported by the service",
    	fault_details_action_not_supported,
    		/* details are submitted both as string (the invalid action URI) and as sub code */
    },
    /* FAULT_CONCURRENCY */
    {
    	FAULT_URI_MANAGEMENT,
    	FAULT_CODE_SENDER,
    	"wsman:Concurrency",
    	"The action could not be completed due to concurrency or locking problems",
    	NULL,	/* no details here */
    },
    /* FAULT_ALREADY_EXISTS */
    {
    	FAULT_URI_MANAGEMENT,
    	FAULT_CODE_SENDER,
    	"wsman:AlreadyExists",
    	"The sender attempted to create a resource which already exists",
    	NULL,	/* no details here */
    },
    /* FAULT_CANNOT_PROCESS_FILTER */
    {
    	FAULT_URI_ENUMERATION,
    	FAULT_CODE_SENDER,
    	"wsen:CannotProcessFilter",
    	"The requested filter could not be processed",
    	NULL,	/* details submitted as parameter */
    },
    /* FAULT_DELIVERY_MODE_REQUESTED_UNAVAILABLE */
    {
    	FAULT_URI_EVENTING,
    	FAULT_CODE_SENDER,
    	"wse:DeliveryModeRequestedUnavailable",
    	"The requested delivery mode is not supported",
    	NULL,	/* details contain all supported delivery modes; not used for now */
    },
    /* FAULT_DELIVERY_REFUSED */
    {
    	FAULT_URI_MANAGEMENT,
    	FAULT_CODE_RECEIVER,
    	"wsman:DeliveryRefused",
    	"The receiver refuses to accept delivery of events and requests that the subscription be canceled",
    	NULL,	/* no details here */
    },
    /* FAULT_DESTINATION_UNREACHABLE */
    {
    	FAULT_URI_ADDRESSING,
    	FAULT_CODE_SENDER,
    	"wsa:DestinationUnreachable",
    	"No route can be determined to reach the destination role defined by the WS-Addressing To",
    	fault_details_dest_unreachable,	/* details are submitted both as string and as sub code */
    },
    /* FAULT_ENCODING_LIMIT */
    {
    	FAULT_URI_MANAGEMENT,
    	FAULT_CODE_SENDER,
    	"wsman:EncodingLimit",
    	"An internal encoding limit was exceeded in a request or would be violated if the message were processed",
    	fault_details_encoding_limit,	/* details are submitted both as string and as sub code */
    },
    /* FAULT_ENDPOINT_UNAVAILABLE */
    {
    	FAULT_URI_ADDRESSING,
    	FAULT_CODE_RECEIVER,
    	"wsa:EndpointUnavailable",
    	"The specified endpoint is currently unavailable",
    	fault_details_ep_unavailable,
    },
    /* FAULT_EVENT_SOURCE_UNABLE_TO_PROCESS */
    {
    	FAULT_URI_EVENTING,
    	FAULT_CODE_SENDER,
    	"wse:EventSourceUnableToProcess",
    	"The event source cannot process the subscription",
    	fault_details_unable_to_process,	/* details are submitted both as string and as sub code */
    },
    /* FAULT_FILTER_DIALECT_REQUESTED_UNAVAILABLE */
    {
    	FAULT_URI_ENUMERATION,
    	FAULT_CODE_SENDER,
    	"wsen:FilterDialectRequestedUnavailable",
    	"The requested filtering dialect is not supported",
    	NULL,	/* details: supported filter dialects; not used for now */
    },
    /* FAULT_FILTERING_NOT_SUPPORTED_WSE */
    {
    	FAULT_URI_EVENTING,
    	FAULT_CODE_SENDER,
    	"wse:FilteringNotSupported",
    	"Filtering over the event source is not supported",
    	NULL,	/* no details here */
    },
    /* FAULT_FILTERING_NOT_SUPPORTED_WSEN */
    {
    	FAULT_URI_ENUMERATION,
    	FAULT_CODE_SENDER,
    	"wsen:FilteringNotSupported",
    	"Filtered enumeration is not supported",
    	NULL,	/* no details here */
    },
    /* FAULT_FILTERING_REQUESTED_UNAVAILABLE */
    {
    	FAULT_URI_EVENTING,
    	FAULT_CODE_SENDER,
    	"wse:FilteringRequestedUnavailable",
    	"The requested filter dialect is not supported",
    	fault_details_filter_unavailable,
    		/* details: this details URIs and a list of supported filter dialects; not used for now */
    },
    /* FAULT_INTERNAL_ERROR */
    {
    	FAULT_URI_MANAGEMENT,
    	FAULT_CODE_RECEIVER,
    	"wsman:InternalError",
    	"The service cannot comply with the request due to internal processing errors",
    	NULL,	/* details submitted as parameter */
    },
    /* FAULT_INVALID_BOOKMARK */
    {
    	FAULT_URI_MANAGEMENT,
    	FAULT_CODE_SENDER,
    	"wsman:InvalidBookmark",
    	"The bookmark supplied with the subscription is not valid",
    	fault_details_invalid_bookmark,
    },
    /* FAULT_INVALID_ENUMERATION_CONTEXT */
    {
    	FAULT_URI_ENUMERATION,
    	FAULT_CODE_RECEIVER,
    	"wsen:InvalidEnumerationContext",
    	"The supplied enumeration context is invalid",
    	NULL,	/* no details here */
    },
    /* FAULT_INVALID_EXPIRATION_TIME_WSE */
    {
    	FAULT_URI_EVENTING,
    	FAULT_CODE_SENDER,
    	"wse:InvalidExpirationTime",
    	"Invalid expiration time",
    	NULL,	/* no details here */
    },
    /* FAULT_INVALID_EXPIRATION_TIME_WSEN */
    {
    	FAULT_URI_ENUMERATION,
    	FAULT_CODE_SENDER,
    	"wsen:InvalidExpirationTime",
    	"The expiration time was not valid",
    	NULL,	/* no details here */
    },
    /* FAULT_INVALID_MESSAGE */
    {
    	FAULT_URI_EVENTING,
    	FAULT_CODE_SENDER,
    	"wse:InvalidMessage",
    	"The request message had unknown or invalid content and could not be processed",
    	NULL,	/* details submitted as parameter */
    },
    /* FAULT_INVALID_MESSAGE_INFORMATION_HEADER */
    {
    	FAULT_URI_ADDRESSING,
    	FAULT_CODE_SENDER,
    	"wsa:InvalidMessageInformationHeader",
    	"A message information header is not valid and the message cannot be processed",
    	NULL,	/* details submitted as parameter */
    },
    /* FAULT_INVALID_OPTIONS */
    {
    	FAULT_URI_MANAGEMENT,
    	FAULT_CODE_SENDER,
    	"wsman:InvalidOptions",
    	"One or more options were not valid",
    	fault_details_invalid_options,
    },
    /* FAULT_INVALID_PARAMETER */
    {
    	FAULT_URI_MANAGEMENT,
    	FAULT_CODE_SENDER,
    	"wsman:InvalidParameter",
    	"An operation parameter was not valid",
    	fault_details_invalid_parameter,
    },
    /* FAULT_INVALID_REPRESENTATION */
    {
    	FAULT_URI_TRANSFER,
    	FAULT_CODE_SENDER,
    	"wxf:InvalidRepresentation",
    	"The XML content was invalid",
    	fault_details_invalid_representation,
    },
    /* FAULT_INVALID_SELECTORS */
    {
    	FAULT_URI_MANAGEMENT,
    	FAULT_CODE_SENDER,
    	"wsman:InvalidSelectors",
    	"The Selectors for the resource were not valid",
    	fault_details_invalid_selectors,
    },
    /* FAULT_MESSAGE_INFORMATION_HEADER_REQUIRED */
    {
    	FAULT_URI_ADDRESSING,
    	FAULT_CODE_SENDER,
    	"wsa:MessageInformationHeaderRequired",
    	"A required header was missing",
    	NULL,	/* details submitted as parameter */
    },
    /* FAULT_QUOTA_LIMIT */
    {
    	FAULT_URI_MANAGEMENT,
    	FAULT_CODE_SENDER,
    	"wsman:QuotaLimit",
    	"The service is busy servicing other requests",
    	NULL,	/* details submitted as parameter */
    },
    /* FAULT_RENAME_FAILURE */
    {
    	FAULT_URI_MANAGEMENT,
    	FAULT_CODE_SENDER,
    	"wsman:RenameFailure",
    	"The Selectors for the resource were not valid",
    	fault_details_rename_failure,
    },
    /* FAULT_SCHEMA_VALIDATION_ERROR */
    {
    	FAULT_URI_MANAGEMENT,
    	FAULT_CODE_SENDER,
    	"wsman:SchemaValidationError",
    	"The supplied SOAP violates the corresponding XML Schema definition",
    	NULL,	/* details submitted as parameter */
    },
    /* FAULT_TIMED_OUT_WSEN */
    {
    	FAULT_URI_ENUMERATION,
    	FAULT_CODE_RECEIVER,
    	"wsen:TimedOut",
    	"The enumerator has timed out and is no longer valid",
    	NULL,	/* no details here */
    },
    /* FAULT_TIMED_OUT_WSMAN */
    {
    	FAULT_URI_MANAGEMENT,
    	FAULT_CODE_RECEIVER,
    	"wsman:TimedOut",
    	"The operation has timed out",
    	NULL,	/* no details here */
    },
    /* FAULT_UNABLE_TO_RENEW */
    {
    	FAULT_URI_EVENTING,
    	FAULT_CODE_SENDER,
    	"wse:UnableToRenew",
    	"The subscription could not be renewed",
    	NULL,	/* details submitted as parameter */
    },
    /* FAULT_UNSUPPORTED_EXPIRATION_TYPE_WSE */
    {
    	FAULT_URI_EVENTING,
    	FAULT_CODE_SENDER,
    	"wse:UnsupportedExpirationType",
    	"The specified expiration type is not supported",
    	NULL,	/* no details here */
    },
    /* FAULT_UNSUPPORTED_EXPIRATION_TYPE_WSEN */
    {
    	FAULT_URI_ENUMERATION,
    	FAULT_CODE_SENDER,
    	"wsen:UnsupportedExpirationType",
    	"The specified expiration type is not supported",
    	NULL,	/* no details here */
    },
    /* FAULT_UNSUPPORTED_FEATURE */
    {
    	FAULT_URI_MANAGEMENT,
    	FAULT_CODE_SENDER,
    	"wsman:UnsupportedFeature",
    	"The specified feature is not supported",
    	fault_details_unsupported_feature,
    },
};

/******************************************************************************
* fault functions                                                             *
******************************************************************************/

/* generate a fault message */
void generate_fault_message(xml_writer_t response, char **fault_uri,
			    int fault_code, int fault_subcode, char *fault_detail) {

    xml_writer_t fault, code, reason;
    
    *fault_uri = faults[fault_code].uri;
    
    fault = print_xml_tag(response, "s", "Fault", NULL, NULL);
    
    code = print_xml_tag(fault, "s", "Code", NULL, NULL);
    print_xml_tag(code, "s", "Value", faults[fault_code].code, NULL);
    if (fault_code != FAULT_NOT_UNDERSTOOD) {
    	xml_writer_t subcode;
    	
    	subcode = print_xml_tag(code, "s", "Subcode", NULL, NULL);
    	print_xml_tag(subcode, "s", "Value", faults[fault_code].subcode, NULL);
    }
    
    reason = print_xml_tag(fault, "s", "Reason", NULL, NULL);
    print_xml_tag(reason, "s", "Text", faults[fault_code].reason, NULL);
    
    if (fault_detail || (faults[fault_code].details && faults[fault_code].details[fault_subcode])) {
    	xml_writer_t detail;
    	
    	detail = print_xml_tag(fault, "s", "Detail", NULL, NULL);
    	
    	if (fault_detail) {
    	    if (fault_code == FAULT_ACTION_NOT_SUPPORTED) {
    	    	print_xml_tag(detail, "wsa", "Action", fault_detail, NULL);
    	    } else if (fault_code == FAULT_CANNOT_PROCESS_FILTER ||
    	    	       fault_code == FAULT_DESTINATION_UNREACHABLE ||
    	    	       fault_code == FAULT_ENCODING_LIMIT ||
    	    	       fault_code == FAULT_INTERNAL_ERROR ||
    	    	       fault_code == FAULT_INVALID_MESSAGE ||
    	    	       fault_code == FAULT_INVALID_MESSAGE_INFORMATION_HEADER ||
    	    	       fault_code == FAULT_QUOTA_LIMIT ||
    	    	       fault_code == FAULT_SCHEMA_VALIDATION_ERROR ||
    	    	       fault_code == FAULT_UNABLE_TO_RENEW) {
    	    	print_xml_tag(detail, "s", "Text", fault_detail, NULL);
    	    } else {
    	    	print_xml_text(detail, fault_detail);
    	    }
    	}
    	if (faults[fault_code].details && faults[fault_code].details[fault_subcode]) {
    	    printf_xml_tag(detail, "wsman", "FaultDetail", "wsman:%s",
    	    		   faults[fault_code].details[fault_subcode]);
    	}
    }
}
