#include <malloc.h>

#include "debug.h"
#include "wsman.h"
#include "resource.h"
#include "standard_actions.h"
#include "res_catalog.h"
#include "xml_writer.h"
#include "fault.h"

static void wsman_add_resource_catalog(xml_writer_t xml, wsman_resource_t *res);

static wsman_instance_t * get_catalog_instance_by_request(wsman_resource_t *, wsman_request_t *, int *, const char *auth_string);
static wsman_instance_t * get_first_catalog_instance(wsman_resource_t *, wsman_request_t *, int *, const char *auth_string);
static wsman_instance_t * get_next_catalog_instance(wsman_resource_t *, wsman_request_t *, wsman_instance_t *, int *, const char *auth_string);
static int get_resource_catalog_instance(xml_writer_t, wsman_request_t *,
					 wsman_instance_t *, int *, int *, char **);
static int get_resource_catalog_fragment(xml_writer_t, wsman_request_t *,
					 wsman_instance_t *, int *, int *, char **);
static int get_resource_catalog_selector(xml_writer_t, wsman_request_t *,
					 wsman_instance_t *, int *, int *, char **);

/******************************************************************************
* type definitions                                                            *
******************************************************************************/

/* our local instances */
typedef struct {
    /* instance structure, must be the first element! */
    wsman_instance_t instance;
    
    /* local additions */
    wsman_resource_t * cat_resource;
} res_catalog_instance_t;

/******************************************************************************
* data                                                                        *
******************************************************************************/

static wsman_selector_t catalog_sel[] = {
    { "ResourceURI", NULL, NULL },
    { NULL, NULL, NULL }
};

static wsman_selector_set_t catalog_sel_set[] = {
    {
    	set_name:		"cat_sel",
    	selectors:		catalog_sel,
    },
    {
    	NULL, NULL
    },
};

static wsman_action_t catalog_act_0[] = {
    { ACTION_URI_GET, resource_get },
    { NULL, NULL }
};
static wsman_action_t catalog_act_1[] = {
    { ACTION_URI_ENUMERATE, resource_enumerate },
    { ACTION_URI_RELEASE, resource_release },
    { ACTION_URI_PULL, resource_pull },
    { NULL, NULL }
};
static char * catalog_sel_ref_0[] = { "cat_sel", NULL };

static wsman_operation_t catalog_op[] = {
    {
    	wsdl_port:		NULL,
    	wsdl_ref:		NULL,
    	actions:		catalog_act_0,
    	selector_set_ref:	catalog_sel_ref_0,
    	option_set_ref:		NULL,
    	schema_ref:		NULL,
    	filter_dialects:	NULL,
    	filters_required:	0,
    	delivery_modes:		NULL,
    },
    {
    	wsdl_port:		NULL,
    	wsdl_ref:		NULL,
    	actions:		catalog_act_1,
    	selector_set_ref:	NULL,
    	option_set_ref:		NULL,
    	schema_ref:		NULL,
    	filter_dialects:	NULL,
    	filters_required:	0,
    	delivery_modes:		NULL,
    },
    {
    	wsdl_port:		NULL,
    	wsdl_ref:		NULL,
    	actions:		NULL,
    	selector_set_ref:	NULL,
    	option_set_ref:		NULL,
    	schema_ref:		NULL,
    	filter_dialects:	NULL,
    	filters_required:	0,
    	delivery_modes:		NULL,
    }
};

wsman_resource_t catalog_resource = {
    resource_uri_namespace:		NAMESPACE_MANAGEMENT,
    resource_uri_ns_shortcut:		"wsman",
#if USE_FEBRUARY_2005_SPEC
    resource_uri:			"system/catalog/2005/02/Catalog",
#else /* USE_FEBRUARY_2005_SPEC */
    resource_uri:			"system/catalog/2005/06/Catalog",
#endif /* USE_FEBRUARY_2005_SPEC */
    ops:				catalog_op,
    sel_set:				catalog_sel_set,
    
    get_instance_by_request:		get_catalog_instance_by_request,
    get_first_instance:			get_first_catalog_instance,
    get_next_instance:			get_next_catalog_instance,
    get_instance_data:			get_resource_catalog_instance,
    get_instance_data_fragment:		get_resource_catalog_fragment,
    get_instance_selectors:		get_resource_catalog_selector,
};

static vector_t * catalog_instances = NULL;

/******************************************************************************
* functions                                                                   *
******************************************************************************/

static void delelem(void *elem) {
    free(elem);
}

static void add_catalog_resource(wsman_resource_t *resource) {
    res_catalog_instance_t *instance = malloc(sizeof(res_catalog_instance_t));
    
    memset(instance, 0, sizeof(res_catalog_instance_t));
    initialize_instance((wsman_instance_t *)instance, &catalog_resource);
    vector_add(catalog_instances, instance);
    instance->cat_resource = resource;
}

int res_catalog_init(void) {
    size_t i, size = vector_size(wsman_resources_in_catalog);
    
    catalog_instances = vector_new(NULL, 0, delelem);
    
    for (i = 0; i < size; i++) {
    	add_catalog_resource((wsman_resource_t *)vector_get(wsman_resources_in_catalog, i));
    }
    
    return add_resource(&catalog_resource, 0);
}

void res_catalog_cleanup(void) {
    if (catalog_instances) {
    	vector_delete(catalog_instances);
    }
}

static void add_value_if_not_nil(xml_writer_t xml, char *name, char *val, int text_only, int namespace) {
    if (text_only) {
    	print_xml_text(xml, val ? val : "xsi:nil");
    } else if (val) {
    	print_xml_tag(xml, namespace ? "cat" : "", name, val, NULL);
    } else {
    	print_xml_tag(xml, namespace ? "cat" : "", name, NULL, "xsi", "nil", "", "true");
    }
}

static inline void add_resource_uri(xml_writer_t xml, wsman_resource_t *res, int text_only, int namespace) {
    char *txt = NULL;
    if (res->resource_uri) {
    	size_t len = strlen(res->resource_uri) + 1;
    	if (res->resource_uri_namespace) {
    	    len += strlen(res->resource_uri_namespace) + 1;
    	}
    	txt = malloc(len);
    	if (res->resource_uri_namespace) {
    	    snprintf(txt, len, "%s/%s", res->resource_uri_namespace, res->resource_uri);
    	} else {
    	    snprintf(txt, len, "%s", res->resource_uri);
    	}
    }
    add_value_if_not_nil(xml, "ResourceURI", txt, text_only, namespace);
    if (txt) free(txt);
}

static inline void add_notes(xml_writer_t xml, wsman_resource_t *res, int text_only, int namespace) {
    add_value_if_not_nil(xml, "Notes", res->notes, text_only, namespace);
}

static inline void add_vendor(xml_writer_t xml, wsman_resource_t *res, int text_only, int namespace) {
    add_value_if_not_nil(xml, "Vendor", res->vendor, text_only, namespace);
}

static inline void add_dispname(xml_writer_t xml, wsman_resource_t *res, int text_only, int namespace) {
    add_value_if_not_nil(xml, "DisplayName", res->display_name, text_only, namespace);
}

static void add_cat_keyword(xml_writer_t xml, wsman_resource_t *res, int idx, int text_only, int namespace) {
    int i;
    int idx2 = 0;
    for (i = 0; res->keywords && res->keywords[i].binding_prefix && res->keywords[i].words; i++) {
    	wsman_keyword_t * kw = &res->keywords[i];
    	int j;
    	for (j = 0; kw->words[j]; j++) {
    	    if (idx == idx2) {
    	    	if (text_only) {
    	    	    printf_xml_text(xml, "%s:%s", kw->binding_prefix, kw->words[j]);
    	    	} else {
    	    	    printf_xml_tag(xml, namespace ? "cat" : "", "Keyword", "%s:%s", kw->binding_prefix, kw->words[j]);
    	    	}
    	    	return;
    	    }
    	    idx2++;
    	}
    }
    add_value_if_not_nil(xml, "Keyword", NULL, text_only, namespace);
}

static void add_cat_keyword_all(xml_writer_t xml, wsman_resource_t *res, int text_only, int namespace) {
    int i, idx = 0;
    for (i = 0; res->keywords && res->keywords[i].binding_prefix && res->keywords[i].words; i++) {
    	wsman_keyword_t * kw = &res->keywords[i];
    	int j;
    	for (j = 0; kw->words[j]; j++) {
    	    add_cat_keyword(xml, res, idx++, text_only, namespace);
    	}
    }
    if (!idx) {
    	add_value_if_not_nil(xml, "Keyword", NULL, text_only, namespace);
    }
}

static void add_cat_keywords(xml_writer_t xml, wsman_resource_t *res, int text_only, int namespace) {
    if (res->keywords) {
    	int i;
    	xml_writer_t keywords;
    	
    	if (!text_only) {
    	    const char **attr = NULL;
    	    int no_attr = 0;
    	    
    	    for (i = 0; res->keywords[i].binding_prefix; i++) {
    	    	wsman_keyword_t * kw = &res->keywords[i];
    	    	int new_no = no_attr + 4;
    	    	if (attr) {
    	    	    attr = realloc(attr, new_no * sizeof(char *));
    	    	} else {
    	    	    attr = malloc(new_no * sizeof(char *));
    	    	}
    	    	attr[no_attr] = "xmlns";
    	    	attr[no_attr + 1] = kw->binding_prefix;
    	    	attr[no_attr + 2] = "";
    	    	attr[no_attr + 3] = kw->binding;
    	    	no_attr = new_no;
    	    }
    	    keywords = print_xml_full_tag(xml, namespace ? "cat" : "", "Keywords", NULL, attr, no_attr / 4);

    	    if (attr) {
    	    	free(attr);
    	    }
    	} else {
    	    keywords = xml;
    	}
    	
    	add_cat_keyword_all(keywords, res, 0, namespace);
    } else {
    	add_value_if_not_nil(xml, "Keywords", NULL, text_only, namespace);
    }
}

static void add_cat_relship(xml_writer_t xml, wsman_resource_t *res, int idx, int text_only, int namespace) {
    int i;
    for (i = 0; res->relships && (res->relships[i].type || res->relships[i].role || res->relships[i].ref); i++) {
    	wsman_relship_t * rel = &res->relships[i];
    	
    	if (i == idx) {
    	    if (!text_only) {
    	    	print_xml_tag(xml, namespace ? "cat" : "", "Relationship", NULL,
    	    	    rel->type ? "" : NULL, rel->type ? "Type" : NULL,
    	    	    rel->type ? "" : NULL, rel->type ? rel->type : NULL,
    	    	    rel->role ? "" : NULL, rel->role ? "Role" : NULL,
    	    	    rel->role ? "" : NULL, rel->role ? rel->role : NULL,
    	    	    rel->ref ? "" : NULL, rel->ref ? "Ref" : NULL,
    	    	    rel->ref ? "" : NULL, rel->ref ? rel->ref : NULL);
    	    }
    	    return;
    	}
    }
    add_value_if_not_nil(xml, "Relationship", NULL, text_only, namespace);
}

static void add_cat_relship_all(xml_writer_t xml, wsman_resource_t *res, int text_only, int namespace) {
    int i, idx = 0;
    for (i = 0; res->relships && (res->relships[i].type || res->relships[i].role || res->relships[i].ref); i++) {
    	add_cat_relship(xml, res, idx++, text_only, namespace);
    }
    if (!idx) {
    	add_value_if_not_nil(xml, "Relationship", NULL, text_only, namespace);
    }
}

static void add_cat_relships(xml_writer_t xml, wsman_resource_t *res, int text_only, int namespace) {
    if (res->relships) {
    	xml_writer_t relships;
    	
    	if (text_only) {
    	    relships = xml;
    	} else {
    	    relships = print_xml_tag(xml, namespace ? "cat" : "", "Relationships", NULL, NULL);
    	}
    	add_cat_relship_all(relships, res, 0, namespace);
    } else {
    	add_value_if_not_nil(xml, "Relationship", NULL, text_only, namespace);
    }
}

static inline void add_cat_compliance(xml_writer_t xml, wsman_resource_t *res UNUSED, int idx, int text_only, int namespace) {
    add_value_if_not_nil(xml, "Compliance", idx > 1 ? NULL : namespace_management, text_only, namespace);
}

static wsman_selector_set_t *get_selset_at_index(wsman_resource_t *res, int idx) {
    int i;
    for (i = 0; res->sel_set && res->sel_set[i].set_name && res->sel_set[i].selectors; i++) {
    	if (i == idx) return &res->sel_set[i];
    }
    return NULL;
}

static void add_cat_selector(xml_writer_t xml, wsman_resource_t *res, int set_index, int sel_index, int text_only, int namespace) {
    wsman_selector_set_t *sel = get_selset_at_index(res, set_index);
    if (sel && sel->selectors) {
    	int i;
    	
    	for (i = 0; sel->selectors[i].sel_name; i++) {
    	    if (i == sel_index) {
    	    	wsman_selector_t *s = &sel->selectors[i];
    	    	if (text_only) {
    	    	    print_xml_text(xml, s->description);
    	    	} else {
    	    	    print_xml_tag(xml, namespace ? "cat" : "",
    	    	    	"Selector",
    	    	    	s->description,
    	    	    	"", "Name", "", s->sel_name,
    	    	    	s->type ? "" : NULL,
    	    	    	s->type ? "Type" : NULL,
    	    	    	s->type ? "" : NULL,
    	    	    	s->type ? s->type : NULL);
    	    	}
    	    	return;
    	    }
    	}
    }
    add_value_if_not_nil(xml, "Selector", NULL, text_only, namespace);
}

static void add_cat_selector_all(xml_writer_t xml, wsman_resource_t *res, int set_index, int text_only, int namespace) {
    wsman_selector_set_t *sel = get_selset_at_index(res, set_index);
    if (sel && sel->selectors) {
    	int i;
    	
    	for (i = 0; sel->selectors[i].sel_name; i++) {
    	    add_cat_selector(xml, res, set_index, i, 0, namespace);
    	}
    } else {
    	add_value_if_not_nil(xml, "Selector", NULL, text_only, namespace);
    }
}

static void add_cat_selset(xml_writer_t xml, wsman_resource_t *res, int idx, int text_only, int namespace) {
    wsman_selector_set_t *sel = get_selset_at_index(res, idx);
    xml_writer_t selset;
    
    if (sel) {
    	if (!text_only) {
    	    selset = print_xml_tag(xml, namespace ? "cat" : "",
    	    	"SelectorSet", NULL, "", "Name", "", sel->set_name);
    	} else {
    	    selset = xml;
    	}
    	add_cat_selector_all(selset, res, idx, 0, namespace);
    	
    	return;
    }
    add_value_if_not_nil(xml, "SelectorSet", NULL, text_only, namespace);
}

static void add_cat_selset_all(xml_writer_t xml, wsman_resource_t *res, int text_only, int namespace) {
    int i, idx = 0;
    for (i = 0; res->sel_set && (res->sel_set[i].set_name && res->sel_set[i].selectors); i++) {
    	add_cat_selset(xml, res, idx++, text_only, namespace);
    }
    if (!idx) {
    	add_value_if_not_nil(xml, "SelectorSet", NULL, text_only, namespace);
    }
}

static wsman_option_set_t *get_optset_at_index(wsman_resource_t *res, int idx) {
    int i;
    for (i = 0; res->opt_set && res->opt_set[i].set_name && res->opt_set[i].options; i++) {
    	if (i == idx) return &res->opt_set[i];
    }
    return NULL;
}

static void add_cat_option(xml_writer_t xml, wsman_resource_t *res, int set_index, int opt_index, int text_only, int namespace) {
    wsman_option_set_t *opt = get_optset_at_index(res, set_index);
    if (opt && opt->options) {
    	int i;
    	
    	for (i = 0; opt->options[i].opt_name; i++) {
    	    if (i == opt_index) {
    	    	wsman_option_t *s = &opt->options[i];
    	    	if (text_only) {
    	    	    print_xml_text(xml, s->description);
    	    	} else {
    	    	    print_xml_tag(xml, namespace ? "cat" : "",
    	    	    	"Option",
    	    	    	s->description,
    	    	    	"", "Name", "", s->opt_name,
    	    	    	s->type ? "" : NULL,
    	    	    	s->type ? "Type" : NULL,
    	    	    	s->type ? "" : NULL,
    	    	    	s->type ? s->type : NULL);
    	    	}
    	    	return;
    	    }
    	}
    }
    add_value_if_not_nil(xml, "Option", NULL, text_only, namespace);
}

static void add_cat_option_all(xml_writer_t xml, wsman_resource_t *res, int set_index, int text_only, int namespace) {
    wsman_option_set_t *opt = get_optset_at_index(res, set_index);
    if (opt && opt->options) {
    	int i;
    	
    	for (i = 0; opt->options[i].opt_name; i++) {
    	    add_cat_option(xml, res, set_index, i, 0, namespace);
    	}
    } else {
    	add_value_if_not_nil(xml, "Option", NULL, text_only, namespace);
    }
}

static void add_cat_optset(xml_writer_t xml, wsman_resource_t *res, int idx, int text_only, int namespace) {
    wsman_option_set_t *opt = get_optset_at_index(res, idx);
    xml_writer_t optset;
    
    if (opt) {
    	if (!text_only) {
    	    optset = print_xml_tag(xml, namespace ? "cat" : "",
    	    	"OptionSet", NULL, "", "Name", "", opt->set_name);
    	} else {
    	    optset = xml;
    	}
    	add_cat_option_all(optset, res, idx, 0, namespace);
    	
    	return;
    }
    add_value_if_not_nil(xml, "OptionSet", NULL, text_only, namespace);
}

static void add_cat_optset_all(xml_writer_t xml, wsman_resource_t *res, int text_only, int namespace) {
    int i, idx = 0;
    for (i = 0; res->opt_set && (res->opt_set[i].set_name && res->opt_set[i].options); i++) {
    	add_cat_optset(xml, res, idx++, text_only, namespace);
    }
    if (!idx) {
    	add_value_if_not_nil(xml, "OptionSet", NULL, text_only, namespace);
    }
}

static wsman_operation_t* get_op_at_index(wsman_resource_t *res, int idx) {
    int i;
    for (i = 0; res->ops && res->ops[i].actions; i++) {
    	if (i == idx) return &res->ops[i];
    }
    return NULL;
}

static void add_cat_action(xml_writer_t xml, wsman_resource_t *res, int op_index, int act_index, int text_only, int namespace) {
    wsman_operation_t *op = get_op_at_index(res, op_index);
    if (op && op->actions) {
    	int i;
    	for (i = 0; op->actions[i].action; i++) {
    	    if (i == act_index) {
    	    	add_value_if_not_nil(xml, "Action", op->actions[i].action, text_only, namespace);
    	    	return;
    	    }
    	}
    }
    add_value_if_not_nil(xml, "Action", NULL, text_only, namespace);
}

static void add_cat_action_all(xml_writer_t xml, wsman_resource_t *res, int op_index, int text_only, int namespace) {
    wsman_operation_t *op = get_op_at_index(res, op_index);
    if (op && op->actions) {
    	int i;
    	
    	for (i = 0; op->actions[i].action; i++) {
    	    add_cat_action(xml, res, op_index, i, 0, namespace);
    	}
    } else {
    	add_value_if_not_nil(xml, "Action", NULL, text_only, namespace);
    }
}

static void add_cat_selsetref(xml_writer_t xml, wsman_resource_t *res, int op_index, int act_index, int text_only, int namespace) {
    wsman_operation_t *op = get_op_at_index(res, op_index);
    if (op && op->selector_set_ref) {
    	int i;
    	for (i = 0; op->selector_set_ref[i]; i++) {
    	    if (i == act_index) {
    	    	if (!text_only) {
    	    	    print_xml_tag(xml, namespace ? "cat" : "", "SelectorSetRef",
    	    	    	NULL, "", "Name", "", op->selector_set_ref[i]);
    	    	}
    	    	return;
    	    }
    	}
    }
    add_value_if_not_nil(xml, "SelectorSetRef", NULL, text_only, namespace);
}

static void add_cat_selsetref_all(xml_writer_t xml, wsman_resource_t *res, int op_index, int text_only, int namespace) {
    wsman_operation_t *op = get_op_at_index(res, op_index);
    if (op && op->selector_set_ref) {
    	int i;
    	
    	for (i = 0; op->selector_set_ref[i]; i++) {
    	    add_cat_selsetref(xml, res, op_index, i, 0, namespace);
    	}
    } else {
    	add_value_if_not_nil(xml, "SelectorSetRef", NULL, text_only, namespace);
    }
}

static void add_cat_optsetref(xml_writer_t xml, wsman_resource_t *res, int op_index, int act_index, int text_only, int namespace) {
    wsman_operation_t *op = get_op_at_index(res, op_index);
    if (op && op->option_set_ref) {
    	int i;
    	for (i = 0; op->option_set_ref[i]; i++) {
    	    if (i == act_index) {
    	    	if (!text_only) {
    	    	    print_xml_tag(xml, namespace ? "cat" : "", "OptionSetRef",
    	    	    	NULL, "", "Name", "", op->option_set_ref[i]);
    	    	}
    	    	return;
    	    }
    	}
    }
    add_value_if_not_nil(xml, "OptionSetRef", NULL, text_only, namespace);
}

static void add_cat_optsetref_all(xml_writer_t xml, wsman_resource_t *res, int op_index, int text_only, int namespace) {
    wsman_operation_t *op = get_op_at_index(res, op_index);
    if (op && op->option_set_ref) {
    	int i;
    	
    	for (i = 0; op->option_set_ref[i]; i++) {
    	    add_cat_optsetref(xml, res, op_index, i, 0, namespace);
    	}
    } else {
    	add_value_if_not_nil(xml, "OptionSetRef", NULL, text_only, namespace);
    }
}

static void add_cat_schemaref(xml_writer_t xml, wsman_resource_t *res, int op_index, int act_index, int text_only, int namespace) {
    wsman_operation_t *op = get_op_at_index(res, op_index);
    if (op && op->schema_ref) {
    	int i;
    	for (i = 0; op->schema_ref[i].uri_prefix[i]; i++) {
    	    if (i == act_index) {
    	    	xml_writer_t p;
    	    	wsman_schema_ref_t * ref = &op->schema_ref[i];
    	    	
    	    	if (text_only) {
    	    	    p = xml;
    	    	} else {
		    p = print_xml_tag(xml, namespace ? "cat" : "", "SchemaRef", NULL,
		    	ref->location ? "" : NULL, ref->location ? "Location" : NULL,
		    	ref->location ? "" : NULL, ref->location ? ref->location : NULL,
		    	ref->uri ? "xmlns" : NULL, ref->uri ? ref->uri_prefix : NULL,
		    	ref->uri ? "" : NULL, ref->uri ? ref->uri : NULL);
    	    	}

    	    	if (ref->schema) {
    	    	    printf_xml_text(p, "%s:%s", ref->uri_prefix, ref->schema);
    	    	}

    	    	return;
    	    }
    	}
    }
    add_value_if_not_nil(xml, "SchemaRef", NULL, text_only, namespace);
}

static void add_cat_schemaref_all(xml_writer_t xml, wsman_resource_t *res, int op_index, int text_only, int namespace) {
    wsman_operation_t *op = get_op_at_index(res, op_index);
    if (op && op->schema_ref) {
    	int i;
    	
    	for (i = 0; op->schema_ref[i].uri_prefix; i++) {
    	    add_cat_schemaref(xml, res, op_index, i, 0, namespace);
    	}
    } else {
    	add_value_if_not_nil(xml, "SchemaRef", NULL, text_only, namespace);
    }
}

static void add_cat_filterdialect(xml_writer_t xml, wsman_resource_t *res, int op_index, int act_index, int text_only, int namespace) {
    wsman_operation_t *op = get_op_at_index(res, op_index);
    if (op && op->filter_dialects) {
    	int i;
    	for (i = 0; op->filter_dialects[i]; i++) {
    	    if (i == act_index) {
    	    	add_value_if_not_nil(xml, "FilterDialect", op->filter_dialects[i], text_only, namespace);
    	    	return;
    	    }
    	}
    }
    add_value_if_not_nil(xml, "FilterDialect", NULL, text_only, namespace);
}

static void add_cat_filterdialect_all(xml_writer_t xml, wsman_resource_t *res, int op_index, int text_only, int namespace) {
    wsman_operation_t *op = get_op_at_index(res, op_index);
    if (op && op->filter_dialects) {
    	int i;
    	
    	for (i = 0; op->filter_dialects[i]; i++) {
    	    add_cat_filterdialect(xml, res, op_index, i, 0, namespace);
    	}
    } else {
    	add_value_if_not_nil(xml, "FilterDialect", NULL, text_only, namespace);
    }
}

static void add_cat_delmode(xml_writer_t xml, wsman_resource_t *res, int op_index, int act_index, int text_only, int namespace) {
    wsman_operation_t *op = get_op_at_index(res, op_index);
    if (op && op->delivery_modes) {
    	int i;
    	for (i = 0; op->delivery_modes[i]; i++) {
    	    if (i == act_index) {
    	    	add_value_if_not_nil(xml, "DeliveryMode", op->delivery_modes[i], text_only, namespace);
    	    	return;
    	    }
    	}
    }
    add_value_if_not_nil(xml, "DeliveryMode", NULL, text_only, namespace);
}

static void add_cat_delmode_all(xml_writer_t xml, wsman_resource_t *res, int op_index, int text_only, int namespace) {
    wsman_operation_t *op = get_op_at_index(res, op_index);
    if (op && op->delivery_modes) {
    	int i;
    	
    	for (i = 0; op->delivery_modes[i]; i++) {
    	    add_cat_delmode(xml, res, op_index, i, 0, namespace);
    	}
    } else {
    	add_value_if_not_nil(xml, "DeliveryMode", NULL, text_only, namespace);
    }
}

static void add_cat_filters_required(xml_writer_t xml, wsman_resource_t *res, int op_index, int text_only, int namespace) {
    wsman_operation_t *op = get_op_at_index(res, op_index);
    add_value_if_not_nil(xml, "FiltersRequired", 
    	op && op->filters_required ? "" : NULL, text_only, namespace);
}

static void add_cat_operation(xml_writer_t xml, wsman_resource_t *res, int idx, int text_only, int namespace) {
    wsman_operation_t *op = get_op_at_index(res, idx);
    
    if (op) {
    	xml_writer_t operation;
    	
    	// add operation header
    	if (text_only) {
    	    operation = xml;
    	} else {
    	    operation = print_xml_tag(xml, namespace ? "cat" : "", "Operation", NULL,
    	    	op->wsdl_port ? "" : NULL,
    	    	op->wsdl_port ? "WsdlPort" : NULL,
    	    	op->wsdl_port ? "" : NULL,
    	    	op->wsdl_port ? op->wsdl_port : NULL,
    	    	op->wsdl_ref ? "" : NULL,
    	    	op->wsdl_ref ? "WsdlRef" : NULL,
    	    	op->wsdl_ref ? "" : NULL,
    	    	op->wsdl_ref ? op->wsdl_ref : NULL);
    	}
    	
    	// add actions
    	if (op->actions && op->actions[0].action) {
    	    add_cat_action_all(operation, res, idx, 0, namespace);
    	}
    	
    	// add selector references
    	if (op->selector_set_ref && op->selector_set_ref[0]) {
    	    add_cat_selsetref_all(operation, res, idx, 0, namespace);
    	}
    	    	
    	// add option references
    	if (op->option_set_ref && op->option_set_ref[0]) {
    	    add_cat_optsetref_all(operation, res, idx, 0, namespace);
    	}
    	
    	// add schema references
    	if (op->schema_ref && op->schema_ref[0].uri_prefix) {
    	    add_cat_schemaref_all(operation, res, idx, 0, namespace);
    	}
    	
    	// add filter dialects
    	if (op->filter_dialects && op->filter_dialects[0]) {
    	    add_cat_filterdialect_all(operation, res, idx, 0, namespace);
    	}
    	if (op->filters_required) {
    	    print_xml_tag(operation, namespace ? "cat" : "", "FiltersRequired", NULL, NULL);
    	}
    	
    	// add delivery modes
    	if (op->delivery_modes && op->delivery_modes[0]) {
    	    add_cat_delmode_all(operation, res, idx, 0, namespace);
    	}
    	
    } else {
    	add_value_if_not_nil(xml, "Operation", NULL, text_only, namespace);
    }
}

static void add_cat_operation_all(xml_writer_t xml, wsman_resource_t *res, int text_only, int namespace) {
    int i, idx = 0;
    for (i = 0; res->ops && res->ops[i].actions; i++) {
    	add_cat_operation(xml, res, idx++, text_only, namespace);
    }
    if (!idx) {
    	add_value_if_not_nil(xml, "Operation", NULL, text_only, namespace);
    }
}

static void add_cat_access(xml_writer_t xml, wsman_resource_t *res, int text_only, int namespace) {
    if (res->ops || res->sel_set || res->opt_set) {
    	xml_writer_t access_node;
    	
    	access_node = print_xml_tag(xml, namespace ? "cat" : "", "Access", NULL, NULL);
    	
    	add_cat_compliance(access_node, res, 0, 0, namespace);
    	
    	// add the operations
    	if (res->ops) {
    	    add_cat_operation_all(access_node, res, 0, namespace);
    	}

    	// add the selector set
    	if (res->sel_set) {
    	    add_cat_selset_all(access_node, res, 0, namespace);
    	}
    	
    	// add the option set
    	if (res->opt_set) {
    	    add_cat_optset_all(access_node, res, 0, namespace);
    	}
    } else {
    	add_value_if_not_nil(xml, "Access", NULL, text_only, namespace);
    }
}

static void wsman_add_resource_catalog(xml_writer_t xml, wsman_resource_t *res) {
    xml_writer_t resource;
    
    // add Resource tag
    resource = print_xml_tag(xml, "cat", "Resource", NULL,
    	"xmlns", "cat", "", NAMESPACE_CATALOG,
    	"", "lang", "", "en-us");
    
    // add general part
    if (res->resource_uri) {
    	add_resource_uri(resource, res, 0, 1);
    }
    if (res->notes) {
    	add_notes(resource, res, 0, 1);
    }
    if (res->vendor) {
    	add_vendor(resource, res, 0, 1);
    }
    if (res->display_name) {
    	add_dispname(resource, res, 0, 1);
    }
    
    // add the keywords (still general part)
    if (res->keywords) {
    	add_cat_keywords(resource, res, 0, 1);
    }
    
    // add the access session
    if (res->ops || res->sel_set || res->opt_set) {
    	add_cat_access(resource, res, 0, 1);
    }
    
    // add the relationships section
    if (res->relships) {
    	add_cat_relships(resource, res, 0, 1);
    }
}

#define RETURN_ERROR(code, detail, det_str)	\
    do {					\
    	*fault_code = code;			\
    	*fault_detail = detail;			\
    	*fault_detail_string = det_str;		\
    	ret = -1;				\
    	goto bail;				\
    } while(0)

#define RET_INVALID_FRAGMENT				\
    do {						\
    	D(D_ERROR, "Fragment %s not found.\n",		\
    	    req->fragment.fragment_full);		\
    	RETURN_ERROR(FAULT_INVALID_REPRESENTATION,	\
    	FAULT_DETAIL_REPRESENTATION_INVALID_FRAGMENT,	\
    	0);						\
    } while(0)

static int add_resource_catalog_fragment(xml_writer_t xml, wsman_request_t *req UNUSED,
					 wsman_instance_t *instance, int *fault_code,
					 int *fault_detail, char **fault_detail_string) {
    int ret = 0;
    wsman_resource_t *res = ((res_catalog_instance_t *)instance)->cat_resource;
    wsman_fragment_t *fragment;
    xml_writer_t frag;
    size_t size = vector_size(req->fragment.fragments);
    
    if (size < 2 ||
    	(fragment = (wsman_fragment_t *)vector_get(req->fragment.fragments, 0))->array_index ||
    	strcasecmp(fragment->fragment, "Resource")) {
    	
    	RET_INVALID_FRAGMENT;
    }
    
    fragment = (wsman_fragment_t *)vector_get(req->fragment.fragments, 1);
    frag = print_xml_tag(xml, "wsman", "XmlFragment", NULL, NULL);
    
    if (size == 2 && !strcasecmp(fragment->fragment, "ResourceURI")) {
    	add_resource_uri(frag, res, req->fragment.text_only, 0);
    } else if (size == 2 && !strcasecmp(fragment->fragment, "Notes")) {
    	add_notes(frag, res, req->fragment.text_only, 0);
    } else if (size == 2 && !strcasecmp(fragment->fragment, "Vendor")) {
    	add_vendor(frag, res, req->fragment.text_only, 0);
    } else if (size == 2 && !strcasecmp(fragment->fragment, "DisplayName")) {
    	add_dispname(frag, res, req->fragment.text_only, 0);
    } else if (!strcasecmp(fragment->fragment, "Keywords")) {
    	if (size == 2 && !fragment->array_index) {
    	    add_cat_keywords(frag, res, req->fragment.text_only, 0);
    	} else if (size >= 3) {
    	    fragment = (wsman_fragment_t *)vector_get(req->fragment.fragments, 2);
    	    if (size == 3 && !strcasecmp(fragment->fragment, "Keyword")) {
    	    	if (fragment->array_index) {
    	    	    add_cat_keyword(frag, res, fragment->array_index - 1, req->fragment.text_only, 0);
    	    	} else if (!req->fragment.text_only) {
    	    	    add_cat_keyword_all(frag, res, req->fragment.text_only, 0);
    	    	} else {
    	    	    RET_INVALID_FRAGMENT;
    	    	}
    	    } else {
    	    	RET_INVALID_FRAGMENT;
    	    }
    	} else {
    	    RET_INVALID_FRAGMENT;
    	}
    } else if (!strcasecmp(fragment->fragment, "Access")) {
    	if (size == 2 && !fragment->array_index) {
    	    add_cat_access(frag, res, req->fragment.text_only, 0);
    	} else if (size >= 3) {
    	    fragment = (wsman_fragment_t *)vector_get(req->fragment.fragments, 2);
    	    if (size == 3 && !strcasecmp(fragment->fragment, "Compliance")) {
    	    	add_cat_compliance(frag, res, fragment->array_index, req->fragment.text_only, 0);
    	    } else if (!strcasecmp(fragment->fragment, "SelectorSet")) {
    	    	if (size == 3) {
    	    	    if (fragment->array_index) {
    	    	    	add_cat_selset(frag, res, fragment->array_index - 1, req->fragment.text_only, 0);
    	    	    } else if (!req->fragment.text_only) {
    	    	    	add_cat_selset_all(frag, res, req->fragment.text_only, 0);
    	    	    } else {
    	    	    	RET_INVALID_FRAGMENT;
    	    	    }
    	    	} else if (size >= 4) {
    	    	    int set_index = fragment->array_index;
    	    	    fragment = (wsman_fragment_t *)vector_get(req->fragment.fragments, 3);
    	    	    if (size == 4 && set_index && !strcasecmp(fragment->fragment, "Selector")) {
    	    	    	if (fragment->array_index) {
    	    	    	    add_cat_selector(frag, res, set_index - 1, fragment->array_index - 1, req->fragment.text_only, 0);
    	    	    	} else if (!req->fragment.text_only) {
    	    	    	    add_cat_selector_all(frag, res, set_index - 1, req->fragment.text_only, 0);
    	    	    	} else {
    	    	    	    RET_INVALID_FRAGMENT;
    	    	    	}
    	    	    } else {
    	    	    	RET_INVALID_FRAGMENT;
    	    	    }
    	    	} else {
    	    	    RET_INVALID_FRAGMENT;
    	    	}
    	    } else if (!strcasecmp(fragment->fragment, "OptionSet")) {
    	    	if (size == 3) {
    	    	    if (fragment->array_index) {
    	    	    	add_cat_optset(frag, res, fragment->array_index - 1, req->fragment.text_only, 0);
    	    	    } else if (!req->fragment.text_only) {
    	    	    	add_cat_optset_all(frag, res, req->fragment.text_only, 0);
    	    	    } else {
    	    	    	RET_INVALID_FRAGMENT;
    	    	    }
    	    	} else if (size >= 4) {
    	    	    int set_index = fragment->array_index;
    	    	    fragment = (wsman_fragment_t *)vector_get(req->fragment.fragments, 3);
    	    	    if (size == 4 && set_index && !strcasecmp(fragment->fragment, "Option")) {
    	    	    	if (fragment->array_index) {
    	    	    	    add_cat_option(frag, res, set_index - 1, fragment->array_index - 1, req->fragment.text_only, 0);
    	    	    	} else if (!req->fragment.text_only) {
    	    	    	    add_cat_option_all(frag, res, set_index - 1, req->fragment.text_only, 0);
    	    	    	} else {
    	    	    	    RET_INVALID_FRAGMENT;
    	    	    	}
    	    	    } else {
    	    	    	RET_INVALID_FRAGMENT;
    	    	    }
    	    	} else {
    	    	    RET_INVALID_FRAGMENT;
    	    	}
    	    } else if (!strcasecmp(fragment->fragment, "Operation")) {
    	    	if (size == 3) {
    	    	    if (fragment->array_index) {
    	    	    	add_cat_operation(frag, res, fragment->array_index - 1, req->fragment.text_only, 0);
    	    	    } else if (!req->fragment.text_only) {
    	    	    	add_cat_operation_all(frag, res, req->fragment.text_only, 0);
    	    	    } else {
    	    	    	RET_INVALID_FRAGMENT;
    	    	    }
    	    	} else if (size >= 4) {
    	    	    int op_index = fragment->array_index;
    	    	    fragment = (wsman_fragment_t *)vector_get(req->fragment.fragments, 3);
    	    	    if (size == 4 && op_index && !strcasecmp(fragment->fragment, "Action")) {
    	    	    	if (fragment->array_index) {
    	    	    	    add_cat_action(frag, res, op_index - 1, fragment->array_index - 1, req->fragment.text_only, 0);
    	    	    	} else if (!req->fragment.text_only) {
    	    	    	    add_cat_action_all(frag, res, op_index - 1, req->fragment.text_only, 0);
    	    	    	} else {
    	    	    	    RET_INVALID_FRAGMENT;
    	    	    	}
    	    	    } else if (size == 4 && op_index && !strcasecmp(fragment->fragment, "SelectorSetRef")) {
    	    	    	if (fragment->array_index) {
    	    	    	    add_cat_selsetref(frag, res, op_index - 1, fragment->array_index - 1, req->fragment.text_only, 0);
    	    	    	} else if (!req->fragment.text_only) {
    	    	    	    add_cat_selsetref_all(frag, res, op_index - 1, req->fragment.text_only, 0);
    	    	    	} else {
    	    	    	    RET_INVALID_FRAGMENT;
    	    	    	}
    	    	    } else if (size == 4 && op_index && !strcasecmp(fragment->fragment, "OptionSetRef")) {
    	    	    	if (fragment->array_index) {
    	    	    	    add_cat_optsetref(frag, res, op_index - 1, fragment->array_index - 1, req->fragment.text_only, 0);
    	    	    	} else if (!req->fragment.text_only) {
    	    	    	    add_cat_optsetref_all(frag, res, op_index - 1, req->fragment.text_only, 0);
    	    	    	} else {
    	    	    	    RET_INVALID_FRAGMENT;
    	    	    	}
    	    	    } else if (size == 4 && op_index && !strcasecmp(fragment->fragment, "SchemaRef")) {
    	    	    	if (fragment->array_index) {
    	    	    	    add_cat_schemaref(frag, res, op_index - 1, fragment->array_index - 1, req->fragment.text_only, 0);
    	    	    	} else if (!req->fragment.text_only) {
    	    	    	    add_cat_schemaref_all(frag, res, op_index - 1, req->fragment.text_only, 0);
    	    	    	} else {
    	    	    	    RET_INVALID_FRAGMENT;
    	    	    	}
    	    	    } else if (size == 4 && op_index && !strcasecmp(fragment->fragment, "FilterDialect")) {
    	    	    	if (fragment->array_index) {
    	    	    	    add_cat_filterdialect(frag, res, op_index - 1, fragment->array_index - 1, req->fragment.text_only, 0);
    	    	    	} else if (!req->fragment.text_only) {
    	    	    	    add_cat_filterdialect_all(frag, res, op_index - 1, req->fragment.text_only, 0);
    	    	    	} else {
    	    	    	    RET_INVALID_FRAGMENT;
    	    	    	}
    	    	    } else if (size == 4 && op_index && !strcasecmp(fragment->fragment, "DeliveryMode")) {
    	    	    	if (fragment->array_index) {
    	    	    	    add_cat_delmode(frag, res, op_index - 1, fragment->array_index - 1, req->fragment.text_only, 0);
    	    	    	} else if (!req->fragment.text_only) {
    	    	    	    add_cat_delmode_all(frag, res, op_index - 1, req->fragment.text_only, 0);
    	    	    	} else {
    	    	    	    RET_INVALID_FRAGMENT;
    	    	    	}
    	    	    } else if (size == 4 && op_index && !fragment->array_index && !strcasecmp(fragment->fragment, "FiltersRequired")) {
    	    	    	add_cat_filters_required(frag, res, op_index - 1, req->fragment.text_only, 0);
    	    	    } else {
    	    	    	RET_INVALID_FRAGMENT;
    	    	    }
    	    	} else {
    	    	    RET_INVALID_FRAGMENT;
    	    	}
    	    } else {
    	    	RET_INVALID_FRAGMENT;
    	    }
    	} else {
    	    RET_INVALID_FRAGMENT;
    	}
    } else if (!strcasecmp(fragment->fragment, "Relationships")) {
    	if (size == 2 && !fragment->array_index) {
    	    add_cat_relships(frag, res, req->fragment.text_only, 0);
    	} else if (size >= 3) {
    	    fragment = (wsman_fragment_t *)vector_get(req->fragment.fragments, 2);;
    	    if (size == 3 && !strcasecmp(fragment->fragment, "Relationship")) {
    	    	if (fragment->array_index) {
    	    	    add_cat_relship(frag, res, fragment->array_index - 1, req->fragment.text_only, 0);
    	    	} else if (!req->fragment.text_only) {
    	    	    add_cat_relship_all(frag, res, req->fragment.text_only, 0);
    	    	} else {
    	    	    RET_INVALID_FRAGMENT;
    	    	}
    	    } else {
    	    	RET_INVALID_FRAGMENT;
    	    }
    	} else {
    	    RET_INVALID_FRAGMENT;
    	}
    } else {
    	RET_INVALID_FRAGMENT;
    }
    
 bail:
    return ret;
}

/* ---- action callbacks ---- */

static int resource_matches_selector(wsman_request_t *req, res_catalog_instance_t *instance, int *fault) {
    int sel;
    if (!instance) return 0;
    D(D_BLABLA, "no_selectors: %d\n", req->no_selectors);
    for (sel = 0; sel < req->no_selectors; sel++) {
    	if (req->selectors[sel].type != SELECTOR_TYPE_SIMPLE) {
    	    D(D_BLABLA, "not simple\n");
    	    continue;
    	}

    	char uri[1000] = "", uri2[1000] = "";
    	if (instance->cat_resource->resource_uri_namespace) {
    	    char *ns_short = xml_find_namespace_shortcut(req->selectors[sel].xml, instance->cat_resource->resource_uri_namespace);
    	    if (ns_short) {
    	    	snprintf(uri2, sizeof(uri2), "%s:%s", ns_short, instance->cat_resource->resource_uri);
    	    }
    	    snprintf(uri, sizeof(uri), "%s/%s", instance->cat_resource->resource_uri_namespace, instance->cat_resource->resource_uri);
    	} else {
    	    snprintf(uri, sizeof(uri), "%s", instance->cat_resource->resource_uri);
    	}

    	if (!strcasecmp(req->selectors[sel].name, "ResourceURI")) {
    	    if (!strcasecmp(req->selectors[sel].value.simple, uri)) {
    	    	D(D_BLABLA, "match 1!\n");
    	    	return 1;
    	    } else if (uri2[0] && !strcasecmp(req->selectors[sel].value.simple, uri2)) {
    	    	D(D_BLABLA, "match 2!\n");
    	    	return 1;
    	    } else {
    	    	D(D_BLABLA, "no match: %s %s %s\n", req->selectors[sel].value.simple, uri, uri2);
    	    	if (fault) {
    	    	    *fault = FAULT_DETAIL_SELECTOR_INVALID_VALUE;
    	    	}
    	    }
    	} else if (fault) {
    	    D(D_BLABLA, "wrong selector: %s\n", req->selectors[sel].name);
    	    *fault = FAULT_DETAIL_SELECTOR_UNEXPECTED_SELECTORS;
    	}
    }
    return 0;
}

static wsman_instance_t * get_catalog_instance_by_request(wsman_resource_t * resource UNUSED,
							  wsman_request_t *req, int *fault, const char *auth_string UNUSED) {
    size_t res;
    
    if (!catalog_instances) {
    	return NULL;
    }
    
    for (res = 0; res < vector_size(catalog_instances); res++) {
    	res_catalog_instance_t *instance =
    	    	(res_catalog_instance_t *)vector_get(catalog_instances, res);
    	if (resource_matches_selector(req, instance, fault)) {
    	    return (wsman_instance_t *)instance;
    	}
    }
    return NULL;
}

static wsman_instance_t * get_first_catalog_instance(wsman_resource_t *res UNUSED, wsman_request_t *req, int *is_last, const char *auth_string UNUSED) {
    size_t size;
    D(D_BLABLA, "Fetching first catalog instance\n");
    if (catalog_instances && (size = vector_size(catalog_instances))) {
    	size_t i;
    	for (i = 0; i < size; i++) {
    	    res_catalog_instance_t *instance = (res_catalog_instance_t *)vector_get(catalog_instances, i);
    	    /* if selectors are given, process them */
    	    if (req->no_selectors == 0 || resource_matches_selector(req, instance, NULL)) {
    	    	*is_last = i == 1;
    	    	return (wsman_instance_t *)instance;
    	    }
    	}
    }
    return NULL;
}

static wsman_instance_t * get_next_catalog_instance(wsman_resource_t *res UNUSED, wsman_request_t *req,
						    wsman_instance_t *instance, int *is_last, const char *auth_string UNUSED) {
    size_t i, j, size = vector_size(catalog_instances);
    
    for (i = 0; i < size && instance; i++) {
    	wsman_instance_t *inst = vector_get(catalog_instances, i);
    	if (inst && inst == instance && (i + 1) < size) {
    	    /* we found the last instance, let's find the next one that suits */
    	    for (j = i + 1; j < size; j++) {
    	    	res_catalog_instance_t *new_inst = vector_get(catalog_instances, j);
    	    	/* if selectors are given, process them */
    	    	if (req->no_selectors == 0 || resource_matches_selector(req, new_inst, NULL)) {
    	    	    *is_last = size <= (j + 1);
    	    	    return (wsman_instance_t *)new_inst;
    	    	}
    	    }
    	}
    }
    return NULL;
}

static int get_resource_catalog_instance(xml_writer_t xml, wsman_request_t *req UNUSED,
					 wsman_instance_t *instance, int *f1 UNUSED,
					 int *f2 UNUSED, char **f3 UNUSED) {
    wsman_add_resource_catalog(xml, ((res_catalog_instance_t *)instance)->cat_resource);
    return 0;
}

static int get_resource_catalog_fragment(xml_writer_t xml, wsman_request_t *req,
					 wsman_instance_t *instance, int *fault_code,
					 int *fault_detail, char **fault_detail_string) {
    return add_resource_catalog_fragment(xml, req, instance, fault_code, fault_detail, fault_detail_string);
}

static int get_resource_catalog_selector(xml_writer_t xml, wsman_request_t *req UNUSED,
					 wsman_instance_t *instance, int *f1 UNUSED,
					 int *f2 UNUSED, char **f3 UNUSED) {
    res_catalog_instance_t *cat_inst = (res_catalog_instance_t *)instance;
    
    xml_writer_t selset = print_xml_tag(xml, "wsman", "SelectorSet", NULL, NULL);
    print_xml_tag(selset, "wsman", "Selector",
    	cat_inst->cat_resource->resource_uri, "", "Name", "", "ResourceURI");
    
    return 0;
}
