#include "debug.h"

#include "resource.h"
#include "fault.h"
#include "xml_writer.h"

#include "res_dummy.h"
#include "res_catalog.h"
#include "res_this.h"
#include "res_cim.h"

/******************************************************************************
* exported data                                                               *
******************************************************************************/

/* all resources in one array */
vector_t * wsman_resources_in_catalog = NULL;

vector_t * wsman_resources_not_in_catalog = NULL;

/******************************************************************************
* exported functions                                                          *
******************************************************************************/

int initialize_resources(void) {
    int ret = 0;
    
    wsman_resources_in_catalog = vector_new(NULL, 0, NULL);
    wsman_resources_not_in_catalog = vector_new(NULL, 0, NULL);
    
    /* initialize all resources, order is important! */
    if ((ret = this_init())
#if USE_DUMMY_RESOURCE
        || (ret = res_dummy_init())
#endif /* USE_DUMMY_RESOURCE */
#if defined(PP_FEAT_CLP)
        || (ret = cim_resources_init())
#endif /* PP_FEAT_CLP */
        /* the catalog must be the last initialized resource! */
        || (ret = res_catalog_init())
       ) {
    	return ret;
    }
    
    return 0;
}

void cleanup_resources(void) {
#if USE_DUMMY_RESOURCE
    res_dummy_cleanup();
#endif /* USE_DUMMY_RESOURCE */
#if defined(PP_FEAT_CLP)
    cim_resources_cleanup();
#endif /* PP_FEAT_CLP */
    res_catalog_cleanup();
    
    if (wsman_resources_in_catalog) {
    	vector_delete(wsman_resources_in_catalog);
    }
    if (wsman_resources_not_in_catalog) {
    	vector_delete(wsman_resources_not_in_catalog);
    }
}

int add_resource(wsman_resource_t *resource, int in_catalog) {
    vector_t *res_vec = in_catalog ? wsman_resources_in_catalog : wsman_resources_not_in_catalog;
    D(D_VERBOSE, "Adding resource %s %s\n", resource->resource_uri,
    	in_catalog ? "to catalog" : "(not in catalog)");

    vector_add(res_vec, resource);

    return 0;
}

static int compare_uri_with_namespace(char *namespace, char *resource_uri, char *uri_with_ns, char divider) {
    if (!namespace || !resource_uri || !uri_with_ns) {
        return 0;
    }

    size_t ns_size = strlen(namespace) + 1;
    size_t uri_size = strlen(resource_uri);
    size_t size = ns_size + uri_size + 1;
    char s[size];
    snprintf(s, size, "%s%c%s", namespace, divider, resource_uri);
    return !strcasecmp(s, uri_with_ns);
}

static int compare_resource_by_uri(wsman_resource_t *res, char *uri, xml_parser_t uri_xml) {
    if (res->resource_uri_ns_shortcut && compare_uri_with_namespace(res->resource_uri_ns_shortcut, res->resource_uri, uri, ':')) {
        return 1;
    }
    
    // try the other possibilities
    if (res->resource_uri_namespace) {
    	char *ns_long = res->resource_uri_namespace;
    	char *ns_short = xml_find_namespace_shortcut(uri_xml, res->resource_uri_namespace);
    	
    	return compare_uri_with_namespace(ns_long, res->resource_uri, uri, '/') || compare_uri_with_namespace(ns_short, res->resource_uri, uri, ':');
    } else {
    	return !strcasecmp(res->resource_uri, uri);
    }
}

wsman_resource_t * find_resource_by_uri(char *uri, xml_parser_t uri_xml) {
    size_t i, size;
    
    size = vector_size(wsman_resources_in_catalog);
    for (i = 0; i < size; i++) {
    	wsman_resource_t *res = vector_get(wsman_resources_in_catalog, i);
    	if (res && compare_resource_by_uri(res, uri, uri_xml)) {
    	    return res;
    	}
    }

    size = vector_size(wsman_resources_not_in_catalog);
    for (i = 0; i < size; i++) {
    	wsman_resource_t *res = vector_get(wsman_resources_not_in_catalog, i);
    	if (res && compare_resource_by_uri(res, uri, uri_xml)) {
    	    return res;
    	}
    }
    
    return NULL;
}

action_t find_action_by_name(wsman_resource_t *resource, char *action) {
    int i;
    
    if (!resource || !resource->ops) {
    	return NULL;
    }
    
    for (i = 0; resource->ops[i].actions; i++) {
    	int j;
    	wsman_operation_t * op = &resource->ops[i];
    	
    	for (j = 0; op->actions && op->actions[j].action; j++) {
    	    if (!strcasecmp(op->actions[j].action, action)) {
    	    	return op->actions[j].function;
    	    }
    	}
    }
    
    return NULL;
}

int initialize_instance(wsman_instance_t * inst, wsman_resource_t * res) {
    inst->resource = res;
    return 0;
}

/*---- dynamically allocating and filling resources ----*/

wsman_resource_t *create_resource(const char *uri_namespace, const char *uri) {
    wsman_resource_t *ret = malloc(sizeof(wsman_resource_t));
    assert(uri);
    memset(ret, 0, sizeof(wsman_resource_t));
    ret->resource_uri = strdup(uri);
    ret->resource_uri_namespace = uri_namespace ? strdup(uri_namespace) : NULL;
    return ret;
}

void resource_set_notes(wsman_resource_t *resource, const char *notes) {
    assert(resource && notes);
    if (resource->notes) free(resource->notes);
    resource->notes = strdup(notes);
}

void resource_set_vendor(wsman_resource_t *resource, const char *vendor) {
    assert(resource && vendor);
    if (resource->vendor) free(resource->vendor);
    resource->vendor = strdup(vendor);
}

void resource_set_dispname(wsman_resource_t *resource, const char *display_name) {
    assert(resource && display_name);
    if (resource->display_name) free(resource->display_name);
    resource->display_name = strdup(display_name);
}

wsman_keyword_t *add_keywords(wsman_resource_t *resource, const char *prefix, const char *binding) {
    int pos = 0;
    assert(resource && prefix && binding);

    if (resource->keywords) {
    	while (resource->keywords[pos].binding_prefix) pos++;
    	resource->keywords = realloc(resource->keywords, (pos + 2) * sizeof(wsman_keyword_t));
    } else {
    	resource->keywords = malloc((pos + 2) * sizeof(wsman_keyword_t));
    }
    
    memset(&resource->keywords[pos], 0, sizeof(wsman_keyword_t));
    memset(&resource->keywords[pos + 1], 0, sizeof(wsman_keyword_t));
    
    resource->keywords[pos].binding_prefix = strdup(prefix);
    resource->keywords[pos].binding = strdup(binding);
    
    return &resource->keywords[pos];
}

void add_keyword(wsman_keyword_t *kw, const char *word) {
    int pos = 0;
    assert(kw && word);
    
    if (kw->words) {
    	while (kw->words[pos]) pos++;
    	kw->words = realloc(kw->words, (2 + pos) * sizeof(char *));
    } else {
    	kw->words = malloc((2 + pos) * sizeof(char *));
    }
    
    kw->words[pos] = strdup(word);
    kw->words[pos + 1] = NULL;
}

wsman_operation_t *add_operation(wsman_resource_t *resource, const char *wsdl_port, const char *wsdl_ref) {
    int pos = 0;
    assert(resource);
    
    if (resource->ops) {
    	while (resource->ops[pos].actions) pos++;
    	resource->ops = realloc(resource->ops, (pos + 2) * sizeof(wsman_operation_t));
    } else {
    	resource->ops = malloc((pos + 2) * sizeof(wsman_operation_t));
    }

    memset(&resource->ops[pos], 0, sizeof(wsman_operation_t));
    memset(&resource->ops[pos + 1], 0, sizeof(wsman_operation_t));
    
    if (wsdl_port) resource->ops[pos].wsdl_port = strdup(wsdl_port);
    if (wsdl_ref) resource->ops[pos].wsdl_ref = strdup(wsdl_ref);
    
    return &resource->ops[pos];
}

void op_add_action(wsman_operation_t *op, const char *action, action_t function) {
    int pos = 0;
    assert(op && action);
    
    if (op->actions) {
    	while (op->actions[pos].action) pos++;
    	op->actions = realloc(op->actions, (pos + 2) * sizeof(wsman_action_t));
    } else {
    	op->actions = malloc((pos + 2) * sizeof(wsman_action_t));
    }

    memset(&op->actions[pos], 0, sizeof(wsman_action_t));
    memset(&op->actions[pos + 1], 0, sizeof(wsman_action_t));
    
    op->actions[pos].action = strdup(action);
    op->actions[pos].function = function;
}

void op_add_selsetref(wsman_operation_t *op, const char *selsetref) {
    int pos = 0;
    assert(op && selsetref);
    
    if (op->selector_set_ref) {
    	while (op->selector_set_ref[pos]) pos++;
    	op->selector_set_ref = realloc(op->selector_set_ref, (pos + 2) * sizeof(char *));
    } else {
    	op->selector_set_ref = malloc((pos + 2) * sizeof(char *));
    }

    op->selector_set_ref[pos] = strdup(selsetref);
    op->selector_set_ref[pos + 1] = NULL;
}

void op_add_optsetref(wsman_operation_t *op, const char *optsetref) {
    int pos = 0;
    assert(op && optsetref);
    
    if (op->option_set_ref) {
    	while (op->option_set_ref[pos]) pos++;
    	op->option_set_ref = realloc(op->option_set_ref, (pos + 2) * sizeof(char *));
    } else {
    	op->option_set_ref = malloc((pos + 2) * sizeof(char *));
    }

    op->option_set_ref[pos] = strdup(optsetref);
    op->option_set_ref[pos + 1] = NULL;
}

void op_add_schemaref(wsman_operation_t *op, const char *uri_prefix,
		      const char *uri, const char *location, const char *schema) {
    
    int pos = 0;
    assert(op);
    
    if (op->schema_ref) {
    	while (op->schema_ref[pos].uri_prefix) pos++;
    	op->schema_ref = realloc(op->schema_ref, (pos + 2) * sizeof(wsman_schema_ref_t));
    } else {
    	op->schema_ref = malloc((pos + 2) * sizeof(wsman_schema_ref_t));
    }

    memset(&op->schema_ref[pos], 0, sizeof(wsman_schema_ref_t));
    memset(&op->schema_ref[pos + 1], 0, sizeof(wsman_schema_ref_t));
    
    if (uri_prefix) op->schema_ref[pos].uri_prefix = strdup(uri_prefix);
    if (uri) op->schema_ref[pos].uri = strdup(uri);
    if (location) op->schema_ref[pos].location = strdup(location);
    if (schema) op->schema_ref[pos].schema = strdup(schema);
}

void op_add_filter_dialect(wsman_operation_t *op, const char *filter_dialect) {
    int pos = 0;
    assert(op && filter_dialect);
    
    if (op->filter_dialects) {
    	while (op->filter_dialects[pos]) pos++;
    	op->filter_dialects = realloc(op->filter_dialects, (pos + 2) * sizeof(char *));
    } else {
    	op->filter_dialects = malloc((pos + 2) * sizeof(char *));
    }

    op->filter_dialects[pos] = strdup(filter_dialect);
    op->filter_dialects[pos + 1] = NULL;
}

void op_set_filters_required(wsman_operation_t *op, int filters_required) {
    assert(op);
    op->filters_required = filters_required;
}

void op_add_delivery_mode(wsman_operation_t *op, const char *delivery_mode) {
    int pos = 0;
    assert(op && delivery_mode);
    
    if (op->delivery_modes) {
    	while (op->delivery_modes[pos]) pos++;
    	op->delivery_modes = realloc(op->delivery_modes, (pos + 2) * sizeof(char *));
    } else {
    	op->delivery_modes = malloc((pos + 2) * sizeof(char *));
    }

    op->delivery_modes[pos] = strdup(delivery_mode);
    op->delivery_modes[pos + 1] = NULL;
}

wsman_selector_set_t *resource_add_selectorset(wsman_resource_t *resource, const char *name) {
    int pos = 0;
    assert(resource && name);
    
    if (resource->sel_set) {
    	while (resource->sel_set[pos].set_name) pos++;
    	resource->sel_set = realloc(resource->sel_set, (pos + 2) * sizeof(wsman_selector_set_t));
    } else {
    	resource->sel_set = malloc((pos + 2) * sizeof(wsman_selector_set_t));
    }

    memset(&resource->sel_set[pos], 0, sizeof(wsman_selector_set_t));
    memset(&resource->sel_set[pos + 1], 0, sizeof(wsman_selector_set_t));
    
    resource->sel_set[pos].set_name = strdup(name);
    
    return &resource->sel_set[pos];
}

void add_selector(wsman_selector_set_t *set, const char *sel_name, const char *type, const char *description) {
    int pos = 0;
    assert(set && sel_name);

    if (set->selectors) {
    	while (set->selectors[pos].sel_name) pos++;
    	set->selectors = realloc(set->selectors, (pos + 2) * sizeof(wsman_selector_t));
    } else {
    	set->selectors = malloc((pos + 2) * sizeof(wsman_selector_t));
    }

    memset(&set->selectors[pos], 0, sizeof(wsman_selector_t));
    memset(&set->selectors[pos + 1], 0, sizeof(wsman_selector_t));
    
    set->selectors[pos].sel_name = strdup(sel_name);
    if (type) set->selectors[pos].type = strdup(type);
    if (description) set->selectors[pos].description = strdup(description);
}

wsman_option_set_t *resource_add_optionset(wsman_resource_t *resource, const char *name) {
    int pos = 0;
    assert(resource && name);
    
    if (resource->opt_set) {
    	while (resource->opt_set[pos].set_name) pos++;
    	resource->opt_set = realloc(resource->opt_set, (pos + 2) * sizeof(wsman_option_set_t));
    } else {
    	resource->opt_set = malloc((pos + 2) * sizeof(wsman_option_set_t));
    }

    memset(&resource->opt_set[pos], 0, sizeof(wsman_option_set_t));
    memset(&resource->opt_set[pos + 1], 0, sizeof(wsman_option_set_t));
    
    resource->opt_set[pos].set_name = strdup(name);
    
    return &resource->opt_set[pos];
}

void add_option(wsman_option_set_t *set, const char *opt_name, const char *type, const char *description) {
    int pos = 0;
    assert(set && opt_name);

    if (set->options) {
    	while (set->options[pos].opt_name) pos++;
    	set->options = realloc(set->options, (pos + 2) * sizeof(wsman_option_t));
    } else {
    	set->options = malloc((pos + 2) * sizeof(wsman_option_t));
    }

    memset(&set->options[pos], 0, sizeof(wsman_option_t));
    memset(&set->options[pos + 1], 0, sizeof(wsman_option_t));
    
    set->options[pos].opt_name = strdup(opt_name);
    if (type) set->options[pos].type = strdup(type);
    if (description) set->options[pos].description = strdup(description);
}

void resource_add_relationship(wsman_resource_t *resource, const char *type, const char *role, const char *ref) {
    int pos = 0;
    assert(resource && (type || role || ref));
    
    if (resource->relships) {
    	while (resource->relships[pos].type || resource->relships[pos].role || resource->relships[pos].ref) pos++;
    	resource->relships = realloc(resource->relships, (pos + 2) * sizeof(wsman_relship_t));
    } else {
    	resource->relships = malloc((pos + 2) * sizeof(wsman_relship_t));
    }

    memset(&resource->relships[pos], 0, sizeof(wsman_relship_t));
    memset(&resource->relships[pos + 1], 0, sizeof(wsman_relship_t));

    if (type) resource->relships[pos].type = strdup(type);
    if (role) resource->relships[pos].role = strdup(role);
    if (ref) resource->relships[pos].ref = strdup(ref);
}

void resource_set_functions(wsman_resource_t *resource,
			    get_instance_by_request_t get_instance_by_request,
			    get_first_instance_t get_first_instance,
			    get_next_instance_t get_next_instance,
			    get_instance_data_t get_instance_data,
			    get_instance_data_t get_instance_data_fragment,
			    get_instance_selectors_t get_instance_selectors,
			    put_instance_data_t put_instance_data,
			    put_instance_data_t put_instance_data_fragment,
			    release_instance_t release_instance,
			    cleanup_instance_t cleanup_instance) {
    assert(resource);
    resource->get_instance_by_request = get_instance_by_request;
    resource->get_first_instance = get_first_instance;
    resource->get_next_instance = get_next_instance;
    resource->get_instance_data = get_instance_data;
    resource->get_instance_data_fragment = get_instance_data_fragment;
    resource->get_instance_selectors = get_instance_selectors;
    resource->put_instance_data = put_instance_data;
    resource->put_instance_data_fragment = put_instance_data_fragment;
    resource->release_instance = release_instance;
    resource->cleanup_instance = cleanup_instance;
}

void delete_resource(wsman_resource_t *res) {
    int i, j;
    
    if (!res) return;

    if (res->resource_uri_namespace)	free(res->resource_uri_namespace);
    if (res->resource_uri)		free(res->resource_uri);
    if (res->notes)			free(res->notes);
    if (res->vendor)			free(res->vendor);
    if (res->display_name)		free(res->display_name);
    
    if (res->keywords) {
    	for (i = 0; res->keywords[i].binding_prefix; i++) {
    	    wsman_keyword_t * kw = &res->keywords[i];
    	    if (kw->binding_prefix)	free(kw->binding_prefix);
    	    if (kw->binding)		free(kw->binding);
    	    if (kw->words) {
    	    	for (j = 0; kw->words[j]; j++) {
    	    	    free(kw->words[j]);
    	    	}
    	    	free(kw->words);
    	    }
    	}
    	
    	free(res->keywords);
    }
    
    if (res->ops) {
    	for (i = 0; res->ops[i].actions; i++) {
    	    wsman_operation_t *op = &res->ops[i];
    	    
    	    if (op->wsdl_port)	free(op->wsdl_port);
    	    if (op->wsdl_ref)	free(op->wsdl_ref);
    	    
    	    if (op->actions) {
    	    	for (j = 0; op->actions[j].action; j++) {
    	    	    free(op->actions[j].action);
    	    	}
    	    	free(op->actions);
    	    }
    	    
    	    if (op->selector_set_ref) {
    	    	for (j = 0; op->selector_set_ref[j]; j++) {
    	    	    free(op->selector_set_ref[j]);
    	    	}
    	    	free(op->selector_set_ref);
    	    }
    	    
    	    if (op->option_set_ref) {
    	    	for (j = 0; op->option_set_ref[j]; j++) {
    	    	    free(op->option_set_ref[j]);
    	    	}
    	    	free(op->option_set_ref);
    	    }
    	    
    	    if (op->schema_ref) {
    	    	for (j = 0; op->schema_ref[j].uri_prefix; j++) {
    	    	    wsman_schema_ref_t *s = &op->schema_ref[j];
    	    	    if (s->uri_prefix)	free(s->uri_prefix);
    	    	    if (s->uri)		free(s->uri);
    	    	    if (s->location)	free(s->location);
    	    	    if (s->schema)	free(s->schema);
    	    	}
    	    	free(op->schema_ref);
    	    }
    	    
    	    if (op->filter_dialects) {
    	    	for (j = 0; op->filter_dialects[j]; j++) {
    	    	    free(op->filter_dialects[j]);
    	    	}
    	    	free(op->filter_dialects);
    	    }
    	    
    	    if (op->delivery_modes) {
    	    	for (j = 0; op->delivery_modes[j]; j++) {
    	    	    free(op->delivery_modes[j]);
    	    	}
    	    	free(op->delivery_modes);
    	    }
    	}
    	
    	free(res->ops);
    }
    
    if (res->sel_set) {
    	for (i = 0; res->sel_set[i].set_name; i++) {
    	    wsman_selector_set_t *set = &res->sel_set[i];
    	    
    	    if (set->selectors) {
    	    	for (j = 0; set->selectors[j].sel_name; j++) {
    	    	    wsman_selector_t *sel = &set->selectors[j];
    	    	    if (sel->sel_name)		free(sel->sel_name);
    	    	    if (sel->type)		free(sel->type);
    	    	    if (sel->description)	free(sel->description);
    	    	}
    	    }
    	    
    	    free(set->set_name);
    	}
    	
    	free(res->sel_set);
    }
    
    if (res->opt_set) {
    	for (i = 0; res->opt_set[i].set_name; i++) {
    	    wsman_option_set_t *set = &res->opt_set[i];
    	    
    	    if (set->options) {
    	    	for (j = 0; set->options[j].opt_name; j++) {
    	    	    wsman_option_t *opt = &set->options[j];
    	    	    if (opt->opt_name)		free(opt->opt_name);
    	    	    if (opt->type)		free(opt->type);
    	    	    if (opt->description)	free(opt->description);
    	    	}
    	    }
    	    
    	    if (set->set_name)	free(set->set_name);
    	}
    	
    	free(res->opt_set);
    }
    
    if (res->relships) {
    	for (i = 0; res->relships[i].type || res->relships[i].role || res->relships[i].ref; i++) {
    	    wsman_relship_t *rel = &res->relships[i];
    	    if (rel->type)	free(rel->type);
    	    if (rel->role)	free(rel->role);
    	    if (rel->ref)	free(rel->ref);
    	}
    	
    	free(res->relships);
    }
    
    free(res);
}
