/**
 * Basic resource/instance definitions.
 * 
 * geo 6.Sep 2006: I added some coarse documentation to this that was
 * derived from sourcecode analysis. Don't expect it to be complete
 * and error-free. Feel free to add comments as you find undocumented
 * stuff.
 */

#ifndef __RESOURCE_H
#define __RESOURCE_H

#include <pp/base.h>
#include <pp/vector.h>

#include "xml_writer.h"
#include "uuid.h"

/******************************************************************************
* data types                                                                  *
******************************************************************************/

/* arrays of <type> are always treated as follows:
   - defined as <type> *
   - last element must be NULL
*/

/* embedded structs, defined below */
typedef struct s_wsman_instance		wsman_instance_t;
typedef struct s_wsman_resource		wsman_resource_t;
typedef struct s_wsman_keyword		wsman_keyword_t;
typedef struct s_wsman_operation	wsman_operation_t;
typedef struct s_wsman_selector_set	wsman_selector_set_t;
typedef struct s_wsman_selector		wsman_selector_t;
typedef struct s_wsman_option_set	wsman_option_set_t;
typedef struct s_wsman_option		wsman_option_t;
typedef struct s_wsman_relship		wsman_relship_t;
typedef struct s_wsman_schema_ref	wsman_schema_ref_t;
typedef struct s_wsman_action		wsman_action_t;


/* one instance of the resource */
struct s_wsman_instance {
    /* the instance's resource */
    wsman_resource_t* resource;
};


/* operation function type */
typedef int (*action_t)(xml_writer_t, wsman_resource_t *,
			wsman_request_t *, char **, int *, int *, char **, char *);
/* yet undocumented */
typedef wsman_instance_t * (*get_instance_by_request_t)(wsman_resource_t* resource,
                                                        wsman_request_t *,
                                                        int *, const char *string);
/* 
 * Get the first instance of a resource, used for resource enumeration by standard_actions.c.
 * May return NULL. 'is_last_instance' must be set to 1 if there are no more instances after
 * the returned instance. Returns a locked instance that must be released and cleaned up after
 * use (relase_instance() and cleanup_instance()).
 */
typedef wsman_instance_t * (*get_first_instance_t)(wsman_resource_t* resource,
                                                   wsman_request_t* request,
                                                   int* is_last_instance,
                                                   const char *auth_string);
/* 
 * Get the next instance of a resource based on a previous instance. Used for resource
 * enumeration by standard_actions.c. May return NULL if there are no more references or
 * the next reference cannot be determined. 'is_last_instance' must be set to 1 if there
 * are no more instances after the returned instance.  Returns a locked instance that must
 * be released and cleaned up after use (relase_instance() and cleanup_instance()).
 * 
 * Attention: the provided handle for previous instance is currently can be released, but not
 * cleaned up. As a result, it may be invalid if the resource does not exist any more. The
 * implementation of this function has to take that into account.
 */
typedef wsman_instance_t * (*get_next_instance_t)(wsman_resource_t* resource,
                                                  wsman_request_t* request,
                                                  wsman_instance_t* previous_instance,
                                                  int* is_last_instance,
                                                  const char *auth_string);
typedef int (*get_instance_data_t)(xml_writer_t, wsman_request_t *, wsman_instance_t *, int *, int *, char **);
typedef int (*get_instance_selectors_t)(xml_writer_t, wsman_request_t *, wsman_instance_t *, int *, int *, char **);
typedef int (*put_instance_data_t)(wsman_instance_t *, wsman_request_t *, int *, int *, char **);
/**
 * Release the provided instance. The instance must not be accessed any more. Some
 * CIM providers currently still make use of the released instance before cleanup,
 * but this will change in the future.
 */
typedef int (*release_instance_t)(wsman_resource_t *, wsman_instance_t *);
/**
 * Clean up the provided instance. Must be called after calling release() to
 * completely free the instance. 
 */
typedef int (*cleanup_instance_t)(wsman_resource_t *, wsman_instance_t *);


/* this include describes one WS-Management resource */
struct s_wsman_resource {
    /* general part */
    char *resource_uri_namespace;	/* fullname of the resource URI namespace */
    char *resource_uri_ns_shortcut;	/* namespace shortcut which always matches (used for wsman:... resources) */
    char *resource_uri;			/* uri of the resource */
    char *notes;			/* notes for the client */
    char *vendor;			/* vendor string */
    char *display_name;			/* display name string */
    
    wsman_keyword_t *keywords;		/* array, keyword table */
    
    /* access block */
    // char *compliance;		/* compliance string; same for all resources */
    wsman_operation_t *ops;		/* operation table */
    wsman_selector_set_t *sel_set;	/* selector set table */
    wsman_option_set_t *opt_set;	/* option set table */
    
    /* relationships */
    wsman_relship_t *relships;		/* relation ship table */
    
    /* open content */
    void *open_content;
    
    /* functions supporting WS-Management operations */
    get_instance_by_request_t get_instance_by_request;
    get_first_instance_t get_first_instance;
    get_next_instance_t get_next_instance;
    
    get_instance_data_t get_instance_data;
    get_instance_data_t get_instance_data_fragment;
    get_instance_selectors_t get_instance_selectors;
    put_instance_data_t put_instance_data;
    put_instance_data_t put_instance_data_fragment;
    release_instance_t release_instance;	/* releases locks on instance */
    cleanup_instance_t cleanup_instance;	/* deletes the instance */
};

/* WS-Management keyword table entry */
struct s_wsman_keyword {
    char *binding_prefix;
    char *binding;
    char **words;
};

struct s_wsman_schema_ref {
    char *uri_prefix;
    char *uri;
    char *location;
    char *schema;
};

struct s_wsman_action {
    char *action;
    action_t function;
};

/* WS-Management operation table entry */
struct s_wsman_operation {
    char *wsdl_port;
    char *wsdl_ref;
    wsman_action_t *actions;		/* table of available actions */
    char **selector_set_ref;		/* selector set reference */
    char **option_set_ref;		/* option set reference */
    wsman_schema_ref_t *schema_ref;	/* schema reference */
    char **filter_dialects;
    int filters_required;
    char **delivery_modes;
};

/* WS-Management selector set */
struct s_wsman_selector_set {
    char *set_name;			/* name of the selector set */
    wsman_selector_t *selectors;
};

struct s_wsman_selector {
    char *sel_name;
    char *type;
    char *description;
};

/* WS-Management option set */
struct s_wsman_option_set {
    char *set_name;
    wsman_option_t *options;
};

struct s_wsman_option {
    char *opt_name;
    char *type;
    char *description;
};

/* WS-Management relationships */
struct s_wsman_relship {
    char *type;
    char *role;
    char *ref;
};

/******************************************************************************
* exported data                                                               *
******************************************************************************/

/* all resources which appear in the catalog */
extern vector_t * wsman_resources_in_catalog;

/* all resources which don't appear in the catalog */
extern vector_t * wsman_resources_not_in_catalog;

/******************************************************************************
* exported functions                                                          *
******************************************************************************/

/* initialize the resources */
int initialize_resources(void);

/* cleanup the resources */
void cleanup_resources(void);

/* dynamically create and update a new resource */
/* NOTE: some functions use realloc() internally,
         don't use first returned pointer after second call! */
wsman_resource_t *create_resource(const char *uri_namespace, const char *uri);
void resource_set_notes(wsman_resource_t *resource, const char *notes);
void resource_set_vendor(wsman_resource_t *resource, const char *vendor);
void resource_set_dispname(wsman_resource_t *resource, const char *display_name);
wsman_keyword_t *add_keywords(wsman_resource_t *resource, const char *prefix, const char *binding);
void add_keyword(wsman_keyword_t *keywords, const char *word);
wsman_operation_t *add_operation(wsman_resource_t *resource, const char *wsdl_port, const char *wsdl_ref);
void op_add_action(wsman_operation_t *op, const char *action, action_t function);
void op_add_selsetref(wsman_operation_t *op, const char *selsetref);
void op_add_optsetref(wsman_operation_t *op, const char *optsetref);
void op_add_schemaref(wsman_operation_t *op, const char *uri_prefix,
		      const char *uri, const char *location, const char *schema);
void op_add_filter_dialect(wsman_operation_t *op, const char *filter_dialect);
void op_set_filters_required(wsman_operation_t *op, int filters_required);
void op_add_delivery_mode(wsman_operation_t *op, const char *delivery_mode);
wsman_selector_set_t *resource_add_selectorset(wsman_resource_t *resource, const char *name);
void add_selector(wsman_selector_set_t *set, const char *sel_name, const char *type, const char *description);
wsman_option_set_t *resource_add_optionset(wsman_resource_t *resource, const char *name);
void add_option(wsman_option_set_t *set, const char *opt_name, const char *type, const char *description);
void resource_add_relationship(wsman_resource_t *resource, const char *type, const char *role, const char *ref);
void resource_set_functions(wsman_resource_t *resource,
			    get_instance_by_request_t get_instance_by_request,
			    get_first_instance_t get_first_instance,
			    get_next_instance_t get_next_instance,
			    get_instance_data_t get_instance_data,
			    get_instance_data_t get_instance_data_fragment,
			    get_instance_selectors_t get_instance_selectors,
			    put_instance_data_t put_instance_data,
			    put_instance_data_t put_instance_data_fragment,
			    release_instance_t release_instance,
			    cleanup_instance_t cleanup_instance);

/* delete one dynamically allocated resource */
void delete_resource(wsman_resource_t *resource);

/* add a new resource */
int add_resource(wsman_resource_t *resource, int in_catalog);

/* find a resource by its resource URI */
wsman_resource_t * find_resource_by_uri(char *uri, xml_parser_t uri_xml);

/* find an action function by the action name */
action_t find_action_by_name(wsman_resource_t *resource, char *action);

/* find an instance by selector set */
wsman_instance_t * find_instance_by_selector(wsman_resource_t *resource,
						 int no_selectors,
						 name_value_t *selectors);

/* initialize one instance */
int initialize_instance(wsman_instance_t *, wsman_resource_t *);

#endif /* __RESOURCE_H */
