#include "wsman.h"
#include "xml_parser.h"
#include "xml_writer.h"
#include "resource.h"
#include "fault.h"
#include "debug.h"
#include "uuid.h"
#include "standard_actions.h"
#include "enumerate_session.h"

#define RETURN_ERROR(code, detail, det_str)	\
    do {					\
    	*fault_code = code;			\
    	*fault_detail = detail;			\
    	*fault_detail_string = det_str;		\
    	ret = -1;				\
    	goto bail;				\
    } while(0)

static int get_instance_epr(xml_writer_t xml, wsman_instance_t *instance, wsman_request_t * req,
			    int *fault_code, int *fault_detail, char **fault_detail_string, char *auth_string UNUSED) {
    int ret = 0;
    xml_writer_t epr, params;

    
    epr = print_xml_tag(xml, "wsa", "EndpointReference", NULL, NULL);
    print_xml_tag(epr, "wsa", "Address", req->wsa_address, NULL);
    params = print_xml_tag(epr, "wsa", "ReferenceParameters", NULL, NULL);
    print_xml_tag(params, "wsman", "ResourceURI", instance->resource->resource_uri, NULL);
    
    if (instance->resource->get_instance_selectors) {
    	ret = instance->resource->get_instance_selectors(params, req, instance, fault_code, fault_detail, fault_detail_string);
    }
    
    return ret;
}

static int get_instance_data(xml_writer_t xml, wsman_request_t *req, wsman_instance_t *instance,
			     int *fault_code, int *fault_detail, char **fault_detail_string, char *auth_string UNUSED) {
    int ret = 0;
    wsman_resource_t *res = instance->resource;
    
    if (req->fragment.fragment_full) {
    	if (res->get_instance_data_fragment) {
    	    ret = res->get_instance_data_fragment(xml, req, instance,
    						  fault_code, fault_detail,
    						  fault_detail_string);
    	} else {
    	    RETURN_ERROR(FAULT_UNSUPPORTED_FEATURE, FAULT_DETAIL_UNSUPPORTED_FRAGMENT_LEVEL_ACCESS, NULL);
    	}
    } else if (res->get_instance_data) {
    	ret = res->get_instance_data(xml, req, instance,
    				     fault_code, fault_detail,
    				     fault_detail_string);
    } else {
    	D(D_ERROR, "resource does not support Get operations.\n");
    	RETURN_ERROR(FAULT_ACTION_NOT_SUPPORTED, 0, req->action_uri);
    }
    
 bail:
    return ret;
}

static int put_instance_data(wsman_request_t *req, wsman_instance_t *instance,
			     int *fault_code, int *fault_detail, char **fault_detail_string, char *auth_string UNUSED) {
    int ret = 0;
    wsman_resource_t *res = instance->resource;
    
    if (req->fragment.fragment_full) {
    	if (res->put_instance_data_fragment) {
    	    ret = res->put_instance_data_fragment(instance, req,
    						  fault_code, fault_detail,
    						  fault_detail_string);
    	} else {
    	    RETURN_ERROR(FAULT_UNSUPPORTED_FEATURE, FAULT_DETAIL_UNSUPPORTED_FRAGMENT_LEVEL_ACCESS, NULL);
    	}
    } else if (res->put_instance_data) {
    	ret = res->put_instance_data(instance, req,
    				     fault_code, fault_detail,
    				     fault_detail_string);
    } else {
    	D(D_ERROR, "instance does not support Put operations.\n");
    	RETURN_ERROR(FAULT_ACTION_NOT_SUPPORTED, 0, req->action_uri);
    }
    
 bail:
    return ret;
}

int resource_get(xml_writer_t xml, wsman_resource_t * res,
		 wsman_request_t * req, char ** action, int *fault_code,
		 int *fault_detail, char **fault_detail_string, char *auth_string) {
    int ret = 0, fault = 0;
    wsman_instance_t *instance = NULL;
    
    assert(res);
    
    /* try to find the correct instance */
    if (!res->get_instance_by_request) {
    	D(D_ERROR, "cannot find instance by request.\n");
    	RETURN_ERROR(FAULT_ACTION_NOT_SUPPORTED, 0, req->action_uri);
    }
    if (!(instance = res->get_instance_by_request(res, req, &fault, auth_string))) {
    	D(D_ERROR, "the requested instance could not be found.\n");
    	RETURN_ERROR(FAULT_INVALID_SELECTORS, fault ? fault : FAULT_DETAIL_SELECTOR_INVALID_VALUE, NULL);
    }
    
    /* add the instance to the stream */
    if ((ret = get_instance_data(xml, req, instance, fault_code,
    				 fault_detail, fault_detail_string, auth_string))) {
    	/* the error variables are already set here */
    	goto bail;
    }
    
    *action = strdup(NAMESPACE_TRANSFER "/GetResponse");
    
 bail:
    if (instance && res->release_instance) {
    	res->release_instance(res, instance);
    }
    if (instance && res->cleanup_instance) {
    	res->cleanup_instance(res, instance);
    }
    return ret;
}

int resource_put(xml_writer_t xml, wsman_resource_t * res,
		 wsman_request_t * req, char ** action, int *fault_code,
		 int *fault_detail, char **fault_detail_string, char *auth_string) {
    int ret = 0, fault = 0;
    wsman_instance_t *instance = NULL;
    
    assert(res);
    
    /* try to find the correct instance */
    if (!res->get_instance_by_request) {
    	D(D_ERROR, "cannot find instance by request.\n");
    	RETURN_ERROR(FAULT_ACTION_NOT_SUPPORTED, 0, req->action_uri);
    }
    if (!(instance = res->get_instance_by_request(res, req, &fault, auth_string))) {
    	D(D_ERROR, "the requested instance could not be found.\n");
    	RETURN_ERROR(FAULT_INVALID_SELECTORS, fault ? fault : FAULT_DETAIL_SELECTOR_INVALID_VALUE, NULL);
    }
    
    /* update the instance */
    if ((ret = put_instance_data(req, instance, fault_code, fault_detail,
    				 fault_detail_string, auth_string))) {
    	/* the error variables are already set here */
    	goto bail;
    }
    
    /* add the instance to the stream */
    if ((ret = get_instance_data(xml, req, instance,
    				 fault_code, fault_detail,
    				 fault_detail_string, auth_string))) {
    	/* the error variables are already set here */
    	goto bail;
    }
    
    *action = strdup(NAMESPACE_TRANSFER "/PutResponse");

 bail:
    if (instance && res->release_instance) {
    	res->release_instance(res, instance);
    }
    if (instance && res->cleanup_instance) {
    	res->cleanup_instance(res, instance);
    }
    return ret;
}

int resource_enumerate(xml_writer_t xml, wsman_resource_t * res,
		       wsman_request_t * req, char ** action, int *fault_code,
		       int *fault_detail, char **fault_detail_string, char *auth_string UNUSED) {
    int ret = 0;
    xml_writer_t resp;
    char context[UUID_LENGTH];
    enumerate_session_t *session;
    
    assert(res);

    D(D_VERBOSE, "Starting enumeration of Resource %s\n", res->resource_uri);
    
    /* check whether enumeration is supported */
    if (!res->get_first_instance ||
    	(req->enum_type != ENUMERATION_TYPE_EPR && !res->get_instance_data)) {
    	
    	D(D_ERROR, "enumeration not supported by the resource.\n");
    	RETURN_ERROR(FAULT_ACTION_NOT_SUPPORTED, 0, req->action_uri);
    }

    /* create a session and add the enumeration context */
    build_uuid(context);
    session = create_session(req->enum_type, context, res, NULL);
    if (!session) {
    	D(D_ERROR, "Could not create enumeration session.\n");
    	RETURN_ERROR(FAULT_INTERNAL_ERROR, 0, "Out of memory");
    }
    
    resp = print_xml_tag(xml, "wsen", "EnumerateResponse", NULL, NULL);
    print_xml_tag(resp, "wsen", "EnumerationContext", context, NULL);
    
    D(D_BLABLA, "Set enumeration context to %s\n", context);
    
    *action = strdup(NAMESPACE_ENUMERATION "/EnumerateResponse");
 bail:
    return ret;
}

int resource_pull(xml_writer_t xml, wsman_resource_t * res,
		 wsman_request_t * req, char ** action, int *fault_code,
		 int *fault_detail, char **fault_detail_string, char *auth_string) {
    int ret = 0;
    wsman_instance_t *instance = NULL, *previous_instance = NULL;
    int is_last_instance = 0;
    xml_writer_t resp, items, item;
    enumerate_session_t *session;
    char uuid[UUID_LENGTH];

    assert(res);

    D(D_VERBOSE, "Pulling of Resource %s, EnumerationContext: %s\n",
    	res->resource_uri, req->enumeration_context);

    /* find the requested instance of the resource; this also starts the enumeration */
    lock_sessions();
    
    session = get_session(req->enumeration_context);
    if (!session) {
    	D(D_ERROR, "invalid enumeration context: %s.\n",
    		req->enumeration_context ? req->enumeration_context : "(none)");
    	RETURN_ERROR(FAULT_INVALID_ENUMERATION_CONTEXT, 0, NULL);
    }
    
    /* find the instance */
    previous_instance = session->instance;
    if (!previous_instance) {
    	/* find the first instance of the resource; this also starts the enumeration */
    	instance = res->get_first_instance(res, req, &is_last_instance, auth_string);
    	if (!res->get_next_instance) {
    	    /* this must be the last one */
    	    is_last_instance = 1;
    	}
    } else {
    	/* use the next instance */
    	instance = res->get_next_instance(res, req, previous_instance, &is_last_instance, auth_string);
    }

    D(D_BLABLA, "Instance is %p (last: %d)\n", instance, is_last_instance);
    
    build_uuid(uuid);
    
    update_session(session, uuid, instance);

    /* add the pull response */
    resp = print_xml_tag(xml, "wsen", "PullResponse", NULL, NULL);    
    print_xml_tag(resp, "wsen", "EnumerationContext", uuid, NULL);
    items = print_xml_tag(resp, "wsen", "Items", NULL, NULL);
    
    if (instance) {
    	switch (session->type) {
    	    case ENUMERATION_TYPE_OBJECT:
    		if ((ret = get_instance_data(items, req, instance, fault_code,
    					     fault_detail, fault_detail_string, auth_string))) {
    		    /* the error variables are already set here */
    		    goto bail;
    		}
    		break;
    	    case ENUMERATION_TYPE_EPR:
    	    	if ((ret = get_instance_epr(items, instance, req, fault_code,
    					  fault_detail, fault_detail_string, auth_string))) {
    		    /* the error variables are already set here */
    		    goto bail;
    		}
    		break;
    	    case ENUMERATION_TYPE_OBJECT_AND_EPR:
    	    	item = print_xml_tag(items, "wsman", "Item", NULL, NULL);
    		if ((ret = get_instance_data(item, req, instance, fault_code,
    					     fault_detail, fault_detail_string, auth_string))) {
    		    /* the error variables are already set here */
    		    goto bail;
    		}
    	    	if ((ret = get_instance_epr(item, instance, req, fault_code,
    					  fault_detail, fault_detail_string, auth_string))) {
    		    /* the error variables are already set here */
    		    goto bail;
    		}
    		break;
    	}
    }
    
    if (!instance || is_last_instance) {
    	D(D_BLABLA, "Next instance not available, ending enumeration.\n");
    	print_xml_tag(resp, "wsen", "EndOfSequence", NULL, NULL);
    } else {
    	D(D_BLABLA, "Next enumeration context: %s\n", uuid);
    }
    
    *action = strdup(NAMESPACE_ENUMERATION "/PullResponse");

 bail:
    /* release the locks of the currently used instance */
    if (instance && res->release_instance) {
    	res->release_instance(res, instance);
    }
    /* the previous instance is not used any longer, delete it */
    if (previous_instance && res->cleanup_instance) {
    	res->cleanup_instance(res, previous_instance);
    }

    if (!instance || is_last_instance) {
    	/* the instance is already cleared, so set it to NULL in the session */
    	update_session(session, uuid, NULL);
    	/* end the enumeration after the last element */
    	if (destroy_session(session)) {
    	    D(D_ERROR, "Could not release session.\n");
    	}
    }

    unlock_sessions();

    return ret;
}

int resource_release(xml_writer_t xml UNUSED, wsman_resource_t * res,
		     wsman_request_t * req, char ** action, int *fault_code UNUSED,
		     int *fault_detail UNUSED, char **fault_detail_string UNUSED, char *auth_string UNUSED) {
    enumerate_session_t *session;

    assert(res);
    
    D(D_VERBOSE, "Releasing enumeration of Resource %s, EnumerationContext: %s\n",
    	res->resource_uri, req->enumeration_context);
    
    lock_sessions();

    session = get_session(req->enumeration_context);
    if (session) {
    	/* this cleans up the instance, too */
    	if (destroy_session(session)) {
    	    D(D_ERROR, "Could not release session.\n");
    	}
    } else {
    	D(D_NOTICE, "invalid enumeration context in Release, ignoring: %s.\n",
    		req->enumeration_context ? req->enumeration_context : "(none)");
    }
    
    *action = strdup(NAMESPACE_ENUMERATION "/ReleaseResponse");

    unlock_sessions();

    return 0;
}
