#include <stdio.h>
#include <string.h>

#include <pp/base.h>
#include <pp/features.h>
#include <pp/wsman.h>
#include <pp/um.h>

#include "debug.h"
#include "wsman.h"
#include "xml_parser.h"
#include "enumerate_session.h"
#include "resource.h"
#include "requests.h"
#include "fault.h"

#define UTF_BOM_NONE	0
#define UTF_BOM_BE	1
#define UTF_BOM_LE	2

/******************************************************************************
* data                                                                        *
******************************************************************************/

char *namespace_xmlschema = NAMESPACE_XMLSCHEMA;
char *namespace_xmlscm_inst = NAMESPACE_XMLSCM_INST;
char *namespace_soap_env = NAMESPACE_SOAP_ENV;
char *namespace_xpath = NAMESPACE_XPATH;
char *namespace_management = NAMESPACE_MANAGEMENT;
char *namespace_addressing = NAMESPACE_ADDRESSING;
char *namespace_enumeration = NAMESPACE_ENUMERATION;
char *namespace_transfer = NAMESPACE_TRANSFER;

/******************************************************************************
* function definitions                                                        *
******************************************************************************/

static int parse_auth(char * auth);

#if WSMAN_FAKE_REPLY

static void test_request_handler(pp_strstream_t * response);

#endif /* WSMAN_FAKE_REPLY */

/******************************************************************************
* function implementations                                                    *
******************************************************************************/

/* main WS Management request handler */
int pp_wsman_process_request(char * auth_string,
			     char * request, size_t req_len,
			     char **response, size_t *resp_len,
			     int secure UNUSED,
			     int is_utf16 UNUSED) {
    int ret = PP_WSMAN_NO_ERROR;
    char userid[128];
    char *ep;
    
    assert(response && auth_string);
    *response = NULL;
    
    if (parse_auth(auth_string)) {
    	return PP_WSMAN_ERROR_AUTH_FAILED;
    }
    
#if WSMAN_FAKE_REPLY
    test_request_handler(response);
#else /* WSMAN_FAKE_REPLY */
    if (!request) request = "";
    
    D(D_BLABLA, "Request:\n%s\n", request);

    /* pass down auth string with just user id; strip off password */    

    snprintf(userid, sizeof(userid), "%s", auth_string);
    ep = strchr(userid, ':');
    if(ep)
        *ep = 0;
    auth_string = userid;
    ret = process_wsman_request(request, req_len, response, resp_len, auth_string);

    D(D_BLABLA, "Response:\n%s\n", *response);
    
#endif /* WSMAN_FAKE_REPLY */

    if (*response) {
    	char *header = "<?xml version=\"1.0\"?>\n";
    	size_t hdr_len = strlen(header);
    	size_t new_len = hdr_len + *resp_len;
    	char *resp_new = malloc(new_len);
    	memcpy(resp_new, header, hdr_len);
    	memcpy(&resp_new[hdr_len], *response, *resp_len);
    	free(*response);
    	*response = resp_new;
    	*resp_len = new_len;
    }
    
    return ret;
}

/* parse the login */
static int parse_auth(char * auth) {
    int ret = -1;
    char * username;
    char * password;
    int server_sid;
    int gid;
    
    D(D_BLABLA, "auth: %s\n", auth);
    
    // copy the auth string
    if (!(auth = strdup(auth))) {
    	return -1;
    }
    
    // get username and password
    username = auth;
    if ((password = strchr(auth, ':'))) {
    	*password = '\0';
    	password++;
    }
    
    // authenticate
    if (pp_um_user_authenticate(username, password, PP_UM_AUTH_NO_FLAGS, &server_sid, &gid) == PP_SUC) {
    	ret = 0;
    }

    free(auth);
    return ret;
}

/* library initialization */
int pp_wsman_init(void) {
    int ret = 0;
    
    if ((ret = uuid_init()) ||
    	(ret = initialize_resources()) ||
    	(ret = enumerate_session_init())) {
    	
    	return ret;
    }
    
    D(D_NOTICE, "WS-Management library successfully initialized.\n");
    
    return ret;
}

/* library cleanup */
void pp_wsman_cleanup(void) {
    enumerate_session_cleanup();
    cleanup_resources();
    uuid_cleanup();
}

/******************************************************************************
* functions for debugging                                                     *
******************************************************************************/

#if WSMAN_FAKE_REPLY

static char* this_test[] = {
//    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(wsman:system/2005/02/this)</wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">http://schemas.xmlsoap.org/ws/2004/09/transfer/Get</wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200</wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "                   <!-- Message ID need not be a GUID but has to be in this format -->"
    "    <wsman:OperationTimeout>PT60.000S</wsman:OperationTimeout> <!-- Optional -->"
    "</env:Header>"
    "<env:Body/>"
    "</env:Envelope>",
//    */
    
//    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(wsman:system/2005/02/thi)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/transfer/Get"
    "    </wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "     </wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "                   <!-- Message ID need not be a GUID but has to be in this format -->"
    "    <wsman:OperationTimeout>PT60.000S</wsman:OperationTimeout> <!-- Optional -->"
    "</env:Header>"
    "<env:Body/>"
    "</env:Envelope>",
//    */

    /*
    "<env:Envelope xml:lang=\"en-US\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\" xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\">"
    "  <env:Header>"
    "        <wsa:To> http:1.2.3.4/wsman?"
    "                   ResourceURI=( wsman:system/catalog/2005/02/Catalog)"
    "        </wsa:To>"
    "        <wsa:ReplyTo>"
    "             <wsa:Address env:mustUnderstand=\"true\">"
    "                    http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "              </wsa:Address>"
    "        </wsa:ReplyTo>"
    "        <wsa:Action env:mustUnderstand=\"true\">"
    "                 http://schemas.xmlsoap.org/ws/2004/09/transfer/Get"
    "         </wsa:Action>"
    "         <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">"
    "                      51200"
    "         </wsman:MaxEnvelopeSize>"
    "         <wsa:MessageID>uuid:70164a7f-a4e4-4b1c-9527-cbd32db9050c</wsa:MessageID>"
    "         <wsman:SelectorSet>"
    "                <wsman:Selector Name=\"ResourceURI\">"
    "                          wsman:root.cim/CIM_Processor"
    "                </wsman:Selector> "
    "         </wsman:SelectorSet>"
    "         <wsman:OperationTimeout>PT60.000S</wsman:OperationTimeout>"
    "  </env:Header>"
    "  <env:Body /> "
    "</env:Envelope>",
    */

    /*
    "<env:Envelope xml:lang=\"en-US\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\" xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\">"
    "  <env:Header>"
    "        <wsa:To> http:1.2.3.4/wsman?"
    "                   ResourceURI=( wsman:system/catalog/2005/02/Catalog)"
    "        </wsa:To>"
    "        <wsa:ReplyTo>"
    "             <wsa:Address env:mustUnderstand=\"true\">"
    "                    http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "              </wsa:Address>"
    "        </wsa:ReplyTo>"
    "        <wsa:Action env:mustUnderstand=\"true\">"
    "                 http://schemas.xmlsoap.org/ws/2004/09/transfer/Get"
    "         </wsa:Action>"
    "         <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">"
    "                      51200"
    "         </wsman:MaxEnvelopeSize>"
    "         <wsa:MessageID>uuid:70164a7f-a4e4-4b1c-9527-cbd32db9050c</wsa:MessageID>"
    "         <wsman:SelectorSet>"
    "                <wsman:Selector Name=\"ResourceURI\">"
    "                          http://schemas.acme.com/2005/02/samples/physicalDisk"
    "                </wsman:Selector> "
    "         </wsman:SelectorSet>"
    "         <wsman:OperationTimeout>PT60.000S</wsman:OperationTimeout>"
    "  </env:Header>"
    "  <env:Body /> "
    "</env:Envelope>",
    */

    /*
    "<env:Envelope xml:lang=\"en-US\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\" xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\">"
    "  <env:Header>"
    "        <wsa:To> http:1.2.3.4/wsman?"
    "                   ResourceURI=( wsman:system/catalog/2005/02/Catalog)"
    "        </wsa:To>"
    "        <wsa:ReplyTo>"
    "             <wsa:Address env:mustUnderstand=\"true\">"
    "                    http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "              </wsa:Address>"
    "        </wsa:ReplyTo>"
    "        <wsa:Action env:mustUnderstand=\"true\">"
    "                 http://schemas.xmlsoap.org/ws/2004/09/transfer/Get"
    "         </wsa:Action>"
    "         <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">"
    "                      51200"
    "         </wsman:MaxEnvelopeSize>"
    "         <wsa:MessageID>uuid:70164a7f-a4e4-4b1c-9527-cbd32db9050c</wsa:MessageID>"
    "         <wsman:SelectorSet>"
    "                <wsman:Selector Name=\"ResourceURI\">"
    "                          http://schemas.dmtf.org/wsman/2005/02/cimv2.9/CIM_NumericSensor"
    "                </wsman:Selector> "
    "         </wsman:SelectorSet>"
    "         <wsman:OperationTimeout>PT60.000S</wsman:OperationTimeout>"
    "  </env:Header>"
    "  <env:Body /> "
    "</env:Envelope>",
    */

    /*
    "<env:Envelope xml:lang=\"en-US\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\" xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\">"
    "  <env:Header>"
    "        <wsa:To> http:1.2.3.4/wsman?"
    "                   ResourceURI=( wsman:system/catalog/2005/02/Catalog)"
    "        </wsa:To>"
    "        <wsa:ReplyTo>"
    "             <wsa:Address env:mustUnderstand=\"true\">"
    "                    http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "              </wsa:Address>"
    "        </wsa:ReplyTo>"
    "        <wsa:Action env:mustUnderstand=\"true\">"
    "                 http://schemas.xmlsoap.org/ws/2004/09/transfer/Get"
    "         </wsa:Action>"
    "         <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">"
    "                      51200"
    "         </wsman:MaxEnvelopeSize>"
    "         <wsa:MessageID>uuid:70164a7f-a4e4-4b1c-9527-cbd32db9050c</wsa:MessageID>"
    "         <wsman:SelectorSet>"
    "                <wsman:Selector Name=\"ResourceURI\">"
    "                          blablabla"
    "                </wsman:Selector> "
    "         </wsman:SelectorSet>"
    "         <wsman:OperationTimeout>PT60.000S</wsman:OperationTimeout>"
    "  </env:Header>"
    "  <env:Body /> "
    "</env:Envelope>",
    */

    /* enumeration test */
    /*
    "<env:Envelope xml:lang=\"en-US\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\" xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\">"
    "  <env:Header>"
    "        <wsa:To> http:1.2.3.4/wsman?"
    "                   ResourceURI=( wsman:system/catalog/2005/02/Catalog)"
    "        </wsa:To>"
    "        <wsa:ReplyTo>"
    "             <wsa:Address env:mustUnderstand=\"true\">"
    "                    http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "              </wsa:Address>"
    "        </wsa:ReplyTo>"
    "        <wsa:Action env:mustUnderstand=\"true\">"
    "                 http://schemas.xmlsoap.org/ws/2004/09/enumeration/Enumerate"
    "         </wsa:Action>"
    "         <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">"
    "                      51200"
    "         </wsman:MaxEnvelopeSize>"
    "         <wsa:MessageID>uuid:70164a7f-a4e4-4b1c-9527-cbd32db9050c</wsa:MessageID>"
    "  </env:Header>"
    "  <env:Body /> "
    "</env:Envelope>",
    */

    /*
    "<env:Envelope xml:lang=\"en-US\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsen=\"http://schemas.xmlsoap.org/ws/2004/09/enumeration\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\" xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\">"
    "  <env:Header>"
    "        <wsa:To> http:1.2.3.4/wsman?"
    "                   ResourceURI=( wsman:system/catalog/2005/02/Catalog)"
    "        </wsa:To>"
    "        <wsa:ReplyTo>"
    "             <wsa:Address env:mustUnderstand=\"true\">"
    "                    http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "              </wsa:Address>"
    "        </wsa:ReplyTo>"
    "        <wsa:Action env:mustUnderstand=\"true\">"
    "                 http://schemas.xmlsoap.org/ws/2004/09/enumeration/Pull"
    "         </wsa:Action>"
    "         <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">"
    "                      51200"
    "         </wsman:MaxEnvelopeSize>"
    "         <wsa:MessageID>uuid:70164a7f-a4e4-4b1c-9527-cbd32db9050c</wsa:MessageID>"
    "  </env:Header>"
    "  <env:Body>"
    "    <wsen:Pull>"
    "      <wsen:EnumerationContext>"
    "        uuid:70164a7f-a4e4-4b1c-9527-cbd32db9050c"
    "      </wsen:EnumerationContext>"
    "    </wsen:Pull>"
    "  </env:Body>"
    "</env:Envelope>",
    */

    /*
    "<env:Envelope xml:lang=\"en-US\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\" xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\">"
    "  <env:Header>"
    "        <wsa:To> http:1.2.3.4/wsman?"
    "                   ResourceURI=( wsman:system/catalog/2005/02/Catalog)"
    "        </wsa:To>"
    "        <wsa:ReplyTo>"
    "             <wsa:Address env:mustUnderstand=\"true\">"
    "                    http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "              </wsa:Address>"
    "        </wsa:ReplyTo>"
    "        <wsa:Action env:mustUnderstand=\"true\">"
    "                 http://schemas.xmlsoap.org/ws/2004/09/enumeration/Release"
    "         </wsa:Action>"
    "         <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">"
    "                      51200"
    "         </wsman:MaxEnvelopeSize>"
    "         <wsa:MessageID>uuid:70164a7f-a4e4-4b1c-9527-cbd32db9050c</wsa:MessageID>"
    "  </env:Header>"
    "  <env:Body /> "
    "</env:Envelope>",
    */

    /*
    "bla",
    */

    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(http://schemas.dmtf.org/wsman/2005/02/cimv2.9/CIM_ComputerSystem)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/transfer/Get"
    "    </wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "     </wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "    <wsman:SelectorSet>"
    "      <wsman:Selector Name=\"CreationClassName\">ComputerSystem</wsman:Selector>"
    "      <wsman:Selector Name=\"Name\">IPMI Controller 32</wsman:Selector>"
    "    </wsman:SelectorSet>"
    "</env:Header>"
    "<env:Body>"
    "</env:Body>"
    "</env:Envelope>",
    */

    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsen=\"http://schemas.xmlsoap.org/ws/2004/09/enumeration\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(http://schemas.dmtf.org/wsman/2005/02/cimv2.9/CIM_ComputerSystem)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/enumeration/Enumerate"
    "    </wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "     </wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "</env:Header>"
    "<env:Body>"
    "</env:Body>"
    "</env:Envelope>",
    */

    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsen=\"http://schemas.xmlsoap.org/ws/2004/09/enumeration\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(http://schemas.dmtf.org/wsman/2005/02/cimv2.9/CIM_ComputerSystem)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/enumeration/Pull"
    "    </wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "     </wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "</env:Header>"
    "  <env:Body>"
    "    <wsen:Pull>"
    "      <wsen:EnumerationContext>"
    "        uuid:70164a7f-a4e4-4b1c-9527-cbd32db9050c"
    "      </wsen:EnumerationContext>"
    "    </wsen:Pull>"
    "  </env:Body>"
    "</env:Envelope>",
    */

    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(http://schemas.dmtf.org/wsman/2005/02/cimv2.9/CIM_ComputerSystem)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/enumeration/Release"
    "    </wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "     </wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "</env:Header>"
    "  <env:Body>"
    "  </env:Body>"
    "</env:Envelope>",
    */

    /* CIM_NumericSensor */
    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(http://schemas.dmtf.org/wsman/2005/02/cimv2.9/CIM_NumericSensor)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/transfer/Get"
    "    </wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "     </wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "    <wsman:SelectorSet>"
    "      <wsman:Selector Name=\"Name\">Dummy Sensor 1 (10.0.32)</wsman:Selector>"
    "    </wsman:SelectorSet>"
    "</env:Header>"
    "<env:Body>"
    "</env:Body>"
    "</env:Envelope>",
    */
    
    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(http://schemas.dmtf.org/wsman/2005/02/cimv2.9/CIM_NumericSensor)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/transfer/Get"
    "    </wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "     </wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "    <wsman:SelectorSet>"
    "      <wsman:Selector Name=\"Name\">blabla</wsman:Selector>"
    "    </wsman:SelectorSet>"
    "</env:Header>"
    "<env:Body>"
    "</env:Body>"
    "</env:Envelope>",
    */
    
    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(http://schemas.dmtf.org/wsman/2005/02/cimv2.9/CIM_NumericSensor)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/transfer/Get"
    "    </wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "     </wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "    <wsman:SelectorSet>"
    "      <wsman:Selector Name=\"CreationClassName\">NumericSensor</wsman:Selector>"
    "      <wsman:Selector Name=\"SystemCreationClassName\">ComputerSystem</wsman:Selector>"
    "      <wsman:Selector Name=\"SystemName\">IPMI Controller 32</wsman:Selector>"
    "      <wsman:Selector Name=\"DeviceID\">10.0.32</wsman:Selector>"
    "    </wsman:SelectorSet>"
    "</env:Header>"
    "<env:Body>"
    "</env:Body>"
    "</env:Envelope>",
    */
    
    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(http://schemas.dmtf.org/wsman/2005/02/cimv2.9/CIM_NumericSensor)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/transfer/Put"
    "    </wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "     </wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "    <wsman:SelectorSet>"
    "      <wsman:Selector Name=\"Name\">Dummy Sensor 1 (10.0.32)</wsman:Selector>"
    "    </wsman:SelectorSet>"
    "</env:Header>"
    "<env:Body>"
    "  <p:CIM_NumericSensor xmlns:p=\"http://schemas.dmtf.org/wsman/2005/02/cimv2.9\">"
    "    <p:LowerThresholdNonCritical>30</p:LowerThresholdNonCritical>"
    "  </p:CIM_NumericSensor>"
    "</env:Body>"
    "</env:Envelope>",
    */

    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(http://schemas.dmtf.org/wsman/2005/02/cimv2.9/CIM_NumericSensor)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/transfer/Put"
    "    </wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "     </wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "    <wsman:SelectorSet>"
    "      <wsman:Selector Name=\"Name\">Dummy Sensor 1 (10.0.32)</wsman:Selector>"
    "    </wsman:SelectorSet>"
    "</env:Header>"
    "<env:Body>"
    "  <CIM_NumericSensor xmlns=\"http://schemas.dmtf.org/wsman/2005/02/cimv2.9\">"
    "    <LowerThresholdNonCritical>35</LowerThresholdNonCritical>"
    "    <LowerThresholdCritical>25</LowerThresholdCritical>"
    "    <LowerThresholdFatal>20</LowerThresholdFatal>"
    "  </CIM_NumericSensor>"
    "</env:Body>"
    "</env:Envelope>",
    */

    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(http://schemas.dmtf.org/wsman/2005/02/cimv2.9/CIM_NumericSensor)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/transfer/Put"
    "    </wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "     </wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "    <wsman:SelectorSet>"
    "      <wsman:Selector Name=\"Name\">Dummy Sensor 1 (10.0.32)</wsman:Selector>"
    "    </wsman:SelectorSet>"
    "</env:Header>"
    "<env:Body>"
    "  <p:CIM_NumericSensor xmlns:p=\"http://schemas.dmtf.org/wsman/2005/02/cimv2.9\">"
    "    <p:BlablaThresholdNonCritical>40</p:BlablaThresholdNonCritical>"
    "  </p:CIM_NumericSensor>"
    "</env:Body>"
    "</env:Envelope>",
    */

    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(http://schemas.dmtf.org/wsman/2005/02/cimv2.9/CIM_NumericSensor)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/transfer/Get"
    "    </wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "     </wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "    <wsman:SelectorSet>"
    "      <wsman:Selector Name=\"CreationClassName\">NumericSensor</wsman:Selector>"
    "      <wsman:Selector Name=\"SystemCreationClassName\">ComputerSystem</wsman:Selector>"
    "      <wsman:Selector Name=\"SystemName\">IPMI Controller 34</wsman:Selector>"
    "      <wsman:Selector Name=\"DeviceID\">10.0.32</wsman:Selector>"
    "    </wsman:SelectorSet>"
    "</env:Header>"
    "<env:Body>"
    "</env:Body>"
    "</env:Envelope>",
    */
    
    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(http://schemas.dmtf.org/wsman/2005/02/cimv2.9/CIM_NumericSensor)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/enumeration/Enumerate"
    "    </wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "     </wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "</env:Header>"
    "<env:Body>"
    "</env:Body>"
    "</env:Envelope>",
    */
    
    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsen=\"http://schemas.xmlsoap.org/ws/2004/09/enumeration\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(http://schemas.dmtf.org/wsman/2005/02/cimv2.9/CIM_NumericSensor)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/enumeration/Pull"
    "    </wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "     </wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "</env:Header>"
    "<env:Body>"
    "    <wsen:Pull>"
    "      <wsen:EnumerationContext>"
    "        uuid:70164a7f-a4e4-4b1c-9527-cbd32db9050c"
    "      </wsen:EnumerationContext>"
    "    </wsen:Pull>"
    "</env:Body>"
    "</env:Envelope>",
    */
    
    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsen=\"http://schemas.xmlsoap.org/ws/2004/09/enumeration\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(wsman:system/catalog/2005/02/Catalog)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/transfer/Get"
    "   </wsa:Action>"
    "   <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "   </wsman:MaxEnvelopeSize>"
    "   <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "   <wsman:SelectorSet>"
    "      <wsman:Selector Name=\"ResourceURI\">wsman:root.cim/CIM_Processor</wsman:Selector>"
    "   </wsman:SelectorSet>"
    "</env:Header>"
    "<env:Body/>"
    "</env:Envelope>",
    */
    
    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsen=\"http://schemas.xmlsoap.org/ws/2004/09/enumeration\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(wsman:root.cim/CIM_Processor)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/transfer/Get"
    "    </wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "     </wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "</env:Header>"
    "<env:Body/>"
    "</env:Envelope>",
    */
    
    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsen=\"http://schemas.xmlsoap.org/ws/2004/09/enumeration\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(wsman:root.cim/CIM_Processor)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/transfer/Get"
    "    </wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "     </wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "   <wsman:SelectorSet>"
    "      <wsman:Selector Name=\"ResourceURI\">wsman:root.cim/CIM_Processor</wsman:Selector>"
    "   </wsman:SelectorSet>"
    "</env:Header>"
    "<env:Body/>"
    "</env:Envelope>",
    */
    
    /* testing malformed XML */
    /*
    "<env:Ennvelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(wsman:system/2005/02/this)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/transfer/Get"
    "    </wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "     </wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "                   <!-- Message ID need not be a GUID but has to be in this format -->"
    "    <wsman:OperationTimeout>PT60.000S</wsman:OperationTimeout> <!-- Optional -->"
    "</env:Header>"
    "<env:Body/>"
    "</env:Envelope>",
    */

    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<ennv:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(wsman:system/2005/02/this)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/transfer/Get"
    "    </wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "     </wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "                   <!-- Message ID need not be a GUID but has to be in this format -->"
    "    <wsman:OperationTimeout>PT60.000S</wsman:OperationTimeout> <!-- Optional -->"
    "</env:Header>"
    "<env:Body/>"
    "</env:Envelope>",
    */

    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI = (wsman:system/2005/02/this)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/transfer/Get"
    "    </wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "     </wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "                   <!-- Message ID need not be a GUID but has to be in this format -->"
    "    <wsman:OperationTimeout>PT60.000S</wsman:OperationTimeout> <!-- Optional -->"
    "</env:Header>"
    "<env:Body/>"
    "</env:Envelope>",
    */

    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(wsman:system/2005/02/this)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/transfer/Get"
    "    </wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "     </wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "                   <!-- Message ID need not be a GUID but has to be in this format"
    "    <wsman:OperationTimeout>PT60.000S</wsman:OperationTimeout> <!-- Optional -->"
    "</env:Header>"
    "<env:Body/>"
    "</env:Envelope>",
    */

    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(wsman:system/catalog/2005/02/Catalog)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/transfer/Get"
    "    </wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "     </wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "                   <!-- Message ID need not be a GUID but has to be in this format -->"
    "    <wsman:OperationTimeout>PT60.000S</wsman:OperationTimeout> <!-- Optional -->"
    "    <wsman:SelectorSet>"
    "      <wsman:Selector>wsman:root.cim/CIM_Processor</wsman:Selector>"
    "    </wsman:SelectorSet>"
    "</env:Header>"
    "<env:Body/>"
    "</env:Envelope>",
    */

    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(wsman:system/catalog/2005/02/Catalog)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/transfer/Get"
    "    </wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "     </wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "                   <!-- Message ID need not be a GUID but has to be in this format -->"
    "    <wsman:OperationTimeout>PT60.000S</wsman:OperationTimeout> <!-- Optional -->"
    "    <wsman:SelectorSet>"
    "      <wsman:Selector Name=\"ResourceURI\"></wsman:Selector>"
    "    </wsman:SelectorSet>"
    "</env:Header>"
    "<env:Body/>"
    "</env:Envelope>",
    */

    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(wsman:system/catalog/2005/02/Catalog)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\">"
    "         http://schemas.xmlsoap.org/ws/2004/09/transfer/Get"
    "    </wsa:Action>"
    "    <wsman:MaxEnvelopeSize env:mustUnderstand=\"true\">51200"
    "     </wsman:MaxEnvelopeSize>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "                   <!-- Message ID need not be a GUID but has to be in this format -->"
    "    <wsman:OperationTimeout>PT60.000S</wsman:OperationTimeout> <!-- Optional -->"
    "    <wsman:SelectorSet"
    "      <wsman:Selector Name=\"ResourceURI\">wsman:root.cim/CIM_Processor</wsman:Selector>"
    "    </wsman:SelectorSet>"
    "</env:Header>"
    "<env:Body/>"
    "</env:Envelope>",
    */

    /*
    "<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsman=\"http://schemas.xmlsoap.org/ws/2005/02/management\">"
    "<env:Header>"
    "   <wsa:To>http://1.2.3.4/wsman?ResourceURI=(wsman:system/catalog/2005/02/Catalog)"
    "   </wsa:To>"
    "   <wsa:ReplyTo>"
    "       <wsa:Address env:mustUnderstand=\"true\">"
    "          http://schemas.xmlsoap.org/ws/2004/08/addressing/role/anonymous"
    "       </wsa:Address>"
    "   </wsa:ReplyTo>"
    "   <wsa:Action env:mustUnderstand=\"true\"/>"
    "    <wsa:MessageID>uuid:E9E02CCC-5EB2-4A46-A753-B58455D9F4B1</wsa:MessageID>"
    "                   <!-- Message ID need not be a GUID but has to be in this format -->"
    "    <wsman:OperationTimeout>PT60.000S</wsman:OperationTimeout> <!-- Optional -->"
    "    <wsman:SelectorSet>"
    "      <wsman:Selector Name=\"ResourceURI\">wsman:root.cim/CIM_Processor</wsman:Selector>"
    "    </wsman:SelectorSet>"
    "</env:Header>"
    "<env:Body/>"
    "</env:Envelope>",
    */

};

static void test_request_handler(pp_strstream_t * response) {
    size_t i;
    int enum_context = 0;
    
    pp_strappendf(response, "<root>\n");
    
    for (i = 0; i < sizeof(this_test) / sizeof(this_test[0]); i++) {
    	pp_strstream_t stream = PP_STRSTREAM_INITIALIZER;
    	char *s = strdup(this_test[i]);
    	
    	pp_strappend(&stream, "");

    	// hack: set the uuid in the pull request
    	if (!enum_context && strstr(s, "EnumerationContext")) {
#if 0
    	    char *c = cim_numericsensor_resource.get_first_instance()->uuid;
    	    D(D_BLABLA, "setting enumeration context to %s\n", c);
    	    char *uuid = strstr(strstr(s, "EnumerationContext"), "uuid:");
    	    memcpy(uuid, c, UUID_LENGTH - 1);
#endif
    	    enum_context = 1;
    	}
    	
    	process_wsman_request(&stream, s, strlen(s), NULL);
    	pp_strappend(&stream, "\n");
    	
    	pp_strappend(response, stream.buf);
    	pp_strstream_free(&stream);
    	    	
    	free(s);
    }
    
    pp_strappendf(response, "</root>\n");
}

#endif /* WSMAN_FAKE_REPLY */
