/**
 * ahb_dma.c
 *
 * KIRA100 AHB DMA kernel module
 *
 * (c) 2006 Peppercon AG, Danny Baumann <daba@raritan.com>
 */

/* 
The usage of this module is as follows:

Drivers can request a DMA channel by calling apb_dma_request_channel. If the driver
got the channel, it can add an action item to the DMA:
            
            apb_dma_parm_t parm;
            parm.src=SRC_ADDR;                      // given source phy addr
            parm.dest=DEST_ADDR;                    // given dest phy addr
            parm.width=APBDMA_WIDTH_32BIT;          // width 32 bit
            parm.sctl=APBDMA_CTL_INC2;              // source increase 2 bytes
            parm.dctl=APBDMA_CTL_FIX;               // dest fix increase
            parm.stype=APBDMA_TYPE_APB;             // indicate source is APB device
            parm.dtype=APBDMA_TYPE_AHB;             // indicate source is AHB device
            parm.size=200;                          // transfer count
            parm.burst=0;                           // use bursts?
            parm.irq=1;                             // trigger irq?
            
            apb_dma_channel_add(priv,&parm);        // call add function
            
The actual transfer is started by calling apb_dma_channel_start. 
*/

#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/wait.h>
#include <asm/types.h>
#include <asm/io.h>
#include <asm/arch/apb_dma.h>

/* local function prototypes */
static int apb_dma_init(void);
static void apb_dma_cleanup(void);
static int apb_dma_get_inc(int, unsigned int);

static apb_dma_data_t dma_data;

static int apb_dma_get_inc(int inc,unsigned int burst)
{
    switch (inc) {
        case APBDMA_CTL_FIX:
            return 0;
        case APBDMA_CTL_INC1:
            return 1;
        case APBDMA_CTL_INC2:
            return 2;
        case APBDMA_CTL_INC4:
            if(burst)
                return 1;
            else
                return 3;
        case APBDMA_CTL_INC8:
            if(burst)
                return 2;
            break;
        case APBDMA_CTL_INC16:
            if(burst)
                return 3;
            break;
        case APBDMA_CTL_DEC1:
            return 5;
        case APBDMA_CTL_DEC2:
            return 6;
        case APBDMA_CTL_DEC4:
            return 7;
        default:
            break;
    }
    return -1;
}

static irqreturn_t apb_dma_interrupt_handler(int irq, void *_data, struct pt_regs *dummy)
{
    apb_dma_data_t *data = _data;
    uint32_t stat, i;

    for (i = 0; i < APB_DMA_MAX_CHANNELS; i++) {
        if (data->data[i]) {
            stat = data->data[i]->base[APBDMA_CHANNEL_CMD];
            if (stat & 0x12) { 
                //interrupt occured - either FinIntSts or ErrIntSts was set
                if (stat & 0x2) data->data[i]->status = INT_TRIGGER;
                else data->data[i]->status = INT_ERROR;
                if (data->irq_handler[i]) data->irq_handler[i](data->isr_data[i]);
                data->data[i]->base[APBDMA_CHANNEL_CMD] &= ~(0x12);
                /* mark channel as inactive */
                data->active_channels &= ~(1 << i);
            }
        }
    }
    
    return IRQ_HANDLED;
}

int __init apb_dma_init(void)
{
    int r, ret;

    memset(&dma_data, 0, sizeof(apb_dma_data_t));

    /* set base address */
    dma_data.base = (u32*)(CPE_APBDMA_VA_BASE);

    /* request irq */
    set_irq_type(IRQ_CPE_APB_DMA, IRQT_HIGH);
    if ((r = request_irq(IRQ_CPE_APB_DMA, apb_dma_interrupt_handler, SA_INTERRUPT, "apb_dma", &dma_data)) == 0) {
        dma_data.irq_requested = 1;
    } else {
        ret = -EIO;
        goto fail;
    }
    
    /* all channels are unused at the beginning */
    dma_data.active_channels = 0;
    
    return 0;
fail:
    return ret;
}

static void apb_dma_cleanup(void)
{
    if (dma_data.irq_requested) free_irq(IRQ_CPE_APB_DMA, &dma_data);
}

int apb_dma_request_channel(apb_dma_irq_handler_t irq_handler, void *isr_data, int channel, apb_dma_channel_data_t **channel_data)
{
    apb_dma_channel_data_t *data;
    
    if ((channel < APB_DMA_MAX_CHANNELS) || (!dma_data.channel_used[channel])) {
        if ((*channel_data = (apb_dma_channel_data_t *)kmalloc(GFP_KERNEL, sizeof(apb_dma_channel_data_t))) == NULL) {
            return -ENOMEM;
        }
        data = *channel_data;
        dma_data.data[channel] = data;
        memset(data, 0, sizeof(apb_dma_channel_data_t));
        data->base = (u32*)(((uint32_t)dma_data.base)+0x80+(channel*0x10));
        data->channel = channel;
        
        dma_data.channel_used[channel] = 1;
        dma_data.irq_handler[channel] = irq_handler;
        dma_data.isr_data[channel] = isr_data;
    } else {
        /* the requested DMA channel is used */
        return -ENODEV;
    }
    return 0;
}

void apb_dma_release_channel(apb_dma_channel_data_t **channel_data)
{
    int channel;
    apb_dma_channel_data_t *data; 
    
    if (channel_data && *channel_data) {
        data = *channel_data;
        channel = data->channel;
        if (channel < APB_DMA_MAX_CHANNELS) {
            dma_data.channel_used[channel] = 0;
            
            /* release memory */
            kfree(data);
            data = NULL;
        }
    }
}

void apb_dma_channel_add(apb_dma_channel_data_t *priv, apb_dma_parm_t *parm)
{
    unsigned int    val;
    unsigned int    sinc,dinc;
    
    priv->base[APBDMA_CHANNEL_SRC] = parm->src;
    priv->base[APBDMA_CHANNEL_DEST] = parm->dest;
    priv->base[APBDMA_CHANNEL_CYC] = parm->size;
    
    if ((sinc = apb_dma_get_inc(parm->sctl,parm->burst)) < 0) {
        printk("Fail to add APB dma src incremental %d with burst %d\n",parm->sctl,parm->burst);
        return;
    }
    if ((dinc = apb_dma_get_inc(parm->dctl,parm->burst)) < 0) {
        printk("Fail to add APB dma dest incremental %d with burst %d\n",parm->dctl,parm->burst);
        return;
    }
    
    val= ((parm->width & 3) << 20) |
         ((priv->hw_handshake_num & 0xf) << 16) |
         ((dinc & 0x7) << 12) |
         ((sinc & 0x7) << 8) |
         ((parm->dtype) ? (1 << 7) : 0) |
         ((parm->stype) ? (1 << 6) : 0) |
         ((parm->irq) ? 0x24 : 0) |
         ((parm->burst) ? (1 << 3) : 0);
    
    priv->base[APBDMA_CHANNEL_CMD] = val;
}

void apb_dma_channel_start(apb_dma_channel_data_t *priv)
{
    /* mark channel as active */
    dma_data.active_channels |= (1 << priv->channel);
   
    priv->base[APBDMA_CHANNEL_CMD] |= 0x1;
}

EXPORT_SYMBOL(apb_dma_request_channel);
EXPORT_SYMBOL(apb_dma_release_channel);
EXPORT_SYMBOL(apb_dma_channel_add);
EXPORT_SYMBOL(apb_dma_channel_start);

module_init(apb_dma_init);
module_exit(apb_dma_cleanup);
