#include <linux/module.h>
#include <linux/sched.h>
#include <linux/spinlock.h>
#include <linux/time.h>
#include <linux/types.h>

#include <asm/io.h>
#include <asm/arch/platform-a320/peppercon.h>
#include <asm/arch/platform-a320/kira100.h>

static volatile u32 * pmu_regs  = (volatile u32*)CPE_PMU_VA_BASE;

spinlock_t pp_kira_gpio_lock = SPIN_LOCK_UNLOCKED;
spinlock_t pp_kira_pmu_lock = SPIN_LOCK_UNLOCKED;

EXPORT_SYMBOL(pp_kira_gpio_lock);

#ifdef PP_FEAT_SYSOK_BLINKING

# define PP_GPIO_STATUS_LED	3
static volatile u32 * gpio_regs = (volatile u32*)CPE_GPIO_VA_BASE;
static struct timer_list status_led_timer;
static unsigned char status_led_state;

static void status_led_toggle(unsigned long data);

void pp_status_led_blinking(unsigned char blinking);
EXPORT_SYMBOL(pp_status_led_blinking);

#endif /* PP_FEAT_SYSOK_BLINKING */


int
pp_time_init(void)
{   
#ifdef PP_FEAT_SYSOK_BLINKING
    unsigned long flags;

    spin_lock_irqsave(&pp_kira_gpio_lock, flags);
    gpio_regs[KIRA_GPIO_PIN_DIR] |= (1 << PP_GPIO_STATUS_LED);
    spin_unlock_irqrestore(&pp_kira_gpio_lock, flags);
    
    pp_status_led_blinking(1);
#endif
    return 0;
}

#ifdef PP_FEAT_SYSOK_BLINKING
void
pp_status_led_blinking(unsigned char blinking)
{
    del_timer_sync(&status_led_timer);
        
    if (blinking) {
	init_timer(&status_led_timer);
	status_led_timer.expires = jiffies + HZ/10;
	status_led_timer.data = 0;
	status_led_timer.function = status_led_toggle;
	add_timer(&status_led_timer);
    } else {
	unsigned long flags;
	    
	// if not blinking, turn on unconditioally
	status_led_state = 1;

	spin_lock_irqsave(&pp_kira_gpio_lock, flags);
	gpio_regs[KIRA_GPIO_DATA_OUT] &= ~(1 << PP_GPIO_STATUS_LED);
	spin_unlock_irqrestore(&pp_kira_gpio_lock, flags);
    }
}

static void
status_led_toggle(unsigned long data)
{
    unsigned long flags;
	
    spin_lock_irqsave(&pp_kira_gpio_lock, flags);
    if (status_led_state) {
	gpio_regs[KIRA_GPIO_DATA_OUT] |= (1 << PP_GPIO_STATUS_LED);
    } else {
	gpio_regs[KIRA_GPIO_DATA_OUT] &= ~(1 << PP_GPIO_STATUS_LED);
    }
    spin_unlock_irqrestore(&pp_kira_gpio_lock, flags);

    status_led_state = (status_led_state) ? 0 : 1;
    status_led_timer.expires = jiffies + HZ/10;
    add_timer(&status_led_timer);
}
#endif


/**
 * function which returns the revision of KIRA100
 */

uint32_t
pp_kira_get_revision(void)
{
    /* we can determine the KIRA revision by two values:
       the VSC revision and the minor revision
       the table for these values is as follows:

       VSC rev      minor rev       KIRA rev
       2.0          0               1.0
       2.0          1               1.1
       2.1          0               2.0
    */
    uint32_t vsc_rev = *((uint32_t*)CPE_VSCREG_VA_BASE) & 0xff;
    uint32_t minor_rev = (pp_kira_get_pmu_reg(KIRA_PMU_REG_MINORREV) 
                          & KIRA_PMU_REG_MINORREV_MINORREV) ? 1 : 0;

    if ((vsc_rev == 0x12) && (minor_rev == 1)) {
	return KIRA_MAKEMAJORMINOR(2, 0);
    } else if ((vsc_rev == 0x02) && (minor_rev == 1)) {
        return KIRA_MAKEMAJORMINOR(1, 1);
    } else {
        return KIRA_MAKEMAJORMINOR(1, 0);
    }
}

EXPORT_SYMBOL(pp_kira_get_revision);

/**
 * functions to access KIRA PMU registers
 */

uint32_t
pp_kira_get_pmu_reg(int reg)
{
    u_long flags;
    uint32_t ret;
    
    spin_lock_irqsave(pp_kira_pmu_lock, flags);
    ret = pmu_regs[reg];
    spin_unlock_irqrestore(pp_kira_pmu_lock, flags);

    return ret;
}

void
pp_kira_set_pmu_reg(int reg, uint32_t val)
{
    u_long flags;

    spin_lock_irqsave(pp_kira_pmu_lock, flags);
    pmu_regs[reg] = val;
    spin_unlock_irqrestore(pp_kira_pmu_lock, flags);
}

uint32_t
pp_kira_get_pmu_reg_bits(int reg, uint32_t mask)
{
    u_long flags;
    u_char ret;

    spin_lock_irqsave(pp_kira_pmu_lock, flags);
    ret = pmu_regs[reg] & mask;
    spin_unlock_irqrestore(pp_kira_pmu_lock, flags);

    return ret;
}

void
pp_kira_set_pmu_reg_bits(int reg, uint32_t set_mask, uint32_t clr_mask)
{
    u_long flags;
    
    spin_lock_irqsave(pp_kira_pmu_lock, flags);
    pmu_regs[reg] = (pmu_regs[reg] & ~clr_mask) | set_mask;
    spin_unlock_irqrestore(pp_kira_pmu_lock, flags);
}

EXPORT_SYMBOL(pp_kira_get_pmu_reg);
EXPORT_SYMBOL(pp_kira_set_pmu_reg);
EXPORT_SYMBOL(pp_kira_get_pmu_reg_bits);
EXPORT_SYMBOL(pp_kira_set_pmu_reg_bits);
