/*
 * FILE NAME: ppc405_pcimcs.c
 *
 * BRIEF MODULE DESCRIPTION:
 * machine check save pci access and configuration routines.
 * These routines may also be used in case a certain PCI
 * resource is physically not present. The according functions
 * will return an error code.
 *
 * Author: Peppercon AG
 *         Thomas Breitfeld <thomas@peppercon.de>
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/module.h>
#include <asm/io.h>
#include <asm/system.h>
#include <asm/machdep.h>
#include <syslib/ppc405_pcimcs.h>
#include <linux/init.h>
#include <asm/pci-bridge.h>

/*
 * NOTE:
 *
 *   after a data machine check the NIP is set to the next instruction in
 *   stream. Normally it should be the "eieio" instruction following our
 *   I/O operation. But there are cases where this isn't true, e.g. if an
 *   interupt occured. In this case the NIP is set to the first
 *   instruction of the interrupt handler. This prevents the execution of
 *   our exception handler. Solution: disable interrupts.
 */
u8 mcs_in_8(volatile u8 * addr, int * error)
{
	u_long flags;
	u32 err;
	u32 data;

	local_irq_save(flags);
        __asm__ __volatile__(
			     "       lbz%U2%X2 %0,%2\n"
			     "1:     eieio\n"
			     "       li %1,0\n"
			     "2:\n"
			     ".section .fixup,\"ax\"\n"
			     "3:     li %1,-1\n"
			     "       b 2b\n"
			     ".section __ex_table,\"a\"\n"
			     "       .align 2\n"
			     "       .long 1b,3b\n"
			     ".text"
			     : "=r" (data), "=r" (err)
			     : "m" (*addr) );
	local_irq_restore(flags);
	*error = err;
	return data;
}

u16 mcs_in_le16(volatile u16 * addr, int * error)
{
	u_long flags;
	u32 err;
	u32 data;

	local_irq_save(flags);
        __asm__ __volatile__(
			     "       lhbrx %0,0,%2\n"
			     "1:     eieio\n"
			     "       li %1,0\n"
			     "2:\n"
			     ".section .fixup,\"ax\"\n"
			     "3:     li %1,-1\n"
			     "       b 2b\n"
			     ".section __ex_table,\"a\"\n"
			     "       .align 2\n"
			     "       .long 1b,3b\n"
			     ".text"
			     : "=r" (data), "=r" (err)
			     : "r" (addr), "m" (*addr) );
	local_irq_restore(flags);
	*error = err;
	return data;
}

u32 mcs_in_le32(volatile u32 * addr, int * error)
{
	u_long flags;
	u32 err;
	u32 data;

	local_irq_save(flags);
	__asm__ __volatile__(
			     "       lwbrx %0,0,%2\n"
			     "1:     eieio\n"
			     "       li %1,0\n"
			     "2:\n"
			     ".section .fixup,\"ax\"\n"
			     "3:     li %1,-1\n"
			     "       b 2b\n"
			     ".section __ex_table,\"a\"\n"
			     "       .align 2\n"
			     "       .long 1b,3b\n"
			     ".text"
			     : "=r" (data), "=r" (err)
			     : "r" (addr), "m" (*addr) );
	local_irq_restore(flags);
	*error = err;
	return data;
}

void mcs_out_8(u32 data, volatile u8 * addr, int * error)
{
	u_long flags;
	u32 err;

	local_irq_save(flags);
        __asm__ __volatile__(
			     "       stb%U0%X0 %2,%0\n"
			     "1:     eieio\n"
			     "       li %1,0\n"
			     "2:\n"
			     ".section .fixup,\"ax\"\n"
			     "3:     li %1,-1\n"
			     "       b 2b\n"
			     ".section __ex_table,\"a\"\n"
			     "       .align 2\n"
			     "       .long 1b,3b\n"
			     ".text"
			     : "=m" (*addr), "=r" (err)
			     : "r" (data) );
	local_irq_restore(flags);
	*error = err;
}

void mcs_out_le16(u32 data, volatile u16 * addr, int * error)
{
	u_long flags;
	u32 err;

	local_irq_save(flags);
        __asm__ __volatile__(
			     "       sthbrx %2,0,%3\n"
			     "1:     eieio\n"
			     "       li %1,0\n"
			     "2:\n"
			     ".section .fixup,\"ax\"\n"
			     "3:     li %1,-1\n"
			     "       b 2b\n"
			     ".section __ex_table,\"a\"\n"
			     "       .align 2\n"
			     "       .long 1b,3b\n"
			     ".text"
			     : "=m" (*addr), "=r" (err)
			     : "r" (data), "r" (addr) );
	local_irq_restore(flags);
	*error = err;
}

void mcs_out_le32(u32 data, volatile u32 * addr, int * error)
{
	u_long flags;
	u32 err;

	local_irq_save(flags);
        __asm__ __volatile__(
			     "       stwbrx %2,0,%3\n"
			     "1:     eieio\n"
			     "       li %1,0\n"
			     "2:\n"
			     ".section .fixup,\"ax\"\n"
			     "3:     li %1,-1\n"
			     "       b 2b\n"
			     ".section __ex_table,\"a\"\n"
			     "       .align 2\n"
			     "       .long 1b,3b\n"
			     ".text"
			     : "=m" (*addr), "=r" (err)
			     : "r" (data), "r" (addr) );
	local_irq_restore(flags);
	*error = err;
}

/*
 *  config space access with caught
 *  Machine Check exception. This is necessary
 *  as we try to detect our VGA chip with an
 *  config cycle, means also in case it is switched
 *  off. The advanced error reporting leads in this
 *  case to a machine check exception that needs
 *  to be handled
 *  mcs - means machine check save
 */
#define MCS_read(val, addr, type, op, errp) *val = op((type)(addr), errp)
#define MCS_write(val, addr, type, op, errp) op((val), (type*)(addr), errp)
 
#define MCS_PCI_CONFIG_OP(rw, size, type, op, mask)                        \
int                                                                        \
mcs_pci_##rw##_config_##size(struct pci_dev *dev, int offset, type value)  \
{                                                                          \
  u32 error;                                                               \
  struct pci_controller* hose = dev->sysdata;                              \
  out_be32(hose->cfg_addr, ((offset & 0xfc) << 24) | (dev->devfn << 16)    \
           | (dev->bus->number << 8) | 0x80);                              \
  MCS_##rw(value, hose->cfg_data + (offset & mask), type, op, &error);     \
  return error;                                                            \
}

MCS_PCI_CONFIG_OP(read, byte, u8*, mcs_in_8, 3)
MCS_PCI_CONFIG_OP(read, word, u16*, mcs_in_le16, 2)
MCS_PCI_CONFIG_OP(read, dword, u32*, mcs_in_le32, 0)
MCS_PCI_CONFIG_OP(write, byte, u8, mcs_out_8, 3)
MCS_PCI_CONFIG_OP(write, word, u16, mcs_out_le16, 2)
MCS_PCI_CONFIG_OP(write, dword, u32, mcs_out_le32, 0)
 
#if 0
/*
 * I'll better leave an example in here to show
 * to what the macros will expand... ;-)
 */
int
mcs_pci_read_config_byte(struct pci_dev *dev, int offset, u8* value)
{
  u32 error;
  struct pci_controller* hose = dev->sysdata;
  out_be32(hose->cfg_addr,
           ((offset & 0xfc) << 24) | (dev->devfn << 16)
           | (dev->bus->number << 8) | 0x80);
  *value = mcs_in_8((u8*)hose->cfg_data + (offset & 3), &error);
  return error;
}
#endif

#define CFG_read(val, addr, type, op) *val = op((type)(addr))
#define CFG_write(val, addr, type, op) op((type*)(addr), (val))
 
#define FIX_PCI_CONFIG_OP(rw, size, type, op, mask)                        \
void                                                                       \
fix_pci_##rw##_config_##size(u32 cfg_addr, u32 cfg_data,                   \
			     int offset, type value)                       \
{                                                                          \
  out_be32((u32*)cfg_addr, ((offset & 0xfc) << 24) | (0 << 16)             \
           | (0 << 8) | 0x80);                                             \
  CFG_##rw(value, cfg_data + (offset & mask), type, op);                   \
}
 
FIX_PCI_CONFIG_OP(read,  byte,  u8*,  in_8,   3)
FIX_PCI_CONFIG_OP(read,  word,  u16*, in_le16,   2)
FIX_PCI_CONFIG_OP(read,  dword, u32*, in_le32,  0)
FIX_PCI_CONFIG_OP(write, byte,  u8,   out_8,   3)
FIX_PCI_CONFIG_OP(write, word,  u16,  out_le16,  2)
FIX_PCI_CONFIG_OP(write, dword, u32,  out_le32, 0)


EXPORT_SYMBOL(mcs_in_8);
EXPORT_SYMBOL(mcs_in_le16);
EXPORT_SYMBOL(mcs_in_le32);
EXPORT_SYMBOL(mcs_out_8);
EXPORT_SYMBOL(mcs_out_le16);
EXPORT_SYMBOL(mcs_out_le32);
EXPORT_SYMBOL(mcs_pci_read_config_byte);
EXPORT_SYMBOL(mcs_pci_read_config_word);
EXPORT_SYMBOL(mcs_pci_read_config_dword);
EXPORT_SYMBOL(mcs_pci_write_config_byte);
EXPORT_SYMBOL(mcs_pci_write_config_word);
EXPORT_SYMBOL(mcs_pci_write_config_dword);
EXPORT_SYMBOL(fix_pci_read_config_byte);
EXPORT_SYMBOL(fix_pci_read_config_word);
EXPORT_SYMBOL(fix_pci_read_config_dword);
EXPORT_SYMBOL(fix_pci_write_config_byte);
EXPORT_SYMBOL(fix_pci_write_config_word);
EXPORT_SYMBOL(fix_pci_write_config_dword);

