/*****************************************************************************/
/* I2C Bus interface initialisation and I2C Commands                         */
/* for PPC405GP		                                                     */
/* Author : AS HARNOIS                                                       */
/* Date   : 13.Dec.00                                                        */
/*****************************************************************************/

#include <linux/config.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/smp.h>
#include <linux/threads.h>
#include <linux/spinlock.h>

#include <asm/system.h>
#include <linux/interrupt.h>

#include <linux/param.h>
#include <linux/string.h>
#include <linux/pci.h>
#include <linux/rtc.h>
#include <linux/console.h>
#include <linux/ide.h>

#include <asm/processor.h>
#include <asm/ibm4xx.h>
#include <asm/machdep.h>
#include <asm/page.h>
#include <asm/kgdb.h>
#include <asm/time.h>
#include <syslib/ppc4xx_i2c.h>

#ifdef DEBUG
# define D(fmt, args...)	{ printk(fmt, ##args); }
#else
# define D(fmt, args...)	{ }
#endif

static void out8(void *ptr, unsigned char val){
 /* Ensure I/O operations complete */
/* __asm__ volatile("eieio");
 * writeb has internal eieio
 */

   writeb(val, ptr);
		
}

static unsigned char in8(void *ptr){

   return readb(ptr);
}


void i2c_init(void)
{
  int divisor;

  D("I2C_INIT\n");
  /* clear lo master address */
  out8(IIC_LMADR,0);
  /* clear hi master address */
  out8(IIC_HMADR,0);
  /* clear lo slave address */
  out8(IIC_LSADR,0);
  /* clear hi slave address */
  out8(IIC_HSADR,0);

  /* Reset status register */
  /* write 1 in SCMP IRQA to clear these fields */
  /* !!! ASH QpenBios says 0x08 */
  /*  out8(IIC_STS, 0x0C);  */
  out8(IIC_STS, 0x08);
  /* write 1 in IRQP IRQD LA ICT XFRA to clear these fields */
  out8(IIC_EXTSTS, 0x8F);


  /* Clock divide Register */
  
  /* set divisor according to OPB frequency */
  if (__res.bi_opb_busfreq <= 20000000)
    divisor = 0x1;
  else if (__res.bi_opb_busfreq <= 30000000)
    divisor = 0x2;
  else if (__res.bi_opb_busfreq <= 40000000)
    divisor = 0x3;
  else if (__res.bi_opb_busfreq <= 50000000)
    divisor = 0x4;
  else
    divisor = 0x5;
  out8(IIC_CLKDIV,divisor);

  /* no interrupts */
  out8(IIC_INTRMSK,0);

  /* clear transfer count */
  out8(IIC_XFRCNT,0);


  /* clear extended control & stat */
  /* write 1 in SRC SRS SWC SWS to clear these fields */
  out8(IIC_XTCNTLSS,0xF0);


  /* Mode Control Register
     Flush Slave/Master data buffer
     Ignore General Call, 100kHz, slave transfers are ignored,
     disable interrupts, exit unknown bus state, enable hold
     SCL
  */
  out8(IIC_MDCNTL, IIC_MDCNTL_FSDB | IIC_MDCNTL_FMDB | IIC_MDCNTL_EUBS);


//  out8(IIC_MDCNTL, ((in8(IIC_MDCNTL))|IIC_MDCNTL_EUBS|IIC_MDCNTL_HSCL));
//sand
//  out8(IIC_MDCNTL, (IIC_MDCNTL_EUBS);
//

  /* clear control reg */
  out8(IIC_CNTL,0x00);


}


static int i2c_transfer(unsigned char command_is_reading, unsigned char address,
		 unsigned short size_to_transfer, unsigned char data[] )
{
  int bytes_transfered;
  int result;
  int status;
  int i, TimeReady, TimeOut;

  result = IIC_OK;
  bytes_transfered = 0;

  D("i2c transfer called\n");

  /* Check init */
  i=10;
  do{
    /* Get status */
    status = in8(IIC_STS);

    i--;
  }while ((status & IIC_STS_PT) && (i>0));
  if (status & IIC_STS_PT)
    {
      result = IIC_NOK_TOUT;
      D("IIC_NOK_TOUT \n");
      return(result);
    }

  /* 7-bit adressing */
  out8(IIC_HMADR,0);

  out8(IIC_LMADR, address);


  /* In worst case, you need to wait 4 OPB clocks */
  /* for data being ready on IIC bus */
  /* (because FIFO is 4 bytes depth) */
  TimeReady = (4/(__res.bi_procfreq/__res.bi_opb_busfreq)) + 1;

  /* IIC time out => wait for TimeOut cycles */
  /* TimeOut /= 2; because waiting-loop is 2-instruction loop */
  TimeOut = __res.bi_procfreq*IIC_TIMEOUT;// * 10;
  TimeOut /= 2;

  while ((bytes_transfered < size_to_transfer) && (result == IIC_OK))
    {
      D("\niic transfer: write %d byte (size %d)\n",bytes_transfered, size_to_transfer);
  
      /* Control register =
	 Normal transfer, 7-bits adressing, Transfer 1 byte, Normal start,
	 Transfer is a sequence of transfers
	 Write/Read, Start transfer */

      /* ACTION => Start - Transfer - Ack - pause */
      /* ACTION - LAST BYTE => Start - Transfer - Nack  - Stop */
      if (command_is_reading)
	{
          D("iic transfer: read command\n");
	  /* issue read command */
	  if (bytes_transfered == size_to_transfer-1)
	    {
	    out8(IIC_CNTL, IIC_CNTL_READ | IIC_CNTL_PT);
	
	    }
	  else
	    {
	      out8(IIC_CNTL, IIC_CNTL_READ | IIC_CNTL_CHT | IIC_CNTL_PT);
	
	    }
	}
      else
	{
          D("iic transfer: write command\n");
	  /* Set buffer */
	  out8(IIC_MDBUF,data[bytes_transfered]);
	
	  i=TimeReady;
	  do{
	    i--;
	  }while(i>0);
	  /* issue write command */
	  if (bytes_transfered == size_to_transfer-1)
	    {
	    out8(IIC_CNTL, IIC_CNTL_PT);
	
	    }
	  else
	    {
	    out8(IIC_CNTL, IIC_CNTL_CHT | IIC_CNTL_PT);
	
	    }
	}

	D("iic transfer in progress\n");
      /* Transfer is in progress */
      i=TimeOut;
      do{
	/* Get status */
	status = in8(IIC_STS);
	
	i--;
      }while ((status & IIC_STS_PT) && (i>0));

      if (status & IIC_STS_PT)
	{
	  D("Transfer pending\n");	
	  result = IIC_NOK_TOUT;
	}
      else if (status & IIC_STS_ERR)
	{
          D("iic transfer: IIC_STS_ERR\n");
	  result = IIC_NOK;
	  status = in8(IIC_EXTSTS);
          D("iic transfer: IIC_EXTSTS: %02x\n",status);
	
	  /* Lost arbitration? */
	  if (status & IIC_EXTSTS_LA)
	    result = IIC_NOK_LA;
	  /* Incomplete transfer? */
	  if (status & IIC_EXTSTS_ICT)
	    result = IIC_NOK_ICT;
	  /* Transfer aborted? */
	  if (status & IIC_EXTSTS_XFRA)
	    result = IIC_NOK_XFRA;
	}
      /* Command is reading => get buffer */
      if ((command_is_reading) && (result == IIC_OK))
	{
	  /* Are there data in buffer */
	  if (status & IIC_STS_MDBS)
	    {
	      i=TimeReady;
	      do{
		i--;
	      }while(i>0);
	      data[bytes_transfered] = in8(IIC_MDBUF);
	
	    }
	  else result = IIC_NOK_DATA;
	}
      bytes_transfered++;
    }
  D("iic transfer: bytes transfered: %d\n",bytes_transfered);
  return(result);
}


int i2c_receive(unsigned char address,
		 unsigned short size_to_expect, unsigned char datain[] )
{
  int status;
  
  status = i2c_transfer(1, address, size_to_expect, datain);
#ifdef CONFIG_PEPPERCON
  if (status != 0) {
  	D("I2C error => status = %d\n", status);
#if 0
  	D("I2C error => status = %02x\n", in8(IIC_STS));
  	D("I2C error => ext status = %02x\n", in8(IIC_EXTSTS));
  	D("I2C error => transfer count  = %02x\n", in8(IIC_XFRCNT));
  	D("I2C error => ext slave status = %02x\n", in8(IIC_XTCNTLSS));
#endif
  }
#endif
  return status;
}


int i2c_send(unsigned char address,
		 unsigned short size_to_send, unsigned char dataout[] )
{
  int status;
  status = i2c_transfer(0, address, size_to_send, dataout);
#if defined(CONFIG_WALNUT405) || defined (CONFIG_PEPPERCON)
  if (status != 0) {
  	D("I2C error => status = %d\n", status);
#if 0
  	D("I2C error => status = %02x\n", in8(IIC_STS));
  	D("I2C error => ext status = %02x\n", in8(IIC_EXTSTS));
  	D("I2C error => transfer count  = %02x\n", in8(IIC_XFRCNT));
  	D("I2C error => ext slave status = %02x\n", in8(IIC_XTCNTLSS));
#endif
  }
#endif
  return status;
}


int i2c_read (unsigned char *addr, int alen, unsigned char *buffer, int len)
{
	int ret = 0;

	D ("I2C read: addr len %d len %x\n", alen, len);
	if (alen == 2) {
		ret = i2c_send(addr[0] << 1, 1, addr+1);
		if (ret) return ret;
		ret = i2c_receive ((addr[0] << 1) | 0x01, len, buffer);
	}
	else if (alen == 3) {
		ret = i2c_send(addr[0] << 1, 2, addr+1);
		if (ret) return ret;
		ret = i2c_receive((addr[0] << 1) | 0x01, len, buffer);
	} else{
		D("I2C read: addr len %d not supported\n", alen);
		ret = -1;
	}
	
	return ret;
}

int i2c_write (unsigned char *addr, int alen, unsigned char *buffer, int len)
{
	unsigned char xbuf[3];
//	unsigned char intaddr[2];
	unsigned short eepaddr;
	int timeout;

	D ("I2C write: addr len %d len %x\n", alen,len);
	if (alen == 2) {
		/* write with ack polling */
		/* XXX this should be improved to allow page write mode XXX - wd */
		while (len-- > 0) {
          		xbuf[0] = addr[1]++; /* increase write offset */
          		xbuf[1] = *buffer++;
          		while (i2c_send(addr[0] << 1, 2, xbuf) != 0) /* single write + ack polling */
            		udelay(100);
		}
	}else if (alen == 3){
		/* write with ack polling */
		/* XXX this should be improved to allow page write mode XXX - wd */
		eepaddr = addr[1];
		eepaddr <<= 8;
		eepaddr |= addr[2];
		while (len-- > 0) {
//          		D("\nwrite %02x to eep (%02x) address %02x%02x\n", xbuf[2],addr[0]<<1,xbuf[0],xbuf[1]);
			xbuf[0] = eepaddr >> 8; /* increase write offset */
          		xbuf[1] = (unsigned char) eepaddr & 0xFF;
          		xbuf[2] = *buffer++;
			eepaddr++;
//          		D("write %02x to eep (%02x) address %02x%02x\n", xbuf[2],addr[0]<<1,xbuf[0],xbuf[1]);
          		timeout = 0;
          		while (i2c_send(addr[0] << 1, 3, xbuf) != 0){ /* single write + ack polling */
            			udelay(100);
            			timeout++;	
            			if (timeout > 10) {
            				D("I2C write timeout after %d * 100 udelays\n",timeout);
            				return -1;
            			}
            		};
//			D("I2C write need %d * 100 udelays\n",timeout);
            		
		}
	}else {
		D("I2C write: addr len %d not supported\n", alen);
		return -1;
	}

	return 0;
}

/*-----------------------------------------------------------------------
 */

int rtc_read (unsigned offset, unsigned char *buffer, unsigned cnt)
{
	unsigned end = offset + cnt;
	unsigned blk_off;
	int ret;

	D("RTC read\n");
	/* Read data until done or would cross a page boundary.
	 * We must write the address again when changing pages
	 * because the next page may be in a different device.
	 */
	while (offset < end) {
		unsigned alen, len, maxlen;
		unsigned char addr[3];

		blk_off = offset & 0xFF;	/* block offset */

		addr[0] = offset >> 16;		/* block number */
		addr[1] = offset >>  8;		/* upper address octet */
		addr[2] = blk_off;		/* lower address octet */
		alen	= 3;

		addr[0] |= CFG_I2C_RTC_ADDR;	/* insert device address */

		maxlen = 0x100 - blk_off;
		len    = end - offset;
		if (len > maxlen)
			len = maxlen;

		if ((ret = i2c_read(addr, alen, buffer, len))) {
			return ret;
		}
		buffer += len;
		offset += len;
	}

	return 0;
}

/*-----------------------------------------------------------------------
 */

int rtc_write (unsigned offset, unsigned char *buffer, unsigned cnt)
{
	unsigned end = offset + cnt;
	unsigned blk_off;
	int ret;

	/* Write data until done or would cross a write page boundary.
	 * We must write the address again when changing pages
	 * because the address counter only increments within a page.
	 */

	D("RTC write\n");
	while (offset < end) {
		unsigned alen, len, maxlen;
		unsigned char addr[3];

		blk_off = offset & 0xFF;	/* block offset */

		addr[0] = 0;			/* block number */
		addr[1] = offset >>  8;		/* upper address octet */
		addr[2] = blk_off;		/* lower address octet */
		alen	= 3;

		addr[0] |= CFG_I2C_RTC_ADDR;	/* insert device address */

		maxlen = 0x100 - blk_off;
		len = end - offset;
		if (len > maxlen)
			len = maxlen;

		if ((ret = i2c_write(addr, alen, buffer, len))) {
			return ret;
		}
		buffer += len;
		buffer += len;
		offset += len;
	}

	return 0;
}


