/*  /drivers/i2c/busses/i2c-asics.c
	Support of the Asics i2c bus
	
	Copyright (C) 2004 Microtronix Datacom Ltd.
	
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
    Written by Wentao Xu <wentao@microtronix.com>
*/

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/wait.h>
#include <linux/pci.h>
#include <linux/i2c.h>
#include <linux/i2c-slave.h>
#include <linux/i2c-algo-asics.h>
#include <asm/io.h>
#include <asm/irq.h>
#include "i2c-asics.h"
#include <syslib/ppc405_pcimcs.h>

MODULE_AUTHOR("Peppercon AG");
MODULE_DESCRIPTION("I2C-Bus adapter routines for Asics i2c controller");
MODULE_LICENSE("GPL");

#define ASICSI2C_IO_SIZE 	64
#define ASICSI2C_CLOCK_DEFAULT	66  /* MHz */
#define ASICSI2C_SCL_DEFAULT	100 /* kHz */
#define ASICSI2C_IRQ_NO_INTMUX	-1

#if defined(LARA_KIMMSI) || defined(LARA_KIMSMI) || defined (LARA_KIMAMD) || defined (LARA_KIMINTEL)
# define ASICS_USE_PCI
# define ASICS_USE_INTMUX
# define ASICS_PCI_VENDOR_ID	0x1743	
# define ASICS_PCI_DEVICE_ID	0x1003
# define ASICS_PCI_BAR_NO	1
# define INTMUX_OFFSET		0x9A0000
# define INTMUX_SIZE		4
#else
# define ASICS_BASE		0x81000000
#endif

#define ASICS_I2C_IRQ		27 /* common irq for all cores */

#define I2C_0_BASE_OFFSET	0x900000
#define I2C_0_INTMUX		1

#define I2C_1_BASE_OFFSET	0x910000
#define I2C_1_INTMUX		2

#define I2C_2_BASE_OFFSET	0x920000
#define I2C_2_INTMUX		3

#define I2C_3_BASE_OFFSET	0x930000
#define I2C_3_INTMUX		4

#define I2C_4_BASE_OFFSET	0x940000
#define I2C_4_INTMUX		5

#define I2C_DDC_BASE_OFFSET	0x9C0000
#define I2C_DDC_INTMUX		15

#define I2C_LAN_BASE_OFFSET	0x9D0000
#define I2C_LAN_INTMUX		16

/******************************************************************************
* debugging                                                                   *
******************************************************************************/

/*
  DBG_LEVEL == 0 --> no debugging
  DBG_LEVEL < 0  --> short debugging
  DBG_LEVEL > 0  --> full debugging
*/

#define DBG_LEVEL 0

#if DBG_LEVEL > 0
#  define DBG(fmt, x...)	printk(KERN_DEBUG "i2c-asics: " fmt, ##x)
#else
#  define DBG(fmt, x...)	((void)0)
#endif
#if DBG_LEVEL > 1
#  define DBG2(fmt, x...) 	DBG( fmt, ##x )
#else
#  define DBG2(fmt, x...) 	((void)0)
#endif
#if DBG_LEVEL > 2
#  define DBG3(fmt, x...) 	DBG( fmt, ##x )
#else
#  define DBG3(fmt, x...) 	((void)0)
#endif

#if DBG_LEVEL < 0
#  define SDBG(fmt, x...) 	printk(/*KERN_DEBUG*/KERN_INFO fmt, ##x)
#else
#  define SDBG(fmt, x...) 	((void)0)
#endif

#define INFO(fmt, x...)		printk(KERN_INFO    "i2c-asics: " fmt, ##x)
#define WARN(fmt, x...)		printk(KERN_WARNING "i2c-asics: " fmt, ##x)
#define CRIT(fmt, x...)		printk(KERN_CRIT    "i2c-asics: " fmt, ##x)
#define ERR(fmt, x...)		printk(KERN_ERR     "i2c-asics: " fmt, ##x)

#define DBG_SEND_MESSAGES_ON_LOAD	0
#define DBG_READ_ADDITIONAL_REGISTERS	0

#if DBG_SEND_MESSAGES_ON_LOAD
#include "i2c-asics-debug.c"
#endif

/******************************************************************************
 * debugging via debug pin                                                    *
 ******************************************************************************/

#define noGPIO_DEBUG

#ifdef GPIO_DEBUG

#define PPC405_GPIO_REGS_ADDR		((u32)0xEF600700)
#define PPC405_GPIO_REGS_SIZE		((size_t)0x20)

static volatile u32 * gpio_regs = NULL;

static void
gpio_init(void) {
	gpio_regs = ioremap_nocache(PPC405_GPIO_REGS_ADDR,
				    PPC405_GPIO_REGS_SIZE);
	
	if (gpio_regs) {
		INFO("GPIO debugging enabled.\n");
	} else {
		WARN("Could not map GPIO registers!\n");
	}
}

static void
gpio_cleanup(void) {
	if (gpio_regs) {
		iounmap((void *)gpio_regs);
		INFO("GPIO debugging disabled.\n");
	} else {
		INFO("Could not unmap GPIO registers, not mapped.\n");
	}
}

static void
gpio_bit_set(unsigned int value)
{
	if (gpio_regs) {
		int bitnum = 31 - 9;
		if (value) {
			set_bit(bitnum, &gpio_regs[0]);
		} else {
			clear_bit(bitnum, &gpio_regs[0]);
		}
	}
}

#else

static void gpio_bit_set(unsigned int value) { }
static void gpio_init(void) { }
static void gpio_cleanup(void) { }

#endif

/******************************************************************************
* board configuration                                                         *
******************************************************************************/

static struct i2c_asics_interface_configuration i2c_asics_interfaces[] = {
    {
    	used:			1,
    	base_offset:		I2C_0_BASE_OFFSET,
    	irq:			ASICS_I2C_IRQ,
    	irq_mux:		I2C_0_INTMUX,
    	scl:			100,
    	clock:			0,
    	slave_addresses:	{ 0x33, 0, 0, 0, 0, 0, 0, 0 },
    	name:			"Asics I2C 0"
    },
    {
    	used:			1,
    	base_offset:		I2C_1_BASE_OFFSET,
    	irq:			ASICS_I2C_IRQ,
    	irq_mux:		I2C_1_INTMUX,
    	scl:			100,
    	clock:			0,
    	slave_addresses:	{ 0x33, 0, 0, 0, 0, 0, 0, 0 },
    	name:			"Asics I2C 1"
    },
    {
    	used:			1,
    	base_offset:		I2C_2_BASE_OFFSET,
    	irq:			ASICS_I2C_IRQ,
    	irq_mux:		I2C_2_INTMUX,
    	scl:			100,
    	clock:			0,
    	slave_addresses:	{ 0x33, 0, 0, 0, 0, 0, 0, 0 },
    	name:			"Asics I2C 2"
    },
    {
    	used:			1,
    	base_offset:		I2C_3_BASE_OFFSET,
    	irq:			ASICS_I2C_IRQ,
    	irq_mux:		I2C_3_INTMUX,
    	scl:			100,
    	clock:			0,
    	slave_addresses:	{ 0x33, 0, 0, 0, 0, 0, 0, 0 },
    	name:			"Asics I2C 3"
    },
    {
    	used:			1,
    	base_offset:		I2C_4_BASE_OFFSET,
    	irq:			ASICS_I2C_IRQ,
    	irq_mux:		I2C_4_INTMUX,
    	scl:			100,
    	clock:			0,
    	slave_addresses:	{ 0x33, 0, 0, 0, 0, 0, 0, 0 },
    	name:			"Asics I2C 4"
    },
    {
    	used:			1,
    	base_offset:		I2C_DDC_BASE_OFFSET,
    	irq:			ASICS_I2C_IRQ,
    	irq_mux:		I2C_DDC_INTMUX,
    	scl:			100,
    	clock:			0,
    	slave_addresses:	{ 0x33, 0, 0, 0, 0, 0, 0, 0 },
    	name:			"Asics I2C DDC"
    },
    {
    	used:			1,
    	base_offset:		I2C_LAN_BASE_OFFSET,
    	irq:			ASICS_I2C_IRQ,
    	irq_mux:		I2C_LAN_INTMUX,
    	scl:			100,
    	clock:			0,
    	slave_addresses:	{ 0x33, 0, 0, 0, 0, 0, 0, 0 },
    	name:			"Asics I2C LAN"
    },
};

#define ASICS_NUM_INTERFACES	(sizeof(i2c_asics_interfaces) / sizeof(struct i2c_asics_interface_configuration))

/******************************************************************************
* function prototypes                                                         *
******************************************************************************/

static int asicsi2c_wait_for_tdone(struct iic_asics_hw *hw);
static irqreturn_t asics_handler(int this_irq, void *dev_id, struct pt_regs *regs);
static int i2c_asics_adapter_configure(struct i2c_adapter *i2c_adap, int initial);
static int asics_hw_init(struct i2c_adapter *i2c_adap, struct iic_asics_hw *hw);
static void asics_slave_init(struct i2c_algo_asics_slave_data * slave_data, int initial);
static int asics_slave_init_adapter(struct i2c_slave_adapter *slave, struct i2c_adapter *adap);
static int i2c_asics_adapter_init(struct i2c_adapter *i2c_adap);
static void asics_hw_exit(struct iic_asics_hw* hw);
static int asics_slave_cleanup_adapter(struct i2c_slave_adapter *slave);
static void i2c_asics_adapter_cleanup(struct i2c_adapter *i2c_adap);
static struct i2c_adapter* i2c_asics_get_adapter_from_hw(struct iic_asics_hw *hw);

/******************************************************************************
* helper functions                                                            *
******************************************************************************/

static char* num_to_bin(uint8_t num)
{
	static char bin[9];
	int i;
	
	for (i = 0; i < 8; i++) {
		bin[i] = (num & (1 << (7 - i))) ? '1' : '0';
	}
	bin[8] = '\0';
	
	return bin;
}

static void asicsi2c_setreg(struct iic_asics_hw *hw, int regno, int val)
{
	DBG3("asicsi2c_setreg: regno %02x (%s), value 0x%02x (%s)\n",
		regno,
		regno == 0 ? "PRERL" : regno == 1 ? "PRERH" : regno == 2 ? "CCR" :
			regno == 3 ? "SR" : regno == 4 ? "DR" : regno == 5 ? "SAR" :
			regno == 13 ? "ESAR" : "unknown",
		val, num_to_bin(val));
	writeb(val, &hw->asics_regs[regno]);
}

static int asicsi2c_getreg_nodbg(struct iic_asics_hw *hw, int regno)
{
	return readb(&hw->asics_regs[regno]);
}

#if DBG_LEVEL > 2
static int asicsi2c_getreg(struct iic_asics_hw *hw, int regno)
{
	int ret = asicsi2c_getreg_nodbg(hw, regno);
	
	DBG3("asicsi2c_getreg: regno %02x (%s), value 0x%02x (%s)\n",
		regno,
		regno == 0 ? "PRERL" : regno == 1 ? "PRERH" : regno == 2 ? "CCR" :
			regno == 3 ? "SR" : regno == 4 ? "DR" : regno == 5 ? "SAR" :
			regno == 13 ? "ESAR" : "unknown",
		ret, num_to_bin(ret));
	
	return ret;
}
#else /* DBG_LEVEL > 2 */
//#define asicsi2c_setreg asicsi2c_setreg_nodbg
#define asicsi2c_getreg asicsi2c_getreg_nodbg
#endif /* DBG_LEVEL > 2 */

#define asics_set_bit(hw, register, bit) \
	asicsi2c_setreg(hw, register, asicsi2c_getreg_nodbg(hw, register) | bit)
#define asics_clear_bit(hw, register, bit) \
	asicsi2c_setreg(hw, register, asicsi2c_getreg_nodbg(hw, register) & ~bit)


static int asicsi2c_wait_for_tdone(struct iic_asics_hw *hw)
{
	int timeout = 2;

	if (hw->irq > 0) {
		DBG("Waiting for transfer to end (interrupt)...\n");
		wait_event_interruptible_timeout(hw->wait, 
					((asicsi2c_getreg(hw, REG_ASICSI2C_SR) & BIT_ASICSI2C_SR_TDONE) != 0) || hw->arbitration_lost, 
					timeout*HZ);
		DBG("...Waiting finished.\n");
	} else {
		DBG("Waiting for transfer to end (polling).\n");
		udelay(100);
	}
	
	if (!(asicsi2c_getreg(hw, REG_ASICSI2C_SR) & BIT_ASICSI2C_SR_TDONE) && !hw->arbitration_lost)
		return -1;
	return 0;
}

/******************************************************************************
* interrupt handler                                                           *
******************************************************************************/

static irqreturn_t asics_handler(int this_irq, void *dev_id, struct pt_regs *regs) 
{
	struct iic_asics_hw *hw= (struct iic_asics_hw *)dev_id;
	struct i2c_adapter *i2c_adap = i2c_asics_get_adapter_from_hw(hw);
	struct i2c_algo_asics_data *adap = (struct i2c_algo_asics_data *) i2c_adap->algo_data;

	uint8_t cr = asicsi2c_getreg_nodbg(hw, REG_ASICSI2C_CCR);
	uint8_t sr = asicsi2c_getreg_nodbg(hw, REG_ASICSI2C_SR);
	
	int arbitration_lost = 0;
	
	gpio_bit_set(1);

	DBG2("---- asics_handler start\n");

#if DBG_READ_ADDITIONAL_REGISTERS
	{
		int i;
		for (i = 7; i < 12; i++) {
			asicsi2c_getreg(hw, i);
		}
	}
#endif /* DBG_READ_ADDITIONAL_REGISTERS */

	SDBG("\n+\n");
#ifdef ASICS_USE_INTMUX
	if (hw->irq_mux != ASICSI2C_IRQ_NO_INTMUX) {
	    u_long mux = readl(&hw->intmux_reg[0]);
	    if (!test_bit(hw->irq_mux, &mux)) {
		// this is not our IRQ
		DBG2("---- asics_handler: not our irq\n");
		SDBG("-\n");
		gpio_bit_set(0);
		return;
	    }
	}
#endif
	
	DBG2("---- asics_handler our irq\n");

	DBG2("asics_handler: CCR=0x%02x (%s)\n", cr, num_to_bin(cr));
	DBG2("asics_handler: SR=0x%02x (%s)\n", sr, num_to_bin(sr));
	DBG2("asics_handler: DR=0x%02x (%s)\n", asicsi2c_getreg_nodbg(hw, REG_ASICSI2C_DR), num_to_bin(asicsi2c_getreg_nodbg(hw, REG_ASICSI2C_DR)));
	SDBG("%x %x\n", cr, sr);
	DBG3("asics_handler: SAR=0x%02x (%s)\n", asicsi2c_getreg_nodbg(hw, REG_ASICSI2C_SAR0), num_to_bin(asicsi2c_getreg_nodbg(hw, REG_ASICSI2C_SAR0)));
	DBG3("asics_handler: ESAR=0x%02x (%s)\n", asicsi2c_getreg_nodbg(hw, REG_ASICSI2C_ESAR), num_to_bin(asicsi2c_getreg_nodbg(hw, REG_ASICSI2C_ESAR)));
	DBG3("asics_handler: PRERL=0x%02x (%s)\n", asicsi2c_getreg_nodbg(hw, REG_ASICSI2C_PRERL), num_to_bin(asicsi2c_getreg_nodbg(hw, REG_ASICSI2C_PRERL)));
	DBG3("asics_handler: PRERH=0x%02x (%s)\n", asicsi2c_getreg_nodbg(hw, REG_ASICSI2C_PRERH), num_to_bin(asicsi2c_getreg_nodbg(hw, REG_ASICSI2C_PRERH)));
	
	/* check for arbitration lost */
	if (sr & BIT_ASICSI2C_SR_AL) {
		DBG2("lost arbitration\n");
		arbitration_lost = hw->arbitration_lost = 1;
		adap->current_state = I2C_ASICS_MODE_IDLE;
		asics_clear_bit(hw, REG_ASICSI2C_SR, BIT_ASICSI2C_SR_AL);
	} else {
		arbitration_lost = hw->arbitration_lost = 0;
	}
	
	/* check for slave mode */
	if (sr & BIT_ASICSI2C_SR_CAS) {
	    DBG2("switching to slave mode\n");
	    if (adap->current_state != I2C_ASICS_MODE_SLAVE) {
		    /* clear old stop condition, if any */
		    if (sr & BIT_ASICSI2C_SR_DSP) {
			DBG2("clear unknown stop cond\n");
			adap->current_state = I2C_ASICS_MODE_IDLE;
			asics_clear_bit(hw, REG_ASICSI2C_SR, BIT_ASICSI2C_SR_DSP);
			sr &= ~BIT_ASICSI2C_SR_DSP;
		    }
	    }
	    adap->current_state = I2C_ASICS_MODE_SLAVE;
	}

	if (!(cr & BIT_ASICSI2C_CC_MS) && !arbitration_lost) {
		DBG3("got slave request, found adapter %s.\n", i2c_adap->name);
		cr = i2c_asics_handle_slave_request(i2c_adap);
		SDBG("%x\n", cr);
		asicsi2c_setreg(hw, REG_ASICSI2C_CCR, cr);
	}
	
	/* check for stop condition */
	if (sr & BIT_ASICSI2C_SR_DSP) {
		DBG2("switching to idle mode\n");
		adap->current_state = I2C_ASICS_MODE_IDLE;
		asics_clear_bit(hw, REG_ASICSI2C_SR, BIT_ASICSI2C_SR_DSP);
	}
	
	/* ack this interrupt */
	asics_clear_bit(hw, REG_ASICSI2C_SR, BIT_ASICSI2C_SR_IF);

	wake_up_interruptible(&hw->wait);
	
#if DBG_READ_ADDITIONAL_REGISTERS
	{
		int i;
		for (i = 7; i < 12; i++) {
			asicsi2c_getreg(hw, i);
		}
	}
#endif /* DBG_READ_ADDITIONAL_REGISTERS */

	DBG2("---- asics_handler finished\n");
	SDBG("-\n");
	gpio_bit_set(0);

	return IRQ_HANDLED;
}


/******************************************************************************
* initialization functions                                                    *
******************************************************************************/

static int check_access(struct iic_asics_hw *hw) {
    u_int8_t dummy;
    int access_error = 0;
    dummy = mcs_in_8((volatile u8 *)&hw->asics_regs[REG_ASICSI2C_CCR], &access_error);
    if (access_error != 0) goto bail;
    mcs_out_8(dummy, (volatile u8 *)&hw->asics_regs[REG_ASICSI2C_CCR], &access_error);
 bail:
    return access_error;
}

static int asics_hw_init(struct i2c_adapter *i2c_adap, struct iic_asics_hw *hw)
{
	int dev_base;
	int ret = 0;
#ifdef ASICS_USE_PCI
	struct pci_dev *pci_dev;
#endif
	hw->initialized = 0;
	hw->irq_registered = 0;
	
#ifdef ASICS_USE_PCI
	pci_dev = pci_find_device(ASICS_PCI_VENDOR_ID, ASICS_PCI_DEVICE_ID, NULL);
	if (pci_dev == NULL) {
	    ERR("PCI device not detected\n");
	    return -ENODEV;
	}
	dev_base = pci_resource_start(pci_dev, ASICS_PCI_BAR_NO);
#else
	dev_base = ASICS_BASE;
#endif
	hw->base = dev_base + hw->base_offset;
	DBG("using physical base address 0x%08x\n", hw->base);
	
	if (!request_mem_region(hw->base, ASICSI2C_IO_SIZE, "i2c (bus adapter)")) {
	    ERR("requested MMIO region (0x%X:2) is in use.\n", hw->base);
	    return -ENODEV;
	}

#ifdef ASICS_USE_INTMUX
	hw->intmux_reg = ioremap_nocache(dev_base + INTMUX_OFFSET, INTMUX_SIZE);
	if (hw->intmux_reg == NULL) {
	    ERR("failed to map IRQ multiplexer\n");
	    ret = -ENODEV;
	    goto error_out;
	}
	DBG("mapped IRQ multiplexer from 0x%08x to 0x%p (%d Byte)\n", dev_base + INTMUX_OFFSET,
	    hw->intmux_reg, INTMUX_SIZE);
#endif
	
	/* map asics regs */
	hw->asics_regs = ioremap_nocache(hw->base, ASICSI2C_IO_SIZE);
	if (hw->asics_regs == NULL) {
	    ERR("failed to map ASICS regs\n");
	    ret = -ENOMEM;
	    goto error_out;
	}
	DBG("mapped ASICS regs from 0x%08x to 0x%p (%d Byte)\n", hw->base,
	    hw->asics_regs, ASICSI2C_IO_SIZE);

	if (check_access(hw)) {
	    WARN("Could not access i2c registers '%s', skipping\n", i2c_adap->name);
	    ret = -ENODEV;
	    goto error_out;
	}
	
	hw->initialized = 1;
	DBG2("asics_hw_init successful\n");

	return 0;
	
 error_out:
	if (hw->asics_regs != NULL) {
	    iounmap((void*)hw->asics_regs);
	    hw->asics_regs = NULL;
	}

#ifdef ASICS_USE_INTMUX
	if (hw->intmux_reg != NULL) {
	    iounmap((void *)hw->intmux_reg);
	    hw->intmux_reg = NULL;
	}    
#endif

	release_mem_region(hw->base, ASICSI2C_IO_SIZE);

	return ret;
}

static void asics_slave_init(struct i2c_algo_asics_slave_data * slave_data, int initial)
{
	if (initial) spin_lock_init(slave_data->slave_lock);
	
	slave_data->block_next_slave_operation = 0;
	slave_data->last_slave_op = I2C_ASICS_SLAVE_OP_NONE;
	
        slave_data->slave_write_ptr = slave_data->slave_write_buffer;
        slave_data->slave_write_count = 0;
        slave_data->slave_read_ptr = slave_data->slave_read_buffer;
        slave_data->slave_read_count = 0;
}

/* register adapter in slave layer */
static int asics_slave_init_adapter(struct i2c_slave_adapter *slave, struct i2c_adapter *adap)
{
	int ret = 0;

	slave->adapter = adap;
	slave->reactivate = i2c_asics_slave_restart_xfer;
	if ((ret = i2c_slave_add_adapter(slave)) != 0){
		CRIT("failed to register i2c slave adapter\n");
		return ret;
	}
	
	return 0;
}

static int i2c_asics_adapter_configure(struct i2c_adapter *i2c_adap, int initial)
{
	struct i2c_algo_asics_data *adap = (struct i2c_algo_asics_data *) i2c_adap->algo_data;
	struct i2c_asics_interface_configuration *config = adap->config;
	struct i2c_algo_asics_slave_data *slave_data = adap->slave_data;
	struct iic_asics_hw *hw = adap->hw;
	unsigned int i;
	int prescale;

	/* clear EN and MS bit */
	asicsi2c_setreg(hw, REG_ASICSI2C_CCR, 0);
	
	/* compute prescale */
	prescale = hw->clock * 1000 / (4 * hw->scl) - 1;
	DBG("scl=%d, clock=%d -> prescale=%d\n", hw->scl, hw->clock, prescale);
	if (prescale < 0) {
	    ERR("prescale error clk_i %d scl %d\n", hw->clock, hw->scl);
	    return -ENODEV;
	}
	
	/* set prescale */
	asicsi2c_setreg(hw, REG_ASICSI2C_PRERL, prescale & 0xff);
	asicsi2c_setreg(hw, REG_ASICSI2C_PRERH, prescale >> 8);
	
	/* error debugging */
	asicsi2c_setreg(hw, REG_ASICSI2C_SR, 8);
	asicsi2c_setreg(hw, REG_ASICSI2C_DR, 0x20);

	if (hw->irq > 0) {
	    if (request_irq(hw->irq, asics_handler, SA_SHIRQ | SA_SAMPLE_RANDOM, "Asics I2C", hw) < 0) {
		ERR("Request irq%d failed\n", hw->irq);
		hw->irq = 0;
		
		/* set EN bit */
		asicsi2c_setreg(hw, REG_ASICSI2C_CCR, BIT_ASICSI2C_CC_ENA);

		hw->irq_registered = 1;
	    } else {
		DBG("Requested irq %d.\n", hw->irq);
		
		/* set EN bit */
		asicsi2c_setreg(hw, REG_ASICSI2C_CCR, BIT_ASICSI2C_CC_ENA | BIT_ASICSI2C_CC_IEN);	
	    }
	}
	
	hw->arbitration_lost = 0;

	asics_slave_init(slave_data, initial);
	for (i = 0; i < NO_SLAVE_ADDRESSES; i++) {
	    i2c_asics_slave_set_address(i2c_adap, i, config->slave_addresses[i]);
	}
	
	return 0;
}

static int i2c_asics_adapter_init(struct i2c_adapter *i2c_adap)
{
	struct i2c_algo_asics_data *adap = (struct i2c_algo_asics_data *) i2c_adap->algo_data;
	struct iic_asics_hw *hw = adap->hw;
	struct i2c_slave_adapter *slave_adapter = adap->slave_adapter;
	
	DBG3("Initializing hardware for %s.\n", i2c_adap->name);
	
	init_waitqueue_head(&hw->wait);
	if (asics_hw_init(i2c_adap, hw) == 0) {
		if (i2c_asics_adapter_configure(i2c_adap, 1) < 0) {
			asics_hw_exit(hw);
			return -1;
		}
		if (i2c_asics_add_bus(i2c_adap) < 0) {
			asics_hw_exit(hw);
			return -1;
		}
		if (asics_slave_init_adapter(slave_adapter, i2c_adap) < 0) {
			asics_hw_exit(hw);
			return -1;
		}
	}
	
	return 0;
}

/******************************************************************************
* cleanup functions                                                           *
******************************************************************************/

static void asics_hw_exit(struct iic_asics_hw* hw)
{
    /* free irq */
    if (hw->irq > 0 && hw->irq_registered) {
	free_irq(hw->irq, hw);
	hw->irq_registered = 0;
    }
    
    /* free register mapping */
    if (hw->asics_regs != NULL) {
	iounmap((void*)hw->asics_regs);
	hw->asics_regs = NULL;
    }
    
#ifdef ASICS_USE_INTMUX
    if (hw->intmux_reg != NULL) {
	iounmap((void *)hw->intmux_reg);
	hw->intmux_reg = NULL;
    }    
#endif
    
    /* free mem region */
    release_mem_region(hw->base, ASICSI2C_IO_SIZE);       
}

/* deregister */
static int asics_slave_cleanup_adapter(struct i2c_slave_adapter *slave)
{
	int ret;
	
	if ((ret = i2c_slave_remove_adapter(slave)) != 0) {
		CRIT("failed to deregister i2c slave adapter\n");
		return ret;
	}
	
	return 0;
}

static void i2c_asics_adapter_cleanup(struct i2c_adapter *i2c_adap)
{
	struct i2c_algo_asics_data *data = (struct i2c_algo_asics_data *) i2c_adap->algo_data;
	struct iic_asics_hw *hw = data->hw;
	struct i2c_slave_adapter *slave_adapter = data->slave_adapter;

	if (hw->initialized == 0) {
	    DBG("i2c_asics_adapter_cleanup: adapter '%s' not used, skipping\n", i2c_adap->name);
	    return;
	}
	
	i2c_asics_del_bus(i2c_adap);
	asics_hw_exit(hw);
	asics_slave_cleanup_adapter(slave_adapter);
}

/******************************************************************************
* module data                                                                 *
******************************************************************************/

/* there are several adapters on the board */

static struct i2c_algo_asics_slave_data asics_slave[ASICS_NUM_INTERFACES];

static struct i2c_slave_adapter asics_slave_adapt[ASICS_NUM_INTERFACES];

static struct iic_asics_hw asics_hw[ASICS_NUM_INTERFACES];

static struct i2c_algo_asics_data asics_data[ASICS_NUM_INTERFACES];

static struct i2c_adapter asics_ops[ASICS_NUM_INTERFACES];


static struct i2c_adapter* i2c_asics_get_adapter_from_hw(struct iic_asics_hw *hw)
{
	int i;
	
	for (i = 0; i < ASICS_NUM_INTERFACES; i++) {
		if (hw == ((struct i2c_algo_asics_data *) asics_ops[i].algo_data)->hw) return &asics_ops[i];
	}
	
	return NULL;
}

static void i2c_asics_init_adapters(void) {
	int i;
	
	memset(asics_ops, 0, sizeof(asics_ops));
	memset(asics_data, 0, sizeof(asics_data));
	memset(asics_hw, 0, sizeof(asics_hw));
	memset(asics_slave_adapt, 0, sizeof(asics_slave_adapt));
	memset(asics_slave, 0, sizeof(asics_slave));

	for (i = 0; i < ASICS_NUM_INTERFACES; i++) {
		if (!i2c_asics_interfaces[i].used) {
			continue;
		}
		
		/* fill i2c_adapter struct */
#warning owner n/a for 2.4
		//asics_ops[i].owner		= THIS_MODULE;
		asics_ops[i].id			= I2C_HW_ASICS_UKIT;
		asics_ops[i].algo_data		= &asics_data[i];
		strncpy(asics_ops[i].name, i2c_asics_interfaces[i].name, sizeof(asics_ops[i].name));
		
		/* fill i2c_algo_asics_data struct */
		asics_data[i].i2c_adap		= &asics_ops[i];
		asics_data[i].hw		= &asics_hw[i];
		asics_data[i].slave_data	= &asics_slave[i];
		asics_data[i].slave_adapter	= &asics_slave_adapt[i];
		asics_data[i].config		= &i2c_asics_interfaces[i];
		asics_data[i].setreg		= asicsi2c_setreg;
		asics_data[i].getreg		= asicsi2c_getreg;
		asics_data[i].wait_for_tdone	= asicsi2c_wait_for_tdone;
		asics_data[i].configure		= i2c_asics_adapter_configure;
		asics_data[i].udelay		= 10,
		asics_data[i].mdelay		= 10,
		asics_data[i].timeout		= 100,
		asics_data[i].current_state	= I2C_ASICS_MODE_IDLE,
		
		/* fill iic_asics_hw struct */
		asics_hw[i].base_offset		= i2c_asics_interfaces[i].base_offset ? i2c_asics_interfaces[i].base_offset : 0;
		asics_hw[i].irq			= i2c_asics_interfaces[i].irq ? i2c_asics_interfaces[i].irq : 0;
		asics_hw[i].irq_mux		= i2c_asics_interfaces[i].irq_mux ? i2c_asics_interfaces[i].irq_mux : 0;
		asics_hw[i].clock		= i2c_asics_interfaces[i].clock ? i2c_asics_interfaces[i].clock : ASICSI2C_CLOCK_DEFAULT;
		asics_hw[i].scl			= i2c_asics_interfaces[i].scl ? i2c_asics_interfaces[i].scl : ASICSI2C_SCL_DEFAULT;
	}
}

/******************************************************************************
* module stuff                                                                *
******************************************************************************/

static int __init i2c_asics_init(void) 
{
	int i;
	
	INFO("i2c Asics adapter driver\n");

	gpio_init();
	mdelay(500);

	/* parameters */
	i2c_asics_init_adapters();
	
	for (i = 0; i < ASICS_NUM_INTERFACES; i++) {
		i2c_asics_adapter_init(&asics_ops[i]);
	}
	
#if DBG_SEND_MESSAGES_ON_LOAD
	send_some_messages(&asics_ops[0]);
#endif
	return 0;
}

static void i2c_asics_exit(void)
{
	int i;

	for (i = 0; i < ASICS_NUM_INTERFACES; i++) {
		i2c_asics_adapter_cleanup(&asics_ops[i]);
	}
	
	gpio_cleanup();
}

module_init(i2c_asics_init);
module_exit(i2c_asics_exit);
