/* ------------------------------------------------------------------------- */
/* 									     */
/* i2c-slave.h - definitions for the i2c bus interface slave mode	     */
/* 									     */
/* ------------------------------------------------------------------------- */
/* Author: Thomas Rendelmann (thre@peppercon.de)                             */

/* 
   Slave functionality is not well supported by the kernel at the moment.
   Because I2C slave mode works differently than the master mode (the slave
   mode is always triggered from outside, namely from the master), it doesn't
   really fit into the kernel's I2C layer. The layer is used for master
   tranfsers which are triggered from inside the system, mostly from an
   application.
   
   In slave mode, a master simply sends data to the slave (or wants to read
   data from it, in this case an address should be sent before anyhow to let
   the slave know which master the data should be sent to). This normally
   causes an interrupt in the I2C device driver. Because the driver cannot
   know what this data is for (because no interaction from the userspace
   preceded the slave operation), the data could not be handled properly with
   the I2C framework.
   
   This is an approach to solve this problem. Within the I2C slave subsystem,
   each device driver should register in the slave handler. This allows the
   subsystem to start also master transfers (which are sometimes needed, a
   lot of systems require both master and slave operations) and to
   distinguish between several I2C adapters in the system. On the other side
   of the system, several logical I2C devices can be found. These devices
   also register in the slave subsystem. The i2c-slave module then can ask
   every logical device whether it is responsible for the received I2C slave
   data (or for the data to be sent). So the data can always be delivered
   from or to the correct logical device. If there is no device responsible
   for the data transfer, the incoming data is thrown away and for outgoing
   data an error on the I2C bus should be signalled (or dummy data
   transferred).
*/

#ifndef __I2C_SLAVE_H
#define __I2C_SLAVE_H

/* Includes ----------------------------------------------------------- */

#include <linux/i2c.h>

/* Constants ---------------------------------------------------------- */

/* device operation modes */
/* Take care when using the I2C_MODE_MASTER_ADDRESS mode, it results in 2
   I2C messages. The first one transmits the sender's own I2C address
   (only 7 bit addresses supported) in one byte, the second one transmits
   the I2C data. This flag might only be used with devices that support
   this!
*/
#define I2C_MODE_MASTER			0x04	/* device works in master mode */
#define I2C_MODE_MASTER_ADDRESS		0x02	/* device sends its own address on op. */
#define I2C_MODE_SLAVE			0x01	/* device works in slave mode */

/* ioctl calls */
#define I2C_DEVICE_SET_TRANSMITTER_MODE	0x0735	/* set the operation mode of the device */
#define I2C_DEVICE_SET_RECEIVER_MODE	0x0736	/* set the operation mode of the device */
/* to change the receiver's slave address, use I2C_SLAVE or I2C_SLAVE_FORCE */
/* to flush the device's buffer, use I2C_SLAVE_FLUSH_BUFFER */

/* Data structures ---------------------------------------------------- */

#ifdef __KERNEL__

struct i2c_slave_device;

/* an i2c adapter */
struct i2c_slave_adapter {
	/* list node */
	struct list_head list;

	/* the physical adapter */
	struct i2c_adapter* adapter;
	
	/* when the adapter could not deliver its data to the device,
	   the i2c bus is halted; this function reactivates the i2c bus */
	int (*reactivate)(struct i2c_adapter* adapter);
	
	/* the currently responsible slave device */
	struct i2c_slave_device *responsible_device;
};

/* a logical i2c (slave) device */
struct i2c_slave_device {
	/* list node */
	struct list_head list;

	/* logical name of the device */
	char name[32];

	/* the I2C adapter to be used */
	struct i2c_adapter* adapter;
	
	/* is the device responsible for the current transfer */
	/* returns nonzero if responsible and 0 otherwise */
	int (*is_responsible)(struct i2c_slave_device*, unsigned int, char*, int);
	
	/* read and write functions */
	/* return the actual number of written/read bytes from/to the
	   logical I2C device or < 0 on error */
	int (*slave_write)(struct i2c_slave_device*, char*, int);
	int (*slave_read)(struct i2c_slave_device*, char*, int);
};

/* Functions ---------------------------------------------------------- */

/* add/remove a physical I2C adapter to the subsystem */
int i2c_slave_add_adapter(struct i2c_slave_adapter* slave_adapter);
int i2c_slave_remove_adapter(struct i2c_slave_adapter* slave_adapter);

/* add/remove a logical I2C device to the subsystem */
int i2c_slave_add_device(struct i2c_slave_device* slave_device);
int i2c_slave_remove_device(struct i2c_slave_device* slave_device);

/* move the data from the locigal device into the physical driver */
/* both functions return the number of actual written/read bytes */
int iic_slave_write(struct i2c_adapter* adapter, unsigned int, char* buf, int len);
int iic_slave_read(struct i2c_adapter* adapter, unsigned int, char* buf, int len, int command);
	/* command is the i2c slave command;
	   if command < 0, no command was transferred */

/* unblock a blocked adapter */
void unblock_adapter(struct i2c_adapter* adapter);

#endif /* __KERNEL__ */

#endif /* __I2C_SLAVE_H */
