#!/bin/bash

###############################################################################
# data
###############################################################################

# list of all supported languages
languages[0  ]="aa";	language_names[0  ]="Afar"
languages[1  ]="ab";	language_names[1  ]="Abkhazian"
languages[2  ]="ae";	language_names[2  ]="Avestan"
languages[3  ]="af";	language_names[3  ]="Afrikaans"
languages[4  ]="ak";	language_names[4  ]="Akan"
languages[5  ]="am";	language_names[5  ]="Amharic"
languages[6  ]="an";	language_names[6  ]="Aragonese"
languages[7  ]="ar";	language_names[7  ]="Arabic"
languages[8  ]="as";	language_names[8  ]="Assamese"
languages[9  ]="av";	language_names[9  ]="Avaric"
languages[10 ]="ay";	language_names[10 ]="Aymara"
languages[11 ]="az";	language_names[11 ]="Azerbaijani"
languages[12 ]="ba";	language_names[12 ]="Bashkir"
languages[13 ]="be";	language_names[13 ]="Belarusian"
languages[14 ]="bg";	language_names[14 ]="Bulgarian"
languages[15 ]="bh";	language_names[15 ]="Bihari"
languages[16 ]="bi";	language_names[16 ]="Bislama"
languages[17 ]="bm";	language_names[17 ]="Bambara"
languages[18 ]="bn";	language_names[18 ]="Bengali"
languages[19 ]="bo";	language_names[19 ]="Tibetan"
languages[20 ]="br";	language_names[20 ]="Breton"
languages[21 ]="bs";	language_names[21 ]="Bosnian"
languages[22 ]="ca";	language_names[22 ]="Catalan"
languages[23 ]="ce";	language_names[23 ]="Chechen"
languages[24 ]="ch";	language_names[24 ]="Chamorro"
languages[25 ]="co";	language_names[25 ]="Corsican"
languages[26 ]="cr";	language_names[26 ]="Cree"
languages[27 ]="cs";	language_names[27 ]="Czech"
languages[28 ]="cu";	language_names[28 ]="Church Slavic"
languages[29 ]="cv";	language_names[29 ]="Chuvash"
languages[30 ]="cy";	language_names[30 ]="Welsh"
languages[31 ]="da";	language_names[31 ]="Danish"
languages[32 ]="de";	language_names[32 ]="German"
languages[33 ]="dv";	language_names[33 ]="Divehi"
languages[34 ]="dz";	language_names[34 ]="Dzongkha"
languages[35 ]="ee";	language_names[35 ]="Ewe"
languages[36 ]="el";	language_names[36 ]="Greek"
languages[37 ]="en";	language_names[37 ]="English"
languages[38 ]="eo";	language_names[38 ]="Esperanto"
languages[39 ]="es";	language_names[39 ]="Spanish"
languages[40 ]="et";	language_names[40 ]="Estonian"
languages[41 ]="eu";	language_names[41 ]="Basque"
languages[42 ]="fa";	language_names[42 ]="Persian"
languages[43 ]="ff";	language_names[43 ]="Fulah"
languages[44 ]="fi";	language_names[44 ]="Finnish"
languages[45 ]="fj";	language_names[45 ]="Fijian"
languages[46 ]="fo";	language_names[46 ]="Faroese"
languages[47 ]="fr";	language_names[47 ]="French"
languages[48 ]="fy";	language_names[48 ]="Western Frisian"
languages[49 ]="ga";	language_names[49 ]="Irish"
languages[50 ]="gd";	language_names[50 ]="Scottish Gaelic"
languages[51 ]="gl";	language_names[51 ]="Galician"
languages[52 ]="gn";	language_names[52 ]="Guarani"
languages[53 ]="gu";	language_names[53 ]="Gujarati"
languages[54 ]="gv";	language_names[54 ]="Manx"
languages[55 ]="ha";	language_names[55 ]="Hausa"
languages[56 ]="he";	language_names[56 ]="Hebrew"
languages[57 ]="hi";	language_names[57 ]="Hindi"
languages[58 ]="ho";	language_names[58 ]="Hiri Motu"
languages[59 ]="hr";	language_names[59 ]="Croatian"
languages[60 ]="ht";	language_names[60 ]="Haitian"
languages[61 ]="hu";	language_names[61 ]="Hungarian"
languages[62 ]="hy";	language_names[62 ]="Armenian"
languages[63 ]="hz";	language_names[63 ]="Herero"
languages[64 ]="ia";	language_names[64 ]="Interlingua"
languages[65 ]="id";	language_names[65 ]="Indonesian"
languages[66 ]="ie";	language_names[66 ]="Interlingue"
languages[67 ]="ig";	language_names[67 ]="Igbo"
languages[68 ]="ii";	language_names[68 ]="Sichuan Yi"
languages[69 ]="ik";	language_names[69 ]="Inupiaq"
languages[70 ]="io";	language_names[70 ]="Ido"
languages[71 ]="is";	language_names[71 ]="Icelandic"
languages[72 ]="it";	language_names[72 ]="Italian"
languages[73 ]="iu";	language_names[73 ]="Inuktitut"
languages[74 ]="ja";	language_names[74 ]="Japanese"
languages[75 ]="jv";	language_names[75 ]="Javanese"
languages[76 ]="ka";	language_names[76 ]="Georgian"
languages[77 ]="kg";	language_names[77 ]="Kongo"
languages[78 ]="ki";	language_names[78 ]="Kikuyu"
languages[79 ]="kj";	language_names[79 ]="Kuanyama"
languages[80 ]="kk";	language_names[80 ]="Kazakh"
languages[81 ]="kl";	language_names[81 ]="Kalaallisut"
languages[82 ]="km";	language_names[82 ]="Khmer"
languages[83 ]="kn";	language_names[83 ]="Kannada"
languages[84 ]="ko";	language_names[84 ]="Korean"
languages[85 ]="kr";	language_names[85 ]="Kanuri"
languages[86 ]="ks";	language_names[86 ]="Kashmiri"
languages[87 ]="ku";	language_names[87 ]="Kurdish"
languages[88 ]="kv";	language_names[88 ]="Komi"
languages[89 ]="kw";	language_names[89 ]="Cornish"
languages[90 ]="ky";	language_names[90 ]="Kirghiz"
languages[91 ]="la";	language_names[91 ]="Latin"
languages[92 ]="lb";	language_names[92 ]="Luxembourgish"
languages[93 ]="lg";	language_names[93 ]="Ganda"
languages[94 ]="li";	language_names[94 ]="Limburgish"
languages[95 ]="ln";	language_names[95 ]="Lingala"
languages[96 ]="lo";	language_names[96 ]="Lao"
languages[97 ]="lt";	language_names[97 ]="Lithuanian"
languages[98 ]="lu";	language_names[98 ]="Luba-Katanga"
languages[99 ]="lv";	language_names[99 ]="Latvian"
languages[100]="mg";	language_names[100]="Malagasy"
languages[101]="mh";	language_names[101]="Marshallese"
languages[102]="mi";	language_names[102]="Maori"
languages[103]="mk";	language_names[103]="Macedonian"
languages[104]="ml";	language_names[104]="Malayalam"
languages[105]="mn";	language_names[105]="Mongolian"
languages[106]="mo";	language_names[106]="Moldavian"
languages[107]="mr";	language_names[107]="Marathi"
languages[108]="ms";	language_names[108]="Malay"
languages[109]="mt";	language_names[109]="Maltese"
languages[110]="my";	language_names[110]="Burmese"
languages[111]="na";	language_names[111]="Nauru"
languages[112]="nb";	language_names[112]="Norwegian Bokmal"
languages[113]="nd";	language_names[113]="North Ndebele"
languages[114]="ne";	language_names[114]="Nepali"
languages[115]="ng";	language_names[115]="Ndonga"
languages[116]="nl";	language_names[116]="Dutch"
languages[117]="nn";	language_names[117]="Norwegian Nynorsk"
languages[118]="no";	language_names[118]="Norwegian"
languages[119]="nr";	language_names[119]="South Ndebele"
languages[120]="nv";	language_names[120]="Navajo"
languages[121]="ny";	language_names[121]="Chichewa"
languages[122]="oc";	language_names[122]="Occitan"
languages[123]="oj";	language_names[123]="Ojibwa"
languages[124]="om";	language_names[124]="Oromo"
languages[125]="or";	language_names[125]="Oriya"
languages[126]="os";	language_names[126]="Ossetian"
languages[127]="pa";	language_names[127]="Panjabi"
languages[128]="pi";	language_names[128]="Pali"
languages[129]="pl";	language_names[129]="Polish"
languages[130]="ps";	language_names[130]="Pashto"
languages[131]="pt";	language_names[131]="Portuguese"
languages[132]="qu";	language_names[132]="Quechua"
languages[133]="rm";	language_names[133]="Raeto-Romance"
languages[134]="rn";	language_names[134]="Kirundi"
languages[135]="ro";	language_names[135]="Romanian"
languages[136]="ru";	language_names[136]="Russian"
languages[137]="rw";	language_names[137]="Kinyarwanda"
languages[138]="sa";	language_names[138]="Sanskrit"
languages[139]="sc";	language_names[139]="Sardinian"
languages[140]="sd";	language_names[140]="Sindhi"
languages[141]="se";	language_names[141]="Northern Sami"
languages[142]="sg";	language_names[142]="Sango"
languages[143]="sh";	language_names[143]="Serbo-Croatian"
languages[144]="si";	language_names[144]="Sinhalese"
languages[145]="sk";	language_names[145]="Slovak"
languages[146]="sl";	language_names[146]="Slovenian"
languages[147]="sm";	language_names[147]="Samoan"
languages[148]="sn";	language_names[148]="Shona"
languages[149]="so";	language_names[149]="Somali"
languages[150]="sq";	language_names[150]="Albanian"
languages[151]="sr";	language_names[151]="Serbian"
languages[152]="ss";	language_names[152]="Swati"
languages[153]="st";	language_names[153]="Southern Sotho"
languages[154]="su";	language_names[154]="Sundanese"
languages[155]="sv";	language_names[155]="Swedish"
languages[156]="sw";	language_names[156]="Swahili"
languages[157]="ta";	language_names[157]="Tamil"
languages[158]="te";	language_names[158]="Telugu"
languages[159]="tg";	language_names[159]="Tajik"
languages[160]="th";	language_names[160]="Thai"
languages[161]="ti";	language_names[161]="Tigrinya"
languages[162]="tk";	language_names[162]="Turkmen"
languages[163]="tl";	language_names[163]="Tagalog"
languages[164]="tn";	language_names[164]="Tswana"
languages[165]="to";	language_names[165]="Tonga"
languages[166]="tr";	language_names[166]="Turkish"
languages[167]="ts";	language_names[167]="Tsonga"
languages[168]="tt";	language_names[168]="Tatar"
languages[169]="tw";	language_names[169]="Twi"
languages[170]="ty";	language_names[170]="Tahitian"
languages[171]="ug";	language_names[171]="Uighur"
languages[172]="uk";	language_names[172]="Ukrainian"
languages[173]="ur";	language_names[173]="Urdu"
languages[174]="uz";	language_names[174]="Uzbek"
languages[175]="ve";	language_names[175]="Venda"
languages[176]="vi";	language_names[176]="Vietnamese"
languages[177]="vo";	language_names[177]="Volapuek"
languages[178]="wa";	language_names[178]="Walloon"
languages[179]="wo";	language_names[179]="Wolof"
languages[180]="xh";	language_names[180]="Xhosa"
languages[181]="yi";	language_names[181]="Yiddish"
languages[182]="yo";	language_names[182]="Yoruba"
languages[183]="za";	language_names[183]="Zhuang"
languages[184]="zh";	language_names[184]="Chinese"
languages[185]="zhs";	language_names[185]="Simplified Chinese"
languages[186]="zht";	language_names[186]="Traditional Chinese"
languages[187]="zu";	language_names[187]="Zulu"

###############################################################################
# helper functions
###############################################################################

# display usage message
usage() {
    echo "Usage:"
    echo ""
    echo "  $0 <command> <options>"
    echo ""
    echo "  possible commands:"
    echo ""
    echo "      -h, -?  .............. display this help"
    echo "      -l  .................. display a list of supported languages"
    echo "      -c <language> ........ create translation files for the given language"
    echo "      -u <language> ........ update translation files for the given language"
    echo "      -d <language> ........ delete translation files for the given language"
    echo "      -z <zip file> ........ create zip file to upload and test localization"
    echo "      -g <tgz file> ........ create tgz file to add to firmware binary"
    echo "      -f <firmware file> ... compile everything and patch firmware file"
    echo ""
}

# list all languages
list_all_languages() {
    echo "Supported languages:"
    echo ""
    local i="0"
    while [ $i -lt ${#languages[*]} ]; do
        echo "    ${languages[$i]} - ${language_names[$i]}"
        i=$[$i+1]
    done
    echo ""
}

# display a text and query for yes/no
# returns 1 if the answer was yes and 0 otherwise
yesno() {
    echo "$1"
    read answer
    local is_yes=0
    if [ "$answer" = "yes" -o "$answer" = "y" ]; then
        is_yes=1
    fi
    return $is_yes
}                                                                         

# checks whether the given language is supported
# returns 1 if it is supported or 0 if not
# this also displays an error message, so the caller doesn't need to do this
language_is_supported() {
    local lang="$1"
    
    if [ "$lang" = "en" ]; then
        echo "English is the default language and cannot be changed."
        return 0
    fi
    
    for i in ${languages[@]}; do
        if [ "$i" = "$lang" ]; then
            return 1
        fi
    done
    
    echo "The given language $lang is not supported."
    return 0
}

###############################################################################
# create a new language
###############################################################################

# create .po file for a new language
# parameter $1: language to create .po file for
create_new_language() {
    if [ "$1" = "" ]; then
        echo "No language given!"
        usage
        return -1
    fi

    local lang="$1"
    
    language_is_supported "$lang"
    if [ $? -ne 1 ]; then
        return -1
    fi

    echo "Creating translation stubs for language $lang."
    
    # check whether we already have $lang.po and ask the user whether he really wants to overwrite it
    if [ -e "$lang.po" ]; then
        yesno "Translation file $lang.po already exists. Are you sure that you want to proceed (yes/no)?"
        if [ $? -ne 1 ]; then
            echo "Aborted"
            return 0
        fi
    fi
    
    # create $lang.po; use $lang.full.po if we have it
    if [ -e "$lang.full.po" ]; then
        cp "$lang.full.po" "$lang.po"
        msgmerge -N -U "$lang.po" erla_all.pot
    else
        msginit -i erla_all.pot -o "$lang.po" -l "$lang"
    fi
    
    local ret=$?
    if [ $ret -ne 0 ]; then
        echo "Error creating translation file."
        return $ret
    fi
    
    echo "Successfully created translation stub file $lang.po"
    echo ""
    return 0
}


###############################################################################
# update language files
###############################################################################

#update language .po file
update_language() {
    if [ "$1" = "" ]; then
        echo "No language given!"
        usage
        return -1
    fi

    local lang="$1"

    language_is_supported "$lang"
    if [ $? -ne 1 ]; then
        return -1
    fi

    echo "Updating translation files for language $lang."
    
    # check whether we already have $lang.po
    if [ ! -e "$lang.po" ]; then
        echo "Language file $lang.po does not exist, cannot update it! You need to create it before."
        return -1
    fi
    
    # check whether we have a language catalog file with the SDK
    local use_cat=0
    if [ -e "$lang.full.po" ]; then
        yesno \
"There comes a language catalog file with this localization SDK.
This will include some new translations, but might overwrite some
of your previously made translations.
Do you want to include this catalog file (yes/no)?"
        if [ $? -eq 1 ]; then
            use_cat=1;
        fi
    fi
    
    # now update the .po file
    if [ $use_cat -eq 1 ]; then
        msgmerge -N -U "$lang.po" -C "$lang.full.po" erla_all.pot
    else
        msgmerge -N -U "$lang.po" erla_all.pot
    fi
    
    local ret=$?
    if [ $ret -ne 0 ]; then
        echo "Error updating translation file."
        return $ret
    fi
    
    echo "Successfully updated translation file $lang.po"
    echo ""
    return 0
}

###############################################################################
# delete language files
###############################################################################

# delete language files
delete_language() {
    if [ "$1" = "" ]; then
        echo "No language given!"
        usage
        return -1
    fi

    local lang="$1"

    echo "Deleting translation files for language $lang."
    
    # check whether we have $lang.po and ask for user confirmation
    if [ -e "$lang.po" ]; then
        yesno "A translation file $lang.po was found. This might contain a lot of translations which will get lost. Are you sure that you want to proceed (yes/no)?"
        if [ $? -ne 1 ]; then
            echo "Aborted"
            return 0
        fi
    fi
    
    # delete all translation files
    rm -f "$lang.po" "$lang.mo" "rclang_$lang.jar"

    echo "Successfully deleted translation files for $lang."
    echo ""
    return 0
}

###############################################################################
# common functionality for updating the compiled translation files
###############################################################################

# get all languages which we have .po files for
get_po_languages() {
    echo `ls *.po 2> /dev/null | grep -v '.full.po' | grep -v '.java.po' | grep -v '^en.po' | sed 's/.po$//'`
}

# compile .po file to .mo file
compile_po_to_mo_file() {
    if [ "$1" = "" ]; then
        echo "No language given!"
        usage
        return -1
    fi

    local lang="$1"
    
    echo "Compiling translation file for language $lang."
    msgfmt -o "$lang.mo" "$lang.po"

    local ret=$?
    if [ $ret -ne 0 ]; then
        echo "Error compiling translation file."
        return $ret
    fi
    
    echo "Successfully compiled translation file $lang.po"
    return 0
}

# compile all .po to .mo files
compile_all_po_to_mo_files() {
    for i in `get_po_languages`; do
        compile_po_to_mo_file $i
        local ret=$?
        if [ $ret -ne 0 ]; then
            return $ret
        fi
    done
    return 0
}

# compile .po file to .jar file
compile_po_to_jar_file() {
    if [ "$1" = "" ]; then
        echo "No language given!"
        usage
        return -1
    fi

    local lang="$1"
    
    echo "Compiling translation file for language $lang (Java)."
    
    # find all the applet related strings
    msggrep --force-po -N nn/pp/*/*.java -o "$lang.java.po" "$lang.po"

    local ret=$?
    if [ $ret -ne 0 ]; then
        echo "Error compiling translation file."
        return $ret
    fi
    
    # compile it
    rm -rf "java_$lang"
    mkdir -p "java_$lang"
    msgfmt -j -d "java_$lang" -l "$lang" -r nn.pp.rclang.Lang "$lang.java.po"
    ret=$?
    rm -f "$lang.java.po"
    
    if [ $ret -ne 0 ]; then
        echo "Error compiling translation file."
        return $ret
    fi
    
    # pack it into a jar archive
    cd "java_$lang"
    jar cvf "../rclang_$lang.jar" *
    ret=$?
    cd ..
    rm -rf "java_$lang"

    if [ $ret -ne 0 ]; then
        echo "Error compiling translation file."
        return $ret
    fi
    
    echo "Successfully compiled translation file $lang.po"
    return 0
}

# compile all .po to .jar files
compile_all_po_to_jar_files() {
    local languages=`get_po_languages`
    
    # get_po_languages doesn't return english, so add it here
    if [ -e en.po ]; then
        languages="en $languages"
    fi

    for i in $languages; do
        compile_po_to_jar_file $i
        local ret=$?
        if [ $ret -ne 0 ]; then
            return $ret
        fi
    done
    return 0
}

# compile all .po files to .mo and .jar
compile_all_po_files() {
    compile_all_po_to_mo_files
    local ret=$?
    if [ $ret -ne 0 ]; then
        return $ret
    fi

    compile_all_po_to_jar_files
    ret=$?
    if [ $ret -ne 0 ]; then
        return $ret
    fi
    
    echo ""
    echo "Successfully compiled all translation files."
    echo ""
    
    return 0
}

# get a list of all files which have to be packaged
get_file_list() {
    ls rclang_*.jar *.mo
    return 0
}

###############################################################################
# creating a zip file
###############################################################################

# create a zip file
create_zip_file() {
    if [ "$1" = "" ]; then
        echo "No filename given!"
        usage
        return -1
    fi

    local filename="$1"

    local langs=`get_po_languages`
    
    # compile all 
    compile_all_po_files
    local ret=$?
    if [ $ret -ne 0 ]; then
        echo "Error compiling the language files."
        return $ret
    fi
    
    # pack them into the zip file
    if [ -e $filename ]; then
        yesno "The zip file $filename already exists. If you proceed, this file will be overwritten. Are you sure that you want to proceed (yes/no)?"
        if [ $? -ne 1 ]; then
            echo "Aborted"
            return 0
        fi
    fi
    
    rm -f $filename
    zip $filename `get_file_list`
    if [ $ret -ne 0 ]; then
        echo "Error creating zip file."
        return $ret
    fi
    
    return 0;
}

# create a tgz file
create_tgz_file() {
    if [ "$1" = "" ]; then
        echo "No filename given!"
        usage
        return -1
    fi

    local filename="$1"
    local force="$2"
    
    if [ "$force" = "" ]; then
        force=1
    fi

    local langs=`get_po_languages`
    
    # compile all 
    compile_all_po_files
    local ret=$?
    if [ $ret -ne 0 ]; then
        echo "Error compiling the language files."
        return $ret
    fi
    
    # pack them into the zip file
    if [ -e $filename -a $force -ne 1 ]; then
        yesno "The tgz file $filename already exists. If you proceed, this file will be overwritten. Are you sure that you want to proceed (yes/no)?"
        if [ $? -ne 1 ]; then
            echo "Aborted"
            return 0
        fi
    fi
    
    rm -f $filename
    tar -cvzf $filename `get_file_list`
    if [ $ret -ne 0 ]; then
        echo "Error creating tgz file."
        return $ret
    fi
    
    return 0;
}

# creaet a tgz file and patch firmware then
patch_firmware() {
    if [ "$1" = "" ]; then
        echo "No filename given!"
        usage
        return -1
    fi

    local filename="$1"
    
    # sanity check
    if [ ! -e "$filename" ]; then
        echo "Error: given firmware file doen not exist!"
        usage
        return -1
    fi
    
    # creating .tgz file
    create_tgz_file "__tmp__.tgz" 1
    local ret=$?
    if [ $ret -ne 0 ]; then
        echo "Error compiling the language files."
        return $ret
    fi
    
    # patch the firmware
    ./replacepart_i18n.pl -a "__tmp__.tgz" "$filename"
    ret=$?
    if [ $ret -ne 0 ]; then
        echo "Error patching firmware."
        return $ret
    fi
    
    #delete .tgz file
    rm -f "__tmp__.tgz"
    
    echo ""
    echo "Successfully patched firmware file $filename."
    echo ""
    
    return 0
}

###############################################################################
# program main function
###############################################################################

if [ "$1" = "" ]; then
    echo "No command given!"
    usage
    exit 0
fi
case "$1" in
    "-h")usage;;
    "-?")usage;;
    "-l")usage; list_all_languages;;
    "-c")create_new_language "$2";;
    "-u")update_language "$2";;
    "-d")delete_language "$2";;
    "-z")create_zip_file "$2";;
    "-g")create_tgz_file "$2" 0;;
    "-f")patch_firmware "$2";;
    *)echo "Unknown command: $1"; usage "$0";;
esac
