#!/usr/bin/perl -w
###############################################################################
#
# mkfirmware - creates two firmware files: old-<filename> and new-<filename>
#
#              The old-style firmware is understood by all firmware releases
#              since 01.01.04. The new-style firmware can be used since
#              firmware release 01.04.00 and includes a build number and a
#              tag (arbitrary string with up to 64 characters).
#

package mkfirmware;

use strict;

#
###############################################################################
# imports
#

use FindBin;
use lib "$FindBin::Bin";
use lib "$FindBin::Bin/cpan";

use Getopt::Std;
use File::Copy;
use File::Path;
use eRIC_Firmware;

#
###############################################################################
# global variables
#

my $debug = 0;
my $col_width = 22;

my $filename;
my $opts  = {};
my $parts = [];
my @include_ids;
my @exclude_ids;
my $usage;
my $usage_base =
  "Usage: \n\n" .
  "  mkfirmware [-f <filename>] [-v <version>]\n".
  "             [-m <min_req_ver>] [-M <min_down_ver>]\n" .
  "             [-t <tag>] [-P <product>] [-b <board>] [-B <subboard>]\n".
  "             [-o <oem>] [-l <lightoem>]\n".
  "             [-h <include_hwids>] [-H <exclude_hwids>] [-W] [-T] [-R] [-O] [-I] \n" .
  "             [-F <flash_map_sel>] [-g <glbl_hdr_ver>]\n" .
  "             -p <part_types>  -s <part_subtypes> <file1> ... <fileN>\n\n" .

  "    -f <filename>      - filename of the firmware file\n" .
  "                         if omitted, the following is used:\n".
  "fw-<board>-<subboard>-<product>-<oem>[_<lightoem>]_<version>-<buildnr>.bin\n" .
  "    -v <version>       - the firmware, if omitted taken from version repo\n". "                          (6 digits)\n" .
  "    -m <min_req_ver>   - minimum required version to update to this fw\n".
  "                         (6 digits)\n" .
  "    -M <min_down_ver>  - minimum allowed downgrade version for this FW\n".
  "                         (6 digits)\n" .
  "    -t <tag>           - optional firmware tag (default: \"Standard Edition\"\n" .
  "    -P <product>       - product type, if omitted taken from\n".
  "                         Config.sh -> \$PP_PRODUCT\n" .
  "    -b <board>         - lara or kira, if omitted taken from\n".
  "                         Config.sh->\$PP_BOARD\n" .
  "    -B <subboard>      - suboard (eg kim), if omitted taken from\n".
  "                         Config.sh->\$PP_SUBBOARD\n" .
  "    -o <oem>           - OEM type,     if omitted taken from\n".
  "                         Config.sh -> \$PP_OEM\n" .
  "    -l <lightoem>      - LIGHTOEM tag\n" .
  "                         if omitted taken from Config.sh -> \$LIGHTOEM\n" .
  "    -h <include_hwids> - list of valid hardware ids for this fw,\n" .
  "                         comma separated, hex, <=0xFF,\n".
  "                         use \$PP_HWID_INT if empty\n" .
  "    -H <exclude_hwids> - list of invalid hardware ids for this fw,\n" .
  "                         comma separated, hex, <=0xFF, exclude none if empty\n" .
  "    -g <glbl_hdr_ver>  - force a global FW header version\n".
  "                         - useful for transition firmwares\n" .
  "    -W                 - disable twin mode \n".
  "                         (i.e. don't create two FW with consecutive \n" .
  "                         build numbers)\n".
  "    -T                 - do *NOT* tag the repo, tag name will be created\n".
  "    -R                 - no Repo operations, don't try to access the CVS,\n".
  "                         for people on the Road (fix build number 9999)\n".
  "    -O                 - if set OEM SDK is created\n\n".
  "    -O                 - if set i18n SDK is created\n\n".
  "    -F <flash_map_sel> - the flash mapping to use:\n".
  "			      0=map_ramfs (Kernel, Initrd, U-boot, KME)\n".
  "			      1=map_jffs  (U-boot, RootFS in JFFS, KME)\n".
  "			    - if omitted, consults pp_features.sh for \n".
  "                           PP_FEAT_JFFS2_ROOTFS\n".
  "    -p <part_type>     - the partition type\n" .
  "                           0=flash, 1=kme_8515, 2=kme_m16l, 3=smartipc, \n".
  "                           4=flash dir,\n" .
  "                           5=ADC atmel, 6=Master Console, 7=Max II jam\n" .
  "                           8=RPC Branch Controller\n" .
  "    -s <part_subtype>  - the partition subtype, \n".
  "                         for details see table in eRIC_Firmware.pm\n" .
  "                           (p=0 [flash]) initrd: 0 (zImage.bin)\n".
  "                                                 1 (initrd.bin)\n".
  "                                                 3 (u-boot.bin)\n" .
  "                                         jffs2:  1 (u-boot.bin)\n" .
  "                           (p=2 [kme16l])      4 (kme_m16l.bin)\n" .
  "                           (p=4 [flash-dir])   0 (oem flashdirectory)\n".
  "                                               1 (jffs2-rootfs)\n" .
  "                           (p=8 [PSoC])  0 (cy8c27)\n".
  "                                         1 (cy8c26)\n".
  "    <fileN>            - the partition files in same order as -p and -s\n" .
  "                         must be the last option of the commandline\n\n";

my $usage_example =
  "Example (note: this will generate 2 fw bins and tag the repo): \n\n" .
  "  ./mkfirmware.pl -O -p 0002 -s 0134 zImage.bin initrd.bin u-boot.bin kme_m16l.bin\n\n" .
  "  ./mkfirmware.pl -O -p 0004 -s 0130 zImage.bin initrd.bin u-boot.bin oem_flashdisk_files.tgz\n\n" .
  "  ./mkfirmware.pl -O -I -p 000447 -s 013020 zImage.bin initrd.bin u-boot.bin oem_flashdisk_files.tgz i18n_flashdisk_files.tgz dvo.jbc\n\n";

my $usage_example_jffs2 =
  "Basic Example (note: this will tag the repo): \n\n" .
  "  ./mkfirmware.pl -p 042 -s 114 u-boot.bin rootfs.tgz kme_m16l.bin\n\n" .
  "OEM Example (note: this will tag the repo): \n\n" .
  "  ./mkfirmware.pl -O -p 0442 -s 1104  u-boot.bin rootfs.tgz oem.tgz kme_m16l.bin\n\n";

my $dir_admin           = "buildnr";
my $dir_version         = "version";
my $dir_buildnr         = $dir_admin."/eric_admin/versions/";
my $fn_fwbuildnr        = "fwbuildnr";
my $fn_fwbuildnr_offset = "fwbuildnr_offset";
my $fn_fwversion        = "fwversion";

#
###############################################################################
# OEM specific data
#

#                    product_oem
my $min_req_vers = { "lara_peppercon"	=> "030205",
		     "1601ip_peppercon"	=> "040000",
		     "0801ip_peppercon" => "040000",
		     "1601ip_raritan"   => "040000",
		     "0801ip_raritan"   => "040000",
		     "0801iptt_peppercon" => "040000",
		     "1601iptt_peppercon" => "040000",
		     "lara_daxten"	=> "040000",
		     "lara_blackbox"	=> "030205",
		     "lara_nti"		=> "030205",
		     "wingle_peppercon"	=> "030500",
		     "wingle_raritan"	=> "030500",
		     "lara_exer"	=> "030205",
		     "lara_adder"       => "030205",
		     "lara_lindy"       => "030205",
		     "rc1_minicom"	=> "010000",
		     "rc1_peppercon"    => "040000",
		     "rc1_syelectronics"=> "040000",
		     "rc1_raritan"	=> "040000",
		     "ericxp_peppercon"	=> "040000",
		     "ericg4_peppercon" => "040100",
		     "laraxp_peppercon"	=> "040000",
		     "lara_digi"	=> "010000",
		     "lara_minicom"	=> "010007",
		     "lara_xceedium"	=> "010007",
		     "lara_sentinel"	=> "010007",
		     "lara_belkin"	=> "010000",
		     "rc1_tripplite"    => "030401",
		     "ripckimxn_belkin"	=> "040003",
	    	     "eric2_peppercon"	=> "040000",
	    	     "flashx4_peppercon"=> "040000",
	    	     "kimtestermst_peppercon" => "040000",
		     "laraxp_lantronix" => "040000",
	             "amddc_peppercon"  => "039900",
	             "msidc_peppercon"  => "039900",
	             "smidc_peppercon"  => "039900",
	             "inteldc_peppercon"=> "039900", 
	             "icpmmd_peppercon" => "040200", 
	             "pdu_peppercon"    => "000900", 
                     "lara_bmc_test_peppercon" => "040000",
	     	     "asmidc_peppercon" => "040000",
	     	     "asmidc_intel" => "040201" };

#
###############################################################################
# hardware ID exceptions (single id is mapped to multiples, e.g. LARA <= 1.3)
#

my $hwid_exceptions = { "0x03" => "0x01,0x02,0x03" };

#
###############################################################################
# prototypes
#
sub parse_hwids($\@);
sub get_buildnr($$$);
sub get_fwversion($$$);
sub get_oem_filepath($$$$$);
sub update_buildnr_tree();
sub commit_buildnr_tree();
sub get_cvs_root();
sub get_config_sh_var($);
sub tag_repo($$$$$$$);
sub build_production_fs($$$$$$$$$);
sub build_oem_sdk($$$$$);
sub build_i18n_sdk($$$$$);
sub build_documentation();
sub set_initrd_version( $$$$$ );

#
###############################################################################
# main program
#

getopts('f:v:m:M:t:P:b:B:o:l:h:H:F:g:p:s:WTROI', $opts);

# '-F' - flash map selector
my $flash_map_name;
my $dflt_map = (get_config_sh_var("PP_FEAT_JFFS2_ROOTFS") eq "1") ? "map_jffs" : "map_ramfs";
my $hint = "";
my $sel;
if (defined $opts->{"F"}) {
    $sel = get_flash_map_sel($opts->{"F"});
} else {
    $sel = get_flash_map_sel($dflt_map);
}
$flash_map_name = get_flash_map_name($sel);
printf ("%-*s: %s\n", $col_width, "Flash mapping", $flash_map_name);

# usage
if ($flash_map_name eq "map_jffs") {
	$usage = $usage_base . $usage_example_jffs2;
} else {
	$usage = $usage_base . $usage_example;
}

# -----------------------------------------------------------------------------
# check command line options
# -----------------------------------------------------------------------------

# -p is required
defined $opts->{p} or die "$usage";


# twin mode
my $twin_mode = 1;
if (defined $opts->{W}) {
    $twin_mode = 0;
}

# -R: do not access repo
# else: get the version information tree into our file system
defined $opts->{R} or update_buildnr_tree();


# argument of -b is the device the firmware is made for
defined $opts->{b} or $opts->{b} = get_config_sh_var("PP_BOARD");
$opts->{b} =~ /^(lara|kira)$/ or
    die "-b parameter is incorrect\n\n$usage";

defined $opts->{B} or $opts->{B} = get_config_sh_var("PP_SUBBOARD");

# argument of -o is the tag for the OEM variant
defined $opts->{o} or $opts->{o} = get_config_sh_var("PP_OEM");

# argument of -l is the tag for the LIGHTOEM variant
defined $opts->{l} or $opts->{l} = get_config_sh_var("PP_LIGHTOEM");

# argument of -P is the product type
defined $opts->{P} or $opts->{P} = get_config_sh_var("PP_PRODUCT");


# argument of -v is the firmware version
if ( defined $opts->{v} ) {
  $opts->{v} =~ /^[0-9]{6}$/ or die "-v parameter is incorrect\n\n$usage";
} else {
  if (defined $opts->{R}) {
    # repoless mode, create stupid number
    $opts->{v} = "999999"
  } else {
    $opts->{v} = get_fwversion($opts->{b}, $opts->{P}, $opts->{o});
  }
}

# argument of -m is the minimum required version for this firmware,
# value from the table is used if no value given
unless (defined $opts->{"m"}) {
  $opts->{"m"} = $min_req_vers->{"$opts->{P}_$opts->{o}"}
}
if (defined $opts->{"m"}) {
  $opts->{m} =~ /^[0-9]{6}$/ or
    die "Incorrect usage of -m \n\n$usage";
} else {
  die "No minimum required version found, use -m or put it into mkfirmware.pl (\$min_req_vers)\n\n$usage";
}

# argument of -M is the minimum downgrade version for this firmware
# currently it is assumed that the minimum downgrade version is properly compiled in the firmware
if (defined $opts->{M}) {
  $opts->{M} =~ /^[0-9]{6}$/ or
    die "Incorrect usage of -M \n\n$usage";
}

# argument of -t contains an optional tag for the firmware;
#     (in twin mode, it affects only the 1st FW - the beta)
my $fw_tag_1 = "Beta";
my $fw_tag_2 = "Standard Edition";
if (defined $opts->{t}) {
  $opts->{t} =~ /^[-_,.;:()A-Za-z0-9 ]{0,64}$/ or
    die "-t parameter is incorrect\n\n$usage";
  $fw_tag_1 = $opts->{t};
}


# argument of -p and -s contains a list of flash partition numbers in which a part
# is written to. The order corresponds to the order of the specified parts on
# the commandline
defined $opts->{p} or die "-s parameter missing\n\n$usage";
$opts->{p} =~ /^[0-8]{1,10}$/ or die "-p parameter is incorrect\n\n$usage";

defined $opts->{s} or die "-s parameter missing\n\n$usage";
$opts->{s} =~ /^[0-9]{1,10}$/ or die "-s parameter is incorrect\n\n$usage";

if (length($opts->{p}) != length($opts->{s})) {
  die "number of entries for -p and -s parameters must be equal"
}

# parse hardware id lists, possible replace using exceptions
defined $opts->{h} or $opts->{h} = get_config_sh_var("PP_HWID_INT");
foreach my $exception (keys %$hwid_exceptions) {
  if ($exception eq $opts->{h}) {
    print "HWID exception \"$exception\", replacing with \"$hwid_exceptions->{$exception}\"\n";
    $opts->{h} = $hwid_exceptions->{$exception};
  }
}
parse_hwids($opts->{h}, @include_ids) or die "error on include hwids\n" ;
if (defined $opts->{H}) {
  parse_hwids($opts->{H}, @exclude_ids) or die "error on exclude hwids\n" ;
}


# HACK: on fast machines there is a problem when the 
# checkout/change/commit happens in the same second,
# so we wait a bit here
sleep 1;

my $fwbuildnr_1;
my $fwbuildnr_2;
if (defined $opts->{R}) {
  # repo-less mode, create invalid build number
  $fwbuildnr_1 = $fwbuildnr_2 = "9999";
} else {
  $fwbuildnr_1 = get_buildnr($opts->{b}, $opts->{P}, $opts->{o});
  $fwbuildnr_2 = "";
  if ($twin_mode) {
    $fwbuildnr_2 = get_buildnr($opts->{b}, $opts->{P}, $opts->{o});
  }
}


# argument of -f is the firmware file name
my ($filename_1_base, $filename_1, $filename_2_base, $filename_2, $lightoem_infix);
if ($opts->{l} ne "") {
    $lightoem_infix = "_".$opts->{l};
} else {
    $lightoem_infix = "";
}
if (defined $opts->{f}) {
    $filename_1_base = $opts->{f};
    $filename_1 = $filename_1_base;
} else {
    $filename_1_base = $opts->{b}."-".$opts->{B}."-".$opts->{P}."-".$opts->{o}.${lightoem_infix}."_".$opts->{v}."-".$fwbuildnr_1;
    $filename_1 = "fw-".$filename_1_base.".bin";
}
if (not $twin_mode) {
    printf ("%-*s: %s\n", $col_width, "FW file", $filename_1);
} else {
    $filename_2_base = $opts->{b}."-".$opts->{B}."-".$opts->{P}."-".$opts->{o}.${lightoem_infix}."_".$opts->{v}."-".$fwbuildnr_2;
    $filename_2 = "fw-".$filename_2_base.".bin";
    printf ("%-*s: %s (Tag: %s)\n", $col_width, "FW file (Beta)", $filename_1, $fw_tag_1);
    printf ("%-*s: %s (Tag: %s)\n", $col_width, "FW file (Std Edition)", $filename_2, $fw_tag_2);
}

# -g - force a specific global FW header version 
#    - useful, when a customer has an FW with, say, hdr ver 2 and wants to
#      update to an FW with hdr ver 3 -> transition FW is needed, with
#      FW_HDR_VER 3 in libpp_firmware, but ver 2 in the FW header
my $glbl_hdr_ver = "";
if (defined $opts->{"g"}) {
    my $ver = $opts->{"g"};
    if ($ver == "2" || $ver == "3") {
	$glbl_hdr_ver = $ver;
    } else {
      die "ERROR: unsupported global header version given with -g!"
    }
}

# -----------------------------------------------------------------------------
# some preprocessing for calls to the firmware manipulation routines
# -----------------------------------------------------------------------------

my @part_types = split //, $opts->{p};
my @part_subtypes  = split //, $opts->{s};
my $rootfs_file = "";

my $nr=0;
foreach my $part_subtype (@part_subtypes) {
    my $file;
    my $part_type = $part_types[$nr++];

    $file = shift @ARGV;

    foreach my $p (@{$parts}) {
	if ($p->{"part_subtype"} == $part_subtype &&
	    $p->{"part_type"} == $part_type) {
	    die "Error: Partition subtype $part_subtype specified twice for partition type $part_type.\n\n$usage";
	}
    }

    # find root filesystem file for next section
    if ($flash_map_name eq "map_jffs") {
      if ($part_type == 4 and $part_subtype == 1) { $rootfs_file = $file; }
    } else {
      if ($part_type == 0 and $part_subtype == 1) { $rootfs_file = $file; }
    }

    my $part;
    $part->{"part_subtype"} = $part_subtype;
    $part->{"file"} = $file;
    $part->{"part_type"} = $part_type;
    defined $file or
      die "Error: More partitions than filenames specified.\n\n$usage";

    push @{$parts}, $part;
}

# -----------------------------------------------------------------------------
# create the firmware file, patch version information
# -----------------------------------------------------------------------------

if (!(defined($opts->{M}))) {
  $opts->{M} = "";
}

set_initrd_version($rootfs_file, $opts->{v}, $fwbuildnr_1, $opts->{M}, $fw_tag_1);
make_firmware($parts, $opts->{v}, $fwbuildnr_1, $fw_tag_1, $opts->{b}, $opts->{o}, $opts->{P}, $opts->{"m"}, @include_ids, @exclude_ids, $flash_map_name, $glbl_hdr_ver) or exit 1;
save_firmware($filename_1) or exit 1;

if ($twin_mode) {
    set_initrd_version($rootfs_file, $opts->{v}, $fwbuildnr_2, $opts->{M}, $fw_tag_2);
    make_firmware($parts, $opts->{v}, $fwbuildnr_2, $fw_tag_2, $opts->{b}, $opts->{o}, $opts->{P}, $opts->{"m"}, @include_ids, @exclude_ids, $flash_map_name, $glbl_hdr_ver) or exit 1;
    save_firmware($filename_2) or exit 1;
}

# -----------------------------------------------------------------------------
# final operations: tag the repo, build OEM-SDK and production filesystem
# -----------------------------------------------------------------------------

# commit version information, if everything went well
defined $opts->{R} or commit_buildnr_tree();

# tag repo, if not prohibited by the user
if ((not defined $opts->{T}) and (not defined $opts->{R})) {
    tag_repo($opts->{b}, $opts->{B}, $opts->{P}, $opts->{v}, $fwbuildnr_1, $fwbuildnr_2, $opts->{o});
}

# build, pack filesystem to copy to production and pack oem sdk template
my ($prod_buildnr, $prod_tag);
if ($twin_mode) {
    # twin mode -> _second_ FW is the Standard Edition
    $prod_buildnr = $fwbuildnr_2;
    $prod_tag = $fw_tag_2;
} else {
    $prod_buildnr = $fwbuildnr_1;
    $prod_tag = $fw_tag_1;
} 
if (defined $opts->{"O"}) {
     build_oem_sdk($opts->{P}, $opts->{v}, $prod_buildnr, $opts->{o}, $opts->{l});
}
if (defined $opts->{"I"}) {
     build_i18n_sdk($opts->{P}, $opts->{v}, $prod_buildnr, $opts->{o}, $opts->{l});
}

# copy build number over to mksdk dir
my $sdkbfile = get_config_sh_var("FW_TOPDIR")."/mksdk/fwbuildinfo";
open(F_SDKBNR, ">".$sdkbfile) or die "Error: ".$!." -> sdkbfile '".$sdkbfile."'\n";
print F_SDKBNR $filename_1_base."\n";
close(F_SDKBNR);
if ($debug) { print "sdk build number file   : ".$sdkbfile."\n" }


#
###############################################################################
# parse hardware ids to array
#
sub parse_hwids ($\@) {
  my $id_string = shift;
  my $hw_ids = shift;
  my $hw_id;

  @$hw_ids = split(/,/, $id_string);

  if ($#{@$hw_ids} > 15) {
    print "Too many entries, maximum is 16\n";
    return 0;
  }

  foreach $hw_id (@$hw_ids) {  
    if (hex($hw_id) > 255) {
      print "Hardware IDs > 0xFF not allowed\n";
      return 0;
    }
    $hw_id = hex($hw_id);
  }

  return 1;
}


#
###############################################################################
# some utils for to retrieve version information
#

sub tag_repo($$$$$$$) {
  my $board = shift;
  my $subboard = shift;
  my $product = shift;
  my $version = shift;
  my $buildnr_1 = shift;
  my $buildnr_2 = shift;
  my $oem = shift;
  my $buildnr =  $buildnr_1 . ($buildnr_2 ? "_$buildnr_2" : "");
  my $tag = "fw-".$board."-".$subboard."-".$product."-".$oem."_".$version."-".$buildnr;

  my @cvscall = ("cvs", "tag", "-c",  $tag);
  chdir "..";
  printf ("%-*s: %s\n", $col_width, "CVS repo tag", $tag);
  system(@cvscall) == 0 or die "Error: system @cvscall failed: $?\n";
  chdir "mkfirmware";
}

sub get_buildnr($$$) {
  my $device = shift;
  my $product = shift;
  my $oem = shift;
  my $path = $dir_buildnr . $fn_fwbuildnr;
  my $path_offset = get_oem_filepath($dir_buildnr, $fn_fwbuildnr_offset, 
				     $device, $product, $oem);
  my $fwbuildnr;
  my $fwbuildnr_offset;

  # get base build number
  open(F_BNR, "+<".$path) or die "Error: ".$!." -> path'".$path."'\n";
  if ($debug) { print "build number file       : ".$path."\n" }
  chop($fwbuildnr = <F_BNR>);
  seek(F_BNR, 0, 0);
  print F_BNR ++$fwbuildnr."\n";
  close(F_BNR);

  # try to get an oem offset
  if ($path_offset ne "") {
    open(F_BNR, "+<".$path_offset) or die "Error: ".$!." -> path_offset '".$path_offset."'\n";
    if ($debug) { print "build number offset file: ".$path_offset."\n" }
    chop($fwbuildnr_offset = <F_BNR>);
    $fwbuildnr += $fwbuildnr_offset;
  }

  return $fwbuildnr;
}

sub get_fwversion($$$) {
  my $device = shift;
  my $product = shift;
  my $oem = shift;
  my $path = get_oem_filepath($dir_version, $fn_fwversion, $device,
                              $product, $oem);
  my $fwversion;

  open(F_FWV, $path) or die "Error: ".$!." -> path '".$path."'\n";
  if ($debug) { print "firmware version file   : ".$path."\n" }
  chop($fwversion = <F_FWV>);
  close(F_FWV);

  return $fwversion;
}

sub get_oem_filepath($$$$$) {
  my $base = shift;
  my $filename = shift;
  my $device = shift;
  my $product = shift;
  my $oem = shift;
  my $path;

  if ($device ne $product) {
    $path = $base."/".$device."_".$product."_".$oem."/".$filename;
    if ( -e $path ) {
      return $path;
    }
  }

  $path = $base."/".$device."_".$oem."/".$filename;
  if ( -e $path ) {
    return $path;
  }

  $path = $base."/".$filename;
  if ( -e $path ) {
    return $path;
  }

  return "";
}

sub update_buildnr_tree() {
  my @cvscall = ("cvs", "-d",  get_cvs_root(), "-Q", "checkout",
		 "eric_admin/versions");
  unless ( -d $dir_admin ) {
    mkdir $dir_admin, 0770 or die "Error: ".$!." -> dir_admin' ".$dir_admin."'\n";
  }
  unlink $dir_buildnr . $fn_fwbuildnr;
  chdir $dir_admin;
  system(@cvscall) == 0 or die "Error: system @cvscall failed: $?\n";
  chdir "..";
}

sub commit_buildnr_tree() {
  my @cvscall = ("cvs", "-Q", "commit", "-m", "by mkfirmware.pl",
		 $dir_buildnr . $fn_fwbuildnr);
  system(@cvscall) == 0 or die "Error: system @cvscall failed: $?\n";
}

sub get_cvs_root() {
  my $cvsroot = `cat CVS/Root`;
  chop($cvsroot);
  return $cvsroot;
}

sub get_config_sh_var($) {
  my $key = shift;
  my $value = `source ../Config.sh; echo \$$key`;
  chop($value);
  return $value;
}

#
###############################################################################
# creates the OEM website documentation
#
sub build_documentation() {

  my $fw_topdir = get_config_sh_var("FW_TOPDIR");
  my $liberic_webs_path = $fw_topdir."/liberic_webs/src";
  my $docu_filepath = "/doc/oem/";
  my $perl_script = "fmparaex.pl";
  my $docu_temp_file = "form_doc_tmpl.txt";
  my $complete_doc_file = $fw_topdir.$docu_filepath."OEM_website_documentation.txt";
  
  my $script = $fw_topdir.$docu_filepath.$perl_script;
  my $tmpl = $fw_topdir.$docu_filepath.$docu_temp_file;
  
  my $command = "$script $tmpl $liberic_webs_path > $complete_doc_file 2>/dev/null";
    
  if (system($command)== 0 ) {
      printf ("%-*s: %s\n", $col_width, "Documentation", $complete_doc_file);
  }
}

#
###############################################################################
# package the oem sdk template fitting to this firmware
#

sub build_oem_sdk($$$$$) {
  my $product = shift;
  my $version = shift;
  my $build = shift;
  my $oem = shift;
  my $lightoem = shift;
  my $template_tgz = "oem_template.tgz";
  my $fw_topdir = get_config_sh_var("FW_TOPDIR");
  my @filelist = ("doc/oem/aspfunctions.txt",
		  "doc/oem/readme.txt",
		  "doc/oem/replacepart_oem.txt",
		  "doc/oem/OEM_website_documentation.txt",
		  "libpp_um/src/pwdhash/pwdhash",
		  "mkfirmware/replacepart_oem.pl",
		  "mkfirmware/replacepart_oem.pm",
		  "mkfirmware/eRIC_Firmware.pm",
		  "mkfirmware/$template_tgz",
		  "build_sys/pp_hw_data.pm");
  my $template_dir = get_config_sh_var("OEM_TEMPLATEDIR");
  my $temp_dir = "oemsdk_${product}_${version}_${build}_${oem}";
  my $tgz_name;
  if (-e $temp_dir && -d $temp_dir) {
    rmtree($temp_dir, 0, 0);
  }
  
  if ($lightoem eq "") {
    $tgz_name = "oemsdk_${product}_${version}_${build}_${oem}.tgz";
  } else {
    $tgz_name = "oemsdk_${product}_${version}_${build}_${oem}_${lightoem}.tgz";
  }
  
  mkpath("$temp_dir", 0);
  
  build_documentation();
     
  system("tar -C $template_dir/ -c -z -f $template_tgz .") == 0
    or ( print "Error: Cannot create $template_tgz\n" and return -1 );

  foreach my $file (@filelist) {
    copy("$fw_topdir/$file", $temp_dir) or
      ( print "Error: Can't copy $fw_topdir/$file ($!)\n" and return -1 );
  }

  system("tar -c -z -f $tgz_name $temp_dir") == 0
    or ( print "Error: Cannot create $tgz_name\n" and return -1 );

  rmtree($template_tgz, 0, 0);
  rmtree($temp_dir, 0, 0);

  printf ("%-*s: %s\n", $col_width, "OEM SDK", $tgz_name);

  return 0;
}

#
###############################################################################
# package the i18n sdk template fitting to this firmware
#

sub build_i18n_sdk($$$$$) {
  my $product = shift;
  my $version = shift;
  my $build = shift;
  my $oem = shift;
  my $lightoem = shift;
  my $template_tgz = "i18n_template.tgz";
  my $fw_topdir = get_config_sh_var("FW_TOPDIR");
  my @filelist = ("doc/oem/replacepart_oem.txt",
		  "mkfirmware/replacepart_i18n.pl",
		  "mkfirmware/replacepart_oem.pm",
		  "mkfirmware/eRIC_Firmware.pm",
		  "mkfirmware/$template_tgz",
		  "mkfirmware/localize_firmware.sh",
		  "build_sys/pp_hw_data.pm");
  my $temp_dir = "i18nsdk_${product}_${version}_${build}_${oem}";
  my $tgz_name;
  if (-e $temp_dir && -d $temp_dir) {
    rmtree($temp_dir, 0, 0);
  }
  
  if ($lightoem eq "") {
    $tgz_name = "i18nsdk_${product}_${version}_${build}_${oem}.tgz";
  } else {
    $tgz_name = "i18nsdk_${product}_${version}_${build}_${oem}_${lightoem}.tgz";
  }
  
  mkpath("$temp_dir", 0);
  
  build_documentation();
     
  foreach my $file (@filelist) {
    copy("$fw_topdir/$file", $temp_dir) or
      ( print "Error: Can't copy $fw_topdir/$file ($!)\n" and return -1 );
  }

  system("tar -c -z -f $tgz_name $temp_dir") == 0
    or ( print "Error: Cannot create $tgz_name\n" and return -1 );

  rmtree($temp_dir, 0, 0);

  printf ("%-*s: %s\n", $col_width, "i18n SDK", $tgz_name);

  return 0;
}

#
###############################################################################
# set the version information for the root filesystem
#

sub set_initrd_version( $$$$$ ) {
  my $rootfs_file = shift;
  my $version = shift;
  my $build_no = shift;
  my $downgrade_version = shift;
  my $tag = shift;
  my $suc;

  if ($rootfs_file eq "") {
    print("********************************************************\n");
    print("*** WARNING: building firmware image without initrd.bin\n");
    print("*** version information will be ignored\n");
    return;
  }

  my $ver_patch_call = "sudo @{[get_hal_wrapper()]} ./patch_version.pl ".
			"-v '$version' -n '$build_no' -t '$tag' ".
			(($downgrade_version ne "") ? "-M '$downgrade_version' " : "").
			"target_root/lib/libpp_firmware.so.1.0";

  mount_initrd($rootfs_file, $flash_map_name);
  $suc = (system($ver_patch_call) == 0);
  umount_initrd_save($rootfs_file, $flash_map_name);
  if (!($suc)) { die "Error: system ($ver_patch_call) failed: $?\n"; }
}
