#!/usr/bin/perl -w
###############################################################################
#
#      replacepart.pl
#        use to add/replace/delete firmware parts
#

use strict;

#
###############################################################################
# imports
#

use FindBin;
use lib "$FindBin::Bin";
use lib "$FindBin::Bin/cpan";

use Getopt::Std;
use eRIC_Firmware;

#
###############################################################################
# prototypes
#
sub get_subpart_usage($$);

#
###############################################################################
# global variables
#

my $opts  = {};

my $usage =
  "Usage: \n\n" .
  "  replacepart.pl [-d] [-a <part_file>] [-p <part_type>] [-s <part_subtype>] \n" .
  "                 [-t <tag>] [-v <version>] [-m <version>] [-M <version>] \n" .
  "                 <firmware_file>\n\n" .
  "    -d                - delete part\n" .
  "    -a <part_file>    - add or replace part with <part_file>\n" .
  "    -p <part_type>    - partition type to remove, add or replace\n" .
  "                        0 (flash)\n" .
  "                        4 (flashdisc directory)\n" .
  "    -s <part_subtype> - part subtype to remove, add or replace, depends on -p \n" .
  "                        (p=0)  " . get_subpart_usage(get_subpart_types(0), 0) . "\n" .
  "                        (p=4)  " . get_subpart_usage(get_subpart_types(4), 0) . "\n" .
  "    -t <tag>          - change firmware tag\n" .
  "    -v <version>      - change firmware version number without dots, eg 040100\n" .
  "    -m <version>      - change firmware minimum requested version\n" .
  "    -M <version>      - change firmware minimum downgrade version\n" .
  "    <firmware_file>   - the firmware file to change (must exist)\n\n";

#
###############################################################################
# main program
#

getopts('da:p:s:t:v:m:M:', $opts);

my $fw_file = shift @ARGV;  # first remaining element after getopts ...

my $have_partfile = 0;
my $have_part_type = 0;
my $have_part_subtype = 0;

my $have_tag = 0;
my $have_version = 0;
my $have_min_ver = 0;
my $have_down_ver = 0;

defined($fw_file) or die "firmware file not specified\n\n$usage";

my $delete_part = 0;
if (defined($opts->{d})) {
    $delete_part = 1;
}

# argument of -v is the firmware version
my $ver;
if (defined($opts->{v})) {
    $opts->{v} =~ /^[0-9]{6}$/ or die "-v parameter is incorrect\n\n$usage";
    $ver = $opts->{v};
    $have_version = 1;
}

my $partfile;
if (defined($opts->{a})) {
    my $pa = $opts->{a};
    if (-r $pa) {
	$partfile = $pa;
	$have_partfile = 1;
    } else {
	die "file \"$pa\" not readable.";
    }
}

my $part_type;
if (defined($opts->{p})) {
    $part_type = int($opts->{p});

    # this check is quite crude and not sufficient
    if (0 <= $part_type && $part_type <= 9) {
	$have_part_type = 1;
    } else {
	die "unsupported part type " . $part_type . ".\n";	
    }
}

my $part_subtype;
if (defined($opts->{s})) {
    $part_subtype = int($opts->{s});
    # this check is quite crude and not sufficient
    if (0 <= $part_subtype && $part_subtype <= 5) {
	$have_part_subtype = 1;
    } else {
	die "unsupported part content " . $part_subtype . ".\n";	
    }
}

my $tag;
if (defined $opts->{t}) {
    $opts->{t} =~ /^[-_,.;:()A-Za-z0-9 ]{0,64}$/ or
        die "-t parameter is incorrect\n\n$usage";
    $tag = $opts->{t};
    $have_tag = 1;
}

# argument of -v is the firmware version
my $version;
if ( defined $opts->{v} ) {
    $opts->{v} =~ /^[0-9]{6}$/ or die "-v parameter is incorrect\n\n$usage";
    $version = $opts->{v};
    $have_version = 1;
}

# argument -m is minimum requested firmware version
my $min_req_ver;
if ( defined $opts->{m} ) {
    $opts->{m} =~ /^[0-9]{6}$/ or die "-m parameter is incorrect\n\n$usage";
    $min_req_ver = $opts->{m};
    $have_min_ver = 1;
}

# argument -M is minimum downgrade version for this fw
my $min_down_ver;
if ( defined $opts->{M} ) {
    $opts->{M} =~ /^[0-9]{6}$/ or die "-m parameter is incorrect\n\n$usage";
    $min_down_ver = $opts->{M};
    $have_down_ver = 1;
}

# some constraints about multiple options
if ($have_partfile && $delete_part) {
    die "replace a part and delete is not meaningful!";
}
if (($have_partfile || $delete_part) && (!$have_part_type || !$have_part_subtype)) {
    die "must specify a part number to replace or delete";
}

##################################
# params gathered, execute 

load_firmware($fw_file) or exit 1; # $fw is set here
print_fw_info_scarce(-1) or exit 1;

print "\n";
if ($delete_part) { print "will delete partition : ", $part_type, ", ", $part_subtype, "\n"; }
if ($have_partfile) { print "will replace subtype ", $part_type, ", ", $part_subtype, " with file ", $partfile, "\n"; }

# version information concerning the binary firmware header
if ($have_tag) {
    print "will set tag to: \"$tag\"\n";
    tag_firmware($tag);
}
if ($have_version) {
    print "will set version to: \"$ver\"\n";
    set_version($ver);
}
if ($have_min_ver) {
    print "will set minimum requested version to: \"$min_req_ver\"\n";
    min_req_ver_firmware($min_req_ver);
}

# save FW parts to disk
my $parts = get_and_save_parts("replacepart_");

if ($delete_part || $have_partfile) {
    # remove the part which should be deleted or replaced
    @{$parts} = grep(!($_->{"part_type"} == $part_type && $_->{"part_subtype"} == $part_subtype), @{$parts});
}

if ($have_partfile) {
    my $par;
    $par->{"file"} = $partfile;
    $par->{"part_type"} = $part_type;
    $par->{"part_subtype"} = $part_subtype;
    push @{$parts}, $par;
}

# now update version information in initrd
# initrd could be any partition, even a new one
if ($have_tag || $have_version || $have_min_ver || $have_down_ver) {
    my $part;
    my $rootfs_file = "";
    foreach $part (@{$parts}) {
        if (($part->{"part_type"}==0) && ($part->{"part_subtype"}==1)) {
            $rootfs_file = $part->{"file"};
        }
    }
    if ($rootfs_file ne "") {
        my $flash_map_name = get_flash_map_name($fw->{"hdr"}->{"flash_map_sel"});
        my $version_call = "sudo @{[get_hal_wrapper()]} ./patch_version.pl";

        if ($have_version) {
            $version_call .= " -v $version";
        }

        if ($have_down_ver) {
            $version_call .= " -M $min_down_ver";
        }

        if ($have_tag) {
            $version_call .= " -t \"$tag\"";
        }

        $version_call .= " target_root/lib/libpp_firmware.so.1.0";

        mount_initrd( $rootfs_file, get_flash_map_name( $fw->{"hdr"}->{"flash_map_sel"} ) );
        my $suc = (system($version_call) == 0);
        umount_initrd_save( $rootfs_file, get_flash_map_name( $fw->{"hdr"}->{"flash_map_sel"} ) );
        if (!($suc)) { die "Error: system $version_call failed: $?\n"; }
    }
}



# reload partitions and recreate the firmware
if ($delete_part || $have_partfile || $have_tag || $have_version || $have_min_ver || $have_down_ver) {
    update_parts ( $parts, 3 ) or exit 1;

    print "\nResulting firmware file:\n";

    save_firmware($fw_file) or exit 1;
    load_firmware($fw_file) or exit 1;
    print_fw_info_scarce(-1) or exit 1;
}


###############################################################################
# subs

sub get_subpart_usage($$) {
  my $subparts = shift;
  my $indent = shift;
  my $ret;
  my $sep = "";

  foreach my $subpart (keys %$subparts) {
    $ret .= $sep . ' ' x $indent . "$subpart " .
      (($subparts->{"$subpart"}->{"name"}) ? ("(" . $subparts->{"$subpart"}->{"name"} . ")") : "");
    $sep = ", ";
  }

  return $ret;
}

#
###############################################################################

