#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <linux/rtc.h>

#ifdef PP_FW_TYPE_PEMX
# define RTC_DEV "/dev/misc/rtc"
#else
# define RTC_DEV "/dev/rtc"
#endif

static int
set_hardware_clock_rtc(const struct tm *new_broken_time) {
    /* ---------------------------------------------------------------------
     * Set the Hardware Clock to the broken down time <new_broken_time>.
     * Use ioctls to "rtc" device /dev/rtc.
     * --------------------------------------------------------------------- */
    int rc = -1;
    int rtc_fd;

    if ((rtc_fd = open(RTC_DEV, O_RDONLY)) == -1) {
	fprintf(stderr, "Unable to open %s", RTC_DEV);
	exit(1);
    }

    rc = ioctl(rtc_fd, RTC_SET_TIME, new_broken_time);

    if (rc == -1) {
	fprintf(stderr, "ioctl(RTC_SET_TIME) to %s failed (%s).\n", RTC_DEV,
		strerror(errno));
	exit(1);
    }

    close(rtc_fd);

    return 0;
}

static float 
time_diff(struct timeval subtrahend, struct timeval subtractor)
{
    /* ---------------------------------------------------------------------
     * The difference in seconds between two times in "timeval" format.
     * --------------------------------------------------------------------- */
    return( (subtrahend.tv_sec - subtractor.tv_sec) +
	    (subtrahend.tv_usec - subtractor.tv_usec) / 1E6 );
}

static void
set_hardware_clock(const time_t newtime)
{
    /* ---------------------------------------------------------------------
     * Set the Hardware Clock to the time <newtime>, in local time zone.
     * --------------------------------------------------------------------- */
    struct tm new_broken_time;  

    /*
     * Time to which we will set Hardware Clock, in broken down format, in
     * the time zone of caller's choice
     */

    new_broken_time = *localtime(&newtime);
    set_hardware_clock_rtc(&new_broken_time);
}

void
set_hardware_clock_exact(const time_t settime, const struct timeval ref_time)
{
    /* ----------------------------------------------------------------------
     * Set the Hardware Clock to the time "settime", in local time zone or
     * UTC, according to "universal".
     *
     * But correct "settime" and wait for a fraction of a second so that
     * "settime" is the value of the Hardware Clock as of system time
     * "ref_time", which is in the past.  For example, if "settime" is
     * 14:03:05 and "ref_time" is 12:10:04.5 and the current system
     * time is 12:10:06.0: Wait .5 seconds (to make exactly 2 seconds since
     * "ref_time") and then set the Hardware Clock to 14:03:07, thus
     * getting a precise and retroactive setting of the clock.
     *
     * (Don't be confused by the fact that the system clock and the Hardware
     * Clock differ by two hours in the above example.  That's just to
     * remind  you that there are two independent time scales here).
     *
     * This function ought to be able to accept set times as fractional
     * times. Idea for future enhancement.
     *
     * --------------------------------------------------------------------- */
    time_t newtime;  /* Time to which we will set Hardware Clock */
    struct timeval now_time;  /* locally used time */

    gettimeofday(&now_time, NULL);
    newtime = settime + (int) time_diff(now_time, ref_time) + 1;
  
    /* Now delay some more until Hardware Clock time newtime arrives */
    do gettimeofday(&now_time, NULL);
    while (time_diff(now_time, ref_time) < newtime - settime);
  
    set_hardware_clock(newtime);
}
