/*
 * (C) Copyright 2002
 * Sysgo Real-Time Solutions, GmbH <www.elinos.com>
 * Marius Groeger <mgroeger@sysgo.de>
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include "cpe.h"
#include <asm/u-boot.h>
#include <asm/global_data.h>
#include "kira/kira100.h"
#include "kira/porting.h"
#include "kira/kira_init.h"

#ifdef PP_FEAT_SYSOK_BLINKING
# define PP_GPIO_STATUS_LED	3
static volatile unsigned char led_state = 0;
#endif

#ifdef PP_FEAT_PSU_ADAPT
# define PP_GPIO_PSU_STDBY	8
# define PP_GPIO_PSU_ONOFF	13
# define PP_GPIO_PSU_ISOLATE	11
#endif

/* ------------------------------------------------------------------------- */


/*
 * Miscelaneous platform dependent initialisations
 */

int board_init(void)
{
    DECLARE_GLOBAL_DATA_PTR;

    /* arch number MACH_TYPE_EDB7312 */
    gd->bd->bi_arch_number = 91;

    /* location of boot parameters */
    /* chkr: FIXME why was this changed from 0x6000 to 0x100? */
    gd->bd->bi_boot_params = 0x100;

    kira_hwid_init();
    kira_init_pinmux();
    kira_gpio_init();
    
    return 0;
}


#define SDRAM_MODULE_WIDTH_4			0
#define SDRAM_MODULE_WIDTH_8			(1<<12)
#define SDRAM_MODULE_WIDTH_16			(2<<12)
#define SDRAM_MODULE_WIDTH_32			(3<<12)

// 16M bit, 64M bit,...
#define SDRAM_MODULE_SIZE_16M			0
#define SDRAM_MODULE_SIZE_64M			(1<<8)
#define SDRAM_MODULE_SIZE_128M			(2<<8)
#define SDRAM_MODULE_SIZE_256M			(3<<8)

#define SDRAM_BUS_WIDTH_8						0
#define SDRAM_BUS_WIDTH_16					(1<<4)
#define SDRAM_BUS_WIDTH_32					(2<<4)

// 1M byte, 2M byte,...
#define SDRAM_BANK_SIZE_1M				0
#define SDRAM_BANK_SIZE_2M				1
#define SDRAM_BANK_SIZE_4M				2
#define SDRAM_BANK_SIZE_8M				3
#define SDRAM_BANK_SIZE_16M				4
#define SDRAM_BANK_SIZE_32M				5
#define SDRAM_BANK_SIZE_64M				6
#define SDRAM_BANK_SIZE_128M			7
#define SDRAM_BANK_SIZE_256M			8
#define SDRAM_BANK_SIZE_512M			9

#define CPE_BANK_REG_BASE 				0x10
#define CPE_SDRAM_CONFIG1 				0x08
#define CPE_BANK_REG_ENABLE				(0x1 << 12)

void sdram_config(int bank_num, u32 bank_base, int data_width, int sdram_size, int bus_width, int bank_size)
{
	u32 config_reg_val;
	
	config_reg_val = (1UL<<28) | ((bank_base>>4)&0x0fff0000) | data_width | sdram_size | bus_width | bank_size;
	
	cpe_outl( CPE_SDRAMC_BASE + CPE_BANK_REG_BASE + bank_num*4, config_reg_val);
}



int dram_init(void)
{
      DECLARE_GLOBAL_DATA_PTR;

#ifdef not_complete_yet
	sdram_config( 0, 0x0,    	SDRAM_MODULE_WIDTH_8, SDRAM_MODULE_SIZE_64M, SDRAM_BUS_WIDTH_32, SDRAM_BANK_SIZE_32M);
	sdram_config( 1, 0x02000000, SDRAM_MODULE_WIDTH_8, SDRAM_MODULE_SIZE_64M, SDRAM_BUS_WIDTH_32, SDRAM_BANK_SIZE_32M);
	sdram_config( 2, 0x04000000, SDRAM_MODULE_WIDTH_8, SDRAM_MODULE_SIZE_64M, SDRAM_BUS_WIDTH_32, SDRAM_BANK_SIZE_32M);
	sdram_config( 3, 0x06000000, SDRAM_MODULE_WIDTH_8, SDRAM_MODULE_SIZE_64M, SDRAM_BUS_WIDTH_32, SDRAM_BANK_SIZE_32M);
#endif /* end_of_not */

	int i;
	u32 config_reg_ptr;
	u32 config_reg_val;
	u32 phys_sdram_size = 0;

	u32 banksize = 0;

	config_reg_val = cpe_inl(CPE_SDRAMC_BASE+CPE_SDRAM_CONFIG1);
	switch (config_reg_val & 0x0f)
		{
			case SDRAM_BANK_SIZE_1M:	banksize = 0x100000;		break;
			case SDRAM_BANK_SIZE_2M:	banksize = 0x200000;		break;
			case SDRAM_BANK_SIZE_4M:	banksize = 0x400000;		break;
			case SDRAM_BANK_SIZE_8M:	banksize = 0x800000;		break;
			case SDRAM_BANK_SIZE_16M:	banksize = 0x1000000;		break;
			case SDRAM_BANK_SIZE_32M:	banksize = 0x2000000;		break;
			case SDRAM_BANK_SIZE_64M:	banksize = 0x4000000;		break;
			case SDRAM_BANK_SIZE_128M:	banksize = 0x8000000;		break;
			case SDRAM_BANK_SIZE_256M:	banksize = 0x10000000;		break;
			case SDRAM_BANK_SIZE_512M:	banksize = 0x20000000;		break;
		}

	for (i=0; i<4; ++i)
	{
		config_reg_ptr = CPE_SDRAMC_BASE + CPE_BANK_REG_BASE + i*4;
		config_reg_val = cpe_inl(config_reg_ptr);
		
		if ( (config_reg_val & CPE_BANK_REG_ENABLE) == 0)
		{
			continue;
		}
		phys_sdram_size += banksize;
		
	}
    gd->bd->bi_dram[0].start = PHYS_SDRAM_0;
    gd->bd->bi_dram[0].size  = phys_sdram_size;

    return 0;
}

void kira_hwid_init(void)
{
    volatile unsigned long *smc_regs  = (unsigned long *)CPE_SRAMC_BASE;
    
    smc_regs[KIRA_SMC_REG_CONFIG_3] =
	KIRA_SMC_REG_CONFIG_BNK_EN |
	(CPE_HWID_BASE & KIRA_SMC_REG_CONFIG_BNK_BASE) |
	(KIRA_SMC_BNK_SIZE_32KB << KIRA_SMC_REG_CONFIG_BNK_SIZE_SHIFT) |
	(KIRA_SMC_BNK_MBW_8BIT << KIRA_SMC_REG_CONFIG_BNK_MBW_SHIFT);    
}

void kira_gpio_init(void)
{
#if defined(PP_FEAT_SYSOK_BLINKING) || defined(PP_FEAT_PSU_ADAPT)
    unsigned long *gpio_regs  = (unsigned long *)CPE_GPIO_BASE;
#endif /* PP_FEAT_SYSOK_BLINKING || PP_FEAT_PSU_ADAPT */
    
#if defined(PP_FEAT_SYSOK_BLINKING)
    /* set GPIO3 (LED) to output */
    gpio_regs[KIRA_GPIO_PIN_DIR]  |= (0x01 << PP_GPIO_STATUS_LED);
    gpio_regs[KIRA_GPIO_DATA_OUT] &= ~(0x01 << PP_GPIO_STATUS_LED);
#endif

#ifdef PP_FEAT_PSU_ADAPT
   
    /* GPIO 13 ONOFF clear bit */    
    gpio_regs[KIRA_GPIO_DATA_OUT] &= ~(0x01 << PP_GPIO_PSU_ONOFF);

    /* GPIO 08 STDBY enable stdby voltage && GPIO 11 ISOLATE enable power button during start*/
    gpio_regs[KIRA_GPIO_DATA_OUT] |= ((0x01 << PP_GPIO_PSU_STDBY) | (0x01 << PP_GPIO_PSU_ISOLATE));

    /*  set pins to output */
    gpio_regs[KIRA_GPIO_PIN_DIR]  |= ((0x01 << PP_GPIO_PSU_STDBY) | (0x01 << PP_GPIO_PSU_ISOLATE) | (0x01 << PP_GPIO_PSU_ONOFF));
    
#endif
}

/**
 * function which returns the revision of KIRA100
 */

unsigned int kira_get_revision(void)
{
    /* we can determine the KIRA revision by two values:
       the VSC revision and the minor revision
       the table for these values is as follows:

       VSC rev      minor rev       KIRA rev
       2.0          0               1.0
       2.0          1               1.1
       2.1          0               2.0
    */
    unsigned int vsc_rev = *((unsigned long*)CPE_VSCREG_BASE) & 0xff;
    unsigned int minor_rev = (*((unsigned long*)CPE_PMU_BASE) & KIRA_PMU_REG_MINORREV_MINORREV) ? 1 : 0;

    if ((vsc_rev == 0x12) && (minor_rev == 1)) return KIRA_MAKEMAJORMINOR(2,0);
    else if ((vsc_rev == 0x02) && (minor_rev == 1)) return KIRA_MAKEMAJORMINOR(1,1);
    else return KIRA_MAKEMAJORMINOR(1,0);
}


unsigned char checkboard (void)
{
    unsigned char  hw_id = 0x00;

#if !defined(KIRA_CPE) && !defined(KIRA_KIMGIGAG4)
    volatile unsigned char *hwid_base = (unsigned char *)CPE_HWID_BASE;  
    hw_id = *hwid_base;
#endif
    
    return hw_id;
}

#if defined CFG_JFFS_CUSTOM_PART
#include <jffs2/jffs2.h>

static struct part_info part;
static int current_part = -1;

struct part_info*
jffs2_part_info(int part_num)
{
    if (part_num == 0) {
	if (current_part == part_num)
		return &part;

        memset(&part, 0, sizeof(part));
        part.offset = CFG_FLASH_BASE;
        part.size = PP_HWID_20_FLASH_SIZE - CFG_MONITOR_LEN;

        /* Mark the struct as ready */
	current_part = part_num;

        return &part;
    }
    return 0;
}
#endif /* CFG_JFFS_CUSTOM_PART */


#ifdef PP_FEAT_SYSOK_BLINKING
void
toggle_led_state(void)
{
    volatile unsigned long *gpio_regs  = (unsigned long *)CPE_GPIO_BASE;
    ulong iflag = disable_interrupts();
    
    unsigned long reg = gpio_regs[KIRA_GPIO_DATA_OUT];
    reg &= ~(0x01 << PP_GPIO_STATUS_LED);
    reg |= ((led_state) << PP_GPIO_STATUS_LED);
    gpio_regs[KIRA_GPIO_DATA_OUT] = reg;
    led_state = (led_state) ? 0 : 1;
    
    if (iflag) enable_interrupts();
}
#endif
