/*****************************************************************************
 *  Analog Devices Analog Flat Panel Interface Driver for initializing,
 *  configuring, and testing the device
 *
 *  FILE: ad9888.c
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential 
 * proprietary information distributed solely pursuant to a confidentiality 
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <common.h>
#include <i2c.h>
#include <ad9888.h>

#ifdef CONFIG_KXGEN2
#include <pca9543a.h>
#endif

ad9888_dev_t addev[AD9888_MAX_DEV];
static u32 ad9888_driver_init = AD9888_UNINIT;
static u32 ad9888_cnt = 0;


#ifdef CONFIG_KXGEN2
static void do_i2c_switch( u32 dev )
{
    /*
     * The AD9888 chips only has 2 valid addresses.  KX2.0 have a maximum
     * of 4 AD9888 chips.  Thus, an i2c switch is required.
     */
    switch( dev ) {
        case 0:
        case 1:
            pca9543a_select_channel(0);
            break;

        case 2:
        case 3:
            pca9543a_select_channel(1);
            break;

        default:
            break;
    }

    return;
}
#endif

s32 ad9888_check_param( u32 dev, char *func )
{
    /* check for initialization */
    if(ad9888_driver_init != AD9888_INITED) {
        printf("%s: AD9888 driver not initialized!", func);
        return( AD9888_FAIL);
    }

    /* check for valid device number */
    if(dev > ad9888_cnt) {
        printf("%s: Invalid device number!", func);
        return( AD9888_FAIL);
    }

    return( AD9888_SUCCESS );
}

u8 ad9888_read( u32 dev, u8 reg )
{
    ad9888_dev_t *plcd;
    u8 data;

    /* get the device structure and base address */
    plcd = &(addev[dev]);

#ifdef CONFIG_KXGEN2
    do_i2c_switch(dev);
#endif

    i2c_read(plcd->ba, reg, 1, &data, 1);

    return (data);
}

void ad9888_write( u32 dev, u8 reg, u8 val )
{
    ad9888_dev_t *plcd;

    /* get the device structure and base address */
    plcd = &(addev[dev]);

#ifdef CONFIG_KXGEN2
    do_i2c_switch(dev);
#endif

    i2c_write(plcd->ba, reg, 1, &val, 1);

    return;
}

s32 ad9888_init( u32 num_devices )
{
    if(num_devices > AD9888_MAX_DEV) {
        printf("%s: Device count is over limit!  Change AD9888_MAX_DEV.\n",
               __FUNCTION__);
        return( AD9888_FAIL);
    }

    ad9888_cnt = num_devices;
    ad9888_driver_init = AD9888_INITED;

    return( AD9888_SUCCESS );
}

s32 ad9888_install( u32 dev, u32 base_addr )
{
    if(ad9888_check_param(dev, __FUNCTION__) != AD9888_SUCCESS) {
        return( AD9888_FAIL);
    }

    /* assign the base address for this AD9888 */
    addev[dev].ba = base_addr;

    return( ad9888_reset(dev) );
}

s32 ad9888_reset( u32 dev )
{
    if(ad9888_check_param(dev, __FUNCTION__) != AD9888_SUCCESS) {
        return( AD9888_FAIL);
    }

    /* reset device info  */
    addev[dev].state = 0;
    addev[dev].imr = 0;

    return( AD9888_SUCCESS );
}

s32 ad9888_test( u32 dev )
{
    ad9888_dev_t *plcd;
    int i, testreg;
    u8  ba;
    u8  rval;
    u8  test_pattern[28] = {
        0x00, 0x11, 0x22, 0x33,
        0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xaa, 0xbb,
        0xcc, 0xdd, 0xee, 0xff,
        0xa5, 0x5a, 0xf0, 0x0f,
        0x01, 0x23, 0x45, 0x67,
        0x89, 0xab, 0xcd, 0xef
    };

    if(ad9888_check_param(dev, __FUNCTION__) != AD9888_SUCCESS) {
        return( AD9888_FAIL);
    }

    /* get the device structure and base address */
    plcd = &(addev[dev]);
    ba = plcd->ba;

#ifdef CONFIG_KXGEN2
    do_i2c_switch(dev);
#endif

    /* verify some of the default values */
#ifdef CHECK_DEFAULT_VALUES
    i2c_read(ba, AD9888_CLAMP_PLACEMENT, 1, &rval, 1);
    if(rval != AD9888_DEF_CLAMP_PLACEMENT) {
        printf("ERROR: CLAMP PLACEMENT read 0x%02x, expected 0x%02x\n",
               rval, AD9888_DEF_CLAMP_PLACEMENT);
        return( AD9888_FAIL );
    }

    i2c_read(ba, AD9888_CLAMP_DURATION, 1, &rval, 1);
    if(rval != AD9888_DEF_CLAMP_DURATION) {
        printf("ERROR: CLAMP DURATION read 0x%02x, expected 0x%02x\n",
               rval, AD9888_DEF_CLAMP_DURATION);
        return( AD9888_FAIL );
    }

    i2c_read(ba, AD9888_RED_GAIN, 1, &rval, 1);
    if(rval != AD9888_DEF_GAIN) {
        printf("ERROR: RED GAIN read 0x%02x, expected 0x%02x\n",
               rval, AD9888_DEF_GAIN);
        return( AD9888_FAIL );
    }

    i2c_read(ba, AD9888_GREEN_GAIN, 1, &rval, 1);
    if(rval != AD9888_DEF_GAIN) {
        printf("ERROR: GREEN GAIN read 0x%02x, expected 0x%02x\n",
               rval, AD9888_DEF_GAIN);
        return( AD9888_FAIL );
    }

    i2c_read(ba, AD9888_BLUE_GAIN, 1, &rval, 1);
    if(rval != AD9888_DEF_GAIN) {
        printf("ERROR: BLUE GAIN read 0x%02x, expected 0x%02x\n",
               rval, AD9888_DEF_GAIN);
        return( AD9888_FAIL );
    }

    i2c_read(ba, AD9888_PRE_COAST, 1, &rval, 1);
    if(rval != AD9888_DEF_COAST) {
        printf("ERROR: PRE-COAST read 0x%02x, expected 0x%02x\n",
               rval, AD9888_DEF_COAST);
        return( AD9888_FAIL );
    }
#endif

    /*
     * We should be able to write to the TEST Registers and read back the
     * same value
     */
    for(testreg = AD9888_TEST_REG1; testreg <= AD9888_TEST_REG2; testreg++) {
        for(i = 0; i < 28; i++) {
            i2c_write(ba, testreg, 1, &test_pattern[i], 1);
            i2c_read(ba, testreg, 1, &rval, 1);
            if(rval != test_pattern[i]) {
                printf("%s: Read 0x%02x Expected 0x%02x\n", __FUNCTION__,
                       rval, test_pattern[i]);
                return( AD9888_FAIL );
            }
        }
    }

    return( AD9888_SUCCESS );
}

void ad9888_print( uchar reg, uchar val )
{
    switch( reg ) {
        case AD9888_CHIP_REV:
            printf("Chip Revision            (00h): 0x%02x\n", val);
            break;
        case AD9888_PLL_DIV_MSB:
            printf("PLL Div MSB              (01h): 0x%02x\n", val);
            break;
        case AD9888_PLL_DIV_LSB:
            printf("PLL Div LSB              (02h): 0x%02x\n", val);
            break;
        case AD9888_VCO:
            printf("VCO/CPMP                 (03h): 0x%02x\n", val);
            break;
        case AD9888_PHASE_ADJUST:
            printf("Phase Adjust             (04h): 0x%02x\n", val);
            break;
        case AD9888_CLAMP_PLACEMENT:
            printf("Clamp Placement          (05h): 0x%02x\n", val);
            break;
        case AD9888_CLAMP_DURATION:
            printf("Clamp Duration           (06h): 0x%02x\n", val);
            break;
        case AD9888_HSYNC_OUTPUT_PULSEWIDTH:
            printf("Hsync Output Pulsewidth  (07h): 0x%02x\n", val);
            break;
        case AD9888_RED_GAIN:
            printf("Red Gain                 (08h): 0x%02x\n", val);
            break;
        case AD9888_GREEN_GAIN:
            printf("Green Gain               (09h): 0x%02x\n", val);
            break;
        case AD9888_BLUE_GAIN:
            printf("Blue Gain                (0Ah): 0x%02x\n", val);
            break;
        case AD9888_RED_OFFSET:
            printf("Red Offset               (0Bh): 0x%02x\n", val);
            break;
        case AD9888_GREEN_OFFSET:
            printf("Green Offset             (0Ch): 0x%02x\n", val);
            break;
        case AD9888_BLUE_OFFSET:
            printf("Blue Offset              (0Dh): 0x%02x\n", val);
            break;
        case AD9888_SYNC_CONTROL1:
            printf("Sync Control 1           (0Eh): 0x%02x\n", val);
            break;
        case AD9888_SYNC_CONTROL2:
            printf("Sync Control 2           (0Fh): 0x%02x\n", val);
            break;
        case AD9888_SYNC_ON_GREEN:
            printf("Sync On Green            (10h): 0x%02x\n", val);
            break;
        case AD9888_SYNC_SEPARATOR_THRESHOLD:
            printf("Sync Separator Threshold (11h): 0x%02x\n", val);
            break;
        case AD9888_PRE_COAST:
            printf("Pre-COAST                (12h): 0x%02x\n", val);
            break;
        case AD9888_POST_COAST:
            printf("Post-COAST               (13h): 0x%02x\n", val);
            break;
        case AD9888_SYNC_DETECT1:
            printf("Sync Detect 1            (14h): 0x%02x\n", val);
            break;
        case AD9888_SYNC_DETECT2:
            printf("Sync Detect 2            (15h): 0x%02x\n", val);
            break;
        case AD9888_TEST_REG1:
            printf("Test Register 1          (16h): 0x%02x\n", val);
            break;
        case AD9888_TEST_REG2:
            printf("Test Register 2          (17h): 0x%02x\n", val);
            break;
        case AD9888_TEST_REG3:
            printf("Test Register 3          (18h): 0x%02x\n", val);
            break;
        case AD9888_TEST_REG4:
            printf("Test Register 4          (19h): 0x%02x\n", val);
            break;

        default:
            printf("Invalid Register Address (0x%x)\n", reg);
    }
    return;
}
